/*
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: shm.sysv.c,v 6.1.1.1 97/01/27 18:23:42 nevin Exp $
 *
 *	Function:	- SYSV shared memory low-level routines
 */

#include <lam_config.h>

#include <errno.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>
#include <terror.h>
#include <typical.h>
#include <t_types.h>

/*
 * global functions
 */
int			_shm_serverinit();
int			_shm_clientinit();
int			_shm_cleanup();

/*
 * external functions
 */
extern void		_sysv_slop_add();
extern void		_sysv_slop_del();

/*
 * global variables
 */
int			_lock_poll;		/* not used in this RPI */
int			_lock_delay;		/* not used in this RPI */

/*
 * local variables
 *
 * Semaphores 0 and 1 are used for cleanup control.
 * Semaphores 2, 3, 4 and 5 are used for transfer buffer locking as follows.
 *
 * 2 - server write lock, client read unlock
 * 3 - client read lock, server write unlock
 * 4 - client write lock, server read unlock
 * 5 - server read lock, client write unlock
 */
static struct sembuf	op_lock[5] = {
				{ 3, -1, 0 },
				{ 4, -1, 0 },
				{ 5, -1, 0 },
				{ 2, -1, 0 },
				{ 0, -1, 0 }
			};

static struct sembuf	op_trylock[4] = {
				{ 3, -1, IPC_NOWAIT },
				{ 4, -1, IPC_NOWAIT },
				{ 5, -1, IPC_NOWAIT },
				{ 2, -1, IPC_NOWAIT },
			};


static struct sembuf	op_unlock[5] = {
				{ 2, 1, 0 },
				{ 5, 1, 0 },
				{ 4, 1, 0 },
				{ 3, 1, 0 },
				{ 0, 1, 0 },
			};

static struct sembuf	sop;

#if HAVE_UNION_SEMUN
static union semun	semctl_arg;
#else
static union {
	int		val;
	struct semid_ds	*buf;
	unsigned short	*array;
} semctl_arg;
#endif


/*
 *	_shm_clientinit
 *
 *	Function:	- client side shared memory initialization
 *	Accepts:	- process
 *			- LAM msg containing server info
 *	Returns:	- 0 or LAMERROR
 */
int
_shm_clientinit(ps, msg)

struct c2c_proc		*ps;
struct nmsg		*msg;

{
	ps->cp_shm = (int) msg->nh_data[1];
	ps->cp_sem = (int) msg->nh_data[2];
/*
 * Initialize the semaphore operations.
 */
	ps->cp_lop = op_lock;
	ps->cp_top = op_trylock;
	ps->cp_uop = op_unlock;
/*
 * Attach the shared memory segment and mark it for removal.
 * It will be automatically removed once both the server and client side
 * processes exit.
 */
	ps->cp_shmbuf = (char *) shmat(ps->cp_shm, (char *) 0, 0);
	if (ps->cp_shmbuf == (char *) -1) return(LAMERROR);
    
	if (shmctl(ps->cp_shm, IPC_RMID, (struct shmid_ds *) 0) < 0) {
		return(LAMERROR);
	}

	ps->cp_shmin = ps->cp_shmbuf;
	ps->cp_shmout = ps->cp_shmbuf + SHMBUFSIZE;

	return(0);
}

/*
 *	_shm_serverinit
 *
 *	Function:	- server side shared memory initialization
 *	Accepts:	- process
 *			- LAM msg to fill with info for client
 *	Returns:	- 0 or LAMERROR
 */
int
_shm_serverinit(ps, msg)

struct c2c_proc		*ps;
struct nmsg		*msg;

{
	int			shmid;
	int			semid;
/*
 * Initialize the semaphore operations.
 */
	ps->cp_lop = op_lock + 2;
	ps->cp_top = op_trylock + 2;
	ps->cp_uop = op_unlock + 2;
/*
 * Create the shared memory segment.
 */
	shmid = shmget(IPC_PRIVATE, SHMBLKSIZE, 0600 | IPC_CREAT);
	if (shmid < 0) return(LAMERROR);
    
	ps->cp_shmbuf = (char *) shmat(shmid, (char *) 0, 0);
	if (ps->cp_shmbuf == (char *) -1) {
		shmctl(shmid, IPC_RMID, (struct shmid_ds *) 0);
		return(LAMERROR);
	}

	ps->cp_shmout = ps->cp_shmbuf;
	ps->cp_shmin = ps->cp_shmbuf + SHMBUFSIZE;
	ps->cp_shm = shmid;
/*
 * Create the semaphores.
 */
	if ((semid = semget(IPC_PRIVATE, 6, 0600 | IPC_CREAT)) < 0) {
		shmctl(shmid, IPC_RMID, (struct shmid_ds *) 0);
		return(LAMERROR);
	}
/*
 * Register id's for cleanup.
 */
	_sysv_slop_add(2, 'm', shmid, 's', semid);
/*
 * Initilize semaphores.
 */
	ps->cp_sem = semid;

	semctl_arg.val = 1;
	if (semctl(semid, 0, SETVAL, semctl_arg) < 0) return(LAMERROR);
	semctl_arg.val = 2;
	if (semctl(semid, 1, SETVAL, semctl_arg) < 0) return(LAMERROR);

	semctl_arg.val = 1;
	if (semctl(semid, 2, SETVAL, semctl_arg) < 0) return(LAMERROR);
	semctl_arg.val = 0;
	if (semctl(semid, 3, SETVAL, semctl_arg) < 0) return(LAMERROR);
    
	semctl_arg.val = 1;
	if (semctl(semid, 4, SETVAL, semctl_arg) < 0) return(LAMERROR);
	semctl_arg.val = 0;
	if (semctl(semid, 5, SETVAL, semctl_arg) < 0) return(LAMERROR);
/*
 * Set information to pass to client.
 */
	msg->nh_data[1] = (int4) shmid;
	msg->nh_data[2] = (int4) semid;

	return(0);
}

/*
 *	_shm_cleanup
 *
 *	Function:	- clean up a process's shared memory structures
 *	Accepts:	- process
 *	Returns:	- 0 or LAMERROR
 */
int
_shm_cleanup(ps)

struct c2c_proc		*ps;

{
	int			val;			/* cleanup counter */

	sop.sem_num = 1;
	sop.sem_op = -1;
	sop.sem_flg = IPC_NOWAIT;
	semctl_arg.val = 0;

	if (semop(ps->cp_sem, op_lock + 4, 1) < 0) return(LAMERROR);
	if (semop(ps->cp_sem, &sop, 1) < 0) return(LAMERROR);

	val = semctl(ps->cp_sem, 1, GETVAL, semctl_arg);
	if (val < 0) return(LAMERROR);

	if (shmdt(ps->cp_shmbuf) < 0) return(LAMERROR);

	if (semop(ps->cp_sem, op_unlock + 4, 1) < 0) return(LAMERROR);

	if (val == 0) {
/*
 * The other side of the shared area has already cleaned up so
 * we can delete the semaphores and deregister the shared structures.
 */
		if (semctl(ps->cp_sem, 0, IPC_RMID, semctl_arg) < 0) {
			return(LAMERROR);
		}
		
		_sysv_slop_del('s', ps->cp_sem);
		_sysv_slop_del('m', ps->cp_shm);
	}

	return(0);
}

/*
 *	_shm_readlock
 *
 *	Function:	- obtain a lock for reading from a process
 *	Accepts:	- process to read from
 *	Returns:	- 0 or LAMERROR
 */
int
_shm_readlock(p)

struct c2c_proc		*p;

{
	do {
		if (semop((p)->cp_sem, (p)->cp_lop, 1) == 0) {
			return(0);
		} else if (errno != EINTR) {
			return(LAMERROR);
		}
	} while (1);
}

/*
 *	_shm_writelock
 *
 *	Function:	- obtain a lock for writing to a process
 *	Accepts:	- process to write to
 *	Returns:	- 0 or LAMERROR
 */
int
_shm_writelock(p)

struct c2c_proc		*p;

{
	do {
		if (semop((p)->cp_sem, (p)->cp_lop + 1, 1) == 0) {
			return(0);
		} else if (errno != EINTR) {
			return(LAMERROR);
		}
	} while (1);
}
