/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: lamattr.c,v 6.1 96/11/23 22:52:51 nevin Rel $
 *
 *	Function:	- manage table of attribute keys
 */

#include <stdlib.h>
#include <string.h>

#include <all_hash.h>
#include <mpi.h>
#include <mpisys.h>
#include <portable.h>
#include <terror.h>
#include <typical.h>

#define DEFATTRSIZE	8
#define DEFKEYSIZE	11			/* prime number */

/*
 * external functions
 */
extern int4		next_prime();

/*
 * static variables
 */
static int		attrcount = 0;
static int		attrsize = 0;
static struct _attrkey	*attrs = 0;

/*
 *	lam_mkkey
 *
 *	Function:	- create a new attribute key
 *	Returns:	- key or ERROR
 */
int
lam_mkkey()

{
	struct _attrkey	*p;
	int		i;
/*
 * Allocate initial attribute array if needed.
 * Assign first position.
 */
	if (attrsize == 0) {
		i = DEFATTRSIZE * sizeof(struct _attrkey);
		attrs = (struct _attrkey *) malloc((unsigned) i);
		if (attrs == 0) {
			return(ERROR);
		}

		memset((char *) attrs, 0, i);
		attrsize = DEFATTRSIZE;
		p = attrs;
		i = 0;
	}
/*
 * Expand attribute array if needed.
 * Assign first position in the expanded side.
 */
	else if (attrsize == attrcount) {
		i = attrsize * sizeof(struct _attrkey);
		p = (struct _attrkey *) realloc((char *)attrs, (unsigned) i+i);
		if (p == 0) {
			return(ERROR);
		}

		attrs = p;
		p += attrsize;
		memset((char *) p, 0, i);
		i = attrsize;
		attrsize += attrsize;
	}
/*
 * Find an empty slot and assign it.
 */
	else {
		for (i = 0, p = attrs; i < attrsize; ++i, ++p) {
			if (p->ak_refcount == 0) {
				break;
			}
		}

		if (i == attrsize) {
			errno = EIMPOSSIBLE;
			return(ERROR);
		}
	}
/*
 * Reserve the slot.
 */
	++attrcount;
	p->ak_refcount = 1;
	return(i);
}

/*
 *	lam_getattr
 *
 *	Function:	- get ptr to attribute
 *	Accepts:	- attribute key
 *	Returns:	- ptr to attribute or NULL
 */
struct _attrkey *
lam_getattr(key)

int			key;

{
	struct _attrkey	*p;

	errno = EINVAL;

	if ((key >= 0) && (key < attrsize)) {
		p = attrs + key;
		if (p->ak_refcount == 0) {
			p = 0;
		}
	} else {
		p = 0;
	}

	return(p);
}

/*
 *	lam_freekey
 *
 *	Function:	- free attribute key
 *	Accepts:	- attribute key
 *	Returns:	- MPI_SUCCESS or error code
 */
int
lam_freekey(key)

int			key;

{
	struct _attrkey	*p;

	p = lam_getattr(key);
	if (p == 0 || (p->ak_flags & LAM_PREDEF) || (p->ak_refcount <= 0)) {
		return(lam_mkerr(MPI_ERR_KEYVAL, 0));
	}

	if (--(p->ak_refcount) == 0) {
		--attrcount;
	}

	return(MPI_SUCCESS);
}

/*
 *	lam_putkey
 *
 *	Function:	- attach a key to a communicator
 *	Accepts:	- communicator
 *			- key
 *	Returns:	- 0 or ERROR
 */
int
lam_putkey(comm, key)

MPI_Comm		comm;
int			key;

{
	struct _attr	elem;
	struct _attrkey	*p;
/*
 * Check if key is valid.
 */
	p = lam_getattr(key);
	if (p == 0) {
		return(ERROR);
	}
/*
 * Allocate the key hash table if needed.
 */
	if (comm->c_keys == 0) {
		comm->c_keys = ah_init(DEFKEYSIZE, (int4) sizeof(struct _attr),
					(int4) MPI_KEYVAL_INVALID, INT4_NIL);
		if (comm->c_keys == 0) {
			return(ERROR);
		}
	}
/*
 * Expand the key hash table if needed.
 */
	else if (ah_count(comm->c_keys) == ah_size(comm->c_keys)) {
		if (ah_expand(comm->c_keys,
				next_prime(1 + ah_size(comm->c_keys)))) {
			return(ERROR);
		}
	}
/*
 * Store new key in hash table.
 */
	elem.a_key = key;
	elem.a_value = 0;

	if (ah_insert(comm->c_keys, &elem)) {
		return(ERROR);
	}

	if (p->ak_refcount > 0) {
		++(p->ak_refcount);
	}

	return(0);
}

/*
 *	lam_getkey
 *
 *	Function:	- get key entry in communicator
 *	Accepts:	- communicator
 *			- key
 *	Returns:	- ptr to entry of NULL
 */
struct _attr *
lam_getkey(comm, key)

MPI_Comm		comm;
int			key;

{
	struct _attr	*p;

	if (comm->c_keys) {
		p = (struct _attr *) ah_find(comm->c_keys, key);
	} else {
		p = 0;
	}

	return(p);
}

/*
 *	lam_delkey
 *
 *	Function:	- delete key from communicator
 *	Accepts:	- communicator
 *			- key
 *	Returns:	- 0 or ERROR
 */
int
lam_delkey(comm, key)

MPI_Comm		comm;
int			key;

{
	return(ah_delete(comm->c_keys, key));
}

/*
 *	lam_nukekeys
 *
 *	Function:	- deallocate all keys (cleanup)
 */
void
lam_nukekeys()

{
	if (attrs) {
		free((char *) attrs);
		attrcount = attrsize = 0;
		attrs = 0;
	}
}
