/*
 *	Software for Humanity
 *	Public Domain
 *	GDB/RBD
 *
 *	This program is freely distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 *	$Id: pathsearch.c,v 6.1 96/11/23 19:58:13 nevin Rel $
 *
 *	Function:	- file searching
 */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <args.h>
#include <sfh.h>

/*
 * PATH environment variable separator
 */
#ifdef WIN32
#define PATHENVSEP	';'
#else
#define PATHENVSEP	':'
#endif

/*
 * public functions
 */
char			*sfh_path_find();
char			*sfh_path_env_find();

/*
 * external functions
 */
char			*getworkdir();

/*
 * local functions
 */
static char		*path_access();
static void		path_env_load();

/*
 *	sfh_path_find
 *
 *	Function:	- locates a file with certain permissions
 *			- environment variable can appear in the form
 *			  $variable at the start of a prefix path
 *			  and will be replaced by the environment value
 *			  if it is defined otherwise the whole prefix
 *			  is ignored
 *			- environment variable must be following by
 *			  a path delimiter or end-of-string
 *
 *	Accepts:	- file name
 *			- array of search directories
 *			- target permissions which must be satisfied
 *
 *	Returns:	- full pathname of located file or NULL
 */
char *
sfh_path_find(fname, pathv, mode)

char			*fname;
char			**pathv;
int			mode;

{
	char		*fullpath;	/* full pathname of search file */
	char		*delimit;	/* ptr to first delimiter in prefix */
	char		*env;		/* ptr to environment var */
	char		*pfix;		/* prefix directory */
	int		i;
/*
 * If absolute path is given, return it without searching.
 */
	if (*fname == STRDIR) {
	    return(path_access(fname, "", mode));
	}
/*
 * Initialize.
 */
	fullpath = 0;
	i = 0;
/*
 * Consider each directory until the file is found.
 * Thus, the order of directories is important.
 */
	while (pathv[i] && !fullpath) {
/*
 * Replace environment variable at the head of the string.
 */
	    if (*pathv[i] == '$') {
		delimit = strchr(pathv[i], STRDIR);

		if (delimit) {
				*delimit = '\0';
		}

		env = getenv(pathv[i] + 1);

		if (delimit) {
				*delimit = STRDIR;
		}

		if (env) {

		    if (!delimit) {
			fullpath = path_access(fname, env, mode);
		    } else {
			pfix = malloc((unsigned) strlen(env) +
				strlen(delimit) + 1);
			if (pfix == 0) return(0);

			strcpy(pfix, env);
			strcat(pfix, delimit);
			fullpath = path_access(fname, pfix, mode);
			free(pfix);
		    }
		}
	    }

	    else {
		fullpath = path_access(fname, pathv[i], mode);
	    }

	    i++;
	}

	return(fullpath);
}

/*
 *	sfh_path_env_find
 *
 *	Function:	- locates a file with certain permissions
 *			  from the list of paths given by the $PATH
 *			  environment variable
 *			- replaces ./ of found path with cwd()/
 *	Accepts:	- file name
 *			- target permissions which must be satisfied
 *	Returns:	- full pathname of located file or NULL
 */
char *
sfh_path_env_find(fname, mode)

char			*fname;
int			mode;

{
	char		**dirv;		/* search directories */
	char		*fullpath;	/* full pathname */
	int		dirc;		/* # search directories */
	int		i;
/*
 * Set the local search paths.
 */
	dirc = 0;
	dirv = 0;
	path_env_load(&dirc, &dirv);
/*
 * Replace the "." path by the current working directory.
 */
	for (i = 0; i < dirc; ++i) {

		if (strcmp(dirv[i], ".") == 0) {
			free(dirv[i]);
			dirv[i] = getworkdir();
			if (dirv[i] == 0) return(0);
		}
	}

	fullpath = sfh_path_find(fname, dirv, mode);
	argvfree(dirv);
	return(fullpath);
}

/*
 *	path_access
 *
 *	Function:	- forms a complete pathname and checks it for
 *			  existance and permissions
 *	Accepts:	- filename
 *			- path prefix
 *			- permissions which must be satisfied
 *	Returns:	- full pathname or NULL
 */
static char *
path_access(fname, path, mode)

char			*fname;
char			*path;
int			mode;

{
	char		*fullpath;	/* full pathname of search file */
/*
 * Allocate space for the full pathname.
 */
	fullpath = malloc((unsigned) strlen(path) + strlen(fname) + 2);
	if (fullpath == 0) return(0);

	if (strlen(path) > 0) {
		strcpy(fullpath, path);
		strcat(fullpath, STRSDIR);
		strcat(fullpath, fname);
	} else {
		strcpy(fullpath, fname);
	}
/*
 * Get status on the full path name to check for existance.
 * Then check the permissions.
 */
	if (access(fullpath, mode)) {
		free(fullpath);
		fullpath = 0;
	}

	return(fullpath);
}

/*
 *	path_env_load
 *
 *	Function:	- loads argument array with $PATH env var
 *	Accepts:	- argv
 *			- argc
 */
static void
path_env_load(pargc, pargv)

int			*pargc;
char			***pargv;

{
	char		*p;		/* favourite pointer */
	char		*path;		/* ptr to path */
	char		saved;		/* saved character */
/*
 * Get the PATH environment variable.
 */
	if ((path = getenv("PATH")) == 0) {
		*pargc =  0;
		return;
	}
/*
 * Loop through the paths (delimited by PATHENVSEP), adding each one to argv.
 */
	while (*path) {
/*
 * Locate the delimiter.
 */
		for (p = path; *p && (*p != PATHENVSEP); ++p);
/*
 * Add the path.
 */
		if (p != path) {
			saved = *p;
			*p = '\0';
			argvadd(pargc, pargv, path);
			*p = saved;
			path = p;
		}
/*
 * Skip past the delimiter, if present.
 */
		if (*path) {
			++path;
		}
	}
}

/*
 * backwards compatibility
 */
char *findpath(fname, pathv, mode) char *fname, **pathv; int mode;
{ return(sfh_path_find(fname, pathv, mode)); }

char *_path_env_find(fname, mode) char *fname; int mode;
{ return(sfh_path_env_find(fname, mode)); }
