/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	JRV/RBD/GDB
 *
 *	$Id: inetexec.c,v 6.1 96/11/24 00:44:54 nevin Rel $
 * 
 *	Function:	- run program on remote UNIX machine
 *	Accepts:	- hostname
 *			- username (or NULL)
 *			- argv structure
 *	Returns:	- 0 or LAMERROR
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/param.h>

#include <lam_config.h>

#include <args.h>
#include <terror.h>
#include <typical.h>

#ifndef MAXPATHLEN
#define MAXPATHLEN	1024
#endif

#ifndef RSH
#define RSH		"/usr/ucb/rsh"
#endif

/*
 * static variables
 */
static char		shellpath[MAXPATHLEN];	/* return of 'echo $SHELL' */

/*
 * static functions
 */
static int		ioexecvp();

int
inetexec(host, username, argv)

char			*host;
char			*username;
char			**argv;

{
	char		**cmdv;			/* rsh command argv */
	int		cmdc;			/* rsh command argc */
	int		fl_csh;			/* csh-flavoured flag */
	int		fl_bash;		/* bash-flavoured flag */
	int		i;			/* favorite counter */
/*
 * Get the user's shell by executing 'echo $SHELL' on remote machine.
 */
	cmdc = 0;
	cmdv = 0;

	argvadd(&cmdc, &cmdv, RSH);
	argvadd(&cmdc, &cmdv, host);
	argvadd(&cmdc, &cmdv, "-n");

	if (username && *username) {
		argvadd(&cmdc, &cmdv, "-l");
		argvadd(&cmdc, &cmdv, username);
	}

	argvadd(&cmdc, &cmdv, "echo $SHELL");

	if (ioexecvp(cmdv, 0, shellpath, sizeof(shellpath))) return(LAMERROR);

	fl_csh = (strstr(shellpath, "csh") != 0) ? TRUE : FALSE;
	fl_bash = (strstr(shellpath, "bash") != 0) ? TRUE : FALSE;
/*
 * Remotely execute the command using "rsh".
 */
	cmdc = 0;
	cmdv = 0;

	argvadd(&cmdc, &cmdv, RSH);
	argvadd(&cmdc, &cmdv, host);
	argvadd(&cmdc, &cmdv, "-n");

	if (username && *username) {
		argvadd(&cmdc, &cmdv, "-l");
		argvadd(&cmdc, &cmdv, username);
	}

#if NEED_RSH_MINUSMINUS
/*
 * Stop rsh from interpreting options for the remote command as
 * rsh options.
 */
        argvadd(&cmdc, &cmdv, "--");
#endif

/*
 * If the user's shell is not based on "csh" or "bash", force the
 * interpretation of the user's .profile script in order to initialize
 * the paths. This works for "sh" and "ksh".
 */
	if (!(fl_csh || fl_bash)) argvadd(&cmdc, &cmdv, "(. ./.profile;");
	for (i = 0; argv[i]; ++i) {
		argvadd(&cmdc, &cmdv, argv[i]);
	}

	if (!(fl_csh || fl_bash)) argvadd(&cmdc, &cmdv, ")");

	return(ioexecvp(cmdv, 1, (char *) 0, 0));
}

/*
 *	ioexecvp
 *
 *	Function:	- execute command (similar to cnfexec)
 *			- can direct command stdout to buffer and/or stdout
 *			- stderr is checked and passed through
 *	Accepts		- command argv
 *			- print stdout flag
 *			- ptr to buffer (for stdout data)
 *			- size of buffer
 *	Returns		- 0 or LAMERROR
 */
static int
ioexecvp(cmdv, showout, outbuff, outbuffsize)

char			**cmdv;
int			showout;
char			*outbuff;
int			outbuffsize;

{
	int		kidstdout[2];		/* child stdout pipe */
	int		kidstderr[2];		/* child stderr pipe */
	int		ret;			/* read() return value */
	int		err;			/* error indicator */
	int		status;			/* exit status */
	int		pid;			/* child process id */
	char		*ptr;			/* buffer pointer */
	char		c;			/* character read in */
/*
 * Create child stdout/stderr pipes and fork the child process (command).
 */
	if (pipe(kidstdout) || pipe(kidstderr)) return(LAMERROR);

	if ((pid = fork()) < 0) {
		return(LAMERROR);
	}

	else if (pid == 0) {				/* child */

		if ((dup2(kidstderr[1], 2) < 0) ||
				(dup2(kidstdout[1], 1) < 0)) {
			perror(cmdv[0]);
			exit(errno);
		}

		if (close(kidstdout[0]) || close(kidstderr[0]) || close(0) ||
				close(kidstdout[1]) || close(kidstderr[1])) {
			perror(cmdv[0]);
			exit(errno);
		}

		execvp(cmdv[0], cmdv);
		exit(errno);
	}

	if (close(kidstdout[1]) || close(kidstderr[1])) return(LAMERROR);

	argvfree(cmdv);
/*
 * Read the child's stdout stream if required.
 * The stream can be written out and/or sent to a buffer.
 * If the stream is longer than the buffer size, it is truncated.
 */
	err = 0;

	if (showout || (outbuff != 0)) {
		ptr = outbuff;

		while ((ret = read(kidstdout[0], &c, 1)) > 0) {

			if (outbuffsize > 0) {
				--outbuffsize;
				*ptr++ = c;
			}

			if (showout) {
				putc(c, stdout);
				fflush(stdout);
			}
		}

		if (ret < 0) {
			err = LAMERROR;
		}
	}
/*
 * Read the child's stderr stream.
 * Assumption: If at least one character exists in the stream,
 * the child process must have aborted with an error.
 */
	ret = read(kidstderr[0], &c, 1);

	if (ret < 0) {
		err = LAMERROR;
	} else if (ret > 0) {

		do {
			putc(c, stderr);
			fflush(stderr);
		} while ((ret = read(kidstderr[0], &c, 1)) > 0);

		errno = EUNKNOWN;
		err = LAMERROR;
	}
/*
 * Close the pipes of the parent process.
 */
	if (close(kidstdout[0]) || close(kidstderr[0])) {
		err = LAMERROR;
	}
/*
 * Wait for the command to exit.
 */
	do {
		if (waitpid(pid, &status, 0) < 0) {
			return(LAMERROR);
		}
	} while (! WIFEXITED(status));
	
	if (WEXITSTATUS(status)) {
		errno = WEXITSTATUS(status);

		if (errno == 1) {
			errno = EUNKNOWN;
		}

		return(LAMERROR);
	}

	return(err);
}
