/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <stdio.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <locale.h>

#include <gtk/gtk.h>

#include "gladeconfig.h"

#include "project.h"
#include "gbwidget.h"
#include "save.h"

#define GB_SAVE_BUFFER_INCREMENT 2048


static GbStatusCode real_save_project_file (gchar * filename);
static void save_named_style_callback (gchar * name,
				       GbStyle * gbstyle,
				       GbWidgetGetArgData * data);
static void save_named_style (gchar * name,
			      GbStyle * gbstyle,
			      GbWidgetGetArgData * data);
static void save_component (GtkWidget * item,
			    GbWidgetGetArgData * data);
static void save_indent (GbWidgetGetArgData * data);

static void save_buffer_add (GbWidgetGetArgData * data, gchar ch);
static void save_buffer_add_string (GbWidgetGetArgData * data, gchar * string);
static void save_buffer_add_chars (GbWidgetGetArgData * data, gchar * chars);
static void save_buffer_add_char (GbWidgetGetArgData * data, gchar ch);


/* We need this to make sure that numbers are output in a portable syntax,
   instead of using the current locale. This code is from glibc info docs. */
GbStatusCode
save_project_file (gchar * filename)
{
  gchar *old_locale, *saved_locale;
  GbStatusCode status;
     
  old_locale = setlocale (LC_NUMERIC, NULL);
  saved_locale = g_strdup (old_locale);
  setlocale (LC_NUMERIC, "C");
  status = real_save_project_file (filename);
  setlocale (LC_NUMERIC, saved_locale);
  g_free (saved_locale);
  return status;
}


static GbStatusCode
real_save_project_file (gchar * filename)
{
  GbWidgetGetArgData data;

  MSG ("Saving project");

  data.status = GB_OK;
  data.filename = filename;

  data.fp = fopen (filename, "w");
  if (data.fp == NULL)
    return GB_FILE_OPEN_ERROR;

  data.buffer_space = GB_SAVE_BUFFER_INCREMENT;
  data.buffer = g_new (gchar, data.buffer_space);
  data.buffer_pos = 0;

  fprintf (data.fp, "<?xml version=\"1.0\"?>\n<GTK-Interface>\n\n");

  MSG ("Saving styles");
  data.indent = 0;

  /* Save default gbstyle first. */
  save_named_style (gb_widget_default_gb_style->name,
		    gb_widget_default_gb_style, &data);
  /* Now save all the other named styles. */
  g_hash_table_foreach (gb_style_hash, (GHFunc) save_named_style_callback,
			&data);

  MSG ("Saving components");
  project_foreach_component ((GtkCallback) save_component, &data);

  fprintf (data.fp, "\n</GTK-Interface>\n");

  g_free (data.buffer);

  fclose (data.fp);
  return data.status;
}


/* This outputs one char, converting <>&" to XML entities. */
static void
save_buffer_add (GbWidgetGetArgData * data, gchar ch)
{
  if (ch == '<')
    save_buffer_add_chars (data, "&lt;");
  else if (ch == '>')
    save_buffer_add_chars (data, "&gt;");
  else if (ch == '&')
    save_buffer_add_chars (data, "&amp;");
  else if (ch == '"')
    save_buffer_add_chars (data, "&quot;");
  else
    save_buffer_add_char (data, ch);
}


static void
save_buffer_add_string (GbWidgetGetArgData * data, gchar * string)
{
  while (*string)
    {
      save_buffer_add (data, *string);
      string++;
    }
}


static void
save_buffer_add_chars (GbWidgetGetArgData * data, gchar * chars)
{
  while (*chars)
    {
      save_buffer_add_char (data, *chars);
      chars++;
    }
}


static void
save_buffer_add_char (GbWidgetGetArgData * data, gchar ch)
{
  /* Expand the buffer if necessary. */
  if (data->buffer_pos == data->buffer_space)
    {
      data->buffer_space += GB_SAVE_BUFFER_INCREMENT;
      data->buffer = g_realloc (data->buffer, data->buffer_space);
    }
  data->buffer[data->buffer_pos++] = ch;
}


static void
save_named_style_callback (gchar * name, GbStyle * gbstyle,
			   GbWidgetGetArgData * data)
{
  /* We've already saved the default GbStyle, so skip it. */
  if (gbstyle == gb_widget_default_gb_style)
    return;
  save_named_style (name, gbstyle, data);
}


static void
save_named_style (gchar * name, GbStyle * gbstyle, GbWidgetGetArgData * data)
{
  gint bytes_written;
  gboolean save_all = FALSE;

  data->buffer_pos = 0;
  data->indent = 0;
  MSG1 ("Saving style: %s", name);
  /* If this is the default style, only save it if it is different to the
     GTK default style, and if it is make sure we save everything. */
  if (gbstyle == gb_widget_default_gb_style)
    {
      if (gbstyle->style == gtk_widget_get_default_style ())
	return;
      else
	save_all = TRUE;
    }
  gb_widget_save_style (gbstyle, data, save_all);
  bytes_written = fwrite (data->buffer, sizeof (gchar), data->buffer_pos,
			  data->fp);
  if (bytes_written != data->buffer_pos)
    {
      MSG2 ("Bytes: %i Written: %i", data->buffer_pos, bytes_written);
      data->status = GB_FILE_WRITE_ERROR;
    }
}


static void
save_component (GtkWidget * component, GbWidgetGetArgData * data)
{
  gint bytes_written;

  data->buffer_pos = 0;
  data->indent = 0;
  gb_widget_save (component, data);

  bytes_written = fwrite (data->buffer, sizeof (gchar), data->buffer_pos,
			  data->fp);
  if (bytes_written != data->buffer_pos)
    {
      MSG2 ("Bytes: %i Written: %i", data->buffer_pos, bytes_written);
      data->status = GB_FILE_WRITE_ERROR;
    }
}


static void
save_indent (GbWidgetGetArgData * data)
{
  gint i;
  for (i = 0; i < data->indent * 2; i++)
    save_buffer_add (data, ' ');
}


gchar *
find_start_of_tag_name (gchar * tag_name)
{
  gint i;
  for (i = 0; i < strlen (tag_name) - 1; i++)
    {
      if (tag_name[i] == ':' && tag_name[i + 1] == ':')
	return &tag_name[i + 2];
    }
  return tag_name;
}


void
save_tag (GbWidgetGetArgData * data, gchar * tag_name)
{
  gchar *tag_name_start = find_start_of_tag_name (tag_name);
  save_indent (data);
  save_buffer_add_char (data, '<');
  save_buffer_add_string (data, tag_name_start);
  save_buffer_add_chars (data, ">\n");
}


void
save_end_tag (GbWidgetGetArgData * data, gchar * tag_name)
{
  gchar *tag_name_start = find_start_of_tag_name (tag_name);
  save_indent (data);
  save_buffer_add_chars (data, "</");
  save_buffer_add_string (data, tag_name_start);
  save_buffer_add_chars (data, ">\n");
}


void
save_newline (GbWidgetGetArgData * data)
{
  save_buffer_add_char (data, '\n');
}




void
save_string (GbWidgetGetArgData * data, gchar * tag_name, gchar * tag_value)
{
  gchar *tag_name_start;

  if (tag_value == NULL)
    return;

  tag_name_start = find_start_of_tag_name (tag_name);
  save_indent (data);
  save_buffer_add_char (data, '<');
  save_buffer_add_string (data, tag_name_start);
  save_buffer_add_char (data, '>');
  save_buffer_add_string (data, tag_value);
  save_buffer_add_chars (data, "</");
  save_buffer_add_string (data, tag_name_start);
  save_buffer_add_chars (data, ">\n");
}


void
save_text (GbWidgetGetArgData * data, gchar * tag_name, gchar * tag_value)
{
  save_string (data, tag_name, tag_value);
}


void
save_int (GbWidgetGetArgData * data, gchar * tag_name, gint tag_value)
{
  gchar buf[32];
  sprintf (buf, "%i", tag_value);
  save_string (data, tag_name, buf);
}


void
save_float (GbWidgetGetArgData * data, gchar * tag_name, gfloat tag_value)
{
  gchar buf[32];
  sprintf (buf, "%g", tag_value);
  save_string (data, tag_name, buf);
}


void
save_bool (GbWidgetGetArgData * data, gchar * tag_name, gint tag_value)
{
  save_string (data, tag_name, tag_value ? "True" : "False");
}


void
save_choice (GbWidgetGetArgData * data, gchar * tag_name, gchar * tag_value)
{
  save_string (data, tag_name, tag_value);
}


void
save_combo (GbWidgetGetArgData * data, gchar * tag_name, gchar * tag_value)
{
  save_string (data, tag_name, tag_value);
}


/* Colors are saved as r,g,b where rgb are 0-255. */
void
save_color (GbWidgetGetArgData * data, gchar * tag_name, GdkColor * tag_value)
{
  gchar buf[32];
  guint16 red = tag_value->red / 256;
  guint16 green = tag_value->green / 256;
  guint16 blue = tag_value->blue / 256;
  sprintf (buf, "%i,%i,%i", red, green, blue);
  save_string (data, tag_name, buf);
}


void
save_bgpixmap (GbWidgetGetArgData * data, gchar * tag_name, gchar * tag_value)
{
  save_string (data, tag_name, tag_value);
}


void
save_dialog (GbWidgetGetArgData * data, gchar * tag_name, gchar * tag_value)
{
  save_string (data, tag_name, tag_value);
}


void
save_filename (GbWidgetGetArgData * data, gchar * tag_name, gchar * tag_value)
{
  save_string (data, tag_name, tag_value);
}


void
save_font (GbWidgetGetArgData * data, gchar * tag_name, gchar * tag_value)
{
  save_string (data, tag_name, tag_value);
}
