

/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <ctype.h>

#include <gtk/gtk.h>

#include "gladeconfig.h"

#include "editor.h"
#include "project.h"
#include "glade_keys_dialog.h"
#include "utils.h"
#include "gbwidget.h"
#include "property.h"
#include "load.h"
#include "save.h"
#include "gtkfontsel.h"

/* These are the standard widget attribute names */
gchar *GbName = "GtkWidget::name";
gchar *GbX = "GtkWidget::x";
gchar *GbY = "GtkWidget::y";
gchar *GbWidth = "GtkWidget::width";
gchar *GbHeight = "GtkWidget::height";
gchar *GbVisible = "GtkWidget::visible";
gchar *GbSensitive = "GtkWidget::sensitive";
gchar *GbTooltip = "GtkWidget::tooltip";
gchar *GbCanDefault = "GtkWidget::can_default";
gchar *GbHasDefault = "GtkWidget::has_default";
gchar *GbCanFocus = "GtkWidget::can_focus";
gchar *GbHasFocus = "GtkWidget::has_focus";
gchar *GbEvents = "GtkWidget::events";
gchar *GbExtEvents = "GtkWidget::extension_events";

gchar *GbStylePropagate = "GtkWidget::style_propagate";
gchar *GbStyleName = "GtkWidget::style_name";
gchar *GbStyleFont = "GtkWidget::style_font";

/* For children of a table */
gchar *GbCellX = "GtkTableChild::cell_x";
gchar *GbCellY = "GtkTableChild::cell_y";
gchar *GbColSpan = "GtkTableChild::col_span";
gchar *GbRowSpan = "GtkTableChild::row_span";
gchar *GbXPad = "GtkTableChild::xpad";
gchar *GbYPad = "GtkTableChild::ypad";
gchar *GbXExpand = "GtkTableChild::xexpand";
gchar *GbYExpand = "GtkTableChild::yexpand";
gchar *GbXShrink = "GtkTableChild::xshrink";
gchar *GbYShrink = "GtkTableChild::yshrink";
gchar *GbXFill = "GtkTableChild::xfill";
gchar *GbYFill = "GtkTableChild::yfill";

/* For children of a box */
gchar *GbPadding = "GtkBoxChild::padding";
gchar *GbExpand = "GtkBoxChild::expand";
gchar *GbFill = "GtkBoxChild::fill";
gchar *GbPack = "GtkBoxChild::pack";

/* For children of a packer */
#ifdef GLD_HAVE_GTK_1_1
gchar *GbPackerSide = "GtkPackerChild::side";
gchar *GbPackerAnchor = "GtkPackerChild::anchor";
gchar *GbPackerExpand = "GtkPackerChild::expand";
gchar *GbPackerFillX = "GtkPackerChild::xfill";
gchar *GbPackerFillY = "GtkPackerChild::yfill";
gchar *GbPackerBorder = "GtkPackerChild::border_width";
gchar *GbPackerPadX = "GtkPackerChild::xpad";
gchar *GbPackerPadY = "GtkPackerChild::ypad";
gchar *GbPackerIPadX = "GtkPackerChild::xipad";
gchar *GbPackerIPadY = "GtkPackerChild::yipad";
#endif

/* Signals page */
gchar *GbSignalName = "GtkWidget::signal_name";
gchar *GbSignalHandler = "GtkWidget::signal_handler";
gchar *GbSignalObject = "GtkWidget::signal_object";
gchar *GbSignalAfter = "GtkWidget::signal_after";
gchar *GbSignalData = "GtkWidget::signal_data";

/* Accelerators page */
gchar *GbAccelKey = "GtkWidget::accelerator_key";
gchar *GbAccelSignal = "GtkWidget::accelerator_signal";


gchar *GbStateTabLabels[] =
{ N_("Normal"), N_("Active"), N_("Prelight"),
  N_("Select"), N_("Insens") };

/* Keys in object data hash used to store color, bg pixmap & filename */
gchar *GbColorKey = "GbColor";
gchar *GbBgPixmapKey = "GbBgPixmap";
gchar *GbBgFilenameKey = "GbBgFilename";
gchar *GbValueWidgetKey = "GbValue";
gchar *GbDialogValueKey = "GbDialogValue";
gchar *GbFontKey = "GbFont";
gchar *GbFontSpecKey = "GbFontSpec";

/* Hashtables of all the label & value widgets on the various property pages */
GHashTable *gb_property_labels = NULL;
GHashTable *gb_property_values = NULL;
GHashTable *gb_property_buttons = NULL;

/* Major components of the property window */
#define GB_PAGE_SIGNALS	5
GtkWidget *win_property = NULL;
GtkWidget *main_notebook = NULL;
GtkWidget *property_widget_notebook;
GtkWidget *special_child_property_notebook;
GtkWidget *apply_button = NULL;
GtkWidget *styles_notebook;

/* Extension mode choices data */
gchar *GbExtensionModeChoices[] =
{"None", "All", "Cursor", NULL};
gint GbExtensionModeValues[] =
{
  GDK_EXTENSION_EVENTS_NONE,
  GDK_EXTENSION_EVENTS_ALL,
  GDK_EXTENSION_EVENTS_CURSOR
};
gchar *GbExtensionModeSymbols[] =
{
  "GDK_EXTENSION_EVENTS_NONE",
  "GDK_EXTENSION_EVENTS_ALL",
  "GDK_EXTENSION_EVENTS_CURSOR"
};

#ifdef GLD_HAVE_GTK_1_1
/* Packer children choices. */
gchar *GbPackerSideChoices[] =
{"Top", "Bottom", "Left", "Right", NULL};
gint GbPackerSideValues[] =
{
  GTK_SIDE_TOP,
  GTK_SIDE_BOTTOM,
  GTK_SIDE_LEFT,
  GTK_SIDE_RIGHT
};
gchar *GbPackerSideSymbols[] =
{
  "GTK_SIDE_TOP",
  "GTK_SIDE_BOTTOM",
  "GTK_SIDE_LEFT",
  "GTK_SIDE_RIGHT"
};

gchar *GbPackerAnchorChoices[] =
{"Center", "North", "North West", "North East", "South", "South West",
 "South East", "West", "East", NULL};
gint GbPackerAnchorValues[] =
{
  GTK_ANCHOR_CENTER,
  GTK_ANCHOR_NORTH,
  GTK_ANCHOR_NORTH_WEST,
  GTK_ANCHOR_NORTH_EAST,
  GTK_ANCHOR_SOUTH,
  GTK_ANCHOR_SOUTH_WEST,
  GTK_ANCHOR_SOUTH_EAST,
  GTK_ANCHOR_WEST,
  GTK_ANCHOR_EAST,
};
gchar *GbPackerAnchorSymbols[] =
{
  "GTK_ANCHOR_CENTER",
  "GTK_ANCHOR_NORTH",
  "GTK_ANCHOR_NORTH_WEST",
  "GTK_ANCHOR_NORTH_EAST",
  "GTK_ANCHOR_SOUTH",
  "GTK_ANCHOR_SOUTH_WEST",
  "GTK_ANCHOR_SOUTH_EAST",
  "GTK_ANCHOR_WEST",
  "GTK_ANCHOR_EAST"
};
#endif

/*
 * Private variables
 */

static GtkTooltips *tooltips;
static GdkColor tooltips_fgcolor =
{0, 0, 0, 0};
static GdkColor tooltips_bgcolor =
{0, 0xffff, 0xffff, 0};

/* The widget whose properties are currently shown (or NULL) */
static GtkWidget *property_widget = NULL;

static GtkStyle *invalid_style;

/* The current table & row used when creating properties */
static GtkWidget *property_table;
static gint property_table_row;

/* This is used when automatically applying properties as they are changed.
   It is on most of the time, except when changes are being made to property
   values which we don't want to result in a callback. */
static gboolean auto_apply;

/* Color selection dialog */
static GtkColorSelectionDialog *colorsel = NULL;
static GtkWidget *color_value = NULL;

/* File selection dialog */
static GtkFileSelection *filesel = NULL;
static GtkWidget *filename_value = NULL;

/* Font selection fialog */
static GtkFontSelectionDialog *fontsel = NULL;

/* Widgets on Accelerators page */
#define ACCEL_MODIFIERS_COL	0
#define ACCEL_KEY_COL		1
#define ACCEL_SIGNAL_COL	2
static GtkWidget *accel_clist;
static GtkWidget *accel_control_button;
static GtkWidget *accel_shift_button;
static GtkWidget *accel_alt_button;

/* Widgets on Signals page */
#define SIGNAL_NAME_COL		0
#define SIGNAL_HANDLER_COL	1
#define SIGNAL_DATA_COL		2
#define SIGNAL_AFTER_COL	3
#define SIGNAL_OBJECT_COL	4
static GtkWidget *signal_clist;

/* Clipboard used for copying/pasting colors - could possibly use GTK clipbd? */
#define GB_COLOR_CLIPBD_EMPTY    1
#define GB_COLOR_CLIPBD_COLOR    2
#define GB_COLOR_CLIPBD_BGPIXMAP 3
#define GB_COLOR_CLIPBD_STATE    4
static gint clipbd_state = GB_COLOR_CLIPBD_EMPTY;
static guchar clipbd_colors[GB_NUM_STYLE_COLORS][4];
static GdkPixmap *clipbd_bgpixmap;
static gchar *clipbd_bgfilename;
static GtkWidget *selected_style_widget = NULL;


/*
 * Private functions
 */

static void property_create ();

static GtkWidget *create_widget_property_page ();
static GtkWidget *create_standard_property_page ();
static GtkWidget *create_special_child_properties ();
static GtkWidget *create_style_property_page ();
static GtkWidget *create_style_page (gchar * state);
static GtkWidget *create_signals_property_page ();
static GtkWidget *create_accelerators_property_page ();

static void property_add (gchar * property_name,
			  gchar * label_string,
			  GtkWidget * value,
			  GtkWidget * dialog_button,
			  gchar * tooltip);

static void on_bool_property_toggle (GtkWidget * value,
				     gpointer data);

static GtkWidget *create_color_preview ();
static void show_color_in_preview (GtkWidget * preview,
				   gpointer data);
static void on_color_draw (GtkWidget * widget,
			   gpointer data);
static void on_color_expose_event (GtkWidget * widget,
				   GdkEvent * event,
				   gpointer data);
static void on_color_select (GtkWidget * widget,
			     gpointer data);

static void show_pixmap_in_drawing_area (GtkWidget * drawing_area,
					 GdkPixmap * gdkpixmap);

static gint expose_pixmap (GtkWidget * drawing_area,
			   GdkEventExpose * event,
			   gpointer data);
static void draw_pixmap (GtkWidget * widget,
			 GdkRectangle * area,
			 gpointer data);
static void draw_pixmap_focus (GtkWidget * widget,
			       gpointer data);
static void real_draw_pixmap (GtkWidget * drawing_area);

static void show_events_dialog (GtkWidget * widget,
				gpointer value);
static void on_events_dialog_ok (GtkWidget * widget,
				 GtkWidget * clist);

static void show_keys_dialog (GtkWidget * widget,
			      gpointer value);
static void on_keys_clist_select (GtkWidget * widget,
				  gint row,
				  gint column,
				  GdkEventButton * bevent,
				  gpointer data);
static void on_keys_dialog_ok (GtkWidget * widget,
			       gpointer data);

static void show_signals_dialog (GtkWidget * widget,
				 gpointer value);
static void on_signals_clist_select (GtkWidget * widget,
				     gint row,
				     gint column,
				     GdkEventButton * bevent,
				     gpointer data);
static void on_signals_dialog_ok (GtkWidget * widget,
				  GtkWidget * clist);

static void on_accelerator_add (GtkWidget * widget,
				GtkWidget * clist);
static void on_accelerator_update (GtkWidget * widget,
				   GtkWidget * clist);
static void on_accelerator_delete (GtkWidget * widget,
				   GtkWidget * clist);
static void on_accelerator_clear (GtkWidget * widget,
				  GtkWidget * clist);
static void on_accelerator_select (GtkWidget * clist,
				   gint row,
				   gint column,
				   GdkEventButton * event,
				   gpointer user_data);

static void on_signal_add (GtkWidget * widget,
			   GtkWidget * clist);
static void on_signal_update (GtkWidget * widget,
			      GtkWidget * clist);
static void on_signal_delete (GtkWidget * widget,
			      GtkWidget * clist);
static void on_signal_clear (GtkWidget * widget,
			     GtkWidget * clist);
static void on_signal_select (GtkWidget * clist,
			      gint row,
			      gint column,
			      GdkEventButton * event,
			      gpointer user_data);

static void on_style_copy (GtkWidget * widget,
			   gpointer data);
static void on_style_copy_all (GtkWidget * widget,
			       gpointer data);
static void on_style_paste (GtkWidget * widget,
			    gpointer data);

/* Currently unused - its all auto-applied
   static void on_apply                 (GtkWidget      *widget,
   gpointer      data);
 */
static void on_property_changed (GtkWidget * widget,
				 GtkWidget * property);

static void set_pixmap (GtkWidget * drawing_area,
			GdkPixmap * gdkpixmap,
			gchar * filename);

static void show_colorsel_dialog (GtkWidget * widget,
				  gpointer data);
static void on_colorsel_dialog_ok (GtkWidget * widget,
				   gpointer data);
static gint close_dialog_event (GtkWidget * widget,
				GdkEvent * event,
				GtkWidget * dialog);
static void close_dialog (GtkWidget * widget,
			  GtkWidget * dialog);

static void show_filesel_dialog (GtkWidget * widget,
				 gpointer data);
static void on_filesel_dialog_ok (GtkWidget * widget,
				  gpointer data);

static void show_font_dialog (GtkWidget * widget,
			      gpointer data);
#if 0
static gint get_font_size_from_spec (gchar * spec);
#endif
static gchar *get_font_name_from_spec (gchar * spec);
static void on_font_dialog_apply (GtkWidget * widget,
				  GtkFontSelectionDialog * fontsel);
static void on_font_dialog_ok (GtkWidget * widget,
			       GtkFontSelectionDialog * fontsel);

static void show_style_dialog (GtkWidget * widget,
			       gpointer value);
static gint add_style_to_clist (gchar * key,
				gpointer data,
				GtkWidget * clist);
static void on_style_clist_select (GtkWidget * widget,
				   gint row,
				   gint column,
				   GdkEventButton * bevent,
				   gpointer data);
static void on_style_dialog_new (GtkWidget * widget,
				 GtkWidget * clist);
static gint create_new_style (GtkWidget * widget,
			      gchar * name,
			      GbStyle * base_gbstyle);
static void on_style_dialog_ok (GtkWidget * widget,
				GtkWidget * clist);
static void on_style_dialog_copy (GtkWidget * widget,
				  GtkWidget * clist);
static void on_style_dialog_rename (GtkWidget * widget,
				    GtkWidget * clist);
static gint rename_style (GtkWidget * widget,
			  gchar * name,
			  GbStyle * gbstyle);
static void on_style_dialog_delete (GtkWidget * widget,
				    GtkWidget * clist);

gchar *GbEventMaskSymbols[GB_EVENT_MASKS_COUNT] =
{
  "GDK_EXPOSURE_MASK",
  "GDK_POINTER_MOTION_MASK",
  "GDK_POINTER_MOTION_HINT_MASK",
  "GDK_BUTTON_MOTION_MASK",
  "GDK_BUTTON1_MOTION_MASK",
  "GDK_BUTTON2_MOTION_MASK",
  "GDK_BUTTON3_MOTION_MASK",
  "GDK_BUTTON_PRESS_MASK",
  "GDK_BUTTON_RELEASE_MASK",
  "GDK_KEY_PRESS_MASK",
  "GDK_KEY_RELEASE_MASK",
  "GDK_ENTER_NOTIFY_MASK",
  "GDK_LEAVE_NOTIFY_MASK",
  "GDK_FOCUS_CHANGE_MASK",
  "GDK_STRUCTURE_MASK",
  "GDK_PROPERTY_CHANGE_MASK",
  "GDK_VISIBILITY_NOTIFY_MASK",
  "GDK_PROXIMITY_IN_MASK",
  "GDK_PROXIMITY_OUT_MASK",
};


gint GbEventMaskValues[GB_EVENT_MASKS_COUNT] =
{
  GDK_EXPOSURE_MASK,
  GDK_POINTER_MOTION_MASK,
  GDK_POINTER_MOTION_HINT_MASK,
  GDK_BUTTON_MOTION_MASK,
  GDK_BUTTON1_MOTION_MASK,
  GDK_BUTTON2_MOTION_MASK,
  GDK_BUTTON3_MOTION_MASK,
  GDK_BUTTON_PRESS_MASK,
  GDK_BUTTON_RELEASE_MASK,
  GDK_KEY_PRESS_MASK,
  GDK_KEY_RELEASE_MASK,
  GDK_ENTER_NOTIFY_MASK,
  GDK_LEAVE_NOTIFY_MASK,
  GDK_FOCUS_CHANGE_MASK,
  GDK_STRUCTURE_MASK,
  GDK_PROPERTY_CHANGE_MASK,
  GDK_VISIBILITY_NOTIFY_MASK,
  GDK_PROXIMITY_IN_MASK,
  GDK_PROXIMITY_OUT_MASK
};


static gchar *GbEventMaskDescriptions[GB_EVENT_MASKS_COUNT] =
{
  N_("When the window needs redrawing"),
  N_("When the mouse moves"),
  N_("Mouse movement hints"),
  N_("Mouse movement with any button pressed"),
  N_("Mouse movement with button 1 pressed"),
  N_("Mouse movement with button 2 pressed"),
  N_("Mouse movement with button 3 pressed"),
  N_("Any mouse button pressed"),
  N_("Any mouse button released"),
  N_("Any key pressed"),
  N_("Any key released"),
  N_("When the mouse enters the window"),
  N_("When the mouse leaves the window"),
  N_("Any change in input focus"),
  N_("Any change in window structure"),
  N_("Any change in X Windows property"),
  N_("Any change in visibility"),
  N_("For cursors in XInput-aware programs"),
  N_("For cursors in XInput-aware programs")
};

/*
 * Create the properties window
 */

static void
property_create ()
{
  GtkWidget *vbox1, *label, *child;
  GdkColormap *colormap;

  /* Create hash table for widgets containing property values */
  gb_property_labels = g_hash_table_new (g_str_hash, g_str_equal);
  gb_property_values = g_hash_table_new (g_str_hash, g_str_equal);
  gb_property_buttons = g_hash_table_new (g_str_hash, g_str_equal);

  /* Create property window */
  win_property = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_uposition (win_property, 220, 0);
  gtk_widget_realize (win_property);

  gtk_signal_connect (GTK_OBJECT (win_property), "delete_event",
		      GTK_SIGNAL_FUNC (property_hide), NULL);

  gtk_window_set_title (GTK_WINDOW (win_property), _("Properties"));
  gtk_container_border_width (GTK_CONTAINER (win_property), 0);

  tooltips = gtk_tooltips_new ();
  colormap = gtk_widget_get_colormap (win_property);
  if (gdk_color_alloc (colormap, &tooltips_fgcolor)
      && gdk_color_alloc (colormap, &tooltips_bgcolor))
    gtk_tooltips_set_colors (tooltips, &tooltips_bgcolor, &tooltips_fgcolor);

  vbox1 = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (win_property), vbox1);
  gtk_widget_show (vbox1);

  /* Create main notebook for different categories of widget properties */
  main_notebook = gtk_notebook_new ();
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (main_notebook), GTK_POS_TOP);
  /*gtk_notebook_set_tab_border (GTK_NOTEBOOK (main_notebook), 2); */
  gtk_box_pack_start (GTK_BOX (vbox1), main_notebook, TRUE, TRUE, 0);
  gtk_container_border_width (GTK_CONTAINER (main_notebook), 2);
  gtk_widget_show (main_notebook);

  /* Create the pages of the main notebook */
  /* NOTE: If you add/remove pages you need to change the GB_PAGE_SIGNALS
     value at the top of this file */
  label = gtk_label_new (_("Widget"));
  gtk_widget_show (label);
  child = create_widget_property_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (main_notebook), child, label);

  label = gtk_label_new (_("Place"));
  gtk_widget_show (label);
  child = create_special_child_properties ();
  gtk_notebook_append_page (GTK_NOTEBOOK (main_notebook), child, label);

  label = gtk_label_new (_("Basic"));
  gtk_widget_show (label);
  child = create_standard_property_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (main_notebook), child, label);

  label = gtk_label_new (_("Style"));
  gtk_widget_show (label);
  child = create_style_property_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (main_notebook), child, label);

  label = gtk_label_new (_("Accel."));
  gtk_widget_show (label);
  child = create_accelerators_property_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (main_notebook), child, label);

  label = gtk_label_new (_("Sig."));
  gtk_widget_show (label);
  child = create_signals_property_page ();
  gtk_notebook_append_page (GTK_NOTEBOOK (main_notebook), child, label);

  /* Apply/Close buttons at bottom */
  /*
     hbox = gtk_hbox_new(TRUE, 5);
     gtk_box_pack_end(GTK_BOX(vbox1), hbox, FALSE, TRUE, 5);
     gtk_widget_show(hbox);

     apply_button = gtk_button_new_with_label("Apply");
     gtk_box_pack_start(GTK_BOX(hbox), apply_button, TRUE, TRUE, 0);
     gtk_widget_show(apply_button);
     gtk_signal_connect (GTK_OBJECT (apply_button), "clicked",
     GTK_SIGNAL_FUNC(on_apply), NULL);

     button = gtk_button_new_with_label("Close");
     gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
     gtk_widget_show(button);
     gtk_signal_connect (GTK_OBJECT (button), "clicked",
     GTK_SIGNAL_FUNC(property_hide), NULL);
   */

  /* Create style to use when a property value is invalid - it's not used
     at present. */
  invalid_style = gtk_style_new ();
  invalid_style->fg[GTK_STATE_NORMAL].red = 65535;
  invalid_style->fg[GTK_STATE_NORMAL].green = 0;
  invalid_style->fg[GTK_STATE_NORMAL].blue = 0;
}


void
property_show (GtkWidget * widget,
	       gpointer data)
{
  if (!win_property)
    property_create ();
  gtk_widget_show (win_property);
  /* This maps the window, which also de-iconifies it according to ICCCM. */
  gdk_window_show (GTK_WIDGET (win_property)->window);
  gdk_window_raise (GTK_WIDGET (win_property)->window);
  property_set_widget (NULL);
}

gint
property_hide (GtkWidget * widget,
	       gpointer data)
{
  gtk_widget_hide (win_property);
  return TRUE;
}


GtkWidget *
property_get_widget ()
{
  return property_widget;
}


void
property_set_widget (GtkWidget * widget)
{
  gchar buffer[128];
  gchar *name;

  if (widget)
    {
      strcpy (buffer, _("Properties: "));
      name = gtk_widget_get_name (widget);
      strncat (buffer, name, 120 - strlen (buffer));
      gtk_window_set_title (GTK_WINDOW (win_property), buffer);
      gtk_widget_set_sensitive (main_notebook, TRUE);
      if (apply_button)
	gtk_widget_set_sensitive (apply_button, TRUE);
      property_widget = widget;
    }
  else
    {
      gtk_window_set_title (GTK_WINDOW (win_property), _("Properties: <none>"));
      property_hide_gbwidget_page ();
      gtk_widget_set_sensitive (main_notebook, FALSE);
      if (apply_button)
	gtk_widget_set_sensitive (apply_button, FALSE);
      /* Need to clear the property widget before clearing the name property,
         since auto-apply may be on, and hence we may clear the name of the
         currently shown widget. */
      property_widget = NULL;
      property_set_string (GbName, NULL);
    }
}


static GtkWidget *
create_widget_property_page ()
{
  GtkWidget *page, *table, *vbox;
  page = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (page),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_viewport_set_shadow_type (GTK_VIEWPORT (GTK_SCROLLED_WINDOW (page)
					      ->viewport), GTK_SHADOW_NONE);
  gtk_widget_show (page);

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox);
  gtk_container_add (GTK_CONTAINER (page), vbox);

  table = gtk_table_new (9, 3, FALSE);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, TRUE, 2);
  gtk_widget_show (table);

  property_set_table_position (table, 0);
  property_add_string (GbName, _("Name:"),
		       _("The name of the widget"));

  property_widget_notebook = gtk_notebook_new ();
  gtk_widget_show (property_widget_notebook);
  gtk_notebook_set_show_tabs (GTK_NOTEBOOK (property_widget_notebook),
			      FALSE);
  gtk_notebook_set_show_border (GTK_NOTEBOOK (property_widget_notebook),
				FALSE);
  gtk_box_pack_start (GTK_BOX (vbox), property_widget_notebook, FALSE, TRUE, 0);
  /*gtk_container_border_width (GTK_CONTAINER (property_widget_notebook), 2); */

  return page;
}


static GtkWidget *
create_standard_property_page ()
{
  GtkWidget *page;

  page = gtk_table_new (9, 3, FALSE);
  gtk_widget_show (page);

  property_set_table_position (page, 0);
  property_add_int_range (GbX, "X:",
			  "The left edge of the widget, relative to its parent (-1 means the default)",
			  -1, 10000, 1, 10, 1);
  property_add_int_range (GbY, "Y:",
			  _("The top edge of the widget, relative to its parent (-1 means the default)"),
			  -1, 10000, 1, 10, 1);
  property_add_int_range (GbWidth, _("Width:"),
	      _("The width of the widget (set to 0 to use the default width)"),
			  0, 10000, 1, 10, 1);
  property_add_int_range (GbHeight, _("Height:"),
	    _("The height of the widget (set to 0 to use the default height)"),
			  0, 10000, 1, 10, 1);

  property_add_bool (GbVisible, _("Visible:"),
		     _("If the widget is initially visible"));
  property_add_bool (GbSensitive, _("Sensitive:"),
		     _("If the widget responds to input"));
  property_add_string (GbTooltip, _("Tooltip:"),
	     _("The tooltip to display if the mouse lingers over the widget"));

  property_add_bool (GbCanDefault, _("Can Default:"),
		     _("If the widget can be the default"));
  property_add_bool (GbHasDefault, _("Has Default:"),
		     _("If the widget is the default"));
  property_add_bool (GbCanFocus, _("Can Focus:"),
		     _("If the widget can accept the input focus"));
  property_add_bool (GbHasFocus, _("Has Focus:"),
		     _("If the widget has the input focus"));

  property_add_dialog (GbEvents, _("Events:"),
		       _("The X events that the widget receives"),
		       FALSE, show_events_dialog);
  property_add_choice (GbExtEvents, _("Ext.Events:"),
		       _("The X Extension events mode"),
		       GbExtensionModeChoices);
  return page;
}


static GtkWidget *
create_style_property_page ()
{
  GtkWidget *page, *table, *label, *child, *hbox, *button;
  int i;

  page = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (page);

  table = gtk_table_new (3, 3, FALSE);
  gtk_box_pack_start (GTK_BOX (page), table, FALSE, TRUE, 0);
  gtk_widget_show (table);

  property_set_table_position (table, 0);
  property_add_bool (GbStylePropagate, _("Propagate:"),
		_("Set True to propagate the style to the widget's children"));
  property_add_dialog (GbStyleName, _("Named Style:"),
	    _("The name of the style, which can be shared by several widgets"),
		       FALSE, show_style_dialog);
  property_add_font (GbStyleFont, _("Font:"),
		     _("The font to use for any text in the widget"));

  /* Create notebook for 5 states */
  styles_notebook = gtk_notebook_new ();
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (styles_notebook), GTK_POS_TOP);
  /*gtk_notebook_set_tab_border (GTK_NOTEBOOK (styles_notebook), 2); */
  gtk_box_pack_start (GTK_BOX (page), styles_notebook, FALSE, TRUE, 0);
  gtk_container_border_width (GTK_CONTAINER (styles_notebook), 0);

  for (i = 0; i < GB_NUM_STYLE_STATES; i++)
    {
      label = gtk_label_new (gettext (GbStateTabLabels[i]));
      gtk_widget_show (label);
      child = create_style_page (GbStateNames[i]);
      gtk_notebook_append_page (GTK_NOTEBOOK (styles_notebook), child, label);
    }
  gtk_widget_show (styles_notebook);

  /* Add/Update/Delete buttons at bottom */
  hbox = gtk_hbox_new (TRUE, 5);
  button = gtk_button_new_with_label (_("Copy"));
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_style_copy), NULL);

  button = gtk_button_new_with_label (_("Copy All"));
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_style_copy_all), NULL);

  button = gtk_button_new_with_label (_("Paste"));
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_style_paste), NULL);

  gtk_box_pack_start (GTK_BOX (page), hbox, FALSE, TRUE, 3);
  gtk_widget_show (hbox);

  return page;
}


static GtkWidget *
create_style_page (gchar * state)
{
  GtkWidget *page, *table;
  int i;
  gchar buffer[128];
  gchar *labels[] =
  {N_("Foreground:"), N_("Background:"), N_("Text:"), N_("Base:"),
  /* FIXME: Delete? "Light:", "Middle:", "Dark:" */ };
  gchar *tooltips[] =
  {N_("Foreground color"), N_("Background color"), N_("Text color"),
   N_("Base color"),
  /* FIXME: Delete?
     "Light color", "Middle color", "Dark color" */ };

  page = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (page);

  /* Add the seven colours for this state */
  table = gtk_table_new (8, 3, FALSE);
  property_set_table_position (table, 0);
  for (i = 0; i < GB_NUM_STYLE_COLORS; i++)
    {
      sprintf (buffer, "GtkStyle::%s[%s]", GbColorNames[i], state);
      property_add_color (buffer, gettext (labels[i]), gettext (tooltips[i]));
    }

  /* Add the background pixmap */
  sprintf (buffer, "GtkStyle::%s[%s]", GbBgPixmapName, state);
  property_add_bgpixmap (buffer, _("Back. Pixmap:"),
		      _("The graphic to use as the background of the widget"));

  gtk_box_pack_start (GTK_BOX (page), table, TRUE, TRUE, 0);
  gtk_widget_show (table);
  return page;
}


static GtkWidget *
create_special_child_properties ()
{
  GtkWidget *page, *table;

  page = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (page);

  special_child_property_notebook = gtk_notebook_new ();
  gtk_widget_show (special_child_property_notebook);
  gtk_notebook_set_show_tabs (GTK_NOTEBOOK (special_child_property_notebook),
			      FALSE);
  gtk_notebook_set_show_border (GTK_NOTEBOOK (special_child_property_notebook),
				FALSE);
  gtk_box_pack_start (GTK_BOX (page), special_child_property_notebook,
		      TRUE, TRUE, 0);

  /* Create table for special properties of widgets in a table */
  table = gtk_table_new (3, 3, FALSE);
  gtk_widget_show (table);
  gtk_notebook_append_page (GTK_NOTEBOOK (special_child_property_notebook),
			    table, NULL);

  property_set_table_position (table, 0);
  property_add_int_range (GbCellX, _("Cell X:"),
			  _("The left edge of the widget in the table"),
			  0, 10000, 1, 10, 1);
  property_add_int_range (GbCellY, _("Cell Y:"),
			  _("The top edge of the widget in the table"),
			  0, 10000, 1, 10, 1);

  property_add_int_range (GbColSpan, _("Col Span:"),
		 _("The number of columns spanned by the widget in the table"),
			  1, 10000, 1, 10, 1);
  property_add_int_range (GbRowSpan, _("Row Span:"),
		    _("The number of rows spanned by the widget in the table"),
			  1, 10000, 1, 10, 1);
  property_add_int_range (GbXPad, _("H Padding:"),
			  _("The horizontal padding"),
			  0, 10000, 1, 10, 1);
  property_add_int_range (GbYPad, _("V Padding:"),
			  _("The vertical padding"),
			  0, 10000, 1, 10, 1);
  property_add_bool (GbXExpand, _("X Expand:"),
		     _("Set True to let the widget expand horizontally"));
  property_add_bool (GbYExpand, _("Y Expand:"),
		     _("Set True to let the widget expand vertically"));
  property_add_bool (GbXShrink, _("X Shrink:"),
		     _("Set True to let the widget shrink horizontally"));
  property_add_bool (GbYShrink, _("Y Shrink:"),
		     _("Set True to let the widget shrink vertically"));
  property_add_bool (GbXFill, _("X Fill:"),
	   _("Set True to let the widget fill its horizontal allocated area"));
  property_add_bool (GbYFill, _("Y Fill:"),
	     _("Set True to let the widget fill its vertical allocated area"));

  /* Create table for special properties of widgets in a box */
  table = gtk_table_new (3, 3, FALSE);
  gtk_widget_show (table);
  gtk_notebook_append_page (GTK_NOTEBOOK (special_child_property_notebook),
			    table, NULL);

  property_set_table_position (table, 0);
  property_add_int_range (GbPadding, _("Padding:"),
			  _("The widget's padding"),
			  0, 10000, 1, 10, 1);
  property_add_bool (GbExpand, _("Expand:"),
		     _("Set True to let the widget expand"));
  property_add_bool (GbFill, _("Fill:"),
		     _("Set True to let the widget fill its allocated area"));
  property_add_bool (GbPack, _("Pack Start:"),
		     _("Set True to pack the widget at the start of the box"));

  /* Create table for special properties of widgets in a packer */
#ifdef GLD_HAVE_GTK_1_1
  table = gtk_table_new (3, 3, FALSE);
  gtk_widget_show (table);
  gtk_notebook_append_page (GTK_NOTEBOOK (special_child_property_notebook),
			    table, NULL);

  property_set_table_position (table, 0);
  property_add_choice (GbPackerSide, _("Side:"),
		       _("The side of the remaining space to place the widget"),
		       GbPackerSideChoices);
  property_add_choice (GbPackerAnchor, _("Anchor:"),
		       _("Where to anchor the widget in its allocated area"),
		       GbPackerAnchorChoices);
  property_add_bool (GbPackerExpand, _("Expand:"),
		     _("Set True to let the widget expand"));
  property_add_bool (GbPackerFillX, _("Fill X:"),
		     _("Set True to let the widget fill its allocated width"));
  property_add_bool (GbPackerFillY, _("Fill Y:"),
		     _("Set True to let the widget fill its allocated height"));
  property_add_int_range (GbPackerBorder, _("Border Width:"),
			  _("The size of the widget's border"),
			  0, 10000, 1, 10, 1);
  property_add_int_range (GbPackerPadX, _("H Padding:"),
			  _("The widget's horizontal external padding"),
			  0, 10000, 1, 10, 1);
  property_add_int_range (GbPackerPadY, _("V Padding:"),
			  _("The widget's vertical external padding"),
			  0, 10000, 1, 10, 1);
  property_add_int_range (GbPackerIPadX, _("H I Padding:"),
			  _("The widget's horizontal internal padding"),
			  0, 10000, 1, 10, 1);
  property_add_int_range (GbPackerIPadY, _("V I Padding:"),
			  _("The widget's vertical internal padding"),
			  0, 10000, 1, 10, 1);
#endif

  return page;
}



gint
property_add_gbwidget_page (GtkWidget * page)
{
  gtk_notebook_append_page (GTK_NOTEBOOK (property_widget_notebook), page, NULL);
  return g_list_length (GTK_NOTEBOOK (property_widget_notebook)->children) - 1;
}


void
property_hide_gbwidget_page ()
{
  gtk_widget_hide (property_widget_notebook);
}


void
property_show_gbwidget_page (gint page)
{
  gtk_notebook_set_page (GTK_NOTEBOOK (property_widget_notebook), page);
  gtk_widget_show (property_widget_notebook);
}


void
property_hide_special_child_page ()
{
  gtk_widget_hide (special_child_property_notebook);
}


void
property_show_special_child_page (gint page)
{
  gtk_notebook_set_page (GTK_NOTEBOOK (special_child_property_notebook), page);
  gtk_widget_show (special_child_property_notebook);
}




static void
on_style_copy (GtkWidget * widget, gpointer data)
{
  guchar *color;
  if (!selected_style_widget)
    {
      show_message_box (_("You need to select a color or background to copy"));
      return;
    }
  if (GTK_IS_PREVIEW (selected_style_widget))
    {
      color = gtk_object_get_data (GTK_OBJECT (selected_style_widget), GbColorKey);
      g_return_if_fail (color != NULL);
      clipbd_colors[0][0] = color[0];
      clipbd_colors[0][1] = color[1];
      clipbd_colors[0][2] = color[2];
      clipbd_state = GB_COLOR_CLIPBD_COLOR;
    }
  else if (GTK_IS_DRAWING_AREA (selected_style_widget))
    {
      clipbd_bgpixmap = gtk_object_get_data (GTK_OBJECT (selected_style_widget),
					     GbBgPixmapKey);
      clipbd_bgfilename = gtk_object_get_data (GTK_OBJECT (selected_style_widget),
					       GbBgFilenameKey);
      clipbd_state = GB_COLOR_CLIPBD_BGPIXMAP;
    }
  else
    {
      g_warning (_("Invalid selection in on_style_copy()"));
    }
}


static void
on_style_copy_all (GtkWidget * widget, gpointer data)
{
  gint page, i;
  guchar *color;
  gchar buffer[128];

  /* Find out which state is currently displayed */
  page = gtk_notebook_current_page (GTK_NOTEBOOK (styles_notebook));

  for (i = 0; i < GB_NUM_STYLE_COLORS; i++)
    {
      sprintf (buffer, "GtkStyle::%s[%s]", GbColorNames[i], GbStateNames[page]);
      color = property_get_color (buffer, NULL, NULL);
      clipbd_colors[i][0] = color[0];
      clipbd_colors[i][1] = color[1];
      clipbd_colors[i][2] = color[2];
    }
  sprintf (buffer, "GtkStyle::%s[%s]", GbBgPixmapName, GbStateNames[page]);
  clipbd_bgpixmap = property_get_bgpixmap (buffer, NULL, NULL,
					   &clipbd_bgfilename);

  clipbd_state = GB_COLOR_CLIPBD_STATE;
}


static void
on_style_paste (GtkWidget * widget, gpointer data)
{
  guchar *color;
  gchar buffer[128];
  gint page, i;
  GtkWidget *value, *preview, *drawing_area, *changed_value = NULL;
  gboolean bgpixmap_changed;
  gchar *filename;

  switch (clipbd_state)
    {
    case GB_COLOR_CLIPBD_EMPTY:
      show_message_box (_("You need to copy a color or background pixmap first"));
      break;
    case GB_COLOR_CLIPBD_COLOR:
      if (!selected_style_widget || !GTK_IS_PREVIEW (selected_style_widget))
	{
	  show_message_box (_("You need to select a color to paste into"));
	  return;
	}
      color = gtk_object_get_data (GTK_OBJECT (selected_style_widget), GbColorKey);
      g_return_if_fail (color != NULL);
      color[0] = clipbd_colors[0][0];
      color[1] = clipbd_colors[0][1];
      color[2] = clipbd_colors[0][2];
      show_color_in_preview (selected_style_widget, NULL);
      value = selected_style_widget->parent;
      on_property_changed (value, value);
      break;
    case GB_COLOR_CLIPBD_BGPIXMAP:
      if (!selected_style_widget || !GTK_IS_DRAWING_AREA (selected_style_widget))
	{
	  show_message_box (_("You need to select a background pixmap to paste into"));
	  return;
	}
      set_pixmap (selected_style_widget, clipbd_bgpixmap, clipbd_bgfilename);
      value = selected_style_widget->parent;
      on_property_changed (value, value);
      break;
    case GB_COLOR_CLIPBD_STATE:
      page = gtk_notebook_current_page (GTK_NOTEBOOK (styles_notebook));

      /* We need to find one color or background pixmap which has changed so
         we can call on_property_changed with it, thus recreating the style.
         We don't want to call on_property_changed multiple times. */
      for (i = 0; i < GB_NUM_STYLE_COLORS; i++)
	{
	  sprintf (buffer, "GtkStyle::%s[%s]", GbColorNames[i], GbStateNames[page]);
	  value = (GtkWidget *) g_hash_table_lookup (gb_property_values, buffer);
	  g_return_if_fail (value != NULL);
	  preview = GTK_BIN (value)->child;
	  g_return_if_fail (GTK_IS_PREVIEW (preview));
	  color = gtk_object_get_data (GTK_OBJECT (preview), GbColorKey);
	  g_return_if_fail (color != NULL);
	  if (color[0] != clipbd_colors[i][0]
	      || color[1] != clipbd_colors[i][1]
	      || color[2] != clipbd_colors[i][2])
	    {
	      color[0] = clipbd_colors[i][0];
	      color[1] = clipbd_colors[i][1];
	      color[2] = clipbd_colors[i][2];
	      show_color_in_preview (preview, NULL);
	      changed_value = value;
	    }
	}

      sprintf (buffer, "GtkStyle::%s[%s]", GbBgPixmapName, GbStateNames[page]);
      value = (GtkWidget *) g_hash_table_lookup (gb_property_values, buffer);
      g_return_if_fail (value != NULL);
      drawing_area = GTK_BIN (value)->child;
      g_return_if_fail (GTK_IS_DRAWING_AREA (drawing_area));
      filename = gtk_object_get_data (GTK_OBJECT (drawing_area), GbBgFilenameKey);
      bgpixmap_changed = FALSE;
      if (filename)
	{
	  if (!clipbd_bgfilename || strcmp (filename, clipbd_bgfilename))
	    bgpixmap_changed = TRUE;
	}
      else
	{
	  if (clipbd_bgfilename)
	    bgpixmap_changed = TRUE;
	}
      if (bgpixmap_changed)
	{
	  property_set_bgpixmap (buffer, clipbd_bgpixmap, clipbd_bgfilename);
	  changed_value = value;
	}

      if (changed_value)
	on_property_changed (changed_value, changed_value);

      break;
    }
}


/*
 * Color selection dialog
 */

static void
show_colorsel_dialog (GtkWidget * widget, gpointer data)
{
  guchar *color;
  gdouble rgb[4];
  int i;

  /* Create the dialog if it doesn't exist yet */
  if (!colorsel)
    {
      colorsel = GTK_COLOR_SELECTION_DIALOG (gtk_color_selection_dialog_new (_("Color Selection Dialog")));
      /* Hide the Help button since it is not used */
      gtk_widget_hide (GTK_WIDGET (colorsel->help_button));
      gtk_signal_connect (GTK_OBJECT (colorsel), "delete_event",
			  GTK_SIGNAL_FUNC (close_dialog_event), colorsel);
      gtk_signal_connect (GTK_OBJECT (colorsel->cancel_button), "clicked",
			  GTK_SIGNAL_FUNC (close_dialog), colorsel);
      gtk_signal_connect (GTK_OBJECT (colorsel->ok_button), "clicked",
			  GTK_SIGNAL_FUNC (on_colorsel_dialog_ok), NULL);
    }

  color_value = GTK_WIDGET (data);
  g_return_if_fail (GTK_IS_FRAME (color_value));

  color = gtk_object_get_data (GTK_OBJECT (GTK_BIN (color_value)->child),
			       GbColorKey);
  g_return_if_fail (color != NULL);
  for (i = 0; i < 3; i++)
    rgb[i] = ((gdouble) color[i]) / 255;

  gtk_color_selection_set_color (GTK_COLOR_SELECTION (colorsel->colorsel), rgb);
  gtk_widget_show (GTK_WIDGET (colorsel));
  /* This maps the window, which also de-iconifies it according to ICCCM. */
  gdk_window_show (GTK_WIDGET (colorsel)->window);
  gdk_window_raise (GTK_WIDGET (colorsel)->window);
}


static void
on_colorsel_dialog_ok (GtkWidget * widget, gpointer data)
{
  guchar *color;
  gdouble rgb[4];
  GtkWidget *preview = GTK_BIN (color_value)->child;
  int i;

  gtk_color_selection_get_color (GTK_COLOR_SELECTION (colorsel->colorsel), rgb);
  color = gtk_object_get_data (GTK_OBJECT (preview), GbColorKey);
  g_return_if_fail (color != NULL);
  for (i = 0; i < 3; i++)
    color[i] = (guchar) 255.0 *rgb[i];
  show_color_in_preview (preview, NULL);
  close_dialog (widget, GTK_WIDGET (colorsel));
  on_property_changed (color_value, color_value);
}


static gint
close_dialog_event (GtkWidget * widget, GdkEvent * event, GtkWidget * dialog)
{
  close_dialog (widget, dialog);
  return TRUE;
}


static void
close_dialog (GtkWidget * widget, GtkWidget * dialog)
{
  gint x, y;

  /* remember position of window for when it is used again */
  gdk_window_get_origin (dialog->window, &x, &y);
  if (windows_x_offset >= 0)
    {
      x -= windows_x_offset;
      y -= windows_y_offset;
    }
  gtk_widget_hide (dialog);
  gtk_widget_set_uposition (dialog, x, y);
}


/*
 * File selection dialog
 */

static void
show_filesel_dialog (GtkWidget * widget, gpointer data)
{
  gchar *filename;
  /* value can be an Entry (for filename properties) or a Frame (for
     background pixmaps) */
  filename_value = GTK_WIDGET (data);

  /* Create the dialog if it doesn't exist yet */
  if (!filesel)
    {
      filesel = GTK_FILE_SELECTION (gtk_file_selection_new (_("Select File")));
      gtk_signal_connect (GTK_OBJECT (filesel), "delete_event",
			  GTK_SIGNAL_FUNC (close_dialog_event), filesel);
      gtk_signal_connect (GTK_OBJECT (filesel->cancel_button), "clicked",
			  GTK_SIGNAL_FUNC (close_dialog), filesel);
      gtk_signal_connect (GTK_OBJECT (filesel->ok_button), "clicked",
			  GTK_SIGNAL_FUNC (on_filesel_dialog_ok), NULL);
    }

  /* set to current file, if there is one */
  if (GTK_IS_ENTRY (filename_value))
    {
      filename = gtk_entry_get_text (GTK_ENTRY (filename_value));
    }
  else
    {
      filename = gtk_object_get_data (GTK_OBJECT (GTK_BIN (filename_value)->child),
				      GbBgFilenameKey);
    }
  if (filename != NULL)
    {
      gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel), filename);
    }
  gtk_widget_show (GTK_WIDGET (filesel));
  /* This maps the window, which also de-iconifies it according to ICCCM. */
  gdk_window_show (GTK_WIDGET (filesel)->window);
  gdk_window_raise (GTK_WIDGET (filesel)->window);
}


static void
on_filesel_dialog_ok (GtkWidget * widget, gpointer data)
{
  GdkPixmap *gdkpixmap = NULL, *old_gdkpixmap;
  gchar *filename, *old_filename;
  GtkWidget *drawing_area;

  filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (filesel));

  /* If the filename ends in '/' it means the user wants to reset the
     pixmap to NULL. */
  if (filename[strlen (filename) - 1] == '/')
    {
      filename = NULL;
    }

  /* For pixmaps we just show the filename */
  if (GTK_IS_ENTRY (filename_value))
    {
      gtk_entry_set_text (GTK_ENTRY (filename_value), filename ? filename : "");
      close_dialog (widget, GTK_WIDGET (filesel));
      on_property_changed (filename_value, filename_value);
      return;
    }

  /* For background pixmaps we show them in the property value widget */
  if (filename)
    {
      gdkpixmap = gdk_pixmap_create_from_xpm (widget->window, NULL,
				       &widget->style->bg[GTK_STATE_NORMAL],
					      filename);
      if (!gdkpixmap)
	{
	  show_message_box (_("Couldn't create pixmap from file\n"));
	  return;
	}
    }

  drawing_area = GTK_BIN (filename_value)->child;
  old_gdkpixmap = gtk_object_get_data (GTK_OBJECT (drawing_area), GbBgPixmapKey);
  if (old_gdkpixmap)
    gdk_pixmap_unref (old_gdkpixmap);
  old_filename = gtk_object_get_data (GTK_OBJECT (drawing_area), GbBgFilenameKey);
  g_free (old_filename);

  gtk_object_set_data (GTK_OBJECT (drawing_area), GbBgPixmapKey, gdkpixmap);
  gtk_object_set_data (GTK_OBJECT (drawing_area), GbBgFilenameKey,
		       g_strdup (filename));

  close_dialog (widget, GTK_WIDGET (filesel));
  show_pixmap_in_drawing_area (drawing_area, gdkpixmap);
  gtk_widget_queue_draw (drawing_area);
  on_property_changed (filename_value, filename_value);
}


/*

 */


static GtkWidget *
create_signals_property_page ()
{
  GtkWidget *page, *table, *hbox, *button;
  gchar *signal_titles[5];

  page = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (page);

  /* List of current signal handlers - Signal/Handler/Data/Options */
  signal_titles[0] = _("Signal");
  signal_titles[1] = _("Handler");
  signal_titles[2] = _("Data");
  signal_titles[3] = _("After");
  signal_titles[4] = _("Object");
  signal_clist = gtk_clist_new_with_titles (5, signal_titles);
  gtk_clist_set_column_width (GTK_CLIST (signal_clist), 0, 150);
  gtk_clist_set_column_width (GTK_CLIST (signal_clist), 1, 150);
  gtk_clist_set_column_width (GTK_CLIST (signal_clist), 2, 80);
  gtk_clist_set_column_width (GTK_CLIST (signal_clist), 3, 50);
  gtk_clist_set_column_width (GTK_CLIST (signal_clist), 4, 80);
  gtk_widget_show (signal_clist);
  gtk_box_pack_start (GTK_BOX (page), signal_clist, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (signal_clist), "select_row",
		      GTK_SIGNAL_FUNC (on_signal_select), NULL);

  /* Mod, Key & Signal fields */
  table = gtk_table_new (3, 3, FALSE);
  gtk_widget_show (table);
  property_set_table_position (table, 0);
  property_add_dialog (GbSignalName, _("Signal:"),
		       _("The signal to add a handler for"), TRUE,
		       show_signals_dialog);
  property_add_string (GbSignalHandler, _("Handler:"),
		       _("The function to handle the signal"));
  property_add_string (GbSignalData, _("Data:"),
		       _("The data passed to the handler"));
  property_add_string (GbSignalObject, _("Object:"),
		       _("The object which receives the signal"));
  property_add_bool (GbSignalAfter, _("After:"),
		     _("If the handler runs after the class function"));
  gtk_box_pack_start (GTK_BOX (page), table, FALSE, TRUE, 5);

  /* Add/Update/Delete buttons at bottom */
  hbox = gtk_hbox_new (TRUE, 2);
  button = gtk_button_new_with_label (_("Add"));
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_signal_add), signal_clist);

  button = gtk_button_new_with_label (_("Update"));
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_signal_update), signal_clist);

  button = gtk_button_new_with_label (_("Delete"));
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_signal_delete), signal_clist);

  button = gtk_button_new_with_label (_("Clear"));
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_signal_clear), signal_clist);

  gtk_box_pack_start (GTK_BOX (page), hbox, FALSE, TRUE, 5);
  gtk_widget_show (hbox);

  return page;
}


static GtkWidget *
create_accelerators_property_page ()
{
  gchar *accel_titles[3];
  GtkWidget *page, *hbox, *label, *button, *table;
  int row;

  page = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (page);

  /* List of current accelerators - Mods/Keys/Signals */
  accel_titles[0] = _("Mod");
  accel_titles[1] = _("Key");
  accel_titles[2] = _("Signal to emit");
  accel_clist = gtk_clist_new_with_titles (3, accel_titles);
  gtk_box_pack_start (GTK_BOX (page), accel_clist, TRUE, TRUE, 0);
  gtk_clist_set_column_width (GTK_CLIST (accel_clist), 0, 30);
  gtk_clist_set_column_width (GTK_CLIST (accel_clist), 1, 120);
  gtk_clist_set_column_width (GTK_CLIST (accel_clist), 2, 200);
  gtk_widget_show (accel_clist);
  gtk_signal_connect (GTK_OBJECT (accel_clist), "select_row",
		      GTK_SIGNAL_FUNC (on_accelerator_select), NULL);

  /* Mod, Key & Signal fields */
  table = gtk_table_new (3, 3, FALSE);
  gtk_widget_show (table);
  row = 0;
  label = gtk_label_new (_("Modifiers:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, row, row + 1,
		    GTK_FILL, 0, 1, 1);

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_widget_show (hbox);
  accel_control_button = gtk_check_button_new_with_label (_("Ctrl"));
  gtk_widget_show (accel_control_button);
  gtk_box_pack_start (GTK_BOX (hbox), accel_control_button, TRUE, TRUE, 0);
  accel_shift_button = gtk_check_button_new_with_label (_("Shift"));
  gtk_widget_show (accel_shift_button);
  gtk_box_pack_start (GTK_BOX (hbox), accel_shift_button, TRUE, TRUE, 0);
  accel_alt_button = gtk_check_button_new_with_label (_("Alt"));
  gtk_widget_show (accel_alt_button);
  gtk_box_pack_start (GTK_BOX (hbox), accel_alt_button, TRUE, TRUE, 0);
  gtk_table_attach (GTK_TABLE (table), hbox, 1, 2, row, row + 1,
		    GTK_EXPAND | GTK_FILL, 0, 1, 1);

  property_set_table_position (table, ++row);
  property_add_dialog (GbAccelKey, _("Key:"),
		       _("The accelerator key"), TRUE, show_keys_dialog);
  property_add_dialog (GbAccelSignal, _("Signal:"),
		       _("The signal to emit when the accelerator is pressed"),
		       TRUE, show_signals_dialog);
  gtk_box_pack_start (GTK_BOX (page), table, FALSE, TRUE, 5);

  /* Add/Update/Delete buttons at bottom */
  hbox = gtk_hbox_new (TRUE, 2);
  button = gtk_button_new_with_label (_("Add"));
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_accelerator_add), accel_clist);

  button = gtk_button_new_with_label (_("Update"));
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_accelerator_update), accel_clist);

  button = gtk_button_new_with_label (_("Delete"));
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_accelerator_delete), accel_clist);

  button = gtk_button_new_with_label (_("Clear"));
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_accelerator_clear), accel_clist);

  gtk_box_pack_start (GTK_BOX (page), hbox, FALSE, TRUE, 5);
  gtk_widget_show (hbox);

  return page;
}


static void
on_color_draw (GtkWidget * widget, gpointer data)
{
  if (widget == selected_style_widget)
    {
      gdk_draw_rectangle (widget->window, widget->style->black_gc, FALSE, 0, 0,
			  widget->allocation.width - 1,
			  widget->allocation.height - 1);
      gdk_draw_rectangle (widget->window, widget->style->white_gc, FALSE, 1, 1,
			  widget->allocation.width - 3,
			  widget->allocation.height - 3);
      gdk_draw_rectangle (widget->window, widget->style->black_gc, FALSE, 2, 2,
			  widget->allocation.width - 5,
			  widget->allocation.height - 5);
    }
}


static void
on_color_expose_event (GtkWidget * widget, GdkEvent * event, gpointer data)
{
  on_color_draw (widget, data);
}


static void
on_color_select (GtkWidget * widget, gpointer data)
{
  if (selected_style_widget == widget)
    return;
  if (selected_style_widget && selected_style_widget != widget)
    gtk_widget_queue_draw (selected_style_widget);
  selected_style_widget = widget;
  gtk_widget_queue_draw (widget);
}


/*
 * Creating property widgets
 */

void
property_set_table_position (GtkWidget * table, gint row)
{
  property_table = table;
  property_table_row = row;
}


GtkWidget*
property_get_table_position (gint *row)
{
  *row = property_table_row;
  return property_table;
}


void
property_add_string (gchar * property_name,
		     gchar * label,
		     gchar * tooltip)
{
  GtkWidget *value = gtk_entry_new ();
  gtk_widget_set_usize (value, 80, -1);
  property_add (property_name, label, value, NULL, tooltip);
  gtk_signal_connect (GTK_OBJECT (value), "changed",
		      GTK_SIGNAL_FUNC (on_property_changed), value);
}


void
property_add_text (gchar * property_name,
		   gchar * label,
		   gchar * tooltip,
		   gint visible_lines)
{
  gint line_height;
  GtkWidget *value = gtk_text_new (NULL, NULL);
  gtk_text_set_editable (GTK_TEXT (value), TRUE);
  line_height = value->style->font->ascent + value->style->font->descent;
  /* We add 6 for the text's border height etc. */
  gtk_widget_set_usize (value, 80, visible_lines * line_height + 6);
  property_add (property_name, label, value, NULL, tooltip);
  gtk_signal_connect (GTK_OBJECT (value), "changed",
		      GTK_SIGNAL_FUNC (on_property_changed), value);
}


void
property_add_int (gchar * property_name,
		  gchar * label,
		  gchar * tooltip)
{
  GtkWidget *value = gtk_entry_new ();
  gtk_widget_set_usize (value, 80, -1);
  property_add (property_name, label, value, NULL, tooltip);
  gtk_signal_connect (GTK_OBJECT (value), "changed",
		      GTK_SIGNAL_FUNC (on_property_changed), value);
}


void
property_add_int_range (gchar * property_name,
			gchar * label,
			gchar * tooltip,
			gint min,
			gint max,
			gint step_increment,
			gint page_increment,
			gint climb_rate)
{
  GtkObject *adjustment = gtk_adjustment_new (min, min, max, step_increment,
					    page_increment, page_increment);
  GtkWidget *value = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment),
					  climb_rate, 0);
  gtk_widget_set_usize (value, 80, -1);
  gtk_signal_connect (GTK_OBJECT (value), "changed",
		      GTK_SIGNAL_FUNC (on_property_changed), value);
  property_add (property_name, label, value, NULL, tooltip);
}


void
property_add_float (gchar * property_name,
		    gchar * label,
		    gchar * tooltip)
{
  GtkWidget *value = gtk_entry_new ();
  gtk_widget_set_usize (value, 80, -1);
  property_add (property_name, label, value, NULL, tooltip);
  gtk_signal_connect (GTK_OBJECT (value), "changed",
		      GTK_SIGNAL_FUNC (on_property_changed), value);
}


void
property_add_float_range (gchar * property_name,
			  gchar * label,
			  gchar * tooltip,
			  gfloat min,
			  gfloat max,
			  gfloat step_increment,
			  gfloat page_increment,
			  gfloat climb_rate,
			  gint decimal_digits)
{
  GtkObject *adjustment = gtk_adjustment_new (min, min, max, step_increment,
					    page_increment, page_increment);
  GtkWidget *value = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment),
					  climb_rate, decimal_digits);
  gtk_widget_set_usize (value, 80, -1);
  gtk_signal_connect (GTK_OBJECT (value), "changed",
		      GTK_SIGNAL_FUNC (on_property_changed), value);
  property_add (property_name, label, value, NULL, tooltip);
}


static void
on_bool_property_toggle (GtkWidget * value, gpointer data)
{
  guint active = GTK_TOGGLE_BUTTON (value)->active;
  GtkWidget *label = GTK_BUTTON (value)->child;
  gtk_label_set (GTK_LABEL (label), active ? _("Yes") : _("No"));
  on_property_changed (value, value);
}


void
property_add_bool (gchar * property_name,
		   gchar * label,
		   gchar * tooltip)
{
  GtkWidget *value = gtk_toggle_button_new_with_label (_("No"));
  gtk_signal_connect (GTK_OBJECT (value), "toggled",
		      GTK_SIGNAL_FUNC (on_bool_property_toggle), NULL);
  property_add (property_name, label, value, NULL, tooltip);
}


void
property_add_choice (gchar * property_name,
		     gchar * label,
		     gchar * tooltip,
		     gchar ** choices)
{
  GtkWidget *value = gtk_option_menu_new ();
  GtkWidget *menu = gtk_menu_new ();

  gint i = 0;
  while (choices[i])
    {
      GtkWidget *menuitem = gtk_menu_item_new_with_label ((gchar *) (choices[i]));
      gtk_menu_append (GTK_MENU (menu), menuitem);
      gtk_widget_show (menuitem);
      gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
			  GTK_SIGNAL_FUNC (on_property_changed), value);
      i++;
    }
  gtk_option_menu_set_menu (GTK_OPTION_MENU (value), menu);
  property_add (property_name, label, value, NULL, tooltip);
}


void
property_add_combo (gchar * property_name,
		    gchar * label,
		    gchar * tooltip,
		    GList * choices)
{
  GtkWidget *value = gtk_combo_new ();
  gtk_widget_set_usize (GTK_COMBO (value)->entry, 60, -1);
  gtk_widget_set_usize (value, 80, -1);
  if (choices)
    gtk_combo_set_popdown_strings (GTK_COMBO (value), choices);
  gtk_signal_connect (GTK_OBJECT (GTK_COMBO (value)->entry), "changed",
		      GTK_SIGNAL_FUNC (on_property_changed), value);
  property_add (property_name, label, value, NULL, tooltip);
}


void
property_add_color (gchar * property_name,
		    gchar * label,
		    gchar * tooltip)
{
  GtkWidget *value, *preview, *dialog_button;
  value = gtk_frame_new (NULL);
  preview = create_color_preview ();
  gtk_widget_set_events (preview, gtk_widget_get_events (preview)
			 | GDK_BUTTON_PRESS_MASK);
  gtk_signal_connect_after (GTK_OBJECT (preview), "expose_event",
			    GTK_SIGNAL_FUNC (on_color_expose_event), value);
  gtk_signal_connect_after (GTK_OBJECT (preview), "draw",
			    GTK_SIGNAL_FUNC (on_color_draw), value);
  gtk_signal_connect (GTK_OBJECT (preview), "button_press_event",
		      GTK_SIGNAL_FUNC (on_color_select), value);

  gtk_container_add (GTK_CONTAINER (value), preview);
  dialog_button = gtk_button_new_with_label ("...");
  gtk_signal_connect (GTK_OBJECT (dialog_button), "clicked",
		      GTK_SIGNAL_FUNC (show_colorsel_dialog), value);
  property_add (property_name, label, value, dialog_button, tooltip);
}


void
property_add_bgpixmap (gchar * property_name,
		       gchar * label,
		       gchar * tooltip)
{
  GtkWidget *value, *drawing_area, *dialog_button;
  value = gtk_frame_new (NULL);
  drawing_area = gtk_drawing_area_new ();
  gtk_widget_set_events (drawing_area, gtk_widget_get_events (drawing_area)
			 | GDK_BUTTON_PRESS_MASK | GDK_EXPOSURE_MASK);
  gtk_drawing_area_size (GTK_DRAWING_AREA (drawing_area), 100, 20);
  gtk_widget_show (drawing_area);
  gtk_signal_connect (GTK_OBJECT (drawing_area), "expose_event",
		      GTK_SIGNAL_FUNC (expose_pixmap), NULL);
  gtk_signal_connect (GTK_OBJECT (drawing_area), "draw",
		      GTK_SIGNAL_FUNC (draw_pixmap), NULL);
  gtk_signal_connect (GTK_OBJECT (drawing_area), "draw_default",
		      GTK_SIGNAL_FUNC (draw_pixmap_focus), NULL);
  gtk_signal_connect (GTK_OBJECT (drawing_area), "draw_focus",
		      GTK_SIGNAL_FUNC (draw_pixmap_focus), NULL);
  gtk_signal_connect (GTK_OBJECT (drawing_area), "button_press_event",
		      GTK_SIGNAL_FUNC (on_color_select), value);
  gtk_container_add (GTK_CONTAINER (value), drawing_area);
  dialog_button = gtk_button_new_with_label ("...");
  gtk_signal_connect (GTK_OBJECT (dialog_button), "clicked",
		      GTK_SIGNAL_FUNC (show_filesel_dialog), value);
  property_add (property_name, label, value, dialog_button, tooltip);
}


void
property_add_dialog (gchar * property_name,
		     gchar * label,
		     gchar * tooltip,
		     gboolean editable,
		     GtkCallback callback)
{
  GtkWidget *value, *dialog_button;
  value = gtk_entry_new ();
  gtk_widget_set_usize (value, 80, -1);
  gtk_entry_set_editable (GTK_ENTRY (value), editable);
  dialog_button = gtk_button_new_with_label ("...");
  gtk_signal_connect (GTK_OBJECT (dialog_button), "clicked",
		      GTK_SIGNAL_FUNC (callback), value);
  gtk_signal_connect (GTK_OBJECT (value), "changed",
		      GTK_SIGNAL_FUNC (on_property_changed), value);
  property_add (property_name, label, value, dialog_button, tooltip);
}


void
property_add_filename (gchar * property_name,
		       gchar * label,
		       gchar * tooltip)
{
  GtkWidget *value, *dialog_button;
  value = gtk_entry_new ();
  gtk_widget_set_usize (value, 80, -1);
  dialog_button = gtk_button_new_with_label ("...");
  gtk_signal_connect (GTK_OBJECT (dialog_button), "clicked",
		      GTK_SIGNAL_FUNC (show_filesel_dialog), value);
  property_add (property_name, label, value, dialog_button, tooltip);
}


void
property_add_font (gchar * property_name,
		   gchar * label,
		   gchar * tooltip)
{
  GtkWidget *value, *dialog_button;
  value = gtk_entry_new ();
  gtk_widget_set_usize (value, 80, -1);
  gtk_entry_set_editable (GTK_ENTRY (value), FALSE);
  dialog_button = gtk_button_new_with_label ("...");
  gtk_signal_connect (GTK_OBJECT (dialog_button), "clicked",
		      GTK_SIGNAL_FUNC (show_font_dialog), value);
  gtk_signal_connect (GTK_OBJECT (value), "changed",
		      GTK_SIGNAL_FUNC (on_property_changed), value);
  property_add (property_name, label, value, dialog_button, tooltip);
}


void
property_add (gchar * property_name,
	      gchar * label_string,
	      GtkWidget * value,
	      GtkWidget * dialog_button,
	      gchar * tooltip)
{
  GtkWidget *label, *eventbox;
  gchar *property_name_copy;

  label = gtk_label_new (label_string);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.3);
  /*gtk_misc_set_padding(GTK_MISC(label), 0, 5); */
  gtk_widget_show (label);

  /* We put the label in the event box so we can set a tooltip. */
  eventbox = gtk_event_box_new ();
  gtk_widget_set_usize (eventbox, 100, -1);
  gtk_widget_show (eventbox);
  gtk_container_add (GTK_CONTAINER (eventbox), label);

  gtk_widget_show (value);
  if (tooltip)
    gtk_tooltips_set_tip (tooltips, eventbox, tooltip, NULL);

  gtk_table_attach (GTK_TABLE (property_table), eventbox, 0, 1,
		    property_table_row, property_table_row + 1,
		    GTK_FILL, GTK_FILL, 1, 1);
  if (dialog_button)
    {
      gtk_table_attach (GTK_TABLE (property_table), value, 1, 2,
			property_table_row, property_table_row + 1,
			GTK_EXPAND | GTK_FILL, 0, 1, 1);
      gtk_widget_show (dialog_button);
      gtk_table_attach (GTK_TABLE (property_table), dialog_button, 2, 3,
			property_table_row, property_table_row + 1,
			0, 0, 1, 1);
    }
  else
    {
      gtk_table_attach (GTK_TABLE (property_table), value, 1, 3,
			property_table_row, property_table_row + 1,
			GTK_EXPAND | GTK_FILL, 0, 1, 1);
    }
  property_table_row++;

  /* Insert property label & value widgets into hash tables */
  property_name_copy = g_strdup (property_name);
  g_hash_table_insert (gb_property_labels, property_name_copy, label);
  g_hash_table_insert (gb_property_values, property_name_copy, value);
  if (dialog_button)
    g_hash_table_insert (gb_property_buttons,
			 property_name_copy,
			 dialog_button);
}


/*
 * Functions for getting/setting properties.
 * NOTE: should also specify whether set properties are copied and
 *       if values returned by get should be freed.
 *       For widgets which use a GtkEntry: string value set is copied,
 *       & don't free string returned by get (it is the actual entry text).
 *
 * NOTE: property_get_*() functions also set the apply flag.
 *       This is used to enable the automatic applying of
 *       properties as they are changed in the property editor. It is set
 *       to TRUE if the property widget matches the to_apply widget.
 */


gchar *
property_get_string (gchar * property_name,
		     GtkWidget * to_apply,
		     gboolean * apply)
{
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  if (apply)
    *apply = (!to_apply || to_apply == widget) ? TRUE : FALSE;
  g_return_val_if_fail (widget != NULL, "");
  return (gtk_entry_get_text (GTK_ENTRY (widget)));
}


void
property_set_string (gchar * property_name,
		     gchar * value)
{
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  g_return_if_fail (widget != NULL);
  gtk_entry_set_text (GTK_ENTRY (widget), value ? value : "");
}


/* Note: returned string must be freed with g_free() */
gchar *
property_get_text (gchar * property_name,
		   GtkWidget * to_apply,
		   gboolean * apply)
{
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  if (apply)
    *apply = (!to_apply || to_apply == widget) ? TRUE : FALSE;
  g_return_val_if_fail (widget != NULL, "");
  return (gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1));
}


void
property_set_text (gchar * property_name,
		   gchar * value)
{
  gchar *text;
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  g_return_if_fail (widget != NULL);
  gtk_editable_delete_text (GTK_EDITABLE (widget), 0, -1);
  text = value ? value : "";
  gtk_text_insert (GTK_TEXT (widget), NULL, NULL, NULL, text, strlen (text));
}


gint
property_get_int (gchar * property_name,
		  GtkWidget * to_apply,
		  gboolean * apply)
{
  gchar *text;
  gint value = 0;
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  if (apply)
    *apply = (!to_apply || to_apply == widget) ? TRUE : FALSE;
  g_return_val_if_fail (widget != NULL, 0);
  if (GTK_IS_SPIN_BUTTON (widget))
    {
      return gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widget));
    }
  else
    {
      text = gtk_entry_get_text (GTK_ENTRY (widget));
      sscanf (text, "%i", &value);
      return value;
    }
}


void
property_set_int (gchar * property_name,
		  gint value)
{
  gchar buffer[128];
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  g_return_if_fail (widget != NULL);
  if (GTK_IS_SPIN_BUTTON (widget))
    {
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (widget), value);
    }
  else
    {
      sprintf (buffer, "%i", value);
      gtk_entry_set_text (GTK_ENTRY (widget), buffer);
    }
}


gfloat
property_get_float (gchar * property_name,
		    GtkWidget * to_apply,
		    gboolean * apply)
{
  gchar *text;
  gfloat value = 0;
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  if (apply)
    *apply = (!to_apply || to_apply == widget) ? TRUE : FALSE;
  g_return_val_if_fail (widget != NULL, 0);
  if (GTK_IS_SPIN_BUTTON (widget))
    {
      return gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (widget));
    }
  else
    {
      text = gtk_entry_get_text (GTK_ENTRY (widget));
      sscanf (text, "%f", &value);
      return value;
    }
}


void
property_set_float (gchar * property_name,
		    gfloat value)
{
  gchar buffer[128];
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  g_return_if_fail (widget != NULL);
  if (GTK_IS_SPIN_BUTTON (widget))
    {
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (widget), value);
    }
  else
    {
      sprintf (buffer, "%g", value);
      gtk_entry_set_text (GTK_ENTRY (widget), buffer);
    }
}


gboolean
property_get_bool (gchar * property_name,
		   GtkWidget * to_apply,
		   gboolean * apply)
{
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  if (apply)
    *apply = (!to_apply || to_apply == widget) ? TRUE : FALSE;
  g_return_val_if_fail (widget != NULL, FALSE);
  return (GTK_TOGGLE_BUTTON (widget)->active ? TRUE : FALSE);
}


void
property_set_bool (gchar * property_name,
		   gint value)
{
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  g_return_if_fail (widget != NULL);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (widget), value);
}


gchar *
property_get_choice (gchar * property_name,
		     GtkWidget * to_apply,
		     gboolean * apply)
{
  GtkWidget *label;
  gchar *label_text;
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  if (apply)
    *apply = (!to_apply || to_apply == widget) ? TRUE : FALSE;
  g_return_val_if_fail (widget != NULL, NULL);
  label = GTK_BUTTON (widget)->child;
  gtk_label_get (GTK_LABEL (label), &label_text);
  return label_text;
}


void
property_set_choice (gchar * property_name,
		     gint value)
{
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  g_return_if_fail (widget != NULL);
  gtk_option_menu_set_history (GTK_OPTION_MENU (widget), value);
}


gchar *
property_get_combo (gchar * property_name,
		    GtkWidget * to_apply,
		    gboolean * apply)
{
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  if (apply)
    *apply = (!to_apply || to_apply == widget) ? TRUE : FALSE;
  g_return_val_if_fail (widget != NULL, "");
  return (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (widget)->entry)));
}


void
property_set_combo (gchar * property_name,
		    gchar * value)
{
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  g_return_if_fail (widget != NULL);
  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (widget)->entry), value ? value : "");
}


void
property_set_combo_strings (gchar * property_name,
			    GList * choices)
{
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  g_return_if_fail (widget != NULL);
  if (choices)
    gtk_combo_set_popdown_strings (GTK_COMBO (widget), choices);
  else
    gtk_list_clear_items (GTK_LIST (GTK_COMBO (widget)->list), 0, -1);
}


guchar *
property_get_color (gchar * property_name,
		    GtkWidget * to_apply,
		    gboolean * apply)
{
  guchar *color;
  GtkWidget *preview;
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  if (apply)
    *apply = (!to_apply || to_apply == widget) ? TRUE : FALSE;
  g_return_val_if_fail (widget != NULL, NULL);
  preview = GTK_BIN (widget)->child;
  g_return_val_if_fail (GTK_IS_PREVIEW (preview), NULL);
  color = gtk_object_get_data (GTK_OBJECT (preview), GbColorKey);
  g_return_val_if_fail (color != NULL, NULL);
  return color;
}


void
property_set_color (gchar * property_name,
		    guchar * value)
{
  GtkWidget *preview;
  guchar *color;
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  g_return_if_fail (widget != NULL);
  preview = GTK_BIN (widget)->child;
  g_return_if_fail (GTK_IS_PREVIEW (preview));
  color = gtk_object_get_data (GTK_OBJECT (preview), GbColorKey);
  g_return_if_fail (color != NULL);
  color[0] = value[0];
  color[1] = value[1];
  color[2] = value[2];
  show_color_in_preview (preview, NULL);
}


GdkPixmap *
property_get_bgpixmap (gchar * property_name,
		       GtkWidget * to_apply,
		       gboolean * apply,
		       gchar ** filename)
{
  GtkWidget *drawing_area;
  GdkPixmap *gdkpixmap;
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  if (apply)
    *apply = (!to_apply || to_apply == widget) ? TRUE : FALSE;
  g_return_val_if_fail (widget != NULL, NULL);
  drawing_area = GTK_BIN (widget)->child;
  g_return_val_if_fail (GTK_IS_DRAWING_AREA (drawing_area), NULL);

  gdkpixmap = gtk_object_get_data (GTK_OBJECT (drawing_area), GbBgPixmapKey);
  *filename = gtk_object_get_data (GTK_OBJECT (drawing_area), GbBgFilenameKey);
  return gdkpixmap;
}


void
property_set_bgpixmap (gchar * property_name,
		       GdkPixmap * gdkpixmap,
		       gchar * filename)
{
  GtkWidget *drawing_area;
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  g_return_if_fail (widget != NULL);
  drawing_area = GTK_BIN (widget)->child;
  g_return_if_fail (GTK_IS_DRAWING_AREA (drawing_area));

  set_pixmap (drawing_area, gdkpixmap, filename);
}


static void
set_pixmap (GtkWidget * drawing_area, GdkPixmap * gdkpixmap, gchar * filename)
{
  GdkPixmap *old_gdkpixmap;
  gchar *old_filename, *filename_copy;

  /* free/unref any existing values */
  old_gdkpixmap = gtk_object_get_data (GTK_OBJECT (drawing_area), GbBgPixmapKey);
  if (old_gdkpixmap)
    gdk_pixmap_unref (old_gdkpixmap);
  old_filename = gtk_object_get_data (GTK_OBJECT (drawing_area), GbBgFilenameKey);
  g_free (old_filename);

  gtk_object_set_data (GTK_OBJECT (drawing_area), GbBgPixmapKey, gdkpixmap);
  filename_copy = filename ? g_strdup (filename) : NULL;
  gtk_object_set_data (GTK_OBJECT (drawing_area), GbBgFilenameKey, filename_copy);

  if (gdkpixmap)
    gdk_pixmap_ref (gdkpixmap);

  show_pixmap_in_drawing_area (drawing_area, gdkpixmap);
}


gpointer
property_get_dialog (gchar * property_name,
		     GtkWidget * to_apply,
		     gboolean * apply)
{
  gpointer dialog_value;
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  if (apply)
    *apply = (!to_apply || to_apply == widget) ? TRUE : FALSE;
  g_return_val_if_fail (widget != NULL, NULL);
  dialog_value = gtk_object_get_data (GTK_OBJECT (widget), GbDialogValueKey);
  if (dialog_value)
    return dialog_value;
  else
    return (gtk_entry_get_text (GTK_ENTRY (widget)));
}


void
property_set_dialog (gchar * property_name,
		     gchar * string,
		     gpointer value)
{
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  g_return_if_fail (widget != NULL);
  if (value)
    gtk_object_set_data (GTK_OBJECT (widget), GbDialogValueKey, value);
  gtk_entry_set_text (GTK_ENTRY (widget), string ? string : "");
}


gchar *
property_get_filename (gchar * property_name,
		       GtkWidget * to_apply,
		       gboolean * apply)
{
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  if (apply)
    *apply = (!to_apply || to_apply == widget) ? TRUE : FALSE;
  g_return_val_if_fail (widget != NULL, "");
  return (gtk_entry_get_text (GTK_ENTRY (widget)));
}


void
property_set_filename (gchar * property_name,
		       gchar * value)
{
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  g_return_if_fail (widget != NULL);
  gtk_entry_set_text (GTK_ENTRY (widget), value ? value : "");
}


GdkFont *
property_get_font (gchar * property_name,
		   GtkWidget * to_apply,
		   gboolean * apply,
		   gchar ** xlfd_fontname)
{
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  if (apply)
    *apply = (!to_apply || to_apply == widget) ? TRUE : FALSE;
  g_return_val_if_fail (widget != NULL, NULL);
  *xlfd_fontname = gtk_object_get_data (GTK_OBJECT (widget), GbFontSpecKey);
  return (gtk_object_get_data (GTK_OBJECT (widget), GbFontKey));
}


void
property_set_font (gchar * property_name,
		   GdkFont * font,
		   gchar * xlfd_fontname)
{
  GdkFont *old_font;
  gchar *old_xlfd_fontname;
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  g_return_if_fail (widget != NULL);

  old_font = gtk_object_get_data (GTK_OBJECT (widget), GbFontKey);
  if (old_font)
    gdk_font_unref (old_font);
  gdk_font_ref (font);
  gtk_object_set_data (GTK_OBJECT (widget), GbFontKey, font);
  old_xlfd_fontname = gtk_object_get_data (GTK_OBJECT (widget), GbFontSpecKey);
  g_free (old_xlfd_fontname);
  gtk_object_set_data (GTK_OBJECT (widget), GbFontSpecKey, g_strdup (xlfd_fontname));
  gtk_entry_set_text (GTK_ENTRY (widget), get_font_name_from_spec (xlfd_fontname));
}


/*
 * Setting properties sensitive/insensitive
 */
void
property_set_sensitive (gchar * property_name, gboolean sensitive)
{
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  g_return_if_fail (widget != NULL);
  gtk_widget_set_sensitive (widget, sensitive);

  widget = (GtkWidget *) g_hash_table_lookup (gb_property_labels,
					      property_name);
  if (widget)
    gtk_widget_set_sensitive (widget, sensitive);

  widget = (GtkWidget *) g_hash_table_lookup (gb_property_buttons,
					      property_name);
  if (widget)
    gtk_widget_set_sensitive (widget, sensitive);
}


/*
 * Setting properties valid/invalid
 */
void
property_set_valid (gchar * property_name, gboolean valid)
{
  GtkWidget *widget = (GtkWidget *) g_hash_table_lookup (gb_property_values,
							 property_name);
  g_return_if_fail (widget != NULL);

  if (valid)
    gtk_widget_set_style (widget, gtk_widget_get_default_style ());
  else
    gtk_widget_set_style (widget, invalid_style);
}


/*
 * Color previews - for showing style colors
 */

static GtkWidget *
create_color_preview ()
{
  GtkWidget *preview;
  guchar *color;

  /* As of GTK 1.0.2 this pushing/popping isn't needed. */
  gtk_widget_push_visual (gtk_preview_get_visual ());
  gtk_widget_push_colormap (gtk_preview_get_cmap ());
  preview = gtk_preview_new (GTK_PREVIEW_COLOR);
  gtk_widget_pop_colormap ();
  gtk_widget_pop_visual ();
  gtk_preview_set_expand (GTK_PREVIEW (preview), TRUE);
  gtk_preview_size (GTK_PREVIEW (preview), 100, 20);
  gtk_widget_show (preview);
  gtk_signal_connect_after (GTK_OBJECT (preview), "size_allocate",
			    GTK_SIGNAL_FUNC (show_color_in_preview), NULL);

  color = g_new (guchar, 4);
  color[0] = color[1] = color[2] = color[3] = 255;
  gtk_object_set_data (GTK_OBJECT (preview), GbColorKey, color);
  return preview;
}


static void
show_color_in_preview (GtkWidget * preview, gpointer data)
{
  static guchar *buf = NULL;
  static gint buf_width = 0;

  gint i, j;
  gint width = preview->allocation.width;
  gint height = preview->allocation.height;
  guchar red, green, blue;
  guchar *color, *p;

  g_return_if_fail (GTK_IS_PREVIEW (preview));
  color = gtk_object_get_data (GTK_OBJECT (preview), GbColorKey);
  g_return_if_fail (color != NULL);
  red = color[0];
  green = color[1];
  blue = color[2];

  if (buf == NULL || (width > buf_width))
    {
      g_free (buf);
      buf_width = width;
      buf = g_new (guchar, 3 * buf_width);
      g_return_if_fail (buf != NULL);
    }

  p = buf;
  for (j = 0; j < width; j++)
    {
      *p++ = red;
      *p++ = green;
      *p++ = blue;
    }

  for (i = 0; i < height; i++)
    gtk_preview_draw_row (GTK_PREVIEW (preview), buf, 0, i, width);

  gtk_widget_queue_draw (preview);
}


/*
 * Pixmap values (displayed as background of drawing area)
 */
static void
show_pixmap_in_drawing_area (GtkWidget * drawing_area, GdkPixmap * gdkpixmap)
{
  g_return_if_fail (GTK_IS_DRAWING_AREA (drawing_area));
  /* The drawing area doesn't get a window until the notebook page is shown! */
  if (drawing_area->window == NULL)
    return;
  if (gdkpixmap)
    {
      gdk_window_set_back_pixmap (drawing_area->window, gdkpixmap, FALSE);
    }
  else
    {
      gdk_window_set_background (drawing_area->window,
				 &drawing_area->style->bg[GTK_STATE_NORMAL]);
    }
  gdk_window_clear (drawing_area->window);
}


static gint
expose_pixmap (GtkWidget * drawing_area, GdkEventExpose * event, gpointer data)
{
  real_draw_pixmap (drawing_area);
  return FALSE;
}


static void
draw_pixmap (GtkWidget * drawing_area, GdkRectangle * area, gpointer data)
{
  real_draw_pixmap (drawing_area);
}


static void
draw_pixmap_focus (GtkWidget * drawing_area, gpointer data)
{
  real_draw_pixmap (drawing_area);
}


static void
real_draw_pixmap (GtkWidget * drawing_area)
{
  GdkPixmap *gdkpixmap;
  g_return_if_fail (GTK_IS_DRAWING_AREA (drawing_area));
  gdkpixmap = gtk_object_get_data (GTK_OBJECT (drawing_area), GbBgPixmapKey);
  show_pixmap_in_drawing_area (drawing_area, gdkpixmap);
  on_color_draw (drawing_area, NULL);
}


/*
 * The Events dialog, used for selecting which X events to receive.
 */

static void
show_events_dialog (GtkWidget * widget, gpointer value)
{
  GtkDialog *dialog;
  GtkWidget *clist, *button;
  gchar *titles[2];
  gchar *row[2];
  int i;
  gchar *event_mask_string;
  gint event_mask_value;

  dialog = GTK_DIALOG (gtk_dialog_new ());
  gtk_window_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);

  gtk_signal_connect_object (GTK_OBJECT (dialog), "delete_event",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));

  gtk_window_set_title (GTK_WINDOW (dialog), _("Select X Events"));

  titles[0] = _("Event Mask");
  titles[1] = _("Description");
  clist = gtk_clist_new_with_titles (2, titles);
  gtk_clist_column_titles_passive (GTK_CLIST (clist));
  gtk_clist_set_column_width (GTK_CLIST (clist), 0, 230);
  gtk_clist_set_column_width (GTK_CLIST (clist), 1, 100);
  gtk_clist_set_selection_mode (GTK_CLIST (clist), GTK_SELECTION_MULTIPLE);
  gtk_clist_set_policy (GTK_CLIST (clist), GTK_POLICY_AUTOMATIC,
			GTK_POLICY_AUTOMATIC);
  gtk_widget_set_usize (clist, 500, 350);
  gtk_widget_show (clist);
  gtk_box_pack_start (GTK_BOX (dialog->vbox), clist, TRUE, TRUE, 0);

  /* Insert events & descriptions */
  gtk_clist_freeze (GTK_CLIST (clist));
  for (i = 0; i < GB_EVENT_MASKS_COUNT; i++)
    {
      row[0] = GbEventMaskSymbols[i];
      row[1] = gettext (GbEventMaskDescriptions[i]);
      gtk_clist_append (GTK_CLIST (clist), row);
    }

  /* Select rows according to current mask setting */
  event_mask_string = gtk_entry_get_text (GTK_ENTRY (value));
  event_mask_value = property_events_string_to_value (event_mask_string);
  for (i = 0; i < GB_EVENT_MASKS_COUNT; i++)
    {
      if (event_mask_value & GbEventMaskValues[i])
	{
	  gtk_clist_select_row (GTK_CLIST (clist), i, 0);
	}
    }
  gtk_clist_thaw (GTK_CLIST (clist));

  button = gtk_button_new_with_label (_("OK"));
  gtk_box_pack_start (GTK_BOX (dialog->action_area), button, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);
  /* Save pointer to value to use when OK pressed */
  gtk_object_set_data (GTK_OBJECT (clist), GbValueWidgetKey, value);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_events_dialog_ok), clist);

  button = gtk_button_new_with_label (_("Cancel"));
  gtk_box_pack_start (GTK_BOX (dialog->action_area), button, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_show (button);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));
  gtk_widget_show (GTK_WIDGET (dialog));
}


static void
on_events_dialog_ok (GtkWidget * widget, GtkWidget * clist)
{
  gint row, mask_value = 0;
  GtkWidget *value;
  GList *selection = GTK_CLIST (clist)->selection;

  while (selection)
    {
      row = GPOINTER_TO_INT (selection->data);
      mask_value |= GbEventMaskValues[row];
      selection = selection->next;
    }

  value = gtk_object_get_data (GTK_OBJECT (clist), GbValueWidgetKey);
  g_return_if_fail (value != NULL);
  gtk_entry_set_text (GTK_ENTRY (value),
		      property_events_value_to_string (mask_value));
  gtk_widget_destroy (gtk_widget_get_toplevel (clist));
}


/* Converts the events gint to a string of 0s and 1s for displaying */
gchar *
property_events_value_to_string (gint event_mask)
{
  static gchar buf[GB_EVENT_MASKS_COUNT + 2];
  int i;

  for (i = 0; i < GB_EVENT_MASKS_COUNT; i++)
    {
      if (event_mask & GbEventMaskValues[i])
	buf[GB_EVENT_MASKS_COUNT - i - 1] = '1';
      else
	buf[GB_EVENT_MASKS_COUNT - i - 1] = '0';
    }
  buf[GB_EVENT_MASKS_COUNT] = '0';
  buf[GB_EVENT_MASKS_COUNT + 1] = '\0';
  return buf;
}


/* Converts the string of 0s and 1s back to a gint event mask */
gint
property_events_string_to_value (gchar * event_string)
{
  gint i, value = 0;

  if (strlen (event_string) < GB_EVENT_MASKS_COUNT)
    return 0;
  for (i = 0; i < GB_EVENT_MASKS_COUNT; i++)
    {
      if (event_string[GB_EVENT_MASKS_COUNT - i - 1] == '1')
	value |= GbEventMaskValues[i];
    }
  return value;
}


/*
 * The Accelerators page
 */

static void
on_accelerator_add (GtkWidget * widget, GtkWidget * clist)
{
  gchar modifiers[4];
  gchar *row[3];
  gchar *key, *signal;

  key = property_get_string (GbAccelKey, NULL, NULL);
  if (strlen (key) == 0)
    {
      show_message_box (_("You need to set the accelerator key"));
      return;
    }
  signal = property_get_string (GbAccelSignal, NULL, NULL);
  if (strlen (signal) == 0)
    {
      show_message_box (_("You need to set the signal to emit"));
      return;
    }

  modifiers[0] = modifiers[1] = modifiers[2] = ' ';
  modifiers[3] = '\0';
  if (GTK_TOGGLE_BUTTON (accel_control_button)->active)
    modifiers[0] = 'C';
  if (GTK_TOGGLE_BUTTON (accel_shift_button)->active)
    modifiers[1] = 'S';
  if (GTK_TOGGLE_BUTTON (accel_alt_button)->active)
    modifiers[2] = 'A';

  row[ACCEL_MODIFIERS_COL] = modifiers;
  row[ACCEL_KEY_COL] = key;
  row[ACCEL_SIGNAL_COL] = signal;
  gtk_clist_append (GTK_CLIST (clist), row);

  /* clear the key & signal fields */
  property_set_string (GbAccelKey, "");
  property_set_string (GbAccelSignal, "");

  on_property_changed (clist, clist);
}


static void
on_accelerator_update (GtkWidget * widget, GtkWidget * clist)
{
  gchar modifiers[4];
  gchar *key, *signal;
  GList *selection = GTK_CLIST (clist)->selection;
  gint row;

  if (!selection)
    return;
  row = GPOINTER_TO_INT (selection->data);

  key = property_get_string (GbAccelKey, NULL, NULL);
  if (strlen (key) == 0)
    {
      show_message_box (_("You need to set the accelerator key"));
      return;
    }
  signal = property_get_string (GbAccelSignal, NULL, NULL);
  if (strlen (signal) == 0)
    {
      show_message_box (_("You need to set the signal to emit"));
      return;
    }

  modifiers[0] = modifiers[1] = modifiers[2] = ' ';
  modifiers[3] = '\0';
  if (GTK_TOGGLE_BUTTON (accel_control_button)->active)
    modifiers[0] = 'C';
  if (GTK_TOGGLE_BUTTON (accel_shift_button)->active)
    modifiers[1] = 'S';
  if (GTK_TOGGLE_BUTTON (accel_alt_button)->active)
    modifiers[2] = 'A';

  gtk_clist_set_text (GTK_CLIST (clist), row, ACCEL_MODIFIERS_COL, modifiers);
  gtk_clist_set_text (GTK_CLIST (clist), row, ACCEL_KEY_COL, key);
  gtk_clist_set_text (GTK_CLIST (clist), row, ACCEL_SIGNAL_COL, signal);

  on_property_changed (clist, clist);
}


static void
on_accelerator_delete (GtkWidget * widget, GtkWidget * clist)
{
  GList *selection = GTK_CLIST (clist)->selection;
  gint row;

  if (!selection)
    return;
  row = GPOINTER_TO_INT (selection->data);
  gtk_clist_remove (GTK_CLIST (clist), row);
  /* clear the key & signal fields */
  property_set_string (GbAccelKey, "");
  property_set_string (GbAccelSignal, "");

  on_property_changed (clist, clist);
}


static void
on_accelerator_clear (GtkWidget * widget, GtkWidget * clist)
{
  property_set_string (GbAccelKey, "");
  property_set_string (GbAccelSignal, "");

  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (accel_control_button), FALSE);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (accel_shift_button), FALSE);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (accel_alt_button), FALSE);
}


static void
on_accelerator_select (GtkWidget * clist, gint row, gint column,
		       GdkEventButton * event, gpointer user_data)
{
  gchar *modifiers, *key, *signal;
  gint len;

  gtk_clist_get_text (GTK_CLIST (clist), row, ACCEL_MODIFIERS_COL, &modifiers);
  gtk_clist_get_text (GTK_CLIST (clist), row, ACCEL_KEY_COL, &key);
  gtk_clist_get_text (GTK_CLIST (clist), row, ACCEL_SIGNAL_COL, &signal);

  len = strlen (modifiers);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (accel_control_button),
			       len >= 1 && modifiers[0] != ' ');
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (accel_shift_button),
			       len >= 2 && modifiers[1] != ' ');
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (accel_alt_button),
			       len >= 3 && modifiers[2] != ' ');

  property_set_string (GbAccelKey, key);
  property_set_string (GbAccelSignal, signal);
}


void
property_clear_accelerators ()
{
  gtk_clist_clear (GTK_CLIST (accel_clist));
  /* clear the key & signal fields */
  property_set_string (GbAccelKey, "");
  property_set_string (GbAccelSignal, "");
}


void
property_add_accelerator (GbAccelerator * accel)
{
  gchar modifiers[4];
  gchar *row[3];

  modifiers[0] = modifiers[1] = modifiers[2] = ' ';
  modifiers[3] = '\0';
  if (accel->modifiers & GDK_CONTROL_MASK)
    modifiers[0] = 'C';
  if (accel->modifiers & GDK_SHIFT_MASK)
    modifiers[1] = 'S';
  /* The Alt key uses GDK_MOD1_MASK */
  if (accel->modifiers & GDK_MOD1_MASK)
    modifiers[2] = 'A';

  row[ACCEL_MODIFIERS_COL] = modifiers;
  row[ACCEL_KEY_COL] = accel->key;
  row[ACCEL_SIGNAL_COL] = accel->signal;
  gtk_clist_append (GTK_CLIST (accel_clist), row);
}


gboolean
property_is_accel_clist (GtkWidget * widget)
{
  return (widget == accel_clist) ? TRUE : FALSE;
}


GList *
property_get_accelerators ()
{
  gint row, len;
  GList *accelerators = NULL;
  GbAccelerator *accel;
  gchar *modifiers, *key, *signal;

  for (row = 0; row < GTK_CLIST (accel_clist)->rows; row++)
    {
      accel = g_new (GbAccelerator, 1);

      gtk_clist_get_text (GTK_CLIST (accel_clist), row,
			  ACCEL_MODIFIERS_COL, &modifiers);
      gtk_clist_get_text (GTK_CLIST (accel_clist), row,
			  ACCEL_KEY_COL, &key);
      gtk_clist_get_text (GTK_CLIST (accel_clist), row,
			  ACCEL_SIGNAL_COL, &signal);

      len = strlen (modifiers);
      accel->modifiers = 0;
      if (len >= 1 && modifiers[0] != ' ')
	accel->modifiers |= GDK_CONTROL_MASK;
      if (len >= 2 && modifiers[1] != ' ')
	accel->modifiers |= GDK_SHIFT_MASK;
      if (len >= 3 && modifiers[2] != ' ')
	accel->modifiers |= GDK_MOD1_MASK;

      accel->key = g_strdup (key);
      accel->signal = g_strdup (signal);
      accelerators = g_list_append (accelerators, accel);
    }
  return accelerators;
}


/*
 * The Accelerator Keys dialog for selecting an accelerator key.
 */

static void
show_keys_dialog (GtkWidget * widget, gpointer value)
{
  GladeKeysDialog *dialog;

  dialog = GLADE_KEYS_DIALOG (glade_keys_dialog_new ());
  gtk_window_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Select Accelerator Key"));
  /* Save pointer to value to use when OK pressed */
  gtk_object_set_data (GTK_OBJECT (dialog), GbValueWidgetKey, value);
  gtk_signal_connect_object (GTK_OBJECT (dialog), "delete_event",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));
  gtk_signal_connect (GTK_OBJECT (dialog->clist), "select_row",
		      GTK_SIGNAL_FUNC (on_keys_clist_select), dialog);
  gtk_signal_connect (GTK_OBJECT (dialog->ok_button), "clicked",
		      GTK_SIGNAL_FUNC (on_keys_dialog_ok), dialog);
  gtk_signal_connect_object (GTK_OBJECT (dialog->cancel_button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));
  gtk_widget_show (GTK_WIDGET (dialog));
}


static void
on_keys_dialog_ok (GtkWidget * widget, gpointer data)
{
  GladeKeysDialog *dialog;
  GtkWidget *value;
  gchar *key_symbol;

  dialog = (GladeKeysDialog*) gtk_widget_get_toplevel (widget);
  key_symbol = glade_keys_dialog_get_key_symbol (dialog);
  if (key_symbol)
    {
      value = gtk_object_get_data (GTK_OBJECT (dialog), GbValueWidgetKey);
      g_return_if_fail (value != NULL);
      gtk_entry_set_text (GTK_ENTRY (value), key_symbol);
    }

  gtk_widget_destroy (GTK_WIDGET (dialog));
}


static void
on_keys_clist_select (GtkWidget * widget, gint row, gint column,
		      GdkEventButton * bevent, gpointer data)
{
  if (bevent && bevent->type == GDK_2BUTTON_PRESS)
    on_keys_dialog_ok (widget, NULL);
}


/*
 * The Signals page
 */

static void
on_signal_add (GtkWidget * widget, GtkWidget * clist)
{
  gchar *row[5];
  gchar *signal, *handler, *object, *data;
  gboolean after;

  signal = property_get_string (GbSignalName, NULL, NULL);
  if (strlen (signal) == 0)
    {
      show_message_box (_("You need to set the signal name"));
      return;
    }
  handler = property_get_string (GbSignalHandler, NULL, NULL);
  if (strlen (handler) == 0)
    {
      show_message_box (_("You need to set the handler for the signal"));
      return;
    }
  object = property_get_string (GbSignalObject, NULL, NULL);
  after = property_get_bool (GbSignalAfter, NULL, NULL);
  data = property_get_string (GbSignalData, NULL, NULL);

  row[SIGNAL_NAME_COL] = signal;
  row[SIGNAL_HANDLER_COL] = handler;
  row[SIGNAL_OBJECT_COL] = object;
  row[SIGNAL_AFTER_COL] = after ? "Y" : "";
  row[SIGNAL_DATA_COL] = data;
  gtk_clist_append (GTK_CLIST (clist), row);

  /* clear the fields */
  property_set_string (GbSignalName, "");
  property_set_string (GbSignalHandler, "");
  property_set_string (GbSignalObject, "");
  property_set_string (GbSignalData, "");

  on_property_changed (clist, clist);
}


static void
on_signal_update (GtkWidget * widget, GtkWidget * clist)
{
  gchar *signal, *handler, *object, *data;
  gboolean after;
  GList *selection = GTK_CLIST (clist)->selection;
  gint row;

  if (!selection)
    return;
  row = GPOINTER_TO_INT (selection->data);

  signal = property_get_string (GbSignalName, NULL, NULL);
  if (strlen (signal) == 0)
    {
      show_message_box (_("You need to set the signal name"));
      return;
    }
  handler = property_get_string (GbSignalHandler, NULL, NULL);
  if (strlen (handler) == 0)
    {
      show_message_box (_("You need to set the handler for the signal"));
      return;
    }
  object = property_get_string (GbSignalObject, NULL, NULL);
  after = property_get_bool (GbSignalAfter, NULL, NULL);
  data = property_get_string (GbSignalData, NULL, NULL);

  gtk_clist_set_text (GTK_CLIST (clist), row, SIGNAL_NAME_COL, signal);
  gtk_clist_set_text (GTK_CLIST (clist), row, SIGNAL_HANDLER_COL, handler);
  gtk_clist_set_text (GTK_CLIST (clist), row, SIGNAL_OBJECT_COL, object);
  gtk_clist_set_text (GTK_CLIST (clist), row, SIGNAL_AFTER_COL,
		      after ? "Y" : "");
  gtk_clist_set_text (GTK_CLIST (clist), row, SIGNAL_DATA_COL, data);

  on_property_changed (clist, clist);
}


static void
on_signal_delete (GtkWidget * widget, GtkWidget * clist)
{
  GList *selection = GTK_CLIST (clist)->selection;
  gint row;

  if (!selection)
    return;
  row = GPOINTER_TO_INT (selection->data);
  gtk_clist_remove (GTK_CLIST (clist), row);
  /* clear the key & signal fields */
  property_set_string (GbSignalName, "");
  property_set_string (GbSignalHandler, "");
  property_set_string (GbSignalObject, "");
  property_set_string (GbSignalData, "");

  on_property_changed (clist, clist);
}


static void
on_signal_clear (GtkWidget * widget, GtkWidget * clist)
{
  property_set_string (GbSignalName, "");
  property_set_string (GbSignalHandler, "");
  property_set_string (GbSignalObject, "");
  property_set_string (GbSignalData, "");
}


static void
on_signal_select (GtkWidget * clist,
		  gint row,
		  gint column,
		  GdkEventButton * event,
		  gpointer user_data)
{
  gchar *signal, *handler, *object, *after, *data;
  gtk_clist_get_text (GTK_CLIST (clist), row, SIGNAL_NAME_COL, &signal);
  gtk_clist_get_text (GTK_CLIST (clist), row, SIGNAL_HANDLER_COL, &handler);
  gtk_clist_get_text (GTK_CLIST (clist), row, SIGNAL_OBJECT_COL, &object);
  gtk_clist_get_text (GTK_CLIST (clist), row, SIGNAL_AFTER_COL, &after);
  gtk_clist_get_text (GTK_CLIST (clist), row, SIGNAL_DATA_COL, &data);
  property_set_string (GbSignalName, signal);
  property_set_string (GbSignalHandler, handler);
  property_set_string (GbSignalObject, object);
  if (!strcmp (after, "Y"))
    property_set_bool (GbSignalAfter, TRUE);
  else
    property_set_bool (GbSignalAfter, FALSE);
  property_set_string (GbSignalData, data);
}


void
property_clear_signals ()
{
  gtk_clist_clear (GTK_CLIST (signal_clist));
  /* clear the fields */
  property_set_string (GbSignalName, "");
  property_set_string (GbSignalHandler, "");
  property_set_string (GbSignalObject, "");
  property_set_string (GbSignalData, "");
}


void
property_add_signal (GbSignal * signal)
{
  gchar *row[5];

  row[SIGNAL_NAME_COL] = signal->name ? signal->name : "";
  row[SIGNAL_HANDLER_COL] = signal->handler ? signal->handler : "";
  row[SIGNAL_OBJECT_COL] = signal->object ? signal->object : "";
  row[SIGNAL_AFTER_COL] = signal->after ? "Y" : "";
  row[SIGNAL_DATA_COL] = signal->data ? signal->data : "";
  gtk_clist_append (GTK_CLIST (signal_clist), row);
}


gboolean
property_is_signal_clist (GtkWidget * widget)
{
  return (widget == signal_clist) ? TRUE : FALSE;
}


GList *
property_get_signals ()
{
  gint row;
  GList *signals = NULL;
  GbSignal *signal;
  gchar *name, *handler, *object, *after, *data;

  for (row = 0; row < GTK_CLIST (signal_clist)->rows; row++)
    {
      signal = g_new (GbSignal, 1);

      gtk_clist_get_text (GTK_CLIST (signal_clist), row,
			  SIGNAL_NAME_COL, &name);
      gtk_clist_get_text (GTK_CLIST (signal_clist), row,
			  SIGNAL_HANDLER_COL, &handler);
      gtk_clist_get_text (GTK_CLIST (signal_clist), row,
			  SIGNAL_OBJECT_COL, &object);
      gtk_clist_get_text (GTK_CLIST (signal_clist), row,
			  SIGNAL_AFTER_COL, &after);
      gtk_clist_get_text (GTK_CLIST (signal_clist), row,
			  SIGNAL_DATA_COL, &data);

      signal->name = strlen (name) > 0 ? g_strdup (name) : NULL;
      signal->handler = strlen (handler) > 0 ? g_strdup (handler) : NULL;
      signal->object = strlen (object) > 0 ? g_strdup (object) : NULL;
      signal->after = !strcmp (after, "Y") ? TRUE : FALSE;
      signal->data = strlen (data) > 0 ? g_strdup (data) : NULL;
      signals = g_list_append (signals, signal);
    }
  return signals;
}


/*
 * The Signals dialog box for selecting a signal to handle.
 */

static void
show_signals_dialog (GtkWidget * widget, gpointer value)
{
  GtkDialog *dialog;
  GtkWidget *clist, *button;
  gint row, i;
  gchar *current_signal, *name;
  gchar *titles[1];
  guint *signals;
  guint nsignals;
  GtkSignalQuery *query_info;
  guint type;
  gpointer class;
  GdkColor *inactive_fg, *inactive_bg;

  if (!property_widget)
    return;

  dialog = GTK_DIALOG (gtk_dialog_new ());
  gtk_window_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);

  gtk_signal_connect_object (GTK_OBJECT (dialog), "delete_event",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));

  gtk_window_set_title (GTK_WINDOW (dialog), _("Select Signal"));

  titles[0] = _("Signals");
  clist = gtk_clist_new_with_titles (1, titles);
  gtk_clist_column_titles_passive (GTK_CLIST (clist));
  gtk_clist_set_policy (GTK_CLIST (clist), GTK_POLICY_AUTOMATIC,
			GTK_POLICY_AUTOMATIC);
  gtk_widget_set_usize (clist, 240, 260);
  gtk_widget_show (clist);
  gtk_box_pack_start (GTK_BOX (dialog->vbox), clist, TRUE, TRUE, 0);

  /* Insert events & descriptions */
  gtk_clist_freeze (GTK_CLIST (clist));

  current_signal = gtk_entry_get_text (GTK_ENTRY (value));
  type = GTK_OBJECT_TYPE (property_widget);
  row = 0;
  inactive_fg = &widget->style->fg[GTK_STATE_INSENSITIVE];
  inactive_bg = &widget->style->bg[GTK_STATE_INSENSITIVE];
  while (type)
    {
      class = gtk_type_class (type);
      signals = GTK_OBJECT_CLASS (class)->signals;
      nsignals = GTK_OBJECT_CLASS (class)->nsignals;

      name = gtk_type_name (type);
      gtk_clist_append (GTK_CLIST (clist), &name);
      gtk_clist_set_foreground (GTK_CLIST (clist), row, inactive_fg);
      gtk_clist_set_background (GTK_CLIST (clist), row, inactive_bg);
      gtk_clist_set_row_data (GTK_CLIST (clist), row, "ClassName");
      row++;

      for (i = 0; i < nsignals; i++)
	{
	  query_info = gtk_signal_query (signals[i]);
	  name = g_strdup (query_info->signal_name);
	  g_strdelimit (name, NULL, '_');
	  gtk_clist_append (GTK_CLIST (clist), &name);
	  gtk_clist_set_shift (GTK_CLIST (clist), row, 0, 0, 10);
	  if (!strcmp (current_signal, name))
	    {
	      gtk_clist_select_row (GTK_CLIST (clist), row, 0);
	    }
	  row++;
	  g_free (name);
	  g_free (query_info);
	}
      type = gtk_type_parent (type);
    }

  gtk_clist_thaw (GTK_CLIST (clist));
  gtk_signal_connect (GTK_OBJECT (clist), "select_row",
		      GTK_SIGNAL_FUNC (on_signals_clist_select), clist);

  button = gtk_button_new_with_label (_("OK"));
  gtk_box_pack_start (GTK_BOX (dialog->action_area), button, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);
  /* Save pointer to value to use when OK pressed */
  gtk_object_set_data (GTK_OBJECT (clist), GbValueWidgetKey, value);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_signals_dialog_ok), clist);

  button = gtk_button_new_with_label (_("Cancel"));
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_show (button);
  gtk_box_pack_start (GTK_BOX (dialog->action_area), button, TRUE, TRUE, 0);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));
  gtk_widget_show (GTK_WIDGET (dialog));
}


static void
on_signals_dialog_ok (GtkWidget * widget, GtkWidget * clist)
{
  gint row, page;
  GtkWidget *value, *handler;
  GList *selection = GTK_CLIST (clist)->selection;
  gchar *name, *handler_text;

  if (selection)
    {
      row = GPOINTER_TO_INT (selection->data);
      value = gtk_object_get_data (GTK_OBJECT (clist), GbValueWidgetKey);
      g_return_if_fail (value != NULL);
      gtk_clist_get_text (GTK_CLIST (clist), row, 0, &name);
      gtk_entry_set_text (GTK_ENTRY (value), name);

      /* If we're on the Signals page, and the current handler is empty,
         insert an initial value of "on_<widget name>_<signal name>" */
      page = gtk_notebook_current_page (GTK_NOTEBOOK (main_notebook));
      if (page == GB_PAGE_SIGNALS)
	{
	  handler = (GtkWidget *) g_hash_table_lookup (gb_property_values,
						       GbSignalHandler);
	  g_return_if_fail (handler != NULL);
	  handler_text = gtk_entry_get_text (GTK_ENTRY (handler));

	  if (strlen (handler_text) == 0)
	    {
	      gchar buf[128];
	      gchar *widget_name;
	      widget_name = gtk_widget_get_name (property_widget);
	      if (strlen (widget_name) + strlen (name) + 5 < 128)
		{
		  sprintf (buf, "on_%s_%s", widget_name, name);
		  gtk_entry_set_text (GTK_ENTRY (handler), buf);
		  gtk_entry_select_region (GTK_ENTRY (handler), 0, -1);
		}
	      gtk_widget_grab_focus (handler);
	    }
	}
    }
  gtk_widget_destroy (gtk_widget_get_toplevel (clist));
}


static void
on_signals_clist_select (GtkWidget * widget, gint row, gint column,
			 GdkEventButton * bevent, gpointer data)
{
  /* Don't allow selection of widget class names */
  if (gtk_clist_get_row_data (GTK_CLIST (widget), row))
    gtk_clist_unselect_row (GTK_CLIST (widget), row, 0);
  if (bevent && bevent->type == GDK_2BUTTON_PRESS)
    {
      on_signals_dialog_ok (widget, widget);
    }
}


/*
   void
   on_apply(GtkWidget *widget, gpointer data)
   {
   if (!property_widget) return;
   gb_widget_apply_properties(property_widget, NULL);
   }
 */

void
property_set_auto_apply (gboolean value)
{
  auto_apply = value;
}


/* This is just used for debugging */
#ifdef GLD_DEBUG
static void
find_hash_value (gchar * key, gpointer data, gpointer property)
{
  if (data == property)
    MSG1 ("  applying property: %s", key);
}
#endif

static void
on_property_changed (GtkWidget * widget, GtkWidget * property)
{
  MSG ("In on_property_changed");
  if (property_widget && auto_apply)
    {
#ifdef GLD_DEBUG
      g_hash_table_foreach (gb_property_values, (GHFunc) find_hash_value,
			    property);
#endif
      gb_widget_apply_properties (property_widget, property);
    }
}


/*
 * Adjustments - handles adding/showing/applying of all 6 properties
 */
void
property_add_adjustment (gchar * Values[], gint label_type)
{
  gchar *default_labels[] =
  {N_("Value:"), N_("Min:"), N_("Max:"), N_("Step Inc:"),
   N_("Page Inc:"), N_("Page Size:")};
  gchar *horz_labels[] =
  {N_("H Value:"), N_("H Min:"), N_("H Max:"), N_("H Step Inc:"),
   N_("H Page Inc:"), N_("H Page Size:")};
  gchar *vert_labels[] =
  {N_("V Value:"), N_("V Min:"), N_("V Max:"), N_("V Step Inc:"),
   N_("V Page Inc:"), N_("V Page Size:")};
  gchar *tips[] =
  {
    N_("The initial value"),
    N_("The minimum value"),
    N_("The maximum value"),
    N_("The step increment"),
    N_("The page increment"),
    N_("The page size"),
  };
  gchar **labels;
  gint i;

  if (label_type == GB_ADJUST_H_LABELS)
    labels = horz_labels;
  else if (label_type == GB_ADJUST_V_LABELS)
    labels = vert_labels;
  else
    labels = default_labels;

  for (i = 0; i < 6; i++)
    property_add_float (Values[i], gettext (labels[i]), gettext (tips[i]));
}


/*
 * The Font dialog
 */

static void
show_font_dialog (GtkWidget * widget, gpointer value)
{
  gchar *current_xlfd_fontname;

  /* Create the dialog if it doesn't exist yet */
  if (!fontsel)
    {
      fontsel = GTK_FONT_SELECTION_DIALOG (gtk_font_selection_dialog_new (NULL));

      /* The OK/Apply/Cancel button */
      gtk_signal_connect (GTK_OBJECT (fontsel), "delete_event",
			  GTK_SIGNAL_FUNC (close_dialog_event), fontsel);
      gtk_signal_connect (GTK_OBJECT (fontsel->ok_button), "clicked",
			  GTK_SIGNAL_FUNC (on_font_dialog_ok), fontsel);
      gtk_widget_show (fontsel->apply_button);
      gtk_signal_connect (GTK_OBJECT (fontsel->apply_button), "clicked",
			  GTK_SIGNAL_FUNC (on_font_dialog_apply), fontsel);
      gtk_signal_connect (GTK_OBJECT (fontsel->cancel_button), "clicked",
			  GTK_SIGNAL_FUNC (close_dialog), fontsel);
    }

  /* Select font according to current setting */
  current_xlfd_fontname = gtk_object_get_data (GTK_OBJECT (value), GbFontSpecKey);
  if (strlen (current_xlfd_fontname) == 0)
    current_xlfd_fontname = GB_DEFAULT_XLFD_FONTNAME;

  gtk_font_selection_dialog_set_font_name (fontsel, current_xlfd_fontname);

  /* Save pointer to value to use when OK/Apply pressed */
  gtk_object_set_data (GTK_OBJECT (fontsel), GbValueWidgetKey, value);

  gtk_widget_show (GTK_WIDGET (fontsel));
  /* This maps the window, which also de-iconifies it according to ICCCM. */
  gdk_window_show (GTK_WIDGET (fontsel)->window);
  gdk_window_raise (GTK_WIDGET (fontsel)->window);
}


/* FIXME: This isn't used at present */
#if 0
static gint
get_font_size_from_spec (gchar * spec)
{
  gint i, size = -1;

  for (i = 0; i < 8; i++)
    {
      spec = strchr (spec, '-');
      if (spec == NULL)
	return -1;
      spec++;
    }
  sscanf (spec, "%i", &size);
  return size;
}
#endif

/* Note: this only works with standard X font specs, e.g.
   -adobe-courier-bold-i-normal--0-0-75-75-m-0-iso8859-1
   It copies the first two fields, changing '-' to ' ' and capitalising the
   first characters of words - after a '-', ' ' or '&'.
   Note: returns pointer to static buffer, so copy it if you want to keep it */
static gchar *
get_font_name_from_spec (gchar * spec)
{
  static gchar buf[128];
  gint i, dashes_found = 0;
  gboolean word_start = TRUE;
  gchar ch;

  if (spec == NULL)
    return "";

  for (i = 0; i < 127; i++)
    {
      ch = spec[i + 1];
      if (ch == '\0')
	break;
      if (ch == '-')
	{
	  dashes_found++;
	  if (dashes_found == 2)
	    break;
	  ch = ' ';
	}
      if (word_start)
	ch = toupper (ch);
      word_start = (ch == ' ' || ch == '&');
      buf[i] = ch;
    }
  buf[i] = '\0';
  return buf;
}


static void
on_font_dialog_apply (GtkWidget * widget, GtkFontSelectionDialog * fontsel)
{
  GtkWidget *value;
  GdkFont *font, *old_font;
  gchar *xlfd_fontname, *old_xlfd_fontname;

  value = gtk_object_get_data (GTK_OBJECT (fontsel), GbValueWidgetKey);
  g_return_if_fail (value != NULL);

  /* Try to create the font, if the font spec has changed */
  xlfd_fontname = gtk_font_selection_dialog_get_font_name (fontsel);
  font = gtk_font_selection_dialog_get_font (fontsel);
  old_xlfd_fontname = gtk_object_get_data (GTK_OBJECT (value), GbFontSpecKey);

  if (!old_xlfd_fontname
      || (xlfd_fontname && strcmp (xlfd_fontname, old_xlfd_fontname)))
    {
      if (font == NULL)
	{
	  show_message_box (_("The requested font is not available."));
	  return;
	}

      old_font = gtk_object_get_data (GTK_OBJECT (value), GbFontKey);
      if (old_font)
	gdk_font_unref (old_font);
      gdk_font_ref (font);
      gtk_object_set_data (GTK_OBJECT (value), GbFontKey, font);
      g_free (old_xlfd_fontname);
      gtk_object_set_data (GTK_OBJECT (value), GbFontSpecKey, g_strdup (xlfd_fontname));
      gtk_entry_set_text (GTK_ENTRY (value), get_font_name_from_spec (xlfd_fontname));
    }
}


static void
on_font_dialog_ok (GtkWidget * widget, GtkFontSelectionDialog * fontsel)
{
  on_font_dialog_apply (widget, fontsel);
  close_dialog (widget, GTK_WIDGET (fontsel));
}


/*
 * The Style dialog, for selecting a style to use/copy for a widget.
 */

static void
show_style_dialog (GtkWidget * widget, gpointer value)
{
  GtkWidget *dialog;
  GtkWidget *hbox, *clist, *vbbox, *button;
  int i;
  gchar *current_style;
  gchar *titles[1];
  gchar *text;
  gchar *row[1];

  dialog = gtk_window_new (GTK_WINDOW_DIALOG);
  gtk_window_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
  gtk_container_border_width (GTK_CONTAINER (dialog), 3);

  gtk_signal_connect_object (GTK_OBJECT (dialog), "delete_event",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));

  gtk_window_set_title (GTK_WINDOW (dialog), _("Select Named Style"));

  hbox = gtk_hbox_new (FALSE, 5);
  gtk_container_add (GTK_CONTAINER (dialog), hbox);
  gtk_widget_show (hbox);

  titles[0] = _("Styles");
  clist = gtk_clist_new_with_titles (1, titles);
  gtk_clist_column_titles_passive (GTK_CLIST (clist));
  gtk_clist_set_policy (GTK_CLIST (clist), GTK_POLICY_AUTOMATIC,
			GTK_POLICY_AUTOMATIC);
  gtk_widget_set_usize (clist, 200, 200);
  gtk_widget_show (clist);
  gtk_box_pack_start (GTK_BOX (hbox), clist, TRUE, TRUE, 0);

  /* Insert styles */
  gtk_clist_freeze (GTK_CLIST (clist));

  /* Add unnamed style first */
  row[0] = GB_STYLE_UNNAMED;
  gtk_clist_append (GTK_CLIST (clist), row);
  g_hash_table_foreach (gb_style_hash, (GHFunc) add_style_to_clist, clist);

  current_style = gtk_entry_get_text (GTK_ENTRY (value));
  if (strlen (current_style) == 0)
    {
      gtk_clist_select_row (GTK_CLIST (clist), 0, 0);
    }
  else
    {
      for (i = 1; i < GTK_CLIST (clist)->rows; i++)
	{
	  gtk_clist_get_text (GTK_CLIST (clist), i, 0, &text);
	  if (!strcmp (current_style, text))
	    {
	      gtk_clist_select_row (GTK_CLIST (clist), i, 0);
	    }
	}
    }

  gtk_clist_thaw (GTK_CLIST (clist));
  gtk_signal_connect (GTK_OBJECT (clist), "select_row",
		      GTK_SIGNAL_FUNC (on_style_clist_select), clist);

  /* Save pointer to value to use when a button is pressed */
  gtk_object_set_data (GTK_OBJECT (clist), GbValueWidgetKey, value);

  /* Create all the buttons */
  vbbox = gtk_vbutton_box_new ();
  gtk_button_box_set_layout (GTK_BUTTON_BOX (vbbox), GTK_BUTTONBOX_START);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (vbbox), 2);
  gtk_widget_show (vbbox);
  gtk_box_pack_start (GTK_BOX (hbox), vbbox, FALSE, TRUE, 0);

  button = gtk_button_new_with_label (_("New"));
  gtk_box_pack_start (GTK_BOX (vbbox), button, FALSE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_style_dialog_new), clist);

  button = gtk_button_new_with_label (_("Rename"));
  gtk_box_pack_start (GTK_BOX (vbbox), button, FALSE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_style_dialog_rename), clist);
  gtk_object_set_data (GTK_OBJECT (clist), "rename_button", button);
  gtk_widget_set_sensitive (button, FALSE);


  button = gtk_button_new_with_label (_("Delete"));
  gtk_box_pack_start (GTK_BOX (vbbox), button, FALSE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_style_dialog_delete), clist);
  gtk_object_set_data (GTK_OBJECT (clist), "delete_button", button);
  gtk_widget_set_sensitive (button, FALSE);

  button = gtk_button_new_with_label (_("Copy"));
  gtk_box_pack_start (GTK_BOX (vbbox), button, FALSE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_style_dialog_copy), clist);
  gtk_object_set_data (GTK_OBJECT (clist), "copy_button", button);
  gtk_widget_set_sensitive (button, FALSE);

  button = gtk_button_new_with_label (_("Cancel"));
  gtk_box_pack_start (GTK_BOX (vbbox), button, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_show (button);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));
  gtk_widget_show (GTK_WIDGET (dialog));

  button = gtk_button_new_with_label (_("OK"));
  gtk_box_pack_start (GTK_BOX (vbbox), button, FALSE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_style_dialog_ok), clist);

}


static gint
add_style_to_clist (gchar * key, gpointer data, GtkWidget * clist)
{
  gint i;
  gchar *text, *row[1];
  row[0] = key;

  /* Leave unnamed style at top */
  for (i = 1; i < GTK_CLIST (clist)->rows; i++)
    {
      gtk_clist_get_text (GTK_CLIST (clist), i, 0, &text);
      if (strcmp (key, text) < 0)
	{
	  gtk_clist_insert (GTK_CLIST (clist), i, row);
	  return i;
	}
    }
  return gtk_clist_append (GTK_CLIST (clist), row);
}


static void
on_style_clist_select (GtkWidget * widget, gint row, gint column,
		       GdkEventButton * bevent, gpointer data)
{
  gchar *text = NULL;
  GtkWidget *copy_button, *rename_button, *delete_button;
  gboolean copy_sensitive = TRUE;
  gboolean rename_sensitive = TRUE;
  gboolean delete_sensitive = TRUE;

  if (bevent && bevent->type == GDK_2BUTTON_PRESS)
    {
      on_style_dialog_ok (widget, widget);
    }

  copy_button = gtk_object_get_data (GTK_OBJECT (widget), "copy_button");
  g_return_if_fail (copy_button != NULL);
  rename_button = gtk_object_get_data (GTK_OBJECT (widget), "rename_button");
  g_return_if_fail (rename_button != NULL);
  delete_button = gtk_object_get_data (GTK_OBJECT (widget), "delete_button");
  g_return_if_fail (delete_button != NULL);

  /* If unnamed style selected, make copy, rename & delete buttons insensitive,
     else if default style selected, make rename & delete insensitive. */
  gtk_clist_get_text (GTK_CLIST (widget), row, 0, &text);
  /* Added this check since it SEGVed once here. */
  g_return_if_fail (text != NULL);
  if (!strcmp (text, GB_STYLE_UNNAMED))
    {
      copy_sensitive = FALSE;
      rename_sensitive = FALSE;
      delete_sensitive = FALSE;
    }
  else if (!strcmp (text, GB_STYLE_DEFAULT))
    {
      rename_sensitive = FALSE;
      delete_sensitive = FALSE;
    }
  gtk_widget_set_sensitive (copy_button, copy_sensitive);
  gtk_widget_set_sensitive (rename_button, rename_sensitive);
  gtk_widget_set_sensitive (delete_button, delete_sensitive);
}


static void
on_style_dialog_new (GtkWidget * widget, GtkWidget * clist)
{
  GList *selection = GTK_CLIST (clist)->selection;
  gint row;
  gchar *text;
  GbWidgetData *wdata;
  GbStyle *base_gbstyle = NULL;

  if (selection)
    {
      row = GPOINTER_TO_INT (selection->data);
      gtk_clist_get_text (GTK_CLIST (clist), row, 0, &text);
      if (!strcmp (text, GB_STYLE_UNNAMED))
	{
	  if (property_widget)
	    {
	      wdata = gtk_object_get_data (GTK_OBJECT (property_widget),
					   GB_WIDGET_DATA_KEY);
	      if (wdata)
		base_gbstyle = wdata->gbstyle;
	    }
	}
      else
	{
	  base_gbstyle = (GbStyle *) g_hash_table_lookup (gb_style_hash, text);
	  g_return_if_fail (base_gbstyle != NULL);
	}
    }

  show_entry_dialog (_("New Style:"), NULL, clist,
		     (GbEntryDialogFunc) create_new_style, base_gbstyle);
}


static gint
create_new_style (GtkWidget * widget, gchar * name, GbStyle * base_gbstyle)
{
  GbStyle *gbstyle, *existing_gbstyle;
  gint row;

  if (strlen (name) == 0)
    {
      show_message_box (_("Invalid style name"));
      return FALSE;
    }

  /* Make sure name is unique */
  existing_gbstyle = (GbStyle *) g_hash_table_lookup (gb_style_hash, name);
  if (existing_gbstyle)
    {
      show_message_box (_("That style name is already in use"));
      return FALSE;
    }

  if (!base_gbstyle)
    base_gbstyle = gb_widget_default_gb_style;
  g_return_val_if_fail (base_gbstyle != NULL, TRUE);

  gbstyle = gb_widget_copy_gb_style (base_gbstyle);
  g_free (gbstyle->name);
  gbstyle->name = g_strdup (name);
  g_hash_table_insert (gb_style_hash, gbstyle->name, gbstyle);

  /* Add style to clist */
  row = add_style_to_clist (name, NULL, widget);
  gtk_clist_select_row (GTK_CLIST (widget), row, 0);

  return TRUE;
}


static void
on_style_dialog_copy (GtkWidget * widget, GtkWidget * clist)
{
  GbWidgetData *wdata;
  gint row, i;
  GtkWidget *value;
  GList *selection = GTK_CLIST (clist)->selection;
  gchar *text;
  GbStyle *gbstyle, *other_gbstyle;

  if (selection)
    {
      row = GPOINTER_TO_INT (selection->data);
      value = gtk_object_get_data (GTK_OBJECT (clist), GbValueWidgetKey);
      g_return_if_fail (value != NULL);

      gtk_clist_get_text (GTK_CLIST (clist), row, 0, &text);
      if (!strcmp (text, GB_STYLE_UNNAMED))
	return;
      other_gbstyle = (GbStyle *) g_hash_table_lookup (gb_style_hash, text);
      g_return_if_fail (other_gbstyle != NULL);

      if (property_widget)
	{
	  wdata = gtk_object_get_data (GTK_OBJECT (property_widget),
				       GB_WIDGET_DATA_KEY);
	  g_return_if_fail (wdata != NULL);
	  gbstyle = wdata->gbstyle;

	  /* If widget is using an unnamed GbStyle, just use the selected GbStyle.
	     else copy the selected GbStyle to the current one. */
	  if (wdata->flags & GB_STYLE_IS_UNNAMED)
	    {
	      gb_widget_set_gb_style (widget, other_gbstyle);
	    }
	  else
	    {
	      g_free (gbstyle->xlfd_fontname);
	      gbstyle->xlfd_fontname = g_strdup (other_gbstyle->xlfd_fontname);
	      for (i = 0; i < GB_NUM_STYLE_STATES; i++)
		{
		  g_free (gbstyle->bg_pixmap_filenames[i]);
		  gbstyle->bg_pixmap_filenames[i]
		    = g_strdup (other_gbstyle->bg_pixmap_filenames[i]);
		}
	      gtk_style_unref (gbstyle->style);
	      gbstyle->style = other_gbstyle->style;
	      gtk_style_ref (gbstyle->style);
	      gb_widget_update_gb_styles (gbstyle, gbstyle);
	    }

	  editor_refresh_widget (property_widget);
	  property_set_auto_apply (FALSE);
	  gb_widget_show_style (property_widget);
	  property_set_auto_apply (TRUE);
	}
    }
  gtk_widget_destroy (gtk_widget_get_toplevel (clist));
}


static void
on_style_dialog_rename (GtkWidget * widget, GtkWidget * clist)
{
  gint row;
  GList *selection = GTK_CLIST (clist)->selection;
  gchar *text;
  GbStyle *gbstyle;

  if (selection)
    {
      row = GPOINTER_TO_INT (selection->data);
      gtk_clist_get_text (GTK_CLIST (clist), row, 0, &text);
      if (!strcmp (text, GB_STYLE_UNNAMED) || (!strcmp (text, GB_STYLE_DEFAULT)))
	return;
      gbstyle = (GbStyle *) g_hash_table_lookup (gb_style_hash, text);
      g_return_if_fail (gbstyle != NULL);
      show_entry_dialog (_("Rename Style To:"), text, clist,
			 (GbEntryDialogFunc) rename_style, gbstyle);
    }
}


static gint
rename_style (GtkWidget * clist, gchar * name, GbStyle * gbstyle)
{
  GbStyle *existing_gbstyle;
  gchar *text, *old_name;
  gint i, row;

  if (strlen (name) == 0)
    {
      show_message_box (_("Invalid style name"));
      return FALSE;
    }

  /* Make sure name is unique */
  existing_gbstyle = (GbStyle *) g_hash_table_lookup (gb_style_hash, text);
  if (existing_gbstyle == gbstyle)
    return TRUE;
  if (existing_gbstyle)
    {
      show_message_box (_("That style name is already in use"));
      return FALSE;
    }

  old_name = gbstyle->name;
  gbstyle->name = g_strdup (name);

  /* Delete old entry in style hash & insert new one */
  g_hash_table_remove (gb_style_hash, old_name);
  g_hash_table_insert (gb_style_hash, gbstyle->name, gbstyle);

  /* Update name in clist */
  for (i = 0; i < GTK_CLIST (clist)->rows; i++)
    {
      gtk_clist_get_text (GTK_CLIST (clist), i, 0, &text);
      if (!strcmp (text, old_name))
	{
	  gtk_clist_remove (GTK_CLIST (clist), i);
	  break;
	}
    }
  row = add_style_to_clist (name, NULL, clist);
  gtk_clist_select_row (GTK_CLIST (clist), row, 0);

  g_free (old_name);

  return TRUE;
}


static void
on_style_dialog_delete (GtkWidget * widget, GtkWidget * clist)
{
  gint row;
  GtkWidget *value;
  GList *selection = GTK_CLIST (clist)->selection;
  gchar *text;
  GbStyle *gbstyle;
  gboolean reshow = FALSE;

  if (selection)
    {
      row = GPOINTER_TO_INT (selection->data);
      value = gtk_object_get_data (GTK_OBJECT (clist), GbValueWidgetKey);
      g_return_if_fail (value != NULL);

      gtk_clist_get_text (GTK_CLIST (clist), row, 0, &text);
      if (!strcmp (text, GB_STYLE_UNNAMED) || (!strcmp (text, GB_STYLE_DEFAULT)))
	return;

      gbstyle = (GbStyle *) g_hash_table_lookup (gb_style_hash, text);
      g_return_if_fail (gbstyle != NULL);

      gtk_clist_remove (GTK_CLIST (clist), row);

      if (property_widget && property_widget->style == gbstyle->style)
	{
	  reshow = TRUE;
	}

      /* Make all widgets which are using the style use the default instead */
      gb_widget_update_gb_styles (gbstyle, gb_widget_default_gb_style);
      gb_widget_destroy_gb_style (gbstyle, TRUE);

      if (reshow)
	{
	  property_set_auto_apply (FALSE);
	  gb_widget_show_style (property_widget);
	  property_set_auto_apply (TRUE);
	}
    }
}


static void
on_style_dialog_ok (GtkWidget * widget, GtkWidget * clist)
{
  GbWidgetData *wdata;
  gint row;
  GtkWidget *value;
  GList *selection = GTK_CLIST (clist)->selection;
  gchar *text;
  GbStyle *gbstyle;

  if (selection)
    {
      row = GPOINTER_TO_INT (selection->data);
      value = gtk_object_get_data (GTK_OBJECT (clist), GbValueWidgetKey);
      g_return_if_fail (value != NULL);

      gtk_clist_get_text (GTK_CLIST (clist), row, 0, &text);

      if (property_widget)
	{
	  wdata = gtk_object_get_data (GTK_OBJECT (property_widget),
				       GB_WIDGET_DATA_KEY);
	  g_return_if_fail (wdata != NULL);

	  /* If <none> is selected, just set the unnamed style flag, so if any
	     changes are made to the style a new GbStyle is created. */
	  if (!strcmp (text, GB_STYLE_UNNAMED))
	    {
	      if (!(wdata->flags & GB_STYLE_IS_UNNAMED))
		{
		  wdata->flags |= GB_STYLE_IS_UNNAMED;
		  property_set_auto_apply (FALSE);
		  gb_widget_show_style (property_widget);
		  property_set_auto_apply (TRUE);
		}
	    }
	  else
	    {
	      gbstyle = (GbStyle *) g_hash_table_lookup (gb_style_hash, text);
	      g_return_if_fail (gbstyle != NULL);
	      wdata->flags &= ~GB_STYLE_IS_UNNAMED;

	      gb_widget_set_gb_style (property_widget, gbstyle);
	      editor_refresh_widget (property_widget);
	      property_set_auto_apply (FALSE);
	      gb_widget_show_style (property_widget);
	      property_set_auto_apply (TRUE);
	    }
	}
    }
  gtk_widget_destroy (gtk_widget_get_toplevel (clist));
}
