/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  Based on GtkRadioButton widget from GTK,
 *  Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>

#include "gladeconfig.h"

#include "palettebutton.h"

#define DEFAULT_LEFT_POS  4
#define DEFAULT_TOP_POS   4
#define DEFAULT_SPACING   7

static void gtk_palette_button_class_init (GtkPaletteButtonClass * klass);
static void gtk_palette_button_init (GtkPaletteButton * palette_button);
static void gtk_palette_button_paint (GtkWidget * widget,
				      GdkRectangle * area);
static void gtk_palette_button_draw (GtkWidget * widget,
				     GdkRectangle * area);
static void gtk_palette_button_draw_focus (GtkWidget * widget);


static GtkCheckButtonClass *parent_class = NULL;


guint
gtk_palette_button_get_type (void)
{
  static guint palette_button_type = 0;

  if (!palette_button_type)
    {
      GtkTypeInfo palette_button_info =
      {
	"GtkPaletteButton",
	sizeof (GtkPaletteButton),
	sizeof (GtkPaletteButtonClass),
	(GtkClassInitFunc) gtk_palette_button_class_init,
	(GtkObjectInitFunc) gtk_palette_button_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };

      palette_button_type = gtk_type_unique (gtk_radio_button_get_type (),
					     &palette_button_info);
    }

  return palette_button_type;
}

static void
gtk_palette_button_class_init (GtkPaletteButtonClass * class)
{
  GtkWidgetClass *widget_class;

  widget_class = (GtkWidgetClass *) class;

  parent_class = gtk_type_class (gtk_radio_button_get_type ());

  widget_class->draw = gtk_palette_button_draw;
  widget_class->draw_focus = gtk_palette_button_draw_focus;
}

static void
gtk_palette_button_init (GtkPaletteButton * palette_button)
{
  GTK_RADIO_BUTTON (palette_button)->group = g_slist_prepend (NULL,
							    palette_button);
}

GtkWidget *
gtk_palette_button_new (GSList * group)
{
  GtkPaletteButton *palette_button;

  palette_button = gtk_type_new (gtk_palette_button_get_type ());

  gtk_radio_button_set_group (GTK_RADIO_BUTTON (palette_button), group);

  return GTK_WIDGET (palette_button);
}

GtkWidget *
gtk_palette_button_new_with_label (GSList * group,
				   const gchar * label)
{
  GtkWidget *palette_button;
  GtkWidget *label_widget;

  palette_button = gtk_palette_button_new (group);
  label_widget = gtk_label_new (label);
  gtk_misc_set_alignment (GTK_MISC (label_widget), 0.0, 0.5);

  gtk_container_add (GTK_CONTAINER (palette_button), label_widget);
  gtk_widget_show (label_widget);

  return palette_button;
}

/*
 * +------------------------------------------------+
 * |                   BORDER                       |
 * |  +------------------------------------------+  |
 * |  |\\\\\\\\\\\\\\\\DEFAULT\\\\\\\\\\\\\\\\\  |  |
 * |  |\\+------------------------------------+  |  |
 * |  |\\| |           SPACING       3      | |  |  |
 * |  |\\| +--------------------------------+ |  |  |
 * |  |\\| |########## FOCUS ###############| |  |  |
 * |  |\\| |#+----------------------------+#| |  |  |
 * |  |\\| |#|         RELIEF            \|#| |  |  |
 * |  |\\| |#|  +-----------------------+\|#| |  |  |
 * |  |\\|1|#|  +     THE TEXT          +\|#|2|  |  |
 * |  |\\| |#|  +-----------------------+\|#| |  |  |
 * |  |\\| |#| \\\\\ ythickness \\\\\\\\\\|#| |  |  |
 * |  |\\| |#+----------------------------+#| |  |  |
 * |  |\\| |########### 1 ##################| |  |  |
 * |  |\\| +--------------------------------+ |  |  |
 * |  |\\| |        default spacing   4     | |  |  |
 * |  |\\+------------------------------------+  |  |
 * |  |\            ythickness                   |  |
 * |  +------------------------------------------+  |
 * |                border_width                    |
 * +------------------------------------------------+
 */

static void
gtk_palette_button_paint (GtkWidget * widget,
			  GdkRectangle * area)
{
  GdkRectangle restrict_area;
  GdkRectangle outer_area;
  GdkRectangle tmp_area;
  GdkRectangle new_area;
  gint xthickness;
  gint ythickness;
  guint8 state;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_BUTTON (widget));

  xthickness = widget->style->klass->xthickness;
  ythickness = widget->style->klass->ythickness;

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      restrict_area.x = xthickness;
      restrict_area.y = ythickness;
      restrict_area.width = GTK_WIDGET (widget)->allocation.width -
	restrict_area.x * 2 - GTK_CONTAINER (widget)->border_width * 2;
      restrict_area.height = GTK_WIDGET (widget)->allocation.height -
	restrict_area.y * 2 - GTK_CONTAINER (widget)->border_width * 2;

      outer_area = restrict_area;

      if (GTK_WIDGET_CAN_DEFAULT (widget))
	{
	  restrict_area.x += DEFAULT_LEFT_POS;
	  restrict_area.y += DEFAULT_TOP_POS;
	  restrict_area.width -= DEFAULT_SPACING;
	  restrict_area.height -= DEFAULT_SPACING;
	}

      if (gdk_rectangle_intersect (area, &restrict_area, &new_area))
	{
	  /* Damon - we don't want the background to prelight. */
	  state = GTK_WIDGET_STATE (widget);
	  if (state == GTK_STATE_PRELIGHT)
	    state = GTK_STATE_NORMAL;
	  if (GTK_TOGGLE_BUTTON (widget)->active)
	    state = GTK_STATE_ACTIVE;

	  gtk_style_set_background (widget->style, widget->window, state);
	  gdk_window_clear_area (widget->window,
				 new_area.x, new_area.y,
				 new_area.width, new_area.height);
	}

      if (GTK_WIDGET_CAN_DEFAULT (widget))
	{
	  /* Now fill spacing area between the default border and the button */

	  /* 1 */ tmp_area = outer_area;
	  tmp_area.width = restrict_area.x - outer_area.x;
	  if (gdk_rectangle_intersect (area, &tmp_area, &new_area))
	    gdk_draw_rectangle (widget->window,
				widget->style->bg_gc[GTK_STATE_NORMAL],
				TRUE,
				new_area.x, new_area.y,
				new_area.width, new_area.height);

	  /* 2 */ tmp_area.x = restrict_area.x + restrict_area.width;

	  if (gdk_rectangle_intersect (area, &tmp_area, &new_area))
	    gdk_draw_rectangle (widget->window,
				widget->style->bg_gc[GTK_STATE_NORMAL],
				TRUE,
				new_area.x, new_area.y,
				new_area.width, new_area.height);

	  /* 3 */ tmp_area.width = restrict_area.width;
	  tmp_area.height = restrict_area.y - outer_area.y;
	  tmp_area.x = restrict_area.x;

	  if (gdk_rectangle_intersect (area, &tmp_area, &new_area))
	    gdk_draw_rectangle (widget->window,
				widget->style->bg_gc[GTK_STATE_NORMAL],
				TRUE,
				new_area.x, new_area.y,
				new_area.width, new_area.height);

	  /* 4 */ tmp_area.y = restrict_area.y + restrict_area.height;

	  if (gdk_rectangle_intersect (area, &tmp_area, &new_area))
	    gdk_draw_rectangle (widget->window,
				widget->style->bg_gc[GTK_STATE_NORMAL],
				TRUE,
				new_area.x, new_area.y,
				new_area.width, new_area.height);
	}
    }
}

static void
gtk_palette_button_draw (GtkWidget * widget,
			 GdkRectangle * area)
{
  GtkButton *button;
  GdkRectangle child_area;
  GdkRectangle tmp_area;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_PALETTE_BUTTON (widget));
  g_return_if_fail (area != NULL);

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      button = GTK_BUTTON (widget);

      tmp_area = *area;
      tmp_area.x -= GTK_CONTAINER (button)->border_width;
      tmp_area.y -= GTK_CONTAINER (button)->border_width;

      gtk_palette_button_paint (widget, &tmp_area);

      if (button->child && gtk_widget_intersect (button->child, &tmp_area, &child_area))
	gtk_widget_draw (button->child, &child_area);

      gtk_widget_draw_default (widget);
      gtk_widget_draw_focus (widget);
    }
}

static void
gtk_palette_button_draw_focus (GtkWidget * widget)
{
  GtkButton *button;
  GtkShadowType shadow_type;
  gint width, height;
  gint x, y;
  GdkGC *bg_gc;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_BUTTON (widget));

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      button = GTK_BUTTON (widget);
      if (GTK_TOGGLE_BUTTON (widget)->active)
	bg_gc = widget->style->bg_gc[GTK_STATE_ACTIVE];
      else if (GTK_WIDGET_STATE (widget) == GTK_STATE_PRELIGHT)
	bg_gc = widget->style->bg_gc[GTK_STATE_NORMAL];
      else
	bg_gc = widget->style->bg_gc[GTK_WIDGET_STATE (widget)];

      x = 0;
      y = 0;
      width = widget->allocation.width - GTK_CONTAINER (widget)->border_width
	* 2;
      height = widget->allocation.height - GTK_CONTAINER (widget)->border_width
	* 2;

      if (GTK_WIDGET_CAN_DEFAULT (widget))
	{
	  x += widget->style->klass->xthickness;
	  y += widget->style->klass->ythickness;
	  width -= 2 * x + DEFAULT_SPACING;
	  height -= 2 * y + DEFAULT_SPACING;
	  x += DEFAULT_LEFT_POS;
	  y += DEFAULT_TOP_POS;
	}

      if (GTK_WIDGET_HAS_FOCUS (widget)
	  && !GTK_TOGGLE_BUTTON (widget)->active
	  && GTK_WIDGET_STATE (widget) != GTK_STATE_ACTIVE)
	{
	  x += 1;
	  y += 1;
	  width -= 2;
	  height -= 2;
	}
      else
	{
	  if (GTK_WIDGET_STATE (widget) == GTK_STATE_ACTIVE)
	    gdk_draw_rectangle (widget->window,
				bg_gc, FALSE,
				x + 1, y + 1, width - 4, height - 4);
	  else
	    gdk_draw_rectangle (widget->window,
				bg_gc, FALSE,
				x + 2, y + 2, width - 5, height - 5);
	}

      if (GTK_TOGGLE_BUTTON (widget)->active
	  || GTK_WIDGET_STATE (widget) == GTK_STATE_ACTIVE)
	shadow_type = GTK_SHADOW_IN;
      else
	shadow_type = GTK_SHADOW_OUT;

      if (GTK_TOGGLE_BUTTON (widget)->active
	  || GTK_WIDGET_STATE (widget) == GTK_STATE_ACTIVE)
	gtk_draw_shadow (widget->style, widget->window,
			 GTK_STATE_ACTIVE, shadow_type,
			 x, y, width, height);
      else if (GTK_WIDGET_STATE (widget) == GTK_STATE_PRELIGHT)
	gtk_draw_shadow (widget->style, widget->window,
			 GTK_WIDGET_STATE (widget), shadow_type,
			 x, y, width, height);
      else
	gdk_draw_rectangle (widget->window,
			    bg_gc, FALSE,
			    x, y, width - 1, height - 1);

      if (GTK_WIDGET_HAS_FOCUS (widget)
	  && !GTK_TOGGLE_BUTTON (widget)->active
	  && GTK_WIDGET_STATE (widget) != GTK_STATE_ACTIVE)
	{
	  x -= 1;
	  y -= 1;
	  width += 2;
	  height += 2;

	  gdk_draw_rectangle (widget->window,
			      widget->style->black_gc, FALSE,
			      x, y, width - 1, height - 1);
	}
    }
}
