/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <locale.h>

#include <gtk/gtk.h>

#include "gladeconfig.h"

#include "project.h"
#include "gbwidget.h"
#include "load.h"
#include "save.h"

#define BUFFER_INCREMENT_SIZE	1024

#define MAX_ENTITY_LEN	16


static GbStatusCode real_load_project_file (gchar * filename);
static gchar load_next_char (GbWidgetSetArgData * data);
static void load_buffer_add (GbWidgetSetArgData * data,
			     gchar ch);
static void load_buffer_add_char (GbWidgetSetArgData * data,
				  gchar ch);

static void add_error_message (GbWidgetSetArgData * data,
			       gchar * property_name,
			       gchar * message,
			       gchar * value);



/* We need this to make sure that numbers are read in a portable syntax,
   instead of using the current locale. This code is from glibc info docs. */
GbStatusCode
load_project_file (gchar * filename)
{
  gchar *old_locale, *saved_locale;
  GbStatusCode status;
     
  old_locale = setlocale (LC_NUMERIC, NULL);
  saved_locale = g_strdup (old_locale);
  setlocale (LC_NUMERIC, "C");
  status = real_load_project_file (filename);
  setlocale (LC_NUMERIC, saved_locale);
  g_free (saved_locale);
  return status;
}


/* FIXME: handle XML comments - could be a problem if in the middle of CDATA */

static GbStatusCode
real_load_project_file (gchar * filename)
{
  GbWidgetSetArgData data;
  gboolean toplevel = TRUE;
  GList *element;

  data.status = GB_OK;
  data.fp = fopen (filename, "r");
  if (data.fp == NULL)
    return GB_FILE_OPEN_ERROR;

  data.filename = filename;
  data.line_number = 1;
  data.buffer_space = BUFFER_INCREMENT_SIZE;
  data.buffer = g_new (gchar, data.buffer_space);
  data.buffer[0] = '\0';
  /* Note: we start with a buffer pos of 1 since 0 is needed to represent an
     empty value when retrieving indices from the properties hash. */
  data.buffer_pos = 1;
  data.token_type = GB_TOKEN_START_TAG;
  data.properties_space = GB_LOADED_PROPERTIES_INC;
  data.properties = g_new (GbLoadedProperty, GB_LOADED_PROPERTIES_INC);
  data.child_properties_space = GB_LOADED_PROPERTIES_INC;
  data.child_properties = g_new (GbLoadedProperty, GB_LOADED_PROPERTIES_INC);
  data.signals = NULL;
  data.accelerators = NULL;
  data.gbstyle = NULL;
  data.error_messages = NULL;

  load_token_skip_whitespace (&data);
  while (data.status == GB_OK)
    {
      toplevel = FALSE;

      if (data.token_type == GB_TOKEN_END_TAG)
	{
	  if (!strcmp (data.buffer + data.token, "gtk-interface"))
	    {
	      MSG1 ("End of Document body element: %s", data.buffer + data.token);
	      toplevel = TRUE;
	      break;
	    }
	}

      if (data.token_type != GB_TOKEN_START_TAG)
	{
	  data.status = GB_START_TAG_EXPECTED;
	  break;
	}

      if (!strcmp (data.buffer + data.token, "widget"))
	{
	  gb_widget_load (NULL, &data, NULL);
	  if (data.status != GB_OK)
	    break;
	}
      else if (!strcmp (data.buffer + data.token, "style"))
	{
	  gb_widget_load_style (&data);
	  if (data.status != GB_OK)
	    break;
	}
      else if (!strcmp (data.buffer + data.token, "gtk-interface"))
	{
	  MSG1 ("Document body element: %s", data.buffer + data.token);
	}
      else if (*(data.buffer + data.token) == '?')
	{
	  MSG1 ("Processing instruction: %s", data.buffer + data.token);
	}
      else
	{
	  data.status = GB_INVALID_ENTITY;
	  break;
	}

      load_token_skip_whitespace (&data);
      toplevel = TRUE;
    }

  g_free (data.buffer);
  g_free (data.properties);
  g_free (data.child_properties);
  fclose (data.fp);

  element = data.error_messages;
  while (element)
    {
      g_warning (element->data);
      g_free (element->data);
      element = element->next;
    }
  g_list_free (data.error_messages);

  /* If we got EOF at the top level then that is OK. */
  if (data.status == GB_EOF && toplevel)
    return GB_OK;
  return data.status;
}


/* This skips any initial data token, then reads a start tag, data, and an end
   tag. */
void
load_element (GbWidgetSetArgData * data, gint * element, gint * cdata)
{
  load_token_skip_whitespace (data);
  if (data->status != GB_OK)
    {
      MSG ("Load error");
      return;
    }
  if (data->token_type != GB_TOKEN_START_TAG)
    {
      data->status = GB_START_TAG_EXPECTED;
      MSG ("Load error");
      return;
    }
  *element = data->token;

  load_token (data);
  if (data->status != GB_OK)
    {
      MSG ("Load error");
      return;
    }
  if (data->token_type == GB_TOKEN_DATA)
    {
      *cdata = data->token;

      load_token (data);
      if (data->status != GB_OK)
	{
	  MSG ("Load error");
	  return;
	}
    }
  else
    {
      /* Empty tag - use special buffer position, 0. */
      *cdata = 0;
    }

  if (data->token_type != GB_TOKEN_END_TAG
      || strcmp (data->buffer + *element, data->buffer + data->token))
    {
      data->status = GB_END_TAG_EXPECTED;
      MSG2 ("start tag:%s end tag:%s", data->buffer + *element,
	    data->buffer + data->token);
      MSG ("Load error");
      return;
    }
  MSG2 ("Loaded element: %s, data: %s", data->buffer + *element,
	data->buffer + *cdata);
}


/* Returns one token of XML - a GB_TOKEN_START_TAG, GB_TOKEN_END_TAG or
   GB_TOKEN_DATA. Start and end tags are converted to lower case. */
void
load_token (GbWidgetSetArgData * data)
{
  gchar ch;

  data->token = data->buffer_pos;

  /* If the last token was data, it finished with a '<'. */
  if (data->token_type == GB_TOKEN_DATA)
    ch = '<';
  else
    {
      ch = load_next_char (data);
      if (data->status != GB_OK)
	return;
    }

  if (ch == '<')
    {
      ch = load_next_char (data);
      if (data->status != GB_OK)
	return;
      if (ch != '/')
	{
	  data->token_type = GB_TOKEN_START_TAG;
	  load_buffer_add (data, ch);
	  if (data->status != GB_OK)
	    return;
	}
      else
	{
	  data->token_type = GB_TOKEN_END_TAG;
	}

      for (;;)
	{
	  ch = load_next_char (data);
	  if (data->status != GB_OK)
	    return;
	  if (ch == '>')
	    {
	      load_buffer_add_char (data, '\0');
	      g_strdown (data->buffer + data->token);
	      return;
	    }
	  else
	    {
	      load_buffer_add (data, ch);
	      if (data->status != GB_OK)
		return;
	    }
	}
    }
  else
    {
      data->token_type = GB_TOKEN_DATA;
      load_buffer_add (data, ch);
      if (data->status != GB_OK)
	return;

      for (;;)
	{
	  ch = load_next_char (data);
	  if (data->status != GB_OK)
	    return;
	  if (ch == '<')
	    {
	      load_buffer_add_char (data, '\0');
	      return;
	    }
	  else
	    {
	      load_buffer_add (data, ch);
	      if (data->status != GB_OK)
		return;
	    }
	}
    }
}


/* This skips any whitespace token and returns the next token. */
void
load_token_skip_whitespace (GbWidgetSetArgData * data)
{
  gchar *pos;

  load_token (data);
  if (data->status != GB_OK || data->token_type != GB_TOKEN_DATA)
    return;

  /* Return if the CDATA is not all whitespace. */
  pos = data->buffer + data->token;
  while (*pos)
    {
      if (*pos != ' ' && *pos != '\n' && *pos != '\r' && *pos != '\t')
	return;
      pos++;
    }

  load_token (data);
}


/* This returns the next character from the file, and sets the status if
   eof or an error occurred. It also increments the line number if a '\n' is
   read. */
static gchar
load_next_char (GbWidgetSetArgData * data)
{
  gint num_read;
  gchar ch;

  num_read = fread (&ch, sizeof (gchar), 1, data->fp);
  if (num_read != 1)
    {
      if (feof (data->fp))
	data->status = GB_EOF;
      else
	data->status = GB_FILE_READ_ERROR;
    }
  else if (ch == '\n')
    {
      data->line_number++;
      MSG1 ("Line: %i", data->line_number);
    }

  return ch;
}


/* This adds the char to the buffer, converting any XML entities if necessary.
 */
static void
load_buffer_add (GbWidgetSetArgData * data, gchar ch)
{
  gchar entity[MAX_ENTITY_LEN];
  gint entity_len;
  gboolean is_entity = FALSE;

  if (ch != '&')
    {
      load_buffer_add_char (data, ch);
      return;
    }

  entity_len = 0;
  for (;;)
    {
      ch = load_next_char (data);
      if (data->status != GB_OK)
	return;
      if (ch == ';')
	{
	  is_entity = TRUE;
	  break;
	}
      entity[entity_len++] = ch;
    }

  if (!is_entity)
    {
      data->status = GB_INVALID_ENTITY;
      return;
    }

  entity[entity_len] = '\0';
  if (!g_strcasecmp (entity, "lt"))
    load_buffer_add_char (data, '<');
  else if (!g_strcasecmp (entity, "gt"))
    load_buffer_add_char (data, '>');
  else if (!g_strcasecmp (entity, "amp"))
    load_buffer_add_char (data, '&');
  else if (!g_strcasecmp (entity, "quot"))
    load_buffer_add_char (data, '"');
  else
    data->status = GB_INVALID_ENTITY;
}


/* This adds the char to the buffer, expanding the buffer if necessary. */
static void
load_buffer_add_char (GbWidgetSetArgData * data, gchar ch)
{
  if (data->buffer_pos == data->buffer_space)
    {
      MSG ("Reallocating buffer");
      data->buffer_space += BUFFER_INCREMENT_SIZE;
      data->buffer = g_realloc (data->buffer, data->buffer_space);
    }
  data->buffer[data->buffer_pos++] = ch;
}


/* This releases any data in the buffer, and is meant to be called after a
   widget has read all of it's properties from the buffer. */
void
load_buffer_release (GbWidgetSetArgData * data)
{
  /* Note: we start with a buffer pos of 1 since 0 is needed to represent an
     empty value when retrieving indices from the properties hash. */
  data->buffer_pos = 1;
}



gchar *
load_get_value (GbWidgetSetArgData * data,
		gchar * property_name)
{
  GbLoadedProperty *properties = data->properties;
  gint nproperties = data->nproperties;
  gchar *tag_name;
  gint i;

  tag_name = find_start_of_tag_name (property_name);

  if (data->loading_type == GB_CHILD_PROPERTIES)
    {
      properties = data->child_properties;
      nproperties = data->nchild_properties;
    }

  for (i = 0; i < nproperties; i++)
    {
      if (!strcmp (data->buffer + properties[i].tag_index, tag_name))
	{
	  MSG2 ("load_get_value %s: %s", property_name,
		data->buffer + properties[i].cdata_index);
	  data->apply = TRUE;
	  return data->buffer + properties[i].cdata_index;
	}
    }

  MSG1 ("load_get_value %s: (NULL)", property_name);
  data->apply = FALSE;
  return NULL;
}


static void
add_error_message (GbWidgetSetArgData * data,
		   gchar * property_name,
		   gchar * message,
		   gchar * value)
{
  gchar buffer[1024];

  sprintf (message, _("Error loading property: %s before line: %i\n  %s: %s\n"),
	   find_start_of_tag_name (property_name), data->line_number,
	   message, value);
  message = g_new (gchar, strlen (buffer) + 1);
  strcpy (message, buffer);
  data->error_messages = g_list_append (data->error_messages, message);
}


/* FIXME: Should the functions returning strings return NULL or "" ? */

gchar *
load_string (GbWidgetSetArgData * data,
	     gchar * property_name)
{
  gchar *value = load_get_value (data, property_name);
  return value ? value : "";
}


gchar *
load_text (GbWidgetSetArgData * data,
	   gchar * property_name)
{
  gchar *value = load_get_value (data, property_name);
  return value ? value : "";
}


gint
load_int (GbWidgetSetArgData * data,
	  gchar * property_name)
{
  gchar *value = load_get_value (data, property_name);
  return value ? atoi (value) : 0;
}


gfloat
load_float (GbWidgetSetArgData * data,
	    gchar * property_name)
{
  gchar *value = load_get_value (data, property_name);
  return value ? atof (value) : 0;
}


gboolean
load_bool (GbWidgetSetArgData * data,
	   gchar * property_name)
{
  gchar *value = load_get_value (data, property_name);
  return load_parse_bool (data, value);
}


gboolean
load_parse_bool (GbWidgetSetArgData * data,
		 gchar * value)
{
  if (value != NULL)
    {
      if (!strncmp (value, "True", 4))
	return TRUE;
      else if (!strncmp (value, "False", 5))
	return FALSE;
      else
	{
	  data->status = GB_INVALID_VALUE;
	  MSG1 ("===Invalid boolean property: %s", value);
	}
    }
  return FALSE;
}


gchar *
load_choice (GbWidgetSetArgData * data,
	     gchar * property_name)
{
  gchar *value = load_get_value (data, property_name);
  return value ? value : "";
}


gchar *
load_combo (GbWidgetSetArgData * data,
	    gchar * property_name)
{
  gchar *value = load_get_value (data, property_name);
  return value ? value : "";
}


guchar *
load_color (GbWidgetSetArgData * data,
	    gchar * property_name)
{
  gchar *value = load_get_value (data, property_name);
  return load_parse_color (data, value);
}


guchar *
load_parse_color (GbWidgetSetArgData * data,
		  gchar * value)
{
  static guchar color[3];
  gint matched, red, green, blue;

  if (value == NULL)
    return NULL;
  matched = sscanf (value, "%i,%i,%i", &red, &green, &blue);
  if (matched != 3)
    {
      data->status = GB_INVALID_VALUE;
      return NULL;
    }

  color[0] = red;
  color[1] = green;
  color[2] = blue;
  return color;
}


GdkPixmap *
load_bgpixmap (GbWidgetSetArgData * data,
	       gchar * property_name,
	       gchar ** filename)
{
  GdkPixmap *gdkpixmap;
  gchar *value = load_get_value (data, property_name);
  *filename = value;
  if (value)
    {
      /* FIXME: What do I do here? We have no widget. Could use the parent,
         or load the pixmap in a realize callback. */
      /*
         gdkpixmap = gdk_pixmap_create_from_xpm (data->holding_widget->window, NULL,
         &data->holding_widget->style->bg[GTK_STATE_NORMAL],
         value);
         if (!gdkpixmap)
         add_error_message(data, property_name, "Couldn't load pixmap: ", value);

         return gdkpixmap;
       */
    }

  return NULL;
}


gpointer
load_dialog (GbWidgetSetArgData * data,
	     gchar * property_name)
{
  gchar *value = load_get_value (data, property_name);
  return value ? value : "";
}


gchar *
load_filename (GbWidgetSetArgData * data,
	       gchar * property_name)
{
  gchar *value = load_get_value (data, property_name);
  return value ? value : "";
}


GdkFont *
load_font (GbWidgetSetArgData * data,
	   gchar * property_name,
	   gchar ** xlfd_fontname)
{
  GdkFont *font;
  gchar *value = load_get_value (data, property_name);
  *xlfd_fontname = value;
  if (value)
    {
      font = gdk_font_load (value);
      if (font == NULL)
	add_error_message (data, property_name, _("Couldn't load font: "), value);

      return font;
    }
  return NULL;
}
