
/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * The Menu Editor window, based on initial work by Javier Arriero Pas
 * and John Looney.
 */

#include <string.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "gladeconfig.h"

#include "glade_menu_editor.h"
#include "glade_keys_dialog.h"
#include "gbwidget.h"

/* How many pixels to indent levels of the menu hierarchy in the clist. */
#define GB_INDENT	10

/* The text to display if the item is a separator. */
#define GB_SEPARATOR_TEXT "---"

/* This sets the order of the clist columns. */
#define GB_MENUED_NUM_COLS 9

#define GLD_COL_LABEL	0
#define GLD_COL_TYPE	1
#define GLD_COL_ACCEL	2
#define GLD_COL_NAME	3
#define GLD_COL_HANDLER	4
#define GLD_COL_ICON	5
#define GLD_COL_ACTIVE	6
#define GLD_COL_GROUP	7
#define GLD_COL_JUSTIFY	8

typedef enum
  {
    GB_MENU_ITEM_NORMAL,
    GB_MENU_ITEM_CHECK,
    GB_MENU_ITEM_RADIO
  }
GbMenuItemType;

/* This holds information on one menu item. */
typedef struct _GbMenuItemData GbMenuItemData;
struct _GbMenuItemData
  {
    gchar *label;		/* Text to display. */
    gchar *name;		/* Widget name. */
    gchar *handler;		/* Handler to call when selected. */
    gchar *icon;		/* Icon filename (not used yet). */
    GbMenuItemType type;	/* Type - normal/check/radio. */
    gboolean right_justify;	/* If the item should be right-justified. */
    gboolean active;		/* If the item is initially active. */
    gchar *group_name;		/* Radio menu item group name. */
    guint8 modifiers;		/* Control/Shift/Alt flags. */
    gchar *key;			/* Name of accelerator key. */

    gint level;			/* Level in menu hierarchy. */
    gboolean generate_name;	/* If the name should be auto-generated. */
    gboolean generate_handler;	/* If the handler should be auto-generated. */
  };


static void glade_menu_editor_class_init (GladeMenuEditorClass * klass);
static void glade_menu_editor_init (GladeMenuEditor * dialog);
static void glade_menu_editor_destroy (GtkObject *object);

static gboolean on_key_press (GtkWidget * widget,
			      GdkEventKey * event,
			      gpointer user_data);
static void on_clist_select_row (GtkWidget * clist,
				 gint row,
				 gint column,
				 GdkEventButton * event,
				 gpointer user_data);
static void on_clist_unselect_row (GtkWidget * clist,
				   gint row,
				   gint column,
				   GdkEventButton * event,
				   gpointer user_data);
static void on_entry_changed (GtkWidget * entry,
			      gpointer user_data);
static gboolean on_label_entry_key_press (GtkWidget * widget,
					  GdkEventKey * event,
					  gpointer user_data);
static void on_radiobutton_toggled (GtkWidget * togglebutton,
				    gpointer user_data);
static void on_checkbutton_toggled (GtkWidget * togglebutton,
				    gpointer user_data);
static void on_right_justify_button_toggled (GtkToggleButton * togglebutton,
					     gpointer user_data);
static void on_state_button_toggled (GtkToggleButton * togglebutton,
				     gpointer user_data);
static void on_accel_key_button_clicked (GtkButton * button,
					 gpointer user_data);
static void on_up_button_clicked (GtkButton * button,
				  gpointer user_data);
static void on_down_button_clicked (GtkButton * button,
				    gpointer user_data);
static void on_left_button_clicked (GtkButton * button,
				    gpointer user_data);
static void on_right_button_clicked (GtkButton * button,
				     gpointer user_data);
static void on_add_button_clicked (GtkWidget * button,
				   gpointer user_data);
static void add_item (GladeMenuEditor * menued,
		      gboolean as_child);
static void on_delete_button_clicked (GtkWidget * widget,
				      gpointer user_data);

static void on_keys_dialog_clist_select (GtkWidget * widget,
					 gint row,
					 gint column,
					 GdkEventButton * bevent,
					 GtkWidget * clist);
static void on_keys_dialog_ok (GtkWidget * widget,
			       GtkWidget * clist);

static gint get_selected_row (GladeMenuEditor * menued);
static GbMenuItemData* get_selected_item (GladeMenuEditor * menued);
static void set_interface_state (GladeMenuEditor * menued);
static gchar *get_accel_string (gchar * key,
				guint8 modifiers);
static void insert_item (GtkCList * clist,
			 GbMenuItemData * item,
			 gint row);
static void ensure_visible (GtkWidget *clist,
			    gint row);
static void update_current_item (GladeMenuEditor * menued);
static gboolean item_property_changed (gchar *new, gchar *old);
static gchar* copy_item_property (gchar *property);
static void clear_form (GladeMenuEditor * menued,
			gboolean full);
static void update_radio_groups (GladeMenuEditor * menued);
static GList* add_radio_group (GList *groups,
			       gchar *group_name);
static void show_item_properties (GladeMenuEditor * menued);
static void insert_items (GtkWidget * clist,
			  GList * items,
			  gint row);
static GList *remove_item_and_children (GtkWidget * clist,
					gint row);
static GtkWidget* create_radio_menu_item (GtkMenuShell *menu,
					  gchar *label,
					  gchar *group_name,
					  GHashTable *group_hash);
static gchar* generate_name (gchar *label,
			     gboolean reserve_id);
static gchar* generate_handler (GladeMenuEditor *menued,
				gint row,
				gchar *label,
				gchar *name);
static void check_generated_handlers (GladeMenuEditor *menued);
static gboolean is_parent (GladeMenuEditor *menued,
			   gint row);
static void set_submenu (GladeMenuEditor *menued,
			 GtkMenuShell    *menu,
			 gint	      level);
static void glade_menu_editor_reset (GladeMenuEditor *menued);


static GtkWindowClass *parent_class = NULL;


guint
glade_menu_editor_get_type (void)
{
  static guint glade_menu_editor_type = 0;

  if (!glade_menu_editor_type)
    {
      GtkTypeInfo glade_menu_editor_info =
      {
	"GladeMenuEditor",
	sizeof (GladeMenuEditor),
	sizeof (GladeMenuEditorClass),
	(GtkClassInitFunc) glade_menu_editor_class_init,
	(GtkObjectInitFunc) glade_menu_editor_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };

      glade_menu_editor_type = gtk_type_unique (gtk_window_get_type (),
						&glade_menu_editor_info);
    }

  return glade_menu_editor_type;
}

static void
glade_menu_editor_class_init (GladeMenuEditorClass * class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  object_class = (GtkObjectClass *) class;
  widget_class = (GtkWidgetClass *) class;

  parent_class = gtk_type_class (gtk_window_get_type ());

  object_class->destroy = glade_menu_editor_destroy;
}


static void
glade_menu_editor_init (GladeMenuEditor * menued)
{

  GtkWidget *vbox2, *vbox1;
  GtkWidget *hbox1;
  GtkWidget *vbox3;
  GtkWidget *table1;
  GtkWidget *eventbox3;
  GtkWidget *eventbox2;
  GtkWidget *eventbox1;
  GtkWidget *table2;
  GSList *table2_group = NULL;
  GtkWidget *table3;
  GtkWidget *accel_key_button;
  GtkWidget *hbox2;
  GtkWidget *label9;
  GtkWidget *label8;
  GtkWidget *hbuttonbox3;
  GtkWidget *arrow1;
  GtkWidget *arrow2;
  GtkWidget *arrow3;
  GtkWidget *arrow4;
  GtkWidget *hbuttonbox2;
  GtkWidget *hseparator1;
  GtkWidget *hbuttonbox1;
  GtkTooltips *tooltips;
  gchar *titles[9];

  tooltips = gtk_tooltips_new ();

  gtk_container_border_width (GTK_CONTAINER (menued), 8);
  gtk_window_set_title (GTK_WINDOW (menued), _ ("Menu Editor"));
  gtk_window_set_policy (GTK_WINDOW (menued), FALSE, TRUE, FALSE);

  vbox2 = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox2);
  gtk_container_add (GTK_CONTAINER (menued), vbox2);

  hbox1 = gtk_hbox_new (FALSE, 6);
  gtk_widget_show (hbox1);
  gtk_box_pack_start (GTK_BOX (vbox2), hbox1, TRUE, TRUE, 0);

  vbox1 = gtk_vbox_new (FALSE, 4);
  gtk_widget_show (vbox1);
  gtk_box_pack_start (GTK_BOX (hbox1), vbox1, TRUE, TRUE, 0);

  titles[GLD_COL_LABEL]		= _("Label");
  titles[GLD_COL_TYPE]		= _("Type");
  titles[GLD_COL_ACCEL]		= _("Accelerator");
  titles[GLD_COL_NAME]		= _("Name");
  titles[GLD_COL_HANDLER]	= _("Handler");
  titles[GLD_COL_ICON]		= _("Icon");
  titles[GLD_COL_ACTIVE]	= _("Active");
  titles[GLD_COL_GROUP]		= _("Group");
  titles[GLD_COL_JUSTIFY]	= _("Right Justify");
  menued->clist = gtk_clist_new_with_titles (9, titles);
  gtk_widget_show (menued->clist);
  GTK_WIDGET_SET_FLAGS (menued->clist, GTK_CAN_FOCUS);
  gtk_signal_connect (GTK_OBJECT (menued->clist), "key_press_event",
		      GTK_SIGNAL_FUNC (on_key_press),
		      NULL);
  gtk_box_pack_start (GTK_BOX (vbox1), menued->clist, TRUE, TRUE, 0);
  gtk_widget_set_usize (menued->clist, 300, -1);
  gtk_signal_connect (GTK_OBJECT (menued->clist), "select_row",
		      GTK_SIGNAL_FUNC (on_clist_select_row), NULL);
  gtk_signal_connect (GTK_OBJECT (menued->clist), "unselect_row",
		      GTK_SIGNAL_FUNC (on_clist_unselect_row), NULL);
  gtk_clist_set_column_width (GTK_CLIST (menued->clist), GLD_COL_LABEL, 144);
  gtk_clist_set_column_width (GTK_CLIST (menued->clist), GLD_COL_TYPE, 42);
  gtk_clist_set_column_width (GTK_CLIST (menued->clist), GLD_COL_ACCEL, 120);
  gtk_clist_set_column_width (GTK_CLIST (menued->clist), GLD_COL_NAME, 100);
  gtk_clist_set_column_width (GTK_CLIST (menued->clist), GLD_COL_HANDLER, 172);
  gtk_clist_set_column_width (GTK_CLIST (menued->clist), GLD_COL_ICON, 172);
  gtk_clist_set_column_width (GTK_CLIST (menued->clist), GLD_COL_ACTIVE, 42);
  gtk_clist_set_column_width (GTK_CLIST (menued->clist), GLD_COL_GROUP, 75);
  gtk_clist_set_column_width (GTK_CLIST (menued->clist), GLD_COL_JUSTIFY, 75);
  gtk_clist_column_titles_show (GTK_CLIST (menued->clist));
  gtk_clist_set_policy (GTK_CLIST (menued->clist), GTK_POLICY_AUTOMATIC,
			GTK_POLICY_ALWAYS);
  gtk_clist_column_titles_passive (GTK_CLIST (menued->clist));

  hbuttonbox3 = gtk_hbutton_box_new ();
  gtk_widget_show (hbuttonbox3);
  gtk_box_pack_start (GTK_BOX (vbox1), hbuttonbox3, FALSE, TRUE, 0);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (hbuttonbox3),
			     GTK_BUTTONBOX_SPREAD);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbuttonbox3), 6);
  gtk_button_box_set_child_size (GTK_BUTTON_BOX (hbuttonbox3), 40, 20);

  menued->up_button = gtk_button_new ();
  gtk_widget_show (menued->up_button);
  gtk_container_add (GTK_CONTAINER (hbuttonbox3), menued->up_button);
  gtk_tooltips_set_tip (tooltips, menued->up_button, _ ("Move the item and its children up one place in the list"), NULL);
  gtk_signal_connect (GTK_OBJECT (menued->up_button), "clicked",
		      GTK_SIGNAL_FUNC (on_up_button_clicked),
		      NULL);

  arrow1 = gtk_arrow_new (GTK_ARROW_UP, GTK_SHADOW_OUT);
  gtk_widget_show (arrow1);
  gtk_container_add (GTK_CONTAINER (menued->up_button), arrow1);

  menued->down_button = gtk_button_new ();
  gtk_widget_show (menued->down_button);
  gtk_container_add (GTK_CONTAINER (hbuttonbox3), menued->down_button);
  gtk_tooltips_set_tip (tooltips, menued->down_button, _ ("Move the item and its children down one place in the list"), NULL);
  gtk_signal_connect (GTK_OBJECT (menued->down_button), "clicked",
		      GTK_SIGNAL_FUNC (on_down_button_clicked),
		      NULL);

  arrow2 = gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_OUT);
  gtk_widget_show (arrow2);
  gtk_container_add (GTK_CONTAINER (menued->down_button), arrow2);

  menued->left_button = gtk_button_new ();
  gtk_widget_show (menued->left_button);
  gtk_container_add (GTK_CONTAINER (hbuttonbox3), menued->left_button);
  gtk_tooltips_set_tip (tooltips, menued->left_button, _ ("Move the item and its children up one level"), NULL);
  gtk_signal_connect (GTK_OBJECT (menued->left_button), "clicked",
		      GTK_SIGNAL_FUNC (on_left_button_clicked),
		      NULL);

  arrow3 = gtk_arrow_new (GTK_ARROW_LEFT, GTK_SHADOW_OUT);
  gtk_widget_show (arrow3);
  gtk_container_add (GTK_CONTAINER (menued->left_button), arrow3);

  menued->right_button = gtk_button_new ();
  gtk_widget_show (menued->right_button);
  gtk_container_add (GTK_CONTAINER (hbuttonbox3), menued->right_button);
  gtk_tooltips_set_tip (tooltips, menued->right_button, _ ("Move the item and its children down one level"), NULL);
  gtk_signal_connect (GTK_OBJECT (menued->right_button), "clicked",
		      GTK_SIGNAL_FUNC (on_right_button_clicked),
		      NULL);

  arrow4 = gtk_arrow_new (GTK_ARROW_RIGHT, GTK_SHADOW_OUT);
  gtk_widget_show (arrow4);
  gtk_container_add (GTK_CONTAINER (menued->right_button), arrow4);

  vbox3 = gtk_vbox_new (FALSE, 6);
  gtk_widget_show (vbox3);
  gtk_box_pack_start (GTK_BOX (hbox1), vbox3, FALSE, TRUE, 0);

  table1 = gtk_table_new (4, 3, FALSE);
  gtk_widget_show (table1);
  gtk_box_pack_start (GTK_BOX (vbox3), table1, FALSE, TRUE, 0);
  gtk_table_set_row_spacings (GTK_TABLE (table1), 4);
  gtk_table_set_col_spacings (GTK_TABLE (table1), 4);

  eventbox3 = gtk_event_box_new ();
  gtk_widget_show (eventbox3);
  gtk_table_attach (GTK_TABLE (table1), eventbox3, 0, 1, 2, 3,
		    GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_tooltips_set_tip (tooltips, eventbox3, _ ("The function to be called when the item is selected"), NULL);

  menued->handler_label = gtk_label_new (_ ("Handler:"));
  gtk_widget_show (menued->handler_label);
  gtk_container_add (GTK_CONTAINER (eventbox3), menued->handler_label);
  gtk_misc_set_alignment (GTK_MISC (menued->handler_label), 0, 0.5);

  eventbox2 = gtk_event_box_new ();
  gtk_widget_show (eventbox2);
  gtk_table_attach (GTK_TABLE (table1), eventbox2, 0, 1, 1, 2,
		    GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_tooltips_set_tip (tooltips, eventbox2, _ ("The name of the widget"),
			NULL);

  menued->name_label = gtk_label_new (_ ("Name:"));
  gtk_widget_show (menued->name_label);
  gtk_container_add (GTK_CONTAINER (eventbox2), menued->name_label);
  gtk_misc_set_alignment (GTK_MISC (menued->name_label), 0, 0.5);

  eventbox1 = gtk_event_box_new ();
  gtk_widget_show (eventbox1);
  gtk_table_attach (GTK_TABLE (table1), eventbox1, 0, 1, 0, 1,
		    GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_tooltips_set_tip (tooltips, eventbox1, _ ("The text of the menu item, or empty for separators. Hit return to add a new item below this one. Ctl-Return adds a child item beneath this one."), NULL);

  menued->label_label = gtk_label_new (_ ("Label:"));
  gtk_widget_show (menued->label_label);
  gtk_container_add (GTK_CONTAINER (eventbox1), menued->label_label);
  gtk_misc_set_alignment (GTK_MISC (menued->label_label), 0, 0.5);

  eventbox1 = gtk_event_box_new ();
  gtk_widget_show (eventbox1);
  gtk_table_attach (GTK_TABLE (table1), eventbox1, 0, 1, 3, 4,
		    GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_tooltips_set_tip (tooltips, eventbox1, _ ("An optional icon to show on the left of the menu item."), NULL);

  menued->icon_label = gtk_label_new (_ ("Icon:"));
  gtk_widget_show (menued->icon_label);
  gtk_container_add (GTK_CONTAINER (eventbox1), menued->icon_label);
  gtk_misc_set_alignment (GTK_MISC (menued->icon_label), 0, 0.5);

  menued->label_entry = gtk_entry_new ();
  gtk_widget_show (menued->label_entry);
  gtk_table_attach (GTK_TABLE (table1), menued->label_entry, 1, 3, 0, 1,
		    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (menued->label_entry), "changed",
		      GTK_SIGNAL_FUNC (on_entry_changed), NULL);
  gtk_signal_connect (GTK_OBJECT (menued->label_entry), "key_press_event",
		      GTK_SIGNAL_FUNC (on_label_entry_key_press),
		      NULL);

  menued->name_entry = gtk_entry_new ();
  gtk_widget_show (menued->name_entry);
  gtk_table_attach (GTK_TABLE (table1), menued->name_entry, 1, 3, 1, 2,
		    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (menued->name_entry), "changed",
		      GTK_SIGNAL_FUNC (on_entry_changed), NULL);

  menued->handler_entry = gtk_entry_new ();
  gtk_widget_show (menued->handler_entry);
  gtk_table_attach (GTK_TABLE (table1), menued->handler_entry, 1, 3, 2, 3,
		    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (menued->handler_entry), "changed",
		      GTK_SIGNAL_FUNC (on_entry_changed), NULL);

  menued->icon_entry = gtk_entry_new ();
  gtk_widget_show (menued->icon_entry);
  gtk_table_attach (GTK_TABLE (table1), menued->icon_entry, 1, 2, 3, 4,
		    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_signal_connect (GTK_OBJECT (menued->icon_entry), "changed",
		      GTK_SIGNAL_FUNC (on_entry_changed), NULL);

  menued->icon_button = gtk_button_new_with_label ("...");
  gtk_widget_show (menued->icon_button);
  gtk_table_attach (GTK_TABLE (table1), menued->icon_button, 2, 3, 3, 4,
		    GTK_FILL, GTK_FILL, 0, 0);

  hbuttonbox2 = gtk_hbutton_box_new ();
  gtk_widget_show (hbuttonbox2);
  gtk_box_pack_start (GTK_BOX (vbox3), hbuttonbox2, FALSE, TRUE, 0);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (hbuttonbox2),
			     GTK_BUTTONBOX_SPREAD);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbuttonbox2), 8);
  gtk_button_box_set_child_size (GTK_BUTTON_BOX (hbuttonbox2), 70, 20);

  menued->add_button = gtk_button_new_with_label (_ ("Add"));
  gtk_widget_show (menued->add_button);
  gtk_container_add (GTK_CONTAINER (hbuttonbox2), menued->add_button);
  gtk_tooltips_set_tip (tooltips, menued->add_button,
			_ ("Add a new item below the selected item."), NULL);
  gtk_signal_connect (GTK_OBJECT (menued->add_button), "clicked",
		      GTK_SIGNAL_FUNC (on_add_button_clicked),
		      NULL);

  menued->delete_button = gtk_button_new_with_label (_ ("Delete"));
  gtk_widget_show (menued->delete_button);
  gtk_container_add (GTK_CONTAINER (hbuttonbox2), menued->delete_button);
  gtk_tooltips_set_tip (tooltips, menued->delete_button,
			_ ("Delete the current item"), NULL);
  gtk_signal_connect (GTK_OBJECT (menued->delete_button), "clicked",
		      GTK_SIGNAL_FUNC (on_delete_button_clicked),
		      NULL);

  menued->type_frame = gtk_frame_new (_ ("Item Type:"));
  gtk_widget_show (menued->type_frame);
  gtk_box_pack_start (GTK_BOX (vbox3), menued->type_frame, FALSE, TRUE, 0);

  table2 = gtk_table_new (3, 3, FALSE);
  gtk_table_set_col_spacings (GTK_TABLE (table2), 4);
  gtk_table_set_row_spacings (GTK_TABLE (table2), 1);
  gtk_widget_show (table2);
  gtk_container_add (GTK_CONTAINER (menued->type_frame), table2);
  gtk_container_border_width (GTK_CONTAINER (table2), 4);

  eventbox1 = gtk_event_box_new ();
  gtk_widget_show (eventbox1);
  gtk_table_attach (GTK_TABLE (table2), eventbox1, 1, 2, 0, 1,
		    GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_tooltips_set_tip (tooltips, eventbox1,
			_ ("If the item is right justified (e.g. for a Help menu)."), NULL);

  menued->right_justify_label = gtk_label_new (_("Right Justify:"));
  gtk_misc_set_alignment (GTK_MISC (menued->right_justify_label), 0, 0.5);
  gtk_widget_show (menued->right_justify_label);
  gtk_container_add (GTK_CONTAINER (eventbox1), menued->right_justify_label);

  menued->right_justify_togglebutton = gtk_toggle_button_new_with_label (_("No"));
  gtk_widget_show (menued->right_justify_togglebutton);
  gtk_table_attach (GTK_TABLE (table2), menued->right_justify_togglebutton,
		    2, 3, 0, 1,
		    GTK_EXPAND | GTK_FILL, 0, 0, 0);

  eventbox1 = gtk_event_box_new ();
  gtk_widget_show (eventbox1);
  gtk_table_attach (GTK_TABLE (table2), eventbox1, 1, 2, 1, 2,
		    GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_tooltips_set_tip (tooltips, eventbox1,
			_ ("If the item is initially on."), NULL);

  menued->state_label = gtk_label_new (_("Active:"));
  gtk_misc_set_alignment (GTK_MISC (menued->state_label), 0, 0.5);
  gtk_widget_show (menued->state_label);
  gtk_container_add (GTK_CONTAINER (eventbox1), menued->state_label);

  menued->state_togglebutton = gtk_toggle_button_new_with_label (_("No"));
  gtk_widget_show (menued->state_togglebutton);
  gtk_table_attach (GTK_TABLE (table2), menued->state_togglebutton, 2, 3, 1, 2,
		    GTK_EXPAND | GTK_FILL, 0, 0, 0);

  hbox2 = gtk_hbox_new (FALSE, 4);
  gtk_widget_show (hbox2);
  gtk_table_attach (GTK_TABLE (table2), hbox2, 1, 3, 2, 3,
		    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

  eventbox1 = gtk_event_box_new ();
  gtk_widget_show (eventbox1);
  gtk_box_pack_start (GTK_BOX (hbox2), eventbox1, FALSE, TRUE, 0);
  gtk_tooltips_set_tip (tooltips, eventbox1,
			_ ("The radio menu item's group. Leave empty to use the parent menu's default group."), NULL);

  menued->group_label = gtk_label_new (_ ("Group:"));
  gtk_misc_set_alignment (GTK_MISC (menued->group_label), 0, 0.5);
  gtk_widget_show (menued->group_label);
  gtk_container_add (GTK_CONTAINER (eventbox1), menued->group_label);

  menued->group_combo = gtk_combo_new ();
  gtk_widget_set_usize (GTK_COMBO (menued->group_combo)->entry, 60, -1);
  gtk_widget_show (menued->group_combo);
  gtk_box_pack_start (GTK_BOX (hbox2), menued->group_combo, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (GTK_COMBO (menued->group_combo)->entry),
		      "changed", GTK_SIGNAL_FUNC (on_entry_changed), NULL);

  menued->radio_radiobutton = gtk_radio_button_new_with_label (table2_group,
							       _ ("Radio"));
  table2_group = gtk_radio_button_group (GTK_RADIO_BUTTON (menued->radio_radiobutton));
  gtk_widget_show (menued->radio_radiobutton);
  gtk_table_attach (GTK_TABLE (table2), menued->radio_radiobutton, 0, 1, 2, 3,
		    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

  menued->check_radiobutton = gtk_radio_button_new_with_label (table2_group,
							       _ ("Check"));
  table2_group = gtk_radio_button_group (GTK_RADIO_BUTTON (menued->check_radiobutton));
  gtk_widget_show (menued->check_radiobutton);
  gtk_table_attach (GTK_TABLE (table2), menued->check_radiobutton, 0, 1, 1, 2,
		    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

  menued->normal_radiobutton = gtk_radio_button_new_with_label (table2_group,
							      _ ("Normal"));
  table2_group = gtk_radio_button_group (GTK_RADIO_BUTTON (menued->normal_radiobutton));
  gtk_widget_show (menued->normal_radiobutton);
  gtk_table_attach (GTK_TABLE (table2), menued->normal_radiobutton, 0, 1, 0, 1,
		    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (menued->normal_radiobutton), TRUE);

  menued->accel_frame = gtk_frame_new (_ ("Accelerator:"));
  gtk_widget_show (menued->accel_frame);
  gtk_box_pack_start (GTK_BOX (vbox3), menued->accel_frame, FALSE, TRUE, 0);

  table3 = gtk_table_new (2, 2, FALSE);
  gtk_widget_show (table3);
  gtk_container_add (GTK_CONTAINER (menued->accel_frame), table3);
  gtk_container_border_width (GTK_CONTAINER (table3), 4);
  gtk_table_set_row_spacings (GTK_TABLE (table3), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table3), 4);

  hbox2 = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox2);
  gtk_table_attach (GTK_TABLE (table3), hbox2, 1, 2, 1, 2,
		    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

  menued->accel_key_entry = gtk_entry_new ();
  gtk_widget_set_usize (menued->accel_key_entry, 100, -1);
  gtk_widget_show (menued->accel_key_entry);
  gtk_box_pack_start (GTK_BOX (hbox2), menued->accel_key_entry,
		      TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (menued->accel_key_entry), "changed",
		      GTK_SIGNAL_FUNC (on_entry_changed), NULL);

  accel_key_button = gtk_button_new_with_label ("...");
  gtk_widget_show (accel_key_button);
  gtk_box_pack_start (GTK_BOX (hbox2), accel_key_button,
		      FALSE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (accel_key_button), "clicked",
		      GTK_SIGNAL_FUNC (on_accel_key_button_clicked),
		      NULL);

  hbox2 = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (hbox2);
  gtk_table_attach (GTK_TABLE (table3), hbox2, 1, 2, 0, 1,
		    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

  menued->accel_ctrl_checkbutton = gtk_check_button_new_with_label (_ ("Ctrl"));
  gtk_widget_show (menued->accel_ctrl_checkbutton);
  gtk_box_pack_start (GTK_BOX (hbox2), menued->accel_ctrl_checkbutton,
		      TRUE, TRUE, 0);

  menued->accel_shift_checkbutton = gtk_check_button_new_with_label (_ ("Shift"));
  gtk_widget_show (menued->accel_shift_checkbutton);
  gtk_box_pack_start (GTK_BOX (hbox2), menued->accel_shift_checkbutton,
		      TRUE, TRUE, 0);

  menued->accel_alt_checkbutton = gtk_check_button_new_with_label (_ ("Alt"));
  gtk_widget_show (menued->accel_alt_checkbutton);
  gtk_box_pack_start (GTK_BOX (hbox2), menued->accel_alt_checkbutton,
		      TRUE, TRUE, 0);

  label9 = gtk_label_new (_ ("Key:"));
  gtk_widget_show (label9);
  gtk_table_attach (GTK_TABLE (table3), label9, 0, 1, 1, 2,
		    GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label9), 0, 0.5);

  label8 = gtk_label_new (_ ("Modifiers:"));
  gtk_widget_show (label8);
  gtk_table_attach (GTK_TABLE (table3), label8, 0, 1, 0, 1,
		    GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label8), 0, 0.5);


  hseparator1 = gtk_hseparator_new ();
  gtk_widget_show (hseparator1);
  gtk_box_pack_start (GTK_BOX (vbox2), hseparator1, FALSE, TRUE, 8);

  hbuttonbox1 = gtk_hbutton_box_new ();
  gtk_widget_show (hbuttonbox1);
  gtk_box_pack_start (GTK_BOX (vbox2), hbuttonbox1, FALSE, TRUE, 0);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (hbuttonbox1), GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbuttonbox1), 8);

  menued->ok_button = gtk_button_new_with_label (_ ("OK"));
  gtk_widget_show (menued->ok_button);
  gtk_container_add (GTK_CONTAINER (hbuttonbox1), menued->ok_button);
  GTK_WIDGET_SET_FLAGS (menued->ok_button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (menued->ok_button);

  menued->apply_button = gtk_button_new_with_label (_ ("Apply"));
  gtk_widget_show (menued->apply_button);
  gtk_container_add (GTK_CONTAINER (hbuttonbox1), menued->apply_button);
  GTK_WIDGET_SET_FLAGS (menued->apply_button, GTK_CAN_DEFAULT);

  menued->cancel_button = gtk_button_new_with_label (_ ("Cancel"));
  gtk_widget_show (menued->cancel_button);
  gtk_container_add (GTK_CONTAINER (hbuttonbox1), menued->cancel_button);
  GTK_WIDGET_SET_FLAGS (menued->cancel_button, GTK_CAN_DEFAULT);

  gtk_signal_connect_after (GTK_OBJECT (menued->normal_radiobutton), "toggled",
			    GTK_SIGNAL_FUNC (on_radiobutton_toggled), NULL);
  gtk_signal_connect_after (GTK_OBJECT (menued->check_radiobutton), "toggled",
			    GTK_SIGNAL_FUNC (on_radiobutton_toggled), NULL);
  gtk_signal_connect_after (GTK_OBJECT (menued->radio_radiobutton), "toggled",
			    GTK_SIGNAL_FUNC (on_radiobutton_toggled), NULL);
  gtk_signal_connect (GTK_OBJECT (menued->right_justify_togglebutton),
		      "toggled",
		      GTK_SIGNAL_FUNC (on_right_justify_button_toggled), NULL);
  gtk_signal_connect (GTK_OBJECT (menued->state_togglebutton), "toggled",
		      GTK_SIGNAL_FUNC (on_state_button_toggled), NULL);
  gtk_signal_connect (GTK_OBJECT (menued->accel_ctrl_checkbutton), "toggled",
		      GTK_SIGNAL_FUNC (on_checkbutton_toggled), NULL);
  gtk_signal_connect (GTK_OBJECT (menued->accel_shift_checkbutton), "toggled",
		      GTK_SIGNAL_FUNC (on_checkbutton_toggled), NULL);
  gtk_signal_connect (GTK_OBJECT (menued->accel_alt_checkbutton), "toggled",
		      GTK_SIGNAL_FUNC (on_checkbutton_toggled), NULL);

  set_interface_state (menued);
}


GtkWidget *
glade_menu_editor_new ()
{
  GladeMenuEditor *menued;

  menued = gtk_type_new (glade_menu_editor_get_type ());

  return GTK_WIDGET (menued);
}

static void
glade_menu_editor_destroy (GtkObject *object)
{
  GladeMenuEditor *menued;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GLADE_IS_MENU_EDITOR (object));

  menued = GLADE_MENU_EDITOR (object);
  glade_menu_editor_reset (menued);
}


/**************************************************************************
 * Signal Handlers
 **************************************************************************/

static void
on_clist_select_row (GtkWidget * clist,
		     gint row,
		     gint column,
		     GdkEventButton * event,
		     gpointer user_data)
{
  GladeMenuEditor *menued;

  menued = GLADE_MENU_EDITOR (gtk_widget_get_toplevel (clist));

  show_item_properties (menued);

  if (event && !GTK_WIDGET_HAS_FOCUS (clist))
    gtk_widget_grab_focus (clist);

  set_interface_state (menued);
}

static void
on_clist_unselect_row (GtkWidget * clist,
		       gint row,
		       gint column,
		       GdkEventButton * event,
		       gpointer user_data)
{
  GladeMenuEditor *menued;

  menued = GLADE_MENU_EDITOR (gtk_widget_get_toplevel (clist));

  clear_form (menued, FALSE);

  if (event && !GTK_WIDGET_HAS_FOCUS (clist))
    gtk_widget_grab_focus (clist);

  set_interface_state (menued);
}

/* This will only call update_current_item if the text is different to the
   corresponding item field, since we don't want to propogate updates when
   we are setting the entry. */
static void
on_entry_changed (GtkWidget * entry,
		  gpointer user_data)
{
  GladeMenuEditor *menued;
  GtkCList *clist;
  GbMenuItemData *item;
  gchar *text, *item_text;
  gboolean changed = FALSE;
  gint row;

  menued = GLADE_MENU_EDITOR (gtk_widget_get_toplevel (entry));
  clist = GTK_CLIST (menued->clist);
  row = get_selected_row (menued);
  if (row == -1)
    return;
  item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (menued->clist),
						    row);

  text = gtk_entry_get_text (GTK_ENTRY (entry));
  if (entry == menued->label_entry)
    item_text = item->label;
  else if (entry == menued->name_entry)
    item_text = item->name;
  else if (entry == menued->handler_entry)
    item_text = item->handler;
  else if (entry == menued->icon_entry)
    item_text = item->icon;
  else if (entry == GTK_COMBO (menued->group_combo)->entry)
    item_text = item->group_name;
  else if (entry == menued->accel_key_entry)
    item_text = item->key;
  else
    return;

  if (item_text == NULL)
    {
      if (strlen (text) > 0)
	changed = TRUE;
    }
  else
    {
      if (strcmp (text, item_text))
	changed = TRUE;
    }

  if (changed)
    {
      if (entry == menued->label_entry)
	{
	  if (item->generate_name)
	    {
	      g_free (item->name);
	      item->name = generate_name (text, FALSE);
	      gtk_entry_set_text (GTK_ENTRY (menued->name_entry),
				  item->name ? item->name : "");
	      gtk_clist_set_text (clist, row, GLD_COL_NAME,
				  item->name ? item->name : "");
	      if (item->generate_handler)
		{
		  g_free (item->handler);
		  item->handler = generate_handler (menued, row, text,
						    item->name);
		  gtk_entry_set_text (GTK_ENTRY (menued->handler_entry),
				      item->handler ? item->handler : "");
		  gtk_clist_set_text (clist, row, GLD_COL_HANDLER,
				      item->handler ? item->handler : "");
		}
	    }
	}
      else if (entry == menued->name_entry)
	{
	  item->generate_name = FALSE;
	  if (item->generate_handler)
	    {
	      g_free (item->handler);
	      item->handler = generate_handler (menued, row, item->label,
						text);
	      gtk_entry_set_text (GTK_ENTRY (menued->handler_entry),
				  item->handler ? item->handler : "");
	      gtk_clist_set_text (clist, row, GLD_COL_HANDLER,
				  item->handler ? item->handler : "");
	    }
	}
      else if (entry == menued->handler_entry)
	{
	  item->generate_handler = FALSE;
	}

      update_current_item (menued);
      set_interface_state (menued);
    }
}

static gboolean
on_label_entry_key_press (GtkWidget * widget,
			  GdkEventKey * event,
			  gpointer user_data)
{
  GladeMenuEditor *menued;

  menued = GLADE_MENU_EDITOR (gtk_widget_get_toplevel (widget));

  /* If the Return key is pressed, we add a new item beneath the selected item.
     This makes it very easy to add several menus. If the Control key is
     pressed, we add the item as a child, otherwise we add it as a sibling. */
  if (event->keyval == GDK_Return)
    {
      if (event->state & GDK_CONTROL_MASK)
	{
	  add_item (menued, TRUE);
	  /* Since we are added a child, we may need to set the parent's
	     handler to NULL if it has been auto-generated. */
	  check_generated_handlers (menued);
	}
      else
	{
	  add_item (menued, FALSE);
	}
      return TRUE;
    }
  return FALSE;
}


static void
on_radiobutton_toggled (GtkWidget * togglebutton,
			gpointer user_data)
{
  GladeMenuEditor *menued;
  GbMenuItemData *item;
  GbMenuItemType type;
  gboolean changed = FALSE;

  menued = GLADE_MENU_EDITOR (gtk_widget_get_toplevel (GTK_WIDGET (togglebutton)));
  item = get_selected_item (menued);
  if (item == NULL)
    return;

  if (togglebutton == menued->normal_radiobutton)
    type = GB_MENU_ITEM_NORMAL;
  else if (togglebutton == menued->check_radiobutton)
    type = GB_MENU_ITEM_CHECK;
  else if (togglebutton == menued->radio_radiobutton)
    type = GB_MENU_ITEM_RADIO;

  if (GTK_TOGGLE_BUTTON (togglebutton)->active)
    {
      if (type != item->type)
	changed = TRUE;
    }
  else
    {
      if (type == item->type)
	changed = TRUE;
    }

  if (changed)
    {
      update_current_item (menued);
      set_interface_state (menued);
    }
}

static void
on_checkbutton_toggled (GtkWidget * togglebutton,
			gpointer user_data)
{
  GladeMenuEditor *menued;
  GbMenuItemData *item;
  guint active;
  guint8 currently_active;

  menued = GLADE_MENU_EDITOR (gtk_widget_get_toplevel (GTK_WIDGET (togglebutton)));
  active = GTK_TOGGLE_BUTTON (togglebutton)->active;
  item = get_selected_item (menued);
  if (item == NULL)
    return;
  if (togglebutton == menued->accel_ctrl_checkbutton)
    currently_active = item->modifiers & GDK_CONTROL_MASK;
  if (togglebutton == menued->accel_shift_checkbutton)
    currently_active = item->modifiers & GDK_SHIFT_MASK;
  if (togglebutton == menued->accel_alt_checkbutton)
    currently_active = item->modifiers & GDK_MOD1_MASK;

  if ((active && !currently_active) || (!active && currently_active))
    {
      update_current_item (menued);
      set_interface_state (menued);
    }
}

static void
on_right_justify_button_toggled (GtkToggleButton * togglebutton,
				 gpointer user_data)
{
  GladeMenuEditor *menued;
  GbMenuItemData *item;
  GtkWidget *label;
  guint active;

  menued = GLADE_MENU_EDITOR (gtk_widget_get_toplevel (GTK_WIDGET (togglebutton)));
  active = GTK_TOGGLE_BUTTON (togglebutton)->active;
  label = GTK_BUTTON (togglebutton)->child;
  gtk_label_set (GTK_LABEL (label), active ? _("Yes") : _("No"));

  item = get_selected_item (menued);
  if (item == NULL)
    return;
  if ((item->right_justify && !active) || (!item->right_justify && active))
    update_current_item (menued);
}

static void
on_state_button_toggled (GtkToggleButton * togglebutton,
			 gpointer user_data)
{
  GladeMenuEditor *menued;
  GbMenuItemData *item;
  GtkWidget *label;
  guint active;

  menued = GLADE_MENU_EDITOR (gtk_widget_get_toplevel (GTK_WIDGET (togglebutton)));
  active = GTK_TOGGLE_BUTTON (togglebutton)->active;
  label = GTK_BUTTON (togglebutton)->child;
  gtk_label_set (GTK_LABEL (label), active ? _("Yes") : _("No"));

  item = get_selected_item (menued);
  if (item == NULL)
    return;
  if ((item->active && !active) || (!item->active && active))
    update_current_item (menued);
}

static void
on_accel_key_button_clicked (GtkButton * button,
			     gpointer user_data)
{
  GladeMenuEditor *menued;
  GladeKeysDialog *dialog;

  menued = GLADE_MENU_EDITOR (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  dialog = GLADE_KEYS_DIALOG (glade_keys_dialog_new ());
  gtk_window_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
  gtk_object_set_data (GTK_OBJECT (dialog->clist), "menued", menued);
  gtk_signal_connect_object (GTK_OBJECT (dialog), "delete_event",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));
  gtk_signal_connect (GTK_OBJECT (dialog->clist), "select_row",
		      GTK_SIGNAL_FUNC (on_keys_dialog_clist_select),
		      dialog->clist);
  gtk_signal_connect (GTK_OBJECT (dialog->ok_button), "clicked",
		      GTK_SIGNAL_FUNC (on_keys_dialog_ok), dialog->clist);
  gtk_signal_connect_object (GTK_OBJECT (dialog->cancel_button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (dialog));
  gtk_widget_show (GTK_WIDGET (dialog));
}

static void
on_keys_dialog_clist_select (GtkWidget * widget, gint row, gint column,
			     GdkEventButton * bevent, GtkWidget * clist)
{
  if (bevent && bevent->type == GDK_2BUTTON_PRESS)
    on_keys_dialog_ok (widget, clist);
}

static void
on_keys_dialog_ok (GtkWidget * widget, GtkWidget * clist)
{
  GladeMenuEditor *menued;
  GladeKeysDialog *dialog;
  gchar *key_symbol;

  dialog = (GladeKeysDialog*) gtk_widget_get_toplevel (widget);
  key_symbol = glade_keys_dialog_get_key_symbol (dialog);
  if (key_symbol)
    {
      menued = gtk_object_get_data (GTK_OBJECT (clist), "menued");
      g_return_if_fail (menued != NULL);
      gtk_entry_set_text (GTK_ENTRY (menued->accel_key_entry), key_symbol);
    }

  gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
on_up_button_clicked (GtkButton * button,
		      gpointer user_data)
{
  GladeMenuEditor *menued;
  GtkWidget *clist;
  GbMenuItemData *item, *prev_item;
  gint row, new_row, i, level;
  GList *items;

  menued = GLADE_MENU_EDITOR (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  clist = menued->clist;
  row = get_selected_row (menued);
  if (row == -1 || row == 0)
    return;

  item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (clist), row);
  level = item->level;

  /* Find the new position of the item and its children. */
  new_row = -1;
  for (i = row - 1; i >= 0; i--)
    {
      prev_item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (clist),
							     i);
      if (prev_item->level == level)
	{
	  new_row = i;
	  break;
	}
      else if (prev_item->level < level)
	break;
    }

  /* Return if we can't move the item up. */
  if (new_row == -1)
    return;

  /* Remove item and children. */
  items = remove_item_and_children (clist, row);

  /* Now insert at new position. */
  insert_items (clist, items, new_row);
  ensure_visible (clist, new_row);

  g_list_free (items);

  gtk_clist_select_row (GTK_CLIST (clist), new_row, 0);
  set_interface_state (menued);
}

static void
on_down_button_clicked (GtkButton * button,
			gpointer user_data)
{
  GladeMenuEditor *menued;
  GtkWidget *clist;
  GbMenuItemData *item, *next_item;
  gint row, new_row, i, level;
  gboolean found_next_item;
  GList *items;

  menued = GLADE_MENU_EDITOR (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  clist = menued->clist;
  row = get_selected_row (menued);
  if (row == -1 || row == GTK_CLIST (clist)->rows - 1)
    return;

  item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (clist), row);
  level = item->level;

  /* Find the new position of the item and its children. */
  new_row = -1;
  found_next_item = FALSE;
  for (i = row + 1; i < GTK_CLIST (clist)->rows; i++)
    {
      next_item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (clist),
							     i);
      /* We have to skip all the children of the next item as well. */
      if (next_item->level == level)
	{
	  if (found_next_item)
	    {
	      new_row = i;
	      break;
	    }
	  else
	    found_next_item = TRUE;
	}
      else if (next_item->level < level)
	break;
    }

  /* Return if we can't move the item up. */
  if (new_row == -1)
    {
      if (found_next_item)
	new_row = i;
      else
	return;
    }

  /* Remove item and children. */
  items = remove_item_and_children (clist, row);
  /* Remember that the new_row needs to be shifted because we deleted items. */
  new_row -= g_list_length (items);

  /* Now insert at new position. */
  insert_items (clist, items, new_row);
  ensure_visible (clist, new_row);

  g_list_free (items);

  gtk_clist_select_row (GTK_CLIST (clist), new_row, 0);
  set_interface_state (menued);
}

static void
on_left_button_clicked (GtkButton * button,
			gpointer user_data)
{
  GladeMenuEditor *menued;
  GtkWidget *clist;
  GbMenuItemData *item;
  gint row, i, level;

  menued = GLADE_MENU_EDITOR (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  clist = menued->clist;
  row = get_selected_row (menued);
  if (row == -1)
    return;
  item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (clist), row);
  level = item->level;
  if (item->level > 0)
    item->level--;
  gtk_clist_set_shift (GTK_CLIST (clist), row, 0, 0, item->level * GB_INDENT);

  for (i = row + 1; i < GTK_CLIST (clist)->rows; i++)
    {
      item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (clist), i);
      if (item->level <= level)
	break;
      item->level--;
      gtk_clist_set_shift (GTK_CLIST (clist), i, 0, 0,
			   item->level * GB_INDENT);
    }
  check_generated_handlers (menued);
  set_interface_state (menued);
}

static void
on_right_button_clicked (GtkButton * button,
			 gpointer user_data)
{
  GladeMenuEditor *menued;
  GtkWidget *clist;
  GbMenuItemData *item, *prev_item;
  gint row, i, level;

  menued = GLADE_MENU_EDITOR (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  clist = menued->clist;
  row = get_selected_row (menued);
  if (row == -1 || row == 0)
    return;
  item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (clist), row);
  prev_item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (clist),
							 row - 1);
  if (prev_item->level < item->level)
    return;

  level = item->level;
  item->level++;
  gtk_clist_set_shift (GTK_CLIST (clist), row, 0, 0, item->level * GB_INDENT);

  for (i = row + 1; i < GTK_CLIST (clist)->rows; i++)
    {
      item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (clist), i);
      if (item->level <= level)
	break;
      item->level++;
      gtk_clist_set_shift (GTK_CLIST (clist), i, 0, 0,
			   item->level * GB_INDENT);
    }

  check_generated_handlers (menued);
  set_interface_state (menued);
}

static void
on_add_button_clicked (GtkWidget * button,
		       gpointer user_data)
{
  GladeMenuEditor *menued;
  menued = GLADE_MENU_EDITOR (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  add_item (menued, FALSE);
}

static gboolean
on_key_press (GtkWidget * widget,
	      GdkEventKey * event,
	      gpointer user_data)
{
  switch (event->keyval)
    {
    case GDK_Delete:
      on_delete_button_clicked (widget, NULL);
      break;
    }

  return FALSE;
}

static void
on_delete_button_clicked (GtkWidget * widget,
			  gpointer user_data)
{
  GladeMenuEditor *menued;
  GtkWidget *clist;
  GbMenuItemData *item;
  gint row, level, i;

  menued = GLADE_MENU_EDITOR (gtk_widget_get_toplevel (GTK_WIDGET (widget)));
  clist = menued->clist;
  row = get_selected_row (menued);
  if (row == -1)
    return;
  item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (clist), row);
  level = item->level;

  gtk_clist_remove (GTK_CLIST (clist), row);
  g_free (item->name);
  g_free (item->label);
  g_free (item->handler);
  g_free (item->icon);
  g_free (item->group_name);
  g_free (item->key);
  g_free (item);

  /* Move all children up a level */
  for (i = row; i < GTK_CLIST (clist)->rows; i++)
    {
      item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (clist), i);
      if (item->level <= level)
	break;
      item->level--;
      gtk_clist_set_shift (GTK_CLIST (clist), i, 0, 0,
			   item->level * GB_INDENT);
    }

  gtk_clist_select_row (GTK_CLIST (clist), row, 0);
  set_interface_state (menued);
}


/**************************************************************************
 * Utility functions
 **************************************************************************/


/* This returns the index of the currently selected row in the clist, or -1
   if no item is currently selected. */
static gint
get_selected_row (GladeMenuEditor * menued)
{
  if (GTK_CLIST (menued->clist)->selection == NULL)
    return -1;
  return GPOINTER_TO_INT (GTK_CLIST (menued->clist)->selection->data);
}

/* This returns the currently selected item, or NULL if no item is currently
   selected. */
static GbMenuItemData*
get_selected_item (GladeMenuEditor * menued)
{
  gint row = get_selected_row (menued);
  if (row == -1)
    return NULL;
  return (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (menued->clist),
						    row);
}

/* This set the sensitivity of the buttons according to the current state. */
static void
set_interface_state (GladeMenuEditor * menued)
{
  GbMenuItemData *item, *tmp_item;
  GtkCList *clist;
  gboolean up_button_sens = FALSE, down_button_sens = FALSE;
  gboolean left_button_sens = FALSE, right_button_sens = FALSE;
  gboolean add_button_sens = FALSE, delete_button_sens = FALSE;
  gboolean justify_sens = FALSE, state_sens = FALSE, group_sens = FALSE;
  gboolean form_sens = FALSE;
  gint index, i;

  clist = GTK_CLIST (menued->clist);

  /* Figure out which of the arrow buttons should be sensitive. */
  /* The Delete button and the entire form are sensitive if an item is
     selected in the clist. */
  index = get_selected_row (menued);
  if (index != -1)
    {
      form_sens = TRUE;
      delete_button_sens = TRUE;

      if (index > 0)
	up_button_sens = TRUE;
      if (index < clist->rows - 1)
	down_button_sens = TRUE;

      item = (GbMenuItemData *) gtk_clist_get_row_data (clist, index);
      if (item->level > 0)
	left_button_sens = TRUE;

      if (index > 0)
	{
	  tmp_item = (GbMenuItemData *) gtk_clist_get_row_data (clist,
								index - 1);
	  if (tmp_item->level >= item->level)
	    right_button_sens = TRUE;
	}

      /* The 'Right Justify' button is sensitive if this is the last item on
	 the top level of the menu. */
      if (item->level == 0)
	{
	  justify_sens = TRUE;
	  for (i = index + 1; i < clist->rows; i++)
	    {
	      tmp_item = (GbMenuItemData *) gtk_clist_get_row_data (clist, i);
	      if (tmp_item->level == 0)
		{
		  justify_sens = FALSE;
		  break;
		}
	    }
	}
    }

  /* The Add button is always sensitive, since empty labels are separators. */
  add_button_sens = TRUE;

  /* Figure out if the radio group widgets should be sensitive. */
  if (GTK_TOGGLE_BUTTON (menued->radio_radiobutton)->active)
    {
      group_sens = TRUE;
      state_sens = TRUE;
    }

  if (GTK_TOGGLE_BUTTON (menued->check_radiobutton)->active)
    state_sens = TRUE;

  /* Now set the sensitivity of the widgets. */
  gtk_widget_set_sensitive (menued->label_label, form_sens);
  gtk_widget_set_sensitive (menued->label_entry, form_sens);
  gtk_widget_set_sensitive (menued->name_label, form_sens);
  gtk_widget_set_sensitive (menued->name_entry, form_sens);
  gtk_widget_set_sensitive (menued->handler_label, form_sens);
  gtk_widget_set_sensitive (menued->handler_entry, form_sens);

  /* FIXME: These are always FALSE for now, until we implement icon support. */
  gtk_widget_set_sensitive (menued->icon_label, FALSE);
  gtk_widget_set_sensitive (menued->icon_entry, FALSE);
  gtk_widget_set_sensitive (menued->icon_button, FALSE);

  gtk_widget_set_sensitive (menued->add_button, add_button_sens);
  gtk_widget_set_sensitive (menued->delete_button, delete_button_sens);

  gtk_widget_set_sensitive (menued->type_frame, form_sens);
  gtk_widget_set_sensitive (menued->right_justify_label, justify_sens);
  gtk_widget_set_sensitive (menued->right_justify_togglebutton, justify_sens);
  gtk_widget_set_sensitive (menued->state_label, state_sens);
  gtk_widget_set_sensitive (menued->state_togglebutton, state_sens);
  gtk_widget_set_sensitive (menued->group_label, group_sens);
  gtk_widget_set_sensitive (menued->group_combo, group_sens);

  gtk_widget_set_sensitive (menued->accel_frame, form_sens);
}


/* This gets a string representing the accelerator key + modifiers.
   It returns a pointer to a static buffer. */
static gchar *
get_accel_string (gchar * key, guint8 modifiers)
{
  static gchar buffer[32];

  buffer[0] = '\0';
  if (modifiers & GDK_CONTROL_MASK)
    strcat (buffer, "C+");
  if (modifiers & GDK_SHIFT_MASK)
    strcat (buffer, "S+");
  if (modifiers & GDK_MOD1_MASK)
    strcat (buffer, "A+");
  if (key)
    strcat (buffer, key);
  return buffer;
}

/* This shows the properties of the item currently selected in the clist. */
static void
show_item_properties (GladeMenuEditor * menued)
{
  GbMenuItemData *item;
  GtkWidget *clist;

  clist = menued->clist;
  item = get_selected_item (menued);
  if (item == NULL)
    return;

  /* Now set them to the item's properties. */
  gtk_entry_set_text (GTK_ENTRY (menued->label_entry),
		      item->label ? item->label : "");
  gtk_entry_set_text (GTK_ENTRY (menued->name_entry),
		      item->name ? item->name : "");
  gtk_entry_set_text (GTK_ENTRY (menued->handler_entry),
		      item->handler ? item->handler : "");
  gtk_entry_set_text (GTK_ENTRY (menued->icon_entry),
		      item->icon ? item->icon : "");

  if (item->type == GB_MENU_ITEM_NORMAL)
    {
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (menued->normal_radiobutton), TRUE);
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (menued->group_combo)->entry),
			  "");
    }
  else if (item->type == GB_MENU_ITEM_CHECK)
    {
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (menued->check_radiobutton), TRUE);
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (menued->group_combo)->entry),
			  "");
    }
  else
    {
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (menued->radio_radiobutton), TRUE);
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (menued->group_combo)->entry),
			  item->group_name ? item->group_name : "");
    }

  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (menued->right_justify_togglebutton), item->right_justify);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (menued->state_togglebutton), item->active);

  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (menued->accel_ctrl_checkbutton), (item->modifiers & GDK_CONTROL_MASK) ? TRUE : FALSE);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (menued->accel_shift_checkbutton), (item->modifiers & GDK_SHIFT_MASK) ? TRUE : FALSE);
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (menued->accel_alt_checkbutton), (item->modifiers & GDK_MOD1_MASK) ? TRUE : FALSE);
  gtk_entry_set_text (GTK_ENTRY (menued->accel_key_entry),
		      item->key ? item->key : "");
}

/* This adds a new default item, with the label 'New Item'.
   It is added to the clist beneath the currently selected item, or at the
   end of the list if no item is selected. If as_child is TRUE it adds the
   item as a child of the selected item, else it adds it as a sibling. */
static void
add_item (GladeMenuEditor * menued,
	  gboolean as_child)
{
  GbMenuItemData *item, *selected_item;
  GtkWidget *clist;
  gint row;

  item = g_new (GbMenuItemData, 1);
  item->label = gb_widget_new_name ("Item");
  item->name = generate_name (item->label, FALSE);
  item->handler = generate_handler (menued, -1, item->label, item->name);
  item->icon = NULL;
  item->type = GB_MENU_ITEM_NORMAL;
  item->right_justify = FALSE;
  item->active = FALSE;
  item->group_name = NULL;
  item->modifiers = 0;
  item->key = NULL;
  item->level = 0;
  item->generate_name = TRUE;
  item->generate_handler = TRUE;

  clist = menued->clist;
  row = get_selected_row (menued);
  if (row != -1)
    {
      selected_item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (clist), row);
      item->level = selected_item->level + (as_child ? 1 : 0);
      insert_item (GTK_CLIST (clist), item, row + 1);
      gtk_clist_select_row (GTK_CLIST (clist), row + 1, 0);
      ensure_visible (clist, row + 1);
    }
  else
    {
      item->level = 0;
      insert_item (GTK_CLIST (clist), item, -1);
      gtk_clist_select_row (GTK_CLIST (clist), GTK_CLIST (clist)->rows - 1, 0);
      ensure_visible (clist, GTK_CLIST (clist)->rows - 1);
    }

  set_interface_state (menued);
  gtk_widget_grab_focus (menued->label_entry);
  gtk_entry_select_region (GTK_ENTRY (menued->label_entry), 0, -1);
}

/* This adds the item to the clist at the given position. */
static void
insert_item (GtkCList * clist, GbMenuItemData * item, gint row)
{
  gchar *rowdata[GB_MENUED_NUM_COLS];

  /* Empty labels are understood to be separators. */
  if (item->label && strlen (item->label) > 0)
    rowdata[GLD_COL_LABEL] = item->label;
  else
    rowdata[GLD_COL_LABEL] = GB_SEPARATOR_TEXT;
  if (item->type == GB_MENU_ITEM_NORMAL)
    rowdata[GLD_COL_TYPE] = "";
  else if (item->type == GB_MENU_ITEM_CHECK)
    rowdata[GLD_COL_TYPE] = "Check";
  else if (item->type == GB_MENU_ITEM_RADIO)
    rowdata[GLD_COL_TYPE] = "Radio";
  rowdata[GLD_COL_ACCEL] = get_accel_string (item->key, item->modifiers);
  rowdata[GLD_COL_NAME] = item->name ? item->name : "";
  rowdata[GLD_COL_HANDLER] = item->handler ? item->handler : "";

  rowdata[GLD_COL_ICON] = item->icon ? item->icon : "";
  rowdata[GLD_COL_ACTIVE] = item->active ? _("Yes") : "";
  rowdata[GLD_COL_GROUP] = item->group_name ? item->group_name : "";
  rowdata[GLD_COL_JUSTIFY] = item->right_justify ? _("Yes") : "";

  if (row >= 0)
    gtk_clist_insert (clist, row, rowdata);
  else
    row = gtk_clist_append (GTK_CLIST (clist), rowdata);

  gtk_clist_set_row_data (GTK_CLIST (clist), row, item);
  gtk_clist_set_shift (GTK_CLIST (clist), row, 0, 0, item->level * GB_INDENT);
}

/* This makes sure the given row is visible. */
static void
ensure_visible (GtkWidget *clist,
		gint row)
{
  if (gtk_clist_row_is_visible (GTK_CLIST (clist), row)
      != GTK_VISIBILITY_FULL)
    gtk_clist_moveto(GTK_CLIST(clist), row, -1, 0.5, 0);
}

/* This updates the currently selected item, updating each field if it is
   different to the item settings. */
static void
update_current_item (GladeMenuEditor * menued)
{
  GbMenuItemData *item;
  GtkCList *clist;
  gchar *name, *label, *handler, *icon, *group_name, *key;
  GbMenuItemType type;
  gint row;
  guint8 modifiers;
  gboolean right_justify, active, update_accelerator = FALSE;

  clist = GTK_CLIST (menued->clist);
  row = get_selected_row (menued);
  if (row == -1)
    return;
  item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (menued->clist),
						    row);

  name = gtk_entry_get_text (GTK_ENTRY (menued->name_entry));
  if (item_property_changed (name, item->name))
    {
      g_free (item->name);
      item->name = copy_item_property (name);
      gtk_clist_set_text (clist, row, GLD_COL_NAME,
			  item->name ? item->name : "");
    }

  label = gtk_entry_get_text (GTK_ENTRY (menued->label_entry));
  if (item_property_changed (label, item->label))
    {
      g_free (item->label);
      item->label = copy_item_property (label);
      gtk_clist_set_text (clist, row, GLD_COL_LABEL,
			  item->label ? item->label : GB_SEPARATOR_TEXT);
    }

  handler = gtk_entry_get_text (GTK_ENTRY (menued->handler_entry));
  if (item_property_changed (handler, item->handler))
    {
      g_free (item->handler);
      item->handler = copy_item_property (handler);
      gtk_clist_set_text (clist, row, GLD_COL_HANDLER,
			  item->handler ? item->handler : "");
    }

  icon = gtk_entry_get_text (GTK_ENTRY (menued->icon_entry));
  if (item_property_changed (icon, item->icon))
    {
      g_free (item->icon);
      item->icon = copy_item_property (icon);
      gtk_clist_set_text (clist, row, GLD_COL_ICON,
			  item->icon ? item->icon : "");
    }

  if (GTK_TOGGLE_BUTTON (menued->normal_radiobutton)->active)
    type = GB_MENU_ITEM_NORMAL;
  else if (GTK_TOGGLE_BUTTON (menued->check_radiobutton)->active)
    type = GB_MENU_ITEM_CHECK;
  else
    type = GB_MENU_ITEM_RADIO;
  if (item->type != type)
    {
      item->type = type;
      if (type == GB_MENU_ITEM_NORMAL)
	gtk_clist_set_text (clist, row, GLD_COL_TYPE, "");
      else if (type == GB_MENU_ITEM_CHECK)
	gtk_clist_set_text (clist, row, GLD_COL_TYPE, "Check");
      else if (type == GB_MENU_ITEM_RADIO)
	gtk_clist_set_text (clist, row, GLD_COL_TYPE, "Radio");
    }

  right_justify = GTK_TOGGLE_BUTTON (menued->right_justify_togglebutton)->active
    ? TRUE : FALSE;
  if (right_justify != item->right_justify)
    {
      item->right_justify = right_justify;
      gtk_clist_set_text (clist, row, GLD_COL_JUSTIFY,
			  right_justify ? _("Yes") : "");
    }

  active = GTK_TOGGLE_BUTTON (menued->state_togglebutton)->active
    ? TRUE : FALSE;
  if (active != item->active)
    {
      item->active = active;
      gtk_clist_set_text (clist, row, GLD_COL_ACTIVE, active ? _("Yes") : "");
    }

  group_name = gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (menued->group_combo)->entry));
  if (item_property_changed (group_name, item->group_name))
    {
      g_free (item->group_name);
      item->group_name = copy_item_property (group_name);
      gtk_clist_set_text (clist, row, GLD_COL_GROUP,
			  item->group_name ? item->group_name : "");
      update_radio_groups (menued);
    }

  key = gtk_entry_get_text (GTK_ENTRY (menued->accel_key_entry));
  if (item_property_changed (key, item->key))
    {
      g_free (item->key);
      item->key = copy_item_property (key);
      update_accelerator = TRUE;
    }

  modifiers = 0;
  if (GTK_TOGGLE_BUTTON (menued->accel_ctrl_checkbutton)->active)
    modifiers |= GDK_CONTROL_MASK;
  if (GTK_TOGGLE_BUTTON (menued->accel_shift_checkbutton)->active)
    modifiers |= GDK_SHIFT_MASK;
  if (GTK_TOGGLE_BUTTON (menued->accel_alt_checkbutton)->active)
    modifiers |= GDK_MOD1_MASK;
  if (modifiers != item->modifiers)
    {
      item->modifiers = modifiers;
      update_accelerator = TRUE;
    }

  if (update_accelerator)
    gtk_clist_set_text (clist, row, GLD_COL_ACCEL,
			get_accel_string (item->key, item->modifiers));

  set_interface_state (menued);
}

/* This checks if the new value is different to the old, but an empty string
   in new is taken to be equal to NULL as well. */
static gboolean
item_property_changed (gchar *new, gchar *old)
{
  if (old == NULL)
    return (new == NULL || strlen (new) == 0) ? FALSE : TRUE;
  if (new == NULL)
    return TRUE;
  if (!strcmp (old, new))
    return FALSE;
  return TRUE;
}

/* This returns a copy of the given property string, or NULL if it is an
   empty string. */
static gchar*
copy_item_property (gchar *property)
{
  if (strlen (property) == 0)
    return NULL;
  return g_strdup (property);
}


/* This clears the form, ready to add a new item. If full is TRUE it resets
   the type checkbuttons, group and accelerator modifiers. When adding items
   full is set to FALSE so the user can add several items of the same type. */
static void
clear_form (GladeMenuEditor * menued,
	    gboolean full)
{
  gtk_entry_set_text (GTK_ENTRY (menued->label_entry), "");
  gtk_entry_set_text (GTK_ENTRY (menued->name_entry), "");
  gtk_entry_set_text (GTK_ENTRY (menued->handler_entry), "");
  gtk_entry_set_text (GTK_ENTRY (menued->icon_entry), "");
  gtk_entry_set_text (GTK_ENTRY (menued->accel_key_entry), "");

  if (full)
    {
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (menued->normal_radiobutton), TRUE);
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (menued->group_combo)->entry),
			  "");
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (menued->accel_ctrl_checkbutton), FALSE);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (menued->accel_shift_checkbutton), FALSE);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (menued->accel_alt_checkbutton), FALSE);
    }
}


/* This recreates the list of available radio groups, and puts them in the
   combo's drop-down list. */
static void
update_radio_groups (GladeMenuEditor * menued)
{
  GbMenuItemData *item;
  GList *groups = NULL;
  gint row;

  for (row = 0; row < GTK_CLIST (menued->clist)->rows; row++)
    {
      item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (menued->clist), row);
      if (item->group_name && strlen (item->group_name) > 0)
	groups = add_radio_group (groups, item->group_name);
    }

  /* We have to block the combo's list's selection changed signal, or it
     causes problems. */
  gtk_signal_handler_block (GTK_OBJECT (GTK_COMBO (menued->group_combo)->list),
			    GTK_COMBO (menued->group_combo)->list_change_id);
  if (groups)
    gtk_combo_set_popdown_strings (GTK_COMBO (menued->group_combo), groups);
  else
    gtk_list_clear_items (GTK_LIST (GTK_COMBO (menued->group_combo)->list),
			  0, -1);
  gtk_signal_handler_unblock (GTK_OBJECT (GTK_COMBO (menued->group_combo)->list),
			    GTK_COMBO (menued->group_combo)->list_change_id);
  g_list_free (groups);
}

static GList*
add_radio_group (GList *groups,
		 gchar *group_name)
{
  GList *tmp_list;
  gint pos, cmp;

  tmp_list = groups;
  pos = 0;
  while (tmp_list)
    {
      cmp = strcmp ((gchar *) tmp_list->data, group_name);
      if (cmp == 0)
	return groups;
      if (cmp > 0)
	break;
      tmp_list = tmp_list->next;
      pos++;
    }
  return g_list_insert (groups, group_name, pos);
}

/* This removes an item and its children from the clist, and returns a list
   of the removed items. */
static GList *
remove_item_and_children (GtkWidget * clist,
			  gint row)
{
  GList *items = NULL;
  GbMenuItemData *item;
  gint level;

  item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (clist), row);
  level = item->level;
  items = g_list_append (items, item);
  gtk_clist_remove (GTK_CLIST (clist), row);

  while (row < GTK_CLIST (clist)->rows)
    {
      item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (clist), row);
      if (item->level > level)
	{
	  items = g_list_append (items, item);
	  gtk_clist_remove (GTK_CLIST (clist), row);
	}
      else
	break;
    }
  return items;
}

/* This inserts the given list of items at the given position in the clist. */
static void
insert_items (GtkWidget * clist,
	      GList * items,
	      gint row)
{
  GbMenuItemData *item;

  while (items)
    {
      item = (GbMenuItemData *) items->data;
      insert_item (GTK_CLIST (clist), item, row++);
      items = items->next;
    }
}


/* This returns the default name of the widget, given its label. The returned
   string should be freed at some point. If reserve_id is TRUE it reserves
   the ID number. We only do this when the widgets is actually created
   otherwise we may reserve lots of unnecessary IDs as the user types in
   the label. */
static gchar*
generate_name (gchar *label,
	       gboolean reserve_id)
{
  gchar id_buf[16];
  gint new_id;
  gchar *name, *src, *dest;
  gchar *separator_name;

  /* For empty labels, i.e. separators, use 'separator'. */
  if (label == NULL || strlen (label) == 0)
    {
      separator_name = _("separator");
      new_id = gb_widget_get_last_id (separator_name) + 1;
      sprintf (id_buf, "%i", new_id);
      name = g_malloc (strlen (separator_name) + strlen (id_buf) + 1);
      strcpy (name, separator_name);
      strcat (name, id_buf);
      if (reserve_id)
	gb_widget_set_last_id (separator_name, new_id);
      return name;
    }

  new_id = gb_widget_get_last_id (label) + 1;
  if (new_id == 1)
    id_buf[0] = '\0';
  else
    sprintf (id_buf, "%i", new_id);

  if (reserve_id)
    gb_widget_set_last_id (label, new_id);

  name = g_malloc (strlen (label) + strlen (id_buf) + 1);
  /* Convert spaces to underscores. */
  for (src = label, dest = name; *src; src++)
    {
      if (*src == ' ')
	*dest++ = '_';
      else if (*src == '.')
	continue;
      else
	*dest++ = *src;
    }
  *dest = '\0';
  strcat (name, id_buf);
  return name;
}

/* This returns the default 'activate' handler name, given the name of the
   item. The returned string should be freed at some point. */
static gchar*
generate_handler (GladeMenuEditor *menued, gint row, gchar *label, gchar *name)
{
  gchar *handler, *start = "on_", *end = "_activate";

  /* For empty labels, i.e. separators, and items with submenus, there is no
     handler by default. */
  if (label == NULL || strlen (label) == 0 || is_parent (menued, row))
    return NULL;

  handler = g_malloc (strlen (name) + strlen (start) + strlen (end) + 1);
  strcpy (handler, start);
  strcat (handler, name);
  strcat (handler, end);
  return handler;
}


/* This makes sure the default handlers are updated as items are moved around.
 */
static void
check_generated_handlers (GladeMenuEditor *menued)
{
  GtkCList *clist;
  GbMenuItemData *item;
  gint row;
  gchar *handler;

  clist = GTK_CLIST (menued->clist);
  for (row = 0; row < clist->rows; row++)
    {
      item = (GbMenuItemData *) gtk_clist_get_row_data (clist, row);
      if (item->generate_handler)
	{
	  handler = generate_handler (menued, row, item->label, item->name);
	  if (item_property_changed (handler, item->handler))
	    {
	      g_free (item->handler);
	      item->handler = handler;
	      gtk_clist_set_text (clist, row, GLD_COL_HANDLER, handler);
	    }
	  else
	    {
	      g_free (handler);
	    }
	}
    }
}


/* This returns TRUE id the item in the given row is a parent, or FALSE
   if the row doesn't exist or isn't a parent. */
static gboolean
is_parent (GladeMenuEditor *menued,
	   gint row)
{
  GtkCList *clist;
  GbMenuItemData *item, *next_item;

  clist = GTK_CLIST (menued->clist);
  if (row < 0 || row >= clist->rows - 1)
    return FALSE;
  item = (GbMenuItemData *) gtk_clist_get_row_data (clist, row);
  next_item = (GbMenuItemData *) gtk_clist_get_row_data (clist, row + 1);
  if (next_item->level > item->level)
    return TRUE;
  return FALSE;
}


/**************************************************************************
 * Public functions
 **************************************************************************/

/* This updates the given widget, based on the settings in the menu editor.
   It removes all the current children of the menu and recreates it. */
void
glade_menu_editor_update_menu (GladeMenuEditor    *menued,
			       GtkMenuShell	  *menu)
{
  GbMenuItemData *item;
  GtkWidget *menuitem, *prev_item = NULL, *child_menu;
  GtkCList *clist;
  GtkMenuShell *current_menu;
  GList *menus;
  GHashTable *group_hash;
  gchar *name;
  gint i, level;
  GbWidgetData *wdata;
#ifdef GLD_HAVE_GTK_1_1
  GtkAccelGroup *accel_group;
#else
  GtkAcceleratorTable *accelerator_table;
#endif

  /* Remove existing children of the menu. */
  while (menu->children)
    {
      gtk_container_remove (GTK_CONTAINER (menu),
			    GTK_WIDGET (menu->children->data));
    }

  /* This seems to be necessary to re-initialise the menu. I don't know why. */
  menu->menu_flag = TRUE;

  /* Now add widgets according to the items in the menu editor clist. */
  level = 0;
  menus = g_list_append (NULL, menu);
  group_hash = g_hash_table_new (g_str_hash, g_str_equal);
  clist = GTK_CLIST (menued->clist);
  for (i = 0; i < clist->rows; i++)
    {
      item = (GbMenuItemData *) gtk_clist_get_row_data (clist, i);

      if (item->level > level)
	{
	  child_menu = gb_widget_new ("GtkMenu", NULL);
	  /* We use the menus GList as a stack, pushing menus onto the
	     front. */
	  menus = g_list_prepend (menus, child_menu);
	  gtk_menu_item_set_submenu (GTK_MENU_ITEM (prev_item), child_menu);
	  level = item->level;
	}
      while (item->level < level)
	{
	  /* This removes/pops the first menu in the list. */
	  menus = g_list_remove_link (menus, menus);
	  level--;
	}
      current_menu = GTK_MENU_SHELL (menus->data);

      if (item->label && strlen (item->label) > 0)
	{
	  if (item->type == GB_MENU_ITEM_NORMAL)
	    menuitem = gtk_menu_item_new_with_label (item->label);
	  else if (item->type == GB_MENU_ITEM_CHECK)
	    {
	      menuitem = gtk_check_menu_item_new_with_label (item->label);
	      if (item->active)
		gtk_check_menu_item_set_state (GTK_CHECK_MENU_ITEM (menuitem),
					       TRUE);
	    }
	  else
	    {
	      menuitem = create_radio_menu_item (current_menu, item->label,
						 item->group_name, group_hash);
	      if (item->active)
		gtk_check_menu_item_set_state (GTK_CHECK_MENU_ITEM (menuitem),
					       TRUE);
	    }
	}
      else
	{
	  /* This creates a separator. */
	  menuitem = gtk_menu_item_new ();
	}
      gtk_widget_show (menuitem);

      if (item->right_justify)
	gtk_menu_item_right_justify (GTK_MENU_ITEM (menuitem));


      /* Turn it into a GbWidget, and add the 'activate' handler and the
	 accelerator. */
      name = item->name;
      if (name == NULL || strlen (name) == 0)
	name = gb_widget_new_name ("menuitem");

      /* If the name and handler are auto-generated, we need to get a real ID
	 now. */
      if (item->generate_name)
	{
	  g_free (item->name);
	  item->name = generate_name (item->label, TRUE);
	  gtk_clist_set_text (clist, i, GLD_COL_NAME,
			      item->name ? item->name : "");
	  if (item->generate_handler)
	    {
	      g_free (item->handler);
	      item->handler = generate_handler (menued, i, item->label,
						item->name);
	      gtk_clist_set_text (clist, i, GLD_COL_HANDLER,
				  item->handler ? item->handler : "");
	    }
	}
      item->generate_name = FALSE;
      item->generate_handler = FALSE;

      /* We can't use gb_widget_create_from () here, since we don't want the
	 name recreated. */
      gtk_widget_set_name (menuitem, item->name);
      wdata = gb_widget_new_widget_data ();
      gb_widget_real_initialize (menuitem, wdata);

      if (item->active)
	wdata->flags |= GB_ACTIVE;
      if (item->handler && strlen (item->handler) > 0)
	{
	  GbSignal *signal = g_new (GbSignal, 1);
	  signal->name = g_strdup ("activate");
	  signal->handler = g_strdup (item->handler);
	  signal->object = NULL;
	  signal->after = FALSE;
	  signal->data = NULL;
	  wdata->signals = g_list_append (wdata->signals, signal);
	}
      if (item->key && strlen (item->key) > 0)
	{
	  GbAccelerator *accel = g_new (GbAccelerator, 1);
	  guint key;
	  accel->modifiers = item->modifiers;
	  accel->key = g_strdup (item->key);
	  accel->signal = g_strdup ("activate");
	  wdata->accelerators = g_list_append (wdata->accelerators, accel);

	  /* We can only add accelerators to menus, not menubars. */
	  if (GTK_IS_MENU (current_menu))
	    {
	      key = glade_keys_dialog_find_key (item->key);
#ifdef GLD_HAVE_GTK_1_1
	      accel_group = GTK_MENU (current_menu)->accel_group;
	      gtk_widget_add_accelerator (menuitem, "activate", accel_group,
					  key, item->modifiers,
					  GTK_ACCEL_VISIBLE);
#else
	      accelerator_table = GTK_MENU (current_menu)->accelerator_table;
	      gtk_widget_install_accelerator (menuitem, accelerator_table,
					      "activate", key,
					      item->modifiers);
#endif
	    }
	}

      /* Add the menuitem to the current menu. */
      gtk_menu_shell_append (current_menu, menuitem);
      prev_item = menuitem;
    }
  g_list_free (menus);
  g_hash_table_destroy (group_hash);

  /* Make sure the displayed item is correct. */
  show_item_properties (menued);
}


/* This tries to find the radio group given a group name.
   If the group name is not NULL, it looks it up in the hash, which contains
   the first widget from each group. If the group doesn't exist yet it adds
   this menuitem to the hash and returns NULL.
   If the group name is NULL, it looks for the first radio menuitem in the
   given menu and uses its group. */
static GtkWidget*
create_radio_menu_item (GtkMenuShell *menu,
			gchar *label,
			gchar *group_name,
			GHashTable *group_hash)
{
  GSList *group = NULL;
  GtkWidget *menuitem;
  GtkWidget *group_widget;
  gboolean insert_into_hash = FALSE;
  GList *child;

  if (group_name)
    {
      group_widget = (GtkWidget*) (g_hash_table_lookup (group_hash,
							group_name));
      if (group_widget)
	group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (group_widget));
      else
	insert_into_hash = TRUE;
    }
  else
    {
      child = menu->children;
      while (child)
	{
	  if (GTK_IS_RADIO_MENU_ITEM (child->data))
	    {
	      group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (child->data));
	      break;
	    }
	  child = child->next;
	}
    }

  menuitem = gtk_radio_menu_item_new_with_label (group, label);
  if (insert_into_hash)
    g_hash_table_insert (group_hash, group_name, menuitem);
  /* FIXME: should use Group from gbradiomenuitem.c. */
  g_free (gtk_object_get_data (GTK_OBJECT (menuitem),
			       "GtkRadioMenuItem::group"));
  gtk_object_set_data (GTK_OBJECT (menuitem), "GtkRadioMenuItem::group",
		       g_strdup (group_name));

  return menuitem;
}


/* This converts the given menu/menubar into the list of items displayed in
   the menu editor. */
void
glade_menu_editor_set_menu	 (GladeMenuEditor *menued,
				  GtkMenuShell    *menu)
{
  /* First clear any current items and radio groups. */
  glade_menu_editor_reset (menued);

  /* Now add each of the menus/menuitems in the given menu. */
  set_submenu (menued, menu, 0);
  update_radio_groups (menued);
}

/* This recursively adds menus. */
static void
set_submenu (GladeMenuEditor *menued,
	     GtkMenuShell    *menu,
	     gint	      level)
{
  GbMenuItemData *item;
  GtkWidget *menuitem;
  GList *child, *tmp_list;
  gchar *label;
  GbWidgetData *wdata;

  child = menu->children;
  while (child)
    {
      if (!GTK_IS_MENU_ITEM (child->data))
	{
	  g_warning ("Menu widget is not a menu item");
	  continue;
	}
      menuitem = GTK_WIDGET (child->data);
      wdata = (GbWidgetData*) gtk_object_get_data (GTK_OBJECT(menuitem),
						   GB_WIDGET_DATA_KEY);
      if (wdata == NULL)
	{
	  g_warning ("Menu widget has no GbWidgetData");
	  continue;
	}

      item = g_new (GbMenuItemData, 1);
      item->name = g_strdup (gtk_widget_get_name (menuitem));
      item->label = NULL;
      item->handler = NULL;
      item->icon = NULL;
      item->type = GB_MENU_ITEM_NORMAL;
      item->right_justify = FALSE;
      item->active = FALSE;
      item->group_name = NULL;
      item->modifiers = 0;
      item->key = NULL;
      item->level = level;
      item->generate_name = FALSE;
      item->generate_handler = FALSE;

      if (GTK_IS_RADIO_MENU_ITEM (menuitem))
	{
	  item->type = GB_MENU_ITEM_RADIO;
	  /* FIXME: should use Group from gbradiomenuitem.c. */
	  item->group_name = g_strdup (gtk_object_get_data (GTK_OBJECT (menuitem), "GtkRadioMenuItem::group"));
	}
      else if (GTK_IS_CHECK_MENU_ITEM (menuitem))
	item->type = GB_MENU_ITEM_CHECK;

      if (GTK_BIN (menuitem)->child
	  && GTK_IS_LABEL (GTK_BIN (menuitem)->child))
	{
	  gtk_label_get (GTK_LABEL (GTK_BIN (menuitem)->child), &label);
	  item->label = g_strdup (label);
	}

      if (GTK_IS_CHECK_MENU_ITEM (menuitem)
	  && GTK_CHECK_MENU_ITEM (menuitem)->active)
	item->active = TRUE;
	
      if (GTK_MENU_ITEM (menuitem)->right_justify)
	item->right_justify = TRUE;

      /* Find 'activate' handler in widget data. */
      tmp_list = wdata->signals;
      while (tmp_list)
	{
	  GbSignal *signal = (GbSignal *) tmp_list->data;
	  if (!strcmp (signal->name, "activate"))
	    {
      item->handler = g_strdup (signal->handler);
	      break;
	    }
	  tmp_list = tmp_list->next;
	}

      /* Find 'activate' accelerator in widget data. */
      tmp_list = wdata->accelerators;
      while (tmp_list)
	{
	  GbAccelerator *accel = (GbAccelerator *) tmp_list->data;
	  if (!strcmp (accel->signal, "activate"))
	    {
	      item->key = g_strdup (accel->key);
	      item->modifiers = accel->modifiers;
	      break;
	    }
	  tmp_list = tmp_list->next;
	}


      insert_item (GTK_CLIST (menued->clist), item, -1);

      if (GTK_MENU_ITEM (menuitem)->submenu)
	{
	  set_submenu (menued,
		       GTK_MENU_SHELL (GTK_MENU_ITEM (menuitem)->submenu),
		       level + 1);
	}

      child = child->next;
    }
}


/* This clears the clist, freeing all GbMenuItemDatas, and resets the radio
   group combo, freeing all the current radio groups. */
static void
glade_menu_editor_reset (GladeMenuEditor *menued)
{
  GbMenuItemData *item;
  gint i;

  for (i = 0; i < GTK_CLIST (menued->clist)->rows; i++)
    {
      item = (GbMenuItemData *) gtk_clist_get_row_data (GTK_CLIST (menued->clist), i);
      g_free (item->name);
      g_free (item->label);
      g_free (item->handler);
      g_free (item->icon);
      g_free (item->group_name);
      g_free (item->key);
      g_free (item);
    }
  gtk_clist_clear (GTK_CLIST (menued->clist));

  gtk_list_clear_items (GTK_LIST (GTK_COMBO (menued->group_combo)->list),
			0, -1);
}
