
/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include "../gb.h"
#include "../glade_keys_dialog.h"

/* Include the 21x21 icon pixmap for this widget, to be used in the palette */
/* This widget has no icon as it doesn't appear in the palette */
/*#include "../graphics/" */

/*
 * This is the GbWidget struct for this widget (see ../gbwidget.h).
 * It is initialized in the init() function at the end of this file
 */
static GbWidget gbwidget;

static gchar *Label = "MenuItem|GtkItem::label";
static gchar *Justify = "GtkMenuItem::right_justify";


/******
 * NOTE: To use these functions you need to uncomment them AND add a pointer
 * to the function in the GbWidget struct at the end of this file.
 ******/

/*
 * Creates a new GtkWidget of class GtkMenuItem, performing any specialized
 * initialization needed for the widget to work correctly in this environment.
 * If a dialog box is used to initialize the widget, return NULL from this
 * function, and call data->callback with your new widget when it is done.
 * If the widget needs a special destroy handler, add a signal here.
 */
GtkWidget *
gb_menu_item_new (GbWidgetNewData * data)
{
  GtkWidget *new_widget;

  if (data->action == GB_CREATING)
    new_widget = gtk_menu_item_new_with_label (data->name);
  else
    new_widget = gtk_menu_item_new ();
  return new_widget;
}



/*
 * Creates the components needed to edit the extra properties of this widget.
 */
static void
gb_menu_item_create_properties (GtkWidget * widget, GbWidgetCreateArgData * data)
{
  property_add_text (Label, _("Label:"), _("The text to display"), 2);
  property_add_bool (Justify, _("Right Justify:"),
		     _("If the menu item is right-justified"));
}



/*
 * Gets the properties of the widget. This is used for both displaying the
 * properties in the property editor, and also for saving the properties.
 */
static void
gb_menu_item_get_properties (GtkWidget * widget, GbWidgetGetArgData * data)
{
  gb_widget_output_child_label (widget, data, Label);
  gb_widget_output_bool (data, Justify, GTK_MENU_ITEM (widget)->right_justify);
}



/*
 * Sets the properties of the widget. This is used for both applying the
 * properties changed in the property editor, and also for loading.
 */
static void
gb_menu_item_set_properties (GtkWidget * widget, GbWidgetSetArgData * data)
{
  gboolean justify;
  GList *tmp_list;
#ifdef GLD_HAVE_GTK_1_1
  GtkAccelGroup *accel_group;
#else
  GtkAcceleratorTable *accelerator_table;
#endif
  guint key;

  gb_widget_input_child_label (widget, data, Label);

  justify = gb_widget_input_bool (data, Justify);
  /* GtkMenuItem doesn't have a function for setting/clearing right-justify */
  if (data->apply)
    {
      GTK_MENU_ITEM (widget)->right_justify = justify;
      gtk_widget_queue_resize (widget);
    }

  /* FIXME: should this be somewhere else? */
  /* If we are loading, install the 'activate' accelerator, if it has one, so
     that is is visible. */
  if (data->action == GB_LOADING)
    {
      tmp_list = data->accelerators;
      while (tmp_list)
	{
	  GbAccelerator *accel = (GbAccelerator *) tmp_list->data;
	  if (!strcmp (accel->signal, "activate"))
	    {
	      key = glade_keys_dialog_find_key (accel->key);
#ifdef GLD_HAVE_GTK_1_1
	      accel_group = GTK_MENU (widget->parent)->accel_group;
	      gtk_widget_add_accelerator (widget, "activate", accel_group, key,
					  accel->modifiers, GTK_ACCEL_VISIBLE);
#else
	      accelerator_table = GTK_MENU (widget->parent)->accelerator_table;
	      gtk_widget_install_accelerator (widget, accelerator_table,
					      "activate", key, accel->modifiers);
#endif
	      break;
	    }
	  tmp_list = tmp_list->next;
	}
    }
}



/*
 * Adds menu items to a context menu which is just about to appear!
 * Add commands to aid in editing a GtkMenuItem, with signals pointing to
 * other functions in this file.
 */
static void
gb_menu_item_create_popup_menu (GtkWidget * widget, GbWidgetCreateMenuData * data)
{
  GtkWidget *menuitem;

  if (GTK_IS_LABEL (GTK_BIN (widget)->child))
    {
      menuitem = gtk_menu_item_new_with_label (_("Remove Label"));
      gtk_widget_show (menuitem);
      gtk_menu_append (GTK_MENU (data->menu), menuitem);
      gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
			  GTK_SIGNAL_FUNC (gb_widget_remove_label), widget);
    }
}



/*
 * Writes the source code needed to create this widget.
 * You have to output everything necessary to create the widget here, though
 * there are some convenience functions to help.
 */
static void
gb_menu_item_write_source (GtkWidget * widget, GbWidgetWriteSourceData * data)
{
  GtkWidget *child = GTK_BIN (widget)->child;
  gchar *label_text;

  if (child && GTK_IS_LABEL (child) && !GB_IS_GB_WIDGET (child))
    {
      gtk_label_get (GTK_LABEL (child), &label_text);
      source_add (data, "  %s = gtk_menu_item_new_with_label (\"%s\");\n",
		  data->wname, source_make_string (label_text));
    }
  else
    {
      source_add (data, "  %s = gtk_menu_item_new ();\n", data->wname);
    }

  gb_widget_write_standard_source (widget, data);

  if (GTK_MENU_ITEM (widget)->right_justify)
    source_add (data, "  gtk_menu_item_right_justify (GTK_MENU_ITEM (%s));\n",
		data->wname);
}



/*
 * Initializes the GbWidget structure.
 * I've placed this at the end of the file so we don't have to include
 * declarations of all the functions.
 */
GbWidget *
gb_menu_item_init ()
{
  /* Initialise the GTK type */
  gtk_menu_item_get_type ();

  /* Initialize the GbWidget structure */
  gb_widget_init_struct (&gbwidget);

  /* Fill in the pixmap struct & tooltip */
  gbwidget.pixmap_struct = NULL;
  gbwidget.tooltip = NULL;

  /* Fill in any functions that this GbWidget has */
  gbwidget.gb_widget_new = gb_menu_item_new;
  gbwidget.gb_widget_create_properties = gb_menu_item_create_properties;
  gbwidget.gb_widget_get_properties = gb_menu_item_get_properties;
  gbwidget.gb_widget_set_properties = gb_menu_item_set_properties;
  gbwidget.gb_widget_create_popup_menu = gb_menu_item_create_popup_menu;
  gbwidget.gb_widget_write_source = gb_menu_item_write_source;

  return &gbwidget;
}
