
/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include "../gb.h"

/* Include the 21x21 icon pixmap for this widget, to be used in the palette */
#include "../graphics/hbox.xpm"

/*
 * This is the GbWidget struct for this widget (see ../gbwidget.h).
 * It is initialized in the init() function at the end of this file
 */
static GbWidget gbwidget;

static gchar *Size = "HBox|GtkBox::size";
static gchar *Homogeneous = "HBox|GtkBox::homogeneous";
static gchar *Spacing = "HBox|GtkBox::spacing";


static void show_hbox_dialog (GbWidgetNewData * data);
static void on_hbox_dialog_ok (GtkWidget * widget, GbWidgetNewData * data);
static void on_hbox_dialog_destroy (GtkWidget * widget, GbWidgetNewData * data);

static void gb_hbox_insert_before (GtkWidget * menuitem, GtkWidget * child);
static void gb_hbox_insert_after (GtkWidget * menuitem, GtkWidget * child);

/******
 * NOTE: To use these functions you need to uncomment them AND add a pointer
 * to the function in the GbWidget struct at the end of this file.
 ******/

/*
 * Creates a new GtkWidget of class GtkHBox, performing any specialized
 * initialization needed for the widget to work correctly in this environment.
 * If a dialog box is used to initialize the widget, return NULL from this
 * function, and call data->callback with your new widget when it is done.
 * If the widget needs a special destroy handler, add a signal here.
 */
GtkWidget *
gb_hbox_new (GbWidgetNewData * data)
{
  GtkWidget *new_widget;

  if (data->action == GB_LOADING)
    {
      new_widget = gtk_hbox_new (FALSE, 0);
      return new_widget;
    }
  else
    {
      show_hbox_dialog (data);
      return NULL;
    }
}


static void
show_hbox_dialog (GbWidgetNewData * data)
{
  GtkWidget *window, *vbox, *hbox, *separator, *hbbox, *label, *spinbutton;
  GtkWidget *button;
  GtkObject *adjustment;

  window = gtk_window_new (GTK_WINDOW_DIALOG);
  gtk_window_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);
  gtk_window_set_title (GTK_WINDOW (window), _("New horizontal box"));

  gtk_signal_connect_object (GTK_OBJECT (window), "delete_event",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (window));
  gtk_signal_connect (GTK_OBJECT (window), "destroy",
		      GTK_SIGNAL_FUNC (on_hbox_dialog_destroy),
		      data);

  vbox = gtk_vbox_new (FALSE, 5);
  gtk_container_add (GTK_CONTAINER (window), vbox);
  /*gtk_container_border_width (GTK_CONTAINER (vbox), 10); */
  gtk_widget_show (vbox);

  hbox = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 5);
  gtk_container_border_width (GTK_CONTAINER (hbox), 10);
  gtk_widget_show (hbox);

  label = gtk_label_new (_("Number of columns:"));
  gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 5);
  gtk_widget_show (label);

  adjustment = gtk_adjustment_new (3, 1, 100, 1, 10, 10);
  spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
  /* save pointer to spinbutton so we can find it easily in the OK handler */
  gtk_object_set_data (GTK_OBJECT (window), "spinbutton", spinbutton);
  gtk_box_pack_start (GTK_BOX (hbox), spinbutton, TRUE, TRUE, 5);
  gtk_widget_set_usize (spinbutton, 50, -1);
  gtk_widget_grab_focus (spinbutton);
  gtk_widget_show (spinbutton);

  separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (vbox), separator, TRUE, TRUE, 5);
  gtk_widget_show (separator);

  hbbox = gtk_hbutton_box_new ();
  gtk_box_pack_start (GTK_BOX (vbox), hbbox, TRUE, TRUE, 5);
  gtk_container_border_width (GTK_CONTAINER (vbox), 10);
  gtk_widget_show (hbbox);

  button = gtk_button_new_with_label (_("OK"));
  gtk_container_add (GTK_CONTAINER (hbbox), button);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_hbox_dialog_ok),
		      data);

  button = gtk_button_new_with_label (_("Cancel"));
  gtk_container_add (GTK_CONTAINER (hbbox), button);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_show (button);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (window));

  gtk_widget_show (window);
  gtk_grab_add (window);
}


static void
on_hbox_dialog_ok (GtkWidget * widget, GbWidgetNewData * data)
{
  GtkWidget *new_widget, *spinbutton, *window, *placeholder;
  gint cols, i;

  window = gtk_widget_get_toplevel (widget);

  /* Only call callback if placeholder/fixed widget is still there */
  if (gb_widget_can_finish_new (data))
    {
      spinbutton = gtk_object_get_data (GTK_OBJECT (window), "spinbutton");
      g_return_if_fail (spinbutton != NULL);
      cols = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spinbutton));

      new_widget = gtk_hbox_new (FALSE, 0);
      for (i = 0; i < cols; i++)
	{
	  placeholder = editor_new_placeholder ();
	  /*gtk_widget_set_usize(placeholder, 60, 80); */
	  gtk_box_pack_start (GTK_BOX (new_widget), placeholder, TRUE, TRUE, 0);
	}
      gb_widget_initialize (new_widget, data);
      (*data->callback) (new_widget, data);
    }
  gtk_widget_destroy (window);
}


static void
on_hbox_dialog_destroy (GtkWidget * widget, GbWidgetNewData * data)
{
  gb_widget_free_new_data (data);
  gtk_grab_remove (widget);
}



/*
 * Creates the components needed to edit the extra properties of this widget.
 */
static void
gb_hbox_create_properties (GtkWidget * widget, GbWidgetCreateArgData * data)
{
  property_add_int_range (Size, _("Size:"), _("The number of widgets in the box"),
			  0, 1000, 1, 10, 1);
  property_add_bool (Homogeneous, _("Homogeneous:"),
		     _("If the children should be the same size"));
  property_add_int_range (Spacing, _("Spacing:"), _("The space between each child"),
			  0, 1000, 1, 10, 1);
}



/*
 * Gets the properties of the widget. This is used for both displaying the
 * properties in the property editor, and also for saving the properties.
 */
static void
gb_hbox_get_properties (GtkWidget * widget, GbWidgetGetArgData * data)
{
  if (data->action != GB_SAVING)
    gb_widget_output_int (data, Size, g_list_length (GTK_BOX (widget)->children));
  gb_widget_output_bool (data, Homogeneous, GTK_BOX (widget)->homogeneous);
  gb_widget_output_int (data, Spacing, GTK_BOX (widget)->spacing);
}



/*
 * Sets the properties of the widget. This is used for both applying the
 * properties changed in the property editor, and also for loading.
 */
static void
gb_hbox_set_properties (GtkWidget * widget, GbWidgetSetArgData * data)
{
  gboolean homogeneous;
  gint spacing, size;

  size = gb_widget_input_int (data, Size);
  if (data->apply)
    gb_widget_update_box_size (widget, size);

  homogeneous = gb_widget_input_bool (data, Homogeneous);
  if (data->apply)
    gtk_box_set_homogeneous (GTK_BOX (widget), homogeneous);

  spacing = gb_widget_input_int (data, Spacing);
  if (data->apply)
    gtk_box_set_spacing (GTK_BOX (widget), spacing);
}


/*
 * Adds menu items to a context menu which is just about to appear!
 * Add commands to aid in editing a GtkHBox, with signals pointing to
 * other functions in this file.
 */
static void
gb_hbox_create_popup_menu (GtkWidget * widget, GbWidgetCreateMenuData * data)
{
  GtkWidget *menuitem;

  g_return_if_fail (data->child != NULL);

  menuitem = gtk_menu_item_new_with_label (_("Insert Before"));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (data->menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (gb_hbox_insert_before), data->child);

  menuitem = gtk_menu_item_new_with_label (_("Insert After"));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (data->menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (gb_hbox_insert_after), data->child);
}


static void
gb_hbox_insert_before (GtkWidget * menuitem, GtkWidget * child)
{
  GtkWidget *box, *newchild;
  guint pos;
  gint expand, fill, padding;
  GtkPackType pack_type;

  box = child->parent;
  gtk_container_block_resize (GTK_CONTAINER (box));
  pos = gb_box_get_pos (GTK_BOX (box), child);
  g_return_if_fail (pos != -1);
  newchild = editor_new_placeholder ();
  gtk_box_query_child_packing (GTK_BOX (box), child,
			       &expand, &fill, &padding, &pack_type);
  if (pack_type == GTK_PACK_START)
    {
      gtk_box_pack_start (GTK_BOX (box), newchild, TRUE, TRUE, 0);
      gtk_box_reorder_child (GTK_BOX (box), newchild, pos);
    }
  else
    {
      gtk_box_pack_end (GTK_BOX (box), newchild, TRUE, TRUE, 0);
      gtk_box_reorder_child (GTK_BOX (box), newchild, pos + 1);
    }
  gtk_container_unblock_resize (GTK_CONTAINER (box));
}

static void
gb_hbox_insert_after (GtkWidget * menuitem, GtkWidget * child)
{
  GtkWidget *box, *newchild;
  guint pos;
  gint expand, fill, padding;
  GtkPackType pack_type;

  box = child->parent;
  gtk_container_block_resize (GTK_CONTAINER (box));
  pos = gb_box_get_pos (GTK_BOX (box), child);
  g_return_if_fail (pos != -1);
  newchild = editor_new_placeholder ();
  gtk_box_query_child_packing (GTK_BOX (box), child,
			       &expand, &fill, &padding, &pack_type);
  if (pack_type == GTK_PACK_START)
    {
      gtk_box_pack_start (GTK_BOX (box), newchild, TRUE, TRUE, 0);
      gtk_box_reorder_child (GTK_BOX (box), newchild, pos + 1);
    }
  else
    {
      gtk_box_pack_end (GTK_BOX (box), newchild, TRUE, TRUE, 0);
      gtk_box_reorder_child (GTK_BOX (box), newchild, pos);
    }
  gtk_container_unblock_resize (GTK_CONTAINER (box));
}


/*
 * Writes the source code needed to create this widget.
 * You have to output everything necessary to create the widget here, though
 * there are some convenience functions to help.
 */
static void
gb_hbox_write_source (GtkWidget * widget, GbWidgetWriteSourceData * data)
{
  if (data->create_widget)
    {
      source_add (data, "  %s = gtk_hbox_new (%s, %i);\n", data->wname,
		  GTK_BOX (widget)->homogeneous ? "TRUE" : "FALSE",
		  GTK_BOX (widget)->spacing);
    }

  gb_widget_write_standard_source (widget, data);
}



/*
 * Initializes the GbWidget structure.
 * I've placed this at the end of the file so we don't have to include
 * declarations of all the functions.
 */
GbWidget *
gb_hbox_init ()
{
  /* Initialise the GTK type */
  gtk_hbox_get_type ();

  /* Initialize the GbWidget structure */
  gb_widget_init_struct (&gbwidget);

  /* Fill in the pixmap struct & tooltip */
  gbwidget.pixmap_struct = hbox_xpm;
  gbwidget.tooltip = _("Horizontal Box");

  /* Fill in any functions that this GbWidget has */
  gbwidget.gb_widget_new = gb_hbox_new;
  gbwidget.gb_widget_create_properties = gb_hbox_create_properties;
  gbwidget.gb_widget_get_properties = gb_hbox_get_properties;
  gbwidget.gb_widget_set_properties = gb_hbox_set_properties;
  gbwidget.gb_widget_write_source = gb_hbox_write_source;
  gbwidget.gb_widget_create_popup_menu = gb_hbox_create_popup_menu;

  return &gbwidget;
}
