
/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>

#include <gtk/gtk.h>
#include "../gb.h"

/* Include the 21x21 icon pixmap for this widget, to be used in the palette */
#include "../graphics/combo.xpm"

/*
 * This is the GbWidget struct for this widget (see ../gbwidget.h).
 * It is initialized in the init() function at the end of this file
 */
static GbWidget gbwidget;

static gchar *Case = "GtkCombo::case_sensitive";
static gchar *Arrows = "GtkCombo::use_arrows";
static gchar *Always = "GtkCombo::use_arrows_always";
static gchar *Items = "GtkCombo::items";

/* Maximum length of text containing all combo popdown strings. */
#define	ITEMS_BUFFER_SIZE	8192

static void add_label (GtkWidget * widget, gchar * items);

static gint is_simple_combo (GtkWidget * widget);
static void is_simple_combo_callback (GtkWidget * widget,
				      gint * num_children);
static void write_items_source_callback (GtkWidget * item,
					 GbWidgetWriteSourceData * data);

/******
 * NOTE: To use these functions you need to uncomment them AND add a pointer
 * to the function in the GbWidget struct at the end of this file.
 ******/

/*
 * Creates a new GtkWidget of class GtkCombo, performing any specialized
 * initialization needed for the widget to work correctly in this environment.
 * If a dialog box is used to initialize the widget, return NULL from this
 * function, and call data->callback with your new widget when it is done.
 * If the widget needs a special destroy handler, add a signal here.
 */
/*
   GtkWidget*
   gb_combo_new(GbWidgetNewData *data)
   {

   }
 */



/*
 * Creates the components needed to edit the extra properties of this widget.
 */
static void
gb_combo_create_properties (GtkWidget * widget, GbWidgetCreateArgData * data)
{
  property_add_bool (Case, _("Case Sens.:"),
		     _("If the searching is case sensitive"));
  property_add_bool (Arrows, _("Use Arrows:"),
		     _("If arrows can be used to change the value"));
  property_add_bool (Always, _("Use Always:"),
		     _("If arrows work even if the value is not in the list"));
  property_add_text (Items, _("Items:"),
		     _("The items in the combo list, one per line"), 5);
}



/*
 * Gets the properties of the widget. This is used for both displaying the
 * properties in the property editor, and also for saving the properties.
 */
static void
gb_combo_get_properties (GtkWidget * widget, GbWidgetGetArgData * data)
{
  gchar items[ITEMS_BUFFER_SIZE];

  gb_widget_output_bool (data, Case, GTK_COMBO (widget)->case_sensitive);
  gb_widget_output_bool (data, Arrows, GTK_COMBO (widget)->use_arrows);
  gb_widget_output_bool (data, Always, GTK_COMBO (widget)->use_arrows_always);

  if (is_simple_combo (widget) >= 0)
    {
      /* Clear the buffer and make sure it's seen as valid. */
      items[0] = '\0';
      gtk_container_foreach (GTK_CONTAINER (GTK_COMBO (widget)->list),
			     (GtkCallback) add_label, &items);
      if (items[0] == -1)
	{
	  gb_widget_output_text (data, Items, "");
	  if (data->action == GB_SHOWING)
	    property_set_sensitive (Items, FALSE);
	}
      else
	{
	  gb_widget_output_text (data, Items, items);
	  if (data->action == GB_SHOWING)
	    property_set_sensitive (Items, TRUE);
	}
    }
}

/* Adds combo label text to buffer, or sets first char to -1 if it won't fit.
 */
static void
add_label (GtkWidget * widget, gchar * items)
{
  gchar *label_text;

  /* Return if list has already been marked invalid (i.e. it is too big). */
  if (items[0] == -1)
    return;

  gtk_label_get (GTK_LABEL (GTK_BIN (widget)->child), &label_text);
  if (strlen (items) + strlen (label_text) + 2 < ITEMS_BUFFER_SIZE)
    {
      strcat (items, label_text);
      strcat (items, "\n");
    }
  else
    items[0] = -1;
}


/* Returns number of items in combo list, or -1 if not a simple combo (not
   all children are labels). */
static gint
is_simple_combo (GtkWidget * widget)
{
  gint num_children = 0;

  gtk_container_foreach (GTK_CONTAINER (GTK_COMBO (widget)->list),
		     (GtkCallback) is_simple_combo_callback, &num_children);
  return num_children;
}


/* Sets flag to FALSE if list item's child widget is not a label. */
static void
is_simple_combo_callback (GtkWidget * widget, gint * num_children)
{
  if (*num_children == -1)
    return;
  if (!GTK_IS_LABEL (GTK_BIN (widget)->child))
    *num_children = -1;
  else
    *num_children += 1;
}


/*
 * Sets the properties of the widget. This is used for both applying the
 * properties changed in the property editor, and also for loading.
 */
static void
gb_combo_set_properties (GtkWidget * widget, GbWidgetSetArgData * data)
{
  gboolean case_sensitive, arrows, arrows_always;
  gchar *items;
  GList *item_list = NULL;

  case_sensitive = gb_widget_input_bool (data, Case);
  if (data->apply)
    gtk_combo_set_case_sensitive (GTK_COMBO (widget), case_sensitive);
  arrows = gb_widget_input_bool (data, Arrows);
  if (data->apply)
    gtk_combo_set_use_arrows (GTK_COMBO (widget), arrows);
  arrows_always = gb_widget_input_bool (data, Always);
  if (data->apply)
    gtk_combo_set_use_arrows_always (GTK_COMBO (widget), arrows_always);

  items = gb_widget_input_text (data, Items);
  if (data->apply)
    {
      gchar *pos = items;
      gchar *items_end = &items[strlen (items)];

      while (pos < items_end)
	{
	  gchar *item_end = strchr (pos, '\n');
	  if (item_end == NULL)
	    item_end = items_end;
	  *item_end = '\0';
	  item_list = g_list_append (item_list, pos);
	  pos = item_end + 1;
	}
      if (item_list)
	gtk_combo_set_popdown_strings (GTK_COMBO (widget), item_list);
      else
	gtk_list_clear_items (GTK_LIST (GTK_COMBO (widget)->list), 0, -1);
      g_list_free (item_list);
    }
  if (data->action == GB_APPLYING)
    g_free (items);
}



/*
 * Adds menu items to a context menu which is just about to appear!
 * Add commands to aid in editing a GtkCombo, with signals pointing to
 * other functions in this file.
 */
/*
   static void
   gb_combo_create_popup_menu(GtkWidget *widget, GbWidgetCreateMenuData *data)
   {

   }
 */



/*
 * Writes the source code needed to create this widget.
 * You have to output everything necessary to create the widget here, though
 * there are some convenience functions to help.
 */
static void
gb_combo_write_source (GtkWidget * widget, GbWidgetWriteSourceData * data)
{
  if (data->create_widget)
    {
      source_add (data, "  %s = gtk_combo_new ();\n", data->wname);
    }

  gb_widget_write_standard_source (widget, data);

  if (GTK_COMBO (widget)->case_sensitive)
    {
      source_add (data, "  gtk_combo_set_case_sensitive (GTK_COMBO (%s), TRUE);\n",
		  data->wname);
    }
  if (!GTK_COMBO (widget)->use_arrows)
    {
      source_add (data, "  gtk_combo_set_use_arrows (GTK_COMBO (%s), FALSE);\n",
		  data->wname);
    }
  if (GTK_COMBO (widget)->use_arrows_always)
    {
      source_add (data,
	      "  gtk_combo_set_use_arrows_always (GTK_COMBO (%s), TRUE);\n",
		  data->wname);
    }

  if (is_simple_combo (widget) > 0)
    {
      source_add_decl (data, "  GList *%s_items = NULL;\n", data->wname);
      gtk_container_foreach (GTK_CONTAINER (GTK_COMBO (widget)->list),
			   (GtkCallback) write_items_source_callback, data);
      source_add (data,
	    "  gtk_combo_set_popdown_strings (GTK_COMBO (%s), %s_items);\n",
		  data->wname, data->wname);
      source_add (data, "  g_list_free (%s_items);\n", data->wname);
    }
}


static void
write_items_source_callback (GtkWidget * item, GbWidgetWriteSourceData * data)
{
  gchar *label_text;
  gtk_label_get (GTK_LABEL (GTK_BIN (item)->child), &label_text);
  source_add (data, "  %s_items = g_list_append (%s_items, \"%s\");\n",
	      data->wname, data->wname, source_make_string (label_text));
}


/*
 * Initializes the GbWidget structure.
 * I've placed this at the end of the file so we don't have to include
 * declarations of all the functions.
 */
GbWidget *
gb_combo_init ()
{
  /* Initialise the GTK type */
  gtk_combo_get_type ();

  /* Initialize the GbWidget structure */
  gb_widget_init_struct (&gbwidget);

  /* Fill in the pixmap struct & tooltip */
  gbwidget.pixmap_struct = combo_xpm;
  gbwidget.tooltip = _("Combo Box");

  /* Fill in any functions that this GbWidget has */
  gbwidget.gb_widget_create_properties = gb_combo_create_properties;
  gbwidget.gb_widget_get_properties = gb_combo_get_properties;
  gbwidget.gb_widget_set_properties = gb_combo_set_properties;
  gbwidget.gb_widget_write_source = gb_combo_write_source;
/*
   gbwidget.gb_widget_new               = gb_combo_new;
   gbwidget.gb_widget_create_popup_menu = gb_combo_create_popup_menu;
 */

  return &gbwidget;
}
