/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>

#include <gdk/gdkx.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "gladeconfig.h"

#include "project.h"
#include "palette.h"
#include "property.h"
#include "gbwidget.h"
#include "utils.h"
#include "editor.h"
#include "tree.h"

/* The pixmap to use for placeholders */
#include "graphics/placeholder.xpm"


#define MIN_WIDGET_WIDTH		16
#define MIN_WIDGET_HEIGHT		16
#define MAX_INITIAL_WIDGET_WIDTH	200
#define MAX_INITIAL_WIDGET_HEIGHT	200
#define DEFAULT_WIDGET_WIDTH		50
#define DEFAULT_WIDGET_HEIGHT		50

#define PLACEHOLDER_WIDTH  32
#define PLACEHOLDER_HEIGHT 16


/* The grid (for fixed containers) */
#define GB_GRID_DOTS	1
#define GB_GRID_LINES	2
gboolean editor_show_grid = TRUE;
static gint editor_grid_horz_spacing = 8;
static gint editor_grid_vert_spacing = 8;
static gboolean editor_grid_style = GB_GRID_DOTS;


/* Snapping to the grid */
#define GB_SNAP_TOP	1 << 1
#define GB_SNAP_BOTTOM	1 << 2
#define GB_SNAP_LEFT	1 << 3
#define GB_SNAP_RIGHT	1 << 4
/*#define GB_SNAP_CENTER        1 << 5  maybe in future */
gboolean editor_snap_to_grid = TRUE;
static gint editor_snap_to_grid_x = GB_SNAP_LEFT | GB_SNAP_RIGHT;
static gint editor_snap_to_grid_y = GB_SNAP_TOP | GB_SNAP_BOTTOM;


/* Dragging (in a fixed container) - remembers which part of the widget is
   being dragged, the offset of the mouse (used when moving) and the initial
   widget rectangle (used when resizing) */
#define GB_DRAG_NONE    1
#define GB_TOP_LEFT     2
#define GB_TOP_RIGHT    3
#define GB_BOTTOM_LEFT  4
#define GB_BOTTOM_RIGHT 5
#define GB_MIDDLE       6
static gint drag_action;
static GtkWidget *dragging_widget = NULL;
static gint drag_offset_x;
static gint drag_offset_y;
static gint drag_widget_x1, drag_widget_y1, drag_widget_x2, drag_widget_y2;

/* Used when a widget is being moved around within a box or table. */
static GtkWidget *moving_widget = NULL;

/* The list of selected widgets */
static GList *selected_widgets = NULL;

/* The cursors used when selecting/adding/moving/resizing widgets */
static GdkCursor *cursor_selector;
static GdkCursor *cursor_add_widget;
static GdkCursor *cursor_add_to_fixed;
static GdkCursor *cursor_move;
static GdkCursor *cursor_top_left;
static GdkCursor *cursor_top_right;
static GdkCursor *cursor_bottom_left;
static GdkCursor *cursor_bottom_right;

/* Struct only used for find_child_at callback */
typedef struct _GbFindChildAtData GbFindChildAtData;
struct _GbFindChildAtData
  {
    gint x;
    gint y;
    GtkWidget *found_child;
  };


/* Static functions */
static void editor_on_widget_realize (GtkWidget *widget,
				      gpointer data);

static gint editor_on_key_press_event (GtkWidget * widget,
				       GdkEventKey * event,
				       gpointer data);
static gint editor_on_key_release_event (GtkWidget * widget,
					 GdkEventKey * event,
					 gpointer data);

static void add_mouse_signals_recursive (GtkWidget *widget,
					 gpointer data);
static gint editor_on_button_press (GtkWidget * signal_widget,
				    GdkEventButton * event,
				    gpointer data);
static gint editor_on_button_release (GtkWidget * widget,
				      GdkEvent * event,
				      gpointer data);

static gint editor_on_motion_notify (GtkWidget * signal_widget,
				     GdkEventMotion * event,
				     gpointer data);

static void placeholder_replace (GtkWidget * placeholder);
static void placeholder_finish_replace (GtkWidget * new_widget,
					GbWidgetNewData * data);

static gint get_notebook_page (GtkWidget * notebook,
			       GtkWidget * widget);

static void add_widget_to_fixed (GtkWidget * parent,
				 gint x,
				 gint y);
static void add_widget_to_fixed_finish (GtkWidget * new_widget,
					GbWidgetNewData * data);
#ifdef GLD_HAVE_GTK_1_1
static void add_widget_to_container (GtkWidget * parent);
static void add_widget_to_container_finish (GtkWidget * new_widget,
					    GbWidgetNewData * data);
#endif
static gint expose_widget (GtkWidget * widget,
			   GdkEventExpose * event,
			   GbWidgetData * wdata);
static gint draw_widget (GtkWidget * widget,
			 GdkRectangle * area,
			 gpointer data);
static void draw_widget_focus (GtkWidget * widget,
			       gpointer data);
static void draw_grid (GtkWidget * widget);
static void paint_widget (GtkWidget * widget);
static void paint_selection (GdkWindow * window,
			     GdkGC * gc,
			     gint x,
			     gint y,
			     gint width,
			     gint height);
static void clear_child_windows (GdkWindow * window,
				 gint x,
				 gint y,
				 gint w,
				 gint h);

static void delete (GtkWidget * widget);
static void delete_placeholder (GtkWidget * widget);

static void on_grid_settings_ok (GtkWidget * widget,
				 gpointer data);
static void editor_redraw_component (GtkWidget * widget,
				     gpointer data);

static void on_snap_settings_ok (GtkWidget * widget,
				 gpointer data);
static gint snap_top_edge (gint y);
static gint snap_bottom_edge (gint y);
static gint snap_left_edge (gint x);
static gint snap_right_edge (gint x);

static gint get_position_in_widget (GtkWidget * widget,
				    gint x,
				    gint y);
static void raise_fixed_child (GtkWidget * widget);

static gint do_drag_action (GtkWidget * widget,
			    GdkEventMotion * event);
static void find_child_at (GtkWidget * widget,
			   GbFindChildAtData * data);

static void move_table_widgets (GtkWidget * table,
				GtkWidget * child1,
				GtkWidget * child2);

static void set_cursor_recursive (GdkWindow *window, GdkCursor *cursor);

void
editor_init ()
{
  /* Create all cursors needed */
  cursor_selector = gdk_cursor_new (GDK_TOP_LEFT_ARROW);
  cursor_add_widget = gdk_cursor_new (GDK_PLUS);
  cursor_add_to_fixed = gdk_cursor_new (GDK_TCROSS);
  cursor_move = gdk_cursor_new (GDK_FLEUR);
  cursor_top_left = gdk_cursor_new (GDK_TOP_LEFT_CORNER);
  cursor_top_right = gdk_cursor_new (GDK_TOP_RIGHT_CORNER);
  cursor_bottom_left = gdk_cursor_new (GDK_BOTTOM_LEFT_CORNER);
  cursor_bottom_right = gdk_cursor_new (GDK_BOTTOM_RIGHT_CORNER);
}


GtkWidget *
editor_new_window ()
{
  GtkWidget *window = gb_widget_new ("GtkWindow", NULL);
  g_return_val_if_fail (window != NULL, NULL);
  return window;
}


GtkWidget *
editor_new_dialog ()
{
  GtkWidget *window = gb_widget_new ("GtkDialog", NULL);
  g_return_val_if_fail (window != NULL, NULL);
  return window;
}


GtkWidget *
editor_new_filesel ()
{
  GtkWidget *window = gb_widget_new ("GtkFileSelection", NULL);
  g_return_val_if_fail (window != NULL, NULL);
  gtk_widget_show (window);
  return window;
}


GtkWidget *
editor_new_colorsel ()
{
  GtkWidget *window = gb_widget_new ("GtkColorSelectionDialog", NULL);
  g_return_val_if_fail (window != NULL, NULL);
  gtk_widget_show (window);
  return window;
}

#ifdef GLD_HAVE_GTK_1_1
GtkWidget *
editor_new_fontsel ()
{
  GtkWidget *window = gb_widget_new ("GtkFontSelectionDialog", NULL);
  g_return_val_if_fail (window != NULL, NULL);
  gtk_widget_show (window);
  return window;
}
#endif

GtkWidget *
editor_new_input_dialog ()
{
  GtkWidget *window = gb_widget_new ("GtkInputDialog", NULL);
  g_return_val_if_fail (window != NULL, NULL);
  gtk_widget_show (window);
  return window;
}


GtkWidget *
editor_new_popupmenu ()
{
  GtkWidget *component = gb_widget_new ("GtkMenu", NULL);
  g_return_val_if_fail (component != NULL, NULL);
  gtk_widget_set_name (component, gb_widget_new_name ("popupmenu"));
  return component;
}


gint
editor_close_window (GtkWidget * widget,
		     GdkEvent * event,
		     gpointer data)
{
  gint x, y;

  /* remember position of window for when it is used again */
  gdk_window_get_origin (widget->window, &x, &y);
  if (windows_x_offset >= 0)
    {
      x -= windows_x_offset;
      y -= windows_y_offset;
    }
  gtk_widget_hide (widget);
  gtk_widget_set_uposition (widget, x, y);
  return TRUE;
}


/*
 * Grid settings dialog
 */
gboolean
editor_get_show_grid ()
{
  return editor_show_grid;
}


void
editor_toggle_show_grid (GtkWidget * widget, gpointer data)
{
  editor_show_grid = GTK_CHECK_MENU_ITEM (widget)->active;
  project_foreach_component (editor_redraw_component, NULL);
}


void
editor_show_grid_settings_dialog (GtkWidget * widget, gpointer data)
{
  GtkWidget *window, *vbox, *table, *label, *button, *hbbox;
  GtkWidget *separator, *spinbutton;
  GtkObject *adjustment;
  GSList *group;

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), _("Grid Options"));
  gtk_window_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), vbox);
  gtk_container_border_width (GTK_CONTAINER (vbox), 10);
  gtk_widget_show (vbox);

  table = gtk_table_new (2, 3, FALSE);
  gtk_box_pack_start (GTK_BOX (vbox), table, TRUE, TRUE, 5);
  gtk_widget_show (table);

  label = gtk_label_new (_("Horizontal Spacing:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		    GTK_FILL, 0, 5, 5);
  gtk_widget_show (label);

  adjustment = gtk_adjustment_new (editor_grid_horz_spacing, 1, 1000, 1,
				   10, 10);
  spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 0, 0);
  gtk_table_attach (GTK_TABLE (table), spinbutton, 1, 3, 0, 1,
		    GTK_EXPAND | GTK_FILL, 0, 5, 5);
  gtk_widget_show (spinbutton);
  gtk_object_set_data (GTK_OBJECT (window), "spinbutton1", spinbutton);

  label = gtk_label_new (_("Vertical Spacing:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
		    GTK_FILL, 0, 5, 5);
  gtk_widget_show (label);

  adjustment = gtk_adjustment_new (editor_grid_vert_spacing, 1, 1000, 1,
				   10, 10);
  spinbutton = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 0, 0);
  gtk_table_attach (GTK_TABLE (table), spinbutton, 1, 3, 1, 2,
		    GTK_EXPAND | GTK_FILL, 0, 5, 5);
  gtk_widget_show (spinbutton);
  gtk_object_set_data (GTK_OBJECT (window), "spinbutton2", spinbutton);

  table = gtk_table_new (1, 3, FALSE);
  gtk_box_pack_start (GTK_BOX (vbox), table, TRUE, TRUE, 0);
  gtk_widget_show (table);

  label = gtk_label_new (_("Grid Style:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		    GTK_FILL, 0, 5, 5);
  gtk_widget_show (label);

  button = gtk_radio_button_new_with_label (NULL, _("Dots"));
  if (editor_grid_style == GB_GRID_DOTS)
    gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
  gtk_table_attach (GTK_TABLE (table), button, 1, 2, 0, 1,
		    GTK_EXPAND | GTK_FILL, 0, 5, 5);
  gtk_widget_show (button);
  gtk_object_set_data (GTK_OBJECT (window), "button1", button);
  group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));

  button = gtk_radio_button_new_with_label (group, _("Lines"));
  if (editor_grid_style == GB_GRID_LINES)
    gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
  gtk_table_attach (GTK_TABLE (table), button, 2, 3, 0, 1,
		    GTK_EXPAND | GTK_FILL, 0, 5, 1);
  gtk_widget_show (button);
  gtk_object_set_data (GTK_OBJECT (window), "button2", button);

  separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (vbox), separator, TRUE, TRUE, 5);
  gtk_widget_show (separator);

  /* Buttons at bottom */
  hbbox = gtk_hbutton_box_new ();
  gtk_box_pack_start (GTK_BOX (vbox), hbbox, TRUE, TRUE, 2);
  gtk_widget_show (hbbox);

  button = gtk_button_new_with_label (_("OK"));
  gtk_container_add (GTK_CONTAINER (hbbox), button);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_grid_settings_ok),
		      NULL);

  button = gtk_button_new_with_label (_("Cancel"));
  gtk_container_add (GTK_CONTAINER (hbbox), button);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_show (button);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (window));

  gtk_widget_show (window);
}


static void
on_grid_settings_ok (GtkWidget * widget, gpointer data)
{
  GtkWidget *window, *spinbutton, *button;

  window = gtk_widget_get_toplevel (widget);
  spinbutton = gtk_object_get_data (GTK_OBJECT (window), "spinbutton1");
  editor_grid_horz_spacing = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON
							       (spinbutton));
  spinbutton = gtk_object_get_data (GTK_OBJECT (window), "spinbutton2");
  editor_grid_vert_spacing = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON
							       (spinbutton));

  button = gtk_object_get_data (GTK_OBJECT (window), "button1");
  if (GTK_TOGGLE_BUTTON (button)->active)
    editor_grid_style = GB_GRID_DOTS;
  else
    editor_grid_style = GB_GRID_LINES;

  gtk_widget_destroy (window);

  /* redraw all windows */
  project_foreach_component (editor_redraw_component, NULL);
}


static void
editor_redraw_component (GtkWidget * widget, gpointer data)
{
  gtk_widget_queue_draw (widget);
}


/*
 * Snap settings
 */
gboolean
editor_get_snap_to_grid ()
{
  return editor_snap_to_grid;
}


void
editor_toggle_snap_to_grid (GtkWidget * widget, gpointer data)
{
  editor_snap_to_grid = GTK_CHECK_MENU_ITEM (widget)->active;
}


void
editor_show_snap_settings_dialog (GtkWidget * widget, gpointer data)
{
  GtkWidget *window, *vbox, *table, *label, *button, *hbbox, *separator;

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), _("Snap Options"));
  gtk_window_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), vbox);
  gtk_container_border_width (GTK_CONTAINER (vbox), 10);
  gtk_widget_show (vbox);

  table = gtk_table_new (4, 2, FALSE);
  gtk_box_pack_start (GTK_BOX (vbox), table, TRUE, TRUE, 5);
  gtk_widget_show (table);

  /* Horizontal snapping */
  label = gtk_label_new (_("Horizontal Snapping:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		    GTK_FILL, 0, 5, 5);
  gtk_widget_show (label);

  button = gtk_check_button_new_with_label (_("Left"));
  if (editor_snap_to_grid_x & GB_SNAP_LEFT)
    gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
  gtk_table_attach (GTK_TABLE (table), button, 1, 2, 0, 1,
		    GTK_EXPAND | GTK_FILL, 0, 5, 5);
  gtk_widget_show (button);
  gtk_object_set_data (GTK_OBJECT (window), "button1", button);

  button = gtk_check_button_new_with_label (_("Right"));
  if (editor_snap_to_grid_x & GB_SNAP_RIGHT)
    gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
  gtk_table_attach (GTK_TABLE (table), button, 1, 2, 1, 2,
		    GTK_EXPAND | GTK_FILL, 0, 5, 5);
  gtk_widget_show (button);
  gtk_object_set_data (GTK_OBJECT (window), "button2", button);

  /* Vertical snapping */
  label = gtk_label_new (_("Vertical Snapping:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
		    GTK_FILL, 0, 5, 5);
  gtk_widget_show (label);

  button = gtk_check_button_new_with_label (_("Top"));
  if (editor_snap_to_grid_y & GB_SNAP_TOP)
    gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
  gtk_table_attach (GTK_TABLE (table), button, 1, 2, 2, 3,
		    GTK_EXPAND | GTK_FILL, 0, 5, 5);
  gtk_widget_show (button);
  gtk_object_set_data (GTK_OBJECT (window), "button3", button);

  button = gtk_check_button_new_with_label (_("Bottom"));
  if (editor_snap_to_grid_y & GB_SNAP_BOTTOM)
    gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
  gtk_table_attach (GTK_TABLE (table), button, 1, 2, 3, 4,
		    GTK_EXPAND | GTK_FILL, 0, 5, 5);
  gtk_widget_show (button);
  gtk_object_set_data (GTK_OBJECT (window), "button4", button);


  separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX (vbox), separator, TRUE, TRUE, 5);
  gtk_widget_show (separator);

  /* Buttons at bottom */
  hbbox = gtk_hbutton_box_new ();
  gtk_box_pack_start (GTK_BOX (vbox), hbbox, TRUE, TRUE, 2);
  gtk_widget_show (hbbox);

  button = gtk_button_new_with_label (_("OK"));
  gtk_container_add (GTK_CONTAINER (hbbox), button);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_snap_settings_ok),
		      NULL);

  button = gtk_button_new_with_label (_("Cancel"));
  gtk_container_add (GTK_CONTAINER (hbbox), button);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_show (button);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (window));

  gtk_widget_show (window);
}


static void
on_snap_settings_ok (GtkWidget * widget, gpointer data)
{
  GtkWidget *window, *button;

  window = gtk_widget_get_toplevel (widget);
  editor_snap_to_grid_x = 0;
  editor_snap_to_grid_y = 0;
  button = gtk_object_get_data (GTK_OBJECT (window), "button1");
  if (GTK_TOGGLE_BUTTON (button)->active)
    editor_snap_to_grid_x |= GB_SNAP_LEFT;
  button = gtk_object_get_data (GTK_OBJECT (window), "button2");
  if (GTK_TOGGLE_BUTTON (button)->active)
    editor_snap_to_grid_x |= GB_SNAP_RIGHT;
  button = gtk_object_get_data (GTK_OBJECT (window), "button3");
  if (GTK_TOGGLE_BUTTON (button)->active)
    editor_snap_to_grid_y |= GB_SNAP_TOP;
  button = gtk_object_get_data (GTK_OBJECT (window), "button4");
  if (GTK_TOGGLE_BUTTON (button)->active)
    editor_snap_to_grid_y |= GB_SNAP_BOTTOM;

  gtk_widget_destroy (window);
}


static gint
snap_top_edge (gint y)
{
  if (editor_snap_to_grid && (editor_snap_to_grid_y & GB_SNAP_TOP))
    {
      y += editor_grid_vert_spacing / 2;
      y -= y % editor_grid_vert_spacing;
    }
  return y;
}


static gint
snap_bottom_edge (gint y)
{
  if (editor_snap_to_grid && (editor_snap_to_grid_y & GB_SNAP_BOTTOM))
    {
      y += editor_grid_vert_spacing / 2;
      y -= y % editor_grid_vert_spacing;
    }
  return y;
}


static gint
snap_left_edge (gint x)
{
  if (editor_snap_to_grid && (editor_snap_to_grid_x & GB_SNAP_LEFT))
    {
      x += editor_grid_horz_spacing / 2;
      x -= x % editor_grid_horz_spacing;
    }
  return x;
}


static gint
snap_right_edge (gint x)
{
  if (editor_snap_to_grid && (editor_snap_to_grid_x & GB_SNAP_RIGHT))
    {
      x += editor_grid_horz_spacing / 2;
      x -= x % editor_grid_horz_spacing;
    }
  return x;
}




GtkWidget *
editor_new_placeholder ()
{
  GtkWidget *placeholder = gtk_drawing_area_new ();
  gtk_widget_set_events (placeholder,
			 gtk_widget_get_events (placeholder)
			 | GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK
			 | GDK_BUTTON_RELEASE_MASK
			 | GDK_POINTER_MOTION_MASK | GDK_BUTTON1_MOTION_MASK);
  gtk_widget_set_usize (placeholder, PLACEHOLDER_WIDTH, PLACEHOLDER_HEIGHT);
  gtk_widget_show (placeholder);
  editor_add_draw_signals (placeholder);
  editor_add_mouse_signals (placeholder);
  gtk_object_set_data (GTK_OBJECT (placeholder), GB_PLACEHOLDER_KEY, "True");
  return placeholder;
}


static void
move_table_widgets (GtkWidget * table, GtkWidget * child1, GtkWidget * child2)
{
  GtkTableChild tchild1, tchild2, *old_tchild1, *old_tchild2;
  gint xoptions1, yoptions1, xoptions2, yoptions2;

  /* This switches the widgets around. I'm not sure it's really what
     we want, especially if we use drag-and-drop in future. */
  gtk_widget_ref (child1);
  gtk_widget_ref (child2);
  old_tchild1 = gb_table_find_child (GTK_TABLE (table), child1);
  old_tchild2 = gb_table_find_child (GTK_TABLE (table), child1);
  if (old_tchild1 && old_tchild2)
    {
      /* We have to copy the table child data as it will be freed. */
      memmove (&tchild1, old_tchild1, sizeof (GtkTableChild));
      memmove (&tchild2, old_tchild2, sizeof (GtkTableChild));

      xoptions1 = (tchild1.xexpand ? GTK_EXPAND : 0)
	| (tchild1.xshrink ? GTK_SHRINK : 0)
	| (tchild1.xfill ? GTK_FILL : 0);
      yoptions1 = (tchild1.yexpand ? GTK_EXPAND : 0)
	| (tchild1.yshrink ? GTK_SHRINK : 0)
	| (tchild1.yfill ? GTK_FILL : 0);
      xoptions2 = (tchild2.xexpand ? GTK_EXPAND : 0)
	| (tchild2.xshrink ? GTK_SHRINK : 0)
	| (tchild2.xfill ? GTK_FILL : 0);
      yoptions2 = (tchild2.yexpand ? GTK_EXPAND : 0)
	| (tchild2.yshrink ? GTK_SHRINK : 0)
	| (tchild2.yfill ? GTK_FILL : 0);

      gtk_container_remove (GTK_CONTAINER (table), child1);
      gtk_container_remove (GTK_CONTAINER (table), child2);

      /* Now we add them back - we switch the positions, but keep the
         same options for each widget. */
      gtk_table_attach (GTK_TABLE (table), child1,
			tchild2.left_attach, tchild2.right_attach,
			tchild2.top_attach, tchild2.bottom_attach,
			xoptions1, yoptions1,
			tchild1.xpadding, tchild1.ypadding);

      gtk_table_attach (GTK_TABLE (table), child2,
			tchild1.left_attach, tchild1.right_attach,
			tchild1.top_attach, tchild1.bottom_attach,
			xoptions2, yoptions2,
			tchild2.xpadding, tchild2.ypadding);
    }
  gtk_widget_unref (child1);
  gtk_widget_unref (child2);
}


static void
placeholder_replace (GtkWidget * placeholder)
{
  char *class_name;

  class_name = palette_get_widget_class ();
  g_return_if_fail (class_name != NULL);
  palette_reset_selection ();
  gb_widget_new_full (class_name, placeholder->parent, placeholder, 0, 0,
		      placeholder_finish_replace, GB_CREATING, NULL);
}


static void
placeholder_finish_replace (GtkWidget * new_widget, GbWidgetNewData * data)
{
  editor_clear_selection (NULL);
  /* Replace placeholder in parent container */
  gb_widget_replace_child (data->parent, data->current_child, new_widget);
  gb_widget_show_properties (new_widget);
  tree_add_widget (new_widget);
}





/* Note: returns last found child if children overlap */
static void
find_child_at (GtkWidget * widget, GbFindChildAtData * data)
{
#if 0
  g_print ("In find_child_at: %s X:%i Y:%i W:%i H:%i\n",
	   gtk_widget_get_name (widget),
	   widget->allocation.x, widget->allocation.y,
	   widget->allocation.width, widget->allocation.height);
#endif
  MSG1 ("In find_child_at widget:%s", gtk_widget_get_name (widget));
  /* Notebook pages are visible but not mapped if they are not showing. */
  if (GTK_WIDGET_VISIBLE (widget) && GTK_WIDGET_MAPPED (widget)
      && (widget->allocation.x <= data->x)
      && (widget->allocation.y <= data->y)
      && (widget->allocation.x + widget->allocation.width > data->x)
      && (widget->allocation.y + widget->allocation.height > data->y))
    {
#if 0
      g_print ("found child:%s", gtk_widget_get_name (widget));
#endif
      data->found_child = widget;
    }
}


/* Checks if point is in the given notebook's tabs. */
static void
find_notebook_tab (GtkWidget * widget, GbFindChildAtData * data)
{
  GList *children;
  GtkNotebookPage *page;

  children = GTK_NOTEBOOK (widget)->children;
  while (children)
    {
      page = children->data;
      if ((page->allocation.x <= data->x)
	  && (page->allocation.y <= data->y)
	  && (page->allocation.x + page->allocation.width > data->x)
	  && (page->allocation.y + page->allocation.height > data->y))
	{
	  /* Make sure this is a GbWidget. */
	  /*if (GB_IS_GB_WIDGET (page->tab_label))*/
	    data->found_child = page->tab_label;
	}
      children = children->next;
    }
}


/* This function is passed a widget which has received a mouse event, and
   the coordinates of that event. It returns the widget which the event is
   really meant for (which could be a descendent of the given widget), and
   the position of the event in the widget's allocated area. */
static GtkWidget *
editor_get_event_widget (GtkWidget *widget, GdkWindow *window, gint x, gint y,
			 gint * x_return, gint * y_return)
{
  GbFindChildAtData data;
  gint win_x, win_y;
  GtkWidget *found_gbwidget = NULL;
  gint found_x = 0, found_y = 0;
  GdkWindow *parent_window;

#if 0
  g_print ("\n\nOriginal:%s X:%i Y:%i\n", gtk_widget_get_name (widget), x, y);
  if (widget->parent)
    g_print ("  Parent: %s\n", gtk_widget_get_name (widget->parent));
#endif

  /* FIXME: GTK bug workaround? - need to translate coords if mouse button was
     pressed in a child window. */
  /* Remember widgets can have other windows besides their main one, and
     when dragging the event may be sent to the parent's window? */
  parent_window = widget->parent ? widget->parent->window : widget->window;
  while (window && window != parent_window)
    {
      gdk_window_get_position (window, &win_x, &win_y);
      x += win_x;
      y += win_y;
      window = gdk_window_get_parent (window);
    }
  if (window != parent_window)
    {
      MSG ("  editor_get_event_widget - unknown window");
      return NULL;
    }

#if 0
  g_print ("  Translated X:%i Y:%i\n", x, y);
#endif

  /* We now have correct coordinates relative to the parent's window.
     Now we find out which widget this event is really for.
     We step down the widget tree, trying to find the widget at the given
     position. We have to translate coordinates for children of widgets with
     windows. We may need to use bin_window for viewport. */
  if (GB_IS_GB_WIDGET (widget) || GB_IS_PLACEHOLDER (widget))
    {
#if 0
      g_print ("Found child:%s\n", gtk_widget_get_name (widget));
#endif
      found_gbwidget = widget;
      found_x = x;
      found_y = y;
    }
  if (!GTK_WIDGET_NO_WINDOW (widget) && widget->parent)
    {
      /* FIXME; use bin_window for viewport ? */
      window = widget->window;
      gdk_window_get_position (window, &win_x, &win_y);
      x -= win_x;
      y -= win_y;
#if 0
      g_print ("  Translated X:%i Y:%i\n", x, y);
#endif
	}
  for (;;)
    {
      if (!GTK_IS_CONTAINER (widget) || GTK_IS_MENU_BAR (widget))
	break;
      data.x = x;
      data.y = y;
      data.found_child = NULL;
      MSG ("...Finding child widget");
      gtk_container_foreach (GTK_CONTAINER (widget),
			     (GtkCallback) find_child_at, &data);
      /* SPECIAL CODE - Check for notebook tabs. */
      if (GTK_IS_NOTEBOOK (widget))
	find_notebook_tab (widget, &data);

      if (data.found_child)
	{
#if 0
	  g_print ("Found child:%s\n", gtk_widget_get_name (data.found_child));
#endif
	  widget = data.found_child;
	  if (GB_IS_GB_WIDGET (widget) || GB_IS_PLACEHOLDER (widget))
	    {
	      found_gbwidget = widget;
	      found_x = x;
	      found_y = y;
	    }
	}
      else
	break;

      if (!GTK_WIDGET_NO_WINDOW (widget))
	{
	  /* FIXME; use bin_window for viewport ? */
	  window = widget->window;
	  gdk_window_get_position (window, &win_x, &win_y);
	  x -= win_x;
	  y -= win_y;
#if 0
	  g_print ("  Translated X:%i Y:%i\n", x, y);
#endif
	}
    }

  if (!found_gbwidget)
    return NULL;

  *x_return = found_x - found_gbwidget->allocation.x;
  *y_return = found_y - found_gbwidget->allocation.y;

#if 0
  g_print ("  Event widget: %s X:%i Y:%i\n",
	   gtk_widget_get_name (found_gbwidget), *x_return, *y_return);
#endif

  return found_gbwidget;
}


static gint
editor_on_button_press (GtkWidget * signal_widget,
			GdkEventButton * event,
			gpointer user_data)
{
  GtkWidget *widget;
  GtkWidget *child1, *parent;
  gint x, y;

#if 0
  g_print ("In editor_on_button_press widget: %s Colormap:%p\n",
	   gtk_widget_get_name (signal_widget),
	   gdk_window_get_colormap (signal_widget->window));
#endif
  widget = editor_get_event_widget (signal_widget, event->window,
				    event->x, event->y, &x, &y);
  if (widget == NULL)
    return FALSE;

  MSG ("...Checking which button pressed");
  if (event->button == 1)
    {
      /* check if there's a widget being moved.
         Note: this will soon be replaced by drag-and-drop code. */
      if (moving_widget)
	{
	  child1 = widget;
	  parent = child1->parent;
	  while (parent && parent != moving_widget->parent)
	    {
	      child1 = parent;
	      parent = child1->parent;
	    }
	  if (parent)
	    {
	      if (GTK_IS_BOX (parent))
		{
		  /* This switches the widgets around. I'm not sure it's really
		     what we want, especially if we use drag-and-drop in
		     future. */
		  guint x1 = gb_box_get_pos (GTK_BOX (parent), child1);
		  guint x2 = gb_box_get_pos (GTK_BOX (parent), moving_widget);
		  gtk_box_reorder_child (GTK_BOX (parent), child1, x2);
		  gtk_box_reorder_child (GTK_BOX (parent), moving_widget, x1);
		}
	      else if (GTK_IS_TABLE (parent))
		{
		  move_table_widgets (parent, child1, moving_widget);
		}
	    }
	  moving_widget = NULL;
	  gtk_signal_emit_stop_by_name (GTK_OBJECT (signal_widget),
					"button_press_event");
	  return TRUE;
	}
      else if (palette_is_selector_on ())
	{
	  gboolean handled;
	  MSG ("...Selecting widget");
	  handled = editor_select_widget (widget, event->state, x, y);
	  if (handled)
	    gtk_signal_emit_stop_by_name (GTK_OBJECT (signal_widget),
					  "button_press_event");
	  return handled;
	}
      else if (GTK_IS_FIXED (widget))
	{
	  add_widget_to_fixed (widget, x, y);
	  gtk_signal_emit_stop_by_name (GTK_OBJECT (signal_widget),
					"button_press_event");
	  return TRUE;
	}
      else if (GB_IS_PLACEHOLDER (widget))
	{
	  placeholder_replace (widget);
	  gtk_signal_emit_stop_by_name (GTK_OBJECT (signal_widget),
					"button_press_event");
	}
#ifdef GLD_HAVE_GTK_1_1
      else if (GTK_IS_PACKER (widget))
	{
	  add_widget_to_container (widget);
	  gtk_signal_emit_stop_by_name (GTK_OBJECT (signal_widget),
					"button_press_event");
	}
#endif
    }
  else if (event->button == 3)
    {
      gb_widget_show_popup_menu (widget, event);
      gtk_signal_emit_stop_by_name (GTK_OBJECT (signal_widget),
				    "button_press_event");
      return TRUE;
    }
  return FALSE;
}


static gint
editor_on_button_release (GtkWidget * widget,
			  GdkEvent * event,
			  gpointer data)
{
  MSG ("In editor_on_button_release");
  if (dragging_widget)
    {
      dragging_widget = NULL;
      drag_action = GB_DRAG_NONE;
      gtk_grab_remove (widget);
    }
  return FALSE;
}


static void
add_widget_to_fixed (GtkWidget * parent, gint x, gint y)
{
  char *class_name;

  class_name = palette_get_widget_class ();
  g_return_if_fail (class_name != NULL);
  palette_reset_selection ();
  gb_widget_new_full (class_name, parent, NULL, x, y,
		      add_widget_to_fixed_finish, GB_CREATING, NULL);
}


static void
add_widget_to_fixed_finish (GtkWidget * new_widget, GbWidgetNewData * data)
{
  GtkRequisition requisition = {0, 0};
  GbWidgetData *wdata;
  gint x = data->x;
  gint y = data->y;
  gint w, h;

  /* I think we should do this first, in case the widget needs to be
     realized in order to calculate its requisition. */
  gtk_container_add (GTK_CONTAINER (data->parent), new_widget);
  gtk_widget_show (new_widget);

  gtk_widget_size_request (new_widget, &requisition);
  w = requisition.width;
  h = requisition.height;
  if (w < MIN_WIDGET_WIDTH || h < MIN_WIDGET_HEIGHT)
    {
      MSG3 ("Default widget size too small: %s W:%i H:%i\n",
	    gtk_widget_get_name (new_widget), w, h);
    }
  if (w > MAX_INITIAL_WIDGET_WIDTH || h > MAX_INITIAL_WIDGET_HEIGHT)
    {
      MSG3 ("Default widget size too big: %s W:%i H:%i\n",
	    gtk_widget_get_name (new_widget), w, h);
    }

  if (w == 0)
    w = DEFAULT_WIDGET_WIDTH;
  if (h == 0)
    h = DEFAULT_WIDGET_HEIGHT;
  w = MAX (w, MIN_WIDGET_WIDTH);
  w = MIN (w, MAX_INITIAL_WIDGET_WIDTH);
  h = MAX (h, MIN_WIDGET_HEIGHT);
  h = MIN (h, MAX_INITIAL_WIDGET_HEIGHT);

  /* FIXME: Make sure a gamma curve is a reasonable size. */
  if (GTK_IS_GAMMA_CURVE (new_widget))
    w = 80;

  /* Children of fixed widgets always have their width & height set
     explicitly. */
  wdata = gtk_object_get_data (GTK_OBJECT (new_widget), GB_WIDGET_DATA_KEY);
  g_return_if_fail (wdata != NULL);
  wdata->flags |= GB_WIDTH_SET | GB_HEIGHT_SET;

  /* Calculate real position & size according to grid & snapping */
  /* FIXME: only snaps top & left at present */
  if (editor_snap_to_grid)
    {
      if (editor_snap_to_grid_x & GB_SNAP_LEFT)
	{
	  /*x += editor_grid_horz_spacing / 2; */
	  x -= x % editor_grid_horz_spacing;
	}
      if (editor_snap_to_grid_y & GB_SNAP_TOP)
	{
	  /*y += editor_grid_vert_spacing / 2; */
	  y -= y % editor_grid_vert_spacing;
	}
    }

  gb_widget_set_usize (new_widget, w, h);
  gtk_widget_set_uposition (new_widget, x, y);
  gb_widget_show_properties (new_widget);
  tree_add_widget (new_widget);
}


#ifdef GLD_HAVE_GTK_1_1
static void
add_widget_to_container (GtkWidget * parent)
{
  char *class_name;

  class_name = palette_get_widget_class ();
  g_return_if_fail (class_name != NULL);
  palette_reset_selection ();
  gb_widget_new_full (class_name, parent, NULL, 0, 0,
		      add_widget_to_container_finish, GB_CREATING, NULL);
}


static void
add_widget_to_container_finish (GtkWidget * new_widget, GbWidgetNewData * data)
{
#if 0
  GtkRequisition requisition = {0, 0};
  GbWidgetData *wdata;
  gint x = data->x;
  gint y = data->y;
  gint w, h;
#endif

  /* I think we should do this first, in case the widget needs to be
     realized in order to calculate its requisition. */
  gtk_container_add (GTK_CONTAINER (data->parent), new_widget);
  gtk_widget_show (new_widget);

#if 0
  gtk_widget_size_request (new_widget, &requisition);
  w = requisition.width;
  h = requisition.height;
  if (w < MIN_WIDGET_WIDTH || h < MIN_WIDGET_HEIGHT)
    {
      MSG3 ("Default widget size too small: %s W:%i H:%i\n",
	    gtk_widget_get_name (new_widget), w, h);
    }
  if (w > MAX_INITIAL_WIDGET_WIDTH || h > MAX_INITIAL_WIDGET_HEIGHT)
    {
      MSG3 ("Default widget size too big: %s W:%i H:%i\n",
	    gtk_widget_get_name (new_widget), w, h);
    }

  if (w == 0)
    w = DEFAULT_WIDGET_WIDTH;
  if (h == 0)
    h = DEFAULT_WIDGET_HEIGHT;
  w = MAX (w, MIN_WIDGET_WIDTH);
  w = MIN (w, MAX_INITIAL_WIDGET_WIDTH);
  h = MAX (h, MIN_WIDGET_HEIGHT);
  h = MIN (h, MAX_INITIAL_WIDGET_HEIGHT);

  /* FIXME: Make sure a gamma curve is a reasonable size. */
  if (GTK_IS_GAMMA_CURVE (new_widget))
    w = 80;

  /* Children of fixed widgets always have their width & height set
     explicitly. */
  wdata = gtk_object_get_data (GTK_OBJECT (new_widget), GB_WIDGET_DATA_KEY);
  g_return_if_fail (wdata != NULL);
  wdata->flags |= GB_WIDTH_SET | GB_HEIGHT_SET;

  gb_widget_set_usize (new_widget, w, h);
  gtk_widget_set_uposition (new_widget, x, y);
#endif

  gb_widget_show_properties (new_widget);
  tree_add_widget (new_widget);
}
#endif

/*
 * Clears all currently selected widgets, except the given widget.
 * Returns TRUE if given widget is selected.
 */
gint
editor_clear_selection (GtkWidget * leave_widget)
{
  GList *child = selected_widgets, *next_child;
  GtkWidget *widget;
  gint selected = FALSE;

  while (child)
    {
      next_child = child->next;
      widget = child->data;
      if (widget == leave_widget)
	{
	  selected = TRUE;
	}
      else
	{
	  selected_widgets = g_list_remove (selected_widgets, widget);
	  tree_select_widget (widget, FALSE);
	  editor_refresh_widget_selection (widget);
	}
      child = next_child;
    }
  return selected;
}


void
editor_remove_widget_from_selection (GtkWidget * widget)
{
  selected_widgets = g_list_remove (selected_widgets, widget);
  tree_select_widget (widget, FALSE);
}


static gint
get_position_in_widget (GtkWidget * widget, gint x, gint y)
{
  gint width = widget->allocation.width;
  gint height = widget->allocation.height;
  if (x < GB_CORNER_WIDTH && y < GB_CORNER_HEIGHT)
    return GB_TOP_LEFT;
  if (x > width - GB_CORNER_WIDTH && y < GB_CORNER_HEIGHT)
    return GB_TOP_RIGHT;
  if (x < GB_CORNER_WIDTH && y > height - GB_CORNER_HEIGHT)
    return GB_BOTTOM_LEFT;
  if (x > width - GB_CORNER_WIDTH && y > height - GB_CORNER_HEIGHT)
    return GB_BOTTOM_RIGHT;
  return GB_MIDDLE;
}


static void
raise_fixed_child (GtkWidget * widget)
{
  GtkFixed *fixed;

  g_return_if_fail (GTK_IS_FIXED (widget->parent));
  fixed = GTK_FIXED (widget->parent);
  /* If widget hasn't got a window, move it to the back of the parent fixed's
     children. If it has got a window, raise it. */
  /* Note: this is slightly naughty as it changes the GtkFixed's GList of
     children, but it's better than removing the widget and adding it again. */
  if (GTK_WIDGET_NO_WINDOW (widget))
    {
      GList *child;
      GtkFixedChild *data;
      child = fixed->children;
      while (child)
	{
	  data = child->data;
	  if (data->widget == widget)
	    {
	      fixed->children = g_list_remove (fixed->children, data);
	      fixed->children = g_list_append (fixed->children, data);
	      break;
	    }
	  child = child->next;
	}
    }
  else
    {
      gdk_window_raise (widget->window);
    }
}


gboolean
editor_select_widget_control (GtkWidget * widget)
{
  /* If widget is currently selected, deslect it, else add it to
     selected. */
  if (g_list_find (selected_widgets, widget))
    {
      selected_widgets = g_list_remove (selected_widgets, widget);
    }
  else
    {
      gb_widget_show_properties (widget);
      selected_widgets = g_list_append (selected_widgets, widget);
      project_clear_component_selection ();
    }
  editor_refresh_widget_selection (widget);
  return TRUE;
}
 
GList *
editor_get_selection ()
{
  return g_list_first (selected_widgets);
}


gboolean
editor_select_widget (GtkWidget * widget, guint modifiers, gint x, gint y)
{
  gint already_selected, page;
  GtkWidget *select_widget, *ancestor;
  gboolean handled = FALSE;

#if 0
  g_print ("=+=+ widget: %s Alloc X:%i Y:%i W:%i H:%i\n",
	   gtk_widget_get_name (widget),
	   widget->allocation.x, widget->allocation.y,
	   widget->allocation.width, widget->allocation.height);
#endif

  /* reset any move/resize action */
  drag_action = GB_DRAG_NONE;

  /* Shift + selection. Step through parents of this widget, until a selected
     widget is found. Then clear selection & select that widgets parent.
     If no parents were selected, or the top-level parent was selected,
     select this widget. */
  if (modifiers & GDK_SHIFT_MASK)
    {
      gint found = FALSE;
      ancestor = widget;
      while (ancestor)
	{
	  if (g_list_find (selected_widgets, ancestor))
	    {
	      found = TRUE;
	      break;
	    }
	  ancestor = ancestor->parent;
	}
      if (found && ancestor->parent != NULL)
	{
	  select_widget = ancestor->parent;
	  /* If widget is not a GbWidget, i.e. it has no GbWidgetData,
	     skip it. */
	  while (select_widget)
	    {
	      if (GB_IS_GB_WIDGET (select_widget))
		break;
	      select_widget = select_widget->parent;
	    }
	}
      else
	{
	  select_widget = widget;
	}
      g_return_val_if_fail (select_widget != NULL, FALSE);
      gb_widget_show_properties (select_widget);
      already_selected = editor_clear_selection (select_widget);
      if (already_selected)
	return FALSE;
      else
	tree_clear_selection ();
      selected_widgets = g_list_append (selected_widgets, select_widget);
      tree_select_widget (select_widget, TRUE);
      project_clear_component_selection ();
      editor_refresh_widget_selection (select_widget);
      return TRUE;
    }

  /* Control + selection. If widget is currently selected, deslect it, else
     add it to the selected widgets. */
  if (modifiers & GDK_CONTROL_MASK)
    {
      /* If widget is currently selected, deslect it, else add it to
         selected. */
      if (g_list_find (selected_widgets, widget))
	tree_select_widget (widget, FALSE);
      else
	tree_select_widget (widget, TRUE);
      editor_refresh_widget_selection (widget);
      return editor_select_widget_control (widget);
    }

  /* Normal selection. If the widget is currently selected, just get the
     data for a possible resize/drag. If it is not selected, clear all
     currently selected widgets, then select this one.
     Also remember where the button press occurred, in case widget is being
     moved or resized (in a fixed container). */
  gb_widget_show_properties (widget);

  if (!g_list_find (selected_widgets, widget))
    {
      editor_clear_selection (NULL);
      tree_clear_selection ();
      selected_widgets = g_list_append (selected_widgets, widget);
      tree_select_widget (widget, TRUE);
      handled = TRUE;
      project_clear_component_selection ();
      /* If parent is a fixed container, move widget to front */
      if (widget->parent && GTK_IS_FIXED (widget->parent))
	raise_fixed_child (widget);

      /* SPECIAL CODE: If the widget or an ancestor is a notebook tab,
         show the page */
      ancestor = widget;
      while (ancestor->parent)
	{
	  if (GTK_IS_NOTEBOOK (ancestor->parent))
	    {
	      page = get_notebook_page (ancestor->parent, ancestor);
	      if (page != -1)
		gtk_notebook_set_page (GTK_NOTEBOOK (ancestor->parent), page);
	      break;
	    }
	  ancestor = ancestor->parent;
	}

      editor_refresh_widget_selection (widget);
    }


  /* NOTE: this will only work in a fixed */
  if (widget->parent && GTK_IS_FIXED (widget->parent))
    {
      drag_action = get_position_in_widget (widget, x, y);
      drag_offset_x = x;
      drag_offset_y = y;

      drag_widget_x1 = widget->allocation.x;
      drag_widget_y1 = widget->allocation.y;
      drag_widget_x2 = drag_widget_x1 + widget->allocation.width;
      drag_widget_y2 = drag_widget_y1 + widget->allocation.height;
      return TRUE;
    }
  return handled;
}


static gint
get_notebook_page (GtkWidget * notebook,
		   GtkWidget * widget)
{
  GList *children;
  GtkNotebookPage *page;
  gint page_num = 0;

  children = GTK_NOTEBOOK (notebook)->children;
  while (children)
    {
      page = children->data;
      if (page->tab_label == widget)
	return page_num;
      children = children->next;
      page_num++;
    }
  return -1;
}


static gint
do_drag_action (GtkWidget * widget, GdkEventMotion * event)
{
  gint mouse_x, mouse_y, new_x = 0, new_y = 0, new_width = 0, new_height = 0;
  GbWidgetData *wdata;

  MSG ("IN do_drag_action");
  if (dragging_widget == NULL)
    {
      dragging_widget = widget;
      gtk_grab_add (widget);
    }
  else
    {
      if (dragging_widget != widget)
	return FALSE;
    }

  wdata = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  g_return_val_if_fail (wdata != NULL, FALSE);

  gtk_widget_get_pointer (widget->parent, &mouse_x, &mouse_y);
  /* FIXME: Use these instead eventually, but may need to translate? */
  /*
     mouse_x = event->x;
     mouse_y = event->y;
   */

  gtk_container_block_resize (GTK_CONTAINER (widget->parent));

  switch (drag_action)
    {
    case GB_TOP_LEFT:
      new_x = snap_left_edge (mouse_x);
      new_y = snap_top_edge (mouse_y);
      new_width = drag_widget_x2 - new_x;
      new_height = drag_widget_y2 - new_y;
      if (new_width < MIN_WIDGET_WIDTH)
	{
	  new_width = MIN_WIDGET_WIDTH;
	  new_x = drag_widget_x2 - new_width;
	}
      if (new_height < MIN_WIDGET_HEIGHT)
	{
	  new_height = MIN_WIDGET_HEIGHT;
	  new_y = drag_widget_y2 - new_height;
	}
      break;

    case GB_TOP_RIGHT:
      new_x = drag_widget_x1;
      new_y = snap_top_edge (mouse_y);
      new_width = snap_right_edge (mouse_x) - new_x;
      new_height = drag_widget_y2 - new_y;
      if (new_width < MIN_WIDGET_WIDTH)
	{
	  new_width = MIN_WIDGET_WIDTH;
	}
      if (new_height < MIN_WIDGET_HEIGHT)
	{
	  new_height = MIN_WIDGET_HEIGHT;
	  new_y = drag_widget_y2 - new_height;
	}
      break;

    case GB_BOTTOM_LEFT:
      new_x = snap_left_edge (mouse_x);
      new_y = drag_widget_y1;
      new_width = drag_widget_x2 - new_x;
      new_height = snap_bottom_edge (mouse_y) - new_y;
      if (new_width < MIN_WIDGET_WIDTH)
	{
	  new_width = MIN_WIDGET_WIDTH;
	  new_x = drag_widget_x2 - new_width;
	}
      if (new_height < MIN_WIDGET_HEIGHT)
	new_height = MIN_WIDGET_HEIGHT;
      break;

    case GB_BOTTOM_RIGHT:
      new_x = drag_widget_x1;
      new_y = drag_widget_y1;
      new_width = snap_right_edge (mouse_x) - new_x;
      new_height = snap_bottom_edge (mouse_y) - new_y;
      if (new_width < MIN_WIDGET_WIDTH)
	new_width = MIN_WIDGET_WIDTH;
      if (new_height < MIN_WIDGET_HEIGHT)
	new_height = MIN_WIDGET_HEIGHT;
      break;

    case GB_MIDDLE:
      new_x = snap_left_edge (mouse_x - drag_offset_x);
      new_y = snap_top_edge (mouse_y - drag_offset_y);
      new_width = widget->allocation.width;
      new_height = widget->allocation.height;
      if (new_x < 0)
	new_x = 0;
      if (new_y < 0)
	new_y = 0;
      break;
    }

  /* Only move/resize widget if values have changed */
  if (new_width != widget->allocation.width
      || new_height != widget->allocation.height)
    {
      gb_widget_set_usize (widget, new_width, new_height);
      wdata->width = new_width;
      wdata->height = new_height;
    }
  if (new_x != widget->allocation.x
      || new_y != widget->allocation.y)
    {
      /* Do we need both? The widget doesn't move without set_uposition() */
      gtk_fixed_move (GTK_FIXED (widget->parent), widget, new_x, new_y);
      gtk_widget_set_uposition (widget, new_x, new_y);
      wdata->x = new_x;
      wdata->y = new_y;
    }

  gtk_container_unblock_resize (GTK_CONTAINER (widget->parent));

  /* Update properties if widget's properties are currently shown */
  if (property_get_widget () == widget)
    {
      property_set_auto_apply (FALSE);
      property_set_int (GbX, new_x);
      property_set_int (GbY, new_y);
      property_set_int (GbWidth, new_width);
      property_set_int (GbHeight, new_height);
      property_set_auto_apply (TRUE);
    }

  return TRUE;
}


static gint
editor_on_motion_notify (GtkWidget * signal_widget,
			 GdkEventMotion * event,
			 gpointer data)
{
  /* We remember the last cursor set and the last window, so we don't set the
     same cursor on the same window repeatedly. */
  static GdkWindow *last_window = NULL;
  static GdkCursor *last_cursor = NULL;

  GtkWidget *widget;
  GdkCursor *cursor = NULL;
  gint x, y, pos;

#if 0
  g_print ("In editor_on_motion_notify\n");
#endif

  widget = editor_get_event_widget (signal_widget, event->window,
				    event->x, event->y, &x, &y);
  if (widget == NULL)
    return FALSE;

#if 0
  g_print ("  widget: %s (%p) X:%i Y:%i\n",
	   gtk_widget_get_name (widget), widget, x, y);
#endif
  if (palette_is_selector_on ())
    {
      if (widget->parent && GTK_IS_FIXED (widget->parent))
	{
	  if (drag_action != GB_DRAG_NONE && event->state & GDK_BUTTON1_MASK)
	    {
	      return do_drag_action (widget, event);
	    }
	  else
	    {
	      pos = get_position_in_widget (widget, x, y);
	      switch (pos)
		{
		case GB_TOP_LEFT:
#if 0
		  g_print ("TOP_LEFT\n");
#endif
		  cursor = cursor_top_left;
		  break;
		case GB_TOP_RIGHT:
#if 0
		  g_print ("TOP_RIGHT\n");
#endif
		  cursor = cursor_top_right;
		  break;
		case GB_BOTTOM_LEFT:
#if 0
		  g_print ("BOTTOM_LEFT\n");
#endif
		  cursor = cursor_bottom_left;
		  break;
		case GB_BOTTOM_RIGHT:
#if 0
		  g_print ("BOTTOM_RIGHT\n");
#endif
		  cursor = cursor_bottom_right;
		  break;
		case GB_MIDDLE:
#if 0
		  g_print ("MIDDLE\n");
#endif
		  cursor = cursor_move;
		  break;
		}
	    }
	}
      else
	{
	  cursor = cursor_selector;
	}
    }
  else
    {
      if (GTK_IS_FIXED (widget)
	  || (widget->parent && GTK_IS_FIXED (widget->parent)))
	cursor = cursor_add_to_fixed;
      else
	cursor = cursor_add_widget;
    }

  if (cursor)
    {
      if (last_window != widget->window || last_cursor != cursor)
	{
	  set_cursor_recursive (widget->window, cursor);
	  last_window = widget->window;
	  last_cursor = cursor;

	}
      gtk_signal_emit_stop_by_name (GTK_OBJECT (signal_widget),
				    "motion_notify_event");
      return TRUE;
    }
  return FALSE;
}


/* I've stopped setting the cursor recursively for now since it stops the
   resize cursor from being shown for resizing clist columns. */
static void
set_cursor_recursive (GdkWindow *window, GdkCursor *cursor)
{
  GList *children;
  GdkWindow *child_window;

  gdk_window_set_cursor (window, cursor);

  children = gdk_window_get_children (window);
  while (children)
    {
      child_window = children->data;
      /*set_cursor_recursive (child_window, cursor);*/
      children = children->next;
    }
  g_list_free (children);
}


/*
 * Adding signals to widgets to allow manipulation, e.g. selecting/drawing
 */

static GdkFilterReturn
filter_function (XEvent *xevent, GdkEvent *event, gpointer data)
{
  /*g_print ("In filter_function\n");*/

  return GDK_FILTER_CONTINUE;
}


static void
set_events_recursive (GdkWindow *window)
{
  GList *children;
  GdkWindow *child_window;
  GdkEventMask mask;

  /*g_print ("Setting events of window: %p\n", window);*/
  mask = gdk_window_get_events (window);
  /*g_print ("Mask: %o -> %o\n", mask, mask | GDK_ALL_EVENTS_MASK);*/
  gdk_window_set_events (window, mask | GDK_POINTER_MOTION_MASK);

  gdk_window_add_filter (window, (GdkFilterFunc) filter_function, NULL);

  children = gdk_window_get_children (window);
  while (children)
    {
      child_window = children->data;
      set_events_recursive (child_window);
      children = children->next;
    }
  g_list_free (children);
}


static void
editor_on_widget_realize (GtkWidget *widget, gpointer data)
{
#if 0
  g_print ("In editor_on_widget_realize widget:%s (%p)\n",
	   gtk_widget_get_name (widget), widget);
#endif
  /* FIXME: Only do this for toplevels? */
  /*set_events_recursive (widget->window);*/

  if (GB_IS_PLACEHOLDER (widget))
    {
      GdkPixmap *placeholder_pixmap;

      /* Create placeholder pixmap */
      /* FIXME: I don't think we really need to create a new pixmap for each
	 placeholder, but there is a problem with some X displays
	 (multi-depth?). */
      placeholder_pixmap = gdk_pixmap_create_from_xpm_d (widget->window,
							 NULL,
				         &widget->style->bg[GTK_STATE_NORMAL],
							 placeholder_xpm);
      if (!placeholder_pixmap)
	{
	  g_warning ("Couldn't create placeholder pixmap\n");
	  /* FIXME: Use a color instead? */
	}
      else
	{
	  gdk_window_set_back_pixmap (widget->window, placeholder_pixmap,
				      FALSE);
	  /* I think we can we free the pixmap immediately. */
	  gdk_pixmap_unref (placeholder_pixmap);
	}
    }
}


/* This adds the button signals to an existing widget (currently only the
   Clist title buttons). */
void
editor_add_mouse_signals_to_existing (GtkWidget * widget)
{
  gtk_signal_connect (GTK_OBJECT (widget), "button_press_event",
		      GTK_SIGNAL_FUNC (editor_on_button_press), NULL);
  gtk_signal_connect (GTK_OBJECT (widget), "button_release_event",
		      GTK_SIGNAL_FUNC (editor_on_button_release), NULL);
}


static void
add_mouse_signals_recursive (GtkWidget *widget, gpointer data)
{
#if 0
  g_print ("Adding mouse signals to:%s (%p)\n", gtk_widget_get_name (widget),
	   widget);
#endif
  if (!GTK_WIDGET_NO_WINDOW (widget))
    gtk_widget_set_events (widget, gtk_widget_get_events (widget)
			   | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);
  gtk_signal_connect (GTK_OBJECT (widget), "button_press_event",
		      GTK_SIGNAL_FUNC (editor_on_button_press), NULL);
  gtk_signal_connect (GTK_OBJECT (widget), "button_release_event",
		      GTK_SIGNAL_FUNC (editor_on_button_release), NULL);

  if (GTK_IS_CONTAINER (widget))
    gtk_container_foreach (GTK_CONTAINER (widget),
			   (GtkCallback) add_mouse_signals_recursive, NULL);
}

/* We need to be careful about passing events on to widgets, especially with
   regard to mouse grabs - in a GtkEntry the mouse is grabbed while selecting
   text, and this can cause all sorts of problems for Glade. */
void
editor_add_mouse_signals (GtkWidget * widget)
{
  /* Widgets without windows will not get events directly from X Windows,
     but they may have child widgets which pass events up to them, e.g.
     a GtkCombo has a GtkEntry which will get X events.
     This doesn't matter too much since we have to call a function to figure
     out which widget the event is for anyway. */
  add_mouse_signals_recursive (widget, NULL);

  gtk_signal_connect_after (GTK_OBJECT (widget), "realize",
			    GTK_SIGNAL_FUNC (editor_on_widget_realize), NULL);
}

void
editor_add_key_signals (GtkWidget * widget)
{
  gtk_signal_connect (GTK_OBJECT (widget), "key_press_event",
		      GTK_SIGNAL_FUNC (editor_on_key_press_event), NULL);
  gtk_signal_connect (GTK_OBJECT (widget), "key_release_event",
		      GTK_SIGNAL_FUNC (editor_on_key_release_event), NULL);
}


void
on_size_allocate (GtkWidget * widget, GbWidgetData * wdata)
{
  MSG ("In on_size_allocate");
  if (property_get_widget () == widget)
    gb_widget_show_position_properties (widget);
}


void
editor_add_draw_signals (GtkWidget * widget)
{
  GbWidgetData *widget_data;

  if (!GTK_WIDGET_NO_WINDOW (widget))
    gtk_widget_set_events (widget, gtk_widget_get_events (widget)
			   | GDK_EXPOSURE_MASK | GDK_POINTER_MOTION_MASK
			   | GDK_BUTTON1_MOTION_MASK);

  widget_data = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);

  gtk_signal_connect_after (GTK_OBJECT (widget), "expose_event",
			    GTK_SIGNAL_FUNC (expose_widget), widget_data);
  gtk_signal_connect_after (GTK_OBJECT (widget), "draw",
			    GTK_SIGNAL_FUNC (draw_widget), NULL);
  gtk_signal_connect_after (GTK_OBJECT (widget), "size_allocate",
			    GTK_SIGNAL_FUNC (on_size_allocate), widget_data);

  /* Needed for button, others? */
  gtk_signal_connect_after (GTK_OBJECT (widget), "draw_default",
			    GTK_SIGNAL_FUNC (draw_widget_focus), NULL);
  gtk_signal_connect_after (GTK_OBJECT (widget), "draw_focus",
			    GTK_SIGNAL_FUNC (draw_widget_focus), NULL);

  /* FIXME: mouse signal - This also needs to be added to all children. */
  gtk_signal_connect (GTK_OBJECT (widget), "motion_notify_event",
		      GTK_SIGNAL_FUNC (editor_on_motion_notify), NULL);

  /* Needed for scrolled window, clist? & possibly other widgets */
  if (GTK_IS_CONTAINER (widget))
    gtk_container_foreach (GTK_CONTAINER (widget),
			   (GtkCallback) editor_add_draw_signals, NULL);
}



static gint
expose_widget (GtkWidget * widget, GdkEventExpose * event,
	       GbWidgetData * widget_data)
{
#if 0
  g_print ("In expose_event widget:%s (%p)\n", gtk_widget_get_name (widget),
	   widget);
  g_print ("Area x:%i y:%i w:%i h:%i\n", event->area.x, event->area.y,
	   event->area.width, event->area.height);
#endif
  /* Ignore spurious exposes before widget is positioned. */
  if (widget->allocation.x == -1 || widget->allocation.y == -1)
    return FALSE;

  /* Here we try to make sure that the widget's size & position are displayed
     correctly in the property window when the widget is created. */
  if (widget_data && (widget_data->flags & GB_INITIAL_EXPOSE))
    {
      widget_data->flags &= ~GB_INITIAL_EXPOSE;
      if (property_get_widget () == widget)
	gb_widget_show_position_properties (widget);
    }
  paint_widget (widget);
  /* FIXME: GTK bug workaround - try to stop multiple exposes of placeholders,
     which happens because the drawing area doesn't have a draw() function
     and the default widget real_draw() function emits an expose event. */
  /*if (GB_IS_PLACEHOLDER (widget)) */
  /*gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "expose_event"); */
  return FALSE;
}


static gint
draw_widget (GtkWidget * widget, GdkRectangle * area, gpointer data)
{
#if 0
  g_print ("In draw_widget: %s (%p) parent: %p\n",
	   gtk_widget_get_name (widget), widget, widget->parent);
#endif
  paint_widget (widget);
  /* FIXME: test */
  /*gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "draw"); */
  return FALSE;
}


static void
draw_widget_focus (GtkWidget * widget, gpointer data)
{
#if 0
  g_print ("In draw_widget_focus: %s (%p) parent: %p\n",
	   gtk_widget_get_name (widget), widget, widget->parent);
#endif
  paint_widget (widget);
}


void
paint_widget (GtkWidget * widget)
{
  gint x, y, w, h;
  GtkWidget *ancestor;
  GdkGC *gc;
  GdkWindow *window;

  /* Check widget is drawable in case it has been deleted. */
  if (!GTK_WIDGET_DRAWABLE (widget))
    return;

  /* If widget is a placeholder, draw the placeholder pixmap in it and a
     3D border around it. */
  if (GB_IS_PLACEHOLDER (widget))
    {
      GdkGC *light_gc = widget->style->light_gc[GTK_STATE_NORMAL];
      GdkGC *dark_gc = widget->style->dark_gc[GTK_STATE_NORMAL];
      gdk_window_get_size (widget->window, &w, &h);
      /* This should not be needed. */
      /*gdk_window_clear (widget->window); */
      gdk_draw_line (widget->window, light_gc, 0, 0, w - 1, 0);
      gdk_draw_line (widget->window, light_gc, 0, 0, 0, h - 1);
      gdk_draw_line (widget->window, dark_gc, 0, h - 1, w - 1, h - 1);
      gdk_draw_line (widget->window, dark_gc, w - 1, 0, w - 1, h - 1);
    }

  /* Draw grid for fixed containers */
  if (GTK_IS_FIXED (widget))
    draw_grid (widget);

  /* highlight selected widgets */
  gc = widget->style->black_gc;
  gdk_gc_set_subwindow (gc, GDK_INCLUDE_INFERIORS);
  ancestor = widget;
  while (ancestor)
    {
      if (g_list_find (selected_widgets, ancestor))
	{
	  if (ancestor->parent)
	    {
	      window = ancestor->parent->window;
	      x = ancestor->allocation.x;
	      y = ancestor->allocation.y;
	      w = ancestor->allocation.width;
	      h = ancestor->allocation.height;
	    }
	  else
	    {
	      window = ancestor->window;
	      x = 0;
	      y = 0;
	      gdk_window_get_size (window, &w, &h);
	    }
	  paint_selection (window, gc, x, y, w, h);
	}
      /* SPECIAL CODE: Don't try to paint ancestors if it is a viewport. */
      if (ancestor != widget && GTK_IS_VIEWPORT (ancestor))
	break;

      ancestor = ancestor->parent;
    }
  /* Reset gc - maybe we should remember the current setting */
  gdk_gc_set_subwindow (gc, GDK_CLIP_BY_CHILDREN);
}


static void
draw_grid (GtkWidget * widget)
{
  GdkGC *gc = widget->style->dark_gc[GTK_STATE_NORMAL];
  gint width = widget->allocation.width;
  gint height = widget->allocation.height;
  gint gridx, gridy;

  /* Note: should we take the border_width into account? - i.e. start the
     grid inside the border. It makes it awkward if you change the border
     size. */
  if (!editor_show_grid)
    return;
  if (editor_grid_style == GB_GRID_DOTS)
    {
      for (gridx = 0; gridx < width; gridx += editor_grid_horz_spacing)
	{
	  for (gridy = 0; gridy < height; gridy += editor_grid_vert_spacing)
	    gdk_draw_point (widget->window, gc, gridx, gridy);
	}
    }
  else
    {
      for (gridx = 0; gridx < width; gridx += editor_grid_horz_spacing)
	gdk_draw_line (widget->window, gc, gridx, 0, gridx, height);
      for (gridy = 0; gridy < height; gridy += editor_grid_vert_spacing)
	gdk_draw_line (widget->window, gc, 0, gridy, width, gridy);
    }
}

static void
paint_selection (GdkWindow * window, GdkGC * gc, gint x, gint y,
		 gint width, gint height)
{
  /* Paint the four corner handles, if there is enough room. */
  if (width > GB_CORNER_WIDTH && height > GB_CORNER_HEIGHT)
    {
      gdk_draw_rectangle (window, gc, TRUE,
			  x, y,
			  GB_CORNER_WIDTH, GB_CORNER_HEIGHT);
      gdk_draw_rectangle (window, gc, TRUE,
			  x, y + height - GB_CORNER_HEIGHT,
			  GB_CORNER_WIDTH, GB_CORNER_HEIGHT);
      gdk_draw_rectangle (window, gc, TRUE,
			  x + width - GB_CORNER_WIDTH, y,
			  GB_CORNER_WIDTH, GB_CORNER_HEIGHT);
      gdk_draw_rectangle (window, gc, TRUE,
			  x + width - GB_CORNER_WIDTH,
			  y + height - GB_CORNER_HEIGHT,
			  GB_CORNER_WIDTH, GB_CORNER_HEIGHT);
    }
  /* Paint the box around the widget. */
  gdk_draw_rectangle (window, gc, FALSE,
		      x, y, width - 1, height - 1);
}


/*
 * Redraw the given widget completely, including all space allocated by its
 * parent (since this may be used for drawing the widget's selection).
 * If widget has no parent (i.e. its a toplevel window) just clear
 * it all and redraw.
 */

void
editor_refresh_widget (GtkWidget * widget)
{
#if 0
  g_print ("In editor_refresh_widget widget: %s (%p)\n",
	   gtk_widget_get_name (widget), widget);
#endif

  editor_refresh_widget_area (widget,
			      widget->allocation.x,
			      widget->allocation.y,
			      widget->allocation.width,
			      widget->allocation.height);
  gtk_widget_draw (widget, NULL);
  /* FIXME: GTK bug workaround to draw Clist children. */
  if (GTK_IS_CLIST (widget))
    gtk_widget_draw_children (widget);
}


void
editor_refresh_widget_selection (GtkWidget * widget)
{
  gint x, y, w, h;

#if 0
  g_print ("In editor_refresh_widget_selection widget: %s (%p)\n",
	   gtk_widget_get_name (widget), widget);
#endif

  x = widget->allocation.x;
  y = widget->allocation.y;
  w = widget->allocation.width;
  h = widget->allocation.height;

  /* Clear the four corners. */
  editor_refresh_widget_area (widget,
			      x, y,
			      GB_CORNER_WIDTH, GB_CORNER_HEIGHT);
  editor_refresh_widget_area (widget,
			      x, y + h - GB_CORNER_HEIGHT,
			      GB_CORNER_WIDTH, GB_CORNER_HEIGHT);
  editor_refresh_widget_area (widget,
			      x + w - GB_CORNER_WIDTH, y,
			      GB_CORNER_WIDTH, GB_CORNER_HEIGHT);
  editor_refresh_widget_area (widget,
			      x + w - GB_CORNER_WIDTH,
			      y + h - GB_CORNER_HEIGHT,
			      GB_CORNER_WIDTH, GB_CORNER_HEIGHT);
  /* Clear the four lines along the edges. */
  editor_refresh_widget_area (widget,
			      x + GB_CORNER_WIDTH, y,
			      w - 2 * GB_CORNER_WIDTH, 1);
  editor_refresh_widget_area (widget,
			      x + GB_CORNER_WIDTH, y + h - 1,
			      w - 2 * GB_CORNER_WIDTH, 1);
  editor_refresh_widget_area (widget,
			      x, y + GB_CORNER_HEIGHT,
			      1, h - 2 * GB_CORNER_HEIGHT);
  editor_refresh_widget_area (widget,
			      x + w - 1,
			      y + GB_CORNER_HEIGHT,
			      1, h - 2 * GB_CORNER_HEIGHT);

  gtk_widget_draw (widget, NULL);
  /* FIXME: GTK bug workaround to draw Clist children. */
  if (GTK_IS_CLIST (widget))
    gtk_widget_draw_children (widget);
}


void
editor_refresh_widget_area (GtkWidget * widget, gint x, gint y, gint w, gint h)
{
  GdkRectangle area;

  area.x = x;
  area.y = y;
  area.width = w;
  area.height = h;

  if (!GTK_WIDGET_DRAWABLE (widget))
    return;

  if (widget->parent)
    {
      gdk_window_clear_area (widget->parent->window, x, y, w, h);
      clear_child_windows (widget->parent->window, x, y, w, h);
    }
  else
    {
      gdk_window_clear_area (widget->window, x, y, w, h);
      clear_child_windows (widget->window, x, y, w, h);
    }
}


/* This clears all child windows which fall within the given rectangle.
   If the rectangle width is -1, then all children are cleared. */
static void
clear_child_windows (GdkWindow * window, gint x, gint y, gint w, gint h)
{
  GList *children;
  GdkWindow *child_window;
  gint win_x, win_y, win_w, win_h;
  XWindowAttributes xwa;
  GdkRectangle area, child, intersection;

  area.x = x;
  area.y = y;
  area.width = w;
  area.height = h;

  children = gdk_window_get_children (window);
  while (children)
    {
      child_window = children->data;
      gdk_window_get_position (child_window, &win_x, &win_y);
      gdk_window_get_size (child_window, &win_w, &win_h);

      child.x = win_x;
      child.y = win_y;
      child.width = win_w;
      child.height = win_h;

      if (gdk_rectangle_intersect (&area, &child, &intersection))
	{
	  /* We need to make sure this is not an InputOnly window, or we get
	     a BadMatch. CList uses InputOnly windows - for resizing columns.*/
	  XGetWindowAttributes (GDK_DISPLAY (),
				GDK_WINDOW_XWINDOW (child_window),
				&xwa);
	  if (xwa.class !=InputOnly)
	    {
	      /* Convert to the childs coordinate space. */
	      intersection.x -= child.x;
	      intersection.y -= child.y;
	      gdk_window_clear_area (child_window,
				     intersection.x, intersection.y,
				   intersection.width, intersection.height);
	      clear_child_windows (child_window,
				   intersection.x, intersection.y,
				   intersection.width, intersection.height);
	    }
	}
      children = children->next;
    }
  g_list_free (children);
}



/*
 * Key events
 */
static gint
editor_on_key_press_event (GtkWidget * widget, GdkEventKey * event,
			   gpointer data)
{
  gboolean stop_signal_emission = FALSE;
  guint key = event->keyval;

  MSG ("In on_key_press_event");
  switch (key)
    {
    case GDK_Delete:
      if (selected_widgets)
	delete (selected_widgets->data);
      stop_signal_emission = TRUE;
      break;
    case GDK_Escape:
      editor_clear_selection (NULL);
      stop_signal_emission = TRUE;
      break;
    }

  if (stop_signal_emission)
    {
      gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");
      return TRUE;
    }
  return FALSE;
}


static gint
editor_on_key_release_event (GtkWidget * widget, GdkEventKey * event,
			     gpointer data)
{
  MSG ("In on_key_release_event");

  return FALSE;
}



/* This is when the 'Select' menuitem on the popup menu is selected */
void
editor_on_select_activate (GtkWidget * menuitem, GtkWidget * widget)
{
  /* Note: could start a weird drag? */
  editor_select_widget (widget, 0, 0, 0);
}



void
editor_on_delete ()
{
  if (selected_widgets)
    delete (selected_widgets->data);
}


/* This is when the 'Delete' menuitem on the popup menu is selected */
void
editor_on_delete_activate (GtkWidget * menuitem, GtkWidget * widget)
{
  delete (widget);
}


static void
delete (GtkWidget * widget)
{
  if (GB_IS_PLACEHOLDER (widget))
    delete_placeholder (widget);
  else
    editor_delete_widget (widget);
}


static void
delete_placeholder (GtkWidget * placeholder)
{
  GtkWidget *parent = placeholder->parent;
  gchar *child_name;

  /* SPECIAL CODE: Don't allow placeholders in clist titles to be deleted. */
  child_name = gtk_object_get_data (GTK_OBJECT (placeholder),
				    GB_CHILD_NAME_KEY);
  if (child_name)
    {
      if (!strcmp (child_name, "CList:title"))
	{
	  /*g_print ("Not deleting clist title placeholder\n");*/
	  return;
	}
    }

  /* Remove widget from the selection */
  editor_clear_selection (NULL);

  /* Can't delete children of a paned or a viewport */
  if (GTK_IS_PANED (parent) || GTK_IS_VIEWPORT (parent))
    return;

  /* For a Clist, we can delete everything except column title widgets */
  if (GTK_IS_CLIST (parent))
    {
      g_warning ("Deleting a widget in a clist - not implemented yet");
      return;
    }

  /* Widgets with these parents can all be deleted OK */
  if (GTK_IS_TOOLBAR (parent) || GTK_IS_TREE (parent) || GTK_IS_LIST (parent))
    {
      gtk_widget_destroy (placeholder);
      return;
    }

  /* For these widgets replace the parent with a placeholder, or delete the
     component if parent is a toplevel widget */
  if (GTK_IS_BIN (parent) || GTK_IS_BUTTON (parent))
    {
      editor_delete_widget (parent);
      return;
    }

  /* For a box, if the placeholder is the only child then replace the box with
     a placeholder, else just delete the placeholder */
  if (GTK_IS_BOX (parent))
    {
      if (g_list_length (GTK_BOX (parent)->children) == 1)
	{
	  editor_delete_widget (parent);
	}
      else
	{
	  gtk_container_remove (GTK_CONTAINER (parent), placeholder);
	  /* Shouldn't really need to do this */
	  gtk_widget_queue_resize (parent);
	}
      return;
    }

  /* For a notebook, if placeholder is the only page, replace the notebook with
     a placeholder, else delete the current notebook page (i.e. placeholder) */
  if (GTK_IS_NOTEBOOK (parent))
    {
      if (g_list_length (GTK_NOTEBOOK (parent)->children) == 1)
	{
	  editor_delete_widget (parent);
	}
      else
	{
	  gtk_notebook_remove_page (GTK_NOTEBOOK (parent),
			 gtk_notebook_current_page (GTK_NOTEBOOK (parent)));
	}
      return;
    }

  /* For a table, can't delete placeholder, unless there is only 1 row or
     column. In this case delete the placeholder, and move all the other
     children up/left. If the table is 1 x 1 then delete the table. */
  if (GTK_IS_TABLE (parent))
    {
      gint nrows, ncols, position = 0, distance_to_move = 0;
      GList *item;
      GtkTableChild *table_child;

      nrows = GTK_TABLE (parent)->nrows;
      ncols = GTK_TABLE (parent)->ncols;
      if (nrows > 1 && ncols > 1)
	return;
      if (nrows == 1 && ncols == 1)
	{
	  editor_delete_widget (parent);
	  return;
	}

      /* Find out where placeholder is */
      item = GTK_TABLE (parent)->children;
      while (item)
	{
	  table_child = (GtkTableChild *) item->data;

	  if (table_child->widget == placeholder)
	    {
	      /* Calculate how far up/left we will have to move the rest of the
	         children */
	      if (nrows == 1)
		{
		  position = table_child->left_attach;
		  distance_to_move = table_child->right_attach
		    - table_child->left_attach;
		}
	      else
		{
		  position = table_child->top_attach;
		  distance_to_move = table_child->bottom_attach
		    - table_child->top_attach;
		}
	      break;
	    }
	  item = item->next;
	}
      /* Shouldn't reach the end of the list */
      g_return_if_fail (item != NULL);
      gtk_widget_destroy (placeholder);

      /* Now step through the table again, moving children up or left */
      item = GTK_TABLE (parent)->children;
      while (item)
	{
	  table_child = (GtkTableChild *) item->data;
	  if (nrows == 1)
	    {
	      if (table_child->left_attach > position)
		{
		  table_child->left_attach -= distance_to_move;
		  table_child->right_attach -= distance_to_move;
		}
	    }
	  else
	    {
	      if (table_child->top_attach > position)
		{
		  table_child->top_attach -= distance_to_move;
		  table_child->bottom_attach -= distance_to_move;
		}
	    }
	  item = item->next;
	}

      /* Now update the tables nrows & ncols */
      if (nrows == 1)
	GTK_TABLE (parent)->ncols -= distance_to_move;
      else
	GTK_TABLE (parent)->nrows -= distance_to_move;

      return;
    }
}


void
editor_delete_widget (GtkWidget * widget)
{
  GtkWidget *placeholder;
  gchar *child_name;
  /*gint w, h; */

  MSG ("In editor_delete_widget");

  /* SPECIAL CODE: Don't allow dialog buttons & widgets to be deleted. */
  child_name = gtk_object_get_data (GTK_OBJECT (widget), GB_CHILD_NAME_KEY);
  if (child_name)
    {
      if (!strcmp (child_name, "FileSel:ok_button")
	  || !strcmp (child_name, "FileSel:cancel_button")
	  || !strcmp (child_name, "ColorSel:ok_button")
	  || !strcmp (child_name, "ColorSel:cancel_button")
	  || !strcmp (child_name, "ColorSel:help_button")
	  || !strcmp (child_name, "FontSel:ok_button")
	  || !strcmp (child_name, "FontSel:cancel_button")
	  || !strcmp (child_name, "FontSel:apply_button")
	  || !strcmp (child_name, "Dialog:vbox")
	  || !strcmp (child_name, "Dialog:action_area")
	  || !strcmp (child_name, "InputDialog:save_button")
	  || !strcmp (child_name, "InputDialog:close_button"))
	{
	  /*g_print ("Not deleting dialog widget\n");*/
	  return;
	}
    }

  /* Remove the widget from the widget tree window. */
  tree_delete_widget (widget);

  /* Set properties widget to NULL, in case the widget or parent is deleted */
  property_set_widget (NULL);

  /* Remove widget from the selection */
  editor_clear_selection (NULL);

  /* If widget is a toplevel widget (i.e. a project component) delete the
     component. */
  if (!widget->parent)
    {
      project_delete_component (widget);
      return;
    }

  /* If the widget's parent is a fixed container remove widget completely. */
  if (GTK_IS_FIXED (widget->parent))
    {
      gtk_widget_destroy (widget);
      return;
    }

  /* Replace normal widget's with a placeholder  & select it so it can also
     be deleted easily using the Delete key. */
  placeholder = editor_new_placeholder ();

  gb_widget_replace_child (widget->parent, widget, placeholder);
  editor_select_widget (placeholder, 0, 0, 0);

  MSG ("Out editor_delete_widget");
}
