/*
 *	cook - file construction tool
 *	Copyright (C) 1997, 1998 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to manipulate command opcodes
 */

#include <errno.h>
#include <ac/stdio.h>
#include <ac/stdlib.h>
#include <ac/string.h>
#include <ac/time.h>
#include <ac/unistd.h>

#include <error.h>
#include <error_intl.h>
#include <expr/position.h>
#include <flag.h>
#include <id.h>
#include <id/variable.h>
#include <mem.h>
#include <opcode/context.h>
#include <opcode/command.h>
#include <opcode/private.h>
#include <option.h>
#include <os.h>
#include <quote.h>
#include <star.h>
#include <str_list.h>
#include <trace.h>


typedef struct opcode_command_ty opcode_command_ty;
struct opcode_command_ty
{
	opcode_ty	inherited;
	int		input;
	expr_position_ty pos;
};


static void destructor _((opcode_ty *));

static void
destructor(that)
	opcode_ty	*that;
{
	opcode_command_ty *this;

	this = (opcode_command_ty *)that;
	expr_position_destructor(&this->pos);
}


/*
 * NAME
 *	echo_command
 *
 * SYNOPSIS
 *	void echo_command(void);
 *
 * DESCRIPTION
 *	The echo_command function is used to determine whether the
 *	command will probably produce output.  Used to see of a
 *	star_eoln needs to be issued.
 *
 * RETURNS
 *	int; 1 if prints, 0 if not
 */

static int echo_command _((string_list_ty *));

static int
echo_command(wlp)
	string_list_ty	*wlp;
{
	static string_ty *echo;
	string_ty	*tmp;
	char		*cp;

	if (!echo)
		echo = str_from_c("echo");
	if (wlp->nstrings < 1 || !str_equal(wlp->string[0], echo))
		return 0;
	tmp = wl2str(wlp, 0, wlp->nstrings - 1, (char *)0);
	for (cp = tmp->str_text; *cp; ++cp)
	{
		if (strchr("^|>", *cp))
		{
			str_free(tmp);
			return 0;
		}
	}
	str_free(tmp);
	return 1;
}


static string_list_ty *id_var_search _((string_ty *));

static string_list_ty *
id_var_search(key)
	string_ty	*key;
{
	id_ty		*idp;
	string_list_ty	*result;

	idp = id_search(key);
	if (!idp)
		return 0;
	result = id_variable_query2(idp);
	if (!result || !result->nstrings)
		return 0;
	return result;
}


/*
 * NAME
 *	spawn - execute a command
 *
 * SYNOPSIS
 *	opcode_status_ty spawn(string_list_ty *args, string_ty *input);
 *
 * DESCRIPTION
 *	The spawn function is used to launch the given command.
 *	The command is not waited for.
 *
 * RETURNS
 *	opcode_status_ty...
 *		opcode_status_error	if something went wrong
 *		opcode_status_wait	if something all went well
 */

static opcode_status_ty spawn _((string_list_ty	*, string_ty *, int *,
	string_ty *));

static opcode_status_ty
spawn(args, input, pid_p, host_binding)
	string_list_ty	*args;
	string_ty	*input;
	int		*pid_p;
	string_ty	*host_binding;
{
	int		j;
	FILE		*fp;
	char		iname[L_tmpnam];
	int		pid;
	opcode_status_ty status;
	static char	**argv;
	static size_t	argvlen;
	string_list_ty	cmd;
	sub_context_ty	*scp;
	char		*shell;

	trace(("execute()\n{\n"/*}*/));
	assert(args);
	assert(args->nstrings > 0);

	/*
	 * build the input file, if required
	 */
	status = opcode_status_error;
	fp = 0;
	if (input)
	{
		/*
		 * He has given a string to be used as input to the command,
		 * so write it out to a file, and then redirect the input.
		 */
		tmpnam(iname);
		fp = fopen(iname, "w+");
		if (!fp)
		{
			scp = sub_context_new();
			sub_errno_set(scp);
			sub_var_set(scp, "File_Name", "%s", iname);
			error_intl(scp, i18n("open $filename: $errno"));
			sub_context_delete(scp);
			goto done;
		}
		if (unlink(iname))
		{
			scp = sub_context_new();
			sub_errno_set(scp);
			sub_var_set(scp, "File_Name", "%s", iname);
			error_intl(scp, i18n("unlink $filename: $errno"));
			sub_context_delete(scp);
			fclose(fp);
			goto done;
		}
		fputs(input->str_text, fp);
		if (ferror(fp) || fseek(fp, 0L, 0))
		{
			scp = sub_context_new();
			sub_errno_set(scp);
			sub_var_set(scp, "File_Name", "%s", iname);
			error_intl(scp, i18n("write $filename: $errno"));
			sub_context_delete(scp);
			fclose(fp);
			goto done;
		}
	}

	/*
	 * See if there is a host_binding in effect.
	 */
	if (host_binding)
	{
		static string_ty *key;
		string_list_ty	*slp;
		string_ty	*s;
		string_ty	*rcmd;
		string_ty	*rcmdq;
		string_ty	*temp_file_name;
		static long	temp_file_number;
		string_ty	*cwd;

		/*
		 * Work out the name of the remote shell command.
		 */
		if (!key)
			key = str_from_c("parallel_rsh");
		slp = id_var_search(key);
		if (slp)
			string_list_copy_constructor(&cmd, slp);
		else
		{
			static string_ty *rsh;

			if (!rsh)
				rsh = str_from_c("rsh");
			string_list_constructor(&cmd);
			string_list_append(&cmd, rsh);
		}

		/*
		 * Add the name of the host.
		 */
		string_list_append(&cmd, host_binding);

		/*
		 * The remote end will need to change directory to where
		 * we are now, since it is *highly* unlikely that we are
		 * in our home directory.
		 *
		 * The comamnd itself will need to be quoted, because
		 * the command is passed to a shell th athe other end,
		 * twice!
		 *
		 * Because rsh(1) always returns an exit status of zero,
		 * we have to make other arrangements to obtain the exit
		 * status.
		 */
		rcmd = wl2str(args, 0, args->nstrings, (char *)0);
		rcmdq = quote(rcmd);
		str_free(rcmd);

		/*
		 * what shell are we using
		 */
		shell = getenv("SHELL");
		if (!shell || !*shell)
			shell = CONF_SHELL;

		/*
		 * We actually need to be a more devious, so that we can
		 * get the exit status back.  Write it to a temporary
		 * file server side, then read and remove it client side.
		 *
		 * This is the server side command.  It gets quoted
		 * again to protect it from the client side shell.
		 */
		temp_file_name =
			str_format(".%d.%ld", getpid(), ++temp_file_number);
		cwd = os_curdir(); /* do not str_free this */
		rcmd =
			str_format
			(
				"cd %S && ( %s %s %S ) ; echo $? > %S/%S",
				cwd,
				shell,
				(option_test(OPTION_ERROK) ? "-c" : "-ce"),
				rcmdq,
				cwd,
				temp_file_name
			);
		str_free(rcmdq);
		rcmdq = quote(rcmd);
		str_free(rcmd);
		string_list_append(&cmd, rcmdq);
		str_free(rcmdq);

		/*
		 * This is the rest of the server side command.
		 */
		s =
			str_format
			(
				"&& exit `cat %S;rm %S`",
				temp_file_name,
				temp_file_name
			);
		string_list_append(&cmd, s);
		str_free(s);
		str_free(temp_file_name);

		/*
		 * Because the command has a semicolon and back quotes,
		 * we need to hand it to a shell (as alluded to above).
		 * Assemble into a single string.
		 */
		rcmd = wl2str(&cmd, 0, cmd.nstrings, (char *)0);
		string_list_destructor(&cmd);

		/*
		 * Build the final command to be executed.
		 * It even cleans up after itself.
		 */
		s = str_from_c(shell);
		string_list_append(&cmd, s);
		str_free(s);
		s = str_from_c("-c");
		string_list_append(&cmd, s);
		str_free(s);
		string_list_append(&cmd, rcmd);
		str_free(rcmd);
	}
	else
	{
		int		magic_characters;

		/*
		 * see if there are any magic shell cheracters
		 */
		magic_characters = 0;
		for (j = 0; j < args->nstrings; ++j)
		{
			char	*s;

			for (s = args->string[j]->str_text; *s; ++s)
			{
				if (strchr("#|=^();&<>*?[]:$`'\"\\\n", *s))
				{
					magic_characters = 1;
					break;
				}
			}
		}

		/*
		 * build the command
		 */
		if (magic_characters)
		{
			string_ty	*str;

			/*
			 * what shell are we using
			 */
			shell = getenv("SHELL");
			if (!shell || !*shell)
				shell = CONF_SHELL;

			string_list_constructor(&cmd);
			str = str_from_c(shell);
			string_list_append(&cmd, str);
			str_free(str);
			if (option_test(OPTION_ERROK))
				str = str_from_c("-c");
			else
				str = str_from_c("-ce");
			string_list_append(&cmd, str);
			str_free(str);
			str = wl2str(args, 0, args->nstrings - 1, (char *)0);
			string_list_append(&cmd, str);
			str_free(str);
		}
		else
			string_list_copy_constructor(&cmd, args);
	}

	/*
	 * build the argv array
	 */
	if (!argv)
	{
		argvlen = cmd.nstrings + 1;
		argv = mem_alloc(argvlen * sizeof(char *));
	}
	else if (argvlen < cmd.nstrings + 1)
	{
		argvlen = cmd.nstrings + 1;
		argv = mem_change_size(argv, argvlen * sizeof(char *));
	}
	if (cmd.nstrings == 0)
	{
		if (fp)
			fclose(fp);
		string_list_destructor(&cmd);
		status = opcode_status_success;
		goto done;
	}
	for (j = 0; j < cmd.nstrings; ++j)
		argv[j] = cmd.string[j]->str_text;
	argv[cmd.nstrings] = 0;

	/*
	 * spawn the child process
	 */
	switch (pid = fork())
	{
	case -1:
		scp = sub_context_new();
		sub_errno_set(scp);
		error_intl(scp, i18n("fork(): $errno"));
		sub_context_delete(scp);
		break;

	case 0:
		/*
		 * child
		 */
		if (fp)
		{
			if (close(0) && errno != EBADF)
			{
				string_ty	*fn0;
				int		err;

				err = errno;
				scp = sub_context_new();
				fn0 = subst_intl(scp, "standard input");
				/* re-use substitution context */
				sub_errno_setx(scp, err);
				sub_var_set(scp, "File_Name", "%S", fn0);
				fatal_intl(scp, i18n("close $filename: $errno"));
				/* NOTREACHED */
				sub_context_delete(scp);
				str_free(fn0);
			}
			if (dup(fileno(fp)) < 0)
			{
				scp = sub_context_new();
				sub_errno_set(scp);
				fatal_intl(scp, i18n("dup(): $errno"));
				/* NOTREACHED */
				sub_context_delete(scp);
			}
		}
		if (argv[0][0] == '/')
			execv(argv[0], argv);
		else
			execvp(argv[0], argv);
		scp = sub_context_new();
		sub_errno_set(scp);
		sub_var_set(scp, "File_Name", "%s", argv[0]);
		fatal_intl(scp, i18n("exec $filename: $errno"));
		/* NOTREACHED */
		sub_context_delete(scp);

	default:
		/*
		 * parent
		 */
		if (fp)
			fclose(fp);
		string_list_destructor(&cmd);
		status = opcode_status_wait;
		trace(("pid = %d;\n", pid));
		*pid_p = pid;
		break;
	}
	done:
	trace(("return %s;\n", opcode_status_name(status)));
	trace((/*{*/"}\n"));
	return status;
}


/*
 * NAME
 *	meter_ptime - print timing info
 *
 * SYNOPSIS
 *	void meter_ptime(double t, char *s);
 *
 * DESCRIPTION
 *	The meter_ptime function is used to print e representation of the
 *	time, in seconds, given in the `t' argument.  The `s' argument is a
 *	title string.
 */

#ifdef HAVE_WAIT3

static void meter_ptime _((double, char *));

static void
meter_ptime(t,s,e)
	double		t;
	char		*s;
{
	long		hour;
	long		min;
	long		sec;
	long		frac;

	frac = t * 1000 + 0.5;
	sec = frac / 1000;
	frac %= 1000;
	min = sec / 60;
	sec %= 60;
	hour = min / 60;
	min %= 60;
	fprintf(stderr, "%2ld:%02ld:%02ld.%03ld %s\n", hour, min, sec, frac, s);
}


/*
 * NAME
 *	meter_print - end a metering interval
 *
 * SYNOPSIS
 *	void meter_print(struct rusage *);
 *
 * DESCRIPTION
 *	The meter_end function is used to end a metering interval and print
 *	the metered information.
 */

static void meter_end _((struct rusage *));

static void
meter_print(rup)
	struct rusage	*rup;
{
	double		usr;
	double		sys;
	
	usr = rup->ru_utime.tv_sec + rup->ru_utime.tv_usec * 1.0e-6;
	meter_ptime(usr, "usr");
	sys = rup->ru_stime.tv_sec + rup->ru_stime.tv_usec * 1.0e-6;
	meter_ptime(sys, "sys");
}

#endif /* HAVE_WAIT3 */


/*
 * NAME
 *	execute
 *
 * SYNOPSIS
 *	opcode_status_ty execute(opcode_ty *, opcode_context_ty *);
 *
 * DESCRIPTION
 *	The execute function is used to execute the given opcode within
 *	the given interpretation context.
 *
 * RETURNS
 *	opcode_status_ty to indicate the result of the execution
 */

static opcode_status_ty execute _((const opcode_ty *, opcode_context_ty *));

static opcode_status_ty
execute(op, icp)
	const opcode_ty	*op;
	opcode_context_ty *icp;
{
	const opcode_command_ty *this;
	opcode_status_ty status;
	string_list_ty	*wlp;
	string_list_ty	*flags_words;
	string_ty	*isp;
	char		*cmd;
	flag_ty		*flags;

	/*
	 * pop the arguments off the value stack
	 */
	trace(("opcode_command::execute()\n{\n"/*}*/));
	this = (const opcode_command_ty *)op;
	status = opcode_status_success;
	isp = 0;
	flags_words = 0;
	flags = 0;
	wlp = 0;
	if (icp->pid)
	{
		icp->pid = 0;
		wlp = icp->wlp;
		icp->wlp = 0;
		goto resume;
	}
	if (this->input)
	{
		string_list_ty	*islp;

		islp = opcode_context_string_list_pop(icp);
		isp = wl2str(islp, 0, islp->nstrings - 1, (char *)0);
		string_list_delete(islp);
	}
	flags_words = opcode_context_string_list_pop(icp);
	wlp = opcode_context_string_list_pop(icp);

	/*
	 * take care of the options
	 */
	flags = flag_recognize(flags_words, &this->pos);
	if (!flags)
	{
		status = opcode_status_error;
		goto done;
	}
	string_list_delete(flags_words);
	flags_words = 0;
	flag_set_options(flags, OPTION_LEVEL_EXECUTE);
	flag_delete(flags);
	flags = 0;

	/*
	 * echo the command if required
	 */
	if (!option_test(OPTION_SILENT))
	{
		string_ty *cp;

		/*
		 * If the command has not been silenced,
		 * form it into a string and echo it.
		 */
		cp = wl2str(wlp, 0, wlp->nstrings - 1, (char *)0);
		error_raw("%s", cp->str_text);
		str_free(cp);
	}

	/*
	 * execute the command if required
	 */
	if (option_test(OPTION_ACTION))
	{
		/*
		 * emit suitable progress stars
		 */
		if (option_test(OPTION_SILENT))
		{
			if (echo_command(wlp))
				star_eoln();
			else
				star_bang();
		}
		else
			star_sync();

		/*
		 * invalidate the stat cache
		 */
		if (option_test(OPTION_INVALIDATE_STAT_CACHE))
		{
			size_t		j;
	
			for (j = 0; j < wlp->nstrings; ++j)
				if (os_clear_stat(wlp->string[j]))
					status = opcode_status_error;
		}

		/*
		 * prepare for metering
		 */
#ifdef HAVE_WAIT3
		if (option_test(OPTION_METERING) && !icp->rup)
			icp->rup = mem_alloc(struct rusage);
#endif

		/*
		 * run the command
		 */
		status = spawn(wlp, isp, &icp->pid, icp->host_binding);
		if (status == opcode_status_wait)
		{
			trace(("...wait\n")); 
			icp->wlp = wlp;
			wlp = 0;
			goto done;
		}
		if (status == opcode_status_error)
			goto done;
resume:
#ifdef HAVE_WAIT3
		if (option_test(OPTION_METERING))
		{
			assert(icp->rup);
			meter_print(icp->rup);
		}
#endif
		/*
		 * Echo the exit status of the command, if it was not
		 * successful.  The error flag will also be examined (it
		 * changes the text of the message, too) to see if we
		 * should terminate successfully or with and error.
		 */
		cmd = (wlp->nstrings ? wlp->string[0]->str_text : "");
		if
		(
			exit_status
			(
				cmd,
				icp->exit_status,
				option_test(OPTION_ERROK)
			)
		)
		{
			status = opcode_status_error;
		}
	}

	/*
	 * rescind the flag settings
	 */
	option_undo_level(OPTION_LEVEL_EXECUTE);

	/*
	 * release the arguments and exit
	 */
	done:
	if (flags_words)
		string_list_delete(flags_words);
	if (flags)
		flag_delete(flags);
	if (wlp)
		string_list_delete(wlp);
	if (isp)
		str_free(isp);
	trace(("return %s;\n", opcode_status_name(status)));
	trace((/*{*/"}\n"));
	return status;
}


/*
 * NAME
 *	script
 *
 * SYNOPSIS
 *	opcode_status_ty script(opcode_ty *, opcode_context_ty *);
 *
 * DESCRIPTION
 *	The script function is used to script the given opcode within
 *	the given interpretation context.
 *
 * RETURNS
 *	opcode_status_ty to indicate the result
 */

static opcode_status_ty script _((const opcode_ty *, opcode_context_ty *));

static opcode_status_ty
script(op, icp)
	const opcode_ty	*op;
	opcode_context_ty *icp;
{
	const opcode_command_ty *this;
	opcode_status_ty status;
	string_list_ty	*wlp;
	string_ty	*isp;
	string_list_ty	*flags_words;
	flag_ty		*flags;

	/*
	 * pop the arguments off the value stack
	 */
	trace(("opcode_command::script()\n{\n"/*}*/));
	this = (const opcode_command_ty *)op;
	status = opcode_status_success;
	if (this->input)
	{
		string_list_ty	*islp;

		islp = opcode_context_string_list_pop(icp);
		isp = wl2str(islp, 0, islp->nstrings - 1, (char *)0);
		string_list_delete(islp);
	}
	else
		isp = 0;
	flags_words = opcode_context_string_list_pop(icp);
	wlp = opcode_context_string_list_pop(icp);

	/*
	 * set the flags
	 */
	flags = flag_recognize(flags_words, &this->pos);
	string_list_delete(flags_words);
	if (!flags)
	{
		status = opcode_status_error;
		goto done;
	}
	flag_set_options(flags, OPTION_LEVEL_EXECUTE);
	flag_delete(flags);

	/*
	 * echo the command if required
	 */
	if (!option_test(OPTION_SILENT))
	{
		string_ty	*s1;
		string_ty	*s2;

		s1 = wl2str(wlp, 0, wlp->nstrings, " ");
		s2 = quoted(s1);
		str_free(s1);
		printf("echo %s\n", s2->str_text);
		str_free(s2);
	}

	/*
	 * script the command if required
	 */
	if (option_test(OPTION_ACTION))
	{
		size_t		j;

		printf("(");
		for (j = 0; j < wlp->nstrings; ++j)
			printf(" %s", wlp->string[j]->str_text);
		printf(" )");

		if (isp)
		{
			char		fubar[50];
			time_t		t;
			int		nl;

			time(&t);
			if (t < 0)
				t = 0;
			sprintf(fubar, "fubar%ldfubar%ldfubar", (long)t, (long)t);

			nl = (isp->str_length && isp->str_text[isp->str_length - 1] != '\n');
			printf
			(
				" << '%s'\n%s%s%s",
				fubar,
				isp->str_text,
				(nl ? "\n" : ""),
				fubar
			);
		}
		printf("\n");

		if (!option_test(OPTION_ERROK))
			printf("test $? -eq 0 || exit 1\n");
	}

	/*
	 * rescind the flag settings
	 */
	option_undo_level(OPTION_LEVEL_EXECUTE);

	/*
	 * release the arguments and exit
	 */
	done:
	string_list_delete(wlp);
	if (isp)
		str_free(isp);
	trace(("return %s;\n", opcode_status_name(status)));
	trace((/*{*/"}\n"));
	return status;
}


/*
 * NAME
 *	disassemble
 *
 * SYNOPSIS
 *	void disassemble(opcode_ty *);
 *
 * DESCRIPTION
 *	The disassemble function is used to disassemble the copdode and
 *	its arguments onto the standard output.  Don't worry about the
 *	location or a trailing newline.
 */

static void disassemble _((const opcode_ty *));

static void
disassemble(op)
	const opcode_ty	*op;
{
	const opcode_command_ty *this;

	trace(("opcode_command::disassemble()\n{\n"/*}*/));
	this = (const opcode_command_ty *)op;
	printf
	(
		"%d\t# %s:%d",
		this->input,
		this->pos.pos_name->str_text,
		this->pos.pos_line
	);
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	method
 *
 * DESCRIPTION
 *	The method variable describes this class.
 *
 * CAVEAT
 *	This symbol is not exported from this file.
 */

static opcode_method_ty method =
{
	"command",
	sizeof(opcode_command_ty),
	destructor,
	execute,
	script,
	disassemble,
};


/*
 * NAME
 *	opcode_command_new
 *
 * SYNOPSIS
 *	opcode_ty *opcode_command_new(void);
 *
 * DESCRIPTION
 *	The opcode_command_new function is used to allocate a new instance
 *	of a command opcode.
 *
 * RETURNS
 *	opcode_ty *; use opcode_delete when you are finished with it.
 */

opcode_ty *
opcode_command_new(input, pp)
	int		input;
	const expr_position_ty *pp;
{
	opcode_ty	*op;
	opcode_command_ty *this;

	trace(("opcode_command_new()\n{\n"/*}*/));
	op = opcode_new(&method);
	this = (opcode_command_ty *)op;
	this->input = input;
	expr_position_copy_constructor(&this->pos, pp);
	trace(("return %08lX;\n", (long)op));
	trace((/*{*/"}\n"));
	return op;
}
