/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994, 1995, 1996, 1997, 1998 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: operating system start point, and command line argument parsing
 */

#include <ac/stddef.h>
#include <ac/string.h>
#include <ac/stdio.h>
#include <ac/stdlib.h>

#include <arglex.h>
#include <builtin.h>
#include <cook.h>
#include <error_intl.h>
#include <help.h>
#include <id.h>
#include <id/variable.h>
#include <lex.h>
#include <listing.h>
#include <option.h>
#include <parse.h>
#include <star.h>
#include <progname.h>
#include <quit.h>
#include <trace.h>
#include <version.h>


enum
{
	arglex_token_action,
	arglex_token_action_not,
	arglex_token_book,
	arglex_token_book_not,
	arglex_token_cascade,
	arglex_token_cascade_not,
	arglex_token_disassemble,
	arglex_token_disassemble_not,
	arglex_token_errok,
	arglex_token_errok_not,
	arglex_token_fingerprint,
	arglex_token_fingerprint_not,
	arglex_token_force,
	arglex_token_force_not,
	arglex_token_include,
	arglex_token_include_cooked,
	arglex_token_include_cooked_not,
	arglex_token_log,
	arglex_token_log_not,
	arglex_token_metering,
	arglex_token_metering_not,
	arglex_token_pairs,
	arglex_token_parallel,
	arglex_token_parallel_not,
	arglex_token_pedantic,
	arglex_token_pedantic_not,
	arglex_token_persevere,
	arglex_token_persevere_not,
	arglex_token_precious,
	arglex_token_precious_not,
	arglex_token_reason,
	arglex_token_reason_not,
	arglex_token_script,
	arglex_token_shallow,
	arglex_token_shallow_not,
	arglex_token_silent,
	arglex_token_silent_not,
	arglex_token_star,
	arglex_token_star_not,
	arglex_token_strip_dot,
	arglex_token_strip_dot_not,
	arglex_token_touch,
	arglex_token_touch_not,
	arglex_token_tty,
	arglex_token_tty_not,
	arglex_token_update,
	arglex_token_update_not

	/*
	 * When you add an option to this list, you must
	 * also add it to the list in cook/builtin/options.c
	 */
};

static arglex_table_ty argtab[] =
{
	{ "-Action",		(arglex_token_ty)arglex_token_action,	},
	{ "-No_Action",		(arglex_token_ty)arglex_token_action_not, },
	{ "-Book",		(arglex_token_ty)arglex_token_book,	},
	{ "-No_Book",		(arglex_token_ty)arglex_token_book_not,	},
	{ "-CAScade",		(arglex_token_ty)arglex_token_cascade, },
	{ "-No_CAScade",	(arglex_token_ty)arglex_token_cascade_not, },
	{ "-Continue",		(arglex_token_ty)arglex_token_persevere, },
	{ "-No_Continue",	(arglex_token_ty)arglex_token_persevere_not, },
	{ "-DISassemble",	(arglex_token_ty)arglex_token_disassemble, },
	{ "-No_DISassemble",	(arglex_token_ty)arglex_token_disassemble_not, },
	{ "-Errok",		(arglex_token_ty)arglex_token_errok,	},
	{ "-No_Errok",		(arglex_token_ty)arglex_token_errok_not,	},
	{ "-FingerPrint",	(arglex_token_ty)arglex_token_fingerprint, },
	{ "-No_FingerPrint",	(arglex_token_ty)arglex_token_fingerprint_not, },
	{ "-Forced",		(arglex_token_ty)arglex_token_force,	},
	{ "-No_Forced",		(arglex_token_ty)arglex_token_force_not,	},
	{ "-Include",		(arglex_token_ty)arglex_token_include,	},
	{ "-\\I*",		(arglex_token_ty)arglex_token_include,	},
	{ "-Include_Cooked",	(arglex_token_ty)arglex_token_include_cooked, },
	{ "-Jobs",		(arglex_token_ty)arglex_token_parallel, },
	{ "-No_Include_Cooked",	(arglex_token_ty)arglex_token_include_cooked_not, },
	{ "-LOg",		(arglex_token_ty)arglex_token_log,	},
	{ "-List",		(arglex_token_ty)arglex_token_log,	},
	{ "-No_LOg",		(arglex_token_ty)arglex_token_log_not,	},
	{ "-No_List",		(arglex_token_ty)arglex_token_log_not,	},
	{ "-Meter",		(arglex_token_ty)arglex_token_metering,	},
	{ "-No_Meter",		(arglex_token_ty)arglex_token_metering_not, },
	{ "-PAirs",		(arglex_token_ty)arglex_token_pairs,	},
	{ "-PARallel",		(arglex_token_ty)arglex_token_parallel,	},
	{ "-No_PARallel",	(arglex_token_ty)arglex_token_parallel_not, },
	{ "-Precious",		(arglex_token_ty)arglex_token_precious,	},
	{ "-No_Precious",	(arglex_token_ty)arglex_token_precious_not, },
	{ "-Reason",		(arglex_token_ty)arglex_token_reason,	},
	{ "-No_Reason",		(arglex_token_ty)arglex_token_reason_not, },
	{ "-SCript",		(arglex_token_ty)arglex_token_script,	},
	{ "-Silent",		(arglex_token_ty)arglex_token_silent,	},
	{ "-No_Silent",		(arglex_token_ty)arglex_token_silent_not, },
	{ "-SHallow",		(arglex_token_ty)arglex_token_shallow,	},
	{ "-No_SHallow",	(arglex_token_ty)arglex_token_shallow_not, },
	{ "-STar",		(arglex_token_ty)arglex_token_star,	},
	{ "-No_STar",		(arglex_token_ty)arglex_token_star_not,	},
	{ "-Strip_Dot",		(arglex_token_ty)arglex_token_strip_dot, },
	{ "-No_Strip_Dot",	(arglex_token_ty)arglex_token_strip_dot_not, },
	{ "-TErminal",		(arglex_token_ty)arglex_token_tty,	},
	{ "-No_TErminal",	(arglex_token_ty)arglex_token_tty_not,	},
	{ "-Touch",		(arglex_token_ty)arglex_token_touch,	},
	{ "-No_Touch",		(arglex_token_ty)arglex_token_touch_not,	},
	{ "-TRace",		(arglex_token_ty)arglex_token_reason,	},
	{ "-No_TRace",		(arglex_token_ty)arglex_token_reason_not,	},
	{ "-Update",		(arglex_token_ty)arglex_token_update,	},
	{ "-No_Update",		(arglex_token_ty)arglex_token_update_not, },
	{ 0, (arglex_token_ty)0, }, /* end marker */
};


/*
 *  NAME
 *	usage - options diagnostic
 *
 *  SYNOPSIS
 *	void usage(void);
 *
 *  DESCRIPTION
 *	Usage is called when the user has made a syntactic or semantic error
 *	on the command line.
 *
 *  CAVEAT
 *	This function does NOT return.
 */

static void usage _((void));

static void
usage()
{
	char		*progname;

	progname = progname_get();
	fprintf(stderr, "usage: %s [ <option>... ][ <filename>... ]\n", progname);
	fprintf(stderr, "       %s -Help\n", progname);
	fprintf(stderr, "       %s -VERSion\n", progname);
	quit(1);
	trace(("to silence warnings\n"));
}


/*
 * NAME
 *	argparse - parse command line
 *
 * SYNOPSIS
 *	void argparse(option_level_ty);
 *
 * DESCRIPTION
 *	The argparse function is used to parse command lines.
 *
 * RETURNS
 *	void
 */

static void argparse _((option_level_ty));

static void
argparse(level)
	option_level_ty	level;
{
	option_number_ty type;
	string_ty	*s;
	sub_context_ty	*scp;

	type = -1;
	switch (arglex())
	{
	case arglex_token_help:
		if (level != OPTION_LEVEL_COMMAND_LINE)
		{
			not_in_env:
			scp = sub_context_new();
			sub_var_set(scp, "Name", "%s", arglex_value.alv_string);
			fatal_intl
			(
				scp,
			       i18n("may not use $name in environment variable")
			);
			/* NOTREACHED */
		}
		help((char *)0, usage);
		quit(0);
	
	case arglex_token_version:
		if (level != OPTION_LEVEL_COMMAND_LINE)
			goto not_in_env;
		version();
		quit(0);

	default:
		break;
	}
	while (arglex_token != arglex_token_eoln)
	{
		switch (arglex_token)
		{
		default:
			generic_argument(usage);
			continue;

		case arglex_token_include:
			if (arglex() != arglex_token_string)
			{
				scp = sub_context_new();
				sub_var_set(scp, "Name", "%s", arglex_token_name(arglex_token_include));
				fatal_intl(scp, i18n("$name needs string"));
				/* NOTREACHED */
			}
			s = str_from_c(arglex_value.alv_string);
			string_list_append_unique(&option.o_search_path, s);
			str_free(s);
			break;

		case arglex_token_reason:
			type = OPTION_REASON;
			normal_on:
			if (option_already(type, level))
			{
				too_many:
				scp = sub_context_new();
				sub_var_set
				(
					scp,
					"Name",
					"%s",
					arglex_value.alv_string
				);
				fatal_intl
				(
					scp,
					i18n("duplicate \"$name\" option")
				);
				/* NOTREACHED */
			}
			option_set(type, level, 1);
			break;

		case arglex_token_reason_not:
			type = OPTION_REASON;
			normal_off:
			if (option_already(type, level))
				goto too_many;
			option_set(type, level, 0);
			break;

		case arglex_token_cascade:
			type = OPTION_CASCADE;
			goto normal_on;

		case arglex_token_cascade_not:
			type = OPTION_CASCADE;
			goto normal_off;

		case arglex_token_disassemble:
			type = OPTION_DISASSEMBLE;
			goto normal_on;

		case arglex_token_disassemble_not:
			type = OPTION_DISASSEMBLE;
			goto normal_off;

		case arglex_token_tty:
			type = OPTION_TERMINAL;
			goto normal_on;

		case arglex_token_tty_not:
			type = OPTION_TERMINAL;
			goto normal_off;

		case arglex_token_precious:
			type = OPTION_PRECIOUS;
			goto normal_on;

		case arglex_token_precious_not:
			type = OPTION_PRECIOUS;
			goto normal_off;

		case arglex_token_log:
			if (option_already(OPTION_LOGGING, level))
				goto too_many;
			option_set(OPTION_LOGGING, level, 1);
			if (arglex() != arglex_token_string)
				continue;
			if (option.o_logfile)
				str_free(option.o_logfile);
			option.o_logfile = str_from_c(arglex_value.alv_string);
			break;

		case arglex_token_log_not:
			type = OPTION_LOGGING;
			goto normal_off;

		case arglex_token_book:
			if (option_already(OPTION_BOOK, level))
				goto too_many;
			option_set(OPTION_BOOK, level, 1);
			if (arglex() != arglex_token_string)
				continue;
			if (option.o_book)
				str_free(option.o_book);
			option.o_book = str_from_c(arglex_value.alv_string);
			break;

		case arglex_token_book_not:
			type = OPTION_BOOK;
			goto normal_off;

		case arglex_token_include_cooked:
			type = OPTION_INCLUDE_COOKED;
			goto normal_on;

		case arglex_token_include_cooked_not:
			type = OPTION_INCLUDE_COOKED;
			goto normal_off;

		case arglex_token_silent:
			type = OPTION_SILENT;
			goto normal_on;

		case arglex_token_silent_not:
			type = OPTION_SILENT;
			goto normal_off;

		case arglex_token_metering:
			type = OPTION_METER;
			goto normal_on;

		case arglex_token_metering_not:
			type = OPTION_METER;
			goto normal_off;

		case arglex_token_touch:
			type = OPTION_TOUCH;
			goto normal_on;

		case arglex_token_touch_not:
			type = OPTION_TOUCH;
			goto normal_off;

		case arglex_token_action:
			type = OPTION_ACTION;
			goto normal_on;

		case arglex_token_action_not:
			type = OPTION_ACTION;
			goto normal_off;

		case arglex_token_persevere:
			type = OPTION_PERSEVERE;
			goto normal_on;

		case arglex_token_persevere_not:
			type = OPTION_PERSEVERE;
			goto normal_off;

		case arglex_token_errok:
			type = OPTION_ERROK;
			goto normal_on;

		case arglex_token_errok_not:
			type = OPTION_ERROK;
			goto normal_off;

		case arglex_token_force:
			type = OPTION_FORCE;
			goto normal_on;

		case arglex_token_force_not:
			type = OPTION_FORCE;
			goto normal_off;

		case arglex_token_fingerprint:
			type = OPTION_FINGERPRINT;
			goto normal_on;

		case arglex_token_fingerprint_not:
			type = OPTION_FINGERPRINT;
			goto normal_off;

		case arglex_token_pairs:
			if (level != OPTION_LEVEL_COMMAND_LINE)
				goto not_in_env;
			if (option.pairs)
				goto too_many;
			option.pairs++;
			break;

		case arglex_token_script:
			if (level != OPTION_LEVEL_COMMAND_LINE)
				goto not_in_env;
			if (option.script)
				goto too_many;
			option.script++;
			break;

		case arglex_token_string:
			if (level != OPTION_LEVEL_COMMAND_LINE)
			{
				if (strchr(arglex_value.alv_string, '='))
				{
					fatal_intl
					(
						0,
			i18n("may not assign variables in environment variable")
					);
				}
				else
				{
					fatal_intl
					(
						0,
			    i18n("may not name targets in environment variable")
					);
				}
			}
			else
			{
				char		*cp;

				cp = strchr(arglex_value.alv_string, '=');
				if (!cp)
				{
					s = str_from_c(arglex_value.alv_string);
					string_list_append(&option.o_target, s);
					str_free(s);
				}
				else
				{
					s = str_from_c(arglex_value.alv_string);
					string_list_append(&option.o_vardef, s);
					str_free(s);
				}
			}
			break;

		case arglex_token_star:
			type = OPTION_STAR;
			goto normal_on;

		case arglex_token_star_not:
			type = OPTION_STAR;
			goto normal_off;

		case arglex_token_strip_dot:
			type = OPTION_STRIP_DOT;
			goto normal_on;

		case arglex_token_strip_dot_not:
			type = OPTION_STRIP_DOT;
			goto normal_off;

		case arglex_token_update:
			type = OPTION_UPDATE;
			goto normal_on;

		case arglex_token_update_not:
			type = OPTION_UPDATE;
			goto normal_off;

		case arglex_token_parallel:
			if (arglex() != arglex_token_number)
			{
				s = str_from_c("parallel_jobs=4");
				string_list_append(&option.o_vardef, s);
				str_free(s);
				continue;
			}
			s = str_format("parallel_jobs=%d", (int)arglex_value.alv_number);
			string_list_append(&option.o_vardef, s);
			str_free(s);
			break;

		case arglex_token_parallel_not:
			s = str_from_c("parallel_jobs=1");
			string_list_append(&option.o_vardef, s);
			str_free(s);
			break;

		case arglex_token_shallow:
			type = OPTION_SHALLOW;
			goto normal_on;

		case arglex_token_shallow_not:
			type = OPTION_SHALLOW;
			goto normal_off;
		}
		arglex();
	}
}


static void set_command_line_goals _((void));

static void
set_command_line_goals()
{
	string_ty	*name;

	name = str_from_c("command-line-goals");
	id_assign(name, id_variable_new(&option.o_target));
	str_free(name);
}


/*
 * NAME
 *	main - initial entry point for cook
 *
 * SYNOPSIS
 *	void main(int argc, char **argv);
 *
 * DESCRIPTION
 *	Main is the initial entry point for cook.
 *
 * RETURNS
 *	Exit is always through exit().
 *	The exit code will be 0 for success, or 1 for some error.
 */

int main _((int, char **));

int
main(argc, argv)
	int		argc;
	char		**argv;
{
	int		retval;

	/*
	 * initialize things
	 * (order is critical here)
	 */
	progname_set(argv[0]);
	str_initialize();
	id_initialize(); 
	lex_initialize();

	/*
	 * parse the command line
	 */
	arglex_init_from_env(argv[0], argtab);
	argparse(OPTION_LEVEL_ENVIRONMENT);

	/*
	 * parse the command line
	 */
	arglex_init(argc, argv, argtab);
	argparse(OPTION_LEVEL_COMMAND_LINE);

	option_tidy_up();

	log_open();

	/*
	 * turn on progress stars if they asked for them
	 */
	if (option_test(OPTION_STAR))
		star_enable();

	/*
	 * read in the cook book
	 *
	 * If there are #include-cooked directives,
	 * we may need to do it more than once.
	 */
	if (!option.o_book)
		fatal_intl(0, i18n("no book found"));
	for (;;)
	{
		int	status;
		long	j;

		builtin_initialize();

		/*
		 * instanciate the command line variable assignments
		 */
		for (j = 0; j < option.o_vardef.nstrings; ++j)
		{
			char		*s;
			char		*cp;
			string_ty	*name;
			string_ty	*value;
			string_list_ty	wl;
	
			s = option.o_vardef.string[j]->str_text;
			cp = strchr(s, '=');
			assert(cp);
			if (!cp)
				continue;
			name = str_n_from_c(s, cp - s); 
			value = str_from_c(cp + 1);
			str2wl(&wl, value, (char *)0, 0);
			str_free(value);
			id_assign(name, id_variable_new(&wl));
			str_free(name);
			string_list_destructor(&wl);
		}

		set_command_line_goals();

		parse(option.o_book);
		status = cook_auto_required();
		if (status < 0)
			quit(1);
		if (!status)
			break;
		id_reset();
		cook_reset();
	}

	/*
	 * work out what to cook.
	 * If no targets have been given, use the first explicit recipe.
	 */
	set_command_line_goals();
	if (!option.o_target.nstrings)
		cook_find_default(&option.o_target);
	assert(option.o_target.nstrings);

	/*
	 * cook the target
	 */
	if (option.pairs)
		retval = cook_pairs(&option.o_target);
	else if (option.script)
		retval = cook_script(&option.o_target);
	else
		retval = cook(&option.o_target);

	quit(retval);
	/*NOTREACHED*/
	return 0;
}
