/*
 *	cook - file construction tool
 *	Copyright (C) 1997 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to manipulate recipes
 */

#include <cook.h>
#include <dir_part.h>
#include <error_intl.h>
#include <graph/file.h>
#include <graph/file_list.h>
#include <graph/recipe.h>
#include <id.h>
#include <opcode/context.h>
#include <match.h>
#include <mem.h>
#include <option.h>
#include <os.h>
#include <recipe.h>
#include <star.h>
#include <stmt.h>
#include <trace.h>


/*
 * NAME
 *	graph_recipe_new
 *
 * SYNOPSIS
 *	graph_recipe_ty *graph_recipe_new(recipe_ty *);
 *
 * DESCRIPTION
 *	The graph_recipe_new function is used to allocate a new graph
 *	recipe instance in dynamic memory.
 *
 * RETURNS
 *	graph_recipe_ty *
 *
 * CAVEAT
 *	Use graph_recipe_delete when you are done with it.
 */

graph_recipe_ty *
graph_recipe_new(rp)
	recipe_ty	*rp;
{
	graph_recipe_ty	*grp;

	trace(("graph_recipe_new()\n{\n"/*}*/));
	grp = mem_alloc(sizeof(graph_recipe_ty));
	grp->reference_count = 1;
	grp->rp = recipe_copy(rp);
	grp->mp = 0;
	grp->input = graph_file_list_nrc_new();
	grp->output = graph_file_list_nrc_new();
	grp->ocp = 0;
	grp->single_thread = 0;
	grp->host_binding = 0;
	trace(("return %08lX;\n", (long)grp));
	trace((/*{*/"}\n"));
	return grp;
}


/*
 * NAME
 *	graph_recipe_delete
 *
 * SYNOPSIS
 *	void graph_recipe_delete(graph_recipe_ty *);
 *
 * DESCRIPTION
 *	The graph_recipe_delete function is used to release the
 *	resources held by a graph recipe instance in dynamic memory.
 */

void
graph_recipe_delete(grp)
	graph_recipe_ty *grp;
{
	trace(("graph_recipe_delete(grp = %08lX)\n{\n"/*}*/, (long)grp));
	assert(grp->reference_count > 0);
	grp->reference_count--;
	if (grp->reference_count > 0)
	{
		trace((/*{*/"}\n"));
		return;
	}
	recipe_delete(grp->rp);
	if (grp->mp)
		match_free(grp->mp);
	graph_file_list_nrc_delete(grp->input);
	graph_file_list_nrc_delete(grp->output);
	if (grp->ocp)
		opcode_context_delete(grp->ocp);
	if (grp->single_thread)
		string_list_delete(grp->single_thread);
	if (grp->host_binding)
		string_list_delete(grp->host_binding);
	mem_free(grp);
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	graph_recipe_copy
 *
 * SYNOPSIS
 *	graph_recipe_ty *graph_recipe_copy(graph_recipe_ty *);
 *
 * DESCRIPTION
 *	The graph_recipe_copy function is used to copy a graph recipe
 *	instance in dynamic memory.
 */

graph_recipe_ty *
graph_recipe_copy(grp)
	graph_recipe_ty *grp;
{
	assert(grp->reference_count > 0);
	grp->reference_count++;
	return grp;
}


/*
 * NAME
 *	graph_recipe_getpid
 *
 * SYNOPSIS
 *	int graph_recipe_getpid(graph_recipe_ty *);
 *
 * DESCRIPTION
 *	The graph_recipe_getpid function is used to get the process-id
 *	of the process to wait for.
 *
 * CAVEAT
 *	Must only be used when graph_recipe_run returns
 *	graph_walk_status_wait.
 */

int
graph_recipe_getpid(grp)
	graph_recipe_ty *grp;
{
	assert(grp);
	assert(grp->ocp);
	return opcode_context_getpid(grp->ocp);
}


/*
 * NAME
 *	graph_recipe_waited
 *
 * SYNOPSIS
 *	void graph_recipe_waited(graph_recipe_ty *, int);
 *
 * DESCRIPTION
 *	The graph_recipe_waited function is used to set the exit status
 *	after a waiting graph walk is about to resume.
 */

void
graph_recipe_waited(grp, status)
	graph_recipe_ty *grp;
	int		status;
{
	assert(grp);
	assert(grp->ocp);
	opcode_context_waited(grp->ocp, status);
}
