/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994, 1997, 1998 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to implement the builtin functions
 *
 * The builtin function all append their results to the supplied
 * `result' word list.  The first word of the `args' word list
 * is the name of the function.
 *
 * all of the functions return 0 in success, or -1 on error.
 *
 * Only a limited set of this are candidates for builtin functions,
 * these are 
 *	- string manipulation [dirname, stringset, ect ]
 *	- environment manipulation [getenv(3), etc]
 *	- stat(3) related functions [exists, mtime, pathname, etc]
 *	- launching OS commands [execute, collect]
 * The above list is though to be exhaustive.
 *
 * This explicitly and forever excluded from being a builtin function
 * is anything which known or understands the format of some secific 
 * class of files.
 *
 * Access to stdio(3) has been thought of, and explicitly avoided.
 * Mostly because a specialist program used through [collect]
 * will almost always be far faster.
 */

#include <builtin/match.h>
#include <error_intl.h>
#include <expr/position.h>
#include <match.h>
#include <trace.h>


/*
 * NAME
 *	builtin_match - wildcard mapping
 *
 * SYNOPSIS
 *	int builtin_match(string_list_ty *result, string_list_ty *args);
 *
 * DESCRIPTION
 *	Fromto is a built-in function of cook, described as follows:
 *	This function requires at least two arguments.
 *	Fromto gives the user access to the wildcard transformations
 *	available to cook.
 *	The first argument is the "from" form,
 *	the second argument is the "to" form.
 *
 * RETURNS
 *	All other arguments are mapped from one to the other.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a string_list_destructor() call.
 */

int
builtin_match(result, args, pp, ocp)
	string_list_ty	*result;
	const string_list_ty *args;
	const expr_position_ty *pp;
	const struct opcode_context_ty *ocp;
{
	int		j;
	match_ty	*field;

	trace(("match\n"));
	assert(result);
	assert(args);
	assert(args->nstrings);
	if (args->nstrings < 2)
	{
		sub_context_ty	*scp;

		scp = sub_context_new();
		sub_var_set(scp, "Name", "%S", args->string[0]);
		error_with_position
		(
			pp,
			scp,
			i18n("$name: requires one or more arguments")
		);
		sub_context_delete(scp);
		return -1;
	}
	for (j = 2; j < args->nstrings; j++)
	{
		field = match(args->string[1], args->string[j], pp);
		if (field == MATCH_ERROR)
			return -1;
		if (field)
		{
			string_list_append(result, str_true);
			match_free(field);
		}
		else
			string_list_append(result, str_false);
	}
	return 0;
}


/*
 * NAME
 *	builtin_match - wildcard mapping
 *
 * SYNOPSIS
 *	int builtin_match(string_list_ty *result, string_list_ty *args);
 *
 * DESCRIPTION
 *	Fromto is a built-in function of cook, described as follows:
 *	This function requires at least two arguments.
 *	Fromto gives the user access to the wildcard transformations
 *	available to cook.
 *	The first argument is the "from" form,
 *	the second argument is the "to" form.
 *
 * RETURNS
 *	All other arguments are mapped from one to the other.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a string_list_destructor() call.
 */

int
builtin_match_mask(result, args, pp, ocp)
	string_list_ty	*result;
	const string_list_ty *args;
	const expr_position_ty *pp;
	const struct opcode_context_ty *ocp;
{
	int		j;
	match_ty	*field;

	trace(("match_mask\n"));
	assert(result);
	assert(args);
	assert(args->nstrings);
	if (args->nstrings < 2)
	{
		sub_context_ty	*scp;

		scp = sub_context_new();
		sub_var_set(scp, "Name", "%S", args->string[0]);
		error_with_position
		(
			pp,
			scp,
			i18n("$name: requires one or more arguments")
		);
		sub_context_delete(scp);
		return -1;
	}
	for (j = 2; j < args->nstrings; j++)
	{
		field = match(args->string[1], args->string[j], pp);
		if (field == MATCH_ERROR)
			return -1;
		if (field)
		{
			string_list_append(result, args->string[j]);
			match_free(field);
		}
	}
	return 0;
}


/*
 * NAME
 *	builtin_fromto - wildcard mapping
 *
 * SYNOPSIS
 *	int builtin_fromto(string_list_ty *result, string_list_ty *args);
 *
 * DESCRIPTION
 *	Fromto is a built-in function of cook, described as follows:
 *	This function requires at least two arguments.
 *	Fromto gives the user access to the wildcard transformations
 *	available to cook.
 *	The first argument is the "from" form,
 *	the second argument is the "to" form.
 *
 * RETURNS
 *	All other arguments are mapped from one to the other.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a string_list_destructor() call.
 */

int
builtin_fromto(result, args, pp, ocp)
	string_list_ty	*result;
	const string_list_ty *args;
	const expr_position_ty *pp;
	const struct opcode_context_ty *ocp;
{
	int		j;
	match_ty	*field;

	trace(("fromto\n"));
	assert(result);
	assert(args);
	assert(args->nstrings);
	if (args->nstrings < 3)
	{
		sub_context_ty	*scp;

		scp = sub_context_new();
		sub_var_set(scp, "Name", "%S", args->string[0]);
		error_with_position
		(
			pp,
			scp,
			i18n("$name: requires two or more arguments")
		);
		sub_context_delete(scp);
		return -1;
	}
	for (j = 3; j < args->nstrings; j++)
	{
		field = match(args->string[1], args->string[j], pp);
		if (field == MATCH_ERROR)
			return -1;
		if (field)
		{
			string_ty	*s;

			s = reconstruct(args->string[2], field, pp);
			match_free(field);
			if (!s)
				return -1;
			string_list_append(result, s);
			str_free(s);
		}
		else
			string_list_append(result, args->string[j]);
	}
	return 0;
}
