/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <linux/linkage.h>

#include "bootinfo.h"
#include "loader.h"
#include "version.h"
#include "loaderlib.h"
#include "mvmebug.h"

/*--------------------------------------------------------------------------*/

extern const FILEMAP *	find_file_map(const char *path);
extern char				debug_mode;

/*--------------------------------------------------------------------------*/

static unsigned long	current_file_position	= 0;
static const FILEMAP	*current_file_map		= NULL;
static const FILEMAP	*current_file_frag		= NULL;
static unsigned long	current_frag_position	= 0;
static unsigned char	disk_id                 = 0xff;

/*--------------------------------------------------------------------------*/
/* Board ID data structure
 *
 * Note, bytes 12 and 13 are board no in BCD (0162,0166,0167,0177,etc)
 */

typedef struct {
        char    bdid[4];
        u_char  rev, mth, day, yr;
        u_short size, reserved;
        u_short brdno;
        char	brdsuffix[2];
        u_long  options;
        u_short clun, dlun, ctype, dnum;
        u_long  option2;

} t_bdid, *p_bdid;

/*--------------------------------------------------------------------------*/

static
u_long
get_brd_id
(	void
)
{	static u_long	brd_id = 0;

	if (brd_id == 0)
	{
		p_bdid			pkt;

		pkt = (p_bdid) MVMEBug_brd_id();

		if (*(ulong *)pkt->bdid != 0x42444944)
		{
			panic("Cannot identify board");
		}

		brd_id = (u_long) pkt->brdno;
	}

	return brd_id;
}

/*--------------------------------------------------------------------------*/

void
loader_init
(	unsigned long	arg
)
{
	disk_id = (unsigned char) arg;

	put_str("\nMotorola MVME Linux Loader V" VERNUMB "\n\n");
}

/*--------------------------------------------------------------------------*/

#define BCD2BIN(x)	(((x)>>4)*10 + ((x)&0x0f))

unsigned long
get_time
(	void
)
{	u_char buf[8];

	MVMEBug_rtc_read(buf);
	return (BCD2BIN(buf[4]) * 24 + BCD2BIN(buf[5])) * 60 + BCD2BIN(buf[6]);
}

/*--------------------------------------------------------------------------*/

void
mem_clear
(	void			*mem,
	unsigned long	count
)
{
	u_long a = (u_long)mem;

	while ((a & 3) && count)
	{
		*(char *)a = 0;
		a++;
		count--;
	}
	while (count > 3)
	{
		*(long *)a = 0;
		a += 4;
		count -= 4;
	}
	while (count)
	{
		*(char *)a = 0;
		a++;
		count--;
	}
}

/*--------------------------------------------------------------------------*/

void
mem_move
(	void			*dest,
	const void		*srce,
	unsigned long	count
)
{
	char			*d = dest;
	const char		*s = srce;

	if (d > s)
	{
		d += count;
		s += count;
		while (count--)
		{
			*--d = *--s;
		}
	}
	else
	{
		while (count--)
		{
			*d++ = *s++;
		}
	}
}

/*--------------------------------------------------------------------------*/

int
mem_cmp
(	void			*mem1,
	void			*mem2,
	unsigned long	count
)
{
	u_char			c   = 0;
	u_char			*p1 = (u_char *) mem1;
	u_char			*p2 = (u_char *) mem2;

	while (c == 0 && count--)
		c = *p1++ - *p2++;

	return (int) c;
}

/*--------------------------------------------------------------------------*/
/* printf style output formatter
 */

int
vsoutput
(	void 		(*output)(const int),
	const char	*fmt,
	va_list		va
)
{	int			c;
	int			pos   = 0;
#define			BUFSZ	15

	while ((c = *fmt++))
	{
	    if (c != '%')
	    {
			(*output)(c);
			pos++;
		}
		else
		{
			enum {
				FF_DEFAULT = 0,
				FF_DUMP    = 0x01,	/* dump string to output	*/
				FF_ALT     = 0x02,	/* #, alternate format		*/
				FF_SHORT   = 0x04,	/* h, short arg				*/
				FF_LONG    = 0x08,	/* l, long arg				*/
				FF_ZERO    = 0x10,	/* 0, zero fill				*/
				FF_LEFT    = 0x20,	/* -, left adjust			*/
				FF_PREC    = 0x40,	/* .*, precision			*/
				FF_NEG     = 0x80	/* signed arg				*/
			}				flags   = FF_DEFAULT;
			long			n;
			unsigned long	u;
			char			buf[BUFSZ + 1];
			char			*p    = buf + BUFSZ;
			int				sign  = '-';
			int				width = 0;
			int				prec  = 0;

			*p = '\0';

			/* scan flags */
			while (1)
			{
				switch (c = *fmt++)
				{
					case '0':
					{
						flags |= FF_ZERO;
						break;
					}

				  	case '#':		/* alternate format */
				  	{
						flags |= FF_ALT;
						break;
					}

					case ' ':		/* blank sign */
					{
						sign = ' ';
						break;
					}

					case '+':		/* +/- sign */
					{
						sign = '+';
						break;
					}

				  	case '-':		/* left justified */
				  	{
						flags |= FF_LEFT;
						break;
					}

				  	default:
				  	{
						goto scan_width;
					}
				}
			}

		scan_width:
			/* scan width */
		  	if (c == '*')
		  	{
		  		/* width from arg list */
				width = va_arg(va, int);
				c     = *fmt++;
			}
			else
			{
				while ('0' <= c && c <= '9')
				{
					width = width * 10 + (c - '0');
					c     = *fmt++;
				}
			}

			if (c == '.')
			{
				/* scan precision */
				flags |= FF_PREC;
				c      = *fmt++;

				if (c == '*')
				{
					/* precision from arg list */
					prec = va_arg(va, int);
					c    = *fmt++;
				}
				else
				{
					while ('0' <= c && c <= '9')
					{
						prec = prec * 10 + (c - '0');
						c    = *fmt++;
					}
				}
			}

			/* length modifiers */
			if (c == 'h')
			{
				flags |= FF_SHORT;
				c      = *fmt++;
			}
			else if (c == 'l')
			{
				flags |= FF_LONG;
				c      = *fmt++;
			}

			/* do conversion */
			switch (c)
			{
				case '%':		/* %% -> % */
				{
					(*output)(c);
					pos++;
					break;
				}

				case 'n':		/* save position */
				{
					*va_arg(va, int *) = pos;
					break;
				}

				case 'c':		/* character */
				{
					u = (flags & FF_SHORT) ? va_arg(va, unsigned short)
					  : (flags & FF_LONG ) ? va_arg(va, unsigned long )
				  	  : va_arg(va, unsigned int);
					*--p = u;
					flags |= FF_DUMP;
					break;
				}

			  	case 's':		/* string */
			  	{
					if ((p = va_arg(va, char *)) == NULL)
					{
						p = "";
					}

					if ((flags & FF_PREC) && str_len(p) > prec)
					{
						pos += prec;

						while (--prec >= 0)
						{
							(*output)(*p++);
						}

						break;
					}

					flags |= FF_DUMP;
					break;
				}

		  		case 'i': case 'd': case 'u': /* decimal */
		  		{
					if (c != 'u')
					{
						/* signed */
						n = (flags & FF_SHORT) ? va_arg(va, short)
						  : (flags & FF_LONG ) ? va_arg(va, long )
						  : va_arg(va, int);

						if (n < 0)
						{
							flags |= FF_NEG;
						}

						u = (n < 0) ? -n : n;
					}
					else
					{
						u = (flags & FF_SHORT) ? va_arg(va, unsigned short)
						  : (flags & FF_LONG ) ? va_arg(va, unsigned long )
						  : va_arg(va, unsigned int);
					}

					do	{
						*--p = '0' + u % 10;
						u   /= 10;
					} while (u != 0);

					prec -= buf + BUFSZ - p;

					while (--prec >= 0)
					{
						*--p = '0';
					}

					if (flags & FF_NEG)
					{
						*--p = '-';
					}
					else
					{
						if (sign != '-')
						{
							*--p = (sign == '+') ? '+' : ' ';
						}
					}

					flags |= FF_DUMP;
					break;
				}

				case 'x': case 'X':	/* hex, Hex */
				{
					u = (flags & FF_SHORT) ? va_arg(va, unsigned short)
					  : (flags & FF_LONG ) ? va_arg(va, unsigned long )
					  : va_arg(va, unsigned int);
					do	{
						*--p = "0123456789ABCDEF"[u%16];
						u   /= 16;
					} while (u);

					prec -= buf + BUFSZ - p;

					while (--prec >= 0)
					{
						*--p = '0';
					}

					if (flags & FF_ALT)
					{
						*--p = 'x';
						*--p = '0';
					}

					flags |= FF_DUMP;
					break;
				}

				case 'o':		/* octal */
				{
					u = (flags & FF_SHORT) ? va_arg(va, unsigned short)
					  : (flags & FF_LONG ) ? va_arg(va, unsigned long )
					  : va_arg(va, unsigned int);
					do	{
						*--p = '0' + u % 8;
						u   /= 8;
					} while (u);

					prec -= buf + BUFSZ - p;

					while (--prec >= 0)
					{
						*--p = '0';
					}

					if ((flags & FF_ALT) && *p != '0')
					{
						*--p = '0';
					}

					flags |= FF_DUMP;
					break;
				}

				default:		/* error */
				{
					(*output)('%');
					(*output)(c);
					pos += 2;
					break;
				}
			}

			/* copy adjusted string "p" to output */
			if (flags & FF_DUMP)
			{
				int len = str_len(p);
				int pad = width - len;

				if (!(flags & FF_LEFT))
				{
					int padder = (flags & FF_ZERO) ? '0' : ' ';

					while (--pad >= 0)
					{
						(*output)(padder);
					}
				}

				while (*p)
				{
					(*output)(*p++);
				}

				if (flags & FF_LEFT)
				{
					while (--pad >= 0)
					{
						(*output)(' ');
					}
				}

				pos += (len < width) ? width : len;
			}
		}
	}

	return pos;
}

/*--------------------------------------------------------------------------*/
/* Print character.
 */

void
put_char
(	const int	c
)
{
	if (c == '\n')
		MVMEBug_putchar('\r');

	MVMEBug_putchar(c);
}

/*--------------------------------------------------------------------------*/
/* Print formated string.
 */

void
Printf
(	const char	*fmt,
	...
)
{	va_list		args;

    va_start(args, fmt);
	vsoutput(put_char, fmt, args);
    va_end(args);
}

/*--------------------------------------------------------------------------*/
/* Print unformated string.
 */

void
put_str
(	const char	*str
)
{
	while (*str)
		put_char(*str++);
}

/*--------------------------------------------------------------------------*/
/* Wait for and return character from keyboard.
 */

int
get_char
(	unsigned long	timeout		/* maximum time to wait for character		*/
)
{
	if (timeout)
	{
		/* get timeout second */
		timeout += get_time();

		while (MVMEBug_getchar_status() == 0)
		{
			/* check for timeout */
			if (get_time() > timeout)
			{
				/* timeout */
				break;
			}
		}
		if (MVMEBug_getchar_status() == 0)
			return -1;
	}

	return MVMEBug_getchar();
}

/*--------------------------------------------------------------------------*/
/* Report fatal error
 */

void
panic
(	const char	*fmt,
	...
)
{	va_list		args;

	put_str("\nLILO Panic: ");
    va_start(args, fmt);
 	vsoutput(put_char, fmt, args);
    va_end(args);
    put_char('\n');
 
    while (1)
		;
}

/*--------------------------------------------------------------------------*/

void
set_disk_id
(	u_long		dlun
)
{
	disk_id = (unsigned char) dlun;
}

/*--------------------------------------------------------------------------*/

int
file_open
(	const char	*path
)
{
    if ((current_file_map = find_file_map(path)) == NULL)
    {
		return -1;
	}

	current_file_position = 0;
	current_frag_position = 0;
	current_file_frag = MAP_FIRSTFRAG(current_file_map);

	if (debug_mode)
	{
		const FILEMAP *p = current_file_frag;
		u_long n = MAP_NUMFRAGS(current_file_map);

		Printf("\nOPEN:  file map at %08lx, %ld entries\n",
			current_file_map, n);
		while (n)
		{
			Printf("  Frag offset %08lx, length %08lx\n",
				p->offset, p->length);
			p++;
			n--;
		}
	}

    return 0;
}

/*--------------------------------------------------------------------------*/

long
file_tell
(	void
)
{
	return current_file_position;
}

/*--------------------------------------------------------------------------*/

int
file_seek
(	int			where,
	int			whence
)
{	int			filesize;
	int			n;

	/* if no open file */
	if (current_file_map == NULL)
	{
		return -1;
	}

	/* get file size */
	filesize = MAP_FILESIZE(current_file_map);

	switch (whence)
	{
		case SEEK_SET:	break;

		case SEEK_CUR:	where += current_file_position;
						break;

		case SEEK_END:	where += filesize;
						break;

		default:		return -1;
	}

	/* only allow seek within extents of file */
    if (where < 0 || where > filesize)
    {
		return -1;
	}

	current_file_position = where;

	current_file_frag     = MAP_FIRSTFRAG(current_file_map);
	current_frag_position = 0;

	n = current_file_position;
	while (n)
	{
		if (n <= current_file_frag->length)
		{
			current_frag_position = n;
			n = 0;
		}
		else
		{
			n -= current_file_frag->length;
			current_file_frag++;
		}
	}
	if (current_frag_position == current_file_frag->length)
	{
		current_file_frag++;
		current_frag_position = 0;
	}

	if (debug_mode)
	{
		Printf("SEEK: current_file_position %08lx\n",
			current_file_position);
		Printf("  frag->offset %08lx, frag->len %08lx\n",
			current_file_frag->offset, current_file_frag->length);
		Printf("  current_frag_position %08lx\n", current_frag_position);
	}
    return where;
}

/*--------------------------------------------------------------------------*/

int
file_read
(	char	*buf,
	int		count
)
{	int		filesize;
	int		cnt = count;
	DSKRD_CMND	cdb;

	if (debug_mode)
		Printf("READ: buf=%08lx, len=%08lx\n", buf, cnt);

	/* if no open file */
	if (current_file_map == NULL)
	{
		return -1;
	}

	/* if illegal count */
	if (cnt < 0)
	{
		return -1;
	}

	filesize = MAP_FILESIZE(current_file_map);

	/* limit to amount of data left */
    if (cnt > (filesize - current_file_position))
    {
		cnt = filesize - current_file_position;
	}

	while (cnt)
	{
		int l = cnt;

		if (l > current_file_frag->length - current_frag_position)
			l = current_file_frag->length - current_frag_position;

		/* read l bytes */
		cdb.clun = 0;
		cdb.dlun = disk_id;
		cdb.status = 0xdead;
		cdb.address = (u_long)buf;
		cdb.block = (current_file_frag->offset + current_frag_position) >> 8;
		cdb.count = ((l + 511) >> 9) << 1;
		cdb.flags = 0;
		cdb.mod = 0;
		if (debug_mode)
			Printf("READ: blk=%08lx, cnt=%08lx\n", cdb.block, cdb.count);
		/* Holes in the file result in fragments at disk offset 0;
		 * our malloc function will have zero-ed memory, so we
		 * can just skip over this read.
		 */
		if (cdb.block)
		{
			MVMEBug_disk_read(&cdb);
			if (cdb.status)
			{
				u_long *p = (u_long *)&cdb;

				Printf ("READ FAILED: status=%08lx\n", cdb.status);
				if (debug_mode)
					Printf("  %08ld %08ld %08ld %08ld\n",
						*p, *(p+1), *(p+2), *(p+3));
				return -1;
			}
		}

		cnt -= l;
		current_frag_position += l;
		current_file_position += l;
		buf += l;
		if (current_frag_position == current_file_frag->length)
		{
			current_file_frag++;
			current_frag_position = 0;
		}
	}

	if (debug_mode)
	{
		Printf("  current_file_position %08lx\n",
			current_file_position);
		Printf("  frag->offset %08lx, frag->len %08lx\n",
			current_file_frag->offset, current_file_frag->length);
		Printf("  current_frag_position %08lx\n", current_frag_position);
	}
    return count;
}

/*--------------------------------------------------------------------------*/

void
file_close
(	void
)
{
	current_file_map = NULL;
}

/*--------------------------------------------------------------------------*/

unsigned long
get_compat_booti_version
(	void
)
{
	switch (get_brd_id())
	{
		case 0x0162:
		case 0x0172: return COMPAT_MVME162_BOOTI_VERSION;
		default    : return COMPAT_MVME167_BOOTI_VERSION;
	}
}

/*--------------------------------------------------------------------------*/

unsigned long
get_booti_version
(	void
)
{
	return MVME16x_BOOTI_VERSION;
}

/*--------------------------------------------------------------------------*/

unsigned long
get_compat_machtype
(	void
)
{
	switch (get_brd_id())
	{
		case 0x0162:
		case 0x0172: return COMPAT_MACH_MVME162;
		default    : return COMPAT_MACH_MVME167;
	}
}

/*--------------------------------------------------------------------------*/

unsigned long
get_machtype
(	void
)
{
	return MACH_MVME16x;
}

/*--------------------------------------------------------------------------*/

int
can_do_symbols
(	void
)
{
	return FALSE;
}

/*--------------------------------------------------------------------------*/

void
clear_symbols
(	void
)
{
}

/*--------------------------------------------------------------------------*/

int
add_symbol
(	char		*data
)
{
	return 0;
}

/*--------------------------------------------------------------------------*/
/*
 *	This assembler code is copied into local storage, and then executed.
 *	It copies the kernel and ramdisk images to their final resting places.
 *
 *	It is called with:
 *
 *      a0 = address of this code (very top of memory)
 *      a1 = kernel destination address (low memory 0x1000)
 *		a2 = kernel source address
 *		a3 = ramdisk destination address (just below this code)
 *		a4 = ramdisk source address
 *		d0 = kernel size + size of bootinfo data rounded up next multiple of 4
 *		d1 = ramdisk size rounded to next multiple of 1K
 *      d2 = non-zero if monitor should be called
 */

__asm(".text\n"
__ALIGN_STR "\n"
".globl " SYMBOL_NAME_STR(startcode_beg) ";\n"
".globl " SYMBOL_NAME_STR(startcode_end) ";\n"
SYMBOL_NAME_STR(startcode_beg) ":
								| /* copy the ramdisk to the top of memory */
								| /* (from back to front) */
		addl	%d1,%a4			| src   = (u_long *) (rd_src + rd_size);
		movel	%a3,%a5
		addl	%d1,%a5			| dest  = (u_long *) (rd_dest + rd_size);

		bras	2f				| do
1:		movel	-(%a4),-(%a5)	|   *--dest = *--src;
2:		cmpl	%a3,%a5			| while (dest > ramdisk_dest)
		jgt		1b				| 

								| /* copy kernel text and data, and bootinfo */
		movel	%a2,%a4			| limit = (u_long *) (kernel_src + kernel_size);
		addl	%d0,%a4
		movel	%a1,%a5			| dest  = (u_long *) kernel_dest

		bras	2f				| do
1:		movel	(%a2)+,(%a5)+	|  *dest++ = *src++;
2:		cmpl	%a4,%a2			| while (src < limit)
		jlt		1b				|

		dc.w	0xf498			| cinva	ic | invalidate instruction cache

		tstl	%d2				| call monitor?
		jeq		1f				| branch if not

		trap	#15
		dc.w	0x0063

1:		jmp		(%a1)			| start kernel or enter BVMBug
"
SYMBOL_NAME_STR(startcode_end) ":
");

/*--------------------------------------------------------------------------*/

void
print_model
(	int		cpu,
	int		hasfpu
)
{
	Printf("MVME%x", get_brd_id());
}

/*-----------------------------< end of file >------------------------------*/
