/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>

#include <sys/ioctl.h>
#include <sys/fcntl.h>
#include <sys/file.h>
#include <sys/types.h>
#include <sys/stat.h>
#define _LINUX_STAT_H	/* prevent inclusion of <linux/stat.h> from
						 * <linux/fs.h> redefintions cause lots of warnings
						 */

#include <linux/hdreg.h>
#include <linux/fs.h>
#include <linux/major.h>
 

#include "loader.h"
#include "vmelilo.h"

extern int stat(const char *file, struct stat *buf);

/*--------------------------------------------------------------------------*/

int
is_scsi_disk
(	unsigned	rdev
)
{
 	return ((MAJOR(rdev) == SCSI_DISK_MAJOR)
	&&		(MINOR(rdev) &  0x0f           ));
}

/*--------------------------------------------------------------------------*/
/* Get the Geometry for the Partition a File belongs to
 */

static
void
geometry
(	const char			*name,
	dev_t				*device,
	u_long				*start
)
{	struct hd_geometry	geo;
	char				devname[PATH_MAX+1];
	struct stat			info;
	DIR					*dir;
	const struct dirent	*dirent;
	int					fh;
	int					found = 0;

	if (stat(name, &info) == -1)
	{
		error_stat(name);
	}

	if (!S_ISREG(info.st_mode))
	{
		die("%s is not a regular file\n", name);
	}
	*device = info.st_dev;

	if (!(dir = opendir("/dev")))
	{
		error_opendir("/dev");
	}

	while ((dirent = readdir(dir)))
	{
		sprintf(devname, "/dev/%s", dirent->d_name);

		if ((stat(devname, &info) != -1)
		&&	S_ISBLK(info.st_mode)
		&&	(info.st_rdev == *device))
		{
			if (MAJOR(*device) == LOOP_MAJOR)
			{
				geo.start = 0;
			}
			else
			{
				if ((fh = open(devname, O_RDONLY)) == -1)
				{
					error_open(devname);
				}
	
				if (ioctl(fh, HDIO_GETGEO, &geo) == -1)
				{
					error_ioctl(devname, "HDIO_GETGEO");
				}
	
				close(fh);
			}
			found = 1;
			break;
		}
	}

	closedir(dir);

	if (found)
	{
		*start = 512 * geo.start;
	}
	else
	{
		die("Can't find special device entry for file `%s'\n", name);
	}
}


/*--------------------------------------------------------------------------*/
/* Create a Map for a File
 */

const
FILEMAP *
create_file_map
(	const char	*name
)
{	int			fh;
	int			size;
	int			i;
	int			extend;
	dev_t		device;
	u_long		numblks;
	u_long		maxblks;
	u_long		offset;
	u_long		start;
	u_long		blksize;
	FILEMAP		*map;

	/* open file */
	if ((name == NULL) || ((fh = open(name, O_RDONLY)) == -1))
	{
		return NULL;
	}

	/* get file size */
	if ((size = lseek(fh, 0, SEEK_END)) == -1)
	{
		error_seek(name);
	}

	/* get geometry of files partition */
	geometry(name, &device, &start);

	/* make sure file is on boot device */
	if (MAJOR(device) != MAJOR(boot_device))
	{
		die("File `%s' must reside on physical device `%s'\n",
				name, config.boot_device_name);
	}

	switch (MAJOR(boot_device))
	{
		case LOOP_MAJOR:
		{
			if (MINOR(device) != MINOR(boot_device))
			{
				die("File `%s' must reside on loopback filesystem\n"
					"associated with `%s'\n",
						name, config.boot_device_name);
			}
			break;
		}

		case RAMDISK_MAJOR:
		{
			if (MINOR(device) != MINOR(boot_device))
			{
				die("File `%s' must reside on ramdisk `%s'\n",
						name, config.boot_device_name);
			}
			break;
		}

		case SCSI_DISK_MAJOR:
		{
			if ((MINOR(device) & 0xf0) != (MINOR(boot_device) & 0xf0))
			{
				die("File `%s' must reside on physical SCSI device `%s'\n",
						name, config.boot_device_name);
			}
			break;
		}

		default:
		{
			die("Major device %d not supported (yet)\n", (int) MAJOR(device));
			break;
		}
	}

	/* get file system block size */
	if (ioctl(fh, FIGETBSZ, &blksize) == -1)
	{
		error_ioctl(name, "FIGETBSZ");
	}

	/* get number of file system blocks */
	maxblks = (size + blksize - 1) / blksize;

	/* allocate for worst case */
	if ((map = malloc((maxblks + 1) * sizeof(FILEMAP))) == NULL)
	{
		error_nomemory();
	}

	numblks = 0;
	for (i = 0; i < maxblks; i++)
	{
		/* convert block offset in file to block offset in partition */
		offset = i;
		if (ioctl(fh, FIBMAP, &offset) == -1)
		{
			error_ioctl(name, "FIBMAP");
		}

		/* if not a hole */
		if (offset)
		{
			/* convert partition block offset to device byte offset */
			offset = start + offset * blksize;
		}

		/* say create a new fragment */
		extend = FALSE;

		/* if some fragments already exist */
		if (numblks != 0)
		{
			/* not a hole */
			if (offset)
			{
				/* if fragment starts at end of last fragment */
				if (offset == (map[numblks].offset + map[numblks].length))
				{
					/* say extend old fragment */
					extend = TRUE;
				}
			}

			/* a hole */
			else
			{
				/* if last fragment was a hole as well */
				if (map[numblks].offset == 0)
				{
					/* say extend old fragment */
					extend = TRUE;
				}
			}
		}

		if (extend)
		{
			/* extend old framgent by one block */
			map[numblks].length += blksize;
		}
		else
		{
			/* new fragment */
			numblks++;
			map[numblks].offset = offset;
			map[numblks].length = blksize;
		}
	}

	/* record file size and number of fragments in first entry */
	MAP_FILESIZE(map) = size;
	MAP_NUMFRAGS(map) = numblks;

	/* finish with file */
	close(fh);

	/* downsize filemap */
	if (numblks != maxblks)
	{
		if ((map = realloc(map, (numblks + 1) * sizeof(FILEMAP))) == NULL)
		{
			error_nomemory();
		}
	}

	return map;
}


/*--------------------------------------------------------------------------*/
/* Create the File Name for the Boot Block Backup
 */

const char *
create_backup_filename
(	void
)
{	int		size;
	char	*s;

	size = strlen(LILO_BACKUPFILEPREFIX) + 7;

	if (root_path)
	{
		size += strlen(root_path);
	}

	if ((s = malloc(size)) == NULL)
	{
		error_nomemory();
	}

	sprintf(s, "%s%s.%02x.%02x",
		root_path ? root_path : "", LILO_BACKUPFILEPREFIX,
			(int)MAJOR(boot_device), (int)MINOR(boot_device));

	return s;
}


/*-----------------------------< end of file >------------------------------*/
