/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#include <stdarg.h>
#include <linux/linkage.h>

#include "bootinfo.h"
#include "loader.h"
#include "version.h"
#include "loaderlib.h"
#include "bvmbug.h"

/*--------------------------------------------------------------------------*/

extern const FILEMAP *	find_file_map(const char *path);
extern char				debug_mode;

/*--------------------------------------------------------------------------*/

static unsigned long	current_file_position	= 0;
static const FILEMAP	*current_file_map		= NULL;

/*--------------------------------------------------------------------------*/

void
loader_init
(	unsigned long	arg
)
{
	put_str("BVME4000/6000 Linux Loader V" VERNUMB "\n\n");
}

/*--------------------------------------------------------------------------*/

void
mem_clear
(	void			*mem,
	unsigned long	count
)
{
	BVMBug_memset(mem, 0, count);
}

/*--------------------------------------------------------------------------*/

void
mem_move
(	void			*dest,
	const void		*srce,
	unsigned long	count
)
{
#if 0
	char			*d = dest;
	const char		*s = srce;

	if (d > s)
	{
		d += count;
		s += count;
		while (count--)
		{
			*--d = *--s;
		}
	}
	else
	{
		while (count--)
		{
			*d++ = *s++;
		}
	}
#else
	BVMBug_memmove(dest, srce, count);
#endif
}

/*--------------------------------------------------------------------------*/

int
mem_cmp
(	void			*mem1,
	void			*mem2,
	unsigned long	count
)
{
	return BVMBug_memcmp(mem1, mem2, count);
}

/*--------------------------------------------------------------------------*/
/* Print character.
 */

void
put_char
(	const int	c
)
{
	BVMBug_putchar(c);
}

/*--------------------------------------------------------------------------*/
/* Print formated string.
 */

void
Printf
(	const char	*fmt,
	...
)
{	va_list		args;

    va_start(args, fmt);
    BVMBug_print(fmt, args);
    va_end(args);
}

/*--------------------------------------------------------------------------*/
/* Print unformated string.
 */

void
put_str
(	const char	*str
)
{
	BVMBug_putstr(str);
}

/*--------------------------------------------------------------------------*/

unsigned long
get_time
(	void
)
{	unsigned long	date;

	BVMBug_gettime(NULL, &date);
	return date;
}

/*--------------------------------------------------------------------------*/
/* Wait for and return character from keyboard.
 */

int
get_char
(	unsigned long	timeout		/* maximum time to wait for character		*/
)
{	int				c;

	if (timeout)
	{
		/* get timeout second */
		timeout += get_time();

		while ((c = BVMBug_getchar_nowait()) == -1)
		{
			/* check for timeout */
			if (get_time() > timeout)
			{
				/* timeout */
				break;
			}
		}

		return c;
	}

	return BVMBug_getchar();
}

/*--------------------------------------------------------------------------*/
/* Report fatal error
 */

void
panic
(	const char	*fmt,
	...
)
{	va_list		args;

	put_str("\nLILO Panic: ");
    va_start(args, fmt);
    BVMBug_print(fmt, args);
    va_end(args);
    put_char('\n');
 
    while (1)
		;
}

/*--------------------------------------------------------------------------*/

int
file_open
(	const char	*path
)
{
    if ((current_file_map = find_file_map(path)) == NULL)
    {
		return -1;
	}

	if (BVMBug_file_open(	MAP_FILESIZE (current_file_map),
							MAP_NUMFRAGS (current_file_map),
							MAP_FIRSTFRAG(current_file_map)) != 0)
	{
		return -1;
	}

	current_file_position = 0;

    return 0;
}

/*--------------------------------------------------------------------------*/

long
file_tell
(	void
)
{
	return current_file_position;
}

/*--------------------------------------------------------------------------*/

int
file_seek
(	int			where,
	int			whence
)
{	int			filesize;

	/* if no open file */
	if (current_file_map == NULL)
	{
		return -1;
	}

	/* get file size */
	filesize = MAP_FILESIZE(current_file_map);

	switch (whence)
	{
		case SEEK_SET:	break;

		case SEEK_CUR:	where += current_file_position;
						break;

		case SEEK_END:	where += filesize;
						break;

		default:		return -1;
	}

	/* only allow seek within extents of file */
    if (where < 0 || where > filesize)
    {
		return -1;
	}

	current_file_position = where;

    return where;
}

/*--------------------------------------------------------------------------*/

int
file_read
(	char	*buf,
	int		count
)
{	int		filesize;

	/* if no open file */
	if (current_file_map == NULL)
	{
		return -1;
	}

	/* if illegal count */
	if (count < 0)
	{
		return -1;
	}

	filesize = MAP_FILESIZE(current_file_map);

	/* limit to amount of data left */
    if (count > (filesize - current_file_position))
    {
		count = filesize - current_file_position;
	}

	if (count)
	{
		if (BVMBug_file_read(buf, current_file_position, count) != 0)
		{
			return -1;
		}

		current_file_position += count;
	}

    return count;
}

/*--------------------------------------------------------------------------*/

void
file_close
(	void
)
{
	/* if file is open */
	if (current_file_map != NULL)
	{
		current_file_map = NULL;
		BVMBug_file_close();
	}
}

/*--------------------------------------------------------------------------*/

unsigned long
get_compat_booti_version
(	void
)
{
	return COMPAT_BVME6000_BOOTI_VERSION;
}

/*--------------------------------------------------------------------------*/

unsigned long
get_booti_version
(	void
)
{
	return BVME6000_BOOTI_VERSION;
}

/*--------------------------------------------------------------------------*/

unsigned long
get_compat_machtype
(	void
)
{
	return COMPAT_MACH_BVME6000;
}

/*--------------------------------------------------------------------------*/

unsigned long
get_machtype
(	void
)
{
	return MACH_BVME6000;
}

/*--------------------------------------------------------------------------*/

int
can_do_symbols
(	void
)
{
	return TRUE;
}

/*--------------------------------------------------------------------------*/

void
clear_symbols
(	void
)
{
	BVMBug_delete_symbol(NULL);
}

/*--------------------------------------------------------------------------*/

int
add_symbol
(	char			*data
)
{	unsigned long	value;
	char			*p;
	int				type;

	/* skip white space */
	while (*data && *data < '!')
		data++;

	/* find end of value */
	for (p = data; *p >= '!'; p++)
		;

	/* premature end of line */
	if (*p == '\0') return 0;

	/* null terminate value */
	*p++ = '\0';

	/* evaluate */
	if (BVMBug_atoi(data, 16, &value) != 0)
	{
		return 0;
	}

	/* skip white space */
	while (*p && *p < '!')
		p++;

	/* premature end of line */
	if (*p == '\0') return 0;

	/* get symbol type */
	type = *p++;	

	/* premature end of line */
	if (*p == '\0') return 0;

	/* skip white space */
	while (*p && *p < '!')
		p++;

	/* premature end of line */
	if (*p == '\0') return 0;

	/* remember start of symbol name */
	data = p;

	/* find end of symbol name */
	while (*p >= '!')
		p++;

	/* null terminate symbol name */
	*p = '\0';

	/* get symbol type */
	type = (type == 'T' || type == 't') ? 0 : 1;

	if (BVMBug_define_symbol(data, value, type) == OUT_OF_MEMORY)
	{
		Printf("\nBVMBug symbol table full\n");
		return -1;
	}

	return 0;
}

/*--------------------------------------------------------------------------*/
/*
 *	This assembler code is copied into BVMBug local storage, and then executed.
 *	It copies the kernel and ramdisk images to their final resting places.
 *
 *	It is called with:
 *
 *      a0 = address of this code (very top of memory)
 *      a1 = kernel destination address (low memory 0x1000)
 *		a2 = kernel source address
 *		a3 = ramdisk destination address (just below this code)
 *		a4 = ramdisk source address
 *		d0 = kernel size + size of bootinfo data rounded up next multiple of 4
 *		d1 = ramdisk size rounded to next multiple of 1K
 *      d2 = non-zero if BVMBug should be called
 */

__asm(".text\n"
__ALIGN_STR "\n"
".globl " SYMBOL_NAME_STR(startcode_beg) ";\n"
".globl " SYMBOL_NAME_STR(startcode_end) ";\n"
SYMBOL_NAME_STR(startcode_beg) ":
								| /* copy the ramdisk to the top of memory */
								| /* (from back to front) */
		addl	%d1,%a4			| src   = (u_long *) (rd_src + rd_size);
		movel	%a3,%a5
		addl	%d1,%a5			| dest  = (u_long *) (rd_dest + rd_size);

		bras	2f				| do
1:		movel	-(%a4),-(%a5)	|   *--dest = *--src;
2:		cmpl	%a3,%a5			| while (dest > ramdisk_dest)
		jgt		1b				| 

								| /* copy kernel text and data, and bootinfo */
		movel	%a2,%a4			| limit = (u_long *) (kernel_src + kernel_size);
		addl	%d0,%a4
		movel	%a1,%a5			| dest  = (u_long *) kernel_dest

		bras	2f				| do
1:		movel	(%a2)+,(%a5)+	|  *dest++ = *src++;
2:		cmpl	%a4,%a2			| while (src < limit)
		jlt		1b				|

		dc.w	0xf498			| cinva	ic | invalidate instruction cache

		tstl	%d2				| call BVMbug?
		jeq		1f				| branch if not

		movel	%a1,-(%a7)		| return to start of kernel (kernel_dest)
		movel	#0xf8000404,%a1	| get BVMBug entry point
		moveq	#22,%d0			| BVMBug enter

1:		jmp		(%a1)			| start kernel or enter BVMBug
"
SYMBOL_NAME_STR(startcode_end) ":
");

/*--------------------------------------------------------------------------*/

void
print_model
(	int		cpu,
	int		hasfpu
)
{
	Printf("BVME%ld00", cpu);
}

/*-----------------------------< end of file >------------------------------*/
