/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include "bvmbug.h"
#include "loader.h"
#include "vmelilo.h"

#include "bvmeloader.data"

/*--------------------------------------------------------------------------*/

void
bvme_init_boot_block
(	void		*buf
)
{	BOOTBLOCK	*bb = (BOOTBLOCK *) buf;
	u_long		maxfrags;
	u_long		n1;
	u_long		n2;

	/* set boot block id */
	strcpy(bb->boot_id, BOOT_BLOCK_ID);

	/* clear out user code or file map */
	memset(bb->boot_data, 0, sizeof(bb->boot_data));

	/* clear boot message (now printed by V1.1.0 loader) */
	memset(bb->boot_message, 0, sizeof(bb->boot_message));

	/* load and run boot loader at address 0x10000 */
	bb->boot_load_address = 0x10000;
	bb->boot_run_offset   = 0;

	/* get maximum number of fragments allowed in boot block */
	maxfrags = sizeof(bb->boot_data) / sizeof(FILEMAP);

	if (MAP_NUMFRAGS(loader_map) > maxfrags)
	{
		n1 = MAP_NUMFRAGS(loader_map) - maxfrags;

		die("Boot loader file is too fragmented for boot block "
			"(%ld fragment%s too many)\n",
					n1, (n1 == 1) ? "" : "s");
	}

	if (f_verbose)
	{
		n1 = MAP_NUMFRAGS(loader_map);
		n2 = maxfrags - MAP_NUMFRAGS(loader_map);

		printf(	"Boot loader file has %ld fragment%s, "
				"enough space for %lu more\n",
				  	n1, (n1 == 1) ? "" : "s", n2);
	}

	/* put file map of loader into boot block */
	memcpy(bb->boot_data, MAP_FIRSTFRAG(loader_map),
					MAP_NUMFRAGS(loader_map) * sizeof(FILEMAP));

	/* say boot block contains a file map */
	bb->boot_start_block = 0;

	/* specify the file size in blocks */
	bb->boot_block_count  =
		(MAP_FILESIZE(loader_map) + 511) / 512;
}

/*--------------------------------------------------------------------------*/

int
bvme_valid_boot_block
(	void		*buf
)
{	BOOTBLOCK	*bb = (BOOTBLOCK *) buf;
	
	return (strcmp(bb->boot_id, BOOT_BLOCK_ID) == 0);
}

/*--------------------------------------------------------------------------*/

LILO_DRIVER bvme_lilo_driver = {
	"BVMBug",
	bvmeloader_code,
	sizeof(bvmeloader_code),
	sizeof(BOOTBLOCK),
	bvme_init_boot_block,
	bvme_valid_boot_block
};

/*-----------------------------< end of file >------------------------------*/
