#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include "elfload.h"

enum { num_load = 10 };
static struct {
  char *data;
  Elf32_Addr base;
  Elf32_Word length, fsz;
  Elf32_Off off;
} loaded[num_load];
static int numloaded = 0;
static int dirty = 0;

extern const char *curfile;

int
read_check_ehdr(int fd, Elf32_Ehdr *ehdr)
{
  ssize_t numread;

  if (lseek(fd, 0, SEEK_SET) == -1)
    {
      fprintf (stderr, "%s: positioning for header: %m\n", curfile);
      return -1;
    }
  
  numread = read(fd, ehdr, sizeof(Elf32_Ehdr));
  if (numread == -1)
    {
      fprintf (stderr, "%s: header read failed: %m\n", curfile);
      return -1;
    }
  else if (numread < (ssize_t)sizeof(Elf32_Ehdr))
    return 2;

  if (*(unsigned *)ehdr->e_ident != *(const unsigned *)ELFMAG
      || ehdr->e_ident[EI_CLASS] != ELFCLASS32
      || ehdr->e_ident[EI_DATA] != ELFDATA2MSB
      || ehdr->e_ident[EI_VERSION] != EV_CURRENT
      || ehdr->e_type != ET_EXEC && ehdr->e_type != ET_DYN
      || ehdr->e_machine != EM_PPC)
      return -2;

  if (ehdr->e_phentsize != sizeof(Elf32_Phdr))
    {
      fprintf(stderr, "%s: Section size was read as %d, not %d!\n",
	      curfile, ehdr->e_phentsize, sizeof(Elf32_Phdr));
      return -1;
    }
  return 0;
}

int
read_segment_list(int fd, const Elf32_Ehdr *ehdr, Elf32_Off *dyn_offset_p)
{
  Elf32_Off dyn_offset;
  Elf32_Word i;

  free_segments();

  if (lseek(fd, ehdr->e_phoff, SEEK_SET) == -1)
    {
      fprintf (stderr, "%s: positioning for sections: %m\n", curfile);
      return -1;
    }
  
  dyn_offset = 0;
  for (i = 0; i < ehdr->e_phnum; i++)
    {
      Elf32_Phdr phdr;
      if (read(fd, &phdr, sizeof(phdr)) != sizeof(phdr))
	{
	  fprintf (stderr, "%s: reading section header %d: %m\n",
		   curfile, i);
	  return -1;
	}
      if (phdr.p_type == PT_DYNAMIC)
	if (dyn_offset == 0)
	  dyn_offset = phdr.p_offset;
	else
	  {
	    fprintf(stderr, "%s: Seems to have multiple dynamic sections.\n",
		    curfile);
	    return -1;
	  }
      else if (phdr.p_type == PT_LOAD)
	{
	  if (numloaded == num_load)
	    {
	      fprintf(stderr, "%s: Too many loadable sections.\n",
		      curfile);
	      return -1;
	    }
	  loaded[numloaded].base = phdr.p_vaddr;
	  loaded[numloaded].off = phdr.p_offset;
	  loaded[numloaded].length = phdr.p_memsz;
	  loaded[numloaded].fsz = phdr.p_filesz;
	  loaded[numloaded].data = 0;
	  numloaded++;
	}
    }
  if (dyn_offset_p)
    *dyn_offset_p = dyn_offset;
  return dyn_offset == 0;
}

int
load_segments(int fd)
{
  int i;
  for (i = 0; i < numloaded; i++)
    {
      loaded[i].data = malloc(loaded[i].length);
      memset(loaded[i].data, 0, loaded[i].length);

      if (lseek(fd, loaded[i].off, SEEK_SET) == -1)
	{
	  fprintf (stderr, "%s: seeking for segment %d: %m\n", curfile, i);
	  return -1;
	}
      if (read(fd, loaded[i].data, loaded[i].fsz) != (ssize_t)loaded[i].fsz)
	{
	  fprintf (stderr, "%s: reading segment %d: %m\n", curfile, i);
	  return -1;
	}
    }
  return 0;
}

void
changed_segment(void)
{ dirty = 1; }

int
write_segments(int fd)
{
  int i;

  if (!dirty)
    return 0;

  for (i = 0; i < numloaded; i++)
    {
      if (lseek(fd, loaded[i].off, SEEK_SET) == -1)
	{
	  fprintf (stderr, "%s: seeking for segment %d: %m\n", curfile, i);
	  return -1;
	}
      if (write(fd, loaded[i].data, loaded[i].fsz) != (ssize_t)loaded[i].fsz)
	{
	  fprintf (stderr, "%s: writing segment %d: %m\n", curfile, i);
	  return -1;
	}
    }
  dirty = 0;
  return 0;
}

char *
find_location(Elf32_Addr where, Elf32_Word length)
{
  int t;
  char *result = 0;

  for (t = 0; t < numloaded; t++)
    if (loaded[t].base <= where
	&& loaded[t].base + loaded[t].length >= where + length)
      result = loaded[t].data + (where - loaded[t].base);

  return result;
}

void
free_segments(void)
{
  while (numloaded > 0)
      if (loaded[--numloaded].data)
	free(loaded[numloaded].data);
  dirty = 0;
}

