/*
 * aboot/disk.c
 *
 * This file is part of aboot, the SRM bootloader for Linux/Alpha
 * Copyright (C) 1996 Linus Torvalds, David Mosberger, and Michael Schwingen.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <config.h>
#include <aboot.h>
#include <bootfs.h>
#include <cons.h>
#include <disklabel.h>
#include <ksize.h>
#include <utils.h>

#include <linux/string.h>

#include <asm/console.h>
#include <asm/system.h>

extern struct bootfs ext2fs;
extern struct bootfs iso;
extern struct bootfs ufs;
extern struct bootfs dummyfs;

struct disklabel * label;
char		boot_file[256];
int		boot_part = -1;	/* boot partition */

static struct bootfs *bootfs[] = {
	&ext2fs,
	&iso,
	&ufs
};

/*
 * Attempt a raw boot (header-less kernel follows right after aboot).
 */
int
load_raw (long dev)
{
	extern char _end;
	long aboot_size = &_end - (char *) BOOT_ADDR;

	printf("aboot: loading raw kernel...\n");

	return cons_read(dev, (void *) start_addr, KERNEL_SIZE,
			 (aboot_size + SECT_SIZE - 1) / SECT_SIZE
			 + BOOT_SECTOR);
}


int
load_aout (int fd)
{
	long nread, nblocks;
	unsigned char *buf;

	buf = malloc(bfs->blocksize);

	/* read a.out header: */

	nread = (*bfs->bread)(fd, 0, 1, buf);
	if (nread != bfs->blocksize) {
		printf("aboot: read returned %ld instead of %ld bytes\n",
		       nread, sizeof(buf));
		return -1;
	}
#ifdef DEBUG
	if (1) {
		int i,j,c;

		for(i = 0; i < 16; i++) {
			for (j = 0; j < 16; j++)
				printf("%02X ", buf[j+16*i]);
			for(j = 0; j < 16; j++) {
				c = buf[j+16*i];
				printf("%c", (c >= ' ') ? c : ' ');
			}
			printf("\n");
		}
		wait_key();
	}
#endif
	if (first_block(buf, bfs->blocksize) < 0) {
		return -1;
	}

	/* read rest of kernel in one big swoop: */
	nblocks = (bytes_to_copy + bfs->blocksize - 1) / bfs->blocksize;
	nread = (*bfs->bread)(fd, 1, nblocks, dest_addr);
	if (nread != nblocks * bfs->blocksize) {
		printf("aboot: read returned %ld instead of %ld bytes\n",
		       nread, nblocks * bfs->blocksize);
		return -1;
	}
	return 0;
}


static long
read_kernel (const char *filename)
{
	volatile int attempt, method;
	long len;
	int fd;
	static struct {
		const char *name;
		int (*func)(int fd);
	} read_method[]= {
		{"plain",	load_aout},
		{"compressed",	uncompress_kernel}
	};
	long res;
#	define NUM_METHODS ((int)(sizeof(read_method)/sizeof(read_method[0])))

#ifdef DEBUG
	printf("read_kernel(%s)\n", filename); wait_key();
#endif

	method = 0;
	len = strlen(filename);
	if (len > 3 && filename[len - 3] == '.'
	    && filename[len - 2] == 'g' && filename[len - 1] == 'z')
	{
		/* if filename ends in .gz we don't try plain method: */
		method = 1;
	}

	for (attempt = 0; attempt < NUM_METHODS; ++attempt) {
		fd = (*bfs->open)(filename);
		if (fd < 0) {
			printf("%s: file not found\n", filename);
			return -1;
		}
		printf("aboot: loading %s %s...\n",
		       read_method[method].name, filename);

		if (!_setjmp(jump_buffer)) {
			res = (*read_method[method].func)(fd);
			(*bfs->close)(fd);
			if (res >= 0) {
				return 0;
			}
		}
		method = (method + 1) % NUM_METHODS;
	}
	return -1;
}


static void
get_disklabel (long dev)
{
	static char lsect[512];
	long nread;

#ifdef DEBUG
	printf("load_label(dev=%lx)\n", dev); wait_key();
#endif
	nread = cons_read(dev, &lsect, LABELOFFSET + sizeof(*label),
			  LABELSECTOR);
	if (nread != LABELOFFSET + sizeof(*label)) {
		printf("aboot: read of disklabel sector failed (nread=%ld)\n",
		       nread);
		return;
	}
	label = (struct disklabel*) &lsect[LABELOFFSET];
	if (label->d_magic  == DISKLABELMAGIC &&
	    label->d_magic2 == DISKLABELMAGIC)
	{
		printf("aboot: valid disklabel found: %d partitions.\n",
		       label->d_npartitions);
	} else {
		printf("aboot: no disklabel found.\n");
		label = 0;
	}
}


struct bootfs *
mount_fs (long dev, int partition)
{
	struct d_partition * part;
	struct bootfs * fs = 0;
	int i;

#ifdef DEBUG
	printf("mount_fs(%lx, %d)\n", dev, partition); wait_key();
#endif
	if (partition == 0) {
		fs = &dummyfs;
		if ((*fs->mount)(dev, 0, 0) < 0) {
			printf("aboot: disk mount failed\n");
			return 0;
		}
	} else if (!label) {
		for (i = 0; i < (int)(sizeof(bootfs)/sizeof(bootfs[0])); ++i) {
			if ((*bootfs[i]->mount)(dev, 0, 1) >= 0) {
				fs = bootfs[i];
				break;
			}
		}
		if (!fs) {
			printf("aboot: unknown filesystem type\n");
			return 0;
		}
	} else {
		if ((unsigned) (partition - 1) >= label->d_npartitions) {
			printf("aboot: invalid partition %u\n", partition);
			return 0;
		}
		part = &label->d_partitions[partition - 1];
		for (i = 0; bootfs[i]->fs_type != part->p_fstype; ++i) {
			if (i + 1
			    >= (int) (sizeof(bootfs)/sizeof(bootfs[0])))
			{
				printf("aboot: don't know how to mount "
				       "partition %d (filesystem type %d)\n",
				       partition, part->p_fstype);
				return 0;
			}
		}
		fs = bootfs[i];
		if ((*fs->mount)(dev, part->p_offset * label->d_secsize, 0)
		    < 0)
		{
			printf("aboot: mount of partition %d failed\n",
			       partition);
			return 0;
		}
	}
	return fs;
}


void
print_config_file (struct bootfs *fs)
{
	int fd, nread, blkno = 0;
	char *buf;

	fd = (*fs->open)(CONFIG_FILE);
	if (fd < 0) {
		printf("%s: file not found\n", CONFIG_FILE);
		return;
	}
	buf = malloc(fs->blocksize + 1);
	if (!buf) {
		printf("aboot: malloc failed!\n");
		return;
	}
	do {
		nread = (*fs->bread)(fd, blkno++, 1, buf);
		buf[nread] = '\0';
		printf("%s", buf);
	} while (nread > 0);
	(*fs->close)(fd);
}


int
get_default_args (struct bootfs *fs, char *str, int num)
{
	int fd, nread, state, line, blkno = 0;
	char *buf, *d, *p;

#ifdef DEBUG
	printf("get_default_args(%s,%d)\n", str, num); wait_key();
#endif
	*str = '\0';
	fd = (*fs->open)(CONFIG_FILE);
	if (fd < 0) {
		printf("%s: file not found\n", CONFIG_FILE);
		return -1;
	}
	buf = malloc(fs->blocksize);
	if (!buf) {
		printf("aboot: malloc failed!\n");
		return -1;
	}
	d = str;
	line = 1;
	state = 2;
	do {
		nread = (*fs->bread)(fd, blkno++, 1, buf);
		p = buf;
		while (p < buf + nread && *p && state != 5) {
			switch (state) {
			      case 0: /* ignore rest of line */
			      case 1: /* in comment */
				if (*p == '\n') state = 2;
				break;

			      case 2: /* after end of line */
				line++;
				if (*p == num) {
					state = 3;	/* found it... */
					break;
				}
				if (*p == '#') {
					state = 1;	/* comment */
					break;
				}
				if (*p == '-') {
					state = 5;	/* end-of-file mark */
					break;
				}
				state = 0;	/* ignore rest of line */
				break;

			      case 3: /* after matched number */
				if (*p == ':') {
					state = 4;	/* copy string */
				} else {
					state = 2;	/* ignore rest */
					printf("aboot: syntax error in line "
					       "%d: `:' expected\n", line);
				}
				break;

			      case 4: /* copy until EOL */
				if (*p == '\n') {
					*d = 0;
					state=5;
				} else {
					*d++ = *p;
				}
				break;

			      default:
			}
			p++;
		}
	} while (nread > 0 && state != 5);
	(*fs->close)(fd);
#ifdef DEBUG
	printf("get_default done\n"); wait_key();
#endif

	if (state != 5) {
		printf("aboot: could not find default config `%c'\n", num);
		return -1;
	}
	return 0;
}


static void
get_boot_device (char buf[], size_t bufsize)
{
	int i;

	if (cons_getenv(ENV_BOOTED_DEV, buf, bufsize) < 0) {
		printf("aboot: Can't get BOOTED_DEV environment variable!\n");
		buf[0] = '\0';
	}
	/* make case blind: */
	for (i = 0; buf[i]; ++i) {
		if (buf[i] >= 'A' && buf[i] <= 'Z') {
			buf[i] += 'a' - 'A';
		}
	}
	printf("aboot: booted_dev=`%s', ", buf);
	if (strncmp(buf, "dva", 3) == 0) {
		buf[0] = 'f'; buf[1] = 'd';
		if (buf[4] == '0') {
		    ++buf[4];
		}
		buf[2] = buf[4] - 1;
		buf[3] = '\0';
	} else if (strncmp(buf, "scsi", 4) == 0 && boot_part > 0) {
		buf[0] = 's'; buf[1] = 'd'; buf[2] = 'a';
		buf[3] = '0' + boot_part;
		buf[4] = '\0';
	} else {
		strcpy(buf, "scd0");
	}
	printf("guessing boot_device=`%s'\n", buf);
}


static void
get_options (long dev)
{
	char buf[256], *p;
	struct bootfs *fs = 0;
	int i, part, opt = 0, need_bootfile = 1, need_bootdevice = 1;
	size_t len, arglen;

#ifdef DEBUG
	printf("get_options(%lx)\n",dev); wait_key();
#endif
	if (kernel_args[0] >= '1' && kernel_args[0] <= '9'
	    && kernel_args[1] == ':' && kernel_args[2] && !kernel_args[3])
	{
		config_file_partition = kernel_args[0] - '0';
		opt = kernel_args[2];
		if (!opt) {
			opt = 'i';
		}
	} else if (kernel_args[0] && kernel_args[1] == '\0') {
		opt = kernel_args[0];
	}
	if (opt) {
		part = label ? config_file_partition : 1;

		kernel_args[0] = '\0';

		do {
			if (opt >= '0' && opt <= '9') {
				if (!fs) {
					fs = mount_fs(dev, part);
					if (!fs) {
						opt = 'i';
						continue;
					}
				}
				if (get_default_args(fs, buf, opt) >= 0)
					break;
				opt = 'i';	/* failed---ask user */
			} else if (opt == 'i' || opt == 'h' || opt == '?') {
				if (opt == 'h' || opt == '?') {
					if (!fs) {
						fs = mount_fs(dev, part);
						if (!fs) {
							opt = 'i';
							continue;
						}
					}
					print_config_file(fs);
				}
				printf("Enter kernel name and arguments:\n"
				       "aboot> ");
				getline(buf, sizeof(buf));
				printf("\n");
				opt = 0;
				if (buf[0] && buf[1] == '\0')
					opt = buf[0];
				else if (buf[0] == '\0')
					opt = 'i';
			}
		} while ((opt >= '0' && opt <= '9') ||
			 opt == 'i' || opt == 'h' || opt == '?');

		p = strchr(buf, ' ');
		if (p) {
			strcpy(kernel_args, p + 1);
			*p = '\0';
		}
		strcpy(boot_file, buf);
#ifdef DEBUG
		printf("\nfile=`%s',args=`%s'\n", boot_file, kernel_args);
#endif
	}
#ifdef DEBUG
	printf("\nfile=`%s',args=`%s'\n", boot_file, kernel_args);
#endif

	/* parse off partition number if any: */
	if (boot_file[0] >= '0' && boot_file[0] <= '9' && boot_file[1] == '/')
	{
		boot_part = boot_file[0] - '0';
		strcpy(boot_file, boot_file + 2);
	}

	/*
	 * Now setup the bootdevice= and bootfile= options if they
	 * aren't there already
	 */
	for (i = 0; kernel_args[i]; ++i) {
	    if (strncmp(kernel_args + i, "bootdevice=", 11) == 0) {
		need_bootdevice = 0;
	    } else if (strncmp(kernel_args + i, "bootfile=", 9) == 0) {
		need_bootfile = 0;
	    }
	}
	len = strlen(kernel_args);
	if (need_bootdevice) {
	    get_boot_device(buf, sizeof(buf));
	    arglen = strlen(buf);

	    if (len + 1 + 11 + arglen > sizeof(kernel_args)) {
		printf("aboot: option string too long---truncating\n");
		return;
	    }

	    if (len > 0) {
		kernel_args[len++] = ' ';
	    }
	    memcpy(kernel_args + len, "bootdevice=", 11); len += 11;
	    memcpy(kernel_args + len, buf, arglen); len += arglen;
	}
	if (need_bootfile) {
	    arglen = strlen(boot_file);

	    if (len + 1 + 9 + arglen > sizeof(kernel_args)) {
		printf("aboot: option string too long---truncating\n");
		return;
	    }

	    if (len > 0) {
		kernel_args[len++] = ' ';
	    }
	    memcpy(kernel_args + len, "bootfile=", 9); len += 9;
	    memcpy(kernel_args + len, boot_file, arglen); len += arglen;
	}
	kernel_args[len] = '\0';
}


static long
load (long dev)
{
	char *fname;
	int part = boot_part;

#ifdef DEBUG
	printf("load(%lx)\n", dev); wait_key();
#endif
	fname = boot_file;
	if (fname[0] == '-' && fname[1] == '\0') {
		/* a single "-" implies raw boot: */
		return load_raw(dev);
	}
	if (part < 0) {
	    part = 1;	/* default to partition 1 */
	}
	bfs = mount_fs(dev, part);
	if (!bfs) {
		printf("aboot: mount of partition %d failed\n", part);
		return -1;
	}
	if (read_kernel(fname) < 0) {
		return -1;
	}
	/* clear bss: */
	memset((char*)bss_start, 0, bss_size);
	return 0;
}


long
load_kernel (void)
{
	char envval[256];
	long result;
	long dev;

	if (cons_getenv(ENV_BOOTED_DEV, envval, sizeof(envval)) < 0) {
		printf("aboot: Can't get BOOTED_DEV environment variable!\n");
		return -1;
	}

	dev = cons_open(envval);
	if (dev < 0) {
		printf("aboot: unable to open boot device `%s': %lx\n",
		       envval, dev);
		return -1;
	}
	dev &= 0xffffffff;
	get_disklabel(dev);

	/* get boot command line: */
	result = cons_getenv(ENV_BOOTED_FILE, boot_file, sizeof(boot_file));
	if (result < 0) {
		printf("aboot: warning: can't get ENV_BOOTED_FILE "
		       "(result=%lx)!\n", result);
		strcpy(boot_file, "vmlinux.gz");
	}
	result = cons_getenv(ENV_BOOTED_OSFLAGS,
			     kernel_args, sizeof(kernel_args));
	if  (result < 0) {
		printf("aboot: warning: can't get ENV_BOOTED_OSFLAGS "
		       "(result=%lx)!\n", result);
		strcpy(kernel_args, "i");
	}
#ifdef DEBUG
	printf("file:%s ,osflags:%s\n", boot_file, kernel_args); wait_key();
#endif

	while (1) {
		get_options(dev);
		result = load(dev);
		if (result != -1)
			break;
		/* load failed---query user interactively */
		strcpy(kernel_args, "i");
	}
#ifdef DEBUG
	printf("load done\n"); wait_key();
#endif	
	cons_close(dev);
	return result;
}
