/*
ICQ.H
header file containing all the main procedures to interface with the ICQ server at mirabilis
*/

#ifndef ICQ_H
#define ICQ_H

#include <vector.h>
#include <qobject.h>
#include <qtimer.h>

#include "icq-defines.h"
#include "socket.h"
#include "packet.h"
#include "file.h"
#include "outputwin.h"
#include "user.h"
#include "remoteserver.h"
#include "icqevent.h"
#include "translate.h"

//CONSTANT DECLARATIONS
#define COLOR_ERROR   1 // COLOR_RED
#define COLOR_SEND    1 // COLOR_GREEN
#define COLOR_RECEIVE 2 // COLOR_BLUE
#define COLOR_DATA    3 // COLOR_YELLOW


//=====ICQ=========================================================================================

class ICQ : public QObject
{
   Q_OBJECT
public:
   ICQ(void);
   ~ICQ(void);
   bool logon(unsigned short logonStatus);
   void logoff(bool reconnect);
   
   ICQEvent *sendMessage(ICQUser *u, const char *m, bool online, unsigned long id = 0);
   ICQEvent *sendReadAwayMsg(ICQUser *, bool, unsigned long id = 0);
   ICQEvent *sendUrl(ICQUser *u, const char *url, const char *description, bool online, unsigned long id = 0);
   ICQEvent *sendChat(ICQUser *u, const char *reason, bool online, unsigned long id = 0);
   void chatRefuse(ICQUser *u, char *reason, unsigned long theSequence);
   void chatAccept(ICQUser *u, unsigned short thePort, unsigned long theSequence);
   void chatCancel(ICQUser *u, unsigned long seq, bool online, unsigned long id = 0);

   LicqTranslator *licqTrans;

   bool nextServer(void);
   void cancelEvent(ICQEvent *&);
   bool setStatus(unsigned short newStatus);
   bool getUserInfo(ICQUser *user);
   void getStatusInfo(struct UserStatusline &us);
   void authorize(unsigned long uinToAuthorize);
   void requestSystemMsg(void);
   void setOutputWin(OutputWin *w);
   int numUsers(void);
   void setSoundEnabled(unsigned short);
   unsigned short soundEnabled(void);
   void addUser(unsigned long id, char *filename, bool newUser);
   void removeUser(ICQUser *);
   bool knownUser(unsigned long);
   void startSearch(char *, char *, char *, char *, unsigned long);
   void setAwayMessage(char *);
   char *awayMessage(void);
   void saveUsers(void);
   void setSounds(const char *, const char *, const char *, 
                  const char *, const char *, const char *);

   ICQOwner icqOwner;
   vector<class ICQUser *> users;
   vector<bool> tcpPorts;
   ICQRemoteServers icqServers;
   unsigned short tcpServerPort, defaultRemotePort;
   char *soundPlayer, *soundMsg, *soundUrl, *soundChat, *soundFile, *soundNotify;
   
protected:
   // protected variables
   unsigned short numUsersVal;
   char remoteServer[64];
   OutputWin *outputWindow;
   bool autoReconnect, allowUpdateUsers;
   unsigned short soundEnabledVal, tmpSoundEnabled;
   unsigned short desiredStatus;
   TCPSocket *tcpSocket;
   UDPSocket udpServer;
   FILE *errorFile;
   QTimer pingTimer;
   vector <ICQEvent *> icqEvents;
   
   // protected functions
   void addToHistory(ICQUser *, char *, const char *);
   void addTcpHeader(Packet &, ICQUser *, unsigned short, unsigned short, char *, unsigned long = 0, bool = true);
   ICQEvent *sendTcp(Packet &, ICQUser *, const char *, unsigned short, unsigned long id = 0);
   ICQEvent *sendOfflineTcp(ICQUser *user, unsigned short cmd, char *cmdStr, char *m, unsigned long id = 0);
   void sendTcpAck(Packet &, ICQUser *, unsigned long);
   
   void unixToDos(const char *, char *);
   void playSound(char *);
   void createAckPacket(Packet &ack, unsigned short theSequence);
   unsigned short processUdpPacket(Packet &packet);
   bool openServer(void);
   void ackUDP(unsigned short);
   void ackTCP(Packet &, ICQUser *, unsigned short, unsigned long);
   void processSystemMessage(Packet &packet, unsigned long checkUin, unsigned short newCommand, time_t timeSent);
   void processTcpPacket(Packet &packet, int sockfd);
   ICQUser *getUserByUIN(unsigned long u);
   void addUser(unsigned long id, TCPSocket &sock, bool newUser);
   void addNewUser(ICQUser *u);
   void moveToTop(ICQUser *u);
   void moveOffline(ICQUser *u);
   void moveOnline(ICQUser *u);
   ICQEvent *sendICQ(INetSocket &sock, Packet &packet, unsigned short cmd, unsigned long theSequence, unsigned long uin = 0, unsigned short subCmd = 0);
   bool errorOn(const char *protocol, const char *reason);

signals:
   void updatedUsers();
   void updatedStatus();
   void doneOwnerFcn(bool isOk, unsigned short cmd);
   void doneUserFcn(bool isOk, ICQEvent *e);
   void doneUserInfo(bool, unsigned long);
   void eventResult(ICQUser *u, unsigned short cmd, bool accepted, unsigned long port);
   void doneSearch(unsigned short, char);
   void userFound(struct UserStatusline &, unsigned long);

public slots:
   void updateContactList();
   void updateAllUsers();
   
protected slots:
   void ping();
   void doneEvent(bool gotAck, int sockfd, int seq);
   void recvUDP(int);
   void recvNewTCP(int);
   void recvTCP(int sockfd);
};

#endif

