/*
 *  themes.c:		Theme dialog
 *
 *  Written by:		Ullrich Hafner
 *  
 *  Copyright (C) 1999 Ullrich Hafner <hafner@bigfoot.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 */

/*
 *  $Date: 1999/12/03 15:34:28 $
 *  $Author: hafner $
 *  $Revision: 1.33 $
 *  $State: Exp $
 */

#include "config.h"

#include <gtk/gtk.h>
#include <proplist.h>
#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#endif /* HAVE_UNISTD_H */
/* unistd.h defines _POSIX_VERSION on POSIX.1 systems. */
#if defined(HAVE_DIRENT_H) || defined(_POSIX_VERSION)
#   include <dirent.h>
#   define NLENGTH(dirent) (strlen ((dirent)->d_name))
#else
#   define dirent direct
#   define NLENGTH(dirent) ((dirent)->d_namlen)
#   ifdef HAVE_SYS_NDIR_H
#       include <sys/ndir.h>
#   endif /* HAVE_SYS_NDIR_H */
#   ifdef HAVE_SYS_DIR_H
#       include <sys/dir.h>
#   endif /* HAVE_SYS_DIR_H */
#   ifdef HAVE_NDIR_H
#       include <ndir.h>
#   endif /* HAVE_NDIR_H */
#endif /* not (HAVE_DIRENT_H or _POSIX_VERSION) */
#include <ctype.h>
#if HAVE_STDLIB_H
#	include <stdlib.h>
#endif /* not HAVE_STDLIB_H */
#include <stdio.h>
#if HAVE_STRING_H
#	include <string.h>
#else /* not HAVE_STRING_H */
#	include <strings.h>
#endif /* not HAVE_STRING_H */
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>

#include "themes.h"
#include "path.h"
#include "load.h"
#include "misc.h"
#include "dialog.h"
#include "window.h"
#include "previews.h"
#include "rimage.h"
#include "error.h"

/*******************************************************************************

			     global variables
  
*******************************************************************************/

extern proplist_t  *windowmaker;
extern proplist_t  *wmconfig;
extern GtkWidget   *main_window;
extern GtkTooltips *tooltips;

/*******************************************************************************

			     local variables
  
*******************************************************************************/

static GtkWidget  *theme_list          = NULL;
static GtkWidget  *remove_button       = NULL;
static GtkWidget  *preview_button      = NULL;
static proplist_t *theme_keys          = NULL;
static GtkWidget  *workspaceback       = NULL;
static GtkWidget  *titlebarback        = NULL;
static GtkWidget  *iconback            = NULL;
static GtkWidget  *open_button         = NULL;
static GtkWidget  *open_menu_button    = NULL;
static GtkWidget  *set_button          = NULL;
static GtkWidget  *set_menu_button     = NULL;
static GtkWidget  *preview_update_menu = NULL;
static GtkWidget  *rename_theme_menu   = NULL;
static bool_t	   previews_active     = NO;

/*******************************************************************************

			       prototypes
  
*******************************************************************************/

static void
theme_gradient (unsigned row, const char *attribute, GtkWidget *pixmap);
static void
compute_preview (GtkWidget *progress_bar, GtkWidget *progress_label,
		 unsigned n, unsigned nelem,
		 const char *name, unsigned width, unsigned height);
static char *
theme_image_filename (unsigned row, const char *attribute, bool_t combined);
static void
cancel_installation (GtkWidget *widget, gpointer ptr);
static gint
context_menu (GtkCTree *ctree, GdkEventButton *event, gpointer ptr);
static void
set_theme (GtkWidget *button, gpointer ptr);
static void
free_memory (gpointer data, gpointer user_data);
#ifdef HAVE_RENAME
static void
rename_theme_dialog (GtkWidget *button, gpointer ptr);
static void
rename_theme (GtkWidget *widget, gpointer ptr);
#endif /* HAVE_RENAME */
static void
remove_theme_dialog (GtkWidget *button, gpointer ptr);
static void
remove_theme (GtkWidget *widget, gpointer ptr);
static void
remove_file (gpointer data, gpointer user_data);
static void
load_theme (GtkWidget *button, gpointer ptr);
static void
process_dir (DIR *dir, char *dirname);
static GtkWidget *
theme_preview (void);
static void
update_preview (GtkCList *clist, gint row, gint column, GdkEvent *event,
		gpointer data);
static void
update_selection (GtkCList *themelist, gint row, gint column, GdkEvent *event,
		  gpointer data);
static proplist_t
get_color (proplist_t texture);
static void
update_pathlist (GtkCombo *combo, bool_t strip_themes);
static gint
hide_install (GtkWidget *widget, gpointer ptr);
#ifdef GETSTYLE
static void
save_theme_dialog (GtkWidget *widget, gpointer ptr);
static gint
hide_window (GtkWidget *widget, gpointer ptr);
static void
set_themename (GtkWidget *widget, gpointer ptr);
static void
activate_themename (GtkWidget *widget, gpointer ptr);
static void
save_theme (GtkWidget *widget, gpointer ptr);
static bool_t
getstyle_call (const char *themename);
#ifdef SETSTYLE
static void
preview_dialog (GtkWidget *button, gpointer ptr);
static void
leave_preview (GtkWidget *button, gpointer ptr);
#endif /* SETSTYLE */
#endif /* GETSTYLE */
static void
open_dirbrowser (GtkWidget *button, gpointer data);
static void
insert_dir (GtkWidget *button, gpointer data);
static void
install_theme_dialog (GtkWidget *widget, gpointer ptr);
static void
install_themes (GtkWidget *widget, gpointer ptr);
static void
open_theme_browser (GtkWidget *button, gpointer data);
static void
append_text (gpointer data, gpointer user_data);
static gint
insert_theme_files (GtkWidget *widget, gpointer ptr);
static void
insert_dnd_elements (GtkCList *clist, const char *start);
static void
init_dnd (GtkWidget *widget);
static void  
drag_data_received (GtkWidget *widget, GdkDragContext *context, gint x, gint y,
		    GtkSelectionData *data, guint info, guint time);

/*******************************************************************************

				public code
  
*******************************************************************************/

GtkWidget *
theme_dialog (GtkWidget *box)
/*
 *  Generate theme dialog window: scrolled theme list, theme path list,
 *  and action button box
 */
{
   GtkWidget *button;
   GtkWidget *bbox = gtk_vbutton_box_new ();
   GtkWidget *hbox = gtk_hbox_new (FALSE, 0);
   
   {
      char *title [3];

      title [0] = _("Theme");
      title [1] = _("Path");
      title [2] = "Suffix";
      theme_list = gtk_clist_new_with_titles (3, title);
   }
   
   gtk_clist_set_auto_sort (GTK_CLIST (theme_list), YES);
   gtk_clist_set_column_auto_resize (GTK_CLIST (theme_list), 0, YES);
   gtk_clist_set_column_visibility (GTK_CLIST (theme_list), 2, NO);
   gtk_clist_column_titles_passive (GTK_CLIST (theme_list));
   gtk_clist_set_selection_mode (GTK_CLIST (theme_list), GTK_SELECTION_EXTENDED);
   gtk_signal_connect (GTK_OBJECT (theme_list), "select_row",
		       GTK_SIGNAL_FUNC (update_preview), NULL);

   gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_SPREAD);

   open_button = button = gtk_button_new_with_label (_("Open"));
   gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, TRUE, 5);
   gtk_tooltips_set_tip (tooltips, button, _("Open selected theme."), NULL);
   gtk_signal_connect (GTK_OBJECT (button), "clicked",
		       GTK_SIGNAL_FUNC (load_theme), NULL);
   gtk_widget_set_sensitive (button, FALSE);
   
#ifdef GETSTYLE
   button = gtk_button_new_with_label (_("Save..."));
   gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, TRUE, 5);
   gtk_tooltips_set_tip (tooltips, button,
			 _("Save current wmakerconf settings as a theme pack."),
			 NULL);
   gtk_signal_connect (GTK_OBJECT (button), "clicked",
		       GTK_SIGNAL_FUNC (save_theme_dialog), NULL);

#ifdef SETSTYLE

   set_button = button = gtk_button_new_with_label (_("Set"));
   gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, TRUE, 5);
   gtk_tooltips_set_tip (tooltips, button,
			 _("Open selected theme, update wmakerconf, "
			   "and save wmakerconf settings."), NULL);
   gtk_signal_connect (GTK_OBJECT (button), "clicked",
		       GTK_SIGNAL_FUNC (set_theme), NULL);
   gtk_widget_set_sensitive (button, FALSE);

   preview_button = button = gtk_button_new_with_label (_("Preview"));
   gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, TRUE, 5);
   gtk_tooltips_set_tip (tooltips, button, _("Preview the selected theme."),
			 NULL);
   gtk_signal_connect (GTK_OBJECT (button), "clicked",
		       GTK_SIGNAL_FUNC (preview_dialog), NULL);
   gtk_widget_set_sensitive (button, FALSE);

#endif /* SETSTYLE */
   
#endif /* GETSTYLE */

   remove_button = button = gtk_button_new_with_label (_("Remove..."));
   gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, TRUE, 5);
   gtk_tooltips_set_tip (tooltips, button,
			 _("Remove themes with associated files "
			   "and directories."), NULL);
   gtk_signal_connect (GTK_OBJECT (button), "clicked",
		       GTK_SIGNAL_FUNC (remove_theme_dialog), NULL);
   gtk_widget_set_sensitive (button, FALSE);

   button = gtk_button_new_with_label (_("Install..."));
   gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, TRUE, 5);
   gtk_tooltips_set_tip (tooltips, button,
			 _("Install themes (local or via WWW request)."),
			 NULL);
   gtk_signal_connect (GTK_OBJECT (button), "clicked",
		       GTK_SIGNAL_FUNC (install_theme_dialog), NULL);

   gtk_box_pack_start (GTK_BOX (box), hbox, TRUE, TRUE, 5);

   {
      GtkWidget *scrolled = gtk_scrolled_window_new (NULL, NULL);

      gtk_scrolled_window_set_placement (GTK_SCROLLED_WINDOW (scrolled),
					 GTK_CORNER_TOP_LEFT);
      gtk_container_set_border_width (GTK_CONTAINER (scrolled), 5);
      gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
				      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
      gtk_container_add (GTK_CONTAINER (scrolled), theme_list);
      gtk_box_pack_start (GTK_BOX (hbox), scrolled, TRUE, TRUE,  0);
   }
   
   gtk_box_pack_start (GTK_BOX (hbox), bbox, FALSE, TRUE, 5);
   
   gtk_widget_show_all (box);

   themes_list ();

   gtk_signal_connect (GTK_OBJECT (theme_list), "button_press_event",
		       GTK_SIGNAL_FUNC (context_menu), NULL);
   
   {
      GtkWidget *main_hbox = gtk_hbox_new (FALSE, 0);
      GtkWidget *main_vbox = gtk_vbox_new (FALSE, 0);

      gtk_box_pack_start (GTK_BOX (box), main_hbox, FALSE, TRUE, 0);
      gtk_box_pack_start (GTK_BOX (main_hbox), main_vbox, TRUE, TRUE, 0);
      gtk_box_pack_start (GTK_BOX (main_hbox), theme_preview (), FALSE, TRUE, 0);
      gtk_widget_show_all (main_hbox);
   
      gtk_clist_select_row (GTK_CLIST (theme_list), 0, 0);
      init_dnd (theme_list);
      
      return main_vbox;
   }
}

void
themes_list (void)
/*
 *  Recursively fill clist widget with list of available themes
 */
{
   unsigned	n;
   proplist_t	*pltheme = PLMakeString ("ThemePath");
   proplist_t	*dirlist = PLGetDictionaryEntry (windowmaker, pltheme);
   
   PLRelease (pltheme);

   gtk_clist_freeze (GTK_CLIST (theme_list));
   gtk_clist_clear (GTK_CLIST (theme_list));
   
   for (n = 0; n < PLGetNumberOfElements (dirlist); n++)
   {
      proplist_t *pldir   = PLGetArrayElement (dirlist, n);
      char	 *dirname = expand_tilde (PLGetString (pldir));
      DIR	 *dir     = opendir (dirname);
      
      if (dir)
	 process_dir (dir, dirname);

      Free (dirname);
   }
   gtk_widget_set_sensitive (remove_button, GTK_CLIST (theme_list)->rows > 0);
   gtk_widget_set_sensitive (preview_button, GTK_CLIST (theme_list)->rows > 0);
   gtk_widget_set_sensitive (set_button, GTK_CLIST (theme_list)->rows > 0);
   gtk_widget_set_sensitive (open_button, GTK_CLIST (theme_list)->rows > 0);

   gtk_clist_thaw (GTK_CLIST (theme_list));
}

#ifdef GETSTYLE
void
get_theme_attributes (void)
/*
 *  Generate list of attributes that define a Window Maker theme
 *
 *  Side effects:
 *	fills global property array of attributes: 'theme_keys'	
 */
{
   char		*tmp_name      = get_temporary_file_name ();
   char		*tmpthemename  = tmp_name ? tmp_name : "/tmp/theme";
   proplist_t	*theme;

   if (!getstyle_call (tmpthemename))
   {
      dialog_popup (DIALOG_ERROR, NULL, NULL,
		    _("Can't write temporary theme file.\n"
		      "Please check `stderr' for more details."));
      return;
   }
   theme = read_proplist (tmpthemename);
   if (!theme)
   {
      dialog_popup (DIALOG_ERROR, NULL, NULL, 
		    _("Can't open temporary theme file.\n"
		      "Please check `stderr' for more details."));
      return;
   }
   theme_keys = PLGetAllDictionaryKeys (theme);
   PLRelease (theme);
}

void
cleanup_themes (void)
{
   PLRelease (theme_keys);
}

#endif /* GETSTYLE */

bool_t
load_attributes (const char *filename, const char *path)
/*
 *  Load attributes from a file 'filename' in directory 'path'.
 *
 *  Return value:
 *	true on success
 */
{
   proplist_t *theme = read_proplist (filename);

   if (theme)
   {
      unsigned	 n;
      proplist_t *keys;
      proplist_t *plupdate    = PLMakeString ("Update");
      proplist_t *plupdateptr = PLMakeString ("UpdatePtr");

      /*
       * Modify style file that doesn't contain resizebarback
       */
      {
	 proplist_t *rsbpl     = PLMakeString ("ResizebarBack");
	 proplist_t *resizebar = PLGetDictionaryEntry (theme, rsbpl);

	 if (!resizebar)
	 {
	    proplist_t *utitle = PLMakeString ("UTitleBack");
	    proplist_t *tmp    = PLGetDictionaryEntry (theme, utitle);

	    PLRelease (utitle);
	    if (tmp)
	    {
	       proplist_t *color = get_color (tmp);
	       if (color)
	       {
		  proplist_t *array;

		  array = PLMakeArrayFromElements (PLMakeString ("solid"),
						   PLRetain (color), NULL);
		  PLInsertDictionaryEntry (theme, rsbpl, array);
		  PLRelease (array);
	       }
	    }
	 }
	 PLRelease (rsbpl);
      }
      /*
       * Modify style file that doesn't contain iconbarback
       */
      {
	 proplist_t *icontitle = PLMakeString ("IconTitleColor");
	 proplist_t *iconback  = PLMakeString ("IconTitleBack");
	 proplist_t *iconpl    = PLGetDictionaryEntry (theme, icontitle);

	 if (!iconpl)
	    iconpl = PLGetDictionaryEntry (theme, iconback);
	 if (!iconpl)
	 {
	    proplist_t *ftitle = PLMakeString ("FTitleColor");
	    proplist_t *fback  = PLMakeString ("FTitleBack");
	    proplist_t *tmp    = PLGetDictionaryEntry (theme, ftitle);

	    PLRelease (ftitle);
	    if (tmp)
	       PLInsertDictionaryEntry (theme, icontitle, tmp);
	    tmp = PLGetDictionaryEntry (theme, fback);
	    PLRelease (fback);
	    if (tmp)
	    {
	       proplist_t *color = get_color (tmp);
	       if (color)
		  PLInsertDictionaryEntry (theme, iconback, color);
	    }
	 }
	 PLRelease (icontitle);
	 PLRelease (iconback);
      }
      keys = PLGetAllDictionaryKeys (theme)      ;
      for (n = 0; n < PLGetNumberOfElements (keys); n++)
      {
	 proplist_t *key    = PLGetArrayElement (keys, n);
	 proplist_t *value  = PLGetDictionaryEntry (theme, key);
	 proplist_t *keydef = PLGetDictionaryEntry (wmconfig, key);

	 if (keydef)
	 {
	    proplist_t *keydata;
	    proplist_t *keyupdate;

	    keydata   = PLGetDictionaryEntry (keydef, plupdateptr);
	    keyupdate = PLGetDictionaryEntry (keydef, plupdate);
		  
	    if (keyupdate && keydata)
	    {
	       gpointer   data;
	       update_fct update;  

	       data   = * (gpointer *) PLGetDataBytes (keydata);
	       update = * (update_fct *) PLGetDataBytes (keyupdate);

	       update (key, data, value, path);
	    }
	    else
	       warning (_("Attribute %s not yet supported."), PLGetString (key));
	 }
	 else
	    warning (_("Attribute %s not yet supported."), PLGetString (key));
      }
      PLRelease (plupdateptr);
      PLRelease (plupdate);
      PLRelease (keys);
      PLRelease (theme);
      return TRUE;
   }
   else
      return FALSE;
}

void
connect_update_function (proplist_t *key, gpointer ptr, update_fct update)
/*
 *  Connect an 'update' function with the current 'key' and
 *  generic datapointer 'ptr'.
 *
 *  Side effects:
 *	wmconfig dictionary Update and UpdatePtr are set
 */
{
   proplist_t *plupdate    = PLMakeString ("Update");
   proplist_t *plupdateptr = PLMakeString ("UpdatePtr");
   proplist_t *keydef      = PLGetDictionaryEntry (wmconfig, key);
   proplist_t *data;
   
   data = PLMakeData ((unsigned char *) &ptr, sizeof (gpointer));
   PLInsertDictionaryEntry (keydef, plupdateptr, data);
   PLRelease (data);

   data = PLMakeData ((unsigned char *) &update, sizeof (update_fct));
   PLInsertDictionaryEntry (keydef, plupdate, data);
   PLRelease (data);

   PLRelease (plupdateptr);
   PLRelease (plupdate);
}

void
generate_previews (GtkWidget *button, gpointer ptr)
{
   unsigned	row;
   GList	*l1 	  = NULL;
   GList	*l2 	  = NULL;
   GList	*l3 	  = NULL;
   GList	**listptr = (GList **) ptr;

   previews_active = YES;
   if (preview_update_menu)
      gtk_widget_set_sensitive (preview_update_menu, FALSE);
   for (row = 0; row < (unsigned) GTK_CLIST (theme_list)->rows; row++)
   {
      char *filename;
      
      filename = theme_image_filename (row, "WorkspaceBack", NO);
      if (filename)
	 l1 = g_list_append (l1, filename);
      filename = theme_image_filename (row, "IconBack", NO);
      if (filename)
	 l2 = g_list_append (l2, filename);
      filename = theme_image_filename (row, "FTitleBack", NO);
      if (filename)
	 l3 = g_list_append (l3, filename);
      filename = theme_image_filename (row, "UTitleBack", NO);
      if (filename)
	 l3 = g_list_append (l3, filename);
      filename = theme_image_filename (row, "PTitleBack", NO);
      if (filename)
	 l3 = g_list_append (l3, filename);
      filename = theme_image_filename (row, "MenuTitleBack", NO);
      if (filename)
	 l3 = g_list_append (l3, filename);
      filename = theme_image_filename (row, "MenuTextBack", NO);
      if (filename)
	 l3 = g_list_append (l3, filename);
   }
   
   {
      GtkWidget *progress_window;
      GtkWidget *progress_label;
      GtkWidget *progress_bar;
      GtkWidget *vbox;

      /*
       *  Generate a progressbar and window
       */
      progress_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
      gtk_window_set_position (GTK_WINDOW (progress_window), GTK_WIN_POS_MOUSE);
      gtk_window_set_title (GTK_WINDOW (progress_window), _("Update previews"));
      gtk_signal_connect (GTK_OBJECT (progress_window), "delete_event",
			  GTK_SIGNAL_FUNC (gtk_true), NULL);
      gtk_widget_set_usize (progress_window, 250, -1);
   
      vbox = gtk_vbox_new (FALSE, 5);
      gtk_container_add (GTK_CONTAINER (progress_window), vbox);
      gtk_container_set_border_width (GTK_CONTAINER (vbox), 10);
      progress_label = gtk_label_new (_("Update previews"));
      gtk_box_pack_start (GTK_BOX (vbox), progress_label, FALSE, TRUE, 0);

      {
	 GtkAdjustment *adj;
	 
	 adj = (GtkAdjustment *) gtk_adjustment_new (0, 1, 300, 0, 0, 0);
	 
	 progress_bar = gtk_progress_bar_new_with_adjustment (adj);
      }

      gtk_box_pack_start (GTK_BOX (vbox), progress_bar, FALSE, TRUE, 5);
      gtk_widget_show_all (progress_window);
      
      {
	 int n;
	 int m = g_list_length (l1) + g_list_length (l2) + g_list_length (l3);

	 if (listptr && *listptr)
	    m += g_list_length (* (GList **) ptr);
	 
	 for (n = 0; n < g_list_length (l1); n++)
	    compute_preview (progress_bar, progress_label, n, m,
			     g_list_nth_data (l1, n), 160, 120);
	 for (n = 0; n < g_list_length (l2); n++)
	    compute_preview (progress_bar, progress_label,
			     n + g_list_length (l1), m,
			     g_list_nth_data (l2, n), 64, 64);
	 for (n = 0; n < g_list_length (l3); n++)
	    compute_preview (progress_bar, progress_label,
			     n + g_list_length (l1) + g_list_length (l2), m,
			     g_list_nth_data (l3, n), 234, 22);
	 if (listptr && *listptr)
	 {
	    GList *list = *listptr;
	    
	    for (n = 0; n < g_list_length (list); n++)
	    {
	       previewdata_t *pd    = g_list_nth_data (list, n);
	       char	     *pname = preview_name (pd->name);
	       char	     *path  = get_pixmap_path (pname);

	       if (!path)
	       {
		  compute_preview (progress_bar, progress_label,
				   n + g_list_length (l1) + g_list_length (l2)
				   + g_list_length (l3), m, pd->name, 64, 64);
		  path = get_pixmap_path (pname);
	       }
	       if (path)
	       {
		  GtkWidget *pixmap;

		  pixmap = gtk_object_get_user_data (GTK_OBJECT (pd->button));
		  make_pixmap (path, 64, 64, pixmap);
		  Free (path);
	       }
	       Free (pname);
	    }
	 }
      }
      gtk_widget_destroy (progress_window);
   }

   g_list_foreach (l1, free_memory, NULL);
   g_list_free (l1);
   g_list_foreach (l2, free_memory, NULL);
   g_list_free (l2);
   g_list_foreach (l3, free_memory, NULL);
   g_list_free (l3);
   previews_active = NO;
   if (preview_update_menu)
      gtk_widget_set_sensitive (preview_update_menu, TRUE);
}

/*******************************************************************************

				private code
  
*******************************************************************************/

static void
process_dir (DIR *dir, char *dirname)
/*
 *  Scan contents of directory. Files are appended to the list of themes
 *  whereas a directory is recursively processed ...
 */
{
   struct dirent *file;
   char		 *row [3];

   row [1] = dirname;
	 
   while ((file = readdir (dir)))
   {
      char *filename;
      char *ext;
      DIR  *subdir;

      if (streq (file->d_name, ".") || streq (file->d_name, ".."))
	 continue;

      {
	 char *path = g_strconcat (dirname, "/", file->d_name, NULL);
	 
	 subdir = opendir (path);
	 if (subdir)			/* .themed or subdir */
	 {
	    if (!strstr (path, ".themed"))
	    {
	       process_dir (subdir, path);
	       Free (path);
	       continue;
	    }
	    closedir (subdir);
	 }
	 Free (path);
      }
      filename = g_strdup (file->d_name);
      ext      = strchr (filename, '.');
      if (ext)
      {
	 row [2] = g_strdup (ext);
	 *ext = 0;
      }
      else
	 row [2] = g_strdup ("");
      row [0] = filename;
      gtk_clist_insert (GTK_CLIST (theme_list), -1, row);

      Free (row [2]);
      Free (filename);
   }
   closedir (dir);
}

#ifdef GETSTYLE

static void
save_theme_dialog (GtkWidget *widget, gpointer ptr)
/*
 *  Save theme dialog window
 */
{
   static GtkWidget *window = NULL;
   static GtkWidget *entry  = NULL;
   static GtkWidget *combo  = NULL;
   
   if (!window)
   {
      GtkBox *vbox;

      combo = gtk_combo_new ();
      gtk_widget_set_usize (combo, 300, -1);
      gtk_widget_set_usize (GTK_COMBO (combo)->entry, 300, -1);
      
      entry  = gtk_entry_new ();
      gtk_object_set_user_data (GTK_OBJECT (entry), GTK_COMBO (combo)->entry);

      window = gtk_dialog_new ();
      gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);

      vbox = GTK_BOX (GTK_DIALOG (window)->vbox);
      
      gtk_window_set_title (GTK_WINDOW (window), _("Save a theme"));

      gtk_signal_connect (GTK_OBJECT (window), "delete_event",
			  GTK_SIGNAL_FUNC (hide_window), NULL);
      
      /*
       *  Path dialog
       */
      {
	 GtkWidget *hbox   = gtk_vbox_new (FALSE, 0);
	 GtkWidget *button = gtk_button_new_with_label (_("Browse..."));
	 GtkWidget *bbox   = gtk_hbutton_box_new ();
	 GtkWidget *frame  = gtk_frame_new (_("Install path"));
	 
	 gtk_frame_set_label_align (GTK_FRAME (frame), 0.5, 0.5);
	 gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
	 gtk_container_add (GTK_CONTAINER (frame), hbox);
	 gtk_box_pack_start (vbox, frame, FALSE, TRUE, 0);

	 gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);
	 
	 gtk_box_pack_start (GTK_BOX (hbox), combo, TRUE, TRUE, 5);
	 gtk_box_pack_start (GTK_BOX (hbox), bbox, FALSE, TRUE, 5);
	 gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, TRUE, 5);
	 gtk_signal_connect (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (open_dirbrowser),
			     GTK_COMBO (combo)->entry);
      }
      /*
       *  Theme name
       */
      {
	 GtkWidget *hbox  = gtk_hbox_new (FALSE, 0);
	 GtkWidget *frame = gtk_frame_new (_("Theme name"));

	 gtk_frame_set_label_align (GTK_FRAME (frame), 0.5, 0.5);
	 gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
	 gtk_container_add (GTK_CONTAINER (frame), hbox);
	 gtk_box_pack_start (vbox, frame, FALSE, TRUE, 0);

	 gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);
	 gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 5);
      }
      /*
       *  Buttons
       */
      {
	 GtkWidget *hbox  = gtk_hbutton_box_new ();
	 GtkWidget *button1, *button2;
	 
	 gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), hbox,
			     TRUE, TRUE, 5);

	 gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbox), 5);
	 gtk_button_box_set_layout (GTK_BUTTON_BOX (hbox),
				    GTK_BUTTONBOX_END);
	 button1 = gtk_button_new_with_label (_("Save"));
	 gtk_box_pack_start (GTK_BOX (hbox), button1, TRUE, TRUE, 5);
	 gtk_widget_set_sensitive (button1, FALSE);
	 gtk_signal_connect (GTK_OBJECT (button1), "clicked",
			     GTK_SIGNAL_FUNC (save_theme), entry);
	 gtk_signal_connect_object (GTK_OBJECT (button1), "clicked",
				    GTK_SIGNAL_FUNC (gtk_widget_hide),
				    GTK_OBJECT (window));
	 GTK_WIDGET_SET_FLAGS (button1, GTK_CAN_DEFAULT);

	 gtk_signal_connect (GTK_OBJECT (entry), "changed",
			     GTK_SIGNAL_FUNC (set_themename), button1);
	 gtk_signal_connect (GTK_OBJECT (entry), "activate",
			     GTK_SIGNAL_FUNC (activate_themename), button1);
	 
	 button2 = gtk_button_new_with_label (_("Cancel"));
	 gtk_box_pack_start (GTK_BOX (hbox), button2, TRUE, TRUE, 5);
	 gtk_signal_connect_object (GTK_OBJECT (button2), "clicked",
				    GTK_SIGNAL_FUNC (gtk_widget_hide),
				    GTK_OBJECT (window));
	 GTK_WIDGET_SET_FLAGS (button2, GTK_CAN_DEFAULT);
	 gtk_object_set_user_data (GTK_OBJECT (button1), button2);
	 gtk_widget_grab_default (button2);
      }
   }

   if (GTK_WIDGET_VISIBLE (window))
      gtk_widget_hide (window);
   else
   {
      update_pathlist (GTK_COMBO (combo), NO);
      gtk_widget_show_all (window);
      gtk_widget_grab_focus (entry);
      gtk_entry_select_region (GTK_ENTRY (entry), 0, -1);
   }
}

static gint
hide_window (GtkWidget *widget, gpointer ptr)
/*
 *  Hide the specified window, don't destroy it.
 */
{
   gtk_widget_hide (widget);

   return TRUE;
}

static void
set_themename (GtkWidget *widget, gpointer ptr)
/*
 *  Toggle sensitivity of 'Save' button.
 */
{
   if (strlen (gtk_entry_get_text (GTK_ENTRY (widget))) > 0)
   {
      gtk_widget_set_sensitive (GTK_WIDGET (ptr), YES);
      gtk_widget_grab_default (GTK_WIDGET (ptr));
   }
   else
   {
      gtk_widget_set_sensitive (GTK_WIDGET (ptr), NO);
      gtk_widget_grab_default (GTK_WIDGET (gtk_object_get_user_data (GTK_OBJECT (ptr))));
   }
}

static void
activate_themename (GtkWidget *widget, gpointer ptr)
{
   if (strlen (gtk_entry_get_text (GTK_ENTRY (widget))) > 0)
      gtk_signal_emit_by_name (GTK_OBJECT (ptr), "clicked");
   else
      gtk_signal_emit_by_name (GTK_OBJECT (gtk_object_get_user_data (GTK_OBJECT (ptr))), "clicked");
}

static void
save_theme (GtkWidget *widget, gpointer ptr)
/*
 *  Actually write theme to disk.
 */
{
   GtkEntry	*name = GTK_ENTRY (ptr);
   GtkEntry	*path = GTK_ENTRY (gtk_object_get_user_data (GTK_OBJECT (ptr)));
   char		*themename = gtk_entry_get_text (name);
   char		*pathname  = expand_tilde (gtk_entry_get_text (path));
   char		*filename  = g_strconcat (pathname, "/", themename, ".themed",
					  NULL);

   Free (pathname);
   /*
    *  Make a new directory
    */
   {
      char *cmd = g_strconcat ("mkdir '", filename, "'", NULL);
      
      if (system (cmd))
      {
	 dialog_popup (DIALOG_ERROR, NULL, NULL,
		       _("Can't write theme.\n"
			 "Please check `stderr' for more details."));
	 Free (cmd);
	 Free (filename);
	 return;
      }
      Free (cmd);
   }

   /*
    *  Write style file to dir.themed
    *  Copy all image files to theme dir
    */
   {
      unsigned	 n;
      proplist_t *theme;
      proplist_t *plname;
      proplist_t *pltype    = PLMakeString ("Type");
      char	 *stylename = g_strconcat (filename, "/style", NULL);
      
      plname = PLMakeString (stylename);
      Free (stylename);
      theme = PLMakeDictionaryFromEntries (NULL, NULL, NULL);

      for (n = 0; n < PLGetNumberOfElements (theme_keys); n++)
      {
	 proplist_t *key   = PLGetArrayElement (theme_keys, n);
	 proplist_t *entry = PLGetDictionaryEntry (wmconfig, key);

	 if (entry && PLIsDictionary (entry))
	 {
	    proplist_t *value = PLGetDictionaryEntry (windowmaker, key);
	    proplist_t *type  = PLGetDictionaryEntry (entry, pltype);

	    if (strcaseeq (PLGetString (type), "Texture") && value
		&& PLIsArray (value) && PLGetNumberOfElements (value) >= 2)
	    {
	       char *ttype = PLGetString (PLGetArrayElement (value, 0));
		  
	       if (strcaseeq (ttype + 1, "pixmap")
		   || (strcaseeq (ttype + 2, "gradient") && *ttype == 't'))
	       {
		  char *name;
		  char *path;

		  name = g_strdup (PLGetString (PLGetArrayElement (value, 1)));
		  path = get_pixmap_path (name);
		  Free (name);
		  if (path)
		  {
		     char	*file;
		     proplist_t	*plfile;
		     char	*cmd;
		     
		     cmd = g_strconcat ("cp ", path, " '", filename, "'", NULL);
			
		     if (system (cmd))
		     {
			dialog_popup (DIALOG_ERROR, NULL, NULL,
				      _("Can't write theme.\nPlease check "
					"`stderr' for more details."));
			Free (filename);
			Free (cmd);
			Free (path);
			PLRelease (pltype);
			PLRelease (plname);
			PLRelease (theme);
			return;
		     }
		     value = PLDeepCopy (value);
		     file = strrchr (path, '/');
		     if (!file)
			plfile = PLRetain (PLMakeString (path));
		     else
			plfile = PLRetain (PLMakeString (file + 1));

		     PLRemoveArrayElement (value, 1);
		     PLInsertArrayElement (value, plfile, 1);
		     Free (cmd);
		     Free (path);
		  }
		  else
		  {
		     dialog_popup (DIALOG_ERROR, NULL, NULL,
				   _("Can't write theme.\nPlease check "
				     "`stderr' for more details."));
		     Free (filename);
		     PLRelease (pltype);
		     PLRelease (plname);
		     PLRelease (theme);
		     return;
		  }
	       }
	    }
	    PLInsertDictionaryEntry (theme, key, value);
	 }
      }
      
      PLSetFilename (theme, plname);
      if (!PLSave (theme, YES))
      {
	 dialog_popup (DIALOG_ERROR, NULL, NULL,
		       _("Can't write theme.\n"
			 "Please check `stderr' for more details."));
	 Free (filename);
	 PLRelease (pltype);
	 PLRelease (plname);
	 PLRelease (theme);
	 return;
      }
      PLRelease (theme);
      PLRelease (pltype);
      PLRelease (plname);
   }
   
   Free (filename);
   themes_list ();
}

static bool_t
getstyle_call (const char *themename)
/*
 *  Start wmaker's getstyle program
 */
{
   char	*cmdline  = g_strconcat (GETSTYLE, " -t '", themename, "'", NULL);
   bool_t success = !system (cmdline);

   Free (cmdline);

   return success;
}

#ifdef SETSTYLE

static void
preview_dialog (GtkWidget *button, gpointer ptr)
/*
 *  Preview selected theme and hide main window.
 */
{
   if (GTK_CLIST (theme_list)->rows > 0 &&
       GTK_CLIST(theme_list)->selection)
   {
      char	*tmp_name     = get_temporary_file_name ();
      char	*tmpthemename = tmp_name ? tmp_name : "/tmp/theme";

      delete_file_or_dir (tmpthemename);
      if (!getstyle_call (tmpthemename))
	 dialog_popup (DIALOG_ERROR, NULL, NULL,
		       _("Can't write temporary theme file.\n"
			 "Please check `stderr' for more details."));
      else
      {
	 GList		*selection   = GTK_CLIST (theme_list)->selection;
	 GList		*previewlist = NULL;
	 unsigned	n;
	 
	 for (n = 0; n < g_list_length (selection); n++)
	 {
	    unsigned	row = GPOINTER_TO_INT (g_list_nth_data (selection, n));
	    char	*pathname, *themename, *suffix, *filename;

	    gtk_clist_get_text (GTK_CLIST (theme_list), row, 0, &themename);
	    gtk_clist_get_text (GTK_CLIST (theme_list), row, 1, &pathname);
	    gtk_clist_get_text (GTK_CLIST (theme_list), row, 2, &suffix);
	       
	    filename 	= g_strconcat (pathname, "/", themename, suffix, NULL);
	    previewlist = g_list_append (previewlist, filename);
	 }
	 previewlist = g_list_append (previewlist, g_strdup (tmpthemename));
	 leave_preview (button, previewlist);
      }
   }
}

static void
leave_preview (GtkWidget *button, gpointer ptr)
/*
 *  Show new theme or restore old theme
 */
{
   GList *list    = (GList *) ptr;
   char  *name	  = list->data;
   char  *cmdline = g_strconcat (SETSTYLE, " '", name, "'", NULL);
      
   gtk_widget_hide (main_window);
   if (system (cmdline))
   {
      if (g_list_length (list) > 1)
	 dialog_popup (DIALOG_ERROR, NULL, NULL,
		       _("Can't show preview of theme `%s'.\n"
			 "Please check `stderr' for more details."), name);
      else
	 dialog_popup (DIALOG_ERROR, NULL, NULL,
		       _("Can't revert to old theme.\n"
			 "Please check `stderr' for more details."));
   }
   Free (cmdline);
   
   if (g_list_length (list) == 1)	/* revert */
   {
      delete_file_or_dir (name);
      gtk_widget_show (main_window);
      Free (name);
      g_list_free (list);
   }
   else
   {
      list = g_list_remove (list, list->data);
      dialog (DIALOG_INFO,
	      g_list_length (list) > 1
	      ? _("Show next preview")
	      : _("Leave preview mode"), NULL, leave_preview, list,
	      "Preview of `%s'", g_basename (name));
      Free (name);
   }
}
#endif /* SETSTYLE */

#endif /* GETSTYLE */

static gint
hide_install (GtkWidget *widget, gpointer ptr)
/*
 *  Hide the specified window, don't destroy it.
 */
{
   GtkWidget *window = gtk_widget_get_toplevel (widget);

   gtk_widget_hide (window);
   gtk_clist_clear (GTK_CLIST (gtk_object_get_user_data (GTK_OBJECT (window))));
   
   return TRUE;
}

static void
update_pathlist (GtkCombo *combo, bool_t strip_themes)
/*
 *  Updata of list of writable theme directories
 */
{
   proplist_t		*pltheme  = PLMakeString ("ThemePath");
   proplist_t		*new      = PLGetDictionaryEntry (windowmaker, pltheme);
   static proplist_t	*old      = NULL;
   static GList	*dirlist          = NULL;
   static GList	*dirlist_stripped = NULL;
   static char	*found            = NULL;
   static char	*found_stripped   = NULL;

   PLRelease (pltheme);

   if (!old || !PLIsEqual (new, old))
   {
      if (old)
	 PLRelease (old);
      old = PLDeepCopy (new);

      if (dirlist)
      {
	 g_list_foreach (dirlist, free_memory, NULL);
	 g_list_free (dirlist);
	 dirlist = NULL;
      }
      if (dirlist_stripped)
      {
	 g_list_foreach (dirlist_stripped, free_memory, NULL);
	 g_list_free (dirlist_stripped);
	 dirlist_stripped = NULL;
      }
      if (found)
	 Free (found);
      found = NULL;
      if (found_stripped)
	 Free (found_stripped);
      found_stripped = NULL;

      /*
       *  Make new path list
       */
      {
	 unsigned n;
	 char	  *home = expand_tilde ("~/GNUstep/Library/WindowMaker/Themes");
	 char	  *home_stripped = expand_tilde ("~/GNUstep/Library/WindowMaker");

	 for (n = 0; n < PLGetNumberOfElements (new); n++)
	 {
	    proplist_t *pldir   = PLGetArrayElement (new, n);
	    char       *dirname = expand_tilde (PLGetString (pldir));

	    /*
	     *  Check whether we have write access
	     */
	    {
	       FILE *waccess;
	       char *filename = g_strconcat (dirname, "/.wmakerconf", NULL);
	       
	       waccess = fopen (filename, "w");
	       if (!waccess)
	       {
		  Free (filename);
		  Free (dirname);
		  continue;
	       }
	       fclose (waccess);
	       delete_file_or_dir (filename);
	       Free (filename);
	    }

	    /*
	     *  Remove /Themes or /Styles suffix
	     */
	    {
	       char *ext = strstr (dirname, "/Themes");
	      
	       if (ext && streq (ext, "/Themes"))
		  *ext = 0;
	       else
	       {
		  ext = strstr (dirname, "/Styles");
		  if (ext && streq (ext, "/Styles"))
		     *ext = 0;
	       }
	    }
	    dirlist_stripped = g_list_append (dirlist_stripped,
					      g_strdup (dirname));
	    dirlist          = g_list_append (dirlist,
					      g_strdup (PLGetString (pldir)));
	    if (streq (dirname, home_stripped))
	       found_stripped = dirname;
	    else
	       Free (dirname);
	    {
	       dirname = expand_tilde (PLGetString (pldir));
	       if (streq (dirname, home))
		  found = dirname;
	       else
		  Free (dirname);
	    }
	 }
	 Free (home_stripped);
	 Free (home);
      }
   }
   
   {
      GList *list = strip_themes ? dirlist_stripped : dirlist;
      char  *dflt = strip_themes ? found_stripped : found;
      
      if (list)
      {
	 char *text = g_strdup (gtk_entry_get_text (GTK_ENTRY (combo->entry)));
	 
	 gtk_combo_set_popdown_strings (combo, list);
	 gtk_entry_set_text (GTK_ENTRY (combo->entry),
			     strlen (text) > 0 ? text
			     : (dflt ? dflt : g_list_nth_data (list, 0)));
	 Free (text);
      }
   }
}

static void
open_dirbrowser (GtkWidget *button, gpointer data)
/*
 *  Open browser to select new directory. 
 */
{
   GtkWidget		*entry = (GtkWidget *) data;
   GtkWidget		*filesel;
   GtkFileSelection	*fs;

   filesel = gtk_file_selection_new (_("Choose directory"));
   fs      = GTK_FILE_SELECTION (filesel);

   gtk_widget_set_sensitive (fs->file_list, FALSE);
   gtk_file_selection_hide_fileop_buttons (fs);
   gtk_window_set_position (GTK_WINDOW (fs), GTK_WIN_POS_MOUSE);
   gtk_signal_connect_object (GTK_OBJECT (fs), "destroy",
			      GTK_SIGNAL_FUNC (gtk_widget_destroy),
			      GTK_OBJECT (filesel));
   gtk_signal_connect (GTK_OBJECT (fs->ok_button), "clicked",
		       GTK_SIGNAL_FUNC (insert_dir), entry);
   gtk_object_set_user_data (GTK_OBJECT (fs->ok_button), filesel);
      
   gtk_signal_connect_object (GTK_OBJECT (fs->cancel_button), "clicked",
			      GTK_SIGNAL_FUNC (gtk_widget_destroy),
			      GTK_OBJECT (filesel));
   gtk_signal_connect_object (GTK_OBJECT (fs->ok_button), "clicked",
			      GTK_SIGNAL_FUNC (gtk_widget_destroy),
			      GTK_OBJECT (filesel));
   {
      char *filename = gtk_entry_get_text (GTK_ENTRY (entry));
	 
      if (streq (filename, ""))
	 gtk_file_selection_set_filename (fs, "~/");
      else
	 gtk_file_selection_set_filename (fs, filename);
   }
   gtk_widget_show_all (filesel);
}

static void
insert_dir (GtkWidget *button, gpointer data)
/*
 *  Insert directory selected with pixmap browser into the directory list. 
 */
{
   GtkFileSelection *fs;

   fs = GTK_FILE_SELECTION (gtk_object_get_user_data (GTK_OBJECT (button)));
   gtk_entry_set_text (GTK_ENTRY (data), gtk_file_selection_get_filename (fs));
}

static void
load_theme (GtkWidget *button, gpointer ptr)
{
   if (GTK_CLIST (theme_list)->rows > 0 &&
       GTK_CLIST(theme_list)->selection)
   {
      char	*pathname, *themename, *suffix, *filename;
      unsigned	row = GPOINTER_TO_INT (GTK_CLIST (theme_list)->selection->data);
      
      gtk_clist_get_text (GTK_CLIST (theme_list), row, 0, &themename);
      gtk_clist_get_text (GTK_CLIST (theme_list), row, 1, &pathname);
      gtk_clist_get_text (GTK_CLIST (theme_list), row, 2, &suffix);
      
      filename = g_strconcat (pathname, "/", themename, suffix, NULL);
      if (!load_attributes (filename, NULL))	/* try new style theme */
      {
	 char *path   = g_strdup (filename);
	 char *fname2 = g_strconcat (filename, "/style", NULL);
	 if (!load_attributes (fname2, path))
	    dialog_popup (DIALOG_ERROR, NULL, NULL,
			  _("Can't open selected theme."));
	 Free (path);
	 Free (fname2);
      }
      Free (filename);
   }
}

static void
set_theme (GtkWidget *button, gpointer ptr)
{
   load_theme (button, ptr);
   save_config_file (button, ptr);
}

#ifdef HAVE_RENAME
static void
rename_theme_dialog (GtkWidget *button, gpointer ptr)
{
   if (GTK_CLIST (theme_list)->rows > 0 &&
       GTK_CLIST(theme_list)->selection &&
       g_list_length (GTK_CLIST(theme_list)->selection) == 1)
   {
      GtkWidget	*window;
      GtkBox	*vbox;
      GtkWidget	*entry = gtk_entry_new ();
      unsigned   row   = GPOINTER_TO_INT (g_list_nth_data (GTK_CLIST(theme_list)->selection, 0));
      char     	*pathname, *themename, *suffix, *filename;

      gtk_clist_get_text (GTK_CLIST (theme_list), row, 0, &themename);
      gtk_clist_get_text (GTK_CLIST (theme_list), row, 1, &pathname);
      gtk_clist_get_text (GTK_CLIST (theme_list), row, 2, &suffix);
      filename = g_strconcat (pathname, "/", themename, suffix, NULL);

      window = gtk_dialog_new ();
      gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);

      vbox = GTK_BOX (GTK_DIALOG (window)->vbox);
      gtk_window_set_title (GTK_WINDOW (window), _("Rename theme"));

      gtk_signal_connect_object (GTK_OBJECT (window), "destroy",
				 GTK_SIGNAL_FUNC (gtk_widget_destroy),
				 GTK_OBJECT (window));
      {
	 GtkWidget *hbox = gtk_hbox_new (FALSE, 0);

	 gtk_container_set_border_width (GTK_CONTAINER (hbox), 15);
	 
	 gtk_box_pack_start (GTK_BOX (hbox),
			     gtk_label_new (_("New theme name: ")),
			     TRUE, TRUE, 0);
	 gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);
	 gtk_box_pack_start (vbox, hbox, FALSE, TRUE, 0);
	 gtk_entry_set_text (GTK_ENTRY (entry), filename);
	 gtk_object_set_user_data (GTK_OBJECT (entry), filename);
	 gtk_entry_select_region (GTK_ENTRY (entry), 0, -1);
      }
      /*
       *  Buttons
       */
      {
	 GtkWidget *hbox = gtk_hbutton_box_new ();
	 GtkWidget *button;
	 
	 gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), hbox,
			     TRUE, TRUE, 0);

	 gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbox), 5);
	 gtk_button_box_set_layout (GTK_BUTTON_BOX (hbox),
				    GTK_BUTTONBOX_END);

	 button = gtk_button_new_with_label (_("OK"));
	 gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 5);
	 gtk_signal_connect (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (rename_theme), entry);
	 gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
				    GTK_SIGNAL_FUNC (gtk_widget_destroy),
				    GTK_OBJECT (window));
	 GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
    
	 button = gtk_button_new_with_label (_("Cancel"));
	 gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 5);
	 gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
				    GTK_SIGNAL_FUNC (gtk_widget_destroy),
				    GTK_OBJECT (window));
	 GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	 gtk_widget_grab_default (button);
      }
      gtk_widget_show_all (window);
   }
}

static void
rename_theme (GtkWidget *widget, gpointer ptr)
/*
 *  Actually rename theme files
 */
{
   GtkEntry *entry = (GtkEntry *) ptr;
   char     *new   = gtk_entry_get_text (GTK_ENTRY (entry));
   char     *old   = gtk_object_get_user_data (GTK_OBJECT (entry));

   if (rename (old, new) != 0)
      dialog_popup (DIALOG_ERROR, NULL, NULL,
		    _("Can't rename `%s' to `%s'."), old, new);
   Free (old);
   themes_list ();
}
#endif /* HAVE_RENAME */

static void
remove_theme_dialog (GtkWidget *button, gpointer ptr)
{
   if (GTK_CLIST (theme_list)->rows > 0 &&
       GTK_CLIST(theme_list)->selection)
   {
      GtkWidget	*window;
      GtkBox	*vbox;

      window = gtk_dialog_new ();
      gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);

      vbox = GTK_BOX (GTK_DIALOG (window)->vbox);
      gtk_window_set_title (GTK_WINDOW (window), _("Remove themes"));

      gtk_signal_connect_object (GTK_OBJECT (window), "destroy",
				 GTK_SIGNAL_FUNC (gtk_widget_destroy),
				 GTK_OBJECT (window));
      {
	 GtkWidget *hbox = gtk_hbox_new (FALSE, 0);

	 gtk_container_set_border_width (GTK_CONTAINER (hbox), 15);
	 
	 gtk_box_pack_start (GTK_BOX (hbox),
			     gtk_label_new (_("Remove selected theme files")),
			     TRUE, TRUE, 0);
	 gtk_box_pack_start (vbox, hbox, FALSE, TRUE, 0);
      }
      {
	 GtkWidget *filelist = gtk_clist_new (1);

	 gtk_clist_set_auto_sort (GTK_CLIST (filelist), YES);
	 gtk_clist_set_column_auto_resize (GTK_CLIST (filelist), 0, YES);
	 gtk_clist_column_titles_passive (GTK_CLIST (filelist));
	 gtk_clist_set_selection_mode (GTK_CLIST (filelist),
				       GTK_SELECTION_EXTENDED);
	 {
	    GtkWidget *scrolled = gtk_scrolled_window_new (NULL, NULL);
	       
	    gtk_container_set_border_width (GTK_CONTAINER (scrolled), 5);
	    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
					    GTK_POLICY_AUTOMATIC,
					    GTK_POLICY_AUTOMATIC);
	    gtk_container_add (GTK_CONTAINER (scrolled), filelist);
	    gtk_box_pack_start (vbox, scrolled, TRUE, TRUE, 0);
	 }
	 /*
	  *  Buttons
	  */
	 {
	    GtkWidget *hbox = gtk_hbutton_box_new ();
	    GtkWidget *button;
	 
	    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), hbox,
				TRUE, TRUE, 0);

	    gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbox), 5);
	    gtk_button_box_set_layout (GTK_BUTTON_BOX (hbox),
				       GTK_BUTTONBOX_END);

	    button = gtk_button_new_with_label (_("Remove"));
	    gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 5);
	    gtk_signal_connect (GTK_OBJECT (button), "clicked",
				GTK_SIGNAL_FUNC (remove_theme), filelist);
	    gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
				       GTK_SIGNAL_FUNC (gtk_widget_destroy),
				       GTK_OBJECT (window));
	    GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
    
	    button = gtk_button_new_with_label (_("Cancel"));
	    gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 5);
	    gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
				       GTK_SIGNAL_FUNC (gtk_widget_destroy),
				       GTK_OBJECT (window));
	    GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	    gtk_widget_grab_default (button);
	 }
	 /*
	  *  List of Files
	  */
	 {
	    unsigned	n;
	    GList	*selection = GTK_CLIST (theme_list)->selection;
      
	    for (n = 0; n < g_list_length (selection); n++)
	    {
	       unsigned row = GPOINTER_TO_INT (g_list_nth_data (selection, n));
	       char	*pathname, *themename, *suffix, *filename;

	       gtk_clist_get_text (GTK_CLIST (theme_list), row, 0, &themename);
	       gtk_clist_get_text (GTK_CLIST (theme_list), row, 1, &pathname);
	       gtk_clist_get_text (GTK_CLIST (theme_list), row, 2, &suffix);
	       
	       filename = g_strconcat (pathname, "/", themename, suffix, NULL);
	       gtk_clist_insert (GTK_CLIST (filelist), -1, &filename);
	       if (streq (suffix, ".themed"))
	       {
		  DIR  *themedir = opendir (filename);
		  if (themedir)
		  {
		     struct dirent *file;
		  
		     while ((file = readdir (themedir)))
		     {
			if (!streq (file->d_name, ".")
			    && !streq (file->d_name, ".."))
			{
			   char *tmp = g_strconcat (filename, "/", file->d_name,
						    NULL);
			   if (streq (file->d_name, ".previews"))
			   {
			      DIR *pdir = opendir (tmp);
			      if (pdir)
			      {
				 struct dirent *f;
				 while ((f = readdir (pdir)))
				    if (!streq (f->d_name, ".")
					&& !streq (f->d_name, ".."))
				    {
				       char *t = g_strconcat (tmp, "/", f->d_name,
							      NULL);
				       gtk_clist_insert (GTK_CLIST (filelist), -1,
							 &t);
				       Free (t);
				    }
				 closedir (pdir);
			      }
			   }
			   gtk_clist_insert (GTK_CLIST (filelist), -1, &tmp);
			   Free (tmp);
			}
		     }
		     closedir (themedir);
		  }
	       }
	       else
	       {
		  const char *attr[] = {"WorkspaceBack", "IconBack",
					"FTitleBack", "UTitleBack",
					"PTitleBack", "MenuTitleBack",
					"MenuTextBack"};
		  unsigned n;

		  for (n = 0; n < sizeof (attr) / sizeof (attr [0]); n++)
		  {
		     char *name = theme_image_filename (row, attr [n], NO);

		     if (name)
		     {
			char *pname = preview_name (name);
			char *path  = get_pixmap_path (pname);
			
			gtk_clist_insert (GTK_CLIST (filelist), -1, &name);
			if (path)
			{
			   gtk_clist_insert (GTK_CLIST (filelist), -1, &path);
			   Free (path);
			}
			Free (pname);
			Free (name);
		     }
		  }
	       }
	       Free (filename);
	    }
	 }
	 gtk_clist_select_all (GTK_CLIST (filelist));
	 gtk_widget_set_usize (window, 500, 300);
	 gtk_widget_show_all (window);
      }
   }
}

static void
remove_theme (GtkWidget *widget, gpointer ptr)
/*
 *  Actually remove theme files
 */
{
   GtkCList	*filelist = GTK_CLIST (ptr);
   GList	*list 	  = g_list_reverse (filelist->selection);

   g_list_foreach (list, remove_file, filelist);
   themes_list ();
}

static void
remove_file (gpointer data, gpointer user_data)
/*
 *  Remove a file
 */
{
   GtkCList	*filelist = GTK_CLIST (user_data);
   gint		row       = GPOINTER_TO_INT (data);
   char		*path;
   
   gtk_clist_get_text (filelist, row, 0, &path);
   delete_file_or_dir (path);
}

static GtkWidget *
theme_preview (void)
{
   GtkWidget *frame    = gtk_frame_new (_("Theme pixmaps"));
   GtkWidget *hbox     = gtk_hbox_new (FALSE, 0);
   GtkWidget *vbox     = gtk_vbox_new (FALSE, 0);
   GtkWidget *ibox;

   gtk_frame_set_label_align (GTK_FRAME (frame), 0.5, 0.5);
   gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
   gtk_container_add (GTK_CONTAINER (frame), vbox);
   gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

   ibox          = gtk_vbox_new (FALSE, 0);
   workspaceback = make_pixmap (PKGDATADIR "/black.xpm", 160, 120, NULL);
   gtk_box_pack_start (GTK_BOX (hbox), ibox, FALSE, TRUE, 0);
   gtk_box_pack_start (GTK_BOX (ibox), workspaceback, TRUE, TRUE, 0);
   gtk_container_set_border_width (GTK_CONTAINER (ibox), 5);

   ibox     = gtk_vbox_new (FALSE, 0);
   iconback = make_pixmap (PKGDATADIR "/black.xpm", 64, 64, NULL);
   gtk_box_pack_start (GTK_BOX (hbox), ibox, FALSE, TRUE, 0);
   gtk_box_pack_start (GTK_BOX (ibox), iconback, TRUE, TRUE, 0);
   gtk_container_set_border_width (GTK_CONTAINER (ibox), 5);

   ibox         = gtk_vbox_new (FALSE, 0);
   titlebarback = make_pixmap (PKGDATADIR "/black.xpm", 234, 22, NULL);
   gtk_box_pack_start (GTK_BOX (vbox), ibox, FALSE, TRUE, 0);
   gtk_box_pack_start (GTK_BOX (ibox), titlebarback, TRUE, TRUE, 0);
   gtk_container_set_border_width (GTK_CONTAINER (ibox), 5);
   
   gtk_widget_show_all (frame);
   
   return frame;
}

static void
update_preview (GtkCList *clist, gint row, gint column, GdkEvent *event,
		gpointer data)
{
   if (GTK_CLIST (clist)->rows > 0 &&
       GTK_CLIST(theme_list)->selection)
   {
      char *filename;
      unsigned n = g_list_length (GTK_CLIST (theme_list)->selection);

      if (open_button)
	 gtk_widget_set_sensitive (open_button, n == 1);
      if (open_menu_button)
	 gtk_widget_set_sensitive (open_menu_button, n == 1);
      if (set_menu_button)
	 gtk_widget_set_sensitive (set_menu_button, n == 1);
      if (set_button)
	 gtk_widget_set_sensitive (set_button, n == 1);
	 
      filename = theme_image_filename (row, "WorkspaceBack", NO);
      if (filename)
      {
	 char *pname = preview_name (filename);
	 char *name  = get_pixmap_path (pname);

	 make_pixmap (name ? name : filename, 160, 120, workspaceback);
	 Free (pname);
	 Free (filename);
	 if (name)
	    Free (name);
      }
      else
	 theme_gradient (row, "WorkspaceBack", workspaceback);
      
      filename = theme_image_filename (row, "IconBack", NO);
      if (filename)
      {
	 char *pname = preview_name (filename);
	 char *name  = get_pixmap_path (pname);

	 make_pixmap (name ? name : filename, 64, 64, iconback);
	 Free (pname);
	 Free (filename);
	 if (name)
	    Free (name);
      }
      else
	 theme_gradient (row, "IconBack", iconback);
      
      filename = theme_image_filename (row, "FTitleBack", NO);
      if (filename)
      {
	 char *pname = preview_name (filename);
	 char *name  = get_pixmap_path (pname);

	 make_pixmap (name ? name : filename, 234, 22, titlebarback);
	 Free (pname);
	 Free (filename);
	 if (name)
	    Free (name);
      }
      else
	 theme_gradient (row, "FTitleBack", titlebarback);
   }
}

static void
free_memory (gpointer data, gpointer user_data)
{
   Free (data);
}

enum {
  TARGET_STRING,
  TARGET_ROOTWIN
};

static GtkTargetEntry target_table[] = {
  { "STRING",     0, TARGET_STRING },
  { "text/plain", 0, TARGET_STRING }
};

static guint n_targets = sizeof (target_table) / sizeof (target_table [0]);

static void
init_dnd (GtkWidget *widget)
{
   gtk_drag_dest_set (widget, GTK_DEST_DEFAULT_DROP, target_table, n_targets, 
		      GDK_ACTION_COPY | GDK_ACTION_MOVE);
   gtk_signal_connect (GTK_OBJECT (widget), "drag_data_received",
		       GTK_SIGNAL_FUNC (drag_data_received), NULL);
}

static void  
drag_data_received (GtkWidget *widget, GdkDragContext *context, gint x, gint y,
		    GtkSelectionData *data, guint info, guint time)
{
   if (data->length >= 0 && data->format == 8)
   {
      char *text = (char *) data->data;

      if (widget == theme_list)
	 install_theme_dialog (widget, text);
      else
	 insert_dnd_elements (GTK_CLIST (widget), text);
      gtk_drag_finish (context, TRUE, FALSE, time);
   }
   else
      gtk_drag_finish (context, FALSE, FALSE, time);
}

static void
insert_dnd_elements (GtkCList *clist, const char *start)
{
   while (strneq (start, "http://", strlen ("http://"))
	  || strneq (start, "ftp://", strlen ("ftp://"))
	  || strneq (start, "file:/", strlen ("file:/")))
   {
      char *end;
	 
      if ((end = strstr (start + 1, "file:/")) ||
	  (end = strstr (start + 1, "http://")) ||
	  (end = strstr (start + 1, "ftp://")))
      {
	 char		*tmp;
	 const char	*last = end;
	    
	 while (isspace (*--last))
	    ;
	 tmp = g_strndup (start, last - start + 1);
	 gtk_clist_insert (clist, -1, &tmp);
	 Free (tmp);
	 start = end;
      }
      else
      {
	 char		*tmp;
	 const char	*last = start + strlen (start);
	    
	 while (isspace (*--last))
	    ;
	 tmp = g_strndup (start, last - start + 1);
	 gtk_clist_insert (clist, -1, &tmp);
	 Free (tmp);
	 start++;
      }
      gtk_clist_select_row (clist, clist->rows - 1, 0);
   }
}

static void
install_theme_dialog (GtkWidget *widget, gpointer ptr)
/*
 *  Install theme dialog window
 */
{
   static GtkWidget *window   = NULL;
   static GtkWidget *combo    = NULL;
   static GtkWidget *filelist = NULL;

   if (!window)
   {
      GtkBox	*vbox;

      filelist = gtk_clist_new (1);
      combo = gtk_combo_new ();
      gtk_widget_set_usize (combo, 300, -1);
      gtk_widget_set_usize (GTK_COMBO (combo)->entry, 300, -1);
      
      window = gtk_dialog_new ();
      gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);

      gtk_object_set_user_data (GTK_OBJECT (window), filelist);
      
      vbox = GTK_BOX (GTK_DIALOG (window)->vbox);
      
      gtk_window_set_title (GTK_WINDOW (window), _("Install themes"));

      gtk_signal_connect (GTK_OBJECT (window), "delete_event",
			  GTK_SIGNAL_FUNC (hide_install), NULL);
      /*
       *  Path dialog
       */
      {
	 GtkWidget *hbox   = gtk_vbox_new (FALSE, 0);
	 GtkWidget *button = gtk_button_new_with_label (_("Browse..."));
	 GtkWidget *bbox   = gtk_hbutton_box_new ();
	 GtkWidget *frame	= gtk_frame_new (_("Install path"));

	 gtk_frame_set_label_align (GTK_FRAME (frame), 0.5, 0.5);
	 gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
	 gtk_container_add (GTK_CONTAINER (frame), hbox);
	 gtk_box_pack_start (vbox, frame, FALSE, TRUE, 0);
	 
	 gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);
	 gtk_box_pack_start (GTK_BOX (hbox), combo, TRUE, TRUE, 5);
	 gtk_box_pack_start (GTK_BOX (hbox), bbox, FALSE, TRUE, 5);
	 gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, TRUE, 5);
	 gtk_signal_connect (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (open_dirbrowser),
			     GTK_COMBO (combo)->entry);
      }
      /*
       *  List of themes
       */
      {
	 GtkWidget *hbox   = gtk_vbox_new (FALSE, 0);
	 GtkWidget *button = gtk_button_new_with_label (_("Browse..."));
	 GtkWidget *bbox   = gtk_hbutton_box_new ();
	 GtkWidget *frame  = gtk_frame_new (_("List of themes"));

	 gtk_frame_set_label_align (GTK_FRAME (frame), 0.5, 0.5);
	 gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
	 gtk_container_add (GTK_CONTAINER (frame), hbox);
	 gtk_box_pack_start (vbox, frame, FALSE, TRUE, 0);

	 gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);


	 init_dnd (filelist);
	 gtk_clist_set_column_auto_resize (GTK_CLIST (filelist), 0, YES);
	 gtk_clist_column_titles_passive (GTK_CLIST (filelist));
	 gtk_clist_set_selection_mode (GTK_CLIST (filelist),
				       GTK_SELECTION_EXTENDED);
	 {
	    GtkWidget *scrolled = gtk_scrolled_window_new (NULL, NULL);
	    gtk_container_set_border_width (GTK_CONTAINER (scrolled), 5);
	    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
					    GTK_POLICY_AUTOMATIC,
					    GTK_POLICY_AUTOMATIC);
	    gtk_container_add (GTK_CONTAINER (scrolled), filelist);
	    gtk_box_pack_start (GTK_BOX (hbox), scrolled, TRUE, TRUE, 0);
	    gtk_widget_set_usize (scrolled, -1, 100);
	 }

	 gtk_box_pack_start (GTK_BOX (hbox), bbox, FALSE, TRUE, 5);
	 gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, TRUE, 5);
	 gtk_signal_connect (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (open_theme_browser), filelist);
	 gtk_object_set_user_data (GTK_OBJECT (filelist),
				   GTK_COMBO (combo)->entry);

	 gtk_tooltips_set_tip (tooltips, button,
			       _("Select themes with a file selection dialog "
				 "(or use drag and drop with your "
				 "file manager and WWW browser)."), NULL);
      }
      /*
       *  Buttons
       */
      {
	 GtkWidget *hbox  = gtk_hbutton_box_new ();
	 GtkWidget *button1, *button2;
	 
	 gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbox), 5);
	 gtk_button_box_set_layout (GTK_BUTTON_BOX (hbox),
				    GTK_BUTTONBOX_END);
	 gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), hbox,
			     FALSE, TRUE, 5);
	 button1 = gtk_button_new_with_label (_("Install"));
	 gtk_widget_set_sensitive (button1, FALSE);
	 gtk_box_pack_start (GTK_BOX (hbox), button1, FALSE, TRUE, 5);
	 gtk_signal_connect (GTK_OBJECT (button1), "clicked",
			     GTK_SIGNAL_FUNC (install_themes), filelist);
	 gtk_signal_connect (GTK_OBJECT (button1), "clicked",
			     GTK_SIGNAL_FUNC (hide_install), NULL);
	 gtk_signal_connect (GTK_OBJECT (filelist), "select_row",
			     GTK_SIGNAL_FUNC (update_selection), button1);
	 gtk_signal_connect (GTK_OBJECT (filelist), "unselect_row",
			     GTK_SIGNAL_FUNC (update_selection), button1);
	 GTK_WIDGET_SET_FLAGS (button1, GTK_CAN_DEFAULT);

	 button2 = gtk_button_new_with_label (_("Cancel"));
	 gtk_box_pack_start (GTK_BOX (hbox), button2, FALSE, TRUE, 5);
	 gtk_signal_connect (GTK_OBJECT (button2), "clicked",
			     GTK_SIGNAL_FUNC (hide_install), NULL);
	 GTK_WIDGET_SET_FLAGS (button2, GTK_CAN_DEFAULT);
	 gtk_widget_grab_default (button2);
	 gtk_object_set_user_data (GTK_OBJECT (button1), button2);
      }
   }

   update_pathlist (GTK_COMBO (combo), YES);
   gtk_widget_show_all (window);
   if (ptr)
      insert_dnd_elements (GTK_CLIST (filelist), ptr);
}

static void
open_theme_browser (GtkWidget *button, gpointer data)
/*
 *  Open file browser to select themes. 
 */
{
   GtkWidget		*clist = (GtkWidget *) data;
   GtkWidget		*filesel;
   GtkFileSelection	*fs;

   filesel = gtk_file_selection_new (_("Choose themes to install"));
   fs      = GTK_FILE_SELECTION (filesel);

   gtk_file_selection_hide_fileop_buttons (fs);
   gtk_window_set_position (GTK_WINDOW (fs), GTK_WIN_POS_MOUSE);
   gtk_signal_connect_object (GTK_OBJECT (fs), "destroy",
			      GTK_SIGNAL_FUNC (gtk_widget_destroy),
			      GTK_OBJECT (filesel));
   gtk_signal_connect (GTK_OBJECT (fs->ok_button), "clicked",
		       GTK_SIGNAL_FUNC (insert_theme_files), clist);
   gtk_object_set_user_data (GTK_OBJECT (fs->ok_button), filesel);

   gtk_clist_set_selection_mode (GTK_CLIST (fs->file_list),
				 GTK_SELECTION_EXTENDED);
   gtk_signal_connect_object (GTK_OBJECT (fs->cancel_button), "clicked",
			      GTK_SIGNAL_FUNC (gtk_widget_destroy),
			      GTK_OBJECT (filesel));
   gtk_signal_connect_object (GTK_OBJECT (fs->ok_button), "clicked",
			      GTK_SIGNAL_FUNC (gtk_widget_destroy),
			      GTK_OBJECT (filesel));
   gtk_file_selection_set_filename (fs, "~/");
   gtk_widget_show_all (filesel);
}

static void
install_themes (GtkWidget *widget, gpointer ptr)
/*
 *  Install selected list of themes
 */
{
   GtkWidget *vbox;
   GtkWidget *progress_window;
   GtkWidget *progress_label;
   GtkWidget *progress_bar;
   GtkCList  *install_list = GTK_CLIST (ptr);
   GList     *files        = install_list->selection;
   unsigned  n;
   char     *themedir = gtk_entry_get_text (GTK_ENTRY (gtk_object_get_user_data (GTK_OBJECT (install_list))));
   gint	     timer;
   pid_t     pid;
   
   gtk_widget_hide (gtk_widget_get_toplevel (widget));
   if (strlen (themedir) == 0)
   {
      dialog_popup (DIALOG_ERROR, NULL, NULL,
		    _("No installation prefix given."));
      return;
   }
   themedir = expand_tilde (themedir);

   /*
    *  Generate a progressbar and window
    */
   progress_window = gtk_dialog_new ();
   gtk_window_set_position (GTK_WINDOW (progress_window), GTK_WIN_POS_MOUSE);
   gtk_window_set_title (GTK_WINDOW (progress_window), _("Install themes"));
   gtk_widget_set_usize (progress_window, 250, -1);
   
   vbox = gtk_vbox_new (FALSE, 5);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (progress_window)->vbox),
		       vbox, TRUE, TRUE, 0);
   gtk_container_set_border_width (GTK_CONTAINER (vbox), 10);
   progress_label = gtk_label_new ("");
   gtk_box_pack_start (GTK_BOX (vbox), progress_label, FALSE, TRUE, 0);

   {
      GtkAdjustment *adj;
	 
      adj = (GtkAdjustment *) gtk_adjustment_new (0, 1, 300, 0, 0, 0);
	 
      progress_bar = gtk_progress_bar_new_with_adjustment (adj);
      gtk_progress_set_activity_mode (GTK_PROGRESS (progress_bar), TRUE);
      gtk_progress_bar_set_activity_step (GTK_PROGRESS_BAR (progress_bar), 5);
      gtk_progress_bar_set_activity_blocks (GTK_PROGRESS_BAR (progress_bar), 4);
      
      timer = gtk_timeout_add (100, progress_timer, progress_bar);
   }

   gtk_box_pack_start (GTK_BOX (vbox), progress_bar, FALSE, TRUE, 5);
   {
      GtkWidget *bbox   = gtk_hbutton_box_new ();
      GtkWidget *cancel = gtk_button_new_with_label (_("Skip"));
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (progress_window)->action_area),
			  bbox, FALSE, TRUE, 5);
      gtk_box_pack_start (GTK_BOX (bbox), cancel, FALSE, TRUE, 0);
      gtk_signal_connect (GTK_OBJECT (cancel), "clicked",
			  GTK_SIGNAL_FUNC (cancel_installation), &pid);
      gtk_signal_connect (GTK_OBJECT (progress_window), "delete_event",
			  GTK_SIGNAL_FUNC (delete_button), cancel);
   }
      
   gtk_widget_show_all (progress_window);
      
   for (n = 0; n < g_list_length (files); n++)
   {
      char *orig_filename     = NULL;
      char *install_filename  = NULL;
      char *unpacked_filename = NULL;
      char *tmp_filename      = NULL;
      
      gtk_clist_get_text (install_list,
			  GPOINTER_TO_INT (g_list_nth_data (files, n)),
			  0, &orig_filename);
      {
	 char *basename = strrchr (orig_filename, '=');

	 if (!basename)
	    basename = strrchr (orig_filename, '/');
	 gtk_label_set_text (GTK_LABEL (progress_label),
			     basename ? basename + 1: orig_filename);
	 gtk_label_set_text (GTK_LABEL (progress_label),
			     basename ? basename + 1: orig_filename);
      }

      while (gtk_events_pending())
	 gtk_main_iteration();

      /*
       *  Get theme via WWW
       */
      if (!strneq (orig_filename, "file:/", strlen ("file:/")))
#ifdef WWWREQUEST      
      {
	 char *tmp_name     = get_temporary_file_name ();
	 char *tmpthemename = tmp_name ? tmp_name : "/tmp/theme";

	 pid = fork ();
	 if (pid)			/* parent */
	 {
	    int status;

	    while (waitpid (pid, &status, WNOHANG) == 0)
	       gtk_main_iteration ();
	    if (WIFSIGNALED (status))
	       continue;
	 }
	 else				/* child */
	 {
	    execl (PERL, PERL, PKGDATADIR "/getfile.pl",
		   orig_filename, tmpthemename, NULL);
#ifdef HAVE__EXIT
	    _exit (1);
#else  /* HAVE__EXIT */
	    exit (1);
#endif /* HAVE__EXIT */
	 }
	 tmp_filename = install_filename = g_strdup (tmpthemename);
      }
#else  /* not WWWREQUEST */
         continue;			/* skip this file */
#endif /* not WWWREQUEST */
      else
	 install_filename = g_strdup (orig_filename + strlen ("file:"));
      {
	 bool_t	gzip = NO, bzip2 = NO, tar = NO;

#if defined(FILETYPE) && defined(PERL)
	 /*
	  *  Test for gzip or bzip2 compression
	  */
	 {
	    char *tmp;
	    tmp = g_strdup_printf ("%s '%s' | %s -n -e 'exit (1) if (/gzip/)'",
				   FILETYPE, install_filename, PERL);
	    if (system (tmp))
	       gzip = YES;
	    else
	    {
	       Free (tmp);
	       tmp = g_strdup_printf ("%s '%s' | %s -n -e 'exit (1) if (/bzip/)'",
				      FILETYPE, install_filename, PERL);
	       if (system (tmp))
		  bzip2 = YES;
	    }
	    Free (tmp);
	 }
#else /* not defined(FILETYPE) && defined(PERL) */
	 gzip = YES;			
#endif /* not defined(FILETYPE) && defined(PERL) */

#ifdef GUNZIP
	 if (gzip)
	 {
	    char *tmp;

	    unpacked_filename = g_strconcat (install_filename, ".tar", NULL);
	    tmp = g_strdup_printf ("%s -c < '%s' > '%s'", GUNZIP,
				   install_filename, unpacked_filename);
	    if (system (tmp))		/* no success ? */
	    {
	       delete_file_or_dir (unpacked_filename);
	       Free (unpacked_filename);
	       unpacked_filename = NULL;
	    }
	    Free (tmp);
	 }
	 else
#endif /* GUNZIP */
      
#ifdef BUNZIP2
	    if (bzip2)
	    {
	       char *tmp;

       unpacked_filename = g_strconcat (install_filename, ".tar", NULL);
	       tmp = g_strdup_printf ("%s < '%s' > '%s'", BUNZIP2,
				      install_filename, unpacked_filename);
	       if (system (tmp))	/* no success ? */
	       {
		  delete_file_or_dir (unpacked_filename);
		  Free (unpacked_filename);
		  unpacked_filename = NULL;
	       }
	       Free (tmp);
	    }
#endif /* BUNZIP2 */

	 if (unpacked_filename)		/* unpacked file has been created ? */
	 {
	    if (tmp_filename)
	       delete_file_or_dir (tmp_filename);
	    tmp_filename = unpacked_filename;
	 }
	 else
	    unpacked_filename = g_strdup (install_filename); /* still the same */

#if defined(FILETYPE) && defined(PERL)
	 {
	    char *tmp;

	    tmp = g_strdup_printf ("%s '%s' | %s -n -e 'exit (1) if (/:.*tar/)'",
				   FILETYPE, unpacked_filename, PERL);
	    if (system (tmp))
	       tar = YES;
	    Free (tmp);
	 }
#else /* not FILETYPE */
	 tar = YES;			/* guess */
#endif /* not FILETYPE */
	    
#ifdef TAR
	 if (tar)
	 {
	    char *tmp = g_strdup_printf ("cd '%s'; %s x <'%s'", themedir, TAR,
					 unpacked_filename);
	    if (system (tmp))
	       dialog_popup (DIALOG_ERROR, NULL, NULL,
			     _("Can't unpack theme file.\n"
			       "Please check `stderr' for more details."));
	    Free (tmp);
	 }
	 else
#endif /* TAR */
	 {
	    char *tmp = g_strdup_printf ("cp -f '%s' '%s'", unpacked_filename,
					 themedir);
	    if (system (tmp))
	       dialog_popup (DIALOG_ERROR, NULL, NULL,
			     _("Can't unpack theme file.\n"
			       "Please check `stderr' for more details."));
	    Free (tmp);
	 }
	 if (tmp_filename)
	    delete_file_or_dir (tmp_filename);
	 Free (unpacked_filename);
	 Free (install_filename);
      }
   }
   gtk_timeout_remove (timer);
   gtk_widget_destroy (progress_window);
   themes_list ();
   Free (themedir);
   gtk_clist_clear (GTK_CLIST (install_list));
}

static void
cancel_installation (GtkWidget *widget, gpointer ptr)
{
   pid_t pid = * (pid_t *) ptr;

   kill (pid, 15);
}

static gint
insert_theme_files (GtkWidget *widget, gpointer ptr)
{
   GtkFileSelection	*fs;
   GList		*list;

   fs 	= GTK_FILE_SELECTION (gtk_object_get_user_data (GTK_OBJECT (widget)));
   list = GTK_CLIST (fs->file_list)->selection;
   gtk_object_set_user_data (GTK_OBJECT (fs), ptr);
   g_list_foreach (list, append_text, fs);

   return TRUE;
}

static void
append_text (gpointer data, gpointer user_data)
{
   GtkFileSelection	*fs       = GTK_FILE_SELECTION (user_data);
   GtkCList		*filelist = GTK_CLIST (fs->file_list);
   gint			row       = GPOINTER_TO_INT (data);
   GtkCList		*themelist;
   char			*path;

   themelist = GTK_CLIST (gtk_object_get_user_data (GTK_OBJECT (fs)));
   gtk_clist_get_text (filelist, row, 0, &path);
   gtk_entry_set_text (GTK_ENTRY (fs->selection_entry), path);

   {
      char *name = gtk_file_selection_get_filename (fs);
      char *tmp  = g_strconcat ("file:", name, NULL);

      gtk_clist_insert (themelist, -1, &tmp);
      gtk_clist_select_row (themelist, GTK_CLIST (themelist)->rows - 1, 0);
      Free (tmp);
   }
}

static void
update_selection (GtkCList *themelist, gint row, gint column, GdkEvent *event,
		  gpointer data)
{
   if (g_list_length (themelist->selection) > 0)
   {
      gtk_widget_set_sensitive (GTK_WIDGET (data), TRUE);
      gtk_widget_grab_default (GTK_WIDGET (data));
   }
   else
   {
      gtk_widget_set_sensitive (GTK_WIDGET (data), FALSE);
      gtk_widget_grab_default (GTK_WIDGET (gtk_object_get_user_data (GTK_OBJECT (data))));
   }
}

static gint
context_menu (GtkCTree *ctree, GdkEventButton *event, gpointer ptr)
/*
 *  Popup context menu (right click)
 */
{
   gint	row;
   gint	column;

   if (!gtk_clist_get_selection_info (GTK_CLIST (theme_list),
				      event->x, event->y, &row, &column)
       || event->button != 3)
      return FALSE;

   gtk_clist_select_row (GTK_CLIST (theme_list), row, 0);
   
   {
      static GtkWidget  *menu = NULL;
      GtkWidget		*menu_item;

      if (!menu)
      {
	 menu = gtk_menu_new ();

	 open_menu_button = menu_item = gtk_menu_item_new_with_label (_("Open"));
	 gtk_menu_append (GTK_MENU (menu), menu_item);
	 gtk_widget_show (menu_item);
	 gtk_signal_connect (GTK_OBJECT(menu_item), "activate",
			     GTK_SIGNAL_FUNC (load_theme), NULL);
	 if (g_list_length (GTK_CLIST (theme_list)->selection) != 1)
	    gtk_widget_set_sensitive (menu_item, FALSE);
	 
#ifdef GETSTYLE
	 menu_item = gtk_menu_item_new_with_label (_("Save..."));
	 gtk_menu_append (GTK_MENU (menu), menu_item);
	 gtk_widget_show (menu_item);
	 gtk_signal_connect (GTK_OBJECT(menu_item), "activate",
			     GTK_SIGNAL_FUNC (save_theme_dialog), NULL);

#ifdef SETSTYLE

	 set_menu_button = menu_item = gtk_menu_item_new_with_label (_("Set"));
	 gtk_menu_append (GTK_MENU (menu), menu_item);
	 gtk_widget_show (menu_item);
	 gtk_signal_connect (GTK_OBJECT(menu_item), "activate",
			     GTK_SIGNAL_FUNC (set_theme), NULL);
	 if (g_list_length (GTK_CLIST (theme_list)->selection) != 1)
	    gtk_widget_set_sensitive (menu_item, FALSE);

	 menu_item = gtk_menu_item_new_with_label (_("Preview"));
	 gtk_menu_append (GTK_MENU (menu), menu_item);
	 gtk_widget_show (menu_item);
	 gtk_signal_connect (GTK_OBJECT(menu_item), "activate",
			     GTK_SIGNAL_FUNC (preview_dialog), NULL);

#endif /* SETSTYLE */
   
#endif /* GETSTYLE */

	 menu_item = gtk_menu_item_new_with_label (_("Remove..."));
	 gtk_menu_append (GTK_MENU (menu), menu_item);
	 gtk_widget_show (menu_item);
	 gtk_signal_connect (GTK_OBJECT(menu_item), "activate",
			     GTK_SIGNAL_FUNC (remove_theme_dialog), NULL);

#ifdef HAVE_RENAME
	 rename_theme_menu = menu_item
			   = gtk_menu_item_new_with_label (_("Rename..."));
	 gtk_menu_append (GTK_MENU (menu), menu_item);
	 gtk_widget_show (menu_item);
	 gtk_signal_connect (GTK_OBJECT(menu_item), "activate",
			     GTK_SIGNAL_FUNC (rename_theme_dialog), NULL);
#endif /* HAVE_RENAME */

	 menu_item = gtk_menu_item_new_with_label (_("Install..."));
	 gtk_menu_append (GTK_MENU (menu), menu_item);
	 gtk_widget_show (menu_item);
	 gtk_signal_connect (GTK_OBJECT(menu_item), "activate",
			     GTK_SIGNAL_FUNC (install_theme_dialog), NULL);

#ifdef PREVIEWS	 
	 preview_update_menu
	    = menu_item = gtk_menu_item_new_with_label (_("Update previews"));
	 gtk_menu_append (GTK_MENU (menu), menu_item);
	 gtk_widget_show (menu_item);
	 gtk_signal_connect (GTK_OBJECT(menu_item), "activate",
			     GTK_SIGNAL_FUNC (generate_previews), NULL);
	 gtk_widget_set_sensitive (preview_update_menu, !previews_active);

#endif /* PREVIEWS */
      }
      
      gtk_widget_set_sensitive (rename_theme_menu,
				GTK_CLIST(theme_list)->selection &&
				g_list_length (GTK_CLIST(theme_list)->selection) == 1);
      gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL, 3, event->time);
   }
   return TRUE;
}

static proplist_t *cache_theme  = NULL;
static char	  *cache_name   = NULL;

static char *
theme_image_filename (unsigned row, const char *attribute, bool_t combined)
{
   char		*themename, *pathname, *suffix, *filename;
   proplist_t	*theme;
   
   gtk_clist_get_text (GTK_CLIST (theme_list), row, 0, &themename);
   gtk_clist_get_text (GTK_CLIST (theme_list), row, 1, &pathname);
   gtk_clist_get_text (GTK_CLIST (theme_list), row, 2, &suffix);
   
   filename = g_strconcat (pathname, "/", themename, suffix, NULL);

   if (cache_theme && streq (filename, cache_name))
      theme = cache_theme;
   else
   {
      if (cache_name && streq (filename, cache_name))	/* theme was invalid */
      {
	 Free (filename);
	 return NULL;
      }
      theme = read_proplist (filename);
      if (!theme)
      {
	 char *fname2 = g_strconcat (filename, "/style", NULL);
	 
	 theme = read_proplist (fname2);
	 Free (fname2);
      }
      if (cache_theme)
	 PLRelease (cache_theme);
      if (cache_name)
	 Free (cache_name);
      cache_name  = filename;
      cache_theme = theme;
      if (!theme || !PLIsDictionary (theme))
      {
	 if (theme)
	    PLRelease (theme);
	 cache_theme = NULL;
	 
	 return NULL;
      }
   }
   
   {
      proplist_t *plattribute = PLMakeString ((char *) attribute);
      proplist_t *value;
      char       *path = NULL;
      
      value = PLGetDictionaryEntry (theme, plattribute);
      if (value && PLIsArray (value) && PLGetNumberOfElements (value) > 2)
      {
	 proplist_t *type  = PLGetArrayElement (value, 0);
	 proplist_t *file  = PLGetArrayElement (value, 1);
	 char	    *ttype = PLGetString (type);
	       
	 if ((!combined && strcaseeq (ttype + 1, "pixmap"))
	     || (combined && strcaseeq (ttype + 2, "gradient") && *ttype == 't'))
	 {
	    path = get_pixmap_path (PLGetString (file));
	    if (!path)
	    {
	       char *tmp = g_strconcat (filename, "/", PLGetString (file),
					NULL);
	       path = get_pixmap_path (tmp);
	       Free (tmp);
	    }
	 }
      }
      PLRelease (plattribute);
      return path;
   }
}

static void
theme_gradient (unsigned row, const char *attribute, GtkWidget *preview)
{
   char		*themename, *pathname, *suffix, *filename;
   proplist_t	*theme;
   int		width, height;
   
   width  = GTK_WIDGET (preview)->requisition.width;
   height = GTK_WIDGET (preview)->requisition.height;
   
   gtk_clist_get_text (GTK_CLIST (theme_list), row, 0, &themename);
   gtk_clist_get_text (GTK_CLIST (theme_list), row, 1, &pathname);
   gtk_clist_get_text (GTK_CLIST (theme_list), row, 2, &suffix);

   filename = g_strconcat (pathname, "/", themename, suffix, NULL);
   if (cache_theme && streq (filename, cache_name))
      theme = cache_theme;
   else
   {
      if (cache_name && streq (filename, cache_name))
      {
	 Free (filename);
	 make_pixmap (PKGDATADIR "/black.xpm", width, height, preview);
	 return;
      }
      theme = read_proplist (filename);
      if (!theme)
      {
	 char *fname2 = g_strconcat (filename, "/style", NULL);
      
	 theme = read_proplist (fname2);
	 Free (fname2);
      }
      if (cache_theme)
	 PLRelease (cache_theme);
      if (cache_name)
	 Free (cache_name);
      cache_name  = filename;
      cache_theme = theme;
      if (!theme || !PLIsDictionary (theme))
      {
	 cache_theme = NULL;
	 if (theme)
	    PLRelease (theme);
	 make_pixmap (PKGDATADIR "/black.xpm", width, height, preview);
	 return;
      }
   }
   
   {
      proplist_t *plattribute = PLMakeString ((char *) attribute);
      proplist_t *value       = PLGetDictionaryEntry (theme, plattribute);
      bool_t	 done = NO;
      
#if defined(PREVIEWS) && !defined(CONVERT)
   
      if (value && PLIsArray (value) && PLGetNumberOfElements (value) > 1)
      {
	 proplist_t *type  = PLGetArrayElement (value, 0);
	 char	    *ttype = PLGetString (type);
	       
	 if (strcaseeq (ttype + 1, "gradient")
	     || strcaseeq (ttype + 2, "gradient"))
	 {
	    gtype_e  gtype;
	    unsigned offset = strcaseeq (ttype + 2, "gradient") ? 1 : 0;
	    
	    if (*(ttype + offset) == 'h')
	       gtype = HGRADIENT;
	    else if (*(ttype + offset) == 'v')
	       gtype = VGRADIENT;
	    else
	       gtype = DGRADIENT;
	    if (offset && *ttype == 't')
	       make_textured_gradient (theme_image_filename (row, attribute, YES),
				       value, width, height, gtype, preview);
	    else
	       make_gradient (value, width, height, gtype, preview);
	    done = YES;
	 }
	 else if (strcaseeq (ttype, "solid"))
	 {
	    make_solid (PLGetString (PLGetArrayElement (value, 1)),
			width, height, preview);
	    done = YES;
	 }
      }
#endif /* defined(PREVIEWS) && !defined(CONVERT) */

      PLRelease (plattribute);
      if (!done)
	 make_pixmap (PKGDATADIR "/black.xpm", width, height, preview);
   }
}

static void
compute_preview (GtkWidget *progress_bar, GtkWidget *progress_label,
		 unsigned n, unsigned nelem,
		 const char *name, unsigned width, unsigned height)
{
   char *pname;
   char *path;
   
   if (!name)
      return;

   pname = preview_name (name);
   path  = get_pixmap_path (pname);

   gtk_progress_bar_update (GTK_PROGRESS_BAR (progress_bar), n / (double) nelem);
   gtk_label_set_text (GTK_LABEL (progress_label), g_basename (name));
   while (gtk_events_pending ())
      gtk_main_iteration ();

   if (!path)
   {
      DIR *dir = opendir (g_dirname (pname));

      if (!dir)				/* Make ~/.wmakerconf directory */
      {
	 char *cmd = g_strconcat ("mkdir '", g_dirname (pname), "'", NULL);
	 if (system (cmd))		/* Write failure */
	 {
	    warning (_("Can't make directory `%s'.\n"
		       "Please check `stderr' for more details."),
		     g_dirname (pname));
	    Free (cmd);
	    Free (pname);
	    return;
	 }
	 Free (cmd);
      }
      else
	 closedir (dir);

      {
#ifdef CONVERT
	 char *cmd = g_strdup_printf ("%s -comment \"\" -geometry %dx%d "
				      "'%s' ppm:'%s'", CONVERT, width, height,
				      name, pname);
#else  /* not CONVERT */
	 char *cmd = g_strdup_printf ("%s '%s' '%s' %d %d", BINDIR "/mkpreview",
				      name, pname, width, height);
#endif /* not CONVERT */
	 if (system (cmd))		/* Write failure */
	 {
	    warning (_("Can't generate preview `%s'.\n"
		       "Please check `stderr' for more details."),
		     pname);
	 }
	 Free (cmd);
      }
   }
   else
      Free (path);
   Free (pname);
}

static proplist_t
get_color (proplist_t texture)
/* setstyle.c from Window Maker utils directory */
{
    char	*str;
    proplist_t  type  = PLGetArrayElement (texture, 0);
    proplist_t  value = 0;

    if (!type)
	return NULL;

    str = PLGetString (type);
    if (streq (str, "solid"))
       value = PLGetArrayElement (texture, 1);
    else if (streq (str + 1, "gradient"))
    {
       if (PLGetNumberOfElements (texture) == 3)
       {
	  GdkColor	color1, color2;
	  char		*c1name = PLGetString (PLGetArrayElement (texture, 1));
	  char		*c2name = PLGetString (PLGetArrayElement (texture, 2));
	  /*
	   *  Compute default color = (color1 + color2) / 2
	   */
	  if (!gdk_color_parse (c1name, &color1))
	     color1.red = color1.green = color1.blue = 65535; /* white */
	  if (!gdk_color_parse (c2name, &color2))
	     color2.red = color2.green = color2.blue = 65535; /* white */

	  color1.red   = (color1.red + color2.red) >> 1;
	  color1.green = (color1.green + color2.green) >> 1;
	  color1.blue  = (color1.blue + color2.blue) >> 1;

	  {
	     char colstr [MAXSTRLEN];

	     sprintf (colstr, "#%02x%02x%02x", /* new color */
		      (unsigned) (color1.red   / 256 + 0.5),
		      (unsigned) (color1.green / 256 + 0.5),
		      (unsigned) (color1.blue  / 256 + 0.5));
	     value = PLMakeString (colstr);
	  }
       }
    }
    else if (streq (str + 2, "gradient")) 
       value = PLGetArrayElement (texture, 1);
    else if (streq (str + 1, "pixmap"))
       value = PLGetArrayElement (texture, 2);

    return value;
}
