namespace eval toolbar {
    variable toolbar

    # Define the toolbar data structure.
    array set toolbar {
	blist.cc ""
	blist.lw ""
	blist.gen ""
	stop ""
	run ""
	reset ""
	pipe ""
	history ""
	shell ""
	clear ""
	home ""
	up ""
	down ""
	end ""
	save ""
	balloon ""
	finish ""
	help ""
	about ""
	exit ""
    }
}

# toolbar::create 
#
#   Create the toolbar for the tkWorld application
#
# Args
#
#   w - The parent window
#
# Returns
#
#   The object name of the parent window

proc toolbar::create { w } {
    global tkWorld
    variable toolbar

    frame $w \
	    -class toolbar

    # These buttons invoke methods on the Command Center.
    set toolbar(blist.cc) ""

    # These button invode methods on the Log Window.
    set toolbar(blist.lw) ""

    # These are just general buttons which invoke tkWorld methods.
    set toolbar(blist.gen) ""

    # My stupid way of creating the toolbar in a somewhat automated
    # fashion using the toolbar and ballonhelp methods.
    foreach b "stop run reset pipe history shell \
	    clear home up down end \
	    save balloon finish help exit" {

	# Create the button with its image and relief.
	set toolbar($b) [button $w.$b \
		-image [toolbar::image::create $b] \
		-command toolbar::$b \
		-relief flat]
	pack $toolbar($b) \
		-side left \
		-pady 1

	# Build the button list for the Command Center and Log Window,
	# configure each button's command, define the balloon help
	# messages, and build the seperator frames where necessary.
	switch $b {
	    stop {
		append toolbar(blist.cc) "$toolbar($b) "
		balloonhelp::for $toolbar($b) \
			"ctrl-z: Stop the current command from\
			executing"
	    }
	    run {
		append toolbar(blist.cc) "$toolbar($b) "
		balloonhelp::for $toolbar($b) \
			"ctrl-x: Execute the commands in the Command\
			Center and view the output in the Log Window"
	    }
	    reset {
		append toolbar(blist.cc) "$toolbar($b) "
		balloonhelp::for $toolbar($b) \
			"ctrl-c: Reset the Command Center by deleting\
			the commands in it"
	    }
	    pipe {
		append toolbar(blist.cc) "$toolbar($b) "
		balloonhelp::for $toolbar($b) \
			"shift-|: Pipe the current command's output to\
			the input of the next command"
	    }
	    history {
		append toolbar(blist.cc) "$toolbar($b) "
		balloonhelp::for $toolbar($b) \
			"ctrl-m: View history of session commands"
	    }
	    shell {
		append toolbar(blist.cc) "$toolbar($b) "
		balloonhelp::for $toolbar($b) \
			"ctrl-s: Save the commands in the Command\
			Center to a shell script"
		pack [seperator::create $w.sep1 sunken] \
			-side left \
			-padx 2 \
			-pady 1 \
			-fill y
	    }
	    clear {
		append toolbar(blist.lw) "$toolbar($b) "
		balloonhelp::for $toolbar($b) \
			"ctrl-c Clear the text in the Log Window"
	    }
	    home {
		append toolbar(blist.lw) "$toolbar($b) "
		balloonhelp::for $toolbar($b) \
			"ctrl-home: Move to the beginning of the\
			Log Window"
	    }
	    up {
		append toolbar(blist.lw) "$toolbar($b) "
		balloonhelp::for $toolbar($b) \
			"Page up: Move up a page in the\
			Log Window"
	    }
	    down {
		append toolbar(blist.lw) "$toolbar($b) "
		balloonhelp::for $toolbar($b) \
			"Page Down: Move down a page in the\
			Log Window"
	    }
	    end {
		append toolbar(blist.lw) "$toolbar($b) "
		balloonhelp::for $toolbar($b) \
			"ctrl-end: Move to the end of the\
			Log Window"
	    }
	    save {
		append toolbar(blist.lw) "$toolbar($b) "
		balloonhelp::for $toolbar($b) \
			"ctrl-s: Save the contents of the Log Window\
			to a file for editing or later viewing"
		pack [seperator::create $w.sep2 sunken] \
			-side left \
			-padx 2 \
			-pady 1 \
			-fill y
	    }
	    balloon {
		append toolbar(blist.gen) "$toolbar($b) "
		balloonhelp::for $toolbar($b) \
			"Turn balloon help off"
	    }
	    help {
		append toolbar(blist.gen) "$toolbar($b) "
		balloonhelp::for $toolbar($b) \
			"Learn about tkWorld by viewing the HTML\
			help files"
	    }
	    finish {
		append toolbar(blist.gen) "$toolbar($b) "
		balloonhelp::for $toolbar($b) \
			"Turn the Start/Finish header display off\
			in the Log Window"
	    }
	    #about {
		#append toolbar(blist.gen) "$toolbar($b) "
		#balloonhelp::for $toolbar($b) \
			"View information about tkWorld"
	    #}
	    exit {
		append toolbar(blist.gen) "$toolbar($b) "
		balloonhelp::for $toolbar($b) \
			"Exit tkWorld"
	    }
	}
    }

    # Initialize the state of the toolbar buttons.
    toolbar::group_state cmd_center disabled
    toolbar::group_state log_window disabled
    toolbar::group_state general active

    return $w
}

# toolbar::button_state --
#
#   Method to change the state of a toolbar button.
#
# Args
#
#   b_list - List of button object names to change the state of.
#   state  - Either active or disabled.
#
# Returns
#
#   None

proc toolbar::button_state { b_list state } {
    variable toolbar

    # Catch my silly mistake of calling state active enabled and
    # other developer common mistakes that are really okay.
    switch $state {
	on -
	enable -
	enabled {
	    set state active
	}
	off -
	disable {
	    set state disabled
	}
    }

    foreach b $b_list {
	# Do not allow the tkWorld developer to disable the balloon,
	# help, about or exit buttons at any time.
	switch $b {
	    $toolbar(balloon) -
	    $toolbar(about) -
	    $toolbar(start) -
	    $toolbar(exit) -
	    $toolbar(history) -
	    $toolbar(help) {
		set state active
	    }
	}

	# Set the requested state of the button.
	$b configure -state $state

	# Make those buttons move up and down or appear greyed out.
	if {$state == "disabled"} {
	    bind $b <Enter> "$b configure -relief flat"
	    bind $b <Leave> "$b configure -relief flat"
	} else {
	    bind $b <Enter> "$b configure -relief raised"
	    bind $b <Leave> "$b configure -relief flat"
	}
    }
}

# toolbar::group_state --
#
#   Method that allows programs in the registry to change the
#   state of all the buttons associated with the Command Center and
#   Log Window in one invocation without having to use the toolbar
#   variable.
#
# Args
#
#   x     - Either cmd_center or log_window.
#   state - Either active or disabled.
#
# Returns
#
#   None.

proc toolbar::group_state { group state } {
    variable toolbar

    # Define the argument which gives the correct button list group.
    switch $group {
	command_center -
	cmd_cntr -
	cmd_center {
	    set x blist.cc
	}
	log_win -
	log_windwo -
	log_window {
	    set x blist.lw
	}
	gen -
	general {
	    set x blist.gen
	}
    }

    # Change the state of all the buttons in the group.
    toolbar::button_state $toolbar($x) $state
}

# toolbar::hide --
#
#   Method to hide the toolbar from the user.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc toolbar::hide { } {
    global tkWorld

    # Hmmm, this is easy...
    grid forget $tkWorld(toolbar) $tkWorld(seperator_1)
}

# toolbar::show --
#
#   Method to repack the toolbar and its seperator so that
#   it is visible to the user.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc toolbar::show { } {
    global tkWorld

    # First repack the toolbar.
    grid $tkWorld(toolbar) \
	    -row 1 \
	    -column 0 \
	    -columnspan 3 \
	    -sticky ew

    # Now put the seperator back.
    grid $tkWorld(seperator_1) \
	    -row 2 \
	    -column 0 \
	    -columnspan 3 \
	    -sticky ew
}

# toolbar::stop --
#
#   Method to stop the current command from exectuing.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc toolbar::stop { } {
    cmd_center::stop
}

# toolbar::run --
#
#   Method to run the commands in the CC from the toolbar.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc toolbar::run { } {
    variable toolbar

    # Invoke the Command Center method to run the commands.
    cmd_center::run

    # Activate the LW button group since there should be output
    # in the LW.
    toolbar::group_state log_window active
}

# toolbar::reset --
#
#   Method to reset the CC from the toolbar.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc toolbar::reset { } {
    cmd_center::reset

    toolbar::group_state cmd_center disabled
}

# toolbar::history --
#
#   Method to view the users session command history
#   through the toolbar.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc toolbar::history { } {
    history::create
}

# toolbar::shell --
#
#   Method to save the commands in the CC to a shell script.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc toolbar::shell { } {
    variable toolbar

    cmd_center::shell

    toolbar::button_state $toolbar(shell) disabled
}

# toolbar::pipe --
#
#   Method to put a pipe character in the CC from the toolbar.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc toolbar::pipe { } {
    cmd_center::pipe
}

# toolbar::clear  --
#
#   Method to clear the LW from the toolbar.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc toolbar::clear { } {
    log_window::clear

    toolbar::group_state log_window disabled
}

# toolbar::home  --
# toolbar::end --
#
#   Method to set the view of the Log Window to the beginning/end.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc toolbar::home { } {
    log_window::home
}

proc toolbar::end { } {
    log_window::end
}

# toolbar::up  --
# toolbar::down --
#
#   Method to move the Log Window up/down a page.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc toolbar::up { } {
    log_window::up
}

proc toolbar::down { } {
    log_window::down
}

# toolbar::finish --
#
#   Toggle the start and finish image display in the CC.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc toolbar::finish { } {
    global tkWorld

    if {$tkWorld(start_finish_display) == "1"} {
	set tkWorld(start_finish_display) 0
    } else {
	set tkWorld(start_finish_display) 1
    }
}

# toolbar::save --
#
#   Method to save the contents of the LW to a file fromt the toolbar.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc toolbar::save { } {
    variable toolbar

    log_window::save

    toolbar::button_state $toolbar(save) disabled
}

# toolbar::balloon --
#
#   Method to on/off toggle the balloon help from the toolbar.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc toolbar::balloon { } {
    # Toggle the balloon help on/off.
    if $balloonhelp::balloonhelp(active) {
	set balloonhelp::balloonhelp(active) 0
    } else {
	set balloonhelp::balloonhelp(active) 1
    }
}

# toolbar::help  --
#
#   Method to invoke the help browser.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc toolbar::help { } {
    global tkWorld

    help::create
    #help::create "index.html" "tkWorld Help"
}

# toolbar::about --
#
#   Method to invoke the About information dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc toolbar::about { } {
    about::create
}

# toolbar::exit  --
#
#   Method to exit tkWorld from the toolbar.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc toolbar::exit { } {
    global tkWorld

    # First destroy all of the existing children to the
    # parent application
    foreach w [winfo children .] {
	destroy $w
    }

    # Let the user know tkWorld is still working.
    cleanup::create

    # First cleanup all of the files in the tmp directory.
    foreach f [glob -nocomplain [file join \
	    $tkWorld(script_dir) "tkWorld.$tkWorld(user).*"]] {
	catch {file delete $f}
    }

    # Kill the information dialog
    cleanup::close

    # Now kill the application completely.
    destroy .
}

#---------------------------------------------------------------------

namespace eval toolbar::image {
    # No private variables
}

# toolbar::image::create --
#
#   Method to create an icon image in the toolbar.
#
# Args
#
#   f - Base name of the image file (i.e. run where the file
#       filename is run.ic.gif)
#
# Returns
#
#   The Tk name of the image (i.e. image4)

proc toolbar::image::create { f } {
    global tkWorld

    # Create the image with the file name as the array name.
    return [image create photo \
	    -file [file join $tkWorld(image_dir) toolbar $f.gif] \
	    -format gif]
}

