namespace eval history {
    variable history

    # Define the history data structure.  Note commands will be denoted
    # by integer indices.
    array set history {
	index 0
	dialog ""
	listbox ""
    }
}

proc history::create { } {
    global tkWorld
    variable history

    # Build the toplevel dialog without grabbing focus if it
    # does not exists.
    if [winfo exists .history] {
	switch -- [wm state $history(dialog)] {
	    normal {
		raise $history(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $history(dialog)
	    }
	}
	focus .history
	return
    }

    set w [toplevel .history \
	    -borderwidth 1 \
	    -relief flat]
    set history(dialog) $w

    # Note the usage of history::history(dialog) because the callback is
    # executed in the global namespace and needs to be reminded to
    # use the history namespace.
    after idle {
	update idletasks
    
	set xmax [winfo screenwidth .history]
	set ymax [winfo screenheight .history]
    
	set x0 [expr ($xmax - [winfo reqwidth $history::history(dialog)])/2]
	set y0 [expr ($ymax - [winfo reqheight $history::history(dialog)])/2]
	
	wm geometry .history "+$x0+$y0"
    }

    set f1 [scroll_listbox::create $w.f1 50]
    #$f1 configure -relief raised -borderwidth 2
    set history(listbox) [scroll_listbox::interior $f1]

    # Create a seperator from the bottom buttons
    set f6 [seperator::create $w.f6]

    # The bottom frame has the okay, reset, help and
    # cancel buttons
    set f7 [frame $w.f7]

    foreach b "Ok Reset Save Load Help Close" {
	set t $b
	set b [string tolower $b]
	button $f7.$b \
		-text $t \
		-width 5 \
		-command history::$b
	switch $b {
	    ok {
		balloonhelp::for $f7.$b \
			"Insert the command into the Command Center"
	    }
	    reset {
		balloonhelp::for $f7.$b \
			"Reset the values of the radio and checkbuttons"
	    }
	    up {
		balloonhelp::for $f7.$b \
			"Move up a page in the Command History"
	    }
	    down {
		balloonhelp::for $f7.$b \
			"Move down a page in the Command History"
	    }
	    save {
		balloonhelp::for $f7.$b \
			"Save the current history to a file"
	    }
	    load {
		balloonhelp::for $f7.$b \
			"Load a tkWorld history file"
	    }
	    help {
		balloonhelp::for $f7.$b \
			"View the online help for using the Command History"
	    }
	    close {
		balloonhelp::for $f7.$b \
			"Close the command dialog"
	    }
	}
    }
    grid $f7.ok $f7.reset $f7.save $f7.load $f7.help $f7.close \
	    -sticky ew \
	    -padx 5 \
	    -pady 1

    pack $f1 $f6 $f7 \
	    -side top \
	    -fill x \
	    -padx 0 \
	    -pady 2

    wm title .history "Command History"
    wm iconbitmap .history \
	    @[file join $tkWorld(image_dir) history.ic.xbm]
    wm resizable .history 0 0

    # Load the current session commands if they exist.  The index
    # for current commands is an integer number. This is done here
    # if the user previously Closed the history dialog box and now
    # they are re-opening it.
    foreach i [lsort [array names history]] {
	if [regexp {^[0-9]*$} $i] {
	    $history(listbox) insert end $history($i)
	}
    }

    # Allow the user to double click on the command they want to
    # insert into the CC.
    bind $history(listbox) <Double-1> history::ok
}

# history::add --
#
#   Method to add a command to the history dialog.
#
# Args
#
#   cmd - The command to insert into the history dialog.
#
# Returns
#
#   None.

proc history::add { cmd } {
    variable history

    # Insert the supplied command in the listbox if it is open
    # and move to the end of the list.
    if [winfo exists $history(listbox)] {
	$history(listbox) insert end $cmd
	$history(listbox) see end
	bind $history(listbox) <Double-1> {history::ok}
    }

    # Store the supplied command in the history array for
    # later use if the user closes the history window and opens it
    # again in the same tkWorld session.
    set history([incr history(index)]) $cmd

}

# history::up --
# history::down --
#
#   Method to move up/down a page in the Command History. 
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc history::up { } {
    variable history

    $history(listbox) yview scroll +1 pages
}

proc history::down { } {
    variable history

    $history(listbox) yview scroll -1 pages
}

# history::save --
#
#   Method to save the current session command history to a file.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc history::save { } {
    global tkWorld
    variable history

    # For the sake of variety, define a few file types.
    set typelist {
	{"All Files" {*}}
	{"Text File" {.txt}}
    }

    # Get the filename from the user at startup
    set file_name [tk_getSaveFile \
	    -defaultextension "" \
	    -filetypes $typelist \
	    -initialdir $tkWorld(user_home) \
	    -initialfile ".tkWorld_history" \
	    -title "Save History File"]

    # Allow the user to cancel the operation.
    if ![string length $file_name] {
	return
    }

    # Open the file or display the error message.
    if [catch {open $file_name "w"} fid] {
	bell; tk_messageBox \
		-type ok \
		-message $fid \
		-icon info \
		-title "Save History Error"
	return
    }

    # Print out the header of the tkWorld history file.
    puts $fid "#++"
    puts $fid "#"
    puts $fid "# $file_name"
    puts $fid "#"
    puts $fid "#   File which contains session command history"
    puts $fid "#"
    puts $fid "# Version"
    puts $fid "#"
    puts $fid "#   tkWorld-$tkWorld(version)"
    puts $fid "#"
    puts $fid "#"
    puts $fid "# Notes"
    puts $fid "#"
    puts $fid "#   THIS IS A tkWorld GENERATED FILE.  DO NOT MODIFY THIS"
    puts $fid "#   FILE UNLESS YOU KNOW WHAT YOU ARE DOING."
    puts $fid "#"
    puts $fid "#------------------------------------------------------------"
    puts $fid "# Name       Date       Comments"
    puts $fid "# ---------- ---------- -------------------------------------"
    puts $fid "# tkWorld    [clock format [clock seconds] -format %Y.%m.%d]\
	    Created."
    puts $fid "#"
    puts $fid "#--"

    # Insert the history start tag.
    puts $fid "<TKWORLD_HISTORY>"

    # Now put the commands in the history file.
    foreach i [lsort [array names history]] {
	# Only insert items from the array with integer indices.
	if [regexp {^[0-9]*$} $i] {
	    puts $fid $history($i)
	}
    }
    
    # Print out the ending tag.
    puts $fid "</TKWORLD_HISTORY>"

    # Execute the close command in the global namespace,
    # not history::close
    ::close $fid
}    

# history::load --
#
#   Method to load a tkWorld history file.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc history::load { } {
    global tkWorld
    variable history

    # For the sake of variety, define a few file types.
    set typelist {
	{"All Files" {*}}
	{"Text File" {.txt}}
    }

    # Get the history filename to load.
    set file_name [tk_getOpenFile \
	    -filetypes $typelist \
	    -initialdir $tkWorld(user_home) \
	    -initialfile ".tkWorld_history" \
	    -title "History File Select"]

    # Allow the user to cancel the operation.
    if ![string length $file_name] {
	return
    }

    # Open the file or display the error message.
    if [catch {open $file_name "r"} fid] {
	bell; tk_messageBox \
		-type ok \
		-message $fid \
		-icon info \
		-title "Load History Error"
	return
    }

    # Parse the history file to ignore the header and start/end tags.
    set tag_cnt 0
    foreach line [split [read $fid] \n] {
	switch -regexp $line {
	    "^\#.*$" {
		# Skip comment lines.
		continue
	    }
	    "^</?TKWORLD_HISTORY>.*$" {
		incr tag_cnt
	    }
	    default {
		# If we found the first tag, then add the lines.
		if {$tag_cnt == 1} {
		    history::add $line
		}
	    }
	}
    }

    # Move to the last command in the history.
    $history(listbox) see end

    # Display an error message when the file is not a proper tkWorld
    # history file.
    set error_code [expr 2 - $tag_cnt]
    if $error_code {
	bell; tk_messageBox \
		-type ok \
		-message "Not a tkWorld history file.  Missing 1 or both\
		of the history tags <TKWORLD_HISTORY> or\
		</TKWORLD_HISTORY>" \
		-icon info \
		-title "Load History File Error"
	history::reset
    }
    
    # Close the file id in the global space and not in the
    # history::close namespace.
    ::close $fid
}

# history::get --
#
#   Method to put the command at index in the CC.
#
# Args
#
#   idx - The index number of the command to retrieve from history.
#
# Returns
#
#   None.

proc history::get { idx } {
    global tkWorld
    variable history

    # First open the history dialog if it is not already open
    # since a user might use the ctrl-p binding in the CC before the
    # dialog is open.
    history::create
    focus $tkWorld(cmd_center)

    # Next get the selection at the supplied index
    set cmd [$history(listbox) get $idx]

    # Now insert the select command in the CC.
    $tkWorld(cmd_center) insert insert $cmd

    # Activate the buttons in the toolbar for the command center.
    toolbar::group_state cmd_center active
    toolbar::button_state $toolbar::toolbar(stop) disabled
}

# history::ok --
#
#   Method to put the selected command in the CC.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc history::ok { } {
    global tkWorld
    variable history

    # First get the current selection
    set cmd [$history(listbox) get [$history(listbox) curselection]]

    # Now insert the select command in the CC.
    $tkWorld(cmd_center) insert insert $cmd

    # Activate the buttons in the toolbar for the command center.
    toolbar::group_state cmd_center active
    toolbar::button_state $toolbar::toolbar(stop) disabled
}

# history::reset --
#
#   Method to reset the listbox in the History Dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc history::reset { } {
    variable history

    # Remove the commands from the listbox.
    if [winfo exists $history(listbox)] {
	$history(listbox) delete 0 end
    }

    # Now remove them from the array.
    foreach i [array names history] {
	if [regexp {^[0-9]*$} $i] {
	    unset history($i)
	}
    }

    set history(index) 0
}

# history::help --
#
#   Method to invoke the History Command Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc history::help { } {
    global tkWorld

    help::create "history/index.html" "Command History Help"
}

# history::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc history::close { } {
    variable history
    
    balloonhelp::cancel
    destroy $history(dialog)
}
