namespace eval cmd_center {
    variable cmd_center

    array set cmd_center {
	script ""
	output ""
	fid ""
	script ""
	index 0
    }
}

# cmd_center::run --
#
#   Method which controls the execution of the commands in the CC. Note
#   this engine is not cross platform.  It depends on the use of the
#   Bourne Shell for command interpretation, or any substitute that sh
#   might point to (ie sh -> /bin/ksh).
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc cmd_center::run { } {
    global tkWorld
    variable cmd_center

    # Build the name of the tmp file.
    set cmd_center(script) [file join $tkWorld(script_dir) \
	    tkWorld.$tkWorld(user).[clock seconds]]
    
    # Open the tmp file.
    set fhandle [open $cmd_center(script) w]

    puts -nonewline $fhandle "cd $tkWorld(working_dir); "

    # Export the command in the CC to the tmp file.
    set cmd [string trimright [$tkWorld(cmd_center) get 1.0 end] \n]
    puts $fhandle $cmd
    close $fhandle

    # Test for an empty command.
    regsub -all "( |\t|\n)*" $cmd "" null
    if ![string length $null] {
	stderr::log 0001
	return
    }

    # Add the current command to the session history in the bare from.
    history::add $cmd

    # Insert the Start image so the user knows where the
    # command output started.
    if $tkWorld(start_finish_display) {
	set l [label $tkWorld(log_window).cmd_center[incr cmd_center(index)] \
		-text "[string range $cmd 0 29]..." \
		-anchor w \
		-foreground #ffffff \
		-background #006400 \
		-width [$tkWorld(log_window) cget -width]]

	# Put the label in the LW.
	$tkWorld(log_window) window create end \
		-window $l \
		-padx 0 \
		-pady 10

	$tkWorld(log_window) insert end \n
    }

    # Detect if the command is being executed in the background.
    # If it is, run the command but do not log the output.  Otherwise
    # proceed with default processing.
    if $cmd_center(bgproc) {
	if [catch {open "|$tkWorld(working_shell) \
		$cmd_center(script) &"} errmsg] {
	    stderr::log 0002 $errmsg
	} else {
	    # Successfully running program in the background
	    cmd_center::reset
	    set cmd_center(bgproc) 0
	    return
	}
    }

    # Execute the commands in the CC.  Use |& cat to log both
    # the output and the error message to the LW.
    if [catch {open "|$tkWorld(working_shell) \
	    $cmd_center(script) |& cat"} cmd_center(fid)] {
	stderr::log 0002 $cmd_center(fid)
    } else {
	# Enable the LW toolbar group.
	toolbar::group_state log_window active

	# flip-flop the Run/Stop buttons.
	toolbar::button_state $toolbar::toolbar(run) disabled
	toolbar::button_state $toolbar::toolbar(stop) active

	# Define the variable that stores the output.
	set cmd_center(output) ""

	# Start buffering the command output to the LW.
	fconfigure $cmd_center(fid) -buffering line
	fileevent $cmd_center(fid) readable cmd_center::log

	# Define the watch cursor for the user everywhere but
	# on the toolbar.
	foreach w "reg_listbox cmd_center log_window" {
	    $tkWorld($w) configure -cursor watch
	}
    }
}

# cmd_center::log --
#
#   Method to log command output to the LW.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc cmd_center::log { } {
    global tkWorld
    variable cmd_center

    # Stop when the pipe is broken, otherwise just put the output
    # in the LW and log it in the CC output variable.
    if [eof $cmd_center(fid)] {
	cmd_center::stop
    } else {
	gets $cmd_center(fid) line
	$tkWorld(log_window) insert end $line\n
	$tkWorld(log_window) see end
	append cmd_center(output) $line
	update idletasks
    }
}

# cmd_center::stop --
#
#   Method to end the command execution.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc cmd_center::stop { } {
    global tkWorld
    variable cmd_center

    # Stop the current process.
    catch {close $cmd_center(fid)}

    # Test for null output and send the user a message.  Also free
    # up the memory associated with output.
    regsub -all "( \t\n)*" $cmd_center(output) "" null
    if ![string length $null] {
	stderr::log 1001
	unset cmd_center(output)
    }

    # Put the cursor back to normal.
    foreach w "reg_listbox cmd_center log_window" {
	$tkWorld($w) configure -cursor top_left_arrow
    }

    # Flip-flop the state of the Run/Stop buttons.
    toolbar::button_state $toolbar::toolbar(run) active
    toolbar::button_state $toolbar::toolbar(stop) disabled

    # Insert the Finished image so the user knows where the
    # command output ended.
    if $tkWorld(start_finish_display) {
	set l [label $tkWorld(log_window).cmd_center[incr cmd_center(index)] \
		-text "[clock format [clock seconds]]" \
		-anchor w \
		-foreground #ffffff \
		-background #990000 \
		-width [$tkWorld(log_window) cget -width]]

	# Put the label in the LW.
	$tkWorld(log_window) window create end \
		-window $l \
		-padx 0 \
		-pady 10

	$tkWorld(log_window) insert end \n
    }

    # Now move the LW all the way down.
    $tkWorld(log_window) see end

    # Make it so the user can just click the command from the
    # history if they want it and not have to hit refresh if they
    # want to type in a new command.  This based on the user starting
    # a new command more often than modifying the last command.
    cmd_center::reset

    # Now remove the script file.
    if [catch {file delete $cmd_center(script)} err] {
	stderr::log 0004 "Error removing temporary script file\
		$tkWorld(script)...\n$err"
    }
}
    
# cmd_center::reset --
#
#   Method to delete the commands(text) in the Command Center.
#
# Args
#
#   None
#
# Returns
#
#   None

proc cmd_center::reset { } {
    global tkWorld

    # Get rid of all the text in the Command Center.
    $tkWorld(cmd_center) delete 1.0 end
}

# cmd_center::shell --
#
#   Method to save the commands in the Command Center to a shell
#   script file.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc cmd_center::shell { } {
    global tkWorld

    # For the sake of variety, define a few default shell script types.
    set typelist {
	{"Shell Script" {.sh}}
	{"Korn Shell Script" {.ksh}}
	{"Tcl Script" {.tcl}}
	{"Perl Script" {.pl}}
	{"Sed Script" {.sed}}
	{"AWK Script" {.awk}}
	{"All Files" {*}}
    }
    
    # Get the filename from the user at startup
    set file_name [tk_getSaveFile \
	    -defaultextension ".sh" \
	    -filetypes $typelist \
	    -initialdir $tkWorld(user_home) \
	    -initialfile "script1" \
	    -title "Save Shell Script"]

    # Allow the user to cancel the operation.
    if ![string length $file_name] {
	return
    }

    # Open the file or display the error message.
    if [catch {open $file_name "w"} fid] {
	bell; tk_messageBox \
		-type ok \
		-message $fid \
		-icon info \
		-title "Error Shell Script Save"
	return
    }

    # Define the script type from the file extension
    switch -- [file extension $file_name] {
	.ksh {
	    set shell ksh
	}
	.tcl {
	    set shell tclsh
	}
	.pl {
	    set shell perl
	}
	.sed {
	    set shell sed
	}
	.awk {
	    set shell awk
	}
	default {
	    set shell sh
	}
    }

    # Find the path of the shell
    if [catch {exec which $shell} path] {
	set path "/bin/sh"
    }

    # Print out the shell script header which everyone will love!
    puts $fid "#!$path"
    puts $fid "#"
    puts $fid "# [file tail $file_name]"
    puts $fid "#"
    puts $fid "#   Shell script generated by tkWorld-$tkWorld(version)"
    puts $fid "#"
    puts $fid "# Version"
    puts $fid "#"
    puts $fid "#   1.0"
    puts $fid "#"
    puts $fid "#------------------------------------------------------------"
    puts $fid "#\n"

    # Now dump the file contents without using $t dump becuase this
    # command keeps all of the special character intact which is not
    # useful in a shell script. Below is an example:
    #
    # User first types: wc -l tkworld.tcl
    # User changes to: wc -l tkWorld.tcl
    #
    # File contents:
    #
    #        wc -l tkW
    #        orld.tcl
    #
    # Not quite what we want so we use $t get instead.

    foreach line [split [$tkWorld(cmd_center) get 1.0 end] \n] {
	    puts $fid $line
    }

    close $fid
}

# cmd_center::pipe --
#
#   Method to insert the pipe character at the end of the current
#   command line.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc cmd_center::pipe { } {
    global tkWorld

    # Insert the pipe character.
    $tkWorld(cmd_center) insert insert "|"
}

# cmd_center::resize --
#
#   Method to configure the width or height of the CC.
#
# Args
#
#   type - Either the value width or height.
#   size - Numeric value for the type.
#
# Returns
#
#   None.

proc cmd_center::resize { type size } {
    global tkWorld

    # Either configure the height or the width, but nothing else
    switch $type {
	height -
	width  {
	    $tkWorld(cmd_center) configure -$type $size
	}
    }
}
