namespace eval find {
    variable find

    # Define the find array structure so that all variables are
    # defined for the callbacks in the radiobuttons and checkbuttons.
    array set find {
	dialog ""
	entry.path ""
	entry.name ""
	entry.regex ""
	path ""
	name ""
	regex ""
	case_insensisitive ""
	time_unit ""
	and0 ""
	and1 ""
	and2 ""
	plus0 ""
	plus1 ""
	plus2 ""
	plus3 ""
	entry.accessed ""
	accessed ""
	entry.changed ""
	changed ""
	entry.modified ""
	modified ""
	entry.user ""
	user ""
	entry.group ""
	group ""
	type ""
	entry.size ""
	size ""
	size.unit ""
	all_perm_read "0"
	all_perm_write "0"
	all_perm_execute "0"
	user_perm_read "0"
	user_perm_write "0"
	user_perm_execute "0"
	group_perm_read "0"
	group_perm_write "0"
	group_perm_execute "0"
	other_perm_read "0"
	other_perm_write "0"
	other_perm_execute "0"
	entry.command ""
	command ""
	exec.cmd ""
	entry.moveto ""
	moveto ""
	entry.copyto ""
	copyto ""
	list.reset ""
	list.clear ""
    }
}

# find::create --
#
#   Method to create the dialog box for the find command.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to run your command, then reopen
#   it to modify it.  By not making this a modal dialog, we do not
#   have to implement any last command saving characteristics since
#   the user can just leave the dialog open.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc find::create { } {
    global tkWorld
    variable find

    # Build the toplevel dialog without grabbing focus if it
    # does not exists.
    if [winfo exists $find(dialog)] {
	switch -- [wm state $find(dialog)] {
	    normal {
		raise $find(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $find(dialog)
	    }
	}
	focus $find(dialog)
	return
    } else {
	set find(dialog) [dialog::create .find Find]
    }

    # The first tab has the start path, name, and regexp options.
    set tab1 [tabnotebook::page [dialog::interior $find(dialog)] "Name"]

    set f1 [frame $tab1.f1 \
	    -class TabnotebookFrame]
    label $f1.label_path \
	    -text "Start Path" \
	    -width 10 \
	    -anchor e
    set find(entry.path) [entry $f1.entry_path \
	    -width 35 \
	    -textvariable find::find(path)]
    label $f1.label_name \
	    -text "Name" \
	    -anchor e \
	    -width 10
    set find(entry.name) [entry $f1.entry_name \
	    -width 20 \
	    -textvariable find::find(name)]
    label $f1.label_regex \
	    -text "Regex" \
	    -anchor e \
	    -width 10
    set find(entry.regex) [entry $f1.entry_regex \
	    -width 20 \
	    -textvariable find::find(regex)]
    checkbutton $f1.case_insensitive \
	    -text "Ignore Case" \
	    -variable find::find(case_insensitive) \
	    -onvalue "1" \
	    -offvalue "0"
    grid $f1.label_path $f1.entry_path - \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f1.label_name $f1.entry_name $f1.case_insensitive \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f1.label_regex $f1.entry_regex ^ \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    focus $find(entry.path)

    # Define the lists for the comparison operators.
    set list_and {"and" "or" "not" ""}
    set list_gt {"greater than" "less than" "equal to" ""}

    # Define the default unit of time.
    set find(time_unit) day

    # The Second tab has the time comparison information.
    set tab2 [tabnotebook::page [dialog::interior $find(dialog)] "Time"]

    # The top frame defines the unit of time (days or minutes).
    set f2_top [frame $tab2.top \
	    -class TabnotebookFrame]
    label $f2_top.label_time \
	    -text "File time measured in:" \
	    -width 18 \
	    -anchor w
    radiobutton $f2_top.minute \
	    -text "Minutes" \
	    -value min \
	    -variable find::find(time_unit)
    radiobutton $f2_top.day \
	    -text "Days" \
	    -value day \
	    -variable find::find(time_unit)
    grid $f2_top.label_time $f2_top.minute $f2_top.day \
	    -padx 2 \
	    -pady 2

    # Define the accessed, changed, and modified relationships.
    set f2_bot [frame $tab2.bot \
	    -class TabnotebookFrame]
    set and0 [ddlistbox::create $f2_bot find::find(and0) $list_and 3]
    label $f2_bot.label_accessed \
	    -text "accessed" \
	    -anchor e \
	    -width 8
    set plus0 [ddlistbox::create $f2_bot find::find(plus0) $list_gt 10]
    set find(entry.accessed) [entry $f2_bot.entry_accessed \
	    -width 5 \
	    -textvariable find::find(accessed)]

    set and1 [ddlistbox::create $f2_bot find::find(and1) $list_and 3]
    label $f2_bot.label_changed \
	    -text "changed" \
	    -anchor e \
	    -width 8
    set plus1 [ddlistbox::create $f2_bot find::find(plus1) $list_gt 10]
    set find(entry.changed) [entry $f2_bot.entry_changed \
	    -width 5 \
	    -textvariable find::find(changed)]

    set and2 [ddlistbox::create $f2_bot find::find(and2) $list_and 3]
    label $f2_bot.label_modified \
	    -text "modified" \
	    -anchor e \
	    -width 8
    set plus2 [ddlistbox::create $f2_bot find::find(plus2) $list_gt 10]
    set find(entry.modified) [entry $f2_bot.entry_modified \
	    -width 5 \
	    -textvariable find::find(modified)]
    grid $and0 $f2_bot.label_accessed $plus0 $f2_bot.entry_accessed \
	    -padx 2 \
	    -pady 2\
	    -sticky w
    grid $and1 $f2_bot.label_changed $plus1 $f2_bot.entry_changed \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $and2 $f2_bot.label_modified $plus2 $f2_bot.entry_modified \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    # Define the file type list
    set list_filetype {Block Character Directory File Link Socket ""}
    set list_sizeop {"512 Bytes" Bytes Kilobytes ""}

    # The third tab has userid/group, type, and size information.
    set tab3 [tabnotebook::page [dialog::interior $find(dialog)] "File"]

    set f3_top [frame $tab3.f3_top \
	    -class TabnotebookFrame]
    label $f3_top.label_user \
	    -text "Userid" \
	    -width 10 \
	    -anchor e
    set find(entry.user) [entry $f3_top.entry_user \
	    -width 20 \
	    -textvariable find::find(user)]
    label $f3_top.label_group \
	    -text "Group" \
	    -width 10 \
	    -anchor e
    set find(entry.group) [entry $f3_top.entry_group \
	    -width 20 \
	    -textvariable find::find(group)]
    grid $f3_top.label_user $f3_top.entry_user - \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f3_top.label_group $f3_top.entry_group - \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    set f3_bot [frame $tab3.bot \
	    -class TabnotebookFrame]

    label $f3_bot.label_type \
	    -text "Type" \
	    -width 5 \
	    -anchor e
    set type0 [ddlistbox::create $f3_bot find::find(type) \
	    $list_filetype 10]
    label $f3_bot.label_size \
	    -text "Size" \
	    -width 5 \
	    -anchor e
    set plus3 [ddlistbox::create $f3_bot find::find(plus3) $list_gt 10]
    set find(entry.size) [entry $f3_bot.entry_size \
	    -width 8 \
	    -textvariable find::find(size)]
    set size0 [ddlistbox::create $f3_bot find::find(size.unit) \
	    $list_sizeop 8]
    grid $f3_bot.label_type $type0 - - \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f3_bot.label_size $plus3 $f3_bot.entry_size $size0 \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    # The 5th tab has file permission information.
    set tab5 [tabnotebook::page [dialog::interior $find(dialog)] "Perm"]
    set f5 [frame $tab5.f5 \
	    -class TabnotebookFrame]

    set col 0
    foreach t "All User Group Other" {
	set x [string tolower $t]
	set cb($x) [checkbox::create $f5.$x "$t"]
	grid $f5.$x \
		-row 0 \
		-column $col \
		-sticky w \
		-padx 5 \
		-pady 0
	incr col
    }
    foreach {a b} {Read 4 Write 2 Execute 1} {
	set t [string tolower $a]
	checkbox::add $cb(all) $a $b "0" find::find(all_perm_$t)
	checkbox::add $cb(user) $a $b "0" find::find(user_perm_$t)
	checkbox::add $cb(group) $a $b "0" find::find(group_perm_$t)
	checkbox::add $cb(other) $a $b "0" find::find(other_perm_$t)
    }
    unset a b col cb x t

    # The command frame.
    set tab6 [tabnotebook::page [dialog::interior $find(dialog)] "Command"]

    set f6 [frame $tab6.f6 \
	    -class TabnotebookFrame]
    radiobutton $f6.command \
	    -text "Execute Command" \
	    -variable find::find(exec.cmd) \
	    -value "command"
    set find(entry.command) [entry $f6.entry_command \
	    -width 30 \
	    -background #d9d9d9 \
	    -state disabled \
	    -textvariable find::find(command)]
    radiobutton $f6.touch \
	    -text "Touch Files" \
	    -variable find::find(exec.cmd) \
	    -value "touch"
    radiobutton $f6.remove \
	    -text "Remove Files" \
	    -variable find::find(exec.cmd) \
	    -value "rm"
    radiobutton $f6.move \
	    -text "Move Files to" \
	    -variable find::find(exec.cmd) \
	    -value "mv"
    set find(entry.moveto) [entry $f6.entry_moveto \
	    -width 30 \
	    -background #d9d9d9 \
	    -state disabled \
	    -textvariable find::find(moveto)]
    radiobutton $f6.copy \
	    -text "Copy Files to" \
	    -variable find::find(exec.cmd) \
	    -value "cp"
    set find(entry.copyto) [entry $f6.entry_copyto \
	    -width 30 \
	    -background #d9d9d9 \
	    -state disabled \
	    -textvariable find::find(copyto)]
    radiobutton $f6.off \
	    -text "Off" \
	    -variable find::find(exec.cmd) \
	    -value ""
    grid $f6.command $f6.entry_command \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f6.touch - \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f6.remove - \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f6.move $f6.entry_moveto \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f6.copy $f6.entry_copyto \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f6.off - \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    # Create all of the tabs at once.
    pack $f1 $f2_top $f2_bot $f3_top $f3_bot $f5 $f6 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    bind $f6.command <ButtonPress-1> {find::toggle_cmd_entry command}
    bind $f6.touch <ButtonPress-1> {find::toggle_cmd_entry touch}
    bind $f6.remove <ButtonPress-1> {find::toggle_cmd_entry remove}
    bind $f6.move <ButtonPress-1> {find::toggle_cmd_entry move}
    bind $f6.copy <ButtonPress-1> {find::toggle_cmd_entry copy}
    bind $f6.off <ButtonPress-1> {find::toggle_cmd_entry off}

    # Define the lists for the reset and clear methods.
    set find(list.reset) "case_insensitive all_perm_read all_perm_write \
	    all_perm_execute user_perm_read user_perm_write \
	    user_perm_execute group_perm_read group_perm_write \
	    group_perm_execute other_perm_read other_perm_write \
	    other_perm_execute and0 and1 and2 plus0 plus1 plus2 \
	    plus3 size.unit type exec.cmd"
    set find(list.clear) "path name regex accessed changed modified \
	    user group size command moveto copyto"
}

# find::ok --
#
#   Method to insert the command the user has created into the CC.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc find::ok { } {
    global tkWorld
    variable find

    # Build the command line argument.
    set cmd_arg ""
    foreach x "regex name time_unit user group type size" {
	switch -- $x {
	    regex -
	    name {
		if [string length $find($x)] {
		    if $find(case_insensitive) {
			append cmd_arg "-i$x \"$find($x)\" "
		    } else {
			append cmd_arg "-$x \"$find($x)\" "
		    }
		}
	    }
	    time_unit {
		# Okay this is the cool part for parsing the find command.
		# First figure out if the time type is days or minutes.
		# Then parse each accessed, changed, and modified value.
		# If their corresponding operator is not null, then insert
		# it and the not null value.
		set t time
		if {$find(time_unit) == "min"} {
		    set t min
		}
		if [string length $find(accessed)] {
		    if [string length $find(and0)] {
			append cmd_arg "-$find(and0) "
		    }
		    set plus0 [find::parse_plusop $find(plus0)]
		    append cmd_arg "-a$t $plus0$find(accessed) "
		}
		if [string length $find(changed)] {
		    if [string length $find(and1)] {
			append cmd_arg "-$find(and1) "
		    }
		    set plus1 [find::parse_plusop $find(plus1)]
		    append cmd_arg "-c$t $plus1$find(changed) "
		}
		if [string length $find(modified)] {
		    if [string length $find(and2)] {
			append cmd_arg "-$find(and2) "
		    }
		    set plus2 [find::parse_plusop $find(plus2)]
		    append cmd_arg "-m$t $plus2$find(modified) "
		}
	    }
	    user {
		# Define the user name.
		if [string length $find($x)] {
		    append cmd_arg "-user $find($x) "
		}
	    }
	    group {
		# Define the group name.
		if [string length $find($x)] {
		    append cmd_arg "-group $find($x) "
		}
	    }
	    type {
		# Define the type option.
		if [string length $find($x)] {
		    set type [find::parse_typeop $find($x)]
		    append cmd_arg "-type $type "
		    unset type
		}
	    }
	    size {
		# Define the size option and its units.
		if [string length $find($x)] {
		    set plus3 [find::parse_plusop $find(plus3)]
		    set size [find::parse_sizeop $find(size.unit)]
		    append cmd_arg "-size $plus3$find($x)$size "
		    unset size
		}
	    }
	    default {
		if [string length $find($x)] {
		    append cmd_arg "-$x \"$find($x)\" "
		}
	    }
	}
    }

    # Now define the permission stuff if they are turned on.
    foreach x "all user group other" {
	set total($x) [expr $find($x\_perm_read) + \
		$find($x\_perm_write) + \
		$find($x\_perm_execute)]
    }
    if {$total(all) > 0} {
	append cmd_arg "-perm \"$total(all)$total(all)$total(all)\" "
    } elseif {$total(user) > 0 || $total(group) > 0 || $total(other) > 0} {
	append cmd_arg "-perm \"$total(user)$total(group)$total(other)\" "
    }
    unset total

    # Parse the command tab.
    if [string length $find(exec.cmd)] {
	switch $find(exec.cmd) {
	    command {
		append cmd_arg "-exec $find(command) \"\{\}\" \\\;"
	    }
	    touch {
		append cmd_arg "-exec touch \"\{\}\" \\\;"
	    }
	    rm {
		append cmd_arg "-exec rm \"\{\}\" \\\;"
	    }
	    mv {
		append cmd_arg "-exec mv \"\{\}\" $find(moveto) \\\;"
	    }
	    cp {
		append cmd_arg "-exec cp \"\{\}\" $find(copyto) \\\;"
	    }
	}
    }

    # Insert the Tcl command list in the Command Center.
    if [string length $find(path)] {
	$tkWorld(cmd_center) insert insert \
		[subst {find $find(path) $cmd_arg}]
    } else {
	$tkWorld(cmd_center) insert insert \
		[subst {find . $cmd_arg}]
    }

    # Activate the buttons in the toolbar for the command center.
    toolbar::group_state cmd_center active
    toolbar::button_state $toolbar::toolbar(stop) disabled
}

# find::parse_plusop --
#
#   Method to eval the plus ddlistbox value.
#
# Args
#
#   plus - Text value of either "greater than", "less than",
#          and "equal to".
#
# Returns
#
#   Either the string "+", "-", or null "".

proc find::parse_plusop { plus } {
    
    switch $plus {
	"greater than" {
	    return "+"
	}
	"less than" {
	    return "-"
	}
	default {
	    return ""
	}
    }
}

# find::parse_typeop --
#
#   Method to eval the type ddlistbox value.
#
# Args
#
#   size_unit - Text value of either "Block", "Character", "Directory",
#               "File", "Link", and "Socket"
#
# Returns
#
#   Either the string "b", "c", "d", "f","l","s", or null "".

proc find::parse_typeop { type } {
    
    switch $type {
	"Block" {
	    return "b"
	}
	"Character" {
	    return "c"
	}
	"Directory" {
	    return "d"
	}
	"File" {
	    return "f"
	}
	"Link" {
	    return "l"
	}
	"Socket" {
	    return "s"
	}
	default {
	    return ""
	}
    }
}

# find::parse_sizeop --
#
#   Method to eval the size unit ddlistbox value.
#
# Args
#
#   size - Text value of either "512 Bytes", "Bytes", "Kilobytes"
#               or null.
#
# Returns
#
#   Either the string "b", "c", "k", or null "".

proc find::parse_sizeop { size } {
    
    switch $size {
	"512 Bytes" {
	    return "b"
	}
	"Bytes" {
	    return "c"
	}
	"Kilobytes" {
	    return "k"
	}
	default {
	    return ""
	}
    }
}

# find::reset --
#
#   Method to reset the radio and checkbuttons in the dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc find::reset { } {
    variable find

    # Allow for the permission elements to maintain an integer value.
    foreach x $find(list.reset) {
	switch -glob $x {
	    all_perm* -
	    user_perm* -
	    group_perm* -
	    other_perm* {
		set find($x) 0
	    }
	    default {
		set find($x) ""
	    }
	}
    }

    # Disable the entry widgets in the Command tab.
    find::toggle_cmd_entry off
}

# find::clear --
#
#   Method to clear entry items of their text and reset the
#   background and foreground properties.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc find::clear { } {
    variable find

    # Reset the data structure elements and bg/fg.
    foreach x $find(list.clear) {
	set find($x) ""
	$find(entry.$x) configure -bg #ffffff -fg #000000
    }

    focus $find(entry.path)

    # Disable the entry widgets in the Command tab.
    find::toggle_cmd_entry off
}

# find::help --
#
#   Method to invoke the Find Command Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc find::help { } {
    global tkWorld

    help::create "help/find.html" "Find Command Help"
}

# find::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc find::close { } {
    variable find
    
    balloonhelp::cancel
    destroy $find(dialog)
}

# find::open --
#
#   Method to add a file to the file entry and move the cursor
#   index of the entry to the end.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc find::open { } {
    variable find

    regsub { } [file::select] {} find(file)
    $find(file_entry) icursor end
}

# find::toggle_cmd_entry --
#
#   Turn an entry on/off if the checkbutton is turned on/off.
#
# Args
#
#   type - Either command, touch, remove, move or copy.
#
# Returns
#
#   None.

proc find::toggle_cmd_entry { type } {
    variable find

    switch $type {
	command {
	    $find(entry.command) configure \
		    -bg #ffffff \
		    -state normal
	    $find(entry.moveto) configure \
		    -bg #d9d9d9 \
		    -state disabled
	    $find(entry.copyto) configure \
		    -bg #d9d9d9 \
		    -state disabled
	}
	touch -
	remove -
	off {
	    $find(entry.command) configure \
		    -bg #d9d9d9 \
		    -state disabled
	    $find(entry.moveto) configure \
		    -bg #d9d9d9 \
		    -state disabled
	    $find(entry.copyto) configure \
		    -bg #d9d9d9 \
		    -state disabled
	}
	move {
	    $find(entry.command) configure \
		    -bg #d9d9d9 \
		    -state disabled
	    $find(entry.moveto) configure \
		    -bg #ffffff \
		    -state normal
	    $find(entry.copyto) configure \
		    -bg #d9d9d9 \
		    -state disabled
	}
	copy {
	    $find(entry.command) configure \
		    -bg #d9d9d9 \
		    -state disabled
	    $find(entry.moveto) configure \
		    -bg #d9d9d9 \
		    -state disabled
	    $find(entry.copyto) configure \
		    -bg #ffffff \
		    -state normal
	}
    }
}