namespace eval chgrp {
    variable chgrp

    # Define the chgrp array structure so that all variables are
    # defined for the callbacks in the radiobuttons and checkbuttons.
    array set chgrp {
	list.reset ""
	list.clear ""
	entry.dir ""
	entry.file ""
	ddlistbox.groups ""
	group ""
	dialog ""
    }
}

# chgrp::create --
#
#   Method to create the dialog box for the chgrp command.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to run your command, then reopen
#   it to modify it.  By not making this a modal dialog, we do not
#   have to implement any last command saving characteristics since
#   the user can just leave the dialog open.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chgrp::create { } {
    global tkWorld
    variable chgrp

    # Put the focus on the chgrp dialog if it is already open.
    if [winfo exists $chgrp(dialog)] {
	switch -- [wm state $chgrp(dialog)] {
	    normal {
		raise $chgrp(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $chgrp(dialog)
	    }
	}
	focus $chgrp(dialog)
	return
    } else {
	set chgrp(dialog) [dialog::create .chgrp Chgrp]
    }

    # There is only 1 tab.
    set tab1 [tabnotebook::page [dialog::interior \
	    $chgrp(dialog)] "Options"]

    # Use a frame to encapsulate the file and dir selections
    # so that the frame can be centered accross the grid columns.
    set f1 [frame $tab1.f1 \
	    -class TabnotebookFrame]

    button $f1.file_button \
	    -text "File" \
	    -width 5 \
	    -command chgrp::open
    set chgrp(entry.file) [entry $f1.file_entry \
	    -width 35 \
	    -textvariable chgrp::chgrp(file)]
    label $f1.dir_label \
	    -text "Directory" \
	    -width 8
    set chgrp(entry.dir) [entry $f1.dir_entry \
	    -width 35 \
	    -textvariable chgrp::chgrp(dir)]
    checkbutton $f1.recursive \
	    -text "Recursive" \
	    -variable chgrp::chgrp(recursive) \
	    -onvalue "R" \
	    -offvalue ""
    grid $f1.file_button $f1.file_entry \
	    -sticky w \
	    -padx 2 \
	    -pady 2
    grid $f1.dir_label $f1.dir_entry $f1.recursive \
	    -sticky w \
	    -padx 2 \
	    -pady 2

    # Build the frame which allows the user to change the group.
    set f2 [frame $tab1.f2 \
	    -class TabnotebookFrame]

    label $f2.label_group \
	    -text "Group Name" \
	    -width 10
    set chgrp(ddlistbox.groups) [ddlistbox::create \
	    $f2 chgrp::chgrp(group) [system::groups]]
    grid $f2.label_group $chgrp(ddlistbox.groups) \
	    -sticky ew \
	    -padx 2 \
	    -pady 2

    # Now build the individual checkbutton options available to
    # the user for the chgrp command.
    set f3 [frame $tab1.f3 \
	    -class TabnotebookFrame]

    checkbutton $f3.changes \
	    -text Changes \
	    -variable chgrp::chgrp(changes) \
	    -onvalue c \
	    -offvalue ""
    checkbutton $f3.silent \
	    -text Silent \
	    -variable chgrp::chgrp(silent) \
	    -onvalue f \
	    -offvalue ""
    checkbutton $f3.verbose \
	    -text Verbose \
	    -variable chgrp::chgrp(verbose) \
	    -onvalue v \
	    -offvalue ""
    grid $f3.changes $f3.silent $f3.verbose \
	    -sticky ew \
	    -padx 5 \
	    -pady 5

    # Build the first tab.
    pack $f1 $f2 $f3 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # Duhhhhhh......
    focus $chgrp(entry.file)

    # Define the lists for the reset and clear methods
    set chgrp(list.reset) "recursive changes silent verbose"
    set chgrp(list.clear) "file dir"
}

# chgrp::ok --
#
#   Method to insert the command the user has created into the CC.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chgrp::ok { } {
    global tkWorld
    variable chgrp

    # Build the command line for the recursive argument.
    set cmd_arg ""
    if [string length $chgrp(recursive)] {
	set cmd_arg "-$chgrp(recursive) "
    }

    set cmd_arg2 ""
    foreach x "changes silent verbose" {
	if [string length $chgrp::chgrp($x)] {
	    append cmd_arg2 $chgrp::chgrp($x)
	}
    }
    if [string length $cmd_arg2] {
	set cmd_arg2 "-$cmd_arg2"
    }
    set cmd_arg "$cmd_arg $cmd_arg2"
    unset cmd_arg2

    # Insert the Tcl command list in the Command Center with the
    # proper formatting of a space between each argument on the
    # command line.  If there are no options given by the user,
    # then don't display it in the CC.

    if [string length $cmd_arg] {
	$tkWorld(cmd_center) insert insert \
		"chgrp $cmd_arg $chgrp(group) "
    } else {
	$tkWorld(cmd_center) insert insert \
		"chgrp $chgrp(group) "
    }
    if [string length $chgrp(dir)] {
	$tkWorld(cmd_center) insert insert "$chgrp(dir) "
    }
    if [string length $chgrp(file)] {
	$tkWorld(cmd_center) insert insert "$chgrp(file) "
    }

    # Activate the buttons in the toolbar for the command center.
    toolbar::group_state cmd_center active
    toolbar::button_state $toolbar::toolbar(stop) disabled
}

# chgrp::reset --
#
#   Method to reset the radio and checkbuttons in the dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chgrp::reset { } {
    variable chgrp

    # Reset all of the list elements.
    foreach x $chgrp(list.reset) {
	set chgrp($x) ""
    }
}

# chgrp::clear --
#
#   Method to clear entry items of their text and reset the
#   background and foreground properties.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chgrp::clear { } {
    variable chgrp

    # Reset the data structure elements and bg/fg.
    foreach x $chgrp(list.clear) {
	set chgrp($x) ""
	$chgrp(entry.$x) configure -bg #ffffff -fg #000000
    }

    focus $chgrp(entry.file)
}

# chgrp::help --
#
#   Method to invoke the Chgrp Command Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chgrp::help { } {
    global tkWorld

    help::create "help/chgrp.html" "Chgrp Command Help"
}

# chgrp::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chgrp::close { } {
    variable chgrp
    
    balloonhelp::cancel
    destroy $chgrp(dialog)
}

# chgrp::open --
#
#   Method to add a file to the file entry and move the cursor
#   index of the entry to the end.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chgrp::open { } {
    variable chgrp

    # Insert the file list and move the cursor.
    $chgrp(entry.file) insert insert [file::select]
    $chgrp(entry.file) icursor end

    # Set the focus on the entry with the file list in it.
    focus $chgrp(entry.file)
}
