/*  Screem:  editor.c,
 *  The main editor, handles insertion of text, context dependant popup menu,
 *  special keypresses etc
 *
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <ctype.h>
#include <gnome.h>

#include "editor.h"
#include "editMenu.h"
#include "fileops.h"
#include "helpers.h"
#include "html.h"
#include "htmlfuncs.h"
#include "menus.h"
#include "page.h"
#include "pageUI.h"
#include "plugin.h"
#include "preferences.h"
#include "preview.h"
#include "site.h"
#include "support.h"

#ifdef USE_GTKEXTEXT
#include "gtkextext.h"
#endif

extern Site *current_site;
extern Page *current_page;
extern GtkWidget *app;
extern Preferences *cfg;

static GtkWidget *editor; /* the current editor in use */
static GtkWidget *attr_menu = NULL;
static gboolean attr_up = FALSE;
static GtkWidget *screem_create_editor( void );

static void screem_html_insert( gchar *text );
static void screem_perl_insert( gchar *text );
static void screem_java_insert( gchar *text );

static gboolean screem_editor_keypress(GtkWidget *widget, GdkEventKey *event);
static gboolean html_key_press( GdkEventKey *event );
static void screem_editor_drop( GtkWidget *widget, GdkDragContext *context,
				gint x, gint y, 
				GtkSelectionData *selectionData,
				guint info, guint time );
static gboolean screem_editor_motion( GtkWidget *widget, 
				      GdkDragContext *context, 
				      gint x, gint y, guint time, 
				      gpointer data );

static gboolean tag_attributes_menu( gint element );
static void attribute_menu( GtkWidget *menu, gchar **attributes, gint *num );
static void no_attribute( GtkWidget *widget, GdkEventKey *event );
static gint xy_to_cursor_pos( gint x, gint y );

static void preview_editor( void );

static GtkWidget *perl_menu( void );
static GtkWidget *java_menu( void );
static void java_compile( GtkWidget *widget );

static void tagAttributes( gint element );

static void free_popup_memory( void );

void build_tag_dialog( GtkWidget *box, gchar **attributes, gint element,
		       gchar *temp );


static gboolean changed = FALSE;
static gboolean frozenhack = FALSE;
static gboolean popup_open = FALSE;

static GList *popup_memory = NULL;  /* a list of memory blocks we need to
				       free when the popup is destroyed */

#define DELAY 5000

static gint insert_offset;  /* hackish but makes it simple */


static GList *names = NULL;
static GList *entries = NULL;

/* popup menu */
static GnomeUIInfo editor_file_menu[] = {
        GNOMEUIINFO_MENU_NEW_ITEM( N_("_New Page..."), 
                                   N_("Add a new page to the site"), 
                                   0, NULL ),
	GNOMEUIINFO_MENU_NEW_ITEM( N_( "_New Blank Page" ),
				   N_("Create a blank page (add if working on a site)" ),
				   GTK_SIGNAL_FUNC( screem_page_create_blank_page ),
				   NULL ),
        GNOMEUIINFO_MENU_REVERT_ITEM(GTK_SIGNAL_FUNC(screem_page_revert_proxy),
				     0 ),
        GNOMEUIINFO_MENU_SAVE_ITEM( GTK_SIGNAL_FUNC( screem_page_save_proxy ),
				    0 ),
        GNOMEUIINFO_MENU_SAVE_AS_ITEM( GTK_SIGNAL_FUNC( screem_page_save_as ),
				       0 ),
        { GNOME_APP_UI_ITEM, N_("_Insert file..."), N_( "Insert file" ),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_OPEN, 0,
          GDK_CONTROL_MASK, NULL },
        GNOMEUIINFO_END
};

static GnomeUIInfo editor_main_menu[] = {
        GNOMEUIINFO_MENU_FILE_TREE( editor_file_menu ),
        GNOMEUIINFO_MENU_EDIT_TREE( edit_menu ),
	GNOMEUIINFO_SUBTREE( N_( "CVS" ), cvs_menu ),
	/*        GNOMEUIINFO_SUBTREE( N_("Move" ), editorMoveMenu ),*/
	GNOMEUIINFO_SEPARATOR,
       	GNOMEUIINFO_END
};

/* DnD menu */
static GnomeUIInfo editor_dnd_menu[] = {
        { GNOME_APP_UI_ITEM, N_("Insert relative filename"),
          N_("Insert relative filename"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_("Insert complete filename"),
          N_("Insert complete filename"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_("Insert tag"), N_("Insert tag"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_( "Insert tag attibute" ),
          N_( "If the drop is into a tag then the data is converted into\
 an attribute for that tag" ), 0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_( "Insert inline" ),
          N_( "Insert the text from the file into the page" ), 0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        GNOMEUIINFO_SEPARATOR,
        { GNOME_APP_UI_ITEM, N_("Cancel drag"), N_("Cancel drag"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        GNOMEUIINFO_END
};

typedef enum _Drop_actions {
        RELATIVE_PATH,
        FULL_PATH,
        TAG,
        ATTRIBUTE,
        INLINE     /* inserts the file contents (text files only) */
} Drop_actions;

typedef enum _Drop_types {
	TARGET_URI_LIST,
	TARGET_URL,
	TARGET_COLOUR
} Drop_types;

static const GtkTargetEntry drop_types[] = {
        { "text/uri-list", 0, TARGET_URI_LIST },
        { "_NETSCAPE_URL", 0, TARGET_URL },
        { "x-url/http", 0, TARGET_URL },
        { "x-url/ftp", 0, TARGET_URL },
        { "application/x-color", 0, TARGET_COLOUR }
};
static const gint num_drop_types = sizeof(drop_types) / sizeof(drop_types [0]);



/**
 * screem_editor_new:
 *
 * create the editor area, returning the scrolled window that contains it,
 * the text widget itself is the "text_widget" data for the scrolled window
 *
 * return values: a GtkWidget
 */
GtkWidget *screem_editor_new()
{
	GtkWidget *sw;

	sw = screem_create_editor();

	gtk_widget_set_usize( sw, cfg->editor_width, cfg->editor_height );

	editor = gtk_object_get_data( GTK_OBJECT( sw ), "text_widget" );

	return sw;
}

static void editor_status_change( GtkEditable *widget, gint pos )
{
	gchar *l;
	GtkWidget *label;

#ifdef USE_GTKEXTEXT
	l = g_strdup_printf( "L: %i C: %i", 
			      gtk_extext_get_line( GTK_EXTEXT( editor ) ),
			      gtk_extext_get_column( GTK_EXTEXT( editor ) ) );
#else
	l = g_strdup_printf( "L: X: C X" );
#endif
	label = gtk_object_get_data( GTK_OBJECT( app ), "editor_status" );
	gtk_label_set( GTK_LABEL( label ), l );

	g_free( l );
}

static GtkWidget *screem_create_editor()
{
	GtkWidget *sw;
	GtkWidget *text;
	GtkStyle *style;
	GdkColor *cb;
	GdkColor *cf;

#ifdef USE_GTKEXTEXT
	text = gtk_extext_new( NULL, NULL );
     	sw = gtk_scrolled_window_new( GTK_EXTEXT( text )->hadj,
				      GTK_EXTEXT( text )->vadj );
#else
	text = gtk_text_new( NULL, NULL );
	sw = gtk_scrolled_window_new( GTK_TEXT( text )->hadj,
				      GTK_TEXT( text )->vadj );
#endif
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ), 
                                        GTK_POLICY_AUTOMATIC,
                                        GTK_POLICY_AUTOMATIC );
	gtk_container_add( GTK_CONTAINER( sw ), text );

	gtk_object_set_data( GTK_OBJECT( sw ), "text_widget", text );

	gtk_signal_connect( GTK_OBJECT( text ), "changed",
			    GTK_SIGNAL_FUNC( screem_editor_changed ), 0 );
	gtk_signal_connect( GTK_OBJECT( text ), "button_press_event",
			    GTK_SIGNAL_FUNC( screem_editor_popup ), 0 );
	gtk_signal_connect( GTK_OBJECT( text ), "key_press_event",
			    GTK_SIGNAL_FUNC( screem_editor_keypress ), 0 );
	gtk_signal_connect( GTK_OBJECT( text ),"move-to-row",
			    GTK_SIGNAL_FUNC( editor_status_change ), NULL );
        gtk_signal_connect( GTK_OBJECT( text ), "move-to-column",
			    GTK_SIGNAL_FUNC( editor_status_change ), NULL );

	gtk_drag_dest_set( text,
                           GTK_DEST_DEFAULT_HIGHLIGHT |
                           GTK_DEST_DEFAULT_DROP,
                           drop_types, num_drop_types,
                           GDK_ACTION_COPY | GDK_ACTION_ASK );

        gtk_signal_connect( GTK_OBJECT( text ), "drag_data_received",
                            GTK_SIGNAL_FUNC( screem_editor_drop ), NULL );
        gtk_signal_connect( GTK_OBJECT( text ), "drag_motion",
                            GTK_SIGNAL_FUNC( screem_editor_motion ), NULL );

	/* set its style */
	style = gtk_style_copy( gtk_widget_get_style( text ) );
#ifdef USE_GTKEXTEXT
	cb = &style->bg[ 0 ];
#else
	cb = &style->base[ 0 ];
#endif
        cb->red = cfg->back[ 0 ];
        cb->green = cfg->back[ 1 ];
        cb->blue = cfg->back[ 2 ];

#ifdef USE_GTKEXTEXT
	cf = &style->fg[ 0 ];
#else
	cf = &style->text[ 0 ];
#endif
        cf->red = cfg->fore[ 0 ];
        cf->green = cfg->fore[ 1 ];
        cf->blue = cfg->fore[ 2 ];

	style->font = cfg->font;
        gtk_widget_set_style( text, style );

#ifndef USE_GTKEXTEXT
	gtk_text_set_word_wrap( GTK_TEXT( text ), TRUE );
        gtk_text_set_line_wrap( GTK_TEXT( text ), TRUE );

	gtk_timeout_add( DELAY, (GtkFunction)hi_trigger, 0 );
#endif	

	return sw;
}

/**
 * screem_editor_popup:
 *
 * displays the editor menu
 *
 * return values: none
 */
gboolean screem_editor_popup( GtkWidget *widget, GdkEventButton *event )
{
	gint cpos;
	gint pos2;
    	gint pos;
	gint num;
	gchar *text;
	gchar *tag;
	gchar *tag2;
	gchar *title;
	gchar **attributes;
	GtkWidget *menu;
	GtkWidget *menu2;
	GtkWidget *menu_item;

	GtkWidget *helper_menu;

	Page *page;

	const gchar *pathname;
	const gchar *mime_type;

	if( current_site )
		page = screem_site_get_current_page( current_site );
	else
		page = current_page;

	g_return_val_if_fail( page != NULL, FALSE );

	pathname = screem_page_get_pathname( page );

	if( event->button == 3 ) {
		popup_open = TRUE;
		gtk_signal_emit_stop_by_name( GTK_OBJECT( widget ),
					      "button_press_event" );

		mime_type = gnome_mime_type( pathname );
	
		cpos = screem_editor_get_pos();
		/* get position of click */
		pos2 = pos = xy_to_cursor_pos( event->x, event->y );
		screem_editor_set_pos( pos );

		/* create the menu */
		menu = gnome_popup_menu_new( editor_main_menu );

		helper_menu = screem_helper_menu();
		gtk_menu_insert( GTK_MENU( menu ), helper_menu, 3 );

		menu2 = screem_build_scripts_menu();
		if( menu2 )
			gtk_menu_append( GTK_MENU( menu ), menu2 );

		/* id the tag we are inside so we can offer attribute 
		   editing */

		text = screem_editor_get_text( 0, -1 );
                if( ( ! strcmp( "text/html", mime_type ) ) &&
		    ( pos = in_tag( text, pos ) ) ) {
                        pos --;
                        tag = next_tag( text + pos, &pos );
                        tag2 = tag_name( tag );
                        pos = binary_search_tags( tag2 );

                        if( pos != -1 ) {
                                if( html_tag_popups[ pos ] ) { 
                                        title = g_strdup_printf( "<%s>",tag2 );
                                        menu_item = gtk_menu_item_new_with_label( title );
                                        gtk_widget_show( menu_item );
                                        gtk_menu_append( GTK_MENU( menu ),
                                                          menu_item );
                                        menu2 = gtk_menu_new();
                                        gtk_menu_item_set_submenu(GTK_MENU_ITEM( menu_item ), menu2 );
                                        attributes = html_tag_popups[ pos ];
                                        num = 0;
                                        attribute_menu(menu2, attributes,
						       &num);
                                        g_free( title );
                                }
				/* add tag attributes dialog menu item */
				menu_item = gtk_menu_item_new_with_label( _( "Tag attributes" ) );
				gtk_widget_show( menu_item );
				gtk_menu_append( GTK_MENU( menu ),menu_item );
				gtk_signal_connect_object( GTK_OBJECT( menu_item ), "activate", GTK_SIGNAL_FUNC( tagAttributes ), (gpointer)pos );
                        }
                        g_free( tag2 );
                } else if( ! strcmp( "text/x-perl", mime_type ) ) {
			/* create perl menu */
			menu_item = perl_menu();
			gtk_menu_prepend( GTK_MENU( menu ), menu_item );

		} else if( ! strcmp( "text/x-java", mime_type ) ) {
			/* create java menu */
			menu_item = java_menu();
			gtk_menu_prepend( GTK_MENU( menu ), menu_item );
		}
                g_free( text );

		/* display menu */
		gnome_popup_menu_do_popup_modal( menu, 0, 0, event, 0 );
		gtk_widget_destroy( menu );
		if( pos2 < cpos )
			cpos += insert_offset;
		screem_editor_set_pos( cpos );
		popup_open = FALSE;

		free_popup_memory();

		return TRUE;
	}
	return FALSE;
}

/**
 * screem_editor_enable:
 *
 * enables editing.
 *
 * return values: none
 */
void screem_editor_enable()
{
	gtk_editable_set_editable( GTK_EDITABLE( editor ), TRUE );
}

/**
 * screem_editor_disable:
 *
 * disables editing.
 *
 * return values: none
 */
void screem_editor_disable()
{
	gtk_editable_set_editable( GTK_EDITABLE( editor ), FALSE );
}

/**
 * screem_editor_clear:
 *
 * clears the editor
 *
 * return values: none
 */
void screem_editor_clear()
{
	screem_editor_delete_forward( 0, -1 );
}


GtkStyle *screem_editor_get_style()
{
	return gtk_style_copy( gtk_widget_get_style( editor ) );
}

gint screem_editor_get_width()
{
	if( ! editor )
		return cfg->editor_width;

	return editor->allocation.width;
}

gint screem_editor_get_height()
{

	if( ! editor )
		return cfg->editor_height;

	return editor->allocation.height;
}

gint screem_editor_get_length()
{
	gint len;

#ifdef USE_GTKEXTEXT
	len = gtk_extext_get_length( GTK_EXTEXT( editor ) );
#else
	len = gtk_text_get_length( GTK_TEXT( editor ) );
#endif

	return len;
}

gchar *screem_editor_get_text( gint from, gint len )
{
	gint end = len;

      	if( end == -1 )
		end = screem_editor_get_length();
	else
		end += from;

	return gtk_editable_get_chars( GTK_EDITABLE( editor ), from, end );
}

/**
 * screem_editor_has_selection:
 *
 * identify if we have selected some text in the editor
 *
 * return values: boolean
 */
gboolean screem_editor_has_selection()
{
	return GTK_EDITABLE( editor )->has_selection;
}

void screem_editor_select_region( gint start, gint len )
{
    	gtk_editable_select_region( GTK_EDITABLE( editor ), start, len );
}

void screem_editor_cut()
{
	gtk_editable_cut_clipboard( GTK_EDITABLE( editor ) );
}

void screem_editor_copy()
{
	gtk_editable_copy_clipboard( GTK_EDITABLE( editor ) );
}

void screem_editor_clear_selection()
{
	gtk_editable_delete_selection( GTK_EDITABLE( editor ) );
}

void screem_editor_paste()
{
	gtk_editable_paste_clipboard( GTK_EDITABLE( editor ) );
}

void screem_editor_set_pos( gint pos )
{
	gtk_editable_set_position( GTK_EDITABLE( editor ), pos );
}

gint screem_editor_get_pos()
{
	return gtk_editable_get_position( GTK_EDITABLE( editor ) );
}

/**
 * screem_editor_delete_forward:
 * @pos: the position to start at
 * @len: the number of chars to delete, -1 to delete everything
 *
 * return values: none
 */
void screem_editor_delete_forward( gint pos, gint len )
{
	if( len == -1 )
		len = ( screem_editor_get_length() - pos );

 	if( ! frozenhack )
#ifdef USE_GTKEXTEXT
		gtk_extext_freeze( GTK_EXTEXT( editor ) );
#else
	gtk_text_freeze( GTK_TEXT( editor ) );
#endif
	gtk_editable_delete_text( GTK_EDITABLE( editor ), pos, pos + len );

	if( ! frozenhack )
#ifdef USE_GTKEXTEXT
		gtk_extext_thaw( GTK_EXTEXT( editor ) );
#else
	gtk_text_thaw( GTK_TEXT( editor ) );
#endif
}


/**
 * screem_editor_buffer_text:
 *
 * takes the text in the editor and places it in the buffer for the
 * page being editted
 *
 * return values: none
 */
void screem_editor_buffer_text()
{
	Page *page;
	gchar *text;

	if( current_site )
		page = screem_site_get_current_page( current_site );
	else
		page = current_page;

	g_return_if_fail( page != NULL );

	text = screem_editor_get_text( 0, -1 );
	g_free( page->data );
	page->data = text;
}

/**
 * screem_editor_insert_markup:
 * @open_element:  the opening element
 * @close_element: the closing element
 *
 * handles inserting markup, including wrapping around selected text
 *
 * return values: none
 */
void screem_editor_insert_markup( gchar *open_element, gchar *close_element )
{
	gboolean has_selection;
	guint start;
	guint end;
	gint pos;
	gint ipos;
	Page *page;

	if( current_site )
		page = screem_site_get_current_page( current_site );
	else
		page = current_page;

	g_return_if_fail( page != NULL );

	pos = screem_editor_get_pos();
	ipos = pos;

	has_selection = GTK_EDITABLE( editor )->has_selection;

	if( has_selection ) {
		start = GTK_EDITABLE( editor )->selection_start_pos;
                end = GTK_EDITABLE( editor )->selection_end_pos;
                start = start < end ? start:end;
		if( open_element )
			screem_editor_insert( start, open_element );
		
		start = GTK_EDITABLE( editor )->selection_start_pos;
                end = GTK_EDITABLE( editor )->selection_end_pos;
                end = start > end ? start:end;
                pos = end;
		if( close_element ) {
                        screem_editor_insert( end, close_element );
                        pos += strlen( close_element );
                }
	} else {
		if( open_element ) {
			screem_editor_insert( pos, open_element );
			pos += strlen( open_element );
		}
		if( close_element ) {
			screem_editor_insert( pos, close_element );
			pos += strlen( close_element );
		}

		if( open_element )
			pos = ipos + strlen( open_element );
		else if( close_element )
			pos = ipos + strlen( close_element );
	}
	screem_editor_set_pos( pos );
}


/**
 * screem_editor_insert_attribute:
 * @attribute: the attribute text to insert
 *
 * inserts an attribute into an element
 *
 * return values: none
 */
void screem_editor_insert_attribute( gchar *attribute )
{
	gint pos;
	gint cpos;
        gint start;
        gint len;
        gchar *insert;
        gchar *text;
        gchar *tag;
        gchar *attr;
	Page *page;

	gchar *temp;

 	if( current_site )
		page = screem_site_get_current_page( current_site );
	else
		page = current_page;

	g_return_if_fail( page != NULL );
	g_return_if_fail( attribute != NULL );

        cpos = pos = screem_editor_get_pos();
        text = screem_editor_get_text( 0, -1 );

        if( ! ( pos = in_tag( text, pos ) ) ) {
		g_free( text );
		return;
        }
        pos --;
        start = pos;
        /* pos == the start of the tag */
        tag = next_tag( text + pos, &pos );
        g_free( text );

  	/* does the tag already have this attribute?
           = match:  [A-Za-z]*=\"[^"]*\"
           other: [A-Za-z]*
        */
        set_use_regex( TRUE );
	temp = g_strconcat( " ", attribute, NULL );
        if( ( attr = find_text( temp, "=", NULL ) ) ) {
                attr = g_strndup( temp, attr - temp );
                insert = g_strdup_printf( "%s=\\\"[^\\\"]*\\\"", attr );
                g_free( attr );
        } else {
                insert = g_strdup( temp );
        }

        /* now we try and replace it */
        set_use_regex( TRUE );
        if( ! ( attr = find_text( tag, insert, temp ) ) ) {
                /* we didn't find it so we just insert it */
                g_free( insert );
                insert = g_strdup_printf( " %s", attribute );
                screem_editor_insert( pos, insert );
		insert_offset = strlen( attribute ) + 1;
        } else {
                /* erase all the tag's attributes */
                tag = attr;
		insert_offset = - ( pos - start + 1 );
                screem_editor_delete_forward( start, pos - start + 1 );
                insert = g_strdup( tag );
                screem_editor_insert( start, insert );
		insert_offset += strlen( insert );
        }

	g_free( tag );
	g_free( insert );
	g_free( temp );

	gtk_editable_changed( GTK_EDITABLE( editor ) );
}

/**
 * screem_editor_insert_file:
 * @filename: the name of the file to insert
 *
 * inserts the contents of a text file into the editor at the
 * current cursor position
 *
 * return values: none
 */
void screem_editor_insert_file( gchar *filename )
{
	FILE *in;
	gint size;
	gint pos;
	gchar buffer[ 1025 ];

	pos = screem_editor_get_pos();

	if( ! ( in = fopen( filename, "r" ) ) )
		return;

	while( ( size = fread( buffer, 1, 1024, in ) ) ) {
		buffer[ size ] = 0;
		screem_editor_insert( pos, buffer );
		pos += size;
	}

	fclose( in );
}

/**
 * screem_editor_insert:
 * @pos:   the postion to insert the text at
 * @text: the text to insert
 *
 * inserts text into the editor at the position requested.
 * if pos > total text length in the editor then the text will be
 * appended to the current text in the editor
 *
 * return values: none
 */
void screem_editor_insert( gint pos, gchar *text )
{
	gint len;
	gint start;
	const gchar *pathname;
	const gchar *mime_type;

	Page *page;

	if( current_site )
		page = screem_site_get_current_page( current_site );
	else
		page = current_page;

	g_return_if_fail( page != NULL );
	g_return_if_fail( text != NULL );

	pathname = screem_page_get_pathname( page );
	mime_type = gnome_mime_type( pathname );

	len = screem_editor_get_length();

	if( pos == -1 )
		pos = screem_editor_get_pos();

	if( pos > len )
		pos = len;

	start = pos;

	len = strlen( text );

	if( ! frozenhack )
#ifdef USE_GTKEXTEXT
		gtk_extext_freeze( GTK_EXTEXT( editor ) );
#else
	gtk_text_freeze( GTK_TEXT( editor ) );
#endif
	if( ! cfg->tag_highlighting ) {
		gtk_editable_insert_text( GTK_EDITABLE( editor ),
					  text, strlen( text ), &pos );
	} else {
#ifndef USE_GTKEXTEXT
		/* not using GTK_EXTEXT and we want highlighting */
		gtk_text_set_point( GTK_TEXT( editor ), pos );
		if( ! strcmp( "text/html", mime_type ) )
			/* we are working on an html file */
			screem_html_insert( text );
		else if( ! strcmp( "text/x-perl", mime_type ) )
			screem_perl_insert( text );
		else {
			gtk_editable_insert_text( GTK_EDITABLE( editor ),
						  text, strlen( text ), &pos );
		}
#else
		gtk_editable_insert_text( GTK_EDITABLE( editor ),
					  text, strlen( text ), &pos );
#endif
	}
	
	
	if( ! frozenhack ) {
#ifdef USE_GTKEXTEXT
		gtk_extext_thaw( GTK_EXTEXT( editor ) );
#else
		gtk_text_thaw( GTK_TEXT( editor ) );
#endif
		gtk_widget_grab_focus( editor );
	}
}

#ifndef USE_GTKEXTEXT
static void screem_html_insert( gchar *text )
{
	gchar *start_chunk;
        gchar *end_chunk;
        gchar *end;
        gint len_chunk;
        gchar *tag;
        gchar *name;
        Hi_colours* hc;

        GdkColor f, b;
        GdkColor *fore;
        GdkColor *back;

	gint len = strlen( text );

	end_chunk = text;
        end = text + len;
        while( end_chunk < end ) {
		start_chunk = end_chunk;
		set_use_regex( TRUE );

  		end_chunk = find_text( end_chunk, "<[^>]*>", NULL );

		if( ! end_chunk ) {
			/* we have some normal text to insert */
			gtk_text_insert(GTK_TEXT( editor ), cfg->font, 
					NULL, NULL, start_chunk, 
					end - start_chunk );
			return;
		} else if( end_chunk > start_chunk ) {
			/* we have some normal text to insert */
			gtk_text_insert( GTK_TEXT( editor ), 
					 cfg->font, NULL, 
					 NULL, start_chunk, 
					 end_chunk - start_chunk );
		}

		/* now insert the tag */
		len_chunk = 0;
		while( end_chunk[ ++len_chunk ] != '>' );

                fore = back = NULL;

		/* get the tag name */
		tag = g_strndup( end_chunk, ++len_chunk );
		name = tag_name( tag );
		g_free( tag );
		/* id the group it belongs to */
		hc = tag_colour( name );
		if( hc->useFore ) {
			f.red = hc->fore[ 0 ];
			f.green = hc->fore[ 1 ];
			f.blue = hc->fore[ 2 ];
			fore = &f;
		}
		if( hc->useBack ) {
			b.red = hc->back[ 0 ];
			b.green = hc->back[ 1 ];
			b.blue = hc->back[ 2 ];
			back = &b;
		}
		gtk_text_insert( GTK_TEXT( editor ), cfg->font, fore, 
                                 back, end_chunk, len_chunk );

		end_chunk += len_chunk;
        }
}

static void screem_perl_insert( gchar *text )
{
	gchar *start;
	gchar *insert;
	gchar *token = text;
	gchar *end;

	gint len;
	gchar c;

	GdkColor f;
	GdkColor b;
	GdkColor *fore;
	GdkColor *back;

	g_return_if_fail( text != NULL );

	end = text + strlen( text );

	while( token < end ) {
		start = token;
		while( ! isspace( *token ) && *token != '(' &&
		       *token != ')' && *token != '{' &&
		       *token != '}' && *token != '#' &&
		       *token != '\"' && *token != ';' && 
		       *token != '\'' && token < end )
			token ++;

		len = token - start;

		insert = g_strndup( start, len );

		/* id what colour to use for insert */
		fore = NULL;
		back = NULL;
		if( ! strcmp( "if", insert ) || ! strcmp( "else", insert ) ) {
			f.red = 0xffff;
			f.green = 0xffff;
			f.blue = 0;
			fore = &f;
		} else if( ! strcmp( "print", insert ) ) {
			f.red = 0;
			f.green = 0xffff;
			f.blue = 0;
			fore = &f;
		}

		if( strlen( insert ) ) {
			gtk_text_insert( GTK_TEXT( editor ), cfg->font, 
					 fore, back, insert, -1 );
		}
		g_free( insert );

		c = *token;

		if( token < end && c == '#' ) {
			/* we have a comment */
			start = token;
			while( token < end && *token != '\n' )
				token ++;
			insert = g_strndup( start, token - start );
			f.red = 0xffff;
			f.green = 0;
			f.blue = 0;
			fore = &f;
			back = NULL;
			gtk_text_insert( GTK_TEXT( editor ), cfg->font, 
					 fore, back, insert, -1 );
			g_free( insert );
		} else if( token < end && ( c == '\"' || c == '\'' ) ) {
			/* we have a string */
			start = token;
			while( token < end  ) {
				if( *(++token) == c && *( token - 1 ) != '\\' )
					break;
			}
			if( token < end )
				token ++;
			insert = g_strndup( start, token - start );
			f.red = 0x6666;
			f.green = 0x6666;
			f.blue = 0xffff;
			fore = &f;
			back = NULL;
			gtk_text_insert( GTK_TEXT( editor ), cfg->font, 
					 fore, back, insert, -1 );
			g_free( insert );
		}

		if( token < end ) {
			gtk_text_insert( GTK_TEXT( editor ), cfg->font, NULL,
					 NULL, token, 1 );
			token ++;
		}
	}

}
#endif

/**
 * screem_editor_changed:
 *
 * the editor has had text inserted or deleted
 *
 * return values: none
 */
void screem_editor_changed()
{
	changed = TRUE;
}

static gboolean screem_editor_keypress( GtkWidget *widget, GdkEventKey *event )
{
	Page *page;

	const gchar *pathname;
	const gchar *mime_type;
         
	if( current_site )
		page = screem_site_get_current_page( current_site );
	else
		page = current_page;

	g_return_val_if_fail( page != NULL, FALSE );

	pathname = screem_page_get_pathname( page );
	mime_type = gnome_mime_type( pathname );

	if( ! strcmp( mime_type, "text/html" ) )
		return html_key_press( event );
	else
		return FALSE;
}

static gboolean html_key_press( GdkEventKey *event )
{
	gchar *text;
	gchar *tag = NULL;
	gchar *tag2;
	gchar *format;
	gint pos;
	gint start;
	gint len;
	gint len2;
	gint num;
	gboolean in = FALSE;
	gboolean empty = FALSE;

	pos = screem_editor_get_pos();

	switch( event->keyval ) {
	case GDK_less: 
	case GDK_greater:
	case GDK_ampersand:
	case GDK_Agrave:
	case GDK_Aacute:
	case GDK_Acircumflex:
	case GDK_Atilde:
	case GDK_Adiaeresis:
	case GDK_Aring:
	case GDK_AE:
	case GDK_Ccedilla:
	case GDK_Egrave:
	case GDK_Eacute:
	case GDK_Ecircumflex:
	case GDK_Ediaeresis:
	case GDK_Igrave:
	case GDK_Iacute:
	case GDK_Icircumflex:
	case GDK_Idiaeresis:
	case GDK_ETH:
	case GDK_Ntilde:
	case GDK_Ograve:
	case GDK_Oacute:
	case GDK_Ocircumflex:
	case GDK_Otilde:
	case GDK_Odiaeresis:
	case GDK_Ooblique:
	case GDK_Ugrave:
	case GDK_Uacute:
	case GDK_Ucircumflex:
	case GDK_Udiaeresis:
	case GDK_Yacute:
	case GDK_THORN:
	case GDK_ssharp:
	case GDK_agrave:
	case GDK_aacute:
	case GDK_acircumflex:
	case GDK_atilde:
	case GDK_adiaeresis:
	case GDK_aring:
	case GDK_ae:
	case GDK_ccedilla:
	case GDK_egrave:
	case GDK_eacute:
	case GDK_ecircumflex:
	case GDK_ediaeresis:
	case GDK_igrave:
	case GDK_iacute:
	case GDK_icircumflex:
	case GDK_idiaeresis:
	case GDK_eth:
	case GDK_ntilde:
	case GDK_ograve:
	case GDK_oacute:
	case GDK_ocircumflex:
	case GDK_otilde:
	case GDK_odiaeresis:
	case GDK_oslash:
	case GDK_ugrave:
	case GDK_uacute:
	case GDK_ucircumflex:
	case GDK_udiaeresis:
	case GDK_yacute:
	case GDK_thorn:
	case GDK_ydiaeresis:
	case GDK_exclamdown:
	case GDK_cent:
	case GDK_sterling:
	case GDK_currency:
	case GDK_yen:
	case GDK_brokenbar:
	case GDK_section:
	case GDK_diaeresis:
	case GDK_copyright:
	case GDK_ordfeminine:
	case GDK_guillemotleft:
	case GDK_notsign:
	case GDK_hyphen:
	case GDK_registered:
	case GDK_macron:
	case GDK_degree:
	case GDK_plusminus:
	case GDK_twosuperior:
	case GDK_threesuperior:
	case GDK_acute:
	case GDK_mu:
	case GDK_paragraph:
	case GDK_periodcentered:
	case GDK_cedilla:
	case GDK_onesuperior:
	case GDK_masculine:
	case GDK_guillemotright:
	case GDK_onequarter:
	case GDK_onehalf:
	case GDK_threequarters:
	case GDK_questiondown:
	case GDK_quotedbl:
	case GDK_Tab:
		/* we're about to insert a character with a special HTML
		   meaning (tag or entity). To make sure that this can
		   be done we have to check if we're already inside another
		   entity. Look for a semicolon after the current position
		   AND an ampersand before the current position.
		   This search is aborted as soon as we find a character that
		   is part of another HTML construct */
		text = screem_editor_get_text( 0, -1 );
		if( pos != 0 )
			empty = ( text[ pos - 1 ] == '&' && 
				  text[ pos ] == ';' );
		else
			empty = FALSE;
		
		if( ! empty && in_entity( text, pos ) ) {
			/* let the widget handle insertion as normal as
			   the entity
			   already exists and is not empty */
			g_free( text );
			return FALSE;
		}

		if( empty ) 
			format = "%s";
		else if( text[ pos ] == ';' )
			format = "&%s";
		else if( pos > 0 && text[ pos - 1 ] == '&' )
			format = "%s;";
		else
			format = "&%s;";

		switch(event->keyval) {
		case GDK_ampersand:
			if( empty ) 
				tag = g_strdup_printf( format , "amp" ) ;
			else 
				tag = g_strdup_printf( format , "" ) ;
			break ;
		case GDK_Agrave:
			tag = g_strdup_printf( format , "Agrave" );
			break ;
		case GDK_Aacute:
			tag = g_strdup_printf( format , "Aacute" );
			break ;
		case GDK_Acircumflex:
			tag = g_strdup_printf( format , "Acirc" );
			break ;
		case GDK_Atilde:
			tag = g_strdup_printf( format , "Atilde" );
			break ;
		case GDK_Adiaeresis:
			tag = g_strdup_printf( format , "Auml" );
			break ;
		case GDK_Aring:
			tag = g_strdup_printf( format , "Aring" );
			break ;
		case GDK_AE:
			tag = g_strdup_printf( format , "AElig" );
			break ;
		case GDK_Ccedilla:
			tag = g_strdup_printf( format , "Ccedil" );
			break ;
		case GDK_Egrave:
			tag = g_strdup_printf( format , "Egrave" );
			break ;
		case GDK_Eacute:
			tag = g_strdup_printf( format , "Eacute" );
			break ;
		case GDK_Ecircumflex:
			tag = g_strdup_printf( format , "Ecirc" );
			break ;
		case GDK_Ediaeresis:
			tag = g_strdup_printf( format , "Euml" );
			break ;
		case GDK_Igrave:
			tag = g_strdup_printf( format , "Igrave" );
			break ;
		case GDK_Iacute:
			tag = g_strdup_printf( format , "Iacute" );
			break ;
		case GDK_Icircumflex:
			tag = g_strdup_printf( format , "Icirc" );
			break ;
		case GDK_Idiaeresis:
			tag = g_strdup_printf( format , "Iuml" );
			break ;
		case GDK_ETH:
			tag = g_strdup_printf( format , "ETH" );
			break ;
		case GDK_Ntilde:
			tag = g_strdup_printf( format , "Ntilde" );
			break ;
		case GDK_Ograve:
			tag = g_strdup_printf( format , "Ograve" );
			break ;
		case GDK_Oacute:
			tag = g_strdup_printf( format , "Oacute" );
			break ;
		case GDK_Ocircumflex:
			tag = g_strdup_printf( format , "Ocirc" );
			break ;
		case GDK_Otilde:
			tag = g_strdup_printf( format , "Otilde" );
			break ;
		case GDK_Odiaeresis:
			tag = g_strdup_printf( format , "Ouml" );
			break ;
		case GDK_Ooblique:
			tag = g_strdup_printf( format , "Oslash" );
			break ;
		case GDK_Ugrave:
			tag = g_strdup_printf( format , "Ugrave" );
			break ;
		case GDK_Uacute:
			tag = g_strdup_printf( format , "Uacute" );
			break ;
		case GDK_Ucircumflex:
			tag = g_strdup_printf( format , "Ucirc" );
			break ;
		case GDK_Udiaeresis:
			tag = g_strdup_printf( format , "Uuml" );
			break ;
		case GDK_Yacute:
			tag = g_strdup_printf( format , "Yacute" );
			break ;
		case GDK_THORN:
			tag = g_strdup_printf( format , "THORN" );
			break ;
		case GDK_ssharp:
			tag = g_strdup_printf( format , "szlig" );
			break ;
		case GDK_agrave:
			tag = g_strdup_printf( format , "agrave" );
			break ;
		case GDK_aacute:
			tag = g_strdup_printf( format , "aacute" );
			break ;
		case GDK_acircumflex:
			tag = g_strdup_printf( format , "acirc" );
			break ;
		case GDK_atilde:
			tag = g_strdup_printf( format , "atilde" );
			break ;
		case GDK_adiaeresis:
			tag = g_strdup_printf( format , "auml" );
			break ;
		case GDK_aring:
			tag = g_strdup_printf( format , "aring" );
			break ;
		case GDK_ae:
			tag = g_strdup_printf( format , "aelig" );
			break ;
		case GDK_ccedilla:
			tag = g_strdup_printf( format , "ccedil" );
			break ;
		case GDK_egrave:
			tag = g_strdup_printf( format , "egrave" );
			break ;
		case GDK_eacute:
			tag = g_strdup_printf( format , "eacute" );
			break ;
		case GDK_ecircumflex:
			tag = g_strdup_printf( format , "ecirc" );
			break ;
		case GDK_ediaeresis:
			tag = g_strdup_printf( format , "euml" );
			break ;
		case GDK_igrave:
			tag = g_strdup_printf( format , "igrave" );
			break ;
		case GDK_iacute:
			tag = g_strdup_printf( format , "iacute" );
			break ;
		case GDK_icircumflex:
			tag = g_strdup_printf( format , "icirc" );
			break ;
		case GDK_idiaeresis:
			tag = g_strdup_printf( format , "iuml" );
			break ;
		case GDK_eth:
			tag = g_strdup_printf( format , "eth" );
			break ;
		case GDK_ntilde:
			tag = g_strdup_printf( format , "ntilde" );
			break ;
		case GDK_ograve:
			tag = g_strdup_printf( format , "ograve" );
			break ;
		case GDK_oacute:
			tag = g_strdup_printf( format , "oacute" );
			break ;
		case GDK_ocircumflex:
			tag = g_strdup_printf( format , "ocirc" );
			break ;
		case GDK_otilde:
			tag = g_strdup_printf( format , "otilde" );
			break ;
		case GDK_odiaeresis:
			tag = g_strdup_printf( format , "ouml" );
			break ;
		case GDK_oslash:
			tag = g_strdup_printf( format , "oslash" );
			break ;
		case GDK_ugrave:
			tag = g_strdup_printf( format , "ugrave" );
			break ;
		case GDK_uacute:
			tag = g_strdup_printf( format , "uacute" );
			break ;
		case GDK_ucircumflex:
			tag = g_strdup_printf( format , "ucirc" );
			break ;
		case GDK_udiaeresis:
			tag = g_strdup_printf( format , "uuml" );
			break ;
		case GDK_yacute:
			tag = g_strdup_printf( format , "yacute" );
			break ;
		case GDK_thorn:
			tag = g_strdup_printf( format , "thorn" );
			break ;
		case GDK_ydiaeresis:
			tag = g_strdup_printf( format , "yuml" );
			break ;
		case GDK_exclamdown:
			tag = g_strdup_printf( format , "iexcl" );
			break ;
		case GDK_cent:
			tag = g_strdup_printf( format , "cent" );
			break ;
		case GDK_sterling:
			tag = g_strdup_printf( format , "pound" );
			break ;
		case GDK_currency:
			tag = g_strdup_printf( format , "curren" );
			break ;
		case GDK_yen:
			tag = g_strdup_printf( format , "yen" );
			break ;
		case GDK_brokenbar:
			tag = g_strdup_printf( format , "brkbar" );
			break ;
		case GDK_section:
			tag = g_strdup_printf( format , "sect" );
			break ;
		case GDK_diaeresis:
			tag = g_strdup_printf( format , "uml" );
			break ;
		case GDK_copyright:
			tag = g_strdup_printf( format , "copy" );
			break ;
		case GDK_ordfeminine:
			tag = g_strdup_printf( format , "ordf" );
			break ;
		case GDK_guillemotleft:
			tag = g_strdup_printf( format , "laqo" );
			break ;
		case GDK_notsign:
			tag = g_strdup_printf( format , "not" );
			break ;
		case GDK_hyphen:
			tag = g_strdup_printf( format , "shy" );
			break ;
		case GDK_registered:
			tag = g_strdup_printf( format , "reg" );
			break ;
		case GDK_macron:
			tag = g_strdup_printf( format , "macr" );
			break ;
		case GDK_degree:
			tag = g_strdup_printf( format , "deg" );
			break ;
		case GDK_plusminus:
			tag = g_strdup_printf( format , "plusmn" );
			break ;
		case GDK_twosuperior:
			tag = g_strdup_printf( format , "suo2" );
			break ;
		case GDK_threesuperior:
			tag = g_strdup_printf( format , "suo3" );
			break ;
		case GDK_acute:
			tag = g_strdup_printf( format , "acute" );
			break ;
		case GDK_mu:
			tag = g_strdup_printf( format , "micro" );
			break ;
		case GDK_paragraph:
			tag = g_strdup_printf( format , "para" );
			break ;
		case GDK_periodcentered:
			tag = g_strdup_printf( format , "middot" );
			break ;
		case GDK_cedilla:
			tag = g_strdup_printf( format , "cedil" );
			break ;
		case GDK_onesuperior:
			tag = g_strdup_printf( format , "sup1" );
			break ;
		case GDK_masculine:
			tag = g_strdup_printf( format , "ordm" );
			break ;
		case GDK_guillemotright:
			tag = g_strdup_printf( format , "raquo" );
			break ;
		case GDK_onequarter:
			tag = g_strdup_printf( format , "fraq14" );
			break ;
		case GDK_onehalf:
			tag = g_strdup_printf( format , "fraq12" );
			break ;
		case GDK_threequarters:
			tag = g_strdup_printf( format , "fraq34" );
			break ;
		case GDK_questiondown:
			tag = g_strdup_printf( format , "iquest" );
			break ;
		case GDK_quotedbl:
			if( empty || in_attribute( text, pos ) )
				tag = g_strdup_printf( format, "quot" );
			else if( in_tag( text, pos ) ) {
				/* if pos - 1 == "=" then insert two, place
				   between */
				if( text[ pos - 1 ] == '=' &&
				    (text[ pos ] == ' ' ||
				     text[ pos ] == '>') ) {
					tag = g_strdup_printf( "\"\"" );
				} else {
					g_free( text );
					return FALSE;
				}
			} else
				tag = g_strdup_printf( format, "quot" );
			break;
		case GDK_Tab:
			if( empty )
				tag = g_strdup_printf( format, "nbsp" );
			else {
				/* insert a normal tab as not in an entity */
				g_free( text );
				return FALSE;
			}
			break;
		case GDK_less:
			if( empty ) 
				tag = g_strdup_printf( format , "lt" ) ;
			else {
				/* We add this manually along with a
				   GDK_greater, unless we are found to be
				   inside a tag */
				if( in_tag( text, pos ) ) {
					/* in tag so let widget do it */
					g_free( text );
					return FALSE;
				}
				tag = g_strdup_printf( "<>" );
			}
			break ;
		case GDK_greater:
			if( empty ) 
				tag = g_strdup_printf( format , "gt" ) ;
			else{
				/* to keep consistant with GDK_less we
				   do a normal insert if we aren't inside
				   and empty entity */
				g_free( text );
				return FALSE;
			}
			break ;
		}
		g_free( text );
		screem_editor_insert( pos, tag );
		if( empty ) 
			pos += ( strlen(tag) + 1 );
		else {
			if( event->keyval == GDK_less ||
			    event->keyval == GDK_ampersand ||
			    (event->keyval == GDK_quotedbl 
			     && text[ pos - 1] == '=') ) 
				pos ++;
			else 
				pos += strlen( tag );
		}
		g_free( tag );
		screem_editor_set_pos( pos );
		gtk_signal_emit_stop_by_name( GTK_OBJECT( editor ), 
					      "key_press_event" );
		return TRUE;
		break;
	case GDK_space:
		if( ( attr_up ) || ( ! cfg->inline_tagging ) )
			return FALSE;

		text = screem_editor_get_text( 0, -1 );

		if( ! ( start = in_tag( text, pos ) ) ) {
			g_free( text );
			return FALSE;
		}

		len = start;
		start --;
		tag = next_tag( &text[ start ], &start );
		tag2 = tag_name( tag );
		num = binary_search_tags( tag2 );
		len2 = strlen( tag2 );
		g_free( text );
		g_free( tag2 );
		g_free( tag );
	
		/* we must know about the tag and also not be positioned
		   inside its name in the editor */
		if( ( num < 0 ) || ( ( pos - len ) < len2 ) )
			return FALSE;
		else if( tag_attributes_menu( num ) ) {
			gtk_signal_emit_stop_by_name( GTK_OBJECT( editor ),
						      "key_press_event" );
			return TRUE;
		}
		break;
	case GDK_slash:
		text = screem_editor_get_text( 0, -1 );
		if( ! ( start = in_tag( text, pos ) ) ) {
			g_free( text );
			return FALSE;
		} else if( text[ start ] != '>' ) {
			/* we already have other characters in the tag */
			g_free( text );
			return FALSE;
		}

		tag2 = current_context( text, start - 2, NULL, NULL );

		if( tag2 ) {
			/* we found a tag that needed closing */
			screem_editor_insert( pos, "/" );
			screem_editor_insert( pos + 1, tag2 );
			g_free( tag2 );
			gtk_signal_emit_stop_by_name( GTK_OBJECT( editor ),
						      "key_press_event" );
		}

		g_free( text );
	   		
		return (gboolean)tag2;

		break;
	default:
		break;
	}
	return FALSE;
}

static void screem_editor_drop( GtkWidget *widget, GdkDragContext *context,
				gint x, gint y, 
				GtkSelectionData *selectionData,
				guint info, guint time )
{
	Page *page;
	GtkWidget *popup;
	gint item;
	gint pos;
	guint16 *colours;
	gchar *text = NULL;
	gchar *dir;
	gchar *temp;
	Drop_actions action;

	gboolean inline_;

	const gchar *pathname;
	const gchar *mime_type;
	const gchar *spath;

	if( current_site ) {
		page = screem_site_get_current_page( current_site );
		spath = screem_site_get_pathname( current_site );
	} else {
		page = current_page;
		spath = NULL;
	}

	g_return_if_fail( page != NULL );

	pathname = screem_page_get_pathname( page );

	/* if we are doing a middle drag then we need to ask the user
	   what to do */
	if( context->action == GDK_ACTION_ASK ) {
                popup = gnome_popup_menu_new( editor_dnd_menu );
                item = gnome_popup_menu_do_popup_modal( popup, 0, 0, 0, 0 );
                switch( item ) {
                case 0: /* insert relative filename */
                        action = RELATIVE_PATH;
                        break;
                case 1: /* insert complete filename */
                        action = FULL_PATH;
                        break;
                case 2: /* insert tag */
                        action = TAG;
                        break;
                case 3: /* insert tag attribute */
                        action = ATTRIBUTE;
                        break;
                case 4: /* insert text inline */
                        action = INLINE;
                        break;
                default:
                        return;
                        break;
                }
        } else {
                action = RELATIVE_PATH;
	}

	/* position the cursor at the drop location */
	pos = xy_to_cursor_pos( x, y );
	screem_editor_set_pos( pos );

	/* handle the different drop types */
	switch( info ) {
	case TARGET_URI_LIST:
		text = selectionData->data + strlen( "file:" );
		text[ strlen( text ) - 2 ] = 0;
		
		mime_type = gnome_mime_type_or_default( text, "unknown" );

		/* are we doing an inline insert? */
		if( action == INLINE ) {
			inline_ = ! strncmp( mime_type, "text/", 
					     strlen( "text/" ) );
			inline_ |= ! strcmp( mime_type, "application/x-php" );
			inline_ |= ! strcmp( mime_type, "application/x-asp" );
			inline_ |= ! strcmp( mime_type, "application/x-cgi" );

			if( inline_ ) {
				screem_editor_insert_file( text );
				return;
				break;
			}
		}

		/* insert using relative path? */
		if( action != FULL_PATH ) {
			dir = g_dirname( pathname );
			chdir( dir );
			g_free( dir );
			text = relative_path( text, spath );
		} else {
			text = g_strdup( text );
		}

		/* inserting a tag? */
		if( action == TAG ) {
			temp = text;
			if( ! strncmp( mime_type, "image/", 6 ) )
				text = g_strdup_printf( "<img src=\"%s\" alt=\"\">", temp );
			else
				text = g_strdup_printf( "<a href=\"%s\"> </a>",
							temp );
			g_free( temp );
		}
		break;
	case TARGET_URL:
		if( action == TAG )
			text = g_strdup_printf( "<a href=\"%s\"> </a>",
						selectionData->data );
		else
			text = g_strdup( selectionData->data );
		break;
	case TARGET_COLOUR:
		colours = (guint16*)selectionData->data;
                temp = g_strdup_printf( "\"#%.2x%.2x%.2x\"",colours[ 0 ] >> 8, 
                                        colours[ 1 ] >> 8, colours[ 2 ] >> 8 );
		switch( action ) {
		case TAG:
                        text = g_strdup_printf( "<font color=%s> </font>",
                                                temp );
                        g_free( temp );
                        break;
		case ATTRIBUTE:
			screem_editor_get_text( 0, -1 );
                        if( ( pos = in_tag( text, pos ) ) ) {
                                /* we are inside a tag */
                                g_free( text );
                                text = g_strdup_printf( " color=%s", temp );
                                screem_editor_insert_attribute( text );
                                g_free( text );
                                return;
                        } else {
                                /* not in a tag, but we will insert it anyway*/
                                g_free( text );
                                text = g_strdup_printf( " color=%s", temp );
                                g_free( temp );
                        }
                        break;
		default:
			text = temp;
			break;
		}
		break;
	default:
		/* we don't know about this type, and as such
		   this case should never execute */
		g_error( "Error: invalid drop type for editor occured\n" );
		break;
	}

	/* insert the text */
	if( text )
		screem_editor_insert( pos, text );

	g_free( text );
}

static gboolean screem_editor_motion( GtkWidget *widget, 
				      GdkDragContext *context, 
				      gint x, gint y, guint time, 
				      gpointer data )
{
	GdkDragAction action;
	
        if( context->suggested_action != GDK_ACTION_ASK )
                action = GDK_ACTION_COPY;
        else
                action = GDK_ACTION_ASK;

        gdk_drag_status( context, action, time );

        return TRUE;
}


gint hi_trigger()
{
	Page *page;
	gchar *text;
	GtkAdjustment *adj;
	gfloat value;
	gint pos;

	gint first_char;

	GList *props;
	GtkPropertyMark mark;
	gint offset;

	gint height;
	gint editor_width;
	gint char_width;
	gint line_width = 0;
	gint i;
	const gint tab_width = 8;
	const int line_wrap_room = 8; /* 8 == LINE_WRAP_ROOM in gtktext.c */
	GdkFont *font = cfg->font;

	if( ! cfg->tag_highlighting )
		return TRUE;

	/* do the highlighting */
	if( current_site )
		page = screem_site_get_current_page( current_site );
	else
		page = current_page;

	if( popup_open )
		return TRUE;

	if( page && changed ) {
		adj = GTK_TEXT( editor )->vadj;
		value = adj->value;

		first_char = GTK_TEXT( editor )->first_line_start_index;

		pos = screem_editor_get_pos();
		text = screem_editor_get_text( first_char, -1 );
		gtk_text_freeze( GTK_TEXT( editor ) );
		frozenhack = TRUE;

		/* calc num of chars to delete (only viewable lines) */
		height = editor->allocation.height;
		height /= ( font->ascent + font->descent );
		/* height = max number of lines that will be on screen */
	    
		editor_width = editor->allocation.width;

		/* now get the total number of chars we are to delete */
		for( i = 0; i < strlen( text ); i ++ ) {
			if( text[ i ] != '\n' ) {
				char_width = gdk_char_width( font, text[ i ] );
				if( text[ i ] == '\t' )
					char_width *= tab_width;
				line_width += char_width;
			}
			if( ( text[ i ] == '\n' ) || 
			    ( line_width > editor_width ) ) {
				line_width = 0;
				if( ( height ) <= 0 )
					break;
				while( ! isspace( text[ i ] ) )
					i --;
				height --;
			}                    
		}
		g_free( text );
		text = screem_editor_get_text( first_char, i );
		screem_editor_delete_forward( first_char, strlen( text ) );
		screem_editor_insert( first_char, text );
		gtk_text_thaw( GTK_TEXT( editor ) );

		frozenhack = FALSE;
		gtk_adjustment_set_value( adj, value );
		screem_editor_set_pos( pos );
		g_free( text );
	}
	
	changed = FALSE;
	return TRUE;
}

Hi_colours* tag_colour( gchar *name )
{
        gchar **group;
        gint num;
        gint gnum;
        Hi_colours *hc;
        GList *list = cfg->colours;

        /* is it a closing tag? */
        if( name[ 0 ] == '/' )
                name ++;

        for( num = 0; ( group = TAG_GROUPS[ num ] ); num ++ ) {
                list = list->next;
                for( gnum = 0; group[ gnum ]; gnum ++ ) {
                        if( ! strcasecmp( name, group[ gnum ] ) )
                                break;
                }
                if( group[ gnum ] )
                        break;
        }

        /* if we haven't found a match then its the default group */
        if( ! group )
                return cfg->colours->data;

        hc = list->data;
        
        if( hc->useFore || hc->useBack )
                return hc;

        return cfg->colours->data;
}

static gboolean tag_attributes_menu( gint element )
{
	gchar **attributes;
	gint num = 0;

	g_return_val_if_fail( element > 0, FALSE );
	g_return_val_if_fail( element < html_tags_size, FALSE );

	if( html_tag_popups[ element ] ) {
		attr_menu = gtk_menu_new();
		attributes = html_tag_popups[ element ];
		attribute_menu( attr_menu, attributes, &num );
		/* attach a callback for a key press, that way the user
		   can ignore the popup and keep typing */
		gtk_signal_connect( GTK_OBJECT( attr_menu ), "key_press_event",
				    no_attribute, 0 );
		gnome_popup_menu_do_popup( attr_menu, 0, 0, 0, 0 );
	}

	return (gboolean)( html_tag_popups[ element ] );
}

static void attribute_menu( GtkWidget *menu, gchar **attributes, gint *num )
{
	GtkWidget *menuitem;
        GtkWidget *menuitem2;
        GtkWidget *menu2 = NULL;
        gchar **sub;
        gchar *title;
        gchar *attr;
        gint n;
        gchar *a = NULL;

        while( attributes[ *num ] ) {
                title = attributes[ *num ];
                menuitem = gtk_menu_item_new_with_label( title );
                gtk_widget_show( menuitem );
                gtk_menu_append( GTK_MENU( menu ), menuitem );
                /* if we have attributes */
                (*num)++;
                attr = attributes[ *num ];
		if( attr ) {
                        menu2 = gtk_menu_new();
                        gtk_signal_connect( GTK_OBJECT( menu2 ),
                                            "key_press_event",
                                            no_attribute, 0 );
                        gtk_menu_item_set_submenu( GTK_MENU_ITEM( menuitem ),
                                                   menu2 );
                } else {
                        /* we can connect a signal to this one as it has
                           no submenus:
                           data = title */
                        a = g_strdup( title );
                        gtk_signal_connect_object( GTK_OBJECT( menuitem ), "activate", screem_editor_insert_attribute, (gpointer) a );
			/* add to list of memory to free */
			popup_memory = g_list_append( popup_memory, a );
                }
                while( ( attr = attributes[ *num ] ) ) {
                                /* special sub menu?  */
                        if( ! strcmp( "*", attr ) ) {
                                if( ! strcmp( "Core", title ) )
                                        sub = core_attributes;
                                else if( ! strcmp( "i18n", title ) )
                                        sub = i18n_attributes;
                                else if( ! strcmp( "Events", title ) )
                                        sub = event_attributes;
                                else
                                        break;
                                n = 0;
                                attribute_menu( menu2, sub, &n );
                                break;
			} else {
                                /* connect the menu up so a click on it can
                                   insert it, but without a value set */
                                menuitem2 = gtk_menu_item_new_with_label( attr );
                                gtk_widget_show( menuitem2 );
                                gtk_menu_append( GTK_MENU( menu2 ),
                                                 menuitem2 );
                                /* connect signal:
                                   data = title + attr */
                                a = g_strdup_printf( "%s\"%s\"", title, attr );
                                gtk_signal_connect_object( GTK_OBJECT( menuitem2 ), "activate", screem_editor_insert_attribute, (gpointer) a );
				popup_memory = g_list_append( popup_memory,a );
                                (*num) ++;
                        }
                }
                (*num) ++;
        }

}

static void no_attribute( GtkWidget *widget, GdkEventKey *event )
{
        gint pos;
        gboolean ret;

        if( ! isascii( event->keyval ) )
                return;

        gtk_signal_emit_stop_by_name( GTK_OBJECT( widget ),"key_press_event" );

        /* if attrMenu is NULL then we must have come from the
           editor popup menu, which doesn't want this functionality */
        if( ! attr_menu )
                return;

        /* insert initial press of space */
	pos = screem_editor_get_pos();
        screem_editor_insert( pos, " " );
	screem_editor_set_pos( ++ pos );
        /* a key was pressed while the attribute menu was open,
           so emit a key_press_event signal on the editor */

        attr_up = TRUE;
	if( event->keyval != ' ' )
		gtk_signal_emit_by_name( GTK_OBJECT( editor ),
					 "key_press_event", event, &ret );
        /* kill the menu */
        gtk_widget_destroy( attr_menu );
        attr_menu = NULL;
	free_popup_memory();
        attr_up = FALSE;
}

static gint xy_to_cursor_pos( gint x, gint y )
{
        gchar *text;
	gint i = 0;

#ifdef USE_GTKEXTEXT
      	GtkExTextLineData *data;
      	text = screem_editor_get_text( 0, -1 );
	data = gtk_extext_get_line_by_offset( GTK_EXTEXT( editor ), y, &i );
	i = data->startpos;
	i += gtk_extext_get_column_by_offset( GTK_EXTEXT( editor ),
					      data, x, &x );
#else
        gchar *line = NULL;
	gint lineWidth = 0;
        gint editorWidth;
        gint height;
        gint start = 0;
        gint charWidth;
        GdkFont *font = cfg->font;
        GtkText *e = GTK_TEXT( editor );
	const int tab_width = 8;
	const int line_wrap_room = 8; /* 8 == LINE_WRAP_ROOM in gtktext.c */

        gdk_window_get_size( e->text_area, &editorWidth, NULL );
        editorWidth -= line_wrap_room; 
	text = screem_editor_get_text( 0, -1 );

        height = font->ascent + font->descent;
        y += e->first_cut_pixels;

        for( i = e->first_line_start_index; i < strlen( text ); i ++ ) {
                if( text[ i ] != '\n' ) {
                        charWidth = gdk_char_width( font, text[ i ] );
                        if( text[ i ] == '\t' )
                                charWidth *= tab_width;
                        lineWidth += charWidth;
                }
                if( ( text[ i ] == '\n' ) || ( lineWidth > editorWidth ) ) {
                        lineWidth = 0;
                        if( ( y -= height ) <= 0 )
                                break;
                        while( ! isspace( text[ i ] ) )
				i --;
			start = i;
                }                       
        }
	line = g_strndup( &text[ start ], i - start );
      	/* wasn't inside the text area */
        if( y <= 0 ) {
                for( start = 0; x > 0; start ++ ) {
                        charWidth = gdk_char_width( font, line[ start ] );
                        if( line[ start ] == '\t' )
                                charWidth *= tab_width;
                        x -= charWidth;
                }
                if( ( start = ( strlen( line ) - start ) ) > 0 )
			i -= start;
        }

        g_free( line );
#endif

	g_free( text );

        return i;
}


static GtkWidget *perl_menu()
{
	GtkWidget *menu_item;
	GtkWidget *menu;

	GtkWidget *menu_item2;

	menu_item = gtk_menu_item_new_with_label( _( "Perl Menu" ) );
	gtk_widget_show( menu_item );

	menu = gtk_menu_new();
	gtk_menu_item_set_submenu( GTK_MENU_ITEM( menu_item ), menu );

	menu_item2 = gtk_menu_item_new_with_label( _( "" ) );
	gtk_widget_show( menu_item2 );
	gtk_menu_append( GTK_MENU( menu ), menu_item2 );

	return menu_item;
}

static GtkWidget *java_menu()
{
	GtkWidget *menu_item;
	GtkWidget *menu;

	GtkWidget *menu_item2;

	menu_item = gtk_menu_item_new_with_label( _( "Java Menu" ) );
	gtk_widget_show( menu_item );

	menu = gtk_menu_new();
	gtk_menu_item_set_submenu( GTK_MENU_ITEM( menu_item ), menu );

	menu_item2 = gtk_menu_item_new_with_label( _( "Compile..." ) );
      	gtk_widget_show( menu_item2 );
	gtk_signal_connect( GTK_OBJECT( menu_item2 ), "activate",
			    GTK_SIGNAL_FUNC( java_compile ), 0 );
	gtk_menu_append( GTK_MENU( menu ), menu_item2 );

	return menu_item;
}

static void java_compile( GtkWidget *widget )
{
	gchar *command;
	const gchar *pathname;

	pathname = screem_page_get_pathname( current_page );

	command = g_strconcat( cfg->java_compiler, " ", pathname, 
			       " 2>&1", NULL );

	execute_command( command );

	g_free( command );
}

/* tag attributes dialog */
static void tagAttributes( gint element )
{
	GtkWidget *dialog;
	gchar *title;

	GtkWidget *sw;
	GtkWidget *box;
	GtkWidget *line;
	GtkWidget *label;
	GtkWidget *combo;

	gchar **attributes;
	gint num = 0;
	gchar *name;
	gint custom = 3;

	GList *combo_list;
  
	gint button;

	gint pos;
	gint len;
	gchar *temp;
	gchar *text;

	gchar *search;
	gchar *value;
	char c;
	GtkWidget *entry;

	GList *list;
	GList *list2;
	
   	gchar *attr;

	/* highlight the tag in the editor window */
	pos = screem_editor_get_pos();
	text = screem_editor_get_text( 0, -1 );
	pos = in_tag( text, pos );
	for( len = (--pos); text[ len ] != '>'; len ++ );
	screem_editor_select_region( pos, ++len );
     	temp = g_strndup( &text[ pos ], len - pos );

	title = g_strdup_printf( _( "Tag attributes: <%s>\n" ), 
				 html_tags[ element ] );

	/* the scrolled window */
	sw = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
					GTK_POLICY_NEVER,
					GTK_POLICY_ALWAYS );
	gtk_widget_set_usize( sw, -1, 200 );
	/* main box */
	box = gtk_vbox_new( FALSE, 0 );
	gtk_scrolled_window_add_with_viewport( GTK_SCROLLED_WINDOW( sw ), box);

  	/* add in the known element attributes for the tag */
	line = gtk_hbox_new( FALSE, 0 );
	label = gtk_label_new( _( "Valid attributes" ) );
	gtk_box_pack_start( GTK_BOX( line ), label, FALSE, FALSE, 4 );
	gtk_box_pack_start( GTK_BOX( box ), line, FALSE, FALSE, 4 );

	attributes = html_tag_popups[ element ];

	names = NULL;
	entries = NULL;
	build_tag_dialog( box, attributes, 0, temp );

	g_free( temp );
	g_free( text );

	/* add 3 custom element boxes to the bottom */
	line = gtk_table_new( 2, 1, FALSE );
	label = gtk_label_new( _( "Custom attributes" ) );
	gtk_table_attach( GTK_TABLE( line ), label, 0, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 4, 4 );
	gtk_box_pack_start( GTK_BOX( box ), line, FALSE, FALSE, 0 );
	while( custom-- ) {
		line = gtk_table_new( 2, 1, FALSE );
		label = gtk_entry_new();
		combo = gtk_entry_new();
		gtk_table_attach( GTK_TABLE( line ), label, 0, 1, 0, 1,
				  GTK_FILL, GTK_FILL, 4, 4 );
		gtk_table_attach( GTK_TABLE( line ), combo, 1, 2, 0, 1,
				  GTK_FILL, GTK_FILL, 4, 4 );
		gtk_box_pack_start( GTK_BOX( box ), line, FALSE, FALSE, 0 );
	}

	dialog = gnome_dialog_new( title,
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CLOSE,
				   NULL );

	gtk_box_pack_start( GTK_BOX( GNOME_DIALOG( dialog )->vbox ), sw,
			    TRUE, TRUE, GNOME_PAD );
	gtk_window_set_modal( GTK_WINDOW( dialog ), TRUE );

	gtk_widget_show_all( sw );
	button = gnome_dialog_run( GNOME_DIALOG( dialog ) );

	if( button == 0 ) {
		/* OK was clicked, insert all attributes that have been set */
		for( list = names, list2 = entries; list; 
		     list = list->next, list2 = list2->next ) {
			value = gtk_entry_get_text( GTK_ENTRY( list2->data ) );
			if( strcmp( value, "" ) ) {
				attr = g_strdup_printf( "%s\"%s\"", 
							(gchar*)list->data,
							value );
				pos = screem_editor_get_pos();
				screem_editor_insert_attribute( attr );
				screem_editor_set_pos( pos );
				g_free( attr );
			}
		}
	}

	g_list_free( names );
	g_list_free( entries );
	g_free( title );

	gtk_widget_destroy( dialog );
}

static void free_popup_memory()
{
	GList *list;

	for( list = popup_memory; list; list = list->next ) {
		g_free( list->data );
	}

	g_list_free( popup_memory );
	
	popup_memory = NULL;
}


void build_tag_dialog( GtkWidget *box, gchar **attributes, gint element,
		       gchar *temp )
{
      	gchar **sub;
	gchar *name;
	gchar *search;
	gchar *value;
	gint num = 0;

	GtkWidget *line;
	GtkWidget *label;
	GtkWidget *combo;
	GtkWidget *entry;

	GList *combo_list;

	gchar c;

	g_return_if_fail( box != NULL );

	while( attributes && ( name = attributes[ num ] ) ) {
		num ++;
		if( attributes[ num ] && ! strcmp( "*", attributes[ num ] ) ) {
			/* one of the standard attribute sets */
			if( ! strcmp( "Core", name ) )
				sub = core_attributes;
			else if( ! strcmp( "i18n", name ) )
				sub = i18n_attributes;
			else if( ! strcmp( "Events", name ) )
				sub = event_attributes;
			else
				break;
			build_tag_dialog( box, sub, 0, temp );
		} else {
			/* add entrys */
			line = gtk_table_new( 2, 1, FALSE );
			label = gtk_entry_new();
			gtk_entry_set_text( GTK_ENTRY( label ), name );
			gtk_entry_set_editable( GTK_ENTRY( label ), FALSE );
			gtk_table_attach( GTK_TABLE( line ), label, 0, 1, 0, 1,
					  GTK_FILL, GTK_FILL, 4, 4 );
			combo = gtk_combo_new();
			entry = GTK_COMBO( combo )->entry;
			gtk_table_attach( GTK_TABLE( line ), combo, 1, 2, 0, 1,
					  GTK_FILL, GTK_FILL, 4, 4 );
			gtk_box_pack_start( GTK_BOX( box ), line, FALSE,
					    FALSE, 0 );
			/* add combo entries */
			combo_list = NULL;
			while( attributes[ num ] ) {
				combo_list = g_list_append( combo_list, g_strdup( attributes[ num ] ) );
				num ++;
			}
			if( combo_list ) {
				gtk_combo_set_popdown_strings( GTK_COMBO( combo ), combo_list );
				g_list_free( combo_list );
			}

			/* 
			 * set the entry value to whatever value the
			 * attribute currently has in the tag, or make it
			 * empty if the attribute isn't used
			 */
			name = g_strconcat( " ", name, NULL );
			search = find_text( temp, name, NULL );
			if( ! search ) {
				gtk_entry_set_text( GTK_ENTRY( entry ), "" );
			} else if( ! strchr( name, '=' ) ) {
				gtk_entry_set_text( GTK_ENTRY( entry ),
						    "1" );
			} else {
				search = strchr( search, '=' );
				if( ( c = *(++search) ) != '"' )
					c = ' ';
				else
					search ++;
				value = search;
				while( *search != c && *search != '>' )
					search ++;
				value = g_strndup( value, search - value );
				gtk_entry_set_text( GTK_ENTRY( entry ),
						    value );
				g_free( value );
			}
			entries = g_list_append( entries, entry );
			names = g_list_append( names, name + 1 );
		}
		num ++;
	}
}
