/*
 * rpmdata.h : describes the data gathered for each RPM analyzed.
 *
 * Copyright (c) 1997 Daniel Veillard <veillard@apocalypse.org>
 * See COPYING for the status of this software.
 *
 * $Id: rpmdata.h,v 1.20 1998/02/23 22:13:57 veillard Exp $
 */

#ifndef __RPMDATA_H__
#define __RPMDATA_H__

#include <rpm/header.h>

/*
 * Structure associated to a local directory containing RPM files.
 */

#define MAX_URLS	256
#define MAX_ARCHS	(sizeof(int) * 8)

typedef struct rpm_dir {
    struct rpm_dir *next;	/* next in list */
    char *color;		/* The bgcolor for the pages :-) */
    char *dir;		        /* where to store the HTML files */
    char *ftp;		        /* The host URL it's mirrored from */
    char *ftpsrc;		/* The host URL where src.rpm can be found */
    char *host;                 /* Hostname for the server */
    char *mail;			/* mail contact for this directory */
    char *maint;		/* Maintainer name for this directory */
    char *name;			/* Name for this repository */
    char *rpmdir;               /* The local repository */
    char *trust;		/* Confidences in the files */
    char *url;		        /* The output URL */
    int   files;		/* number of files */
    int   installbase;		/* Is this from an installed base */
    int   size;			/* aggregated size of files */
    int   nb_mirrors;		/* numbers of mirrors */
    char *mirrors[MAX_URLS];    /* Urls for download */
} rpmDir, *rpmDirPtr;

/*
 * structure associated with an architecture.
 */

typedef struct rpm_arch {
    struct rpm_arch *next;	/* next in the list */
    char *os;			/* the Operating System name */
    char *arch;			/* the CPU architecture */
} rpmArch, *rpmArchPtr;

/*
 * structure associated with an rpm
 */

#define MAX_RES		64
#define MAX_REQU	64

typedef struct rpm_data {
    struct rpm_data *next;	/* next in the full list of RPMs */
    struct rpm_data *nextSoft;	/* next software */
    struct rpm_data *nextArch;	/* RPM for same software on other archs */
    struct rpm_data *nextHash;	/* next in the hash list */
    rpmDirPtr dir;		/* directory infos */
    char *subdir;		/* subdirectory holding the rpm */
    char *filename;		/* name of the file */
    char *name;			/* name of the software */
    char *version;		/* version of the software */
    char *release;		/* software release */
    char *url;                  /* URL for the software */
    char *arch;                 /* the architecture system */
    char *os;                   /* the target system */
    char *distribution;		/* general OS distribution */
    char *vendor;		/* general OS vendor */
    char *packager;		/* the packager */
    char *group;		/* type of software */
    char *summary;		/* 1 line summary */
    char *description;          /* short description */
    char *copyright;            /* software copyright */
    char *changelog;            /* changelog */
    char *srcrpm;		/* source RPM */
    time_t date;		/* date of packaging */
    time_t stamp;		/* modification file of the archive */
    int_32 size;		/* size of the software */
    char *host;			/* build host */
    int   nb_resources;	/* #of resources provided */
    struct rpm_resource *
       resources[MAX_RES + 1];/* list of them */
    int   nb_requires;	        /* #of resources required */
    struct rpm_resource *
          requires[MAX_REQU];	/* list of them */
    char *filelist;             /* the filelist */
} rpmData, *rpmDataPtr;

/*
 * structure associated with a resource
 */

#define MAX_PROVIDE 75

typedef struct rpm_resource {
    struct rpm_resource *next;	/* next in the list */
    char *name;			/* name of the resource */
    int   nb_provider;		/* #of resources provided */
    time_t stamp;		/* max stamp of the provider */
    struct rpm_data *
          provider[MAX_PROVIDE];/* list of them */
} rpmRess, *rpmRessPtr;


extern const char *archNames[MAX_ARCHS];
extern int nbArchs;

extern rpmDataPtr rpmList;
extern rpmDataPtr rpmSoftwareList; /* avoid different archs clash */
extern rpmRessPtr ressList;
extern rpmArchPtr archList;
extern rpmDataPtr rpmInstalledList;
extern rpmRessPtr ressInstalledList;
extern rpmDirPtr dirList;

extern void rpmAddSoftware(rpmDataPtr rpm);
extern void rpmDataPrint(rpmDataPtr rpm);
/* extern void rpmDataPrintAll(void); */
extern rpmRessPtr rpmRessAdd(char *ress, rpmDataPtr rpm, int installed);
extern rpmRessPtr rpmRequAdd(char *requ, rpmDataPtr rpm, int installed);
extern rpmArchPtr rpmArchAdd(char *os, char *arch);
extern int rpmOpen(char *nameRpm, rpmDirPtr dir, const char *subdir);
extern int rpmDirScanAll(void);
extern char *extractEMail(char *string);

extern void rpmGroupSort(int installed);
extern void rpmDistribSort(int installed);
extern void rpmVendorSort(int installed);
extern void rpmDateSort(int installed);
extern void rpmNameSort(int installed);

#endif /* __RPMDATA_H__ */
