package HNS::Hnf::Command;
################################################################
# HNS::Hnf::Command
#
#  hnf command class
#
# $Id: Command.pm,v 1.8.2.1 1999/07/01 12:29:01 kenji Exp $
################################################################

use strict qw(vars subs);
use HNS::System;
use HNS::Template;
use SimpleDB::Hash;

use vars qw(%Entities);
use vars qw($Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $CountName $OmittableEnd);


################################################################
# entity definition
# used in $AllowCommands

$Entities{'Link'} = ['LINK', 'RLINK', 'URL'];
$Entities{'Decoration'} = ['FONT', 'STRIKE', 'LSTRIKE', 'STRONG'];
$Entities{'Image'} = ['IMG', 'MARK'];
$Entities{'Replace'} = ['ALIAS'];

$Entities{'Inline'} = [$Entities{'Link'}, $Entities{'Decoration'},
		       $Entities{'Image'}, $Entities{'Replace'}];

$Entities{'Cite'} = ['CITE', 'PRE'];
$Entities{'List'} = ['UL', 'OL'];
$Entities{'Comment'} = ['FN'];
$Entities{'Block'} = [$Entities{'Cite'}, $Entities{'List'},
		      $Entities{'Comment'}];

$Entities{'Flow'} = [$Entities{'Inline'}, $Entities{'Block'}];

$Entities{'New'} = ['NEW', 'LNEW'];
$Entities{'Sub'} = ['SUB', 'LSUB'];



sub new ($$)
{
    my ($class, $name) = @_;
    my $self = {name=>$name,    # name of command
		attr=>[],       # array of attributes
		arg_content=>undef, # content in argument
		parent=>undef,  # parent command
		pos=>undef,     # position which new command is inserted
		content=>[]};   # content element(command or text)
    bless $self, $class;
}
################################################################
# insert new command at legal position
# if $elem is not HNS::Hnf::Command object,
# it's regarded as a command name, and insert new object
sub InsertCommand($$)
{
    my ($self, $elem) = @_;
    unless (ref $elem){
	$elem = new HNS::Hnf::Command($elem);
    }
    my $pos = $self->{pos} || $self;
    $pos->PushContent($elem);
    $self->{pos} = $elem;
    return $elem;
}
# push new command or plain text as content
sub PushContent($@)
{
    my $self = shift;
    for (@_){
	if (ref $_){
	    $_->{parent} = $self;
#	    print "push parent:";
#	    print $_->{name}, "-";
#	    print $_->{parent}->{name} . "<br>";
	}
	push(@{$self->{content}}, $_);
    }
}
sub UnshiftContent($@)
{
    my $self = shift;
    for (@_){
	if (ref $_){
	    $_->{parent} = $self;
	}
	unshift(@{$self->{content}}, $_);
    }
}

# traverse the tree-structure
sub Traverse
{
    my ($self, $callback) = @_;

    if (&$callback($self, 1)) {	
	for (@{$self->{content}}) {
	    if (ref $_) {
		$_->Traverse($callback);
	    } else {
		&$callback($_, 1);
	    }
	}
	&$callback($self, 0) unless $self->IsOneline;
    }
    $self;
}
################################################################
# access method to static variable of object class
sub get_static_variable ($$)    # private
{
    my ($self, $var_name) = @_;
    my $full_var_name = (ref $self) . "::$var_name";
    return $$full_var_name if defined $$full_var_name;

    # if undefined the class, get from parent class
    my $tmp = (ref $self) . "::ISA";
    my @isa = @$tmp;
    for my $class (@isa){
	$full_var_name = "${class}::$var_name";
	return $$full_var_name if defined $$full_var_name;
	my $tmp = "${class}::ISA";
	push(@isa, @$tmp);
    }
    return undef;
}
sub CountName ($)
{
    return shift->get_static_variable('CountName');
}
sub OmittableEnd ($)
{
    return shift->get_static_variable('OmittableEnd');
}
sub IsOneline ($)
{
    return shift->get_static_variable('IsOneline');
}
sub IsBeginSection($)
{
    return shift->get_static_variable('IsBeginSection');
}
# check $cmd_name is permitted as content of $self
sub allowed($$)
{
    my ($self, $cmd_name) = @_;

#    print "<br>check allowed: $cmd_name: in " . ($self->{name}) . ": ";
    my $AllowCommands = $self->get_static_variable('AllowCommands');
    return 1 unless defined $AllowCommands;
    my @tmp = @$AllowCommands;
    for (@tmp){
	if (ref $_){                    # entity assc array
	    push(@tmp, @$_);
	} elsif ($cmd_name eq $_){      # command type
	    return 1;
	}
    }
    return 0;                  # not allowed $cmd_name in $self
}
################################################################
# translate to HTML
sub AsHTML ($$)
{
    my ($self, $start, $params) = @_;
#    my ($year, $month, $day, $newCount, $subCount, $fnCount, $cat_img) = @$params;
#    print "NC:$newCount\n";
    my $class = ref $self;
    my $cmd_name = $self->{name};

#    print "$cmd_name($start)<br>";
    # template
    my $template = sprintf("%s::%sTemplate",
			   $class,
			   ($start)?'':'End');
    my $comment = ($start) ? "<!-- $cmd_name -->": "";

    my $templ = new HNS::Template;
    $params->{content} = $self->{arg_content};
    for (1..3){
	$params->{$_} = $self->{attr}->[$_];
    }
    return $templ->Expand($$template, $params);
}
################################################################
################################################################
# derived class from HNS::Hnf::Command
# top command (implicit);
package HNS::Hnf::Command::HNF;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command);
$OmittableEnd = 1;
$AllowCommands = ['CAT'];

################################################################
# body commands:
# (*) if you want to add new command, you must account it in
#     $HNS::Hnf::[A-Z][a-z]+::entity

# static variable:
# $Template            [str]  begin template
# $EndTemplate        [str]  end templte
# $NumAttr              [int]  number of attributes
# $IsOneline             [bool] no need end command
# $IsBeginSection       [bool] beginning of section (Section)
# $AllowCommands      [array]commands which can be element of self class
# $CountName          [str] if countable, set name for use of counter variable
# $OmittableEnd            [bool] ]is end-command omittable

################################################################
package HNS::Hnf::Command::CAT;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $CountName $OmittableEnd);
#use vars qw(%DB);

$Template = "[%var]";
@ISA = qw(HNS::Hnf::Command);

$AllowCommands = [$HNS::Hnf::Command::Entities{'New'}];

sub AsHTML() {undef}

################################################################
# New Entities
package HNS::Hnf::Command::New;;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);


@ISA = qw(HNS::Hnf::Command);
$IsBeginSection = 1;
$OmittableEnd = 1;
$CountName = 'new';
$AllowCommands = [$HNS::Hnf::Command::Entities{'Sub'},
		   $HNS::Hnf::Command::Entities{'Flow'}];

package HNS::Hnf::Command::NEW;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::New);
$Template = q(<dt><font size='+1'>
	      <a name='%year%month%day%new' href='?%year%month%high&to=%year%month%day%new#%year%month%day%new'>#%new</a>
	      <strong>%cat %content</strong></font></dt><dd>);

$EndTemplate = "</dd>\n";

package HNS::Hnf::Command::LNEW;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::New);
$Template = q(<dt><font size='+1'>
	      <a name='%year%month%day%new' href='?%year%month%high&to=%year%month%day%new#%year%month%day%new'>#%new</a>
	      <strong>%cat <a href="%1">%content</a></strong></font></dt>
	      <dd>);
	      
$EndTemplate = "</dd>\n";
$NumAttr = 1;

################################################################
# Sub Entities
package HNS::Hnf::Command::Sub;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command);
$CountName = 'sub';
$OmittableEnd = 1;
$AllowCommands = [$HNS::Hnf::Command::Entities{'Flow'}];

package HNS::Hnf::Command::SUB;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Sub);
$Template = q(<dd>
  <a name='%year%month%day%{new}S%sub'
  href='?%year%month%high#%year%month%day%{new}S%sub&to=%year%month%day%{new}S%sub'></a>
  <strong>%content</strong>: );

package HNS::Hnf::Command::LSUB;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Sub);
$Template = q(<dd>
  <a name='%year%month%day%{new}S%sub'
  href='?%year%month%high#%year%month%day%{new}S%sub&to=%year%month%day%{new}S%sub'></a>
  <strong><a href="%1">%content</a></strong>: );

$NumAttr = 1;

################################################################
# Inline Elements
# these element has no content commands,
# so $EndTemplate, $AllowCommands has no means.

package HNS::Hnf::Command::Inline;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command);
$IsOneline = 1;

# Link Entity
package HNS::Hnf::Command::Link;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Inline);

# LINK
package HNS::Hnf::Command::LINK;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Link);
$Template = qq(<a href="%1">%content</a> );
$NumAttr = 1;

# RLINK
package HNS::Hnf::Command::RLINK;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Link);
use vars qw(%DB);
$Template = qq(<a href="%url%2">%content</a>);
$NumAttr = 2;
%DB = ();
sub AsHTML($$$)
{
    my ($self, $start, $params) = @_;

#    if ($HNS::System::RlinkFlag ne 'ON'){
#	return HNS::Hnf::Warning::Message('CannotUse', $self->{name}) .
#	    $self->{element};
#    } else {
    unless (defined %DB){
	tie %DB, 'SimpleDB::Hash',
	"$HNS::System::DiaryDir/conf/rlink.txt", 1;
    }
    my $url = $DB{$self->{attr}->[1]};
    unless ($url){
	return &HNS::Hnf::Warning::Message('NotAccounted',
					   $self->{name},undef,
					   $self->{attr}->[1]);
    } else {
	$params->{url} = $url;
	return $self->SUPER::AsHTML($start, $params);
    }
}

# URL
package HNS::Hnf::Command::URL;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Link);

$Template = qq(<img alt="*" src="./icons/mark01.gif" width="16" height="16"><a href="%1">%content(%1)</a>);
$NumAttr = 1;

# Decoration Commands
package HNS::Hnf::Command::Decoration; 
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Inline);

# STRIKE
package HNS::Hnf::Command::STRIKE;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Decoration);
$Template = q(<strike>%content</strike>);

# LSTRIKE
package HNS::Hnf::Command::LSTRIKE;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Decoration);
$Template = q(<strike><a href="%1">%content</a></strike>);
$NumAttr = 1;

# STRONG
package HNS::Hnf::Command::STRONG;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Decoration);
$Template = q(<strong>%content</strong>);

# FONT
package HNS::Hnf::Command::FONT;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Decoration);
$Template = q(<font %1="%2">%content</font>);
$NumAttr = 2;

# Image Commands
package HNS::Hnf::Command::Image;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Inline);

# IMG 
package HNS::Hnf::Command::IMG;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

use Image::Size;

@ISA = qw(HNS::Hnf::Command::Image);
$Template = q(<img %align src="%2" alt="%content" width="%width" height="%height">);
$NumAttr = 2;

sub AsHTML ($$$)
{
    my ($self, $start, $params) = @_;
    my %loc = (r=>'align="right"', l=>'align="left"' , n=>'' );
    my ($src, $alt) = ($self->{attr}->[2], $self->{arg_content});
    my $align = $loc{$self->{attr}->[1]};
    my ($width, $height) = imgsize($src, $HNS::System::ImgWidthMaxSize);
    
    $params->{align} = $align;
    $params->{width} = $width;
    $params->{height} = $height;
    return $self->SUPER::AsHTML($start, $params);
#    return qq(<img src="$src" alt="$alt" align="$align" $width_height>\n);
}

# MARK
package HNS::Hnf::Command::MARK;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

use Image::Size;

@ISA = qw(HNS::Hnf::Command::Image);
use vars qw(%List);
$Template = qq(<img src="%src" alt="%1" width=%width height=%height>);
$NumAttr = 1;
%List = ("(^^)" => "icons/nomal_13.gif",
	 "(^^;" => "icons/ase_13.gif",
	 "(;_;)" => "icons/naku_13.gif",
	 "v(^^)" => "icons/v_13.gif",
	 "!!" => "icons/neko_13.gif",
	 "??" => "icons/hatena_13.gif");

sub AsHTML ($$$){
    my ($self, $start, $params) = @_;
    my $mark = $self->{attr}->[1];
    my $src = $List{$mark};
    my ($width, $height) = imgsize($src);
    $params->{src} = $src;
    $params->{width} = $width;
    $params->{height} = $height;
    return $self->SUPER::AsHTML($start, $params);
#    return qq(<img src="$src" alt="$mark" width="$width" height="$height">\n);
}

# Replace Commands
# ALIAS
package HNS::Hnf::Command::ALIAS;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Inline);
use vars qw(%DB);
$Template = "%term";
%DB = ();

sub AsHTML ($$$)
{
    my ($self, $start, $params) = @_;
    my $term = $self->{arg_content};
#    if ($HNS::System::AliasFlag ne 'ON'){
#	return HNS::Hnf::Warning::Message('CannotUse', $self->{name}) .
#	    $term;
#    } else {
    unless (defined %DB){                              # if first use
	tie %DB, 'SimpleDB::Hash',                     # then tie hash
	"$HNS::System::DiaryDir/conf/alias.txt", 1;
    }
    my $replaced_term = $DB{$term};
    unless ($replaced_term){
	return HNS::Hnf::Warning::Message('NotAccounted', $self->{name},undef,
					  $term);
    } else {
	$params->{term} = $replaced_term;
	$self->SUPER::AsHTML($start, $params);
    }
}

################################################################
# Block Commands
package HNS::Hnf::Command::Block;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command);

# Cite Commands
package HNS::Hnf::Command::Cite;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Block);

# PRE
package HNS::Hnf::Command::PRE;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Cite);
$AllowCommands = [$HNS::Hnf::Command::Entities{'Inline'}];
$Template = "<pre>";
$EndTemplate = "</pre>";

# CITE
package HNS::Hnf::Command::CITE;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Cite);
$Template = "<blockquote>";
$EndTemplate = "</blockquote>";
$AllowCommands = [$HNS::Hnf::Command::Entities{'Flow'}];

# List Commands
package HNS::Hnf::Command::List;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Block);
$AllowCommands = ['LI'];

# UL
package HNS::Hnf::Command::UL;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::List);
$Template = "<ul>";
$EndTemplate = "</ul>";

# OL
package HNS::Hnf::Command::OL;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::List);
$Template = "<ol>";
$EndTemplate = "</ol>";

# LI
package HNS::Hnf::Command::LI;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Block);
$Template = q(<li>%content);
$EndTemplate = "</li>";
$OmittableEnd = 1;
$AllowCommands = [$HNS::Hnf::Command::Entities{'Flow'}];

# HR
package HNS::Hnf::Command::HR;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Block);
$IsOneline = 1;
$Template = q(<hr>\n);

# Comment Commands
package HNS::Hnf::Command::Comment;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);

@ISA = qw(HNS::Hnf::Command::Block);

# FN
package HNS::Hnf::Command::FN;
use vars qw(@ISA $Template $EndTemplate $NumAttr $IsOneline $AllowCommands
	    $IsBeginSection $CountName $OmittableEnd);
use vars qw($HeaderTemplate $FooterTemplate $ContentTemplate);

@ISA = qw(HNS::Hnf::Command::Comment);

$Template = qq(<a href="?%year%month%high&to=%year%month%day%{new}F%fn#%year%month%day%{new}F%fn">
	      <small>*%fn</small></a>);

$HeaderTemplate = qq(<hr width="40%" align="left"><dt><small>\n);
$FooterTemplate = qq(</small></dd>);

$ContentTemplate = qq(<dd><a name="%year%month%day%{new}F%fn"
		      href="?%year%month%high&to=%year%month%day%{new}F%fn#%year%month%day%{new}F%fn">
		      <small>*%fn</small></a>:%content);

$AllowCommands = ['LINK', 'STRIKE'];

1;

