/*
 * image.c - creates textured background and other pixmap stuff.
 */


#include <stdio.h>
#include <stdlib.h>

#include <string.h>
#include <ctype.h>
#include <math.h>

#include "arena.h"

#define Byte w_rot_ih_eByte 	/* - ,    ... */
#  include PNG_HEADER_FILE_FOR_INCLUSION_IN_ARENA
#undef Byte

#include "defs.h"
#include "XGetHClrs.h"
#include "bridge.h"
#include "colour_types.h"
#include "colour.h"
#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
#  include "debug.h"
# else
#  include "error.h"
#endif
#include "dither.h"
#include "gif_arena.h"
#include "image.h"
#include "image_i_types.h"
#include "image_types.h"
#include "jpeg_arena.h"
#include "types.h"
#include "main.h"
#include "png_arena.h"
#include "status.h"
#include "util.h"
#include "xbm_arena.h"
#include "xpm_arena.h"


Pixmap smile, frown, note_pixmap, caution_pixmap, warning_pixmap;
Image* images;  /* linked list of images */
Image *note_image, *caution_image, *warning_image;  /* standard icons */
Image* default_image = NULL;
Image* arena_bg_image = NULL;

unsigned int tileWidth, tileHeight;
unsigned char* tileData = NULL;

/* Barely used colourmap */
XColor papercols[256];

static XColor paperrgb[3] = {{ 0, 230, 218, 194, 0, 0 },
			     { 0, 220, 209, 186, 0, 0 },
			     { 0, 210, 199, 177, 0, 0 }};


#if 0	/* janet 21/07/95: declared and defined but not used */
#  include "smile.xbm"
#  include "frown.xbm"
#endif
#include "www.xbm"
#include "note.xbm"
#include "caution.xbm"
#include "warning.xbm"


Image* NewImage(void)
{
 Image* theImage;

 if ((theImage = (Image*)Arena_CAlloc(1, sizeof(Image), False)))
   {
    theImage->mask = theImage->pixmap = default_pixmap;
    return theImage;
   }
  else
   return NULL;
}


Bool DelImage(Image* theImage)
{
#ifdef ARENA_DEBUG
 char Iam[] = "DelImage";
#endif


 if (theImage)
   {
    Pixmap thePixmap;

    if ((thePixmap = theImage->pixmap) != default_pixmap)
      {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
       if (BRIDGE_TRACE)
	 Arena_TracePrint(Iam,
			  " freeing pixmap %ld"
			  " (of image "POINTER_FORMAT").\n",
			  thePixmap, theImage);
#endif
       XFreePixmap(display, thePixmap);
      }

    if ((thePixmap = theImage->mask) != default_pixmap)
      {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
       if (BRIDGE_TRACE)
	 Arena_TracePrint(Iam, " freeing mask pixmap: %ld"
			  " (of image "POINTER_FORMAT").\n",
			  thePixmap, theImage);
#endif
       XFreePixmap(display, thePixmap);
      }

    Free(theImage->url);
    Free(theImage);
    return True;
   }
  else
   return False;
}


Image* DefaultImage(void)
{
 if (default_image) return default_image;

 default_pixmap = XCreatePixmapFromBitmapData(display, win, www_bits,
					      www_width, www_height,
					      textColour, transparent, depth);

 default_image = NewImage();
 default_image->pixmap =  XCreatePixmapFromBitmapData(display, win, www_bits,
						      www_width, www_height,
						      textColour, transparent,
						      depth);
 default_image->mask   = default_pixmap;
 default_image->width  = www_width;
 default_image->height = www_height;

 return default_image;
}


unsigned char *Transparent(unsigned char *p, int x, int y)
{
 unsigned int i;
 unsigned char *s;

 if (tileData)
   {
    x = x % tileWidth;
    y = y % tileHeight;
    i = y * tileWidth + x;

    if (depth == 24)
      {
       s = tileData + 4 * i;

       *p++ = *s++;
       *p++ = *s++;
       *p++ = *s++;
       *p++ = *s++;

       return p;
      }

    if (depth == 16)
      {
       s = tileData + 2 * i;

       *p++ = *s++;
       *p++ = *s++;

       return p;
      };

    *p++ = gamma_table[tileData[i]];
    return p;
   }

 if (depth == 24)
   {
    *p++ = (transparent       ) & 0xFF;   /* LSB first! */
    *p++ = (transparent  >>  8) & 0xFF;
    *p++ = (transparent  >> 16) & 0xFF;
    *p++ = '\0';
    return p;
   }

/* must be GPixelShift ...??? */

 if (depth == 16) 
   {
    *p++ = (transparent      ) & 0xFF;   /* LSB first! */
    *p++ = (transparent  >> 8) & 0xFF;
    return p;
   };

 *p++ = transparent;
 return p;
}


/*
 * Create new ImageData structure and allocate the buffers of proper size.
 */
ImageData* NewImageData(size_t theImageLength, size_t theImageMaskLength)
{
 ImageData* theImageData = NULL;
#ifdef ARENA_DEBUG
 char Iam[] = "NewImageData";
#endif


#ifdef ARENA_DEBUG
 if (theImageLength < 0)
   {
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " the given image length is incorrect.\n");
    return NULL;
   }

 if (theImageMaskLength < 0)
   {
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " the given image mask length is incorrect.\n");
    return NULL;
   }
# else
 if ((theImageLength < 0) || (theImageMaskLength < 0)) return NULL;
#endif

 if ((theImageData = (ImageData*)Arena_CAlloc(1, sizeof(ImageData), False))
     == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " failed to alloc memory for ImageData struct.\n");
# else
    Arena_PrintError(_("Cannot allocate space for image data structure.\n"));
#endif
    return NULL;
   }

 if (theImageLength)
   {
    if ((theImageData->image = (unsigned char*)Arena_CAlloc(theImageLength,
						         sizeof(unsigned char),
							    False))
	== NULL)
      {
#ifdef ARENA_DEBUG
       if (IMAGE_TRACE)
	 Arena_TracePrint(Iam, " failed to allocate memory for image data.\n");
# else
       Arena_PrintError(_("Cannot allocate space for image data.\n"));
#endif
       Free(theImageData);
       return NULL;
      }
   }
  else
   theImageData->image = NULL;

 if (theImageMaskLength)
   {
    if ((theImageData->mask = (unsigned char*)Arena_CAlloc(theImageMaskLength,
						        sizeof(unsigned char),
							   False))
	== NULL)
      {
#ifdef ARENA_DEBUG
       if (IMAGE_TRACE)
	 Arena_TracePrint(Iam, " failed to allocate memory for mask data.\n");
# else
       Arena_PrintError(_("Cannot allocate space for image mask data.\n"));
#endif
       Free(theImageData->image);
       Free(theImageData);
       return NULL;
      }
   }
  else
   theImageData->mask = NULL;

 return theImageData;
}


/*
 * Free ImageData structure with it's contents.
 */
void FreeImageData(ImageData* theImageData)
{
 if (theImageData)
   {
    Free(theImageData->mask);
    Free(theImageData->image);
    free(theImageData);
   }
}


/*
 * Create new ImageXyoke structure with no constituents.
 */
ImageXyoke* NewImageXyoke(void)
{
 ImageXyoke* theImageXcouple = NULL;
#ifdef ARENA_DEBUG
 char Iam[] = "NewImageXyoke";
#endif


 if ((theImageXcouple = (ImageXyoke*)Arena_MAlloc(sizeof(ImageXyoke), False)))
   {
    theImageXcouple->mask = theImageXcouple->image = None;
    return theImageXcouple;
   }
  else
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " Failed to alloc mem.\n");
#endif
    Warn(_("Failed to allocate space for image X yoke"));
    return NULL;
   }
}


/*
 * Free ImageXyoke structure with it's contents.
 */
void FreeImageXyoke(ImageXyoke* theImageXcouple)
{
 if (theImageXcouple)
   {
   /*
    * Be aware:
    * XDestroyImage() frees both the image structure AND
    * the data pointed to by the image structure.
    */
    if (theImageXcouple->image) XDestroyImage(theImageXcouple->image);
    if (theImageXcouple->mask)  XDestroyImage(theImageXcouple->mask);
    Free(theImageXcouple);
   }
}


/*
 * Create ImageData filling context.
 */
ImageAssembly* NewImageAssembly(ArenaImageingCapability imageing,
				unsigned int theWidth, unsigned int theHeight,
				unsigned int theDepth,
				unsigned int theMaskDepth)
{
#ifdef ARENA_DEBUG
 char Iam[] = "NewImageAssembly";
#endif


 if (theWidth && theHeight && theDepth)
   {
    unsigned int i_ppb, m_ppb;   /* pixels per byte */
    unsigned int i_bpp, m_bpp;   /* bytes per pixel */
    unsigned int i_bpl, m_bpl;   /* bytes per  line */
    unsigned long i_size, m_size;

    switch (theDepth)
      {
       case 1:
       case 2:
       case 4:
	 i_ppb = 8/theDepth;
         i_bpp = 0;
	 break;

       case 8:
	 i_ppb = 1;
	 i_bpp = 1;
	 break;

       case 16:
	 i_ppb = 0;
	 i_bpp = 2;
	 break;

       case 12:
       case 24:
	 i_ppb = 0;
	 i_bpp = 4;
	 break;

       default:
#ifdef ARENA_DEBUG
	 Arena_TracePrint(Iam, " Display depth %d unsupported.\n", theDepth);
# else
	 Arena_PrintError(_(" Images for display depth %d unsupported\n"),
			  theDepth);
#endif
	 return NULL;
	 break;
      }
    /* End ``switch (theDepth)'' */

    i_bpl  = i_bpp ? (theWidth * i_bpp) : ((theWidth + i_ppb - 1)/i_ppb);
    i_size = i_bpl * theHeight;

    switch (theMaskDepth)
      {
       case 0:
	 /* No image mask (no transparency) */
	 m_ppb = 0;
         m_bpp = 0;
	 m_bpl = 0;
	 break;

       case 1:
	 m_ppb = 8;
         m_bpp = 0;
	 m_bpl = (theWidth + m_ppb - 1)/m_ppb;
	 break;

       default:
#ifdef ARENA_DEBUG
	 Arena_TracePrint(Iam, " Mask depth %d unsupported.\n", theMaskDepth);
# else
	 Arena_PrintError(_(" Images masks of depth %d unsupported\n"),
			  theMaskDepth);
#endif
	 return NULL;
	 break;
      }
    /* End ``switch (theMaskDepth)'' */

    m_size = m_bpl ? (m_bpl * theHeight) : 0;


    /*
     * Allocate the image aasembly context structure and initialze it.
     */
    {
     ImageData* theImageData;

     if ((theImageData = NewImageData(i_size, m_size)))
       {
        ImageAssembly* theImageAssembly;


        if ((theImageAssembly = Arena_MAlloc(sizeof(ImageAssembly), False)))
	  {
#ifdef ARENA_DEBUG
	   if (IMAGE_TRACE && VERBOSE_TRACE)
	     Arena_TracePrint(Iam,
			      " %dx%d, imaging %d\n"
			      "\t           depth: %d/%d,\n"
			      "\t            size: %d/%d,\n"
			      "\tpixels per  byte: %d/%d,\n"
			      "\t bytes per pixel: %d/%d,\n"
			      "\t bytes per  line: %d/%d.\n",
			      theWidth, theHeight, imageing,
			      theDepth, theMaskDepth,
			      i_size, m_size,
			      i_ppb, m_ppb,
			      i_bpp, m_bpp,
			      i_bpl, m_bpl);
#endif
	   theImageAssembly->imageing = imageing;
	   theImageAssembly->width  = theWidth;
	   theImageAssembly->height = theHeight;
	   theImageAssembly->depth  = theDepth;
	   theImageAssembly->mask_depth  = theMaskDepth;
	   theImageAssembly->image_size  = i_size;
	   theImageAssembly->mask_size   = m_size;
	   theImageAssembly->image_ppb   = i_ppb;
	   theImageAssembly->mask_ppb    = m_ppb;
	   theImageAssembly->image_bpp   = i_bpp;
	   theImageAssembly->mask_bpp    = m_bpp;
	   theImageAssembly->image_bpl   = i_bpl;
	   theImageAssembly->mask_bpl    = m_bpl;
	   theImageAssembly->image_shift = 0;
	   theImageAssembly->mask_shift  = 0x01;
	   theImageAssembly->newbyte     = True;
	   theImageAssembly->x = 0;
	   theImageAssembly->y = 0;
	   theImageAssembly->image_p     = theImageData->image;
	   theImageAssembly->mask_p      = theImageData->mask;
	   theImageAssembly->data = theImageData;

	   return theImageAssembly;
	  }
         else
	  {
#ifdef ARENA_DEBUG
	   if (IMAGE_TRACE)
	     Arena_TracePrint(Iam, " no space for image assembly context.\n");
#endif
	   return NULL;
	  }
       }
      else
       {
#ifdef ARENA_DEBUG
        if (IMAGE_TRACE)
	  Arena_TracePrint(Iam, " no space left for image data.\n");
#endif
        return NULL;
       }
    }
   }
  else
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! Image size turns out to be 0.\n");
#endif
    return NULL;
   }
}


/*
 * Put next pixel to the ImageData in the context.
 */
Bool NextPixel_to_ImageAssembly(ImageAssembly* theImageAssembly,
				unsigned char* thePixelP,
				unsigned int thePixelMask)
{
#ifdef ARENA_DEBUG
 char Iam[] = "NextPixel_to_ImageAssembly";
#endif

 if (theImageAssembly)
   {
    if (theImageAssembly->y < theImageAssembly->height)
      {
       if (theImageAssembly->x < theImageAssembly->width)
	 {
	  if (theImageAssembly->mask_depth == 1)
	    {
	    /* Simple transparency support only */

	     /* First put next mask pixel */
	     if (thePixelMask)
	       {
		*(theImageAssembly->mask_p) |= theImageAssembly->mask_shift;
	       }
	      else
	       {
	       /* Transparent pixel */
		*(theImageAssembly->mask_p) &= ~(theImageAssembly->mask_shift);
	       }

	     /* Then advance mask running stuff */
	     if (theImageAssembly->mask_shift == 0x80)
	       {
		theImageAssembly->mask_shift = 0x01;
		(theImageAssembly->mask_p)++;
	       }
	      else
	       (theImageAssembly->mask_shift) <<= 1;
	    }


	  if (theImageAssembly->image_bpp)
	    {
	     int i;

	     for (i = theImageAssembly->image_bpp; i--; )
	       *(theImageAssembly->image_p)++ = *thePixelP++;
	    }
	   else
	    {
	     *(theImageAssembly->image_p) |= *thePixelP <<
					     theImageAssembly->image_shift;

	     theImageAssembly->image_shift += depth;
	     theImageAssembly->image_shift %= 8;
	     if (theImageAssembly->image_shift)
	       {
		theImageAssembly->newbyte = False;
	       }
	      else
	       {
		theImageAssembly->newbyte = True;
		*(++(theImageAssembly->image_p)) = 0;
	       }
	    }

	  theImageAssembly->x++;

	  if (theImageAssembly->x == theImageAssembly->width)
	    {
	     theImageAssembly->y++;
	     theImageAssembly->x = 0;

	     if (theImageAssembly->image_bpp == 0)
	       {
		if (theImageAssembly->image_shift)
		  {
		  /* make sure we start on a new byte for the new line */
		   theImageAssembly->image_shift = 0;
		   (theImageAssembly->image_p)++;
		   theImageAssembly->newbyte = True;
		  }
	       }

	     if (theImageAssembly->mask_shift != 0x01)
	       {
	       /* Force aligning to byte border */
		theImageAssembly->mask_shift = 0x01;
		(theImageAssembly->mask_p)++;
	       }
	    }

	  return True;
	 }
        else
	 {
#ifdef ARENA_DEBUG
	  if (IMAGE_TRACE)
	    Arena_TracePrint(Iam,
			     " ERROR! The indent too large (%d >= %d).\n",
			     theImageAssembly->x, theImageAssembly->width);
#endif
	  return False;
	 }
      }
     else
      {
#ifdef ARENA_DEBUG
       if (IMAGE_TRACE)
	 Arena_TracePrint(Iam,
			  " ERROR! The offset too large (%d >= %d).\n",
			  theImageAssembly->y, theImageAssembly->height);
#endif
       return False;
      }
   }
  else
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The image assembly context is NULL.\n");
#endif
    return False;
   }
}


/*
 * Destroy ImageData filling context and return the resulting ImageData.
 */
ImageData* FinishImageAssembly(ImageAssembly* theImageAssembly)
{
#ifdef ARENA_DEBUG
 char Iam[] = "FinishImageAssembly";
#endif

 if (theImageAssembly)
   {
    ImageData* theImageData = theImageAssembly->data;

    Free(theImageAssembly);
    return theImageData;
   }
  else
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The image assembly context is NULL.\n");
#endif
    return NULL;
   }
}


/*
 * Create XImage combination from image data.
 * Cut off the ImageData buffers --- this is a preventive measure:
 *   it's necessary as XCreateImage() function
 *   just links the provided buffer into created XImage structure,
 *   i.e. moving the data buffer from image data domain of control
 *   to the XImage domain, thus as XDestroyImage() frees XImage structure
 *   with all it's subordinate datum, it would also free the data buffer.
 */
ImageXyoke* processImage_data2image(ImageData* theImageData,
				    unsigned int theWidth,
				    unsigned int theHeight,
				    unsigned int depth)
{
 ImageXyoke* theImageXpair = NULL;
#ifdef ARENA_DEBUG
 char Iam[] = "processImage_data2image";
#endif


 if (theImageData == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The image is NULL.\n");
#endif
    return NULL;
   }

 if (theImageData->image == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The image data is NULL.\n");
#endif
    return NULL;
   }


 if ((theImageXpair = NewImageXyoke()) == NULL)
   {
    Warn(_("Failed to allocate new image X pair"));
    return NULL;
   }

 if ((theImageXpair->image = XCreateImage(display,
					  DefaultVisual(display, screen),
					  depth, ZPixmap, 0,
					  theImageData->image,
					  theWidth, theHeight,
					  (depth == 24 ?
					   32 : (depth == 16 ? 16 : 8)),
					  0))
     == 0)
   {
    Warn(_("Failed to create XImage"));
    return NULL;
   }

 /* QingLong.28-04-97: we should set these */
 theImageXpair->image->byte_order       = LSBFirst;
 theImageXpair->image->bitmap_bit_order = LSBFirst;

 /* QingLong.12-09-97: Cut off the image data buffer from ImageData */
 theImageData->image = NULL;

 if (theImageData->mask)
   {
    if ((theImageXpair->mask = XCreateImage(display,
					    DefaultVisual(display, screen),
					    1, ZPixmap, 0, theImageData->mask,
					    theWidth, theHeight,
					    8, 0))
	== 0)
      {
       Warn(_("Failed to create mask XImage"));
       return NULL;
      }

    /* QingLong.28-04-97: we should set these */
    theImageXpair->mask->byte_order       = LSBFirst;
    theImageXpair->mask->bitmap_bit_order = LSBFirst;

    /* QingLong.12-09-97: Cut off the image mask data buffer from ImageData */
    theImageData->mask = NULL;
   }

 return theImageXpair;
}


/*
 * Create XImage from image pair. The returned buffers are malloc'ed.
 */
Bool XprocessLoadedImage(ImageXyoke* theImageXcouple, Image* theImage)
{
 int theWidth = 0, theHeight = 0;
 Pixmap pixmap, mask_pixmap = None;
 GC drawGC;
#ifdef ARENA_DEBUG
 char Iam[] = "XprocessLoadedImage";
#endif


 if (theImageXcouple == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The image is NULL.\n");
#endif
    return False;
   }

 if (theImageXcouple->image == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The image data is NULL.\n");
#endif
    return False;
   }

 if (theImage == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! The output image is NULL.\n");
#endif
    return False;
   }


 theWidth  = theImageXcouple->image->width;
 theHeight = theImageXcouple->image->height;

#ifdef ARENA_DEBUG
 if ((theWidth <= 0) || (theHeight <= 0))
   {
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		       " ERROR! Invalid image geometry: %d x %d.\n",
		       theWidth, theHeight);
    return False;
   }
#endif


 if ((pixmap = XCreatePixmap(display, win, theWidth, theHeight, depth))
     == None)
   {
    Warn(_("Failed to create Pixmap"));
    return False;
   }

 if (theImageXcouple->mask)
   {
    if ((mask_pixmap = XCreatePixmap(display, win, theWidth, theHeight, 1))
	== None)
      {
       Warn(_("Failed to create mask Pixmap:"));
       return False;
      }
   }


 drawGC = XCreateGC(display, pixmap, 0, NULL);
 XSetFunction(display, drawGC, GXcopy);
 XPutImage(display, pixmap, drawGC, theImageXcouple->image,
	   0, 0, 0, 0, theWidth, theHeight);
 XFreeGC(display, drawGC);


 if (mask_pixmap)
   {
    drawGC = XCreateGC(display, mask_pixmap, 0, NULL);
    XPutImage(display, mask_pixmap, drawGC, theImageXcouple->mask,
	      0, 0, 0, 0, theWidth, theHeight);
    XFreeGC(display, drawGC);
   }


 theImage->pixmap = pixmap;
 theImage->mask   = mask_pixmap;

 theImage->width  = theWidth;
 theImage->height = theHeight;

 return True;
}


/*
 * create a textured background as paper
 */
ImageData* CreateBackground(unsigned int width, unsigned int height,
			    unsigned int depth)
{
 unsigned char *data, *p;
 ImageData* arena_bg_image_data;
 int size, i, j;
 unsigned long cs[3]; /* howcome 21/9/94 */
 unsigned long int ulp;

 if (depth == 8)
   {
   /* howcome 4/10/94: changed last arg to GetColour */

    int i;
    for (i = 0; i < 3; i++)
      {
       int r, g, b;
       r = paperrgb[i].red;
       g = paperrgb[i].green;
       b = paperrgb[i].blue;
       if (!GetColour(r, g, b, &cs[i])) return NULL;
       paperrgb[i].pixel = cs[i];
       papercols[cs[i]].red   = r;
       papercols[cs[i]].green = g;
       papercols[cs[i]].blue  = b;
      }

    size = width * height;
   }
 else
   if (depth == 24)
     size = width * height * 4;
    else
     if (depth == 16)
       size = width * height *2;
      else
       return NULL;

 arena_bg_image_data = NewImageData(size, 0);
 if (arena_bg_image_data == NULL) return NULL;

 p = data = arena_bg_image_data->image;

 srand(0x6000);

 if (depth == 8)
   {
    for (i = 0; i < height; ++i)
      for (j = 0; j < width; ++j)
	{
	/* howcome 21/9/94:
	   rand returns different ranges on different platforms, therefore: */

	  *p++ = paperrgb[rand() % 3].pixel;
	}
   }
  else
   if (depth == 24)
     {
      for (i = 0; i < height; ++i)
	for (j = 0; j < width; ++j)
	  {
	   int col,r,g,b;

	   col = rand() % 3;

	   r = paperrgb[col].red;
	   g = paperrgb[col].green;
	   b = paperrgb[col].blue;

	   GetColour(r, g, b, &ulp);

	   *p++ = (ulp      ) & 0xFF;   /* LSB first! */
	   *p++ = (ulp >>  8) & 0xFF;
	   *p++ = (ulp >> 16) & 0xFF;
	   *p++ = '\0';
	  }
     }
    else
     if (depth == 16) 
       {
        for (i = 0; i < height; ++i)
	  for (j = 0; j < width; ++j)
            {
	     int col,r,g,b;

	     col = rand() % 3;
	     r = paperrgb[col].red;
	     g = paperrgb[col].green;
	     b = paperrgb[col].blue;

	     GetColour(r, g, b, &ulp);

	     *p++ = ((char*)&ulp)[0];   /* LSB first! */
	     *p++ = ((char*)&ulp)[1];
            }
       }

 return arena_bg_image_data;
}


Bool MakePaper(void)
{
 gc_fill = XCreateGC(display, win, 0, 0);
 XSetFunction(display, gc_fill, GXcopy);

 if (UsePaper)
   {
    if (depth == 8 || depth == 16 || depth == 24)
      {
       ImageData* arena_bg_image_data;

       tileWidth = tileHeight = 64;

       /* howcome 15/3/95: commented out XQueryTile since it on some
	  platforms set tile* to 0. On other servers it returns a low
	  number which dosn't look good */

       /* XQueryBestTile(display, win,
	                 tileWidth, tileHeight, &tileWidth, &tileHeight); */

       arena_bg_image_data = CreateBackground(tileWidth, tileHeight, depth);
       if (arena_bg_image_data)
	 {
	  ImageXyoke* arena_bg_image_yoke;

	  /* As the processImage_data2image() cuts off `mask' and
	   * `data' fields from ImageData structure,
           * we should set `tileData' pointer to the image data buffer
           * prior to call to processImage_data2image().
	   */
	  tileData = arena_bg_image_data->image;

	  arena_bg_image_yoke = processImage_data2image(arena_bg_image_data,
							tileWidth, tileHeight,
							depth);
	  if (arena_bg_image_yoke)
	    {
	     if (arena_bg_image == NULL) arena_bg_image = NewImage();
	     if (arena_bg_image)
	       {
		if (XprocessLoadedImage(arena_bg_image_yoke, arena_bg_image))
		  {
		   XGCValues values;
		   unsigned int valuemask;


		   /* Ensure that the image data is cut off before freeing
		    * to avoid image data freeing, as we need it for tileData.
		    */
		   arena_bg_image_data->image = NULL;
		   FreeImageData(arena_bg_image_data);

		   /* FreeImageXyoke() uses XDestroyImage(),
		    * which frees both the image structure AND
		    * the data pointed to by the image structure.
		    * As we already have freed `arena_bg_image_data' by
		    * call to FreeImageData(), we should cut `data' fileds from
		    * `image' and `mask' fields of `arena_bg_image_yoke' prior
		    * to call to FreeImageXyoke().
		    */
		   if (arena_bg_image_yoke->image)
		     arena_bg_image_yoke->image->data = NULL;
		   if (arena_bg_image_yoke->mask)
		     arena_bg_image_yoke->mask->data  = NULL;
		   FreeImageXyoke(arena_bg_image_yoke);

		   valuemask = GCTile|GCFillStyle;
		   values.tile = arena_bg_image->pixmap;
		   values.fill_style = FillTiled;
		   XChangeGC(display, gc_fill, valuemask, &values);

		   return True;
		  }
	       }
	    }
	 }
      }
    /* End ``if (depth == 8 || depth == 16 || depth == 24)'' */

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    Arena_TracePrint("MakePaper",
		     " Failed to create background paper!\n");
# else
    Arena_PrintError(_("Failed to create background paper!\n"));
#endif
    Free(tileData);
    UsePaper = False;
   }
 /* End ``if (UsePaper)'' */

 XSetForeground(display, gc_fill, windowColour);
 return False;
}


ProcessImage ProcessLoadedImage(Doc* doc)
{
 Image* image = NULL;
 ImageXyoke* imageXcouple = NULL;
 Block block;
 HTAtom* a = NULL;
#ifdef ARENA_DEBUG
 char Iam[] = "ProcessLoadedImage";
#endif

 if (doc)
   if (doc->anchor)
     a = HTAnchor_format(HTAnchor_parent((HTAnchor*)(doc->anchor)));

 if (a == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		       " doc "POINTER_FORMAT" MIME type UNdefined!\n", doc);
#endif
    return FAILURE;
   }

#ifdef ARENA_DEBUG
 if (IMAGE_TRACE && VERBOSE_TRACE)
   Arena_TracePrint(Iam,
		    " doc "POINTER_FORMAT" has type \"%s\".\n", doc, a->name);
#endif

 block.next = 0;  /*NewDoc.hdrlen; */
 block.size = doc->loaded_length; /* NewDoc.length; */
 block.buffer = doc->content_buffer;

 if (doc->loaded_length == 0)
   {
    doc->state = DOC_REJECTED;
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      {
       Arena_TracePrint(Iam,
			" doc "POINTER_FORMAT" has zero length contents!\n",
			doc);
       if (doc->image)
	 Arena_TracePrint(Iam,
			  " ERROR!? doc "POINTER_FORMAT
			  " image "POINTER_FORMAT" (non NULL).\n",
			  doc, doc->image);
      }
#endif
    return FAILURE;
   }

 if (doc->content_buffer == NULL)
   {
   /* probably externally viewed image */
    doc->state = DOC_EXTERNAL;
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      {
       Arena_TracePrint(Iam, " doc "POINTER_FORMAT" has no contents!\n", doc);
       if (doc->image)
	 Arena_TracePrint(Iam,
			  " ERROR!? doc "POINTER_FORMAT
			  " image "POINTER_FORMAT" (non NULL).\n",
			  doc, doc->image);
      }
#endif
    return FAILURE;
   }

 if ((image = NewImage()))
   {
    image->npixels = 0;
   }
  else
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " ERROR! Failed to allocate new image struct!\n");
#endif
    Warn(_("Failed to alloc X image"));
    return FAILURE;
   }

 if (a == gif_atom)
   {
    if ((imageXcouple = LoadGIFimage(&block, depth)) == NULL)
      {
       Warn(_("Failed to load GIF image: %s"), doc->url);
       Free(block.buffer);
       doc->state = DOC_REJECTED;
       return FAILURE;
      }
   }
  else
   if (a == xpm_atom)
     {
      if ((imageXcouple = LoadXPMimage(&block, depth)) == NULL)
	{
	 Warn(_("Failed to load XPM image: %s"), doc->url);
	 Free(block.buffer);
	 doc->state = DOC_REJECTED;
	 return FAILURE;
	}
     }
    else
     if (a == xbm_atom)
       {
	if ((imageXcouple = LoadXBMimage(&block, depth)) == NULL)
	  {
	   Warn(_("Failed to load XBM image: %s"), doc->url);
	   Free(block.buffer);
	   doc->state = DOC_REJECTED;
	   return FAILURE;
	  }
       }
#ifdef JPEG
      else
       if (a == jpeg_atom)
	 {
	  if ((imageXcouple = LoadJPEGimage(&block, depth)) == NULL)
	    {
	     Warn(_("Failed to load JPEG image: %s"), doc->url);
	     Free(block.buffer);
	     doc->state = DOC_REJECTED;
	     return FAILURE;
	    }
	 }
#endif /* JPEG */

#ifdef PNG
        else
	 if (a == png_atom || a == png_exp_atom)
	   {
	    if ((imageXcouple = LoadPNGimage(&block, depth)) == NULL)
	      {
	       Warn(_("Failed to load PNG image: %s"), doc->url);
	       /* Free(block.buffer); */
	       doc->state = DOC_REJECTED;
	       return FAILURE;
	      }
	   }
#endif /* PNG */

	  else
	   {
	    Warn(_("Failed to load unknown image format: %s"), doc->url);
            Free(block.buffer);
            doc->state = DOC_REJECTED;
            return FAILURE;
	   }

 Free(block.buffer);

 doc->content_buffer = NULL; /* howcome 4/12/94:
				no need to keep the image source around ??? */


 if (XprocessLoadedImage(imageXcouple, image))
   {
    doc->image = image;
    doc->state = DOC_PROCESSED;
    FreeImageXyoke(imageXcouple);
    return NEWIMAGE;
   }
  else
   {
    Warn(_("Failed to create X image: %s"), doc->url);
    DelImage(image);
   }

 doc->image = NULL;
 doc->state = DOC_REJECTED;
 FreeImageXyoke(imageXcouple);
 return FAILURE;
}


Bool InitProgressiveImage(Doc *doc)
{
 Bool TotalSuccess = True;
 HTFormat format = NULL;

 if (doc == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint("InitProgressiveImage", " the given doc is NULL.\n");
#endif
    return False;
   }

 if (doc->anchor == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint("InitProgressiveImage",
		       " the doc "POINTER_FORMAT" has no anchor.\n", doc);
#endif
    return False;
   }

 format = HTAnchor_format(HTAnchor_parent((HTAnchor*)(doc->anchor)));

 if (doc->image)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint("InitProgressiveImage",
		       "\n\tthe given doc "POINTER_FORMAT
		       " has already been initialized?\n",
		       doc);
#endif
    return False;
   }

 if ((doc->image = NewImage()) == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint("InitProgressiveImage",
		       " failed to malloc for the doc "POINTER_FORMAT".\n",
		       doc);
#endif
    return False;
   }

 doc->image->npixels = 0;

#ifdef PROGRESSIVE_PNG
 if (format == png_atom || format == png_exp_atom)
   {
    if ((TotalSuccess = InitProgressivePNGimage(doc->image)))
      Announce(_("Processing PNG image %s..."), doc->url); 
   }
#endif

 return TotalSuccess;
}


ProcessImage ProcessProgressiveImage(Doc* doc,
				     unsigned long thechunk_begin,
				     unsigned long thechunk_length)
{
 ImageXyoke* imageXcouple = NULL;
 Image* image;
 Block block;
 HTAtom* a = NULL;
#ifdef ARENA_DEBUG
 char Iam[] = "ProcessProgressiveImage";
#endif


 if (doc == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " the doc is NULL.\n");
#endif
    return FAILURE;
   }

 if (doc->anchor == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		       " the given doc has no anchor.\n");
#endif
    return FAILURE;
   }

 a = HTAnchor_format(HTAnchor_parent((HTAnchor*)(doc->anchor)));

 if ((image = doc->image) == NULL)
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam,
		       " doc "POINTER_FORMAT" image is not initialized?\n",
		       doc);
#endif
    return FAILURE;
   }

 block.next   = (size_t)thechunk_begin;	/* NewDoc.hdrlen; */
 block.size   = doc->loaded_length;	/* NewDoc.length; */
 block.buffer = doc->content_buffer;

 if (doc->content_buffer == NULL)	/* probably externally viewed image */
   {
    doc->state = DOC_EXTERNAL;
    return FAILURE;
   }

#ifdef PROGRESSIVE_PNG
 if (a == png_atom || a == png_exp_atom)
   {
    if ((imageXcouple = ProcessProgressivePNGimage(doc->image,
						   &block, depth,
						   thechunk_length))
	== NULL)
      {
       Warn(_("Failed to load PNG image: %s"), doc->url);
#  if 0
       Free(block.buffer);
#  endif
       doc->state = DOC_REJECTED;
#  ifdef ARENA_DEBUG
       if (IMAGE_TRACE && VERBOSE_TRACE)
	 if (doc->image)
	   Arena_TracePrint(Iam,
			    " freeing the PNG image "POINTER_FORMAT".\n",
			    doc->image);
#  endif
       Free(doc->image);
       return FAILURE;
      }
   }
  else
   {
    Warn(_("Failed to load unknown image format: %s"), doc->url);
    Free(block.buffer);
    doc->state = DOC_REJECTED;
#  ifdef ARENA_DEBUG
    if (IMAGE_TRACE && VERBOSE_TRACE)
      if (doc->image)
	Arena_TracePrint(Iam,
			 " freeing the image "POINTER_FORMAT".\n",
			 doc->image);
#  endif
    Free(doc->image);
    return FAILURE;
   }
# else
 return ProcessLoadedImage(doc);
#endif /* PNG */


 if (image)
   {
    image->npixels = 0;

    if (XprocessLoadedImage(imageXcouple, image))
      {
       doc->image = image;
       doc->state = DOC_PROCESSED;
       FreeImageXyoke(imageXcouple);
       return NEWIMAGE;
      }
     else
      {
       Warn(_("Failed to create X image: %s"), doc->url);
       DelImage(image);
      }
   }
  else
   {
    Warn(_("Failed to alloc X image"));
   }

 doc->image = NULL;
 doc->state = DOC_REJECTED;
 FreeImageXyoke(imageXcouple);
 return FAILURE;
}


Image* GetImage(char *href, int hreflen, Bool reload)
{
 Doc* doc = NULL;
#ifdef ARENA_DEBUG
 char Iam[] = "GetImage";
#endif

 /* check for null name */

 if ((doc = GetInline(href, hreflen, reload)))
   {
    if (doc->image)
      return (doc->image);
     else
      {
       if (doc->state == DOC_LOADED)
	 return ((ProcessLoadedImage(doc) == FAILURE) ? NULL : doc->image);
        else
	 {
#ifdef ARENA_DEBUG
	  if (IMAGE_TRACE)
	    Arena_TracePrint(Iam,
			     " The doc "POINTER_FORMAT" isn't lodaded"
			     " (state %d).\n",
			     doc, doc->state);
#endif
	  return NULL;
	 }
      }
   }

 return NULL;
}



void FreeImages(int cloned)
{
 Image* im;

 while (images)
   {
   /* deallocate colours */

    if (!cloned && images->npixels > 0)
      XFreeColors(display, colourmap, images->pixels, images->npixels, 0);

    /* free pixmap and image structure */

    if (!cloned && images->pixmap != default_pixmap)
      XFreePixmap(display, images->pixmap);

    im = images;
    images = im->next;
    Free(im->url);

    if (im->npixels > 0) Free(im->pixels);

#ifdef ARENA_DEBUG
    if (IMAGE_TRACE && VERBOSE_TRACE)
      if (im)
	Arena_TracePrint("FreeImages",
			 " freeing the image "POINTER_FORMAT".\n", im);
#endif
    Free(im);
   }
}


/* don't add this to images list to avoid trouble when
 * freeing document images
 */
Image* MakeIcon(char* name,
		char* bits,
		unsigned int width, unsigned int height,
		unsigned int depth)
{
 Image* image = NULL;
 ImageXyoke* theImageXcouple = NULL;
#ifdef ARENA_DEBUG
 char Iam[] = "MakeIcon";
#endif

 if (width == 0 || height == 0) return NULL;

 if ((image = NewImage()))
   {
    if ((theImageXcouple = ProcessXBM(bits, width, height, depth)))
      {
       if (XprocessLoadedImage(theImageXcouple, image))
	 {
	  Free(theImageXcouple);

	  image->url     = name;
	  image->npixels = 0;
	  image->pixels  = 0;
	  image->next    = NULL;

	  return image;
	 }
        else
	 {
#ifdef ARENA_DEBUG
	  if (IMAGE_TRACE)
	    Arena_TracePrint(Iam,
			     " failed to create X image for icon \"%s\".\n",
			     name);
#endif
	  Free(theImageXcouple);
	  Free(image);
	  return NULL;
	 }
      }
     else
      {
#ifdef ARENA_DEBUG
       if (IMAGE_TRACE)
	 Arena_TracePrint(Iam,
			  " failed to process XBM image for icon \"%s\".\n",
			  name);
#endif
       Free(image);
       return NULL;
      }
   }
  else
   {
#ifdef ARENA_DEBUG
    if (IMAGE_TRACE)
      Arena_TracePrint(Iam, " failed to alloc image for icon \"%s\".\n", name);
#endif
    return NULL;
   }
}


void MakeIcons(unsigned int depth)
{
 extern Image* note_image;
 extern Image* caution_image;
 extern Image* warning_image;


 note_image    = MakeIcon("note", note_bits,
			  (unsigned int)note_width,
			  (unsigned int)note_height,
			  depth);
 caution_image = MakeIcon("caution", caution_bits,
			  (unsigned int)caution_width,
			  (unsigned int)caution_height,
			  depth);
 warning_image = MakeIcon("warning", warning_bits,
			  (unsigned int)warning_width,
			  (unsigned int)warning_height,
			  depth);
}
