/*
    libparted
    Copyright (C) 1998-2000  Andrew Clausen  <clausen@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    I can also be contacted at:

    Andrew Clausen
    18 Shaw St
    Ashwood, 3147
    Victoria, Australia

*/

#include "fat.h"
#include "traverse.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* extremely ugly hack: stick everything that obviously isn't an unmovable file
 * in here.  Note: DAT is a bit dubious.  Unfortunately, it's used by the
 * registry, so it'll be all over the place :-(
 */
static char*	movable_extensions[] = {
	"",
	"1ST",
	"AVI",
	"BAK", "BAT", "BMP",
	"CFG", "COM", "CSS",
	"DAT", "DLL", "DOC", "DRV",
	"EXE",
	"FAQ", "FLT", "FON",
	"GID", "GIF",
	"HLP", "HTT", "HTM",
	"ICO", "INI",
	"JPG",
	"LNK", "LOG",
	"KBD",
	"ME", "MID", "MSG",
	"OCX", "OLD",
	"PIF", "PNG", "PRV",
	"RTF",
	"SCR", "SYS",
	"TMP", "TTF", "TXT",
	"URL",
	"WAV",
	"VBX", "VOC", "VXD",
	NULL
};

static char*
get_extension (char* file_name)
{
	char*		ext;

	ext = strrchr (file_name, '.');
	if (!ext)
		return "";
	if (strchr (ext, '\\'))
		return "";
	return ext + 1;
}

static int
is_movable_system_file (char* file_name)
{
	char*		ext = get_extension (file_name);
	int		i;

	for (i = 0; movable_extensions [i]; i++) {
		if (strcasecmp (ext, movable_extensions [i]) == 0)
			return 1;
	}

	return 0;
}

/*
    prints out the sequence of clusters for a given file chain, beginning
    at start_cluster.
*/
static void
print_chain (PedFileSystem* fs, FatCluster start)
{
	FatSpecific*	fs_info = FAT_SPECIFIC (fs);
	FatCluster	clst;
	int		this_row;

	this_row = 0;
	for (clst = start; !fat_table_is_eof (fs_info->fat, clst);
	     clst = fat_table_get (fs_info->fat, clst)) {
		printf ("  %d", clst);
		if (++this_row == 7) {
			printf ("\n");
			this_row = 0;
		}
	}
	printf ("\n");
}

/*
    traverse the FAT for a file/directory, marking each entry's flag
    to "flag".
*/
static int
flag_traverse_fat (PedFileSystem* fs, const char* chain_name, FatCluster start,
		   int flag)
{
	FatSpecific*	fs_info = FAT_SPECIFIC (fs);
	FatCluster	clst;

	for (clst = start; !fat_table_is_eof (fs_info->fat, clst);
	     clst = fat_table_get (fs_info->fat, clst)) {
		if (!clst) {
			ped_exception_throw (PED_EXCEPTION_FATAL,
				PED_EXCEPTION_CANCEL,
				_("Bad FAT: unterminated chain for %s"),
				chain_name);
			return 0;
		}

		if (clst >= fs_info->fat->cluster_count + 2) {
			ped_exception_throw (PED_EXCEPTION_FATAL,
				PED_EXCEPTION_CANCEL,
				_("Bad FAT: cluster %d outside filesystem "
				  "in chain for %s"),
				(int) clst, chain_name);
			return 0;
		}

		if (fs_info->fat_flag_map [clst]) {
			ped_exception_throw (PED_EXCEPTION_FATAL,
				PED_EXCEPTION_CANCEL,
				_("Bad FAT: cluster %d is cross-linked for %s"),
				(int) clst, chain_name);
			return 0;
		}

		if (flag == FAT_FLAG_DIRECTORY)
			fs_info->total_dir_clusters++;
		fs_info->fat_flag_map [clst] = flag;
	}

	return 1;
}

/*
    recursively traverses a directory, flagging all clusters in the process.
    It frees the traverse_info structure before returning.
*/
static int
flag_traverse_dir (FatTraverseInfo* trav_info) {
	PedFileSystem*		fs = trav_info->fs;
	FatDirEntry*		this_entry;
	FatTraverseInfo*	subdir_trav_info;
	char			file_name [512];
	char*			file_name_start;
	FatCluster		first_cluster;

	strcpy (file_name, trav_info->dir_name);
	file_name_start = file_name + strlen (file_name);

	while ( (this_entry = fat_traverse_next_dir_entry (trav_info)) ) {
		if (!fat_dir_entry_has_first_cluster (this_entry, fs))
			continue;
		if (this_entry->name [0] == '.')
			continue;	/* skip . and .. entries */

		fat_dir_entry_get_name (this_entry, file_name_start);
		first_cluster = fat_dir_entry_get_first_cluster(this_entry, fs);
        
#ifdef VERBOSE
		printf ("%s: ", file_name);
		print_chain (fs, first_cluster);
#endif

		if (fat_dir_entry_is_system_file (this_entry)
		    && !is_movable_system_file (file_name)) {
			if (ped_exception_throw (
				PED_EXCEPTION_WARNING,
				PED_EXCEPTION_IGNORE_CANCEL,
				_("The file %s is marked as a system file.  "
				"This means moving it could cause some "
				"programs to stop working."),
				file_name)
					!= PED_EXCEPTION_IGNORE)
				return 0;
		}

		if (fat_dir_entry_is_directory (this_entry)) {
			if (!flag_traverse_fat (fs, file_name, first_cluster,
						FAT_FLAG_DIRECTORY))
				return 0;

			subdir_trav_info = fat_traverse_directory (trav_info,
								   this_entry);
			if (!flag_traverse_dir (subdir_trav_info))
				return 0;
		} else if (fat_dir_entry_is_file (this_entry)) {
			if (!flag_traverse_fat (fs, file_name, first_cluster,
						FAT_FLAG_FILE)) 
				return 0;
		}
	}

	fat_traverse_complete (trav_info);
	return 1;
}

/*  
    fills in fat_flag_map.  Each FAT entry is flagged as either
    FAT_FLAGS_FREE, FAT_FLAGS_FILE or FAT_FLAGS_DIRECTORY.
*/
int
fat_flag_clusters (PedFileSystem* fs) {
	FatSpecific*		fs_info = FAT_SPECIFIC (fs);
	FatTraverseInfo*	trav_info;
    
	/* set all clusters to unused as a default */
	memset (fs_info->fat_flag_map, FAT_FLAG_FREE,
		fs_info->fat->cluster_count + 2);
	fs_info->total_dir_clusters = 0;

	if (fs_info->fat_type == FAT_TYPE_FAT32) {
		trav_info = fat_traverse_begin (fs, fs_info->root_cluster,
						"\\");
		if (!flag_traverse_dir (trav_info))
			return 0;
		if (!flag_traverse_fat (fs, _("the root directory"),
                                        fs_info->root_cluster,
                                        FAT_FLAG_DIRECTORY))
			return 0;
	} else {
		trav_info = fat_traverse_begin (fs, FAT_ROOT, "\\");
		if (!flag_traverse_dir (trav_info))
			return 0;
	}
	return 1;
}

