/*
    ext2_mkfs.c -- ext2 fs creator
    Copyright (C) 1999 Lennert Buytenhek <buytenh@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char _ext2_mkfs_c[] = "$Id: ext2_mkfs.c,v 1.4 1999/10/10 07:57:55 buytenh Exp $";

#define USE_EXT2_IS_DATA_BLOCK
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <uuid/uuid.h>
#include "ext2.h"

/* formula grabbed from linux ext2 kernel source */
static __inline__ int is_root(int x, int y)
{
	if (!x)
		return 1;

	while (1)
	{
		if (x == 1)
			return 1;

		if (x % y)
			return 0;

		x /= y;
	}
}

static __inline__ int is_group_sparse(int sparsesbfs, int group)
{
	if (!sparsesbfs)
		return 1;

	if (is_root(group, 3) || is_root(group, 5) || is_root(group, 7))
		return 1;

	return 0;
}

/* has implicit parameter 'sb' !! */
#define is_sparse(group) is_group_sparse(sb->s_feature_ro_compat & EXT2_FEATURE_RO_COMPAT_SPARSE_SUPER, (group))

int ext2_mkfs_write_main(struct ext2_dev_handle *handle,
			 struct ext2_super_block *sb,
			 struct ext2_group_desc *gd)
{
	int freeit;
	int i;
	int numgroups;
	int gdblocks;
	unsigned char *sbbuf;

	freeit = 0;
	sbbuf = (unsigned char *)sb;
	if (sb->s_log_block_size)
	{
		sbbuf = malloc(1024 << sb->s_log_block_size);
		memset(sbbuf, 0, 1024 << sb->s_log_block_size);
		memcpy(sbbuf+1024, sb, 1024);
		freeit = 1;
	}

	numgroups = howmany(sb->s_blocks_count - sb->s_first_data_block, sb->s_blocks_per_group);
	gdblocks = howmany(numgroups * sizeof(struct ext2_group_desc), 1024 << sb->s_log_block_size);

	for (i=0;i<numgroups;i++)
	{
		if (is_sparse(i))
		{
			int offset;

			offset = sb->s_first_data_block + i * sb->s_blocks_per_group;

			if (!handle->ops->write(handle->cookie, sbbuf,
					        offset, 1))
				return 0;
			if (!handle->ops->write(handle->cookie, gd, offset+1,
						gdblocks))
				return 0;
		}
	}

	if (freeit)
		ped_free(sbbuf);
	return 1;
}

int ext2_mkfs_write_meta(struct ext2_dev_handle *handle,
			 struct ext2_super_block *sb,
			 struct ext2_group_desc *gd)
{
	int blocksize;
	int gdtsize;
	int i;
	int itsize;
	int numgroups;
	unsigned char *bb;
	unsigned char *ib;
	unsigned char *zero;

	blocksize = 1 << (sb->s_log_block_size + 13);

	numgroups = howmany(sb->s_blocks_count - sb->s_first_data_block,
			    sb->s_blocks_per_group);
	itsize = howmany(sizeof(struct ext2_inode) * sb->s_inodes_per_group,
			 (1024 << sb->s_log_block_size));
	gdtsize = howmany(sizeof(struct ext2_group_desc) * numgroups,
			  (1024 << sb->s_log_block_size));

	bb = ped_malloc(1024 << sb->s_log_block_size);
	if (!bb) goto error;
	ib = ped_malloc(1024 << sb->s_log_block_size);
	if (!ib) goto error_free_bb;
	zero = ped_malloc((1024 << sb->s_log_block_size) * itsize);
	if (!zero) goto error_free_zero;

	memset(zero, 0, (1024 << sb->s_log_block_size) * itsize);

	for (i=0;i<numgroups;i++)
	{
		int admin;
		blk_t bbblock;
		int groupsize;
		int groupoffset;
		blk_t ibblock;
		int j;

		groupoffset = i*sb->s_blocks_per_group + sb->s_first_data_block;
		groupsize = min(sb->s_blocks_count - groupoffset, sb->s_blocks_per_group);

		admin = itsize + 2;
		bbblock = groupoffset;
		ibblock = groupoffset + 1;
		if (is_sparse(i))
		{
			admin += gdtsize + 1;
			bbblock = groupoffset + gdtsize + 1;
			ibblock = groupoffset + gdtsize + 2;
		}

		{
			memset(bb, 0, 1024 << sb->s_log_block_size);
			if (is_sparse(i))
				for (j=0;j<gdtsize+1;j++)
					bb[j>>3] |= _bitmap[j&7];

			j = bbblock - groupoffset;
			bb[j>>3] |= _bitmap[j&7];

			j = ibblock - groupoffset;
			bb[j>>3] |= _bitmap[j&7];

			for (j=0;j<itsize;j++)
			{
				int k = j + gdtsize + 3;

				bb[k>>3] |= _bitmap[k&7];
			}

			for (j=groupsize;j<blocksize;j++)
				bb[j>>3] |= _bitmap[j&7];

			if (!handle->ops->write(handle->cookie, bb, bbblock, 1))
				goto error_free_zero;
		}

		{
			memset(ib, 0, 1024 << sb->s_log_block_size);

			for (j=sb->s_inodes_per_group;j<blocksize;j++)
				bb[j>>3] |= _bitmap[j&7];

			if (!handle->ops->write(handle->cookie, ib, ibblock, 1))
				goto error_free_zero;
		}

		if (!handle->ops->write(handle->cookie, zero,
					groupoffset + gdtsize + 3, itsize))
			goto error_free_zero;

		gd[i].bg_block_bitmap = bbblock;
		gd[i].bg_inode_bitmap = ibblock;
		gd[i].bg_inode_table = groupoffset + gdtsize + 3;
		gd[i].bg_free_blocks_count = groupsize - admin;
		gd[i].bg_free_inodes_count = sb->s_inodes_per_group;
		gd[i].bg_used_dirs_count = 0;
		gd[i].bg_pad = 0;
		gd[i].bg_reserved[0] = 0;
		gd[i].bg_reserved[1] = 0;
		gd[i].bg_reserved[2] = 0;

		sb->s_free_blocks_count += groupsize - admin;
	}

	ped_free(zero);
	ped_free(ib);
	ped_free(bb);
	return 1;

error_free_zero:
	ped_free(zero);
error_free_ib:
	ped_free(ib);
error_free_bb:
	ped_free(bb);
error:
	return 0;
}

int ext2_mkfs_create_lost_and_found_inode(struct ext2_fs *fs)
{
	struct ext2_buffer_head *bh;
	blk_t blocks[12];
	__u32 data[fs->blocksize / 4];
	int i;
	struct ext2_inode inode;

	for (i=0;i<12;i++)
	{
		if (!(blocks[i] = ext2_find_free_block(fs)))
			return 0;

		if (!ext2_set_block_state(fs, blocks[i], 1, 1))
			return 0;
	}

	memset(data, 0, fs->blocksize);
	data[0] = 11;
	data[1] = (1 << 16) | 12;
	data[2] = 0x2e;
	data[3] = 2;
	data[4] = (2 << 16) | (fs->blocksize - 12);
	data[5] = 0x2e2e;
	bh = ext2_bcreate(fs, blocks[0]);
	if (!bh)
		return 0;
	memcpy(bh->data, data, fs->blocksize);
	bh->dirty = 1;
	ext2_brelse(bh, 1);

	memset(data, 0, fs->blocksize);
	data[0] = 0;
	data[1] = fs->blocksize;
	for (i=1;i<12;i++)
	{
		bh = ext2_bcreate(fs, blocks[i]);
		memcpy(bh->data, data, fs->blocksize);
		bh->dirty = 1;
		ext2_brelse(bh, 1);
	}

	memset(&inode, 0, sizeof(struct ext2_inode));
	inode.i_mode = S_IFDIR | 0755;
	inode.i_uid = 0;
	inode.i_size = 12 * fs->blocksize;
	inode.i_atime = 1;
	inode.i_ctime = 1;
	inode.i_mtime = 1;
	inode.i_dtime = 0;
	inode.i_gid = 0;
	inode.i_links_count = 2;
	inode.i_blocks = (12 * fs->blocksize) >> 9;
	inode.i_flags = 0;
	for (i=0;i<12;i++)
		inode.i_block[i] = blocks[i];

	if (!ext2_write_inode(fs, 11, &inode))
		return 0;
	fs->gd[0].bg_used_dirs_count++;
	fs->metadirty |= EXT2_META_GD;

	return 1;
}

int ext2_mkfs_create_root_inode(struct ext2_fs *fs)
{
	struct ext2_buffer_head *bh;
	blk_t block;
	__u32 data[fs->blocksize / 4];
	struct ext2_inode inode;

	if (!(block = ext2_find_free_block(fs)))
		return 0;
	if (!ext2_set_block_state(fs, block, 1, 1))
		return 0;

	memset(data, 0, fs->blocksize);
	data[0]  = 2;
	data[1]  = (1 << 16) | 12;
	data[2]  = 0x2e;
	data[3]  = 2;
	data[4]  = (2 << 16) | 12;
	data[5]  = 0x2e2e;
	data[6]  = 11;
	data[7]  = (10 << 16) | (fs->blocksize - 24);
	data[8]  = 0x74736f6c;
	data[9]  = 0x756f662b;
	data[10] = 0x646e;
	bh = ext2_bcreate(fs, block);
	memcpy(bh->data, data, fs->blocksize);
	bh->dirty = 1;
	if (!ext2_brelse(bh, 1))
		return 0;

	memset(&inode, 0, sizeof(struct ext2_inode));
	inode.i_mode = S_IFDIR | 0755;
	inode.i_uid = 0;
	inode.i_size = fs->blocksize;
	inode.i_atime = 1;
	inode.i_ctime = 1;
	inode.i_mtime = 1;
	inode.i_dtime = 0;
	inode.i_gid = 0;
	inode.i_links_count = 3;
	inode.i_blocks = fs->blocksize >> 9;
	inode.i_flags = 0;
	inode.i_block[0] = block;

	if (!ext2_write_inode(fs, 2, &inode))
		return 0;
	fs->gd[0].bg_used_dirs_count++;
	fs->metadirty |= EXT2_META_GD;

	return 1;
}

int ext2_reserve_inodes(struct ext2_fs *fs)
{
	int i;

	for (i=1;i<12;i++)
		if (!ext2_set_inode_state(fs, i, 1, 1))
			return 0;
	return 1;
}


struct ext2_fs *ext2_mkfs(struct ext2_dev_handle *handle,
			  blk_t numblocks,
			  int log_block_size,
			  blk_t blocks_per_group,
			  int inodes_per_group,
			  int sparse_sb,
			  int reserved_block_percentage)
{
	struct ext2_super_block sb;
	struct ext2_group_desc *gd;
	int numgroups;
	int first_block;

	if (log_block_size == -1)
		log_block_size = 12;

	handle->ops->set_blocksize(handle->cookie, log_block_size);

	if (numblocks == -1)
		numblocks = handle->ops->get_size(handle->cookie);

	if (blocks_per_group == -1)
		blocks_per_group = 8 << log_block_size;

	first_block = (log_block_size == 10) ? 1 : 0;

	numgroups = howmany(numblocks - first_block, blocks_per_group);

	if (inodes_per_group == -1)
	{
		int mult;

		inodes_per_group = numblocks / (numgroups << 3);

		mult = (1 << log_block_size) / sizeof(struct ext2_group_desc);
		inodes_per_group += mult - 1;
		inodes_per_group /= mult;
		inodes_per_group *= mult;
	}

	if (sparse_sb == -1)
		sparse_sb = 0;

	if (reserved_block_percentage == -1)
		reserved_block_percentage = 5;

	gd = ped_malloc(numgroups * sizeof(struct ext2_group_desc)
			+ (1 << log_block_size));
	if (!gd)
		return NULL;

	memset(&sb, 0, 1024);

	sb.s_inodes_count = numgroups * inodes_per_group;
	sb.s_blocks_count = numblocks;
	sb.s_r_blocks_count = ((u_int64_t)numblocks * reserved_block_percentage) / 100;
	sb.s_free_blocks_count = 0;
	sb.s_free_inodes_count = sb.s_inodes_count;
	sb.s_first_data_block = first_block;
	sb.s_log_block_size = log_block_size - 10;
	sb.s_log_frag_size = sb.s_log_block_size;
	sb.s_blocks_per_group = blocks_per_group;
	sb.s_frags_per_group = blocks_per_group;
	sb.s_inodes_per_group = inodes_per_group;
	sb.s_mtime = 0;
	sb.s_wtime = 0;
	sb.s_mnt_count = 0;
	sb.s_max_mnt_count = 30;
	sb.s_magic = 0xEF53;
	sb.s_state = EXT2_VALID_FS;
	sb.s_errors = 0;
	sb.s_minor_rev_level = 1;
	sb.s_lastcheck = 0;
	sb.s_checkinterval = 0;
	sb.s_creator_os = 0;
	sb.s_rev_level = 0;
	sb.s_def_resuid = 0;
	sb.s_def_resgid = 0;
	sb.s_first_ino = 12;
	sb.s_inode_size = 128;
	sb.s_block_group_nr = 0;
	sb.s_feature_compat = 0;
	sb.s_feature_incompat = 0;
	sb.s_feature_ro_compat = 0;
	if (sparse_sb)
		sb.s_feature_ro_compat |= EXT2_FEATURE_RO_COMPAT_SPARSE_SUPER;
	uuid_generate(sb.s_uuid);
	memset(sb.s_volume_name, 0, 16);
	memset(sb.s_last_mounted, 0, 64);
	sb.s_algorithm_usage_bitmap = 0;
	sb.s_prealloc_blocks = 0;
	sb.s_prealloc_dir_blocks = 0;
	sb.s_padding1 = 0;

	if (!ext2_mkfs_write_meta(handle, &sb, gd)) {
		ped_free(gd);
		return NULL;
	}
	if (!ext2_mkfs_write_main(handle, &sb, gd)) {
		ped_free(gd);
		return NULL;
	}

	ped_free(gd);

	{
		struct ext2_fs *fs;

		fs = ext2_open(handle, 0);
		if (!ext2_reserve_inodes(fs)) goto error_close_fs;
		if (!ext2_mkfs_create_root_inode(fs)) goto error_close_fs;
		if (!ext2_mkfs_create_lost_and_found_inode(fs))
			goto error_close_fs;

		return fs;

	error_close_fs:
		ext2_close(fs);
		return NULL;
	}
}
