/*
    ext2_meta.c -- ext2 metadata mover
    Copyright (C) 1999 Lennert Buytenhek <buytenh@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char _ext2_meta_c[] = "$Id: ext2_meta.c,v 1.1 1999/09/02 12:07:14 buytenh Exp $";

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include "ext2.h"

int ext2_metadata_push(struct ext2_fs *fs, blk_t newsize)
{
	int   i;
	int   newgdblocks;
	blk_t newitoffset;

	newgdblocks = howmany(newsize - fs->sb.s_first_data_block, fs->sb.s_blocks_per_group);
	newgdblocks = howmany(newgdblocks * sizeof(struct ext2_group_desc), fs->blocksize);
	newitoffset = newgdblocks + 3;

	if (newitoffset <= fs->itoffset)
		return 1;

	for (i=0;i<fs->numgroups;i++)
	{
		blk_t diff;
		int   j;
		blk_t fromblock;
		blk_t start;

		start = (i * fs->sb.s_blocks_per_group)
			+ fs->sb.s_first_data_block;

		if (fs->gd[i].bg_inode_table >= start + newitoffset
		    && fs->gd[i].bg_block_bitmap >= start + newitoffset - 2
		    && fs->gd[i].bg_inode_bitmap >= start + newitoffset - 1)
			continue;

		diff = newitoffset - (fs->gd[i].bg_inode_table - start);

		/* inode table */
		fromblock = fs->gd[i].bg_inode_table + fs->inodeblocks;

		if (fs->opt_debug)
		{
			for (j=0;j<diff;j++)
				if (ext2_get_block_state(fs, fromblock+j))
				{
					fprintf(stderr,
						"error: block relocator "
						"should have relocated "
						"%i\n",
						fromblock);

					return 0;
				}
		}

		for (j=0;j<diff;j++)
			if (!ext2_set_block_state(fs, fromblock+j, 1, 0))
				return 0;

		if (!ext2_move_blocks(fs,
				      fs->gd[i].bg_inode_table,
				      fs->inodeblocks,
				      fs->gd[i].bg_inode_table + diff))
			return 0;
		fs->gd[i].bg_inode_table += diff;
		fs->metadirty |= EXT2_META_GD;

		if (fs->opt_safe)
			if (!ext2_sync(fs))
				return 0;

		/* block bitmap and inode bitmap */
		fromblock = fs->gd[i].bg_inode_table;
		if (ext2_is_group_sparse(fs, i))
		{
			if (!ext2_copy_block(fs, fs->gd[i].bg_inode_bitmap,
					     fs->gd[i].bg_inode_bitmap+diff))
				return 0;
			fs->gd[i].bg_inode_bitmap += diff;
                        fs->metadirty |= EXT2_META_GD;

			if (fs->opt_safe)
				if (!ext2_sync(fs))
					return 0;

			if (!ext2_copy_block(fs, fs->gd[i].bg_block_bitmap,
					     fs->gd[i].bg_block_bitmap+diff))
				return 0;
			fs->gd[i].bg_block_bitmap += diff;
			fs->metadirty |= EXT2_META_GD;

			if (fs->opt_safe)
				if (!ext2_sync(fs))
					return 0;

			fromblock = fs->gd[i].bg_block_bitmap;
		}

		ext2_zero_blocks(fs, fromblock-diff, diff);
		for (j=0;j<diff;j++)
			if (!ext2_set_block_state(fs, fromblock+j-diff, 0, 0))
				return 0;

		if (fs->opt_verbose)
			fprintf(stderr,
				"ext2_metadata_push: group %i/%i\r",
				i+1, fs->numgroups);
	}

	fs->itoffset = newitoffset;

	if (fs->opt_verbose)
		fprintf(stderr, "\n");

	return 1;
}
