/*
 * JFBTERM - 
 * Copyright (c) 1999 Noritoshi Masuichi (nmasu@ma3.justnet.ne.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	notice, this list of conditions and the following disclaimer in the
 *	documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY NORITOSHI MASUICHI ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL NORITOSHI MASUICHI BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <string.h>

#include <config.h>

#include <font.h>
#include <pcf.h>
#include <util.h>
#include <message.h>
#include <csv.h>

u_int gFontsWidth = 0;
u_int gFontsHeight = 0;

u_char sgFontsDefaultGlyph[] = {0x80};
u_char sgFontsDefaultDGlyph[] = {0xC0};

TFont gFont[] = {
					/* 1st element shall be ASCII */
	{ tfont_default_glyph,	"ASCII", 1, 1,
		TFONT_FT_SINGLE|TFONT_FT_94CHAR| 0x42,
		FH_LEFT, TFONT_ALIAS, NULL, sgFontsDefaultGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
					/* 2nd element shall be JIS X 0208 */
	{ tfont_default_glyph,	"JISX0208-1978", 2, 1,
		TFONT_FT_DOUBLE | TFONT_FT_94CHAR | 0x40,
		FH_LEFT, TFONT_ALIAS, NULL, sgFontsDefaultDGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"JISX0208-1983", 2, 1,
		TFONT_FT_DOUBLE | TFONT_FT_94CHAR | 0x42,
		FH_LEFT, TFONT_ALIAS, NULL, sgFontsDefaultDGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"CODEPAGE-437", 1, 1,
		TFONT_FT_SINGLE|TFONT_FT_96CHAR| 'U',
		FH_RIGHT, TFONT_ALIAS, NULL, sgFontsDefaultGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"ALT-ASCII", 1, 1,
		TFONT_FT_SINGLE|TFONT_FT_94CHAR| '0',
		FH_LEFT, TFONT_ALIAS, NULL, sgFontsDefaultGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"JISX0201-ROMAN", 1, 1,
		TFONT_FT_SINGLE|TFONT_FT_94CHAR| 0x4A,
		FH_RIGHT, TFONT_ALIAS, NULL, sgFontsDefaultGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"JISX0201-KATAKANA", 1, 1,
		TFONT_FT_SINGLE|TFONT_FT_94CHAR| 0x49,
		FH_RIGHT, TFONT_ALIAS, NULL, sgFontsDefaultGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"GB2312", 2, 1,
		TFONT_FT_DOUBLE | TFONT_FT_94CHAR | 0x41,
		FH_LEFT, TFONT_ALIAS, NULL, sgFontsDefaultDGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"KSX1001", 2, 1,
		TFONT_FT_DOUBLE | TFONT_FT_94CHAR | 0x43,
		FH_LEFT, TFONT_ALIAS, NULL, sgFontsDefaultDGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"ISO8859-1", 1, 1,
		TFONT_FT_SINGLE|TFONT_FT_96CHAR| 0x41,
		FH_RIGHT, TFONT_ALIAS, NULL, sgFontsDefaultGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"ISO8859-2", 1, 1,
		TFONT_FT_SINGLE|TFONT_FT_96CHAR| 0x42,
		FH_RIGHT, TFONT_ALIAS, NULL, sgFontsDefaultGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"ISO8859-3", 1, 1,
		TFONT_FT_SINGLE|TFONT_FT_96CHAR| 0x43,
		FH_RIGHT, TFONT_ALIAS, NULL, sgFontsDefaultGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"ISO8859-4", 1, 1,
		TFONT_FT_SINGLE|TFONT_FT_96CHAR| 0x44,
		FH_RIGHT, TFONT_ALIAS, NULL, sgFontsDefaultGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"ISO8859-5", 1, 1,
		TFONT_FT_SINGLE|TFONT_FT_96CHAR| 0x4C,
		FH_RIGHT, TFONT_ALIAS, NULL, sgFontsDefaultGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"ISO8859-6", 1, 1,
		TFONT_FT_SINGLE|TFONT_FT_96CHAR| 0x41,
		FH_RIGHT, TFONT_ALIAS, NULL, sgFontsDefaultGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"ISO8859-7", 1, 1,
		TFONT_FT_SINGLE|TFONT_FT_96CHAR| 0x46,
		FH_RIGHT, TFONT_ALIAS, NULL, sgFontsDefaultGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"ISO8859-8", 1, 1,
		TFONT_FT_SINGLE|TFONT_FT_96CHAR| 0x48,
		FH_RIGHT, TFONT_ALIAS, NULL, sgFontsDefaultGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"ISO8859-9", 1, 1,
		TFONT_FT_SINGLE|TFONT_FT_96CHAR| 0x4D,
		FH_RIGHT, TFONT_ALIAS, NULL, sgFontsDefaultGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"ISO8859-10", 1, 1,
		TFONT_FT_SINGLE|TFONT_FT_96CHAR| 0x56,
		FH_RIGHT, TFONT_ALIAS, NULL, sgFontsDefaultGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"JISX0212", 2, 1,
		TFONT_FT_DOUBLE | TFONT_FT_94CHAR | 0x44,
		FH_LEFT, TFONT_ALIAS, NULL, sgFontsDefaultDGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{ tfont_default_glyph,	"ITU-TT.101ChinesePrimarySet", 2, 1,
		TFONT_FT_DOUBLE | TFONT_FT_94CHAR | 0x45,
		FH_LEFT, TFONT_ALIAS, NULL, sgFontsDefaultDGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
	{	tfont_default_glyph,	NULL, 1, 1,
		0x00000000,
		FH_LEFT, TFONT_ALIAS, NULL, sgFontsDefaultDGlyph, NULL,
		0xFF, 0x00, 0xFF, 0x00, 0x00, 1, 1
	},
};

void tfont_ary_final(void)
{
	TFont* ce = gFont;
	while (ce->name) {
		if (ce->aliasF & TFONT_OWNER) {
			tfont_final(ce);
		}
		ce++;
	}
}

int tfont_ary_search_idx(const char* na)
{
	TFont* p;
	int i = 0;
	for (p = gFont ; p->name ; p++, i++) {
		if (strncasecmp(p->name, na, strlen(p->name)) == 0) {
			return i;
		}
	}	
	return -1;
}

TFont* tfont_ary_search(const char* na)
{
	TFont* p;
	for (p = gFont ; p->name ; p++) {
		if (strncasecmp(p->name, na, strlen(p->name)) == 0) {
			return p;
		}
	}	
	return NULL;
}

void tfont_final(TFont* p)
{
	util_free(p->glyph);
	util_free(p->dglyph);
	util_free(p->bitmap);
}

const u_char* tfont_default_glyph(
	TFont* p,
	u_char b1,
	u_char b2)
{
	u_int i;

	if (b1 < p->colf || p->coll < b1 || b2 < p->rowf || p->rowl < b2) {
		return p->dglyph;	
	} else {
		i = (b1 - p->colf) + (b2  - p->rowf) * p->colspan;
		return p->glyph[i];
	}
}

static void tfont_setup_font(TFont* pf, const char* fname, FONTSET_HALF hf)
{
	char cmd[1024];
	FILE* fp;
	TPcf pcf;
	int piped;
	int nf = strlen(fname);

	tpcf_init(&pcf);

	if ((nf > 3 && strcmp(".gz", fname+nf-3) == 0) ||
	    (nf > 2 && strcmp(".Z", fname+nf-2) == 0)) {
#ifndef JFB_GZIP_PATH
#error		JFB_GZIP_PATH is not set.
#else
		strcpy(cmd, JFB_GZIP_PATH " -dc < \"");
#endif
		strcat(cmd, fname);
		strcat(cmd, "\"");
		fp = popen(cmd, "r");
		piped = 1;
	} else {
		fp = fopen(fname, "r");
		piped = 0;
	}
	if (!fp) {
		printf("PCF : CANNOTOPEN : %s\n", fname);
		return;
	}
	tpcf_load(&pcf, fp);
	if (piped) {
		pclose(fp);
	} else {
		fclose(fp);
	}

	tpcf_as_tfont(&pcf, pf);
	pf->fhalf = hf;

	tpcf_final(&pcf);
}

static void tfont_alias(TFont* dst, TFont* src, FONTSET_HALF hf)
{
	dst->conv = src->conv;
	dst->width = src->width;
	dst->height = src->height;
	dst->fhalf = hf;
	dst->aliasF = TFONT_ALIAS;
	dst->glyph = src->glyph;
	dst->dglyph = src->dglyph;
	dst->bitmap = src->bitmap;
	dst->colf = src->colf; 
	dst->coll = src->coll;
	dst->rowf = src->rowf;
	dst->rowl = src->rowl;
	dst->colspan = src->colspan;
	dst->bytew = src->bytew;
	dst->bytec = src->bytec;
}

static void tfont_fontlist_glyph_size(void)
{
	TFont* p = gFont;
	u_int w;

	while (p->name) {
		if (gFontsHeight < p->height) {
			gFontsHeight = p->height;
		}
		w = p->width;
		w = (p->fsignature & TFONT_FT_DOUBLE) ? (w+1)/2 : w;
		if (gFontsWidth < w) {
			gFontsWidth = w;
		}
		p++;
	}
}

int tfont_is_valid(TFont* p)
{
	return (p->width == 0 || p->height == 0) ? 0 : 1;
}

void tfont_setup_fontlist(TCapValue* vp)
{
	static const char* types[] = {"pcf", "alias", NULL}; 
	static const char* sides[] = {"L", "R", NULL}; 

	const char* srn;
	const char* sty;
	const char* shf;
	const char* sph;

	int type = 0;
	int side = 0;

	TFont* dst;
	TFont* src;

	TCsv farg;
	
	for (; vp ; vp = vp->next) {
		tcsv_init(&farg, vp->value);
		if (farg.cap != 4) {
			print_message("FONT : Skipped (BAD FORMAT)\n");
			goto FINALIZE;
		}
		srn = tcsv_get_token(&farg);
		sty = tcsv_get_token(&farg);
		shf = tcsv_get_token(&farg);
		sph = tcsv_get_token(&farg);

		print_message("FONT : (%d) [%s]/%s/%s://%s/\n", farg.cap,
				 srn, sty, shf, sph);

		type = util_search_string(sty, types);
		side = util_search_string(shf, sides);

		if (type == -1 || side == -1) {
			print_message("FONT : Skipped (BAD FORMAT)\n");
			goto FINALIZE;
		}

		if (!(dst = tfont_ary_search(srn))) {
			print_message("FONT : Skipped (Unknown FONTSET=`%s'.)\n", srn);
			goto FINALIZE;
		}
		switch (type) {
		case 0:		/* pcf file */
			tfont_setup_font(dst, sph,
					 (side == 0) ? FH_LEFT : FH_RIGHT);
			break;
		case 1:		/* alias */
			if (!(src = tfont_ary_search(sph))) {
				print_message("FONT : Skipped (Unknown ALIAS FONTSET=`%s'.)\n", sph);
				continue;
			}
			tfont_alias(dst, src,
					(side == 0) ? FH_LEFT : FH_RIGHT);
			break;
		}

	FINALIZE:	
		tcsv_final(&farg);
	}

	tfont_fontlist_glyph_size();

	if (!tfont_is_valid(&(gFont[0]))) {
		die("FONT : ISO8859 not loaded.\n");
	}
}

