/* gxset - GTK interface to xset(1)

   Copyright (C) 1999 Ren Seindal (rene@seindal.dk)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"

#include "guistuff.h"
#include "main.h"

void
display_message(gchar *msg)
{
    GtkWidget *dialog;

    dialog = create_dialog();
    gtk_label_set(GTK_LABEL(find_widget(dialog, "message")), msg);

    gtk_grab_add(dialog);
    gtk_widget_show(dialog);
}

gboolean
on_delete_event                        (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
    gtk_main_quit();
    return FALSE;
}


void
on_button_ok_clicked                   (GtkButton       *button,
                                        gpointer         user_data)
{
    GtkWidget *gxset;
    ApplicationState *appstate;
    GString *msg;

    gxset = GTK_WIDGET(user_data);
    appstate = get_application_state(gxset);
    get_gui_state(gxset, appstate);

    msg = apply_settings(appstate);
    g_string_free(msg, TRUE);

    gtk_main_quit();
}


void
on_button_apply_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{
    GtkWidget *gxset;
    ApplicationState *appstate;
    GString *msg;

    gxset = GTK_WIDGET(user_data);
    appstate = get_application_state(gxset);
    get_gui_state(gxset, appstate);

    msg = apply_settings(appstate);
    display_message(msg->str);
    g_string_free(msg, TRUE);

    /* Update application state with applied changes */
    xset_info_delete(appstate->orig);
    xset_info_delete(appstate->cur);
    appstate->orig = xset_info_read();
    appstate->cur = xset_info_dup(appstate->orig);

    set_gui_state(gxset, appstate);
}


void
on_button_save_done(GtkWidget *w, gpointer user_data)
{
    GtkWidget *gxset;
    ApplicationState *appstate;
    GtkFileSelection *file_selection;
    gchar *file;
    GString *msg;

    file = (gchar *)gtk_object_get_user_data(GTK_OBJECT(w));

    if (file) {
	file_selection = GTK_FILE_SELECTION(w);
	gxset = GTK_WIDGET(user_data);
	appstate = get_application_state(gxset);

	g_free(appstate->save_file_name);
	appstate->save_file_name = g_strdup(file);

	get_gui_state(gxset, appstate);

	msg = save_settings(appstate);
	display_message(msg->str);
	g_string_free(msg, TRUE);
    }
}

void
on_button_save_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
    GtkWidget *gxset;
    ApplicationState *appstate;
    GtkWidget *file_selection;

    gxset = GTK_WIDGET(user_data);
    appstate = get_application_state(gxset);

    file_selection = create_file_selector();

    gtk_file_selection_set_filename(GTK_FILE_SELECTION(file_selection), 
				    appstate->save_file_name);

    gtk_signal_connect(GTK_OBJECT(file_selection), "destroy",
		       GTK_SIGNAL_FUNC(on_button_save_done),
		       gxset);

    gtk_grab_add(file_selection);
    gtk_widget_show(file_selection);
}

void
on_button_cancel_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
    gtk_main_quit();
}



void
on_keyclick_onoff_toggled              (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
    GtkWidget *gxset = GTK_WIDGET(user_data);
    gboolean on = gtk_toggle_button_get_active(togglebutton);

    gtk_widget_set_sensitive(lookup_widget(gxset, "keyclick_volume"), on);
}


void
on_keyrepeat_onoff_toggled             (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
    GtkWidget *gxset = GTK_WIDGET(user_data);
    gboolean on = gtk_toggle_button_get_active(togglebutton);

    gtk_widget_set_sensitive(lookup_widget(gxset, "keyrepeat_delay"), on);
    gtk_widget_set_sensitive(lookup_widget(gxset, "keyrepeat_rate"), on);
}


void
on_bell_onoff_toggled                  (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
    GtkWidget *gxset = GTK_WIDGET(user_data);
    gboolean on = gtk_toggle_button_get_active(togglebutton);

    gtk_widget_set_sensitive(lookup_widget(gxset, "bell_volume"), on);
    gtk_widget_set_sensitive(lookup_widget(gxset, "bell_pitch"), on);
    gtk_widget_set_sensitive(lookup_widget(gxset, "bell_duration"), on);
}


void
on_beep_button_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
    gdk_beep();
}


void
on_scrsaver_onoff_toggled              (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
    GtkWidget *gxset = GTK_WIDGET(user_data);
    gboolean on = gtk_toggle_button_get_active(togglebutton);

    gtk_widget_set_sensitive(lookup_widget(gxset, "scrsaver_blank"), on);
    gtk_widget_set_sensitive(lookup_widget(gxset, "scrsaver_expose"), on);
    gtk_widget_set_sensitive(lookup_widget(gxset, "scrsaver_delay"), on);
    gtk_widget_set_sensitive(lookup_widget(gxset, "scrsaver_cycle"), on);
}


void
on_dpms_onoff_toggled                  (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
    GtkWidget *gxset = GTK_WIDGET(user_data);
    gboolean on = gtk_toggle_button_get_active(togglebutton);

    gtk_widget_set_sensitive(lookup_widget(gxset, "dpms_standby"), on);
    gtk_widget_set_sensitive(lookup_widget(gxset, "dpms_suspend"), on);
    gtk_widget_set_sensitive(lookup_widget(gxset, "dpms_off"), on);
}




static void
on_font_new_done(GtkWidget *font_dialog, gpointer user_data)
{
    GtkWidget *gxset;
    ApplicationState *appstate;
    gchar *path;

    gxset = GTK_WIDGET(user_data);
    appstate = get_application_state(gxset);
    path = gtk_object_get_user_data(GTK_OBJECT(font_dialog));

    if (path) {
	font_add_new(appstate, (gchar *)path);
	g_free(path);

	set_font_path(gxset, appstate);
    }
}


void
on_font_button_new_clicked             (GtkButton       *button,
                                        gpointer         user_data)
{
    GtkWidget *gxset;
    GtkWidget *font_dialog;

    gxset = GTK_WIDGET(user_data);

    font_dialog = font_dialog_new(NULL);

    gtk_signal_connect(GTK_OBJECT(font_dialog), "destroy",
		       GTK_SIGNAL_FUNC(on_font_new_done),
		       gxset);

    gtk_grab_add(font_dialog);
    gtk_widget_show(font_dialog);
}


static void
on_font_edit_done(GtkWidget *font_dialog, gpointer user_data)
{
    GtkWidget *gxset;
    ApplicationState *appstate;
    gchar *path;

    gxset = GTK_WIDGET(user_data);
    appstate = get_application_state(gxset);
    path = gtk_object_get_user_data(GTK_OBJECT(font_dialog));

    if (path) {
	font_set_current(appstate, (gchar *)path);
	g_free(path);

	set_font_path(gxset, appstate);
    }
}

void
on_font_button_edit_clicked            (GtkButton       *button,
                                        gpointer         user_data)
{
    GtkWidget *gxset;
    ApplicationState *appstate;
    GtkWidget *font_dialog;

    gxset = GTK_WIDGET(user_data);
    appstate = get_application_state(gxset);

    font_dialog = font_dialog_new(font_get_current(appstate));

    gtk_signal_connect(GTK_OBJECT(font_dialog), "destroy",
		       GTK_SIGNAL_FUNC(on_font_edit_done),
		       gxset);

    gtk_grab_add(font_dialog);
    gtk_widget_show(font_dialog);
}
   

void
on_font_button_up_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
    GtkWidget *gxset;
    ApplicationState *appstate;

    gxset = GTK_WIDGET(user_data);
    appstate = get_application_state(gxset);

    if (font_move_up(appstate)) {
	set_font_path(gxset, appstate);
    }
}


void
on_font_button_down_clicked            (GtkButton       *button,
                                        gpointer         user_data)
{
    GtkWidget *gxset;
    ApplicationState *appstate;

    gxset = GTK_WIDGET(user_data);
    appstate = get_application_state(gxset);

    if (font_move_down(appstate)) {
	set_font_path(gxset, appstate);
    }
}


void
on_font_button_remove_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
    GtkWidget *gxset;
    ApplicationState *appstate;

    gxset = GTK_WIDGET(user_data);
    appstate = get_application_state(gxset);

    if (font_remove(appstate)) {
	set_font_path(gxset, appstate);
    }
}


void
on_font_path_select_row                (GtkCList        *clist,
                                        gint             row,
                                        gint             column,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
    GtkWidget *gxset;
    ApplicationState *appstate;

    gxset = GTK_WIDGET(user_data);
    appstate = get_application_state(gxset);
    appstate->font_path_selected = row;

    adjust_font_buttons(gxset, appstate);
}


void
on_font_path_unselect_row              (GtkCList        *clist,
                                        gint             row,
                                        gint             column,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
    GtkWidget *gxset;
    ApplicationState *appstate;

    gxset = GTK_WIDGET(user_data);
    appstate = get_application_state(gxset);
    appstate->font_path_selected = -1;

    adjust_font_buttons(gxset, appstate);
}



/* 
 * New font directory entry editor
 */

void
on_font_browse_done(GtkWidget *w, gpointer user_data)
{
    GtkWidget *gxset;
    GtkFileSelection *file_selection;
    gchar *file;

    file_selection = GTK_FILE_SELECTION(w);
    gxset = GTK_WIDGET(user_data);

    file = (gchar *)gtk_object_get_user_data(GTK_OBJECT(w));

    if (file) {
	gtk_entry_set_text(GTK_ENTRY(find_widget(gxset, "font_dir_entry")),
			   file);
    }
}

void
on_font_browse_button_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
    GtkWidget *gxset;
    ApplicationState *appstate;

    GtkEntry *entry;
    GtkWidget *file_selection;

    gxset = GTK_WIDGET(user_data);
    appstate = get_application_state(gxset);

    file_selection = create_file_selector();

    entry = GTK_ENTRY(find_widget(gxset, "font_dir_entry"));
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(file_selection), 
				    gtk_entry_get_text(entry));

    gtk_signal_connect(GTK_OBJECT(file_selection), "destroy",
		       GTK_SIGNAL_FUNC(on_font_browse_done),
		       gxset);

    gtk_grab_add(file_selection);
    gtk_widget_show(file_selection);
}


void
on_font_ok_button_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
    GtkWidget *font_dialog;
    gchar *new_path;

    font_dialog = GTK_WIDGET(user_data);
    new_path = font_dialog_get_data(font_dialog);

    if (new_path)
	gtk_object_set_user_data(GTK_OBJECT(font_dialog), new_path);

    gtk_widget_destroy(font_dialog);
}



/* 
 * File selector dialog callbacks
 */

void
on_file_sel_ok_button_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
    GtkFileSelection *file_selection;

    file_selection = GTK_FILE_SELECTION(user_data);

    /* this returns the entry to the font dialog */
    gtk_object_set_user_data(GTK_OBJECT(file_selection),
			     gtk_file_selection_get_filename(file_selection));
    gtk_widget_destroy(GTK_WIDGET(file_selection));
}

