/******************************************************************************\
 gnofin/exp-text.c   $Revision: 1.2 $
 Copyright (C) 1999 Marty Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <gnome.h>

#include "gnofin.h"
#include <errno.h>
#include "view.h"
#include "imp-qif.h"

char* qifFilePtr;
char* qifFileSavePtr;
long  qifPos = 0;

gchar account_name[1024];
gchar account_info[1024];

int qif_find_keyword (char* token);
int qif_read_file (FILE* stream, long fileSize);
void qif_read_line (int* eof, char* textLine);
void qif_trim_left (char* textLine);
int qif_find_one_of (char ch, char* textLine);
gboolean ProcessBankAccountTransactions (FinView* view, char* qifPtr, char*token);
void qif_req_n_string (char* token, char* textLine, int count);
void qif_get_amount (char* textLine, money_t* amount);

//-------------------------------------------------------------------------
gboolean qif_import (void* vw, FILE * stream, gchar* filename)
{

  char* qifPtr;
  char* savptr;
  char*	token;
  int	iToken;
  long 	fileSize;
  int	eof;
  int	iPos;
  FinView* view = (FinView*)vw;

  fin_trace ("");

  qifFileSavePtr = qifFilePtr;

  // get length of file
  fseek (stream, 0, SEEK_END);
  fileSize = ftell (stream);
  fin_trace ("file size: %ld", fileSize);
  rewind (stream);

  qifPtr = calloc (256, 1);
  fin_trace ("qifPtr: %ld", (long)qifPtr);
  if (qifPtr == NULL) {
    fin_trace ("No memory available for processing QIF file.");
    return FALSE;
  }

  savptr = qifPtr;

  token = calloc (256, 1);
  if (token == NULL) {
    fin_trace ("No memocery available for processing QIF file.");
    return FALSE;
  }

  // Ensure that we have a valid QIF file
  if (fread(qifPtr, sizeof (char), 5, stream) != 5) {
    fin_trace ("File read error");
    free (qifPtr);
    return FALSE;
  }

  *(qifPtr+6) = 0;
  fin_trace ("Token: %s", qifPtr);

  iToken = qif_find_keyword (qifPtr);
  fin_trace ("iToken: %d", iToken);

  switch (iToken) {
    case TYPEMARKER_TOK:
    case ACCOUNTMARKER_TOK:
      break;
    default:
      fin_trace ("Invalid QIF file");
      free (qifPtr);
      free (token);
      return FALSE;
  }

  rewind (stream);

  // read file into memory for higher performance in parsing
  if (!qif_read_file (stream, fileSize)) {
    fin_trace ("Error reading QIF file into memory.");
    free (qifPtr);
    free (token);
    return FALSE;
  }

  strcpy (account_name, filename);
  strcpy (account_info, "Imported qif file");

  // parse file and build new accounting set
  while (1) {
    // read in next line
    qifPtr = savptr;
    qif_read_line (&eof, qifPtr);

    if (eof) {
      free (qifFileSavePtr);
      free (qifPtr);
      free (token);
      return TRUE;
    }

    fin_trace ("Text line: %s", qifPtr);

    qif_trim_left (qifPtr);

    // check for a zero length line
    if (strlen (qifPtr) == 0)
      continue;

    // get next token
    iPos = qif_find_one_of (' ', qifPtr);
    fin_trace ("iPos: %d", iPos);
    if (iPos > 0)
      strncpy (token, qifPtr, iPos);
    else
      strcpy (token, qifPtr);

    fin_trace ("qifPtr: %s", token);
    fin_trace ("qifPtr length: %d", strlen (token));
    iToken = qif_find_keyword (token);
    fin_trace ("iToken: %d", iToken);

    switch (iToken) {
      case TYPEBANK_TOK:
	      if (!ProcessBankAccountTransactions (view, qifPtr, token)) {
          free (qifFileSavePtr);
          free (qifPtr);
          free (token);
          return FALSE;
	      }
	      break;
      case TYPECASH_TOK:
        fin_error_dialog (view->window, "Cash account transaction register is not supported.");
	      break;
      case TYPECCARD_TOK:
          fin_error_dialog (view->window, "Credit Card account transaction register is not supported.");
	      break;
      case TYPEINVST_TOK:
          fin_error_dialog (view->window, "Investment account transaction register is not supported.");
	      break;
      case TYPEOTHA_TOK:
          fin_error_dialog (view->window, "Asset account transaction register is not supported.");
	      break;
      case TYPEOTHL_TOK:
          fin_error_dialog (view->window, "Liability account transaction register is not supported.");
	      break;
      case ACCOUNT_TOK:
          fin_error_dialog (view->window, "Account List is not supported.");
	      break;
      case TYPECAT_TOK:
          fin_error_dialog (view->window, "Category List is not supported.");
	      break;
      case TYPECLASS_TOK:
          fin_error_dialog (view->window, "Class List is not supported.");
	      break;
      case TYPEMEMORIZED_TOK:
          fin_error_dialog (view->window, "Memorized transaction list is not supported.");
	      break;
      default:
	      fin_trace ("Unknown token");
	      break;
    }		

    break;
  }

  free (qifFileSavePtr);
  fin_trace ("Freed qifFileSavePtr");
  free (qifPtr);
  fin_trace ("Freed qifPtr");
  free (token);
  fin_trace ("Freed token");

  fin_trace ("Exit Qif_Import");

  return TRUE;
	
}

//-------------------------------------------------------------------------
gboolean ProcessBankAccountTransactions (FinView* view, char* qifPtr, char*token)
{

  char* savptr = qifPtr;
  int eof;
  int iToken;
  gint day, month, year;
  gint check_no;
  gint have_amount;
  gint cleared = 0;
  money_t amount;
  gint itype = 0;
  gint itype_d2 = 0;
  FILE* descInfo = fopen ("/tmp/descInfo", "w");
  FILE* transInfo = fopen ("/tmp/transInfo", "w");
  FILE* import;
  gint info_index = 0;
  gint ipos;
  long filesize;
  char* fileptr;

  if (descInfo == NULL || transInfo == NULL) {
    fin_error_dialog (view->window, "Failed to open temporary files for importing QIF file");
    fin_trace ("Failed to open temporary files for importing QIF file");
    fclose (descInfo);
    remove ("/tmp/descInfo");
    fclose (transInfo);
    remove ("/tmp/transInfo");
    return FALSE;
  }
  
  fin_trace ("");

  while (1) {
    // read in next line
    qifPtr = savptr;
    qif_read_line (&eof, qifPtr);

    if (eof) {
      break;
    }

    fin_trace ("Text line: %s", qifPtr);

    qif_trim_left (qifPtr);

    // check for a zero length line
    if (strlen (qifPtr) == 0)
      continue;

    // get Bank Account type
    qif_req_n_string (token, qifPtr, 1);

    fin_trace ("token: %s", token);
    fin_trace ("qifPtr: %s", qifPtr);
    fin_trace ("token length: %d", strlen (token));
    iToken = qif_find_keyword (token);
    fin_trace ("iToken: %d", iToken);

    switch (iToken) {
      case NONINVEST_DATE_TOK:
        check_no = 0;
        have_amount = 0;
        cleared = 0;
        amount = 0;
        itype = 0;

        ipos = qif_find_one_of ('/', qifPtr);
        fin_trace ("ipos: %d  qifPtr: %s", ipos, qifPtr);
        strncpy (token, qifPtr, ipos);
        sscanf (token, "%2d", &month);
        qifPtr = qifPtr + (ipos+1);

        ipos = qif_find_one_of ('/', qifPtr);
        strncpy (token, qifPtr, ipos);
        sscanf (token, "%2d", &day);
        qifPtr = qifPtr + (ipos+1);

        sscanf (qifPtr, "%2d", &year);
        qifPtr = qifPtr + 2;

        // Y2K compliance fix
        if (year > 20) {
            year += 1900;
        }
        else
            year += 2000;

        fin_trace ("month: %d  day: %d  year: %d", month, day, year);
        fin_trace ("date cleared");
        fin_trace ("date set");
	      break;
      case NONINVEST_NUM_TOK:
        if (*qifPtr >= '1' && *qifPtr <= '9') {  // check/ref number
          sscanf (qifPtr, "%d", &check_no); 
          fin_trace ("Check No: %d", check_no);
        }
        else
            check_no = 0;
        break;
      case NONINVEST_AMOUNT_TOK:
        fin_money_parse (qifPtr, &amount); 
        fin_trace ("amount: %d", amount);
        if (amount > 0) { // deposit
          itype = 3;  // deposit
        }
        else {
          itype = 2;  // check
        }
        have_amount = 1;
        break;
      case NONINVEST_CLEARED_TOK:
        cleared = 1;
        break;
      case NONINVEST_PAYEE_TOK:
	      fprintf (descInfo, "%s\n", qifPtr);
        break;
    case ENDOFENTRY_TOK:
        fprintf (transInfo, "%d %d %d %d %d %d %d %d %ld\n", day, month, year, itype, check_no, itype_d2, info_index, cleared, amount);  
        fin_trace ("record stored");
	      info_index++;
        break;
    }

  }

  fin_trace ("Close desc and trans temporary files");

  fclose (descInfo);
  fclose (transInfo);

  // Now take these two files and create a !FIN file for openin the account

  import = fopen ("/tmp/importqif", "w");
  if (import == NULL)
  {
    fin_error_dialog (view->window, "Failed to open temporary import file for importing QIF file");
    fin_trace ("Failed to open temporary import file for importing QIF file");
    return FALSE;
  }

  // Write out header
  fprintf(import, "%s\n%s\n", "FIN!", "0.4");

  // Write out the number of info entries
  fprintf (import, "%d\n", info_index);

  // Read the info entries into memory

  descInfo = fopen ("/tmp/descInfo", "r");

  // get length of file
  fseek (descInfo, 0, SEEK_END);
  filesize = ftell (descInfo);
  rewind (descInfo);

  fileptr = calloc (filesize, 1);
  if (fileptr == NULL) {
    fin_trace ("Failed to allocate memory for importing QIF file.");
    goto errorXit;
  }

  if (fread (fileptr, sizeof (char), filesize, descInfo) != filesize) {
    fin_trace ("Error reading temporary info descriptions file.");
    goto errorXit;
  }

  // Write out info descriptions
  if (fwrite (fileptr, filesize, 1, import) != 1) {
    fin_trace ("Size to write: %ld  Error writing temporary import QIF file.", 
               filesize);
    goto errorXit;
  }

  free (fileptr);
  fclose (descInfo);
  remove ("/tmp/descInfo");

  // Write out account name and info
  fprintf (import, "%d\n%s\n%s\n", 1, account_name, account_info);

  // Write out the number of transaction entries
  fprintf (import, "%d\n", info_index);

  // Read the transactions entries into memory

  transInfo = fopen ("/tmp/transInfo", "r");

  // get length of file
  fseek (transInfo, 0, SEEK_END);
  filesize = ftell (transInfo);
  rewind (transInfo);

  fileptr = calloc (filesize, 1);
  if (fileptr == NULL) {
    fin_trace ("Failed to allocate memory for importing QIF file.");
    goto errorXit2;
  }

  if (fread (fileptr, sizeof (char), filesize, transInfo) != filesize) {
    fin_trace ("Error reading temporary transInfo file.");
    goto errorXit2;
  }

  // Write out transactions
  if (fwrite (fileptr, filesize, 1, import) != 1) {
    fin_trace ("Size to write: %ld  Error writing temporary import QIF file.", 
               filesize);
    goto errorXit2;
  }

  free (fileptr);
  fclose (transInfo);
  remove ("/tmp/transInfo");

  fclose (import);

  return TRUE;

  errorXit:
  fclose (descInfo);
  remove ("/tmp/descInfo");
  errorXit2:
  fclose (transInfo);
  remove ("/tmp/transInfo");
  fclose (import);
  remove ("/tmp/importqif");

  return FALSE;

}

//-------------------------------------------------------------------------
int qif_find_keyword (char* token)
{
  int i;

  fin_trace("");

  for (i = 0; i < QIF_NUMOFKEYWORDS; i++) {
    if (strcmp (token, keywords[i].name) == 0)
      return keywords[i].value;
  }

  return 0;

}

//-------------------------------------------------------------------------
int qif_read_file (FILE* stream, long fileSize)
{

  fin_trace ("");

  qifFilePtr = calloc (fileSize, 1);
  if (qifFilePtr == NULL) {
    fin_trace ("Failed to allocate memory for QIF file.");
    return 0;
  }

  if (fread (qifFilePtr, sizeof (char), fileSize, stream) != fileSize) {
    fin_trace ("Error reading QIF file.");
    return 0;
  }

  return 1;

}

//-------------------------------------------------------------------------
void qif_read_line (int* eof, char* textLine)
{

  char ch = 0;
  char*	savePtr = textLine;

  fin_trace ("");

  *eof = 0;

  do {
    ch = *qifFilePtr++;
    qifPos++;
    if (ch != 0x0a && ch != 0x0d && ch != 0x00)
      *textLine++ = ch;
    if (ch == 0x00) {
      fin_trace ("EOF Reached");
      *eof = 1;
      break;
    }
				
  } while (ch != 0x0a);

  *textLine = 0;

  textLine = savePtr;

}

//-------------------------------------------------------------------------
void qif_trim_left (char* textLine)
{

  char* ptr;
  char* tempPtr = textLine;

  fin_trace ("");

  if (*textLine == ' ') {
    ptr = calloc (strlen (textLine) +1, 1);
    while (*tempPtr++ == ' ');
      strcpy (ptr, tempPtr);
      strcpy (textLine, ptr);
      free (ptr);
  }
	
}

//-------------------------------------------------------------------------
int qif_find_one_of (char ch, char* textLine)
{

  char* ptr;

  fin_trace ("");

  ptr = strchr (textLine, ch);

  fin_trace ("ptr: %s", ptr);

  if (ptr == NULL) // didn't find ch terminator
    return 0;

  return (int)ptr-(int)textLine;

}

//-------------------------------------------------------------------------
void qif_req_n_string (char* token, char* textLine, int count)
{

  char* ptr;

  fin_trace ("");

  strncpy (token, textLine, count);
  *(token+count) = 0;
  ptr = calloc (strlen (textLine) +1, 1);
  strcpy (ptr, textLine+count);
  strcpy (textLine, ptr);
  free (ptr);

}

//-------------------------------------------------------------------------
void qif_get_amount (char* textLine, money_t* amount)
{

  char* ptr = textLine;
  char* savptr = textLine;
  int len = strlen (textLine);
  char ch;

  fin_trace ("");

  while (len > 0) {
    len--;
    ch = *textLine++;
    if (ch == ',')
      continue;
    *ptr++ = ch;    
  }

  *ptr = 0;
  
  textLine = savptr;

  fin_trace ("money amount: %s", textLine);

  fin_money_parse (textLine, amount); 

}

//-------------------------------------------------------------------------
FinImportFilter FinImport_QIF =
{
  "Quicken Interchange Format",
  "Created by Marty Fisher <mfisher@jagger.me.berkeley.edu>",
  qif_import,
};

