/*
 * Copyright (c) 1997, 1998  Motoyuki Kasahara
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/*
 * This program requires the following Autoconf macros:
 *   AC_C_CONST
 *   AC_HEADER_STDC
 *   AC_CHECK_HEADERS(string.h, memory.h, stdlib.h)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <syslog.h>

#if defined(STDC_HEADERS) || defined(HAVE_STRING_H)
#include <string.h>
#if !defined(STDC_HEADERS) && defined(HAVE_MEMORY_H)
#include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */
#else /* not STDC_HEADERS and not HAVE_STRING_H */
#include <strings.h>
#endif /* not STDC_HEADERS and not HAVE_STRING_H */

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include "permission.h"

#ifdef USE_FAKELOG
#include "fakelog.h"
#endif

#ifdef __STDC__
static int parse_address(const char *, int *);
static int parse_address_netmask(const char *, int *, int *);
#else
static int parse_address();
static int parse_address_netmask();
#endif


/*
 * Initialize `permission'.
 */
void
initialize_permission(permission)
    Permission *permission;
{
    permission->next = NULL;
}


/*
 * Free all nodes in `permission'.
 */
void
clear_permission(permission)
    Permission *permission;
{
    Permission *pm, *next;

    pm = permission->next;
    while (pm != NULL) {
	next = (Permission *)pm->next;
	if (pm->hostname != NULL)
	    free(pm->hostname);
	free(pm);
	pm = next;
    }

    permission->next = NULL;
}


/*
 * Add `pattern' to the list `permission'.
 *
 * If `permission' is a valid IPv4 address, it is recognized as an
 * address.  If `permission' is a valid `address/address' format, it
 * is recognized as a pair of address and netmask.  Otherwise `pattern'
 * is recognized as a hostname.
 *
 * If succeeded, 0 is returned.  Otherwise, -1 is returned.
 */
int
add_permission(permission, pattern)
    Permission *permission;
    const char *pattern;
{
    Permission *pm = NULL;
    const char *pat;
    int addr, mask;

    /*
     * Allocate memeories for new node and a hostname.
     */
    pm = (Permission *)malloc(sizeof(Permission));
    if (pm == NULL) {
	syslog(LOG_ERR, "memory exhausted");
	goto failed;
    }
    pm->not = 0;
    pm->hostname = NULL;
    pm->address = 0;
    pm->netmask = 0;

    pat = pattern;
    if (*pat == '!') {
	pm->not = 1;
	pat++;
    }

    if (0 <= parse_address_netmask(pat, &addr, &mask)) {
	pm->address = addr;
	pm->netmask = mask;
    } else if (0 <= parse_address(pat, &addr)) {
	pm->address = addr;
	pm->netmask = 0xffffffff;
    } else {
	pm->hostname = (char *)malloc(strlen(pat) + 1);
	if (pm->hostname == NULL) {
	    syslog(LOG_ERR, "memory exhausted");
	    goto failed;
	}
	strcpy(pm->hostname, pat);
    }

    /*
     * Insert the new node into the list `perm'.
     */
    pm->next = permission->next;
    permission->next = pm;

    return 0;

    /*
     * If an error occurs...
     */
  failed:
    if (pm != NULL) {
	if (pm->hostname != NULL)
	    free(pm->hostname);
	free(pm);
    }
    return -1;
}


/*
 * Examine access permission to `hostname' and `address'.
 * If access is permitted, 1 is returned.  Otherwise 0 is returned.
 */
int
test_permission(permission, hostname, address, func)
    const Permission *permission;
    const char *hostname;
    const char *address;
#ifdef __STDC__
    int (*func)(const char *, const char *);
#else
    int (*func)();
#endif
{
    const Permission *pm;
    int flag = 0;
    int addrval;

    if (parse_address(address, &addrval) < 0)
	return 0;

    for (pm = permission->next; pm != NULL; pm = (Permission *)pm->next) {
	if (pm->hostname != NULL) {
	    if (func(pm->hostname, hostname))
		flag = pm->not ? 0 : 1;
	} else {
	    if ((addrval & pm->netmask) == pm->address)
		flag = pm->not ? 0 : 1;
	}
    }

    return flag;
}


/*
 * Count the entires of the permission list, and return it.
 */
int
count_permission(permission)
    const Permission *permission;
{
    const Permission *pm;
    int count = 0;

    for (pm = permission->next; pm != NULL; pm = (Permission *)pm->next)
	count++;

    return count;
}


/*
 * Convert an IP address pattern (e.g. "127.0.0.1") to an integer.
 * The result value is stored into `address'.
 * 
 * If `pattern' represents a valid address, 0 is returned.  Otherwise
 * -1 is returned.
 */
static int
parse_address(pattern, address)
    const char *pattern;
    int *address;
{
    static const char separators[] = {'.', '.', '.', '\0'};
    const char *pat = pattern;
    int oct;
    int i;

    *address = 0;
    for (i = 0; i < 4; i++) {
	oct = 0;
	if (*pat < '0' || '9' < *pat)
	    return -1;
	while ('0' <= *pat && *pat <= '9') {
	    oct = oct * 10 + (*pat - '0');
	    pat++;
	}
	if (*pat != separators[i])
	    return -1;
	if (oct < 0 || 255 < oct)
	    return -1;
	*address = (*address << 8) + oct;
	pat++;
    }
    return 0;
}


/*
 * Convert an IP address/netmask pattern (e.g. "192.24.1.0/255.255.255.0")
 * to two integers.  The result values are stored into `address' and
 * `netmask'.
 * 
 * If `pattern' represents a valid address/netmask pair, 0 is returned.
 * Otherwise -1 is returned.
 */
static int
parse_address_netmask(pattern, address, netmask)
    const char *pattern;
    int *address;
    int *netmask;
{
    static const char separators[] = {'.', '.', '.', '/', '.', '.', '.', '\0'};
    const char *pat = pattern;
    int oct;
    int i;

    *address = 0;
    for (i = 0; i < 4; i++) {
	oct = 0;
	if (*pat < '0' || '9' < *pat)
	    return -1;
	while ('0' <= *pat && *pat <= '9') {
	    oct = oct * 10 + (*pat - '0');
	    pat++;
	}
	if (*pat != separators[i])
	    return -1;
	if (oct < 0 || 255 < oct)
	    return -1;
	*address = (*address << 8) + oct;
	pat++;
    }

    *netmask = 0;
    for (i = 4; i < 8; i++) {
	oct = 0;
	if (*pat < '0' || '9' < *pat)
	    return -1;
	while ('0' <= *pat && *pat <= '9') {
	    oct = oct * 10 + (*pat - '0');
	    pat++;
	}
	if (*pat != separators[i])
	    return -1;
	if (oct < 0 || 255 < oct)
	    return -1;
	*netmask = (*netmask << 8) + oct;
	pat++;
    }

    return 0;
}
