/* 
 *	atp.c v1.2   - February 1, 1995
 *	
 *	Ascii to Postscript converter
 *
 *	Copyright (C) 1995 Paolo Severini {lendl@dist.dist.unige.it}
 *
 *	Permission to use and modify this software and its
 *	documentation for any purpose other than its incorporation
 *	into a commercial product is hereby granted without fee.
 *	Permission to copy and distribute this software and its
 *	documentation only for non-commercial use is also granted
 *	without fee, provided, however, that the above copyright
 *	notice appear in all copies, that both that copyright notice
 *	and this permission notice appear in supporting documentation.
 * 	The author makes no representations about the suitability
 * 	of this software for any purpose.  It is provided ``as is'',
 *	without express or implied warranty.
 */
	
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#define ROMAN	0
#define ITALIC  1
#define BOLD	2
#define COMMENT	3
#define PREPROC 1

#ifdef MSDOS
#define BUF_DIM 64000
#else
#define BUF_DIM 128000
#endif

/* list of standard paper sizes from Adobe's PPD. */

typedef struct documentmedia {
    char *name;
    short width, height;
} DocumentMedia;
		
DocumentMedia papersizes[] = {
    "Letter",		 612,  792,
    "letter",		 612,  792,
    "LetterSmall",	 612,  792,
    "Tabloid",		 792, 1224,
    "Ledger",		1224,  792,
    "Legal",		 612, 1008,
    "Statement",	 396,  612,
    "Executive",	 540,  720,
    "A3",		 842, 1190,
    "a3",                842, 1190,
    "A4",		 595,  842,
    "a4",                595,  842,
    "A4Small",		 595,  842,
    "A5",		 420,  595,
    "a5",		 420,  595,
    "B4",		 729, 1032,
    "b4",		 729, 1032,
    "B5",		 516,  729,
    "b5",		 516,  729,
    "Folio",		 612,  936,
    "Quarto",		 610,  780,
    "10x14",		 720, 1008,
    NULL,		   0,    0
};

char *Version="atp v1.2 - February 1, 1995";
FILE *fout;
extern char *font_names[20];
extern short font_widths[16][256];	/* from widths.c */
long 	LeftFrameWidth,		/* length of the date frame */
	RightFrameWidth, 	/* length of the page-number frame */
	FrameHeight,
 	Widths[4][256],		/* widths of the characters of the fonts roman, italic, 
				   bold and for the font used for the C remarks */
	PageWidth=0,
	PageHeight, 	
	TextWidth,
	TextHeight, 		 
	STARTY,			/* Y of the first line */
	ENDY,			/* Y of the last line */
	MaxLength,		/* length of a line */
	Xpos,			/* current X position */
	Xgap,			/* space between two colums */
	XleftFrame,		/* X of the date frame, and of the 1st column */
	FontDim=700,		/* default size for text (7pts) */
                                /* v1.1b -- changed from 8pts */
	TitleDim=1600,		/* default size for page headings */ 
	TitleY,			/* Y of page headings */
	x,
	y,
	xw,
	yw,
	line_number=1,
	m_top,			/* Margins */
	m_bottom,
	m_left,
	m_right;
short   LandscapeMode=1, 	/* 0: portrait  1: landscape */
	Decy,			/* space between two lines */
	opengraph=0, 		/* number of left braces opened */
	comment=0,		/* 1 within a C remark */
	chardef=0,
	string=0,		/* 1 within a C string */
	outfile=0,		/* if 1 write the output in a file */
	Cmode=0,		/* if 1 the input is considered a C source */
	Columns=0,		/* number of columns to print with */
	TabSize=8,
	Ncopies=1,		/* number of copies of each page */
	NumberMode=0,		/* if 1 enable line number printing */
	TruncateMode=0,		/* if 1 truncate long lines */
	Print_Postscript=0,	/* if 1 do not pass through PostScript files */
	is_a_postscript_file=0,
	Modify_Matrix=0,	/* if 1 the font is modified by a transformation matrix */
	Level1=0,		/* if 1 the output is in Postscript Level 1 format */
	Modify_comm_matrix=0,
	HeaderMode=1,		/* if 1 print the header */
	FancyMode=1,		/* if 1 paint the frames under title, date e page number */
	ShadowMode=1, 		/* if 1 paint a shadow around the header chars */
	ManualPageMode=0,
	RaggedMode=0,
	WatermarkMode=0,
	wm=ROMAN,		/* font type used to print the current char: */
	oldwm=-1,				/*   ROMAN, BOLD, COMMENT... */
	n_file=0,		/* # of input files translated */
	cut_lines=0,		/* # of lines cutted */
	newpage=0,
	column=0,
	page=1,
     	endpage=0,
    	ic=0, 			/* count 0...TabSize-1 */
	startline=1, 		/* 1 when a '\n' is found */
	newline=1,		/* 1 if a '\n' is found or the line is folded */
	doc_pages=0;

 /* transformation matrices, in Postscript format */	
float 	slim_matrix[4]={0.83, 0.0, 0.0, 1.0},
	normal_matrix[4]={1,0,0,1},
	title_matrix[4]={1,0,0,1},
	Matrix[4]={1,0,0,1},
	comment_matrix[4]={0.83,0,0,1};

char 	*font_wm[4]={"Rf","If","Bf","Cm"},
	fontname[4][64],
	wm_font[256],
	titlefont[64],
	title[512],
	wm_string[512],
	date_s[16],
	datestr[32],
	file_in[128];

#ifdef MSDOS
char *bs,*ms; /*to be malloc'ed*/
#else
unsigned char bs[BUF_DIM]; /* buffer for the char read from the input stream */
char 	      ms[BUF_DIM]; /* font type with which each char is printed (ROMAN, BOLD...) */
#endif
long 	is=0;		/* current position in bs[] and ms[] */


/* Forward declarations if in ANSI mode */

#ifdef _STDC_
void print_prolog();
void print_end();
void print_ps_matrix(float *);
void manage_8bit();
void start_page();
void print_string(unsigned char *);
void outs(unsigned char*, short);
void empty_buffer(short);
void look_for_header();
void translate_file(FILE *);
void usage();
void error_s(char *);
long width(char *, char *, long, float *);
short font_id(char *);
long select_font(char *, char *, float *);
void prepare_fonts(char *);
#endif



/* Print a postscript matrix (the last two values are always 0,0) */

void print_ps_matrix(m)
float *m;
  {
    short i;
    fputc('[',fout);
    for (i=0;i<4;i++)
	fprintf(fout,"%.2f ",m[i]);
    fprintf(fout,"0 0]");
    return;
  }



/* Change the Encoding vector of the fonts, for handle 8 bit characters,
   using the level2 procedure 'ISOLatinEncoding'. */

void manage_8bit()
  {
    if (Level1)
      {
	fprintf(fout,"/ChgFnt { pop findfont exch makefont exch scalefont setfont } def\n");
	return;
      }
    fprintf(fout,"   %% change fonts using ISO Latin1 characters\n");
    fprintf(fout,"/ChgFnt\n");
    fprintf(fout,"{  dup FontDirectory exch known\n");
    fprintf(fout,"   { exch pop }\n");
    fprintf(fout,"   { dup 3 2 roll ReEncode } ifelse\n");
    fprintf(fout,"   findfont exch makefont exch scalefont setfont\n");
    fprintf(fout,"} def\n");
    fprintf(fout,"/ReEncode {\n");
    fprintf(fout,"findfont dup length dict begin\n");
    fprintf(fout,"  {  1 index /FID ne {def} {pop pop} ifelse } forall\n");
    fprintf(fout,"  /Encoding ISOLatin1Encoding def\n");
    fprintf(fout,"  currentdict end definefont pop\n");
    fprintf(fout,"} def\n");   
    return;
  }



/* Write a string in PostScript format */

void print_string(s)
unsigned char *s;
  {
    unsigned char c;

    fputc('(',fout);
    while ((c=(unsigned char)*s++))
	if (c>=160 && !Level1)
	    fprintf(fout,"\\%3o",c);
	else if (c==')' || c=='(' || c=='\\')
	  {
	    fputc('\\',fout);
	    fputc(c,fout);
	  }
	else
	    fputc(c,fout);
    fputc(')',fout);
  }



/* Print a postscript prolog, with the postscript procedures utilized 
   for select the fonts, painting the frame, shadowing and underlining
   the characters, and so on. */

void print_prolog()
  {	
    long cx,FrameLength,right;
    long ytitle,ydate,ypagn;
    float mat[4],cosf;
    short j;

    doc_pages=0;
	/* Arrange the position of the header frames */
    right=PageWidth-m_right-RightFrameWidth;
    cx=m_left+(PageWidth-m_right-m_left)/2;
    FrameLength=right-LeftFrameWidth-XleftFrame;
    if (FrameLength<0)
	FrameLength=0;
    ytitle=TitleY+100+TitleDim/5.5;
    ydate=TitleY+90+TitleDim/5.5;
    ypagn=TitleY+40+TitleDim/5.5;

    fprintf(fout,"%%!PS-Adobe-%.1f\n%%Creator: %s\n",(Level1 ? 1.0:2.0),Version);
    fprintf(fout,"%%%%CreationDate: %s",datestr);
    fprintf(fout,"%%%%Orientation: %s\n",LandscapeMode ? "Landscape":"Portrait");
    fprintf(fout,"%%%%Pages: (atend)\n");
    fprintf(fout,"%%%%PageOrder: Ascend\n");
    fprintf(fout,"%%%%Title: %s\n",title);
    fprintf(fout,"%%%%DocumentFonts: %s %s %s %s ",fontname[ROMAN],
				fontname[ITALIC],fontname[BOLD],titlefont);
    fprintf(fout,"%s \n", (Cmode ? fontname[COMMENT]:""));
    fprintf(fout,"%%%%EndComments\n\n");
    fprintf(fout,"/Centre  { dup stringwidth pop 2 div neg 0 rmoveto } def\n");
    manage_8bit();

/* PostScript procedures: /Rf, /If, /Bf, /Cm select the fonts for the
   normal, italic, bold text and for the C comments */
    if (Cmode)
      {
        fprintf(fout,"/Cm { %ld ",FontDim);
        print_ps_matrix(comment_matrix);
        fprintf(fout," /%s /%s-Latin1 ChgFnt } def\n",fontname[COMMENT],fontname[COMMENT]);
      }
    for (j=ROMAN;j<=BOLD;j++)
      {
        fprintf(fout,"/%s { %ld ",font_wm[j],FontDim);
        print_ps_matrix(Matrix);
        fprintf(fout," /%s /%s-Latin1 ChgFnt } def\n",fontname[j],fontname[j]);
      }

/* /Gpnf, /Titlef, /Datef select the fonts for the heading (page number,title and date) 
   /Gl paints the left frame,
   /Gr paints the right frame,
   /Gm paints the central frame,
   /Header paints the three frames in the appropriate gray shade */
    if (HeaderMode)
      {
	for (j=0;j<4;j++)
	    mat[j]=title_matrix[j];
    	fprintf(fout,"/Gpnf { %ld ",TitleDim-150);
	print_ps_matrix(mat);
    	fprintf(fout," /%s /%s-Latin1 ChgFnt } def\n",titlefont,titlefont);
    	fprintf(fout,"/Titlef { %ld ",TitleDim);
	print_ps_matrix(mat);
    	fprintf(fout," /%s /%s-Latin1 ChgFnt } def\n",titlefont,titlefont);
    	fprintf(fout,"/Datef { %ld ",TitleDim-300);
	mat[0]*=0.83;
	print_ps_matrix(mat);
	fprintf(fout," /%s /%s-Latin1 ChgFnt } def\n",titlefont,titlefont);
	if (FancyMode)
	  {
    	    fprintf(fout,"/Gl        { newpath moveto %ld 0 rlineto 0 %ld rlineto -%ld 0 rlineto fill } def\n",
				LeftFrameWidth,FrameHeight,LeftFrameWidth);
    	    fprintf(fout,"/Gr        { newpath moveto %ld 0 rlineto 0 %ld rlineto -%ld 0 rlineto fill } def\n",
				RightFrameWidth,FrameHeight,RightFrameWidth);
    	    fprintf(fout,"/Gm        { newpath moveto %ld 0 rlineto 0 %ld rlineto -%ld 0 rlineto fill } def\n",
				FrameLength,FrameHeight,FrameLength);
    	    fprintf(fout,"/Header    { 0.80 setgray %ld %ld Gl %ld %ld Gr 0.93 setgray %ld %ld Gm } def\n", 
				XleftFrame,TitleY,right,TitleY,XleftFrame+LeftFrameWidth,TitleY);
	  }
      }

/* (string) U : print and underline the string, 
   x y h (string) x-200 y-200 Fr : print the string at (x,y) and paint around 
   				   it a shaded frame of height h. */
    fprintf(fout,"/U { currentpoint 3 -1 roll dup stringwidth pop 0 currentpoint\n");
    fprintf(fout,"     %ld sub newpath M rlineto stroke 3 1 roll M S} def\n",FontDim/9);
    fprintf(fout,"/M {moveto} def\n/S {show} def\n");
    fprintf(fout,"/Fr { Bf newpath moveto 0.94 setgray dup stringwidth pop 350 add dup 0\n");
    fprintf(fout,"  rlineto neg 3 2 roll 0 exch rlineto 0 rlineto fill 3 1 roll 0 setgray } def\n");
    fprintf(fout,"/#copies %hd def\n",Ncopies);
    fprintf(fout,"initgraphics\n");
    fprintf(fout,"1 100 div dup scale\n");
    if (LandscapeMode)
        fprintf(fout,"90 rotate 0 -%ld translate\n",PageHeight);

    if (WatermarkMode)
      {
	cosf=((float)PageWidth)/PageHeight;
	fprintf(fout,"/Wmf { 1 [%.2f 1 -1 %.2f 0 0] ",cosf,cosf);
	fprintf(fout,"/%s /%s-Latin1 ChgFnt } def\n",wm_font,wm_font);
	fprintf(fout,"/Wm { gsave dup Wmf stringwidth %ld ",PageHeight);
	fprintf(fout,"exch div 0.7 mul dup currentfont exch\n\tscalefont setfont ");
	fprintf(fout,"%ld %ld M ",PageWidth/2,PageHeight/2); 
	if (PageWidth>PageHeight)
	    fprintf(fout,"2000 exch 2.5 div neg ");
	else
	    fprintf(fout,"3.5 div -3500 ");
	fprintf(fout,"rmoveto pop dup\n");
	fprintf(fout,"\tstringwidth 2 div neg\n\texch 2 div neg exch rmoveto 0.96 setgray");
	fprintf(fout,"\n\tS grestore } def\n");
      }
    fprintf(fout,"/StartPage { /SavedPage save def\n  ");
    if (WatermarkMode)
      {
	print_string(wm_string);
        fprintf(fout," Wm\n");
      }

    if (HeaderMode)
      {
	if (FancyMode)
	    fprintf(fout,"  Header\n");
    	if (ShadowMode)
      	  {
            fprintf(fout,"  Datef  0 setgray %ld %ld M Centre dup S %ld %ld M Centre 1 setgray dup S\n",
		XleftFrame+LeftFrameWidth/2+120,ydate,XleftFrame+LeftFrameWidth/2,ydate+120);
            fprintf(fout,"  %ld %ld M Centre false charpath 0 setgray stroke\n",XleftFrame+LeftFrameWidth/2,ydate+120);
            fprintf(fout,"  Gpnf %ld %ld M Centre dup S %ld %ld M Centre 1 setgray dup S\n",right+RightFrameWidth/2+120,ypagn,right+RightFrameWidth/2,ypagn+120);
            fprintf(fout,"  %ld %ld M Centre false charpath 0 setgray stroke\n", right+RightFrameWidth/2,ypagn+120);
            fprintf(fout,"  Titlef %ld %ld M Centre dup S %ld %ld M Centre 1 setgray dup S\n",cx+120,ytitle,cx,ytitle+120);
            fprintf(fout,"  %ld %ld M Centre false charpath 0 setgray stroke\n",cx,ytitle+120);
      	  }
    	else
      	  {
	    fprintf(fout,"  Datef 0 setgray %ld %ld M Centre S\n",XleftFrame+LeftFrameWidth/2,ydate);
	    fprintf(fout,"  Gpnf %ld %ld M Centre S\n",right+RightFrameWidth/2,ypagn);
            fprintf(fout,"  Titlef %ld %ld M Centre S\n",cx,ytitle);
	  }
      }
    fprintf(fout,"  Rf 0 setgray } def\n");
    fprintf(fout,"/EndPage { ");
    if (Columns==2)
	fprintf(fout,"1 setlinewidth %ld %ld M 0 -%ld rlineto stroke ",cx,
	  STARTY+FontDim-350,TextHeight+400);
    fprintf(fout,"showpage SavedPage restore ");
    fprintf(fout,"} def\n%%EndProlog\n");
  }



/* End the postscript output */

void print_end()
  {
    if (!is_a_postscript_file)
    	fprintf(fout,"%%%%Trailer\n%%%%Pages: %hd\n%%%%EOF\n",doc_pages);
    return;
  }



/* Initiate a new page */

void start_page()
  { 
    page=newpage;
    doc_pages++;
    fprintf(fout,"%%%%Page: %hd %hd\n",doc_pages,doc_pages);
    if (HeaderMode)
    	fprintf(fout,"(%s) (%hd) (%s) ",title,page,date_s);
    fprintf(fout,"StartPage ");
    y=STARTY;
    x=xw=XleftFrame;
    yw=newpage=0;
  }



/* Write a string in the postscript page, at the (xw,yw) coordinates.
   If mode=1 the string is the header of a function, and the function
   print it using the bold font, and within a frame */

void outs(s,mode)
unsigned char *s;
short mode;
  {
    short i=0;
    unsigned char *p;

    if (!*s)
	return;
    if (newpage)
      {
	start_page();
	fprintf(fout,"%s ",font_wm[(oldwm=wm)]);
      }
    else if (wm!=oldwm)
	fprintf(fout,"%s ",font_wm[(oldwm=wm)]);
    p=s;
    if (y!=yw || mode)
      {
	while (*p==0x20)
	    p++;				/* skip the initial spaces */
	if ((i=p-s))
	  {
	    if (y!=yw)				/* and move the X coordinate */
    	    	xw+=(Widths[ROMAN][0x20]*i)/1000L;
	    else
	    	xw+=(Widths[wm][0x20]*i)/1000L;
	  }
	if (!*p)				/* if the string contains */
	    return;				/* only spaces, exit */
	yw=y;
	if (mode==0)				/* else move the currentpoint */
	    fprintf(fout,"\n%ld %ld M ",xw,yw);
      }
    if (mode==0)
      {
	print_string(p);
	fputc(( (wm==ITALIC && !Cmode) ? 'U':'S'),fout);
	fputc(0x20,fout);
      }
    else					/* print the string within a frame */
      {
	s=p+strlen((char*)p)-1;
	i=0;
    	while (*s==0x20)
	  {
	    i++;
	    *s--='\0';
	  }
        fprintf(fout,"\n%ld %ld %ld ",xw,yw,FontDim+200);
	print_string(p);
        fprintf(fout," %ld %ld Fr M S Rf ",xw-200,yw-250);
	xw+=(Widths[BOLD][0x20]/1000L)*i;
      }
    while (*p)
	xw+=(Widths[wm][*p++]/1000L);		/* update the X coordinate */
    return;
  }



/* Write in the Postscript output file the strings contained in the array
   bs[], using the font specified by the array ms[].
   If mode=1 the strings are printed in bold and a shaded frame is painted
   around them */

void empty_buffer(mode)
short mode;
  {
    long i, j, k, tmp, spos[768];
    short c='\0';
    unsigned char s[768];
    char s2[16];

    i=j=0;
    while (i<is)
      {
	if (NumberMode && newline)
	  {
	    if (startline)
		sprintf(s2,"%4ld:",line_number);
	    else
		strcpy(s2,"     ");
	    wm=ROMAN;
	    outs(s2,0);
	    for (k=0;s2[k];k++)
		Xpos+=Widths[ROMAN][(short)s2[k]];
	    yw=-1;
	  }
	wm=ms[i];
        while (!endpage && Xpos<MaxLength && i<is && (c=bs[i])!='\n' && wm==ms[i])
      	  {
	    if (c==0x0c)
		endpage=1;
	    else
	      {
	    	if (c=='\t')
	      	  {
		    Xpos+=Widths[wm][0x20]*(TabSize-ic%TabSize);
		    for (k=ic%TabSize;k<TabSize;k++,ic++)
		      {
			spos[j]=i+1;
		    	s[j++]=0x20;
		      }
	      	  }
	    	else
	      	  {
	    	    Xpos+=Widths[wm][c];
		    spos[j]=i;
	    	    s[j++]=c;
		    ic++;
	          }
	      }
	    i++;
	  }
	if (c!='\n' && RaggedMode && j>1)
	  {
	    tmp=j-1;
	    while (tmp>0 && s[tmp]!=0x20 && s[tmp]!='\t')
		tmp--; 
	    if (tmp)
	      { /*s[tmp]='\0'; fprintf(fout,"\ni:%ld -> %ld j:%ld tmp:%ld s:(%s)",i,i-(j-tmp),j,tmp,s); */
		i=spos[j=tmp];
	      }
	  }
	s[j]='\0';
	outs(s,((wm!=ROMAN)? 0:mode));
	j=0;
	if ( (newline=((startline=(c=='\n')) || Xpos>=MaxLength)) || endpage )
	  {
	    if (startline || endpage)
	      {
	    	line_number++;
		i++;
	      }
	    else if (TruncateMode)
	      {
		i++;
        	while (i<is && (c=bs[i])!='\n' && wm==ms[i])
      	  	    i++;
		line_number++;
		cut_lines++;
		startline=1;
	      }	 
	    y+=Decy;
	    Xpos=0L;
	    ic=0;
	    if ((y<ENDY || endpage) && !newpage)
	      {
		y=STARTY;
		yw=endpage=0;
		if (++column==Columns)
	          {
		    fprintf(fout,"\nEndPage\n");
		    column=0;
		    newpage=page+1;
	          }
	        else
		    x+=(TextWidth+Xgap);
	      }
	    xw=x;
	  }
      }
    is=0;
    return;
  }



/* Search for a function definition header: go backward the array bs[],
   skipping the comments and the preprocessor directives (ms[i]!=ROMAN);
   recognizes a function header if find a sequence as: ;.....(......)
		                                   or: }.....(......).
   Call empty_buffer() to write the chars contained in bs[] */

void look_for_header()
  {
    long i, j, k;
    short c='\0';

    j=is;
    i=is-2;
    while (ms[i]!=ROMAN || (c=bs[i])!=')')
      {
	if (i<=0 || ((c=='(' || c=='{' || c=='}') && ms[i]==ROMAN) )
	    return;
	i--;
      }
    k=i+1;
    i--;
    while (ms[i]!=ROMAN || (c=bs[i])!='(')
      {
	if (i<=0 || ((c==')' || c==';' || c=='}' || c=='}') && ms[i]==ROMAN) )
	    return;
	i--;
      }
    i--;
    while (ms[i]!=ROMAN || ((c=bs[i])!=';' && c!='}'))
      {
	if ((c=='{' || c==',' || c=='(') && ms[i]==ROMAN)
	    return;
	if (--i<0)
	    break;
      }
    if (i>=0)
      {
    	is=++i;
    	empty_buffer(0);
      }
    else
	i=0;
    for (is=0;i<k;i++,is++)
      {
	bs[is]=bs[i];
	ms[is]=ms[i];
      }
    empty_buffer(1);
    for (is=0;i<j;i++,is++)
      {
	bs[is]=bs[i];
	ms[is]=ms[i];
      }
    empty_buffer(0);
    return;
  }



/* Convert in PostScript an ASCII file.
   The input file is splitted in chunks of BUF_DIM chars, and copied in 
   the vector bs[]. The vector ms[] specifies, for each char, if it will
   be printed using the roman, italic or bold font.
   C input files are splitted in functions counting the number of '{' and 
   '}' met */

void translate_file(fin)
FILE *fin;
  {
    char begin[8];
    char PS_begin[8]; 
    short c='\0', oldc='\n';
    short i;
    short nwm=ROMAN;

    strcpy(PS_begin,"%!PS");
    if (!Print_Postscript)		/* postscript file ? */
      {
    	for (i=0;i<4;i++)
	    begin[i]=getc(fin);
    	for (i=3;i>=0;i--)
	    ungetc(begin[i],fin);
    	if (!strncmp(begin,PS_begin,4))
      	  {
	    is_a_postscript_file=1;
	    while ((c=getc(fin))!=EOF)
	    	fputc(c,fout);
	    return;
          }
      }
	    	
    if (!n_file || outfile)
      	print_prolog();
    line_number=startline=newline=newpage=page=1;
    Xpos=is=column=ic=endpage=0;
    wm=-1;
	
    if (Cmode)
      {
	while ((c=getc(fin))!=EOF)
	  {
	    if (c=='\r' || c==0x1a)
		continue;
	    bs[is]=c;
	    ms[is++]=nwm;
	    if (c=='"' && !comment && !chardef)
    	        string^=1;
	    else if (c==0x27 && !comment && !string)
	        chardef^=1;
            else if (!string && !chardef)
	      {
		if (c=='\n')
		  {
		    if (nwm==COMMENT && !(comment&=1))
		    	nwm=ROMAN;
		    else if (nwm==PREPROC && oldc!='\\')
			nwm=ROMAN;
		  }
	        else if (c=='*' && oldc=='/' && !comment)
	      	  {
		    nwm=COMMENT;
	    	    comment|=1;
		    if (is>1)
			ms[is-2]=nwm;
		    ms[is-1]=nwm;
	      	  }
	    	else if (c=='/' && oldc=='*' && comment&1)
	      	  {
		    nwm=ROMAN;
		    comment=0;
	      	  }
		else if (c=='/' && oldc=='/' && !comment)
		  { 
		    nwm=COMMENT;
	    	    comment|=2;
		    if (is>1)
			ms[is-2]=nwm;
		    ms[is-1]=nwm;
	     	  }
	    	else if (!comment)
		  {
		    if (c=='#' && oldc=='\n')
		      {
			nwm=PREPROC;
			ms[is-1]=nwm;
		      }
		    if (c=='{' && ++opengraph==1)
			look_for_header();
	    	    else if (c=='}' && --opengraph==0)
			empty_buffer(0);
		  }
	      }
	    oldc=c;
	  }
	empty_buffer(0); 
      }
    else
      {
	if ((oldc=getc(fin))==EOF)
	    ungetc(oldc,fin);
	while ((c=getc(fin))!=EOF)
	  {
	    if (c=='\r' || c==0x1a)
		continue;
	    else if (c==0x1b)
	      {
		c=getc(fin);
		c='\n';
	      }
	    else if (c==0x08)
	      {
	        do {
		    c=getc(fin);
		    if (c==oldc)
		    	nwm=BOLD;
		    else if (oldc=='_')
		        nwm=ITALIC;
		    oldc=c;
		    c=getc(fin);
		  } while (c==0x08);
	      }
	    else
	   	nwm=ROMAN;
	    bs[is]=oldc;
	    oldc=c;
	    ms[is++]=nwm;
	    if (is>=BUF_DIM)
		empty_buffer(0);
	  }
	bs[is]=oldc;
	ms[is++]=ROMAN;
	empty_buffer(0);
      }
    if (yw||column)
    	fprintf(fout,"\nEndPage\n");
    if (TruncateMode && cut_lines)
	fprintf(stderr,"Warning: file %s: %hd lines cutted\n",file_in,cut_lines);
  }



/* Print the usage message */

void usage()
  {
    fprintf(stderr,"Usage: atp [-BCghlNoprRsvVW] [-c columns] [-t tabsize] ");
    fprintf(stderr,"[-f font] [-F headings_font] [-cf comments_font] ");
    fprintf(stderr,"[-T papersize] [-title Title] [-date Date] ");
    fprintf(stderr,"[-M Tm,Bm,Lm,Rm [cm|in] ]  [-wm string [font] ]  [-# copies] "); 
    fprintf(stderr,"[-fonts] [-level1] [-postscript] files...\n");
    exit(-1);
  }



/* Print an error message */

void error_s(s)
char *s;
  {
    fprintf(stderr,"%s\n",s);
    exit(-1);
  }



/* Return the position of the font 'name' in the array font_names[] or
   -1 if the font is unknown */

short font_id(name)
char *name;
  {
    short i;

    for (i=0;strlen(font_names[i]);i++)
        if (!strcmp(font_names[i],name))
            break;
    if (!strlen(font_names[i]))
	return -1;
    else
	return i;
  }



/* Calculate the width of the string 's', printed in the font
   'font' and dimension 'dim', and modified by the matrix 'm'
   Return the width, or -1 if the fontname is unknown */

long width (s,font,dim,m)
char *s; char *font;
long dim; float *m;
  {
    long wid=0;
    short id;

    if ((id=font_id(font))<0)
	return -1;
    if (!strncmp(font,"Courier",7))
	wid=dim*.6*strlen(s);
    else
      {
	while (*s)
	    wid+=font_widths[id][(short)*s++];
    	wid*=(dim*m[0]/1000.0);
      }
    return wid;
  }



/* Manage the font selection: 
   input:	*p  (string in the format Courier8:1,0,0,1 or Courier8.)
   output:	*name=fontname (for instance, "Courier")
		*m=transformation matrix ([1,0,0,1])
   return 100*font dimension  (800) */

long select_font(p,name,m)
char *p; char *name; float *m;
  {
    short i=0;
    char num[64];
    long dim;

    while (*p && (*p<'0' || *p>'9'))
	name[i++]=*p++;
    name[i]='\0';
    i=0;
    while (*p && *p!=':')
	num[i++]=*p++;
    num[i]='\0';
    dim=100*atof(num);
    if (*p)
	if (sscanf(p+1,"%f,%f,%f,%f",&m[0],&m[1],&m[2],&m[3])!=4)
	    error_s("Insert the matrix as 'n1,n2,n3,n4', ni = floating points number");			
    if ((i=font_id(name))<0)
      {
        fprintf(stderr,"Unknown fontname %s\n",name);
	exit (-1);
      }
    if (dim<500 || dim>5000)
      {
    	fprintf(stderr,"Error: insert font as FontnameFontsize (i.e. Courier10,Helvetica-Bold8,...)\n");
    	fprintf(stderr,"Fontname must be a valid PostScript font name,\n");
    	fprintf(stderr,"Fontsize must be between 5 and 50\n");
    	exit(-1);
      }
    else
	return dim;
  }



void prepare_fonts(font)
char *font;
  {
    short i,j,bold,italic;

    for (i=0;strlen(font_names[i]);i++)
	if (!strcmp(font_names[i],font))
	    break;
    if (!strlen(font_names[i]))
	error_s("Invalid fontname");
    bold= i ^ BOLD;
    italic= i ^ ITALIC;
    strcpy(fontname[ROMAN],font_names[i]);
    strcpy(fontname[ITALIC],font_names[italic]);
    strcpy(fontname[BOLD],font_names[bold]);
    if (!strncmp("Courier",font,7))
	for (j=0;j<256;j++)
	    Widths[ROMAN][j]=Widths[ITALIC][j]=Widths[BOLD][j]=FontDim*600L;
    else 
	for (j=0;j<256;j++)
	  {
	    Widths[ROMAN][j]=FontDim*font_widths[i][j];
	    Widths[ITALIC][j]=FontDim*font_widths[italic][j];
	    Widths[BOLD][j]=FontDim*font_widths[bold][j];
	  }
    if (Cmode)
      {
    	for (i=0;strlen(font_names[i]);i++)
	    if (!strcmp(font_names[i],fontname[COMMENT]))
	    	break;
    	if (!strlen(font_names[i]))
	    error_s("Invalid fontname for comments");
    	if (!strncmp("Courier",fontname[COMMENT],7))
	    for (j=0;j<256;j++)
	    	Widths[COMMENT][j]=FontDim*600L;
    	else 
	    for (j=0;j<256;j++)
	    	Widths[COMMENT][j]=FontDim*font_widths[i][j];
      }

/* appl. matrix transformation */
    for (i=ROMAN;i<=BOLD;i++)
	for (j=0;j<256;j++)
	    Widths[i][j]*=Matrix[0];
    if (Cmode)
	for (j=0;j<256;j++)
	    Widths[COMMENT][j]*=comment_matrix[0]; 

    Decy=-1.096*FontDim*Matrix[3];
    return;
  }



int main(argc,argv) 
int argc;
char *argv[];
  {
    FILE *fin;
    time_t tim=time(NULL);
    char filename[128], textfont[128];
    char *c, option[64];		
    short i, Mod_Margins=0;
    long tmp, mtop[2], mbottom[2], mleft[2], mright[2];
    float mt,mb,ml,mr,mmul;

    mtop[0]=2200;    mtop[1]=1800;	/* default margins for Portrait */
    mbottom[0]=4800; mbottom[1]=4300;	/* 	     and Landscape mode */
    mleft[0]=5200;   mleft[1]=3600;
    mright[0]=3600;  mright[1]=3600;
#ifdef MSDOS
      bs = ( unsigned char *)malloc(BUF_DIM);
      ms = (char *)malloc(BUF_DIM);
      if(!bs || !ms){
              fprintf(stderr,"Memory allocation failure\n");
              exit(-2);
      }
#endif
    strcpy(datestr,ctime(&tim));
    strcpy(title,"");
    for(i=0;i<12;i++)
      {
	date_s[i]=datestr[i+4];
      }
    if (date_s[4]==' ')
      {
	date_s[4]=date_s[5];
	date_s[5]=' ';
      }	
    date_s[i]='\0';
    strcpy(wm_font,"Times-BoldItalic");
    strcpy(textfont,"Courier");
    strcpy(titlefont,"Helvetica-Bold");
    strcpy(fontname[COMMENT],"Courier-BoldItalic");  /* v1.1b -- changed from Palatino-Italic */

    argv++;
    argc--;
    while (argc && **argv=='-')
      {
	argc--;
        strcpy(option,*argv++);
        if (!strcmp(option,"-level1") || !strcmp(option,"-l1"))
            Level1=1;
        else if (!strcmp(option,"-fonts"))
          {
	    for (i=0;strlen(font_names[i]);i+=2)
		fprintf(stderr,"%-28s  %-28s\n",font_names[i],font_names[i+1]);
	    exit(0);
          }
	else if (!strcmp(option,"-title"))
	  {
	    argc--;
	    strcpy(title,*argv++);
	  }
        else if (!strcmp(option, "-date")) /* v1.1b -- added */
	  {
	    argc--;
	    strncpy(date_s, *argv++, 12);
	  }
	else if (!strcmp(option,"-#"))
	  {
	    argc--;
	    Ncopies=atoi(*argv++);
	    if (Ncopies<1)
	      {
		fprintf(stderr,"Wrong ncopies\n"); 
		usage();
	      }
	  }
	else if (!strcmp(option,"-postscript"))
	    Print_Postscript=1;
	else if (!strcmp(option,"-cf")) 
	  {
	    select_font(*argv,fontname[COMMENT],comment_matrix);
	    if (strchr(*argv,':'))
	    	Modify_comm_matrix=1;
	    else
		for (i=0;i<4;i++)
		    comment_matrix[i]=normal_matrix[i];
	    argc--;
	    argv++;
	  }
	else if (!strcmp(option,"-wm")) 
	  {
	    if (*argv)
	      {
		if (strlen(*argv)<=3)
		    sprintf(wm_string," %s ",*argv);
		else
		    strcpy(wm_string,*argv);
		WatermarkMode=1;
		argc--;
		argv++;
		if (*argv && ((i=font_id(*argv)) >=0) )
		  {
		    strcpy(wm_font,font_names[i]);
		    argc--;
		    argv++;
		  }
	      }
	    else
		error_s("Expecting string after -wm");
	  }
	else
	  {
	    c=option;
	    while (*(++c))
	      {
                if (*c=='o')
                    outfile=1;
                else if (*c=='p')
                  {
                    LandscapeMode=0;
                    FontDim=1000;
                  }
                else if (*c=='N')
                    NumberMode=1;
                else if (*c=='s')
                    ShadowMode=0;
                else if (*c=='W')
                    TruncateMode=1;
                else if (*c=='B')
                    HeaderMode=0;
                else if (*c=='g')
                    FancyMode=0;
        	else if (*c=='l')
		    ManualPageMode=1;
        	else if (*c=='r' || *c=='R')
		    RaggedMode=1;
                else if (*c=='C')
                    Cmode=1;
                else if (*c=='v' || *c=='V')
                    error_s(Version);
		else
		  {
		    if (!*argv)
		    	usage();
		    else
		    	argc--;
		    if (*c=='c')
	  	      {
	    	    	Columns=atoi(*argv++);
	    	    	if (Columns<1 || Columns>8)
			    error_s("Invalid columns number");
	  	      }
		    else if (*c=='t')
	  	      {
	    	    	TabSize=atoi(*argv++);
	    	    	if (TabSize<1 || TabSize>40)
			    error_s("Wrong tabsize\n"); 
	  	      }
		    else if (*c=='T')
	  	      {
	    	    	for (i=0;papersizes[i].name;i++)
			    if (!strcmp(*argv,papersizes[i].name))
		    	    	break;
	    	    	PageHeight=100L*papersizes[i].height;
	    	    	PageWidth=100L*papersizes[i].width;
	    	    	argv++;
	  	      }
		    else if (*c=='M')
	  	      {
    			if (*argv && sscanf(*argv,"%f,%f,%f,%f",&mt,&mb,&ml,&mr)==4)
      			  {
			    argv++;
			    if (*argv && !strcmp(*argv,"in"))
			      {
				mmul=7200;
				argv++;
				argc--;
			      }
			    else if (*argv && !strcmp(*argv,"cm"))
			      {
				mmul=2834.64;
                                argv++;
                                argc--;
                              }
			    else
				mmul=100.0;
			    m_top=mt*mmul;
			    m_bottom=mb*mmul;
			    m_left=ml*mmul;
			    m_right=mr*mmul;
			    Mod_Margins=1;
      			  }	
			else
    			    error_s("Insert margins as 'top,bottom,left,right [cm | in]'");			
		      }
		    else if (*c=='f') 
	  	      {
		    	FontDim=select_font(*argv,textfont,Matrix);
	    	    	if (strchr(*argv,':'))
			    Modify_Matrix=1;
		    	argv++;
		      }
		    else if (*c=='F') 
	  	      {
		    	TitleDim=select_font(*argv,titlefont,title_matrix);
		    	argv++; 
	  	      }
		    else
	    	    	usage();
		  }
	      }
	  }
      }

    if (!Columns)
	Columns=(LandscapeMode) ? 2:1;
    if (RaggedMode)
	TruncateMode=0;
    if (!PageWidth)
      {
#ifdef LETTER_SIZE         /* v1.1b -- added choice */
	PageWidth=61200;
	PageHeight=79200;
#else
        PageWidth=59500;
        PageHeight=84200;
#endif
      }
    if (LandscapeMode)
      {
	tmp=PageHeight;
	PageHeight=PageWidth;
	PageWidth=tmp;
      }
    if (ManualPageMode)
      {
	Columns= (LandscapeMode) ? 2:1;
	Modify_Matrix=1;
	HeaderMode=0;
      }
    if (!HeaderMode)
	TitleDim=FrameHeight=0;
    else
      {
        FrameHeight=1.15*TitleDim*title_matrix[3];
        if (FrameHeight<1400)
          {
             TitleDim=1200;
             FrameHeight=1400;
             title_matrix[0]/=title_matrix[3];
             title_matrix[3]=1.0;
          }
      }
    if (Cmode)
	TruncateMode=0;
   
/* Set margins */ 
    if (!Mod_Margins)
      {
    	i= (PageWidth > PageHeight) ? 1:0;
    	m_top=mtop[i];
    	m_bottom=mbottom[i];
    	m_left=mleft[i];
    	m_right=mright[i];
      }
    ENDY=m_bottom;
    XleftFrame=m_left;
    Xgap=(PageWidth-7200)/19.25-200*(Columns-2);
    TextWidth=(PageWidth-m_left-m_right-Xgap*(Columns-1))/Columns;
    if (TextWidth < 5000)
	error_s("Wrong margins: text width too small !!!");
    TitleY=PageHeight-FrameHeight-m_top;
    if (HeaderMode)
    	STARTY=TitleY-((PageWidth > PageHeight) ? 1700:2200);
    else
	STARTY=PageHeight-m_top-1000;
    if ((TextHeight=STARTY-ENDY) <8000)
	error_s("Wrong margins: text height too small !!!");
    MaxLength=1000*TextWidth;
    LeftFrameWidth=300+0.83*width(date_s,titlefont,TitleDim,title_matrix);
    if (LeftFrameWidth<7200)
	LeftFrameWidth=7200;
    RightFrameWidth=300+width("48",titlefont,TitleDim,title_matrix);
    if (RightFrameWidth<4800)
	RightFrameWidth=4800;
    if (ManualPageMode)
        FontDim=TextHeight/71.6;	/* 66 lines per page: 71.6 ~ 65.3*1.096 */

    if (!Modify_Matrix && Cmode && !strncmp(textfont,"Courier",7))
	for (i=0;i<4;i++)
	    Matrix[i]=slim_matrix[i];
    else if (Cmode && Modify_Matrix && !Modify_comm_matrix)
    	for (i=0;i<4;i++)
            comment_matrix[i]=Matrix[i];
    
    prepare_fonts(textfont);

    if (!argc) 				/* read from standard input */
      {
	fin=stdin;
	strcpy(file_in,"stdin");
        if (outfile)
          {
	    if (strlen(title))
	    	sprintf(filename,"%s.ps",title);
	    else
		sprintf(filename,"stdin.ps");
    	    if ((fout=fopen(filename,"w"))==NULL)
      	      {
	    	fprintf(stderr,"Can't open the file %s\n",filename);
	    	return(1);
      	      }
          }
    	else
	    fout=stdout;
        translate_file(fin);
	print_end();
      }
    else
      {
	if (!strlen(title))
            strcpy(title,*argv);
	if (!outfile)
	    fout=stdout;
	while (argc)	 		/* read from input files */
          {
     	    if ((fin=fopen(*argv,"rb"))==NULL)
      	      {
	    	fprintf(stderr,"Can't open the file %s\n",*argv);
	    	return(1);
              }
	    strcpy(file_in,*argv);
			
	    if (outfile)
	      {
            	sprintf(filename,"%s.ps",*argv);
                if ((fout=fopen(filename,"w"))==NULL)
                  {
            	    fprintf(stderr,"Can't open the file %s\n",filename);
            	    return(1);
              	  }
		translate_file(fin);
		print_end();
	      }
	    else
       	    	translate_file(fin);
   	    argv++;
   	    if (--argc)
    	    	strcpy(title,*argv);
	    fclose(fin);
	    if (outfile)
    	        fclose(fout);
	    n_file++;
	  }
	if (!outfile)
	    print_end();
      }
    return 0;
  }
