/*
** oidentd - ojnk ident daemon
** Copyright (C) 1998,1999,2000 Ryan McCabe <odin@numb.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License, version 2,
** as published by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA
**
** $Id: oidentd.c,v 1.11 2000/01/15 18:48:52 odin Exp $
*/

#define _GNU_SOURCE
#include <config.h>
#include <stdlib.h>
#include <signal.h>
#include <oidentd.h>

static int setup_listen(int listen_port, u_long listen_addr);
static int drop_privs(uid_t new_uid, gid_t new_gid);
static int service_request(int sock);
static int read_permline(FILE *f, u_char *buf, int bufl);
static int check_perm(const u_char *name, u_char *suser, size_t slen);
static int open_ispoof(const u_char *path, const struct passwd *pw);
static int check_noident(const u_char *homedir);
static int go_background(int ignore);
static int get_port(const u_char *name);
static int valid_number(const u_char *p);
static void main_loop(void);
static void sigchld(int sig);
static void sigalrm(int sig);
static void print_usage(const u_char *whoami);
static void print_version(void);
static void fix_ident(uid_t cur_uid, const u_char *nam, u_char *suser, size_t len);
static u_char *hostlookup(u_long lookup_addr, u_char *hostname, size_t len);

#if defined(__linux__) && defined(MASQ_SUPPORT)
#	define OPTSTRING "a:Ac:def:Fg:himnNoO:p:P:qrsSt:T:u:vVwWx:"
	int fwdport, fsock;
	u_long proxy;
#else
#	define OPTSTRING "a:Ac:deg:hinNoO:p:qrsSt:T:u:vVwWx:"
#endif

u_char *ret_os = "UNIX";
u_char *charset = NULL;
u_int flags = 0;

static u_char *failuser = NULL;

static u_int timeout = DEFAULT_TIMEOUT, wtimeout = DEFAULT_WTIMEOUT;
static int port;
static u_long addr;
static uid_t uid = 65534;
static gid_t gid = 65534;

int main(int argc, char **argv) {
	int opt;
	struct passwd *pw;
	struct group *gr;

	port = get_port(DEFAULT_PORT);
	addr = htonl(INADDR_ANY);

	pw = getpwnam("nobody");
	if (pw != NULL)
		uid = pw->pw_uid;

	gr = getgrnam("nobody");
	if (gr != NULL)
		gid = gr->gr_gid;
	else {
		gr = getgrnam("nogroup");
		if (gr != NULL)
			gid = gr->gr_gid;
	}
	
	while ((opt = getopt(argc, argv, OPTSTRING)) != EOF) {
		switch (opt) {
			case 'a':
				if (get_addr(optarg, &addr) == -1) {
					fprintf(stderr, "Fatal: Unknown host: %s\n", optarg);
					exit(-1);
				}
				break;
			case 'A':
				flags |= ALL;
				break;
			case 'c':
				charset = optarg;
				break;
			case 'd':
				flags |= DEBUG;
				break;
			case 'e':
				flags |= HIDEE;
				break;
#if defined(__linux__) && defined(MASQ_SUPPORT)
			case 'f':
				flags |= FWD;
				fwdport = get_port(optarg);
				break;
			case 'F':
				flags |= FWD;
				fwdport = get_port(DEFAULT_PORT);
				break;
			case 'm':
				flags |= MASQ;
				break;
			case 'P':
				flags |= PROXY;
				if (get_addr(optarg, &proxy) == -1) {
					fprintf(stderr, "Fatal: Unknown host: %s\n", optarg);
					exit(-1);
				}
				break;
#endif
			case 'g':
				flags |= GID;
				if (valid_number(optarg))
					gid = (gid_t) atol(optarg);
				else {
					gr = getgrnam(optarg);
					if (gr == NULL) {
						fprintf(stderr, "Fatal: Unknown group: %s\n", optarg);
						exit(-1);
					}
					gid = gr->gr_gid;
				}
				break;
			case 'i':
				flags |= INETD;
				break;
			case 'n':
				flags |= NUMERIC;
				break;
			case 'N':
				flags |= NOIDENT;
				break;
			case 'o':
				flags |= HIDEO;
				break;
			case 'O':
				ret_os = optarg;
				break;
			case 'p':
				port = get_port(optarg);
				break;
			case 'q':
				flags |= QUIET;
				break;
			case 'r':
				flags |= RANDOM;
				break;
			case 's':
				flags |= SPOOF;
				break;
			case 'S':
				flags |= (SPOOF | REVSPOOF);
				break;
			case 't':
				if (!valid_number(optarg)) {
					fprintf(stderr, "Fatal: Bad timeout value: %s\n", optarg);
					exit(-1);
				}
				timeout = strtoul(optarg, NULL, 10);
				break;
			case 'T':
				if (!valid_number(optarg)) {
					fprintf(stderr, "Fatal: Bad wtimeout value: %s\n", optarg);
					exit(-1);
				}
				wtimeout = strtoul(optarg, NULL, 10);
				break;
			case 'u':
				flags |= UID;
				if (valid_number(optarg))
					uid = (uid_t) atol(optarg);
				else {
					pw = getpwnam(optarg);
					if (pw == NULL) {
						fprintf(stderr, "Fatal: Unknown user: %s\n", optarg);
						exit(-1);
					}
					uid = pw->pw_uid;
				}
				break;
			case 'v':
			case 'V':
				print_version();
				exit(0);
			case 'w':
				flags |= (WAIT | INETD);
				break;
			case 'W':
				flags |= WRAPPED;
				break;
			case 'x':
				failuser = optarg;
				break;
			case 'h':
			default:
				print_usage(argv[0]);
				exit(0);
		}
	}

	if ((flags & DEBUG) && (flags & QUIET)) {
		fprintf(stderr, "Fatal: The debug and quiet flags are incompatible.\n");
		exit(-1);
	}

	if ((flags & NUMERIC) && (flags & RANDOM)) {
		fprintf(stderr,
			"Fatal: The numeric and random flags are incompatible\n");
		exit(-1);
	}

	if (!VALID_PORT(port)) {
		fprintf(stderr, "Fatal: Bad port: %d\n", port);
		exit(-1);
	}

	main_loop();

	exit(0);
}

/*
** Setup the daemon and handle new connections.
*/

static void main_loop(void) {
	int listenfd = -1, connectfd, len = sizeof(struct sockaddr);
	struct sockaddr cliaddr;

	if (!(flags & INETD)) {
		listenfd = setup_listen(port, addr);

		if (listenfd == -1) {
			fprintf(stderr, "Fatal: Unable to setup listening socket.\n");
			exit(-1);
		}

		if (go_background(listenfd) == -1) {
			fprintf(stderr, "Fatal: Couldn't properly fork into the back.\n");
			exit(-1);
		}

		if (drop_privs(uid, gid) == -1) {
			fprintf(stderr, "Fatal: Couldn't drop privileges.\n");
			exit(-1);
		}
	}

	openlog("oidentd", LOG_PID | LOG_CONS | LOG_NDELAY, FACILITY);

	if (flags & RANDOM) {
		struct timeval tv;

		gettimeofday(&tv, NULL);
		srandom(tv.tv_sec ^ (tv.tv_usec << 11));
	}

#ifdef USE_KMEM
	if (k_open()) {
		if (flags & DEBUG)
			syslog(DPRI, "Error: Can't open kernel memory device, exiting");
		exit(-1);
	}
#endif

	signal(SIGALRM, sigalrm);
	signal(SIGCHLD, sigchld);

	if (!(flags & INETD)) {
		for (;;) {
			connectfd = accept(listenfd, &cliaddr, &len);

			if (connectfd == -1)
				continue;

			if (!fork()) {
				close(listenfd);
				alarm(timeout);
				service_request(connectfd);
				exit(0);
			}
			close(connectfd);
		}
	} else {
		if (!(flags & WAIT)) {
			if (drop_privs(uid, gid) == -1) {
				if (flags & DEBUG)
					syslog(DPRI, "Fatal: Couldn't drop privs.");
				exit(-1);
			}
			alarm(timeout);
			service_request(STDIN_FILENO);
			close(STDIN_FILENO);
			exit(0);
		} else {
			int ret = -1;
			fd_set readfds;
			struct timeval tv;

			if (listen(STDIN_FILENO, 3) == -1) {
				if (flags & DEBUG)
					syslog(DPRI, "Fatal: listen: %s", strerror(errno));
				exit(-1);
			}

			if (drop_privs(uid, gid) == -1) {
				if (flags & DEBUG)
					syslog(DPRI, "Fatal: Couldn't drop privs.");
				exit(-1);
			}
			
			for (;;) {

				FD_ZERO(&readfds);

				do {
					FD_SET(STDIN_FILENO, &readfds);

					tv.tv_sec = wtimeout;
					tv.tv_usec = 0;

					ret = select(STDIN_FILENO + 1, &readfds, NULL, NULL, &tv);
				} while (ret < 0 && errno == EINTR);

				if (ret == 0)
					exit(0);

				else {
					connectfd = accept(STDIN_FILENO, &cliaddr, &len);

					if (connectfd == -1) {
						if (flags & DEBUG)
							syslog(DPRI, "Error: accept: %s", strerror(errno));
						continue;
					}

					if (!fork()) {
						close(STDIN_FILENO);
						alarm(timeout);
						service_request(connectfd);
						exit(0);
					}

					close(connectfd);
				}
			}
		}
	}
}

/*
** Setup the listening socket.
*/

static int setup_listen(int listen_port, u_long listen_addr) {
	const int one = 1;
	int listenfd, ret;
	struct sockaddr_in sl_sin;

	memset(&sl_sin, 0, sizeof(sl_sin));

	sl_sin.sin_family = AF_INET;
	sl_sin.sin_port = htons(listen_port);
	sl_sin.sin_addr.s_addr = listen_addr;

	listenfd = socket(PF_INET, SOCK_STREAM, 0);

	if (listenfd == -1) {
		if (flags & DEBUG)
			syslog(DPRI, "Error: socket: %s", strerror(errno));
		return (-1);
	}

	ret = setsockopt(listenfd, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one));

	if (ret == -1) {
		if (flags & DEBUG)
			syslog(DPRI, "Error: setsockopt: %s", strerror(errno));
		return (-1);
	}

	ret = bind(listenfd, (struct sockaddr *) &sl_sin, sizeof(struct sockaddr));
	
	if (ret == -1) {
		if (flags & DEBUG)
			syslog(DPRI, "Error: bind: %s", strerror(errno));
		return (-1);
	}

	if (listen(listenfd, 3) == -1) {
		if (flags & DEBUG)
			syslog(DPRI, "Error: listen: %s", strerror(errno));
		return (-1);
	}

	return (listenfd);
}

/*
** Drop privileges and run with specified uid/gid.
*/

static int drop_privs(uid_t new_uid, gid_t new_gid) {
	int ret;

	if (flags & GID) {
		if (setgid(new_gid) == -1) {
			if (flags & DEBUG)
				syslog(DPRI, "Error: setgid(%u): %s",
					new_gid, strerror(errno));
			return (-1);
		}
	}

	if (flags & UID) {
		struct passwd *pw = getpwuid(new_uid);

		if (pw == NULL) {
			if (flags & DEBUG) {
				syslog(DPRI, "Error: getpwuid(%u): %s",
					new_uid, strerror(errno));
			}

			return (-1);
		}

		ret = initgroups(pw->pw_name, ((flags & GID) ? new_gid : pw->pw_gid));

		if (ret == -1) {
			if (flags & DEBUG) {
				syslog(DPRI, "Error: initgroups(%s, %u): %s",
					pw->pw_name, gid, strerror(errno));
			}

			return (-1);
		}

		if (setuid(new_uid) == -1) {
			if (flags & DEBUG) {
				syslog(DPRI, "Error: setuid(%u): %s",
					new_uid, strerror(errno));
			}

			return (-1);
		}
	}
	
	return (0);
}

/*
** Handle the client's request: read the client data and send the identd reply.
*/

static int service_request(int sock) {
	int lport, fport, con_uid, len = sizeof(struct sockaddr_in);
	uid_t orig_uid;
	u_char line[128], suser[MAX_ULEN], orig_suser[MAX_ULEN];
	struct sockaddr_in sr_sin;
	struct in_addr laddr, faddr;
	struct passwd *pw;

	if (getpeername(sock, (struct sockaddr *) &sr_sin, &len) == -1) {
		if (flags & DEBUG)
			syslog(DPRI, "Error: getpeername: %s", strerror(errno));

		return (-1);
	}

	faddr = sr_sin.sin_addr;

	if (!(flags & (WRAPPED | QUIET))) {
		u_char host_buf[256];

		syslog(PRIORITY, "Connection from %s:%d",
			hostlookup(sr_sin.sin_addr.s_addr, host_buf, sizeof(host_buf)),
			htons(sr_sin.sin_port));
	}

	if (getsockname(sock, (struct sockaddr *) &sr_sin, &len) == -1) {
		if (flags & DEBUG)
			syslog(DPRI, "Error: getsockname: %s", strerror(errno));

		return (-1);
	}

	laddr = sr_sin.sin_addr;
	len = sockread(sock, line, sizeof(line));

	if (len <= 0)
		return (-1);

	len = sscanf(line, "%d , %d", &lport, &fport);

	if (len < 2 || !VALID_PORT(lport) || !VALID_PORT(fport)) {
		dprintf(sock, "%d , %d : ERROR : %s\r\n",
				lport, fport, ERROR("INVALID-PORT"));

		if (!(flags & QUIET)) {
			syslog(PRIORITY, "[%s] %d , %d : ERROR : INVALID-PORT",
				inet_ntoa(faddr), lport, fport);
		}

		return (0);
	}

#ifdef __linux__
	con_uid = get_user(lport, fport, &laddr, &faddr);
#else
	con_uid = get_user(htons(lport), htons(fport), &laddr, &faddr);
#endif

	if (con_uid == -1) {
#if defined(__linux__) && defined(MASQ_SUPPORT)
		if (flags & MASQ) {
			if (!masq(sock, lport, fport, &faddr))
				return (0);
		}
#endif
		if (failuser != NULL) { 
			dprintf(sock, "%d , %d : USERID : %s%s%s : %s\r\n", lport, fport,
					OS(ret_os), (charset != NULL ? " , " : ""),
					(charset != NULL ? charset : (u_char *) ""), failuser);

			if (!(flags & QUIET)) {
				syslog(PRIORITY, "[%s] Failed lookup: %d , %d : (returned %s)",
					inet_ntoa(faddr), lport, fport, failuser);
			}
		} else {
			dprintf(sock, "%d , %d : ERROR : %s\r\n",
					lport, fport, ERROR("NO-USER"));

			if (!(flags & QUIET)) {
				syslog(PRIORITY, "[%s] %d , %d : ERROR : NO-USER",
					inet_ntoa(faddr), lport, fport);
			}
		}

		return (0);
	}

	pw = getpwuid(con_uid);

	if (pw == NULL) {
		dprintf(sock, "%d , %d : ERROR : %s\r\n",
				lport, fport, ERROR("NO-USER"));

		if (!(flags & QUIET)) {
			syslog(PRIORITY, "[%s] %d , %d : ERROR : getpwuid",
				inet_ntoa(faddr), lport, fport);
		}

		return (0);
	}
	
	if (flags & NOIDENT) {
		if (check_noident(pw->pw_dir)) {
			dprintf(sock, "%d , %d : ERROR : %s\r\n",
					lport, fport, ERROR("HIDDEN-USER"));

			if (!(flags & QUIET)) {
				syslog(PRIORITY, "[%s] %d , %d : ERROR : HIDDEN-USER (%s)",
					inet_ntoa(faddr), lport, fport, pw->pw_name);
			}

			return (0);
		}
	}

	strncpy(orig_suser, pw->pw_name, sizeof(orig_suser));
	orig_suser[sizeof(orig_suser) - 1] = '\0';
	orig_uid = pw->pw_uid;

	if (flags & SPOOF) {
		int ret = check_perm(pw->pw_name, suser, sizeof(suser));

		if (ret == SPOOF_SET)
			goto response;

		/*
		** root is always allowed to spoof identd replies
		*/
		if (ret == SPOOF_OK || pw->pw_uid == 0) {
			u_char *spath = NULL;
			int fd;

			spath = xmalloc(strlen(pw->pw_dir) + sizeof(ISPOOF) + 1);
			strcpy(spath, pw->pw_dir);
			strcat(spath, ISPOOF);

			fd = open_ispoof(spath, pw);

			if (fd != -1) {
				u_char *temp = NULL;

				len = read(fd, suser, sizeof(suser) - 1);
				close(fd);
				if (len > 0) {
					suser[len] = '\0';

					temp = strchr(suser, '\n');

					if (temp != NULL)
						*temp = '\0';
				} else {
					if (flags & DEBUG) {
						syslog(DPRI, "Error reading file \"%s\" : %s",
							spath, strerror(errno));
					}

					fix_ident(pw->pw_uid, pw->pw_name, suser, sizeof(suser));
				}
			} else
				fix_ident(pw->pw_uid, pw->pw_name, suser, sizeof(suser));

			free(spath);
		} else 
			fix_ident(pw->pw_uid, pw->pw_name, suser, sizeof(suser));

		if (pw->pw_uid) {
			struct passwd *check = NULL;
			check = getpwnam(suser);

			if (check && check->pw_uid != orig_uid) {
				syslog(LOG_CRIT, "User %s tried to masquerade as user %s.",
					orig_suser, check->pw_name);

				fix_ident(orig_uid, orig_suser, suser, sizeof(suser));
			}

			if (fport < 1024 && !(flags & ALL))
				fix_ident(orig_uid, orig_suser, suser, sizeof(suser));
		}
	}

	if (!(flags & SPOOF))
		fix_ident(orig_uid, orig_suser, suser, sizeof(suser));


response:

	dprintf(sock, "%d , %d : USERID : %s%s%s : %s\r\n", lport, fport,
		OS(ret_os), (charset != NULL ? " , " : ""),
		(charset != NULL ? charset : (u_char *) ""), suser);

	if (!(flags & QUIET)) {
		syslog(PRIORITY, "[%s] Successful lookup: %d , %d : %s (%s)",
			inet_ntoa(faddr), lport, fport, suser, orig_suser);
	}

	return (0);
}

/*
** Read a line from the identd.spoof file.  If too long, return empty
** Strips leading and trailing whitespace
** Returns EOF or 0
*/

static int read_permline(FILE *f, u_char *buf, int bufl) {
	int c,i;
	bufl--;		/* Reserve space for \0 */

	for (i = 0 ; ;) {
		c = getc(f);

		switch (c) {
			case EOF:
				if (!i)
					return EOF;
				/* FALLTHROUGH */
			case '\n':
				if (i >= bufl)
					i = 0;
				while (i > 0 && isspace(buf[i - 1]))
					i--;
				buf[i] = '\0';

				return (0);
			default:
				if (i == 0 && isspace(c))
					break;
				if (i < bufl)
					buf[i++] = c;
		}
	}

	/* NOTREACHED */
	return (0);
}

/*
** Check to make sure user is authorized to spoof identd replies
** (is in the /etc/identd.spoof file).
*/

static int check_perm(const u_char *name, u_char *suser, size_t slen) {
	FILE *fp = NULL;
	u_char line[24], *p;
	int ret = SPOOF_NOPERM;

	fp = fopen(USERFILE, "r");

	if (fp == NULL) {
		if (flags & DEBUG) {
			syslog(DPRI, "Cannot open file: %s: %s",
				USERFILE, strerror(errno));
		}

		return (flags & REVSPOOF ? 1 : 0);
	}

	while (read_permline(fp, line, sizeof(line)) != EOF) {
		if (*line == '#')
			continue;

		p = strchr(line, ':');
		if (p != NULL)
			*p++ = '\0';

		if (!strcmp(line, name)) {
			if (p == NULL || *p == '\0')
				ret = SPOOF_OK;
			else {
				ret = SPOOF_SET;
				strncpy(suser, p, slen);
				suser[slen - 1] = '\0';
			}

			break;
		}
	}

	fclose(fp);

	return ((flags & REVSPOOF) ? !ret : ret);
}

/*
** Generate a pseudo-random identd reply or toss pw_name back 
** depending on the value of the randomize flag.
*/

static void fix_ident(uid_t cur_uid, const u_char *nam, u_char *suser, size_t len) {
#ifdef NEW_RANDOM
	u_short i;
	static const u_char valid[] = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
#endif

	if (flags & NUMERIC) {
#ifdef HAVE_SNPRINTF
		snprintf(suser, len, "%d", cur_uid);
#else
		sprintf(suser, "%d", cur_uid);
#endif
		return;
	} else if (flags & RANDOM) {
		if (!(flags & INETD)) {
			struct timeval tv;

			gettimeofday(&tv, NULL);
			srandom(tv.tv_sec ^ (tv.tv_usec << 11));
		}

#	ifndef NEW_RANDOM
#		ifdef HAVE_SNPRINTF
		snprintf(suser, len, "%s%u", UPREFIX, random() % 100000);
#		else /* ! HAVE_SNPRINTF */
		sprintf(suser, "%s%u", UPREFIX, random() % 100000);
#		endif /* HAVE_SNPRINTF */
#	else /* ! NEW_RANDOM */

		for (i = 0 ; i < len - 1 ; i++)
			suser[i] = valid[random() % (sizeof(valid) - 1)];

		suser[i] = '\0';
#	endif /* NEW_RANDOM */
	} else {
		strncpy(suser, nam, len);
		suser[len - 1] = '\0';
	}
}

/*
** Open an ".ispoof" file if it exists and matches the uid of the requestee.
*/

static int open_ispoof(const u_char *path, const struct passwd *pw) {
	struct stat st;
	int fd;

	if (stat(path, &st) == -1) {
		if (flags & DEBUG)
			syslog(DPRI, "Error: stat: %s: %s", path, strerror(errno));

		return (-1);
	}

	if (st.st_uid != pw->pw_uid) {
		/*
		** Someone's up to no good.
		*/
		syslog(LOG_CRIT,
			"Refused to read \"%s\" during request for %s (owner is UID %d)",
			path, pw->pw_name, st.st_uid);

		return (-1);
	}

	fd = open(path, O_RDONLY);

	if (fd == -1) {
		if (flags & DEBUG) {
			syslog(DPRI, "Error reading file \"%s\" : %s", path,
				strerror(errno));
		}

		return (-1);
	}

	if (fstat(fd, &st) == -1) {
		if (flags & DEBUG)
			syslog(DPRI, "Error: stat: %s: %s", path, strerror(errno));

		close(fd);
		return (-1);
	}

	/*
	** Close the stat-open race.
	*/
	if (st.st_uid != pw->pw_uid) {
		syslog(LOG_CRIT,
			"Refused to read \"%s\" during request for %s (owner is UID %d)",
			path, pw->pw_name, st.st_uid);

		close(fd);
		return (-1);
	}

	return (fd);
}

/*
** Read a socket. (UNP v2)
*/

ssize_t sockread(int fd, u_char *buf, size_t len) {
	u_char c;
	u_int i;
	ssize_t ret;

	if (!buf)
		return (-1);

	for (i = 1; i < len ; i++) {
		ojnk:
			if ((ret = read(fd, &c, 1)) == 1) {
				*buf++ = c;
				if (c == '\n')
					break;
			} else if (ret == 0) {
				if (i == 1)
					return (0);
				else
					break;
			} else {
				if (errno == EINTR)
					goto ojnk;

				return (-1);
			}
	}

	*buf = '\0';
	return (i);
}

/*
** Check for the existence of a ".noident" file.
*/

static int check_noident(const u_char *homedir) {
	u_char *file = NULL;
	struct stat sbuf;
	char ret = 0;

	file = (u_char *) xmalloc(strlen(homedir) + sizeof(NOIDENTF) + 1);
	strcpy(file, homedir);
	strcat(file, NOIDENTF);

	if (!lstat(file, &sbuf))
		ret = 1;

	free(file);
	return (ret);
}

/*
** Go background.
*/

static int go_background(int ignore) {
	int fd;
	
	switch (fork()) {
		case -1:
			return (-1);
		case 0:
			break;
		default:
			_exit(0);
	}

	if (setsid() == -1)
		return (-1);

	chdir("/");
	umask(DEFAULT_UMASK);

	fd = open("/dev/null", O_RDWR);
	
	if (fd == -1)
		return (-1);

	dup2(fd, 0);
	dup2(fd, 1);
	dup2(fd, 2);

	for (fd = 3; fd < 64 ; fd++) {
		if (fd != ignore)
			close(fd);
	}
	
	return (0);
}

void *xmalloc(size_t size) {
	void *temp = malloc(size);

	if (temp == NULL) {
		if (flags & DEBUG)
			syslog(DPRI, "fatal: malloc: no memory left.");

		exit(-1);
	}

	return (temp);
}

#ifndef HAVE_DPRINTF
int dprintf(int fd, const char *fmt, ...) {
	va_list ap;
#ifdef HAVE_VASPRINTF
	u_char *buf;
	ssize_t ret;

	va_start(ap, fmt);
	ret = vasprintf((char **) &buf, fmt, ap);
	va_end(ap);

	ret = write(fd, buf, ret);
	free(buf);

	return (ret);
#else /* ! HAVE_VASPRINTF */
	u_char buf[4096];

	va_start(ap, fmt);
	/* XXX */
#ifdef HAVE_VSNPRINTF
	vsnprintf(buf, sizeof(buf), fmt, ap);
#else
	/*
	** This is safe for this program, as no string can ever be anywhere close
	** to 4k.  Also, all data that is dprintf'd either comes from the kernel
	** or the user who started the daemon.
	*/
	vsprintf(buf, fmt, ap);
#endif
	va_end(ap);

	return (write(fd, buf, strlen(buf)));
#endif /* HAVE_VASPRINTF */
}
#endif /* HAVE_DPRINTF */

/*
** Return an IP address and/or hostname.
*/

static u_char *hostlookup(u_long lookup_addr, u_char *hostname, size_t len) {
	struct hostent *host = NULL;
	struct in_addr in;

	in.s_addr = lookup_addr;

	host = gethostbyaddr((char *) &in, sizeof(struct in_addr), AF_INET);

	if (host != NULL) {
		if (strlen(host->h_name) + sizeof(" (123.123.123.123)") < len) {
#ifdef HAVE_SNPRINTF
				snprintf(hostname, len, "%s (%s)", host->h_name, inet_ntoa(in));
#else
				sprintf(hostname, "%s (%s)", host->h_name, inet_ntoa(in));
#endif
			return (hostname);
		}
	}

	strncpy(hostname, inet_ntoa(in), len);
	hostname[len - 1] = '\0';

	return (hostname);
}

/*
** Get the port associated with a tcp service name.
*/

static int get_port(const u_char *name) {
	struct servent *servent;

	if (valid_number(name))
		return (atoi(name));

	servent = getservbyname(name, "tcp");
	if (servent != NULL)
		return (ntohs(servent->s_port));

	return (-1);
}

/*
** Return a 32-bit, network byte ordered ipv4 address.
*/

int get_addr(const u_char *const hostname, u_long *g_addr) {
#ifdef HAVE_INET_ATON
	struct in_addr in;

	if (inet_aton(hostname, &in)) {
		*g_addr = in.s_addr;

		return (0);
#else
	*g_addr = inet_addr(hostname);
	if (*g_addr != -1UL) {

		return (0);
#endif
	} else {
		struct hostent *host = gethostbyname(hostname);
		if (host) {
			*g_addr = *((u_long *) host->h_addr);
			return (0);
		}
	}

	return (-1);
}

/*
** Returns non-zero if p points to a valid decimal number
*/

static int valid_number(const u_char *p) {
	/*
	** Make sure that the user isn't really trying to specify an octal
	** number... the only valid number that starts with zero is zero
	** itself
	**/
	if (*p == '0')
		return (p[1] == '\0');
	if (*p == '-')
		p++;
	if (*p == '\0' || *p == '0')
		return (0);
	for (; *p != '\0' ; p++) {
		if (!isdigit(*p))
			return (0);
	}

	return (1);
}

/*
** Handle SIGCHLD.
*/

static void sigchld(int sig) {
	while (waitpid(-1, &sig, WNOHANG) > 0) ;

	signal(SIGCHLD, sigchld);
}

/*
** Handle SIGALRM.
*/

static void sigalrm(int sig) {
	if (!(flags & QUIET))
		syslog(PRIORITY, "Timeout for request.  Closing connection.");

	exit(0);
	(void) sig;
}

static void print_usage(const u_char *whoami) {
	print_version();
	printf("Usage: %s [options]\n"
"  -a <address>\tBind to <address>. (Defaults to INADDR_ANY)\n"
"  -A\t\tWhen spoofing is enabled, enable users to spoof\n"
"    \t\tident on connections to privileged ports.\n"
"  -c <charset>\tSpecify an alternate charset. (Defaults to \"US-ASCII\")\n"
"  -d\t\tEnable debugging.\n"
"  -e\t\tReturn \"UNKNOWN-ERROR\" for all errors.\n"
#if defined(__linux__) && defined(MASQ_SUPPORT)
"  -f <port>\tForward requests for masqueraded hosts to the host on <port>\n"
"  -F\t\tSame as -f, but always use the default port (113) by default\n"
#endif
"  -g <gid>\tRun with specified gid.\n"
"  -i\t\tRun from inetd.\n"
#if defined(__linux__) && defined(MASQ_SUPPORT)
"  -m\t\tEnable support for IP masquerading.\n"
#endif
"  -n\t\tReturn UIDs instead of usernames\n"
"  -N\t\tAllow identd hiding via \".noident\"\n"
"  -o\t\tReturn \"OTHER\" instead of the operating system.\n"
"  -p <port>\tListen for connections on specified port. (Defaults to %s)\n"
"  -q\t\tSuppress normal logging.\n"
#if defined(__linux__) && defined(MASQ_SUPPORT)
"  -P <host>\thost acts as a proxy, forwarding connections to us.\n"
#endif
"  -r\t\tRandomize identd replies.\n"
"    \t\t\tNote: The -n and -r options are incompatible.\n"
"  -s\t\tAllow identd spoofing.\n"
"  -S\t\tSame as -s but allow all users but those listed in\n"
"    \t\t " USERFILE " to spoof replies.\n"
"  -t <seconds>\tWait for <seconds> before closing connection. (Defaults to %d)\n"
"  -T <seconds>\toidentd will remain accepting connections when run\n"
"    \t\twith -w for <seconds>.\n"
"  -u <uid>\tRun with specified uid.\n"
"  -v/-V\t\tDisplay version information and exit.\n"
"  -w\t\tWait mode.\n"
"  -x <string>\tIf a query fails, pretend it succeeded, returning <string>\n"
"  -W\t\toidentd is wrapped. (tcp wrappers)\n"
"  -h\t\tThis help message.\n", whoami, DEFAULT_PORT, DEFAULT_TIMEOUT);
}

static void print_version(void) {
	printf("%s version %s\nby Ryan McCabe <%s>\n%s\n",
		PACKAGE, VERSION, EMAIL, URL);
}
