/* 
 *  gstalker stock charter
 * 
 *  Copyright (c) 1998 Stefan S. Stratigakos
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 */

#include "gstalker.h"
#include <dirent.h>



/***********************************************************************************
converts old chart data to ascii version
**********************************************************************************/
void convert_old_chart (gchar *symbol)
{
  gint tint;
  FILE *infile, *outfile;
  GString *tstring;
  extern struct record config;
  extern struct old_header oheader;
  extern struct old_data odata;
  

  tstring = g_string_new (NULL);
  
  g_string_sprintf (tstring, "%s%s", config.datapath, symbol);
  infile = fopen(tstring->str, "rb");
  fread(&oheader, sizeof (oheader), 1, infile);
  
  /* save index data */
  save_index_file_type (symbol, "chart");
  save_index_symbol (oheader.symbol);
  save_index_name (oheader.symbol, oheader.name);
  g_string_sprintf (tstring, "%ld", oheader.first_date);
  save_index_first_date (oheader.symbol, tstring->str);
  g_string_sprintf (tstring, "%ld", oheader.last_date);
  save_index_last_date (oheader.symbol, tstring->str);
  save_index_records (oheader.symbol, oheader.records);
  save_index_line_pixelspace (oheader.symbol, oheader.pixelspace);
  save_index_ohlc_pixelspace (oheader.symbol, 6);
  save_index_threshold (oheader.symbol, oheader.threshold);
  
  /* convert the chart data */
  g_string_sprintf (tstring, "%s%s", config.datapath, oheader.symbol);
  outfile = fopen (tstring->str, "w");
  for (tint = 0; tint < oheader.records; tint++)
  {
	fread(&odata, sizeof (odata), 1, infile);
	g_string_sprintf (tstring, "%ld %g %g %g %g %ld %ld\n", odata.date, odata.open,
				  odata.high, odata.low, odata.close, odata.volume, odata.openint);
	fputs (tstring->str, outfile);
  }
  fclose (infile);
  fclose (outfile);
}
/***********************************************************************************
gets the gstalker file type field from the index file and puts the result into gstring
***********************************************************************************/
void
get_index_file_type (gchar *symbol)
{
  gchar *tstringp;
  GString *tstring;
  extern struct record config;
  extern GString *gstring;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, "=/Data/FILE");
  tstringp = gnome_config_get_string (tstring->str);
  g_string_assign (gstring, tstringp);
  g_free (tstringp);
  g_string_free (tstring, TRUE);
}
/***********************************************************************************
gets the gstalker symbol field from the index file and puts the result into gstring
***********************************************************************************/
void
get_index_symbol (gchar *symbol)
{
  gchar *tstringp;
  GString *tstring;
  extern struct record config;
  extern GString *gstring;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, "=/Data/SYMBOL");
  tstringp = gnome_config_get_string (tstring->str);
  g_string_assign (gstring, tstringp);
  g_free (tstringp);
  g_string_free(tstring, TRUE);
}
/***********************************************************************************
gets the name field from the index file and puts the result into gstring
***********************************************************************************/
void
get_index_name (gchar *symbol)
{
  gchar *tstringp;
  GString *tstring;
  extern struct record config;
  extern GString *gstring;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, "=/Data/NAME");
  tstringp = gnome_config_get_string (tstring->str);
  g_string_assign (gstring, tstringp);
  g_free (tstringp);
  g_string_free(tstring, TRUE);
}
/***********************************************************************************
gets the gstalker first_date field from the index file and puts the result into gstring
***********************************************************************************/
void
get_index_first_date (gchar *symbol)
{
  gchar *tstringp;
  GString *tstring;
  extern struct record config;
  extern GString *gstring;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, "=/Data/FIRST_DATE");
  tstringp = gnome_config_get_string (tstring->str);
  g_string_assign (gstring, tstringp);
  g_free (tstringp);
  g_string_free(tstring, TRUE);
}
/***********************************************************************************
gets the gstalker last_date field from the index file and puts the result into gstring
***********************************************************************************/
void
get_index_last_date (gchar *symbol)
{
  gchar *tstringp;
  GString *tstring;
  extern struct record config;
  extern GString *gstring;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, "=/Data/LAST_DATE");
  tstringp = gnome_config_get_string (tstring->str);
  g_string_assign (gstring, tstringp);
  g_free (tstringp);
  g_string_free(tstring, TRUE);
}
/***********************************************************************************
gets the records field from the index file and returns the result as an int
***********************************************************************************/
gint
get_index_records (gchar *symbol)
{
  gint tint;
  GString *tstring;
  extern struct record config;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, "=/Data/RECORDS");
  tint = gnome_config_get_int (tstring->str);
  g_string_free(tstring, TRUE);
  return tint;
}
/***********************************************************************************
gets the line_pixelspace field from the index file and returns the result as an int
***********************************************************************************/
gint
get_index_line_pixelspace (gchar *symbol)
{
  gint tint;
  GString *tstring;
  extern struct record config;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, 					   "=/Data/LINE_PIXELSPACE");
  tint = gnome_config_get_int (tstring->str);
  g_string_free(tstring, TRUE);
  return tint;
}
/***********************************************************************************
gets the ohlc_pixelspace field from the index file and returns the result as an int
***********************************************************************************/
gint
get_index_ohlc_pixelspace (gchar *symbol)
{
  gint tint;
  GString *tstring;
  extern struct record config;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, 					   "=/Data/OHLC_PIXELSPACE");
  tint = gnome_config_get_int (tstring->str);
  g_string_free(tstring, TRUE);
  return tint;
}
/***********************************************************************************
gets the threshold field from the index file and returns the result as a float
***********************************************************************************/
gfloat
get_index_threshold (gchar *symbol)
{
  gfloat tfloat;
  GString *tstring;
  extern struct record config;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, "=/Data/THRESHOLD");
  tfloat = gnome_config_get_float (tstring->str);
  g_string_free(tstring, TRUE);
  return tfloat;
}
/***********************************************************************************
saves the gstalker file type field to the index file
***********************************************************************************/
void
save_index_file_type (gchar *symbol, gchar *type)
{
  GString *tstring;
  extern struct record config;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, "=/Data/FILE");
  gnome_config_set_string (tstring->str, type);
  g_string_free (tstring, TRUE);
  gnome_config_sync ();
}
/***********************************************************************************
saves the symbol field to the index file
***********************************************************************************/
void
save_index_symbol (gchar *symbol)
{
  GString *tstring;
  extern struct record config;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, "=/Data/SYMBOL");
  gnome_config_set_string (tstring->str, symbol);
  g_string_free(tstring, TRUE);
  gnome_config_sync ();
}
/***********************************************************************************
saves the name field to the index file
***********************************************************************************/
void
save_index_name (gchar *symbol, gchar *name)
{
  GString *tstring;
  extern struct record config;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, "=/Data/NAME");
  gnome_config_set_string (tstring->str, name);
  g_string_free(tstring, TRUE);
  gnome_config_sync ();
}
/***********************************************************************************
saves the first_date field to the index ***********************************************************************************/
void
save_index_first_date (gchar *symbol, gchar *first_date)
{
  GString *tstring;
  extern struct record config;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, "=/Data/FIRST_DATE");
  gnome_config_set_string (tstring->str, first_date);
  g_string_free(tstring, TRUE);
  gnome_config_sync ();
}
/***********************************************************************************
save the last_date field to the index file
***********************************************************************************/
void
save_index_last_date (gchar *symbol, gchar *last_date)
{
  GString *tstring;
  extern struct record config;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, "=/Data/LAST_DATE");
  gnome_config_set_string (tstring->str, last_date);
  g_string_free(tstring, TRUE);
  gnome_config_sync ();
}
/***********************************************************************************
save the records field to the index file
***********************************************************************************/
void
save_index_records (gchar *symbol, gint records)
{
  GString *tstring;
  extern struct record config;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, "=/Data/RECORDS");
  gnome_config_set_int (tstring->str, records);
  g_string_free(tstring, TRUE);
  gnome_config_sync ();
}
/***********************************************************************************
save the line_pixelspace field to the index file
***********************************************************************************/
void
save_index_line_pixelspace (gchar *symbol, gint line_pixelspace)
{
  GString *tstring;
  extern struct record config;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, 					   "=/Data/LINE_PIXELSPACE");
  gnome_config_set_int (tstring->str, line_pixelspace);
  g_string_free(tstring, TRUE);
  gnome_config_sync ();
}
/***********************************************************************************
save the ohlc_pixelspace to the index file
***********************************************************************************/
void
save_index_ohlc_pixelspace (gchar *symbol, gint ohlc_pixelspace)
{
  GString *tstring;
  extern struct record config;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, 					   "=/Data/OHLC_PIXELSPACE");
  gnome_config_set_int (tstring->str, ohlc_pixelspace);
  g_string_free(tstring, TRUE);
  gnome_config_sync ();
}
/***********************************************************************************
save the threshold field to the index file
***********************************************************************************/
void
save_index_threshold (gchar *symbol, gfloat threshold)
{
  GString *tstring;
  extern struct record config;
  
  
  tstring = g_string_new(NULL);
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, "=/Data/THRESHOLD");
  gnome_config_set_float (tstring->str, threshold);
  g_string_free(tstring, TRUE);
  gnome_config_sync ();
}
/****************************************************************************************
Create a new chart
****************************************************************************************/
gint 
create_new_chart (gchar *symbol)
{
  gint tint=0;
  GString *tstring, *tstring2;
  extern struct record config;


  tstring = g_string_new(NULL);
  tstring2 = g_string_new(NULL);
  
  g_string_sprintf (tstring, "%s%s%s%s", "=", config.indexpath, symbol, "=/Data/");
  
  /* save FILE data */
  g_string_sprintf (tstring2, "%s%s", tstring->str, "FILE");
  gnome_config_set_string (tstring2->str, "chart");

  /* save SYMBOL data */
  g_string_sprintf (tstring2, "%s%s", tstring->str, "SYMBOL");
  gnome_config_set_string (tstring2->str, symbol);
  
  /* save NAME data */
  g_string_sprintf (tstring2, "%s%s", tstring->str, "NAME");
  gnome_config_set_string (tstring2->str, symbol);

  /* save FIRST_DATE data */
  g_string_sprintf (tstring2, "%s%s", tstring->str, "FIRST_DATE");
  gnome_config_set_string (tstring2->str, "99999999");

  /* save LAST_DATE data */
  g_string_sprintf (tstring2, "%s%s", tstring->str, "LAST_DATE");
  gnome_config_set_string (tstring2->str, "0");

  /* save RECORDS data */
  g_string_sprintf (tstring2, "%s%s", tstring->str, "RECORDS");
  gnome_config_set_int (tstring2->str, 0);

  /* save LINE_PIXELSPACE data */
  g_string_sprintf (tstring2, "%s%s", tstring->str, "LINE_PIXELSPACE");
  gnome_config_set_int (tstring2->str, 1);

  /* save OHLC_PIXELSPACE data */
  g_string_sprintf (tstring2, "%s%s", tstring->str, "OHLC_PIXELSPACE");
  gnome_config_set_int (tstring2->str, 6);

  /* save THRESHOLD data */
  g_string_sprintf (tstring2, "%s%s", tstring->str, "THRESHOLD");
  gnome_config_set_float (tstring2->str, 0);

  g_string_free(tstring, TRUE);
  g_string_free(tstring2, TRUE);
  gnome_config_sync ();

  return tint;
}
/*******************************************************************************************
Insert a new record into a chart
******************************************************************************************/
void 
insert_chart_record (gchar *symbol, gchar *date, gchar *data)
{
  gint flag=0, records;
  GString *tstring, *temppath, *first_date, *last_date;
  gchar buffer[1024], tdate[10];
  FILE *infile=0, *tempfile=0;
  extern struct record config;
  extern GString *gstring;
  
  
  tstring = g_string_new(NULL);
  temppath = g_string_new(NULL);
  first_date = g_string_new(NULL);
  last_date = g_string_new(NULL);
  
  /* get some info for the chart */
  records = get_index_records(symbol);
  get_index_first_date(symbol);
  g_string_assign (first_date, gstring->str);
  get_index_last_date(symbol);
  g_string_assign (last_date, gstring->str);
  
  
  if (records == 0)
  {
  	flag = 1;
  	g_string_assign (first_date, date);
  	g_string_assign (last_date, date);
  }
  else
  {
  	/* check if data is the newest record */
  	if (atol (date) < atol(first_date->str))
  	{
  		/* yes, so update the string */
  		g_string_assign (first_date, date);
  	
  		/* we have to insert as first record */
  		flag = 2;
  	}
  	else
  	{
  		/* check if data is the oldest record */
  		if (atol (date) > atol(last_date->str))
  		{
  			/* yes, so update the string */
  			g_string_assign (last_date, date);
  	
  			/* great, we only have to append the data */
  			flag = 1;
  		}
  	}
  }
  
  /* check if the record already exists */
  if (! flag)
  {
  	/* loop through the file and check for record with same date */
  	g_string_sprintf (tstring, "%s%s", config.datapath, symbol);
  	infile = fopen(tstring->str, "r");
  	while (fgets (buffer, sizeof(buffer), infile) != NULL)
  	{
		/* get the date */
  		sscanf (buffer, "%s", tdate);
  			
  		/* check for equality */
  		if (atol(date) == atol(tdate))
  		{
  			flag = 3;
  			break;
  		}
  	}
  }
  
  switch (flag)
  {
  	case 0: /* here we have to insert a record into the chart file */
  	
  		/* reset the flag for later use */
  		flag = 0;
  		
  		/* increment the record count */
		records++;
  		
  		/* open a temp file */
  		g_string_sprintf (temppath, "%s%s", config.datapath, "temp.txt");
		tempfile = fopen(temppath->str, "w");
		
		/* update the index file */
		save_index_records(symbol, records);
		save_index_first_date(symbol, first_date->str);
		save_index_last_date(symbol, last_date->str);
		
		/* open up the chart file */
		if (infile)
			rewind(infile);
		else
		{
			g_string_sprintf (tstring, "%s%s", config.datapath, symbol);
			infile = fopen(tstring->str, "r");
		}
		
		/* loop through the chart file and write out the records to the temp file */
  		while (fgets (buffer, sizeof(buffer), infile) != NULL)
  		{
  			sscanf (buffer, "%s", tdate);
  			if (atol(tdate) < atol(date))
  				fputs(buffer, tempfile);
  			else
  			{
  				if (! flag)
  				{
  					/* save the new record */
  					fputs (data, tempfile);
  					fputs ("\n", tempfile);
  						
  					/* save the current record as well */
  					fputs (buffer, tempfile);
  					flag++;
  				}
  				else
  				{
  					/* save the current record */
  					fputs (buffer, tempfile);
  				}
  			}
  		}
  		
  		/* we are done inserting */
  		fclose(infile);
  		fclose(tempfile);
  		
  		/* delete the old chart file and rename the tempfile to the new chart file */
  		g_string_sprintf (tstring, "%s%s", config.datapath, symbol);
  		unlink(tstring->str);
  		rename(temppath->str, tstring->str);
  		break;
  		
  	case 1: /* here we just append the record to the file */
		/* increment the record count */
		records++;
  		
		/* update the index file */
		save_index_records(symbol, records);
		save_index_first_date(symbol, first_date->str);
		save_index_last_date(symbol, last_date->str);
				
		/* open up the data file for appending */
		g_string_sprintf (tstring, "%s%s", config.datapath, symbol);
		tempfile = fopen(tstring->str, "a");
		
		/* save the new record */
		fputs (data, tempfile);
		fputs ("\n", tempfile);
		
  		/* we are done appending */
  		fclose(tempfile);
  		break;
  		
	case 2: /* here we insert new record as the first one */
		/* increment the record count */
		records++;
  		
  		/* open a temp file */
  		g_string_sprintf (temppath, "%s%s", config.datapath, "temp.txt");
		tempfile = fopen(temppath->str, "w");

		/* update the index file */
		save_index_records(symbol, records);
		save_index_first_date(symbol, first_date->str);
		save_index_last_date(symbol, last_date->str);
		
		/* save the new record */
		fputs (data, tempfile);
		fputs ("\n", tempfile);
		
		/* open up the chart file */
		if (infile)
			rewind(infile);
		else
		{
			g_string_sprintf (tstring, "%s%s", config.datapath, symbol);
			infile = fopen(tstring->str, "r");
		}
		
		/* loop through the chart file and write out the records to the temp file */
  		while (fgets (buffer, sizeof(buffer), infile) != NULL)
			fputs(buffer, tempfile);
  		
  		/* we are done inserting */
  		fclose(infile);
  		fclose(tempfile);
  		
  		/* delete the old chart file and rename the tempfile to the new chart file */
  		g_string_sprintf (tstring, "%s%s", config.datapath, symbol);
  		unlink(tstring->str);
  		rename(temppath->str, tstring->str);
  		break;
  	
  	case 3: /* here we have to overwrite an existing record */
  		/* reset the flag for later use */
  		flag = 0;
  		
  		/* open a temp file */
  		g_string_sprintf (temppath, "%s%s", config.datapath, "temp.txt");
		tempfile = fopen(temppath->str, "w");
		
		/* open up the chart file */
		if (infile)
			rewind(infile);
		else
		{
			g_string_sprintf (tstring, "%s%s", config.datapath, symbol);
			infile = fopen(tstring->str, "r");
		}
		
		/* loop through the chart file and write out the records to the temp file */
  		while (fgets (buffer, sizeof(buffer), infile) != NULL)
  		{
  			sscanf (buffer, "%s", tdate);
  			if (atol(tdate) != atol(date))
  				fputs(buffer, tempfile);
  			else
  			{
  				if (! flag)
  				{
  					/* save the new record */
  					fputs (data, tempfile);
  					fputs ("\n", tempfile);
  					
  					/* update the flag so we don't write it again */
  					flag++;
  				}
  				else
  				{
  					/* save the current record */
  					fputs (buffer, tempfile);
  				}
  			}
  		}
  		
  		/* we are done inserting */
  		fclose(infile);
  		fclose(tempfile);
  		
  		/* delete the old chart file and rename the tempfile to the new chart file */
  		g_string_sprintf (tstring, "%s%s", config.datapath, symbol);
  		unlink(tstring->str);
  		rename(temppath->str, tstring->str);
  		break;
  		
  	default: break;
  }
  g_string_free(tstring, TRUE);
  g_string_free(temppath, TRUE);
  g_string_free(first_date, TRUE);
  g_string_free(last_date, TRUE);
}
/******************************************************************************************
Loads chart data into memory and displays it.
*******************************************************************************************/
void 
load_file (gchar *symbol)
{
  FILE *infile;
  gint records=0;
  GString *tstring;
  gchar buffer[1024], date_string[25];
  guint tuint=0, oldweek, newweek;
  gfloat tfloat=0, open=0, high=0, low=0, close=0, wopen=0, whigh=0, wlow=9999999, wclose=0;
  gfloat mopen=0, mhigh=0, mlow=9999999, mclose=0;
  glong date, volume, openint, wvolume=0, wopenint=0;
  glong mvolume=0, mopenint=0;
  GtkWidget *dialog;
  GDateMonth oldmonth, newmonth;
  extern GDate *gdate;
  extern gchar *mess_cant_open_chart;
  extern gfloat maxhigh, maxlow, range, current_threshold;
  extern gulong volume_high;
  extern gint chartflag, current_records, current_line_pixelspace, current_ohlc_pixelspace;
  extern gint array_size;
  extern struct record config;
  extern GtkWidget *main_window;
  extern GArray *open_array, *high_array, *low_array, *close_array, *volume_array;
  extern GArray *openint_array, *date_array;
  extern GString *current_symbol, *current_name, *current_file_type, *current_first_date;
  extern GString *current_last_date, *gstring;
  


  tstring = g_string_new(NULL);
  maxhigh = 0;
  maxlow = 9999999;
  volume_high = 0;
  
  /* check if file exists */
  g_string_sprintf (tstring, "%s%s", config.datapath, symbol);
  infile = fopen (tstring->str, "r");
  if (! infile)
  	return;
  else
  	fclose (infile);
  
  /* check for older chart version */
  g_string_sprintf (tstring, "%s%s", config.indexpath, symbol);
  infile = fopen (tstring->str, "r");
  if (! infile)
  {
	/* it's an older file so convert it to new format */
	convert_old_chart (symbol);  
  }
  else
  	fclose (infile);

  /* free mem for chart data */
  if (date_array)
  {
    	g_array_free (date_array, TRUE);
    	date_array = g_array_new(FALSE, FALSE, sizeof(glong));
  }
  else
  	date_array = g_array_new(FALSE, FALSE, sizeof(glong));
  	
  if (open_array)
  {
	g_array_free (open_array, TRUE);
	open_array = g_array_new(FALSE, FALSE, sizeof(gfloat));
  }
  else
  	open_array = g_array_new(FALSE, FALSE, sizeof(gfloat));
  	
  if (high_array)
  {
    	g_array_free (high_array, TRUE);
    	high_array = g_array_new(FALSE, FALSE, sizeof(gfloat));
  }
  else
  	high_array = g_array_new(FALSE, FALSE, sizeof(gfloat));
  	
  if (low_array)
  {
    	g_array_free (low_array, TRUE);
    	low_array = g_array_new(FALSE, FALSE, sizeof(gfloat));
  }
  else
  	low_array = g_array_new(FALSE, FALSE, sizeof(gfloat));
  	
  if (close_array)
  {
    	g_array_free (close_array, TRUE);
    	close_array = g_array_new(FALSE, FALSE, sizeof(gfloat));
  }
  else
  	close_array = g_array_new(FALSE, FALSE, sizeof(gfloat));
  	
  if (volume_array)
  {
    	g_array_free (volume_array, TRUE);
    	volume_array = g_array_new(FALSE, FALSE, sizeof(glong));
  }
  else
  	volume_array = g_array_new(FALSE, FALSE, sizeof(glong));
  	
  if (openint_array)
  {
    	g_array_free (openint_array, TRUE);
    	openint_array = g_array_new(FALSE, FALSE, sizeof(glong));
  }
  else
  	openint_array = g_array_new(FALSE, FALSE, sizeof(glong));

  /* load the chart header data */
  get_index_symbol (symbol);
  g_string_assign (current_symbol, gstring->str);
  get_index_name (symbol);
  g_string_assign (current_name, gstring->str);
  get_index_file_type (symbol);
  g_string_assign (current_file_type, gstring->str);
  get_index_first_date (symbol);
  g_string_assign (current_first_date, gstring->str);
  get_index_last_date (symbol);
  g_string_assign (current_last_date, gstring->str);
  current_records = get_index_records (symbol);
  current_line_pixelspace = get_index_line_pixelspace (symbol);
  current_ohlc_pixelspace = get_index_ohlc_pixelspace (symbol);
  current_threshold = get_index_threshold (symbol);
  
  /* open the chart file */
  g_string_sprintf (tstring, "%s%s", config.datapath, symbol);
  infile = fopen(tstring->str, "r");
  if (! infile)
  {
  	dialog = gnome_app_error (GNOME_APP(main_window), mess_cant_open_chart);
	gtk_widget_show(dialog);
	return;
  }
    
  if (strstr(config.chart_type, "Weekly"))
  {
  	array_size = 0;
  	fgets (buffer, sizeof(buffer), infile);
  	sscanf (buffer, "%s", date_string);
  	
	create_gdate_from_string (date_string);
	  
  	oldweek = g_date_monday_week_of_year (gdate);
  	rewind(infile);
  	
  	while (fgets (buffer, sizeof(buffer), infile) != NULL)
      	{
		/* tokenize the data */
	  	sscanf (buffer, "%s %g %g %g %g %ld %ld", date_string, &open, &high, &low, &close,
	  	  									 &volume, &openint);
	  	  									 
  		create_gdate_from_string (date_string);
  		newweek = g_date_monday_week_of_year (gdate);
  		
  		if (newweek != oldweek)
  		{
  			oldweek = newweek;
  		   chart_array_append(atol(tstring->str),wopen,whigh,wlow,wclose,wvolume,wopenint);
  		    g_string_assign (tstring, date_string);
  			wopen = open;
  			whigh = high;
  			wlow = low;
  			wclose = close;
  			wvolume = volume;
  			wopenint = openint;
  			array_size++;
  		}
  		else
  		{
  			g_string_assign (tstring, date_string);
  			wclose = close;
  			wvolume = wvolume + volume;
  			wopenint = openint;
  			
  			/* find the highest close value */
	  		if (high != 0)
	    		{
				if (high > whigh)
					whigh = high;
	    		}
	  		else
	    		{
				if (close > whigh)
					whigh = close;
	    		}
	    
	  		/* find the lowest close value */
	  		if (low != 0)
	   		{
				if (low < wlow)
					wlow = low;
	    		}
	  		else
	    		{
				if (close < wlow)
					wlow = close;
	    		}
	    
	  		/* find the highest volume value */
	  		if (volume > volume_high)
	    			volume_high = volume;
  		}
  	}
  	chart_array_append(atol(tstring->str),wopen,whigh,wlow,wclose,wvolume,wopenint);
  	array_size++;
  }
  
  if (strstr (config.chart_type, "Daily"))
  {
  	/* check for maximum data points to load */
  	if (config.bars == 0)
    	{
      		/* loop through all the records */
      		while (fgets (buffer, sizeof(buffer), infile) != NULL)
      		{
	  		/* tokenize the data */
	  		sscanf (buffer, "%ld %g %g %g %g %ld %ld", &date, &open, &high, &low, &close,
	  	  									 &volume, &openint);
	  	  									 
	  		chart_array_append (date, open, high, low, close, volume, openint);
		}
    	}
  	else
    	{
      		/* get the smallest number of data to load, either total records or maximum */
      		if (current_records < config.bars)
			records = current_records;
      		else
      		{
			records = config.bars;
			
			/* read through the file until you get to the right one */
      			for (tuint = 0; tuint < (current_records - records); tuint++)
				fgets (buffer, sizeof(buffer), infile);
		}
	
      		/* loop through and load the remaining records */
      		while (fgets (buffer, sizeof(buffer), infile) != NULL)
      		{
      	  		sscanf (buffer, "%ld %g %g %g %g %ld %ld", &date, &open, &high, &low, &close,
	  	  									 &volume, &openint);
	  		chart_array_append (date, open, high, low, close, volume, openint);
		}
    	}
  }
  
  if (strstr(config.chart_type, "Monthly"))
  {
  	if (current_line_pixelspace < 3)
  		current_line_pixelspace = 3;
  		
  	array_size = 0;
  	fgets (buffer, sizeof(buffer), infile);
  	sscanf (buffer, "%s", date_string);
  	
	create_gdate_from_string (date_string);
	  
  	oldmonth = g_date_month (gdate);
  	rewind(infile);
  	
  	while (fgets (buffer, sizeof(buffer), infile) != NULL)
      	{
		/* tokenize the data */
	  	sscanf (buffer, "%s %g %g %g %g %ld %ld", date_string, &open, &high, &low, &close,
	  	  									 &volume, &openint);
	  	  									 
  		create_gdate_from_string (date_string);
  		newmonth = g_date_month (gdate);
  		
  		if (newmonth != oldmonth)
  		{
  			oldmonth = newmonth;
  		   chart_array_append(atol(tstring->str),mopen,mhigh,mlow,mclose,mvolume,mopenint);
  			g_string_assign (tstring, date_string);
  			mopen = open;
  			mhigh = high;
  			mlow = low;
  			mclose = close;
  			mvolume = volume;
  			mopenint = openint;
  			array_size++;
  		}
  		else
  		{
  			g_string_assign (tstring, date_string);
  			mclose = close;
  			mvolume = wvolume + volume;
  			mopenint = openint;
  			
  			/* find the highest close value */
	  		if (high != 0)
	    		{
				if (high > mhigh)
					mhigh = high;
	    		}
	  		else
	    		{
				if (close > mhigh)
					mhigh = close;
	    		}
	    
	  		/* find the lowest close value */
	  		if (low != 0)
	   		{
				if (low < mlow)
					mlow = low;
	    		}
	  		else
	    		{
				if (close < mlow)
					mlow = close;
	    		}
	    
	  		/* find the highest volume value */
	  		if (volume > volume_high)
	    			volume_high = volume;
  		}
  	}
  	chart_array_append (atol(tstring->str), mopen, mhigh, mlow, mclose, mvolume, mopenint);
  	array_size++;
  }
  
  fclose(infile);
  
  /* calculate the range between the highest and lowest close for chart drawing */
  tfloat = (maxhigh - maxlow) / 100;
  maxhigh = maxhigh + tfloat;
  maxlow = maxlow - tfloat;
  range = maxhigh - maxlow;
  
  update_name_display ();

  /* update the flag so we know there is an active chart */
  chartflag = 1;
  
  /* update the data bar */
  update_stats_bar ();
  
  /* create the ma arrays */
  if (config.moving_status)
  	create_ma_array(config.moving_type);
  if (config.moving2_status)
  	create_ma2_array(config.moving2_type);
  if (config.moving3_status)
  	create_ma3_array(config.moving3_type);
  
  /* i have to do this in order update the display properly */
  resize_chart ();
  g_string_free(tstring, TRUE);
}
/***********************************************************************************
scans datapath for all charts and inserts info into the specified clist box
************************************************************************************/
void 
show_chart_list (GtkWidget * chartlist)
{
  GString *tstring;
  struct dirent **dirlist;
  gint tint, tint2;
  extern struct record config;


  tstring = g_string_new(NULL);
  gtk_clist_freeze (GTK_CLIST (chartlist));
  gtk_clist_clear (GTK_CLIST (chartlist));
  
  /* get the list of files in the data directory */
  tint = scandir (config.datapath, &dirlist, NULL, alphasort);
  
  /* check if the directory is empty */
  if (tint < 2)
    return;
    
  /* skip the . and .. directory files */
  tint2 = tint;
  for (tint = 2; tint < tint2; tint++)
    {
      /* get the chart info to put into the clist box */
      display_chart_details (dirlist[tint]->d_name, chartlist);
    }
  gtk_clist_thaw (GTK_CLIST (chartlist));
  g_string_free(tstring, TRUE);
}
/********************************************************************************************
This function gets the symbol, name, first date and last date data from filename and
appends that into the clist.
********************************************************************************************/
void 
display_chart_details (gchar *filename, GtkWidget * clist)
{
  GString *tstring, *tstring2, *tstring3, *tstring4;
  gchar *line[4];
  extern GString *gstring;


  tstring = g_string_new(NULL);
  tstring2 = g_string_new(NULL);
  tstring3 = g_string_new(NULL);
  tstring4 = g_string_new(NULL);
  
  get_index_symbol (filename);
  g_string_assign (tstring, gstring->str);
  line[0] = tstring->str;
  get_index_name (filename);
  g_string_assign (tstring2, gstring->str);
  line[1] = tstring2->str;  
      	
  /* get the first date field */
  /* convert the date from YYYYMMDD to MM/DD/YYYY */
  get_index_first_date (filename);
  g_string_assign (tstring3, gstring->str);
  convert_date(tstring3->str);
  g_string_assign (tstring3, gstring->str);
  line[2] = tstring3->str;
      	
  /* get the last date field */
  /* convert the date from YYYYMMDD to MM/DD/YYYY */
  get_index_last_date (filename);
  g_string_assign (tstring4, gstring->str);
  convert_date(tstring4->str);
  g_string_assign (tstring4, gstring->str);
  line[3] = tstring4->str;
      	
  gtk_clist_append (GTK_CLIST (clist), line);
  
  g_string_free(tstring, TRUE);
  g_string_free(tstring2, TRUE);      	
  g_string_free(tstring3, TRUE);
  g_string_free(tstring4, TRUE);
}


