/*
    Copyright (C) 1997-1998  Ulric Eriksson <ulric@edu.stockholm.se>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston,
    MA 02111-1307, USA.
*/

/*
 * fonts.c
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#ifdef NOPE
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#endif

#include "../common/common.h"
#include "../common/cmalloc.h"
#include "../common/oldfonts.h"
#include "../common/fonts.h"
#include "fonts.h"

#ifdef NOPE
static Display *dpy;
#endif

siag_color color_table[1000];
static siag_fontname fontname_table[256];
siag_font font_table[256];
static int ncolor, nfontname, nfont;

static struct {
	char *alias;
	char *name;
} fontalias[256];

static int nfontalias = 0;

static int lastfont;

static int compar(const void *p, const void *q)
{
	return strcmp(*(const char **)p, *(const char **)q);
}

/* ---
returns a null-terminated, sorted list of all known fonts.
   Caller must free
*/

char **font_list(int *n)
{
	int i, j;
	char **list = (char **)cmalloc((nfontalias+nfontname+1)*sizeof(char *));
	for (i = 0; i < nfontalias; i++)
		list[i] = cstrdup(fontalias[i].alias);
	for (j = 0; j < nfontname; j++)
		list[i+j] = cstrdup(fontname_table[j].name);
	list[i+j] = NULL;
	qsort(list, i+j, sizeof(char *), compar);
	*n = i+j;
	return list;
}

/* ---
returns a null-terminated, sorted list of all known colours.
   Caller must free
*/

char **color_list(int *n)
{
	int i;
	char **list = (char **)cmalloc((ncolor+1)*sizeof(char *));
	for (i = 0; i < ncolor; i++)
		list[i] = cstrdup(color_table[i].name);
	list[i] = NULL;
	qsort(list, i, sizeof(char *), compar);
	*n = i;
	return list;
}

#ifdef NOPE
/* ---
The index is into the font_table. The index must have come from
   lookup_font, so we know that the font is already loaded
*/

Font get_font(int index)
{
	return font_table[index].fs->fid;
}
#endif

#ifdef NOPE
/* ---
index to font_table
*/

XFontStruct *font_struct(int index)
{
	return font_table[index].fs;
}
#endif

#ifdef NOPE
/* ---
index to font_table
*/

int font_ascent(int index)
{
  return font_struct(index)->max_bounds.ascent;
}
#endif

#ifdef NOPE
/* ---
index to font table
*/

int font_descent(int index)
{
  return font_struct(index)->max_bounds.descent;
}
#endif

#ifdef NOPE
/* ---
index to font table
*/

int font_height(int index)
{
  return font_ascent(index)+font_descent(index);
}
#endif

#ifdef NOPE
/* ---
index to color_table. The index must come from lookup_color,
so we know that the colour has already been allocated
*/

unsigned long get_color(int index)
{
	return color_table[index].color.pixel;
}
#endif

/* ---
Take an index into the format table (i.e. a new-style format code) and
decode it into the components specified by the mask.
*/

void decode_format(int n, long mask, sfmt *fmt)
{
	int font = format_table[n].font;

	if (mask & SFMT_FAMILY) {
		fmt->family = fontname_table[font_table[font].name].name;
		if (!fmt->family) fmt->family = "Helvetica";
	}
	if (mask & SFMT_SIZE)
		fmt->size = font_table[font].size;
	if (mask & SFMT_BOLD)
		fmt->bold = font_table[font].bold;
	if (mask & SFMT_ITALIC)
		fmt->italic = font_table[font].italic;
	if (mask & SFMT_ULINE)
		fmt->uline = format_table[n].uline;
	if (mask & SFMT_FG) {
		fmt->fg = color_table[format_table[n].fg].name;
		if (!fmt->fg) fmt->fg = "black";
	}
	if (mask & SFMT_BG) {
		fmt->bg = color_table[format_table[n].bg].name;
		if (!fmt->bg) fmt->bg = "white";
	}
	if (mask & SFMT_BORDERS)
		fmt->borders = format_table[n].siagfmt & BORDER_MASK;
	if (mask & SFMT_VADJ)
		fmt->vadj = format_table[n].siagfmt & VADJ_MASK;
	if (mask & SFMT_HADJ)
		fmt->hadj = format_table[n].siagfmt & HADJ_MASK;
	if (mask & SFMT_STYLE)
		fmt->style = format_table[n].style;
}

/* ---
Encode the components from fmt specified by the mask into a new-style
format code, allocating colours and fonts as necessary.
*/

int encode_format(long mask, sfmt *fmt)
{
	char *family = "Helvetica";
	int fg = 0, bg = 0;
	int size = 120;
	int bold = 0, italic = 0, uline = 0;
	int siagfmt = 0;
	int style = 0;

	if (mask & SFMT_FAMILY) {
		family = lookup_fontalias(fmt->family);
		if (!family) family = "Helvetica";
	}
	if (mask & SFMT_SIZE)
		size = fmt->size;
	if (mask & SFMT_BOLD)
		bold = fmt->bold;
	if (mask & SFMT_ITALIC)
		italic = fmt->italic;
	if (mask & SFMT_ULINE)
		uline = fmt->uline;
	if (mask & SFMT_FG) {
		fg = lookup_color(fmt->fg);
		if (fg == -1) fg = 0;	/* black */
	}
	if (mask & SFMT_BG) {
		bg = lookup_color(fmt->bg);
		if (bg == -1) bg = 7;	/* white */
	}
	if (mask & SFMT_BORDERS)
		siagfmt |= fmt->borders;
	if (mask & SFMT_VADJ)
		siagfmt |= fmt->vadj;
	if (mask & SFMT_HADJ)
		siagfmt |= fmt->hadj;
	if (mask & SFMT_STYLE)
		style = fmt->style;
	return lookup_format(lookup_font(family, size, bold, italic),
				uline, fg, bg, style, siagfmt);
}

/* ---
Here is the whole song and dance to get a format id for Siag versions >= 3.1.

Start by deciding on font family, point size, weight and slant.
Get an index into the font table by calling lookup_font with
the family name as argument along with the size, weight and slant.
The font is automatically loaded, if necessary.

Next pick foreground and background colours. The colours are loaded
by calling lookup_color with the colour name as argument. An index into
the colour table is returned.

Finally we are ready to call lookup_format. Decide if the character
should be underlined. If the application is Siag, also combine the
cell attributes. Pass the font index, the colour index, the underlining
flag and the Siag attributes to lookup_format. An index into the
format table is returned, and it is this index we can use to replace
the old style formats.

Yes, this is more complex than the old stuff. But it is also in fact
easier to extract individual pieces of the formats, since they are
normal C structures. To my eyes, font_table[format_table[fmt].font].size
is much clearer than (fmt & SIZE_MASK)>>SIZE_SHIFT. And it is less
prone to errors. And it leaves room for future expansion, for example
for overstrike and double underline, or bidirectional printing.
--- */

#ifdef NOPE
/* ---
Initialise the X format code.
*/

void init_format(Display *d)
{
	dpy = d;
}
#endif

/* ---
returns the index of a structure with the requested color, or -1
   for failure. Pixel value in color_table[i].color.pixel is allocated
*/

int lookup_color(char *name)
{
	int i;
#ifdef NOPE
	Colormap cmap;
#endif

	if (!name) {
		fprintf(stderr, "Some dumbass called me with null colour!\n");
		abort();
	}
#ifdef NOPE
	if (!dpy) return 0;

	cmap = DefaultColormap(dpy, DefaultScreen(dpy));
#endif

	for (i = 0; i < ncolor; i++) {
		if (!cstrcasecmp(name, color_table[i].name))
			break;
	}
	if (i == ncolor) return -1;	/* fail */
#ifdef NOPE
	if (color_table[i].need_init) {
		color_table[i].color.red = color_table[i].red;
		color_table[i].color.green = color_table[i].green;
		color_table[i].color.blue = color_table[i].blue;
		if (!XAllocColor(dpy, cmap, &color_table[i].color))
			return -1;
		color_table[i].need_init = 0;
	}
#endif
	return i;
}

/* ---
returns the index of a font with the specified name, or -1 for failure
*/

int lookup_fontname(char *name)
{
	int i;

	for (i = 0; i < nfontname; i++) {
		if (!cstrcasecmp(name, fontname_table[i].name))
			return i;
	}
	return -1;
}

#ifdef NOPE
/* ---
for the given font with the given properties, encode X and PS strings.
   example (result in x_name will vary depending on resolution):

	font = index of font with name "Helvetica"
	size = 12
	bold = 0
	italic = 1;

	x_name = "-adobe-helvetica-medium-o-*--*-120-75-75-*-*-iso8859-1"
	ps_name = "Helvetica-Oblique"
*/

void encode_font(Display *dpy, int font, int size, int bold, int italic,
		char *x_name, char *ps_name)
{
	char *x_fmt, *ps_fmt;
	int i = 0;
	int res_x, res_y;	/* screen resolution */
#if 0
	int screen = DefaultScreen(dpy);

	res_x = DisplayWidth(dpy, screen)/(DisplayWidthMM(dpy, screen)/25.4);
	res_y = DisplayHeight(dpy, screen)/(DisplayHeightMM(dpy, screen)/25.4);
#else	/* cheat to get ps right */
	res_x = res_y = 72;
#endif

	if (bold) i += 2;
	if (italic) i += 1;

	x_fmt = fontname_table[font].x_name[i];
	if (!x_fmt) x_fmt = fontname_table[0].x_name[0];
	sprintf(x_name, x_fmt, size, res_x, res_y);
	ps_fmt = fontname_table[font].ps_name[i];
	if (!ps_fmt) ps_fmt = fontname_table[0].ps_name[0];
	strcpy(ps_name, ps_fmt);
}
#endif

/* ---
return index into font table or -1 for failure. Load font if necessary.
*/

int lookup_font(char *fname, int size, int bold, int italic)
{
	int i, name;
#ifdef NOPE
	char x_name[1000], ps_name[1000];
#endif

	name = lookup_fontname(fname);
	if (name == -1) name = 0;	/* fallback value */

	for (i = 0; i < nfont; i++) {
		if (font_table[i].name == name &&
			font_table[i].size == size &&
			font_table[i].bold == bold &&
			font_table[i].italic == italic)
				return i;	/* already loaded */
	}
	nfont++;
	font_table[i].name = name;
	font_table[i].size = size;
	font_table[i].bold = bold;
	font_table[i].italic = italic;
#ifdef NOPE
	encode_font(dpy, name, size, bold, italic, x_name, ps_name);
	font_table[i].x_name = cstrdup(x_name);
	font_table[i].ps_name = cstrdup(ps_name);
	font_table[i].fs = XLoadQueryFont(dpy, x_name);
	if (font_table[i].fs == NULL) {
		font_table[i].fs =
			XLoadQueryFont(dpy, "*helvetica-medium-r*12*");
	}
	if (font_table[i].fs == NULL) {
		fprintf(stderr, "Panic: can't load any fonts!\n");
		exit(EXIT_FAILURE);
	}
#endif
	return i;
}

/* ---
X and PS fonts are registered separately because they are device
dependent on different devices. This is slightly lame because they will
probably be called from the same Scheme wrapper function, and they are
in the same C translation unit. Never mind.
*/

int register_x_font(char *name, char *x, char *xi, char *xb, char *xbi)
{
	int i;

	for (i = 0; i < nfontname; i++) {
		if (!strcasecmp(name, fontname_table[i].name)) break;
	}
	if (i == nfontname) {
		nfontname++;
		fontname_table[i].name = cstrdup(name);
		fontname_table[i].ps_name[0] = NULL;
		fontname_table[i].ps_name[1] = NULL;
		fontname_table[i].ps_name[2] = NULL;
		fontname_table[i].ps_name[3] = NULL;
		fontname_table[i].t1_name[0] = NULL;
		fontname_table[i].t1_name[1] = NULL;
		fontname_table[i].t1_name[2] = NULL;
		fontname_table[i].t1_name[3] = NULL;
	}
	fontname_table[i].x_name[0] = cstrdup(x);
	fontname_table[i].x_name[1] = cstrdup(xi);
	fontname_table[i].x_name[2] = cstrdup(xb);
	fontname_table[i].x_name[3] = cstrdup(xbi);
	return i;
}

/* ---
As above but for Postscript.
*/

int register_ps_font(char *name, char *ps, char *psi, char *psb, char *psbi)
{
	int i;

	for (i = 0; i < nfontname; i++) {
		if (!cstrcasecmp(name, fontname_table[i].name)) break;
	}
	if (i == nfontname) {
		nfontname++;
		fontname_table[i].name = cstrdup(name);
		fontname_table[i].x_name[0] = NULL;
		fontname_table[i].x_name[1] = NULL;
		fontname_table[i].x_name[2] = NULL;
		fontname_table[i].x_name[3] = NULL;
		fontname_table[i].t1_name[0] = NULL;
		fontname_table[i].t1_name[1] = NULL;
		fontname_table[i].t1_name[2] = NULL;
		fontname_table[i].t1_name[3] = NULL;
	}
	fontname_table[i].ps_name[0] = cstrdup(ps);
	fontname_table[i].ps_name[1] = cstrdup(psi);
	fontname_table[i].ps_name[2] = cstrdup(psb);
	fontname_table[i].ps_name[3] = cstrdup(psbi);
	/* and now, a piece written in Hackish */
	fontname_table[i].iso8859_1 = strcmp(ps, psi)?1:0;
	return i;
}

int register_t1_font(char *name, char *ps, char *psi, char *psb, char *psbi)
{
	int i;

	for (i = 0; i < nfontname; i++) {
		if (!cstrcasecmp(name, fontname_table[i].name)) break;
	}
	if (i == nfontname) {
		nfontname++;
		fontname_table[i].name = cstrdup(name);
		fontname_table[i].x_name[0] = NULL;
		fontname_table[i].x_name[1] = NULL;
		fontname_table[i].x_name[2] = NULL;
		fontname_table[i].x_name[3] = NULL;
		fontname_table[i].ps_name[0] = NULL;
		fontname_table[i].ps_name[1] = NULL;
		fontname_table[i].ps_name[2] = NULL;
		fontname_table[i].ps_name[3] = NULL;
	}
	fontname_table[i].t1_name[0] = cstrdup(ps);
	fontname_table[i].t1_name[1] = cstrdup(psi);
	fontname_table[i].t1_name[2] = cstrdup(psb);
	fontname_table[i].t1_name[3] = cstrdup(psbi);
	/* and now, a piece written in Hackish */
	fontname_table[i].iso8859_1 = strcmp(ps, psi)?1:0;
	return i;
}

/* ---
handle mapping between e.g. Arial and Helvetica

Store the name of a font we don't have and the name of a replacement
*/

int font_alias(char *alias, char *name)
{
	int i;

	for (i = 0; i < nfontalias; i++) {
		if (!cstrcasecmp(alias, fontalias[i].alias)) break;
	}
	if (i == nfontalias) {
		nfontalias++;
		fontalias[i].alias = cstrdup(alias);
	}
	fontalias[i].name = cstrdup(name);
	return i;
}

/* ---
If there is a font with exactly the alias name, return that. Otherwise
   if there is an alias, return the name of the real font. Otherwise NULL.
*/

char *lookup_fontalias(char *alias)
{
	int i;

	if (alias == NULL) return NULL;
	if (lookup_fontname(alias) >= 0) return alias;
	for (i = 0; i < nfontalias; i++) {
		if (!cstrcasecmp(alias, fontalias[i].alias))
			return fontalias[i].name;
	}
	return NULL;
}

/* ---
stores a colour name along with the red, green and blue values,
   or overwrites an existing colour with the same name.
   The colour is not allocated until it is looked up in lookup_color.
*/

int register_color(char *name,
	unsigned short red, unsigned short green, unsigned short blue)
{
	int i;

	for (i = 0; i < ncolor; i++)
		if (!cstrcasecmp(name, color_table[i].name)) break;
	if (i == ncolor) {
		ncolor++;
		color_table[i].name = cstrdup(name);
	}
	color_table[i].red = red;
	color_table[i].green = green;
	color_table[i].blue = blue;
	color_table[i].need_init = 1;
	return i;
}

/* ---
fmt is an index into fontname_table
*/

char *family2name(int fmt)
{
	return fontname_table[fmt].name;
}

/* ---
fmt is index into font_table
*/

char *size2name(int fmt)
{
	static char b[80];

	sprintf(b, "%d", font_table[fmt].size/10);
	return b;
}

/* ---
fmt is index into color_table
*/

char *color2name(int fmt)
{
	return color_table[fmt].name;
}

/* ---
Postscript font name
*/

char *ps_fontname(int fmt)
{
	return font_table[format_table[fmt].font].ps_name;
}

/* ---
Postscript font size
*/

int ps_fontsize(int fmt)
{
	return font_table[format_table[fmt].font].size / 10;
}

/* ---
Define Latin-1 fonts
*/

void ps_makefonts(FILE *fp)
{
	int i, j;

	for (i = 0; i < nfontname; i++) {
		if (!fontname_table[i].iso8859_1)
			continue;	/* don't bother */
		for (j = 0; j < 3; j++) {
			if (j && !strcmp(fontname_table[i].ps_name[0],
					fontname_table[i].ps_name[j]))
				continue;	/* done already */
			fprintf(fp, "/%s findfont\n",
				fontname_table[i].ps_name[j]);

			fprintf(fp, "dup length dict begin\n");
			fprintf(fp, "  {1 index /FID ne {def} {pop pop} ifelse} forall\n");
			fprintf(fp, "  /Encoding ISOLatin1Encoding def\n");
			fprintf(fp, "  currentdict\n");
			fprintf(fp, "end\n");
			fprintf(fp, "/%s-ISOLatin1 exch definefont pop\n",
				fontname_table[i].ps_name[j]);
		}
	}
}

/* ---
Set the font
*/

void ps_set_font(FILE *fp, int newfont)
{
	int name;

	if (newfont == lastfont)
		return;
	lastfont = newfont;
	if (newfont == -1) return;	/* starter mark */
	name = font_table[newfont].name;
	fprintf(fp, "/%s%s findfont\n",
		ps_fontname(newfont),
		fontname_table[name].iso8859_1?"-ISOLatin1":"");
	fprintf(fp, "%d scalefont\n", ps_fontsize(newfont));
	fprintf(fp, "setfont\n");
}

