/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Stuart Levy, Tamara Munzner, Mark Phillips */

#include "mgP.h"
#include "mgopenglP.h"
#include "polylistP.h"

#ifdef GLUT
# include <GL/glut.h>
#else
# include <GL/gl.h>
#endif

#ifndef alloca
#include <alloca.h>
#endif

void	mgopengl_polygon( int nv, HPoint3 *v, int nn, Point3 *n,
	      	          int nc,ColorA *c );
void	mgopengl_mesh( int wrap,int nu,int nv,HPoint3 *p, Point3 *n,ColorA *c );
void	mgopengl_line( HPoint3 *p1, HPoint3 *p2 );
void	mgopengl_polyline( int nv, HPoint3 *verts, int nc, ColorA *colors, int wrap );
void	mgopengl_polylist(  int np, Poly *p, int nv, Vertex *v, 
			     int plflags );
void	mgopengl_drawnormal(HPoint3 *p, Point3 *n);

void	mgopengl_closer();
void	mgopengl_farther();

#ifndef NO_ZNUDGE
# define mgopengl_v4ffunc(v)  glVertex4fv(&(v)->x);
#else
# define mgopengl_v4ffunc(v)  mgopengl_v4fcloser(v)
#endif


/*-----------------------------------------------------------------------
 * Function:	mgopengl_polygon
 * Description:	draw a polygon
 * Author:	mbp, munzner
 * Date:	Mon Jul 29 16:53:56 1991
 * Notes:	See mg.doc.
 *
 *              do later: Different shading cases separated into different
 *		loops for speed.
 */
void
mgopengl_polygon(int nv,  HPoint3 *V, 
	     int nn,  Point3 *N, 
	     int nc,  ColorA *C)
{
  register int i;	
  register HPoint3 *v;
  register Point3 *n;
  register ColorA *c;
  int cinc, ninc;
  int flag;


  flag = _mgc->astk->ap.flag;
  if ((_mgc->astk->mat.override & MTF_DIFFUSE) && !_mgc->astk->useshader)
    nc = 0;
  cinc = (nc > 1);
  ninc = (nn > 1);
  if(nc == 0)
    C = (ColorA*)&_mgc->astk->ap.mat->diffuse;


  /* reestablish correct drawing color if necessary */

  if (flag & APF_FACEDRAW) {
    MAY_LIGHT();
    glColorMaterial(GL_FRONT_AND_BACK, _mgopenglc->lmcolor);glEnable(GL_COLOR_MATERIAL);
    glBegin(GL_POLYGON);
    if (nc <= 1)
	D4F(&(_mgc->astk->ap.mat->diffuse));
    for (n = N, c = C, v = V, i = 0; i<nv; ++i, ++v) {
	if (nc) { D4F(c); c += cinc; }
	if (nn) { N3F(n,v); n += ninc; }
	glVertex4fv((float *)v);
    }
    glEnd();
  }

  if( flag & (APF_EDGEDRAW|APF_NORMALDRAW) ) {
    if(_mgopenglc->znudge) mgopengl_closer();
    glDisable(GL_COLOR_MATERIAL);
    DONT_LIGHT();
    if (flag & APF_EDGEDRAW) {
	glColor3fv((float *)&_mgc->astk->ap.mat->edgecolor);
	glBegin(GL_LINE_LOOP);
	for (v = V, i = 0; i<nv; ++i, ++v)
	  mgopengl_v4ffunc(v);
	glEnd();
    }

    if (flag & APF_NORMALDRAW) {
	glColor3fv((float *)&_mgc->astk->ap.mat->normalcolor);
	for (n = N, v = V, i = 0; i<nv; ++i, ++v, n += ninc) {
	    mgopengl_drawnormal(v, n);
	}
    }
    if(_mgopenglc->znudge) mgopengl_farther();
  }
}

void
mgopengl_quads(int count,  HPoint3 *V, Point3 *N, ColorA *C)
{
  int i;	
  register int k;
  register HPoint3 *v;
  register Point3 *n;
  register ColorA *c;
  ColorA cs[4];
  int flag;

#define QUAD(stuff)  { \
		register int k = 4;		\
		glBegin(GL_POLYGON);			\
		do { stuff; } while(--k > 0);	\
		glEnd();			\
	}
  if(count <= 0)
    return;
  flag = _mgc->astk->ap.flag;
  if ((_mgc->astk->mat.override & MTF_DIFFUSE) && !_mgc->astk->useshader)
    C = NULL;
  
  /* reestablish correct drawing color if necessary */

  if (flag & APF_FACEDRAW) {
    glColorMaterial(GL_FRONT_AND_BACK, _mgopenglc->lmcolor);glEnable(GL_COLOR_MATERIAL);

    MAY_LIGHT();

    i = count;
    v = V; c = C; n = N;
    if(c) {
	if(n) {
	    do {
		QUAD( (D4F(c++), N3F(n++,v), glVertex4fv((float*)v++)) );
	    } while(--i > 0);
	} else {
	    /* Colors, no normals */
	    do {
		QUAD( (D4F(c++), glVertex4fv((float*)v++)) );
	    } while(--i > 0);
	}
    } else {
	c = (ColorA*)&_mgc->astk->ap.mat->diffuse;
	if(n) {
	    D4F(c);
	    do {
		QUAD( (N3F(n++, v), glVertex4fv((float*)v++)) );
	    } while(--i > 0);
	} else {
	    D4F(c);
	    do {
		QUAD( (glVertex4fv((float*)v++)) );
	    } while(--i > 0);
	}
    }
  }

  if( flag & (APF_EDGEDRAW|APF_NORMALDRAW) ) {
    if(_mgopenglc->znudge) mgopengl_closer();
    glDisable(GL_COLOR_MATERIAL);
    DONT_LIGHT();

    if (flag & APF_EDGEDRAW) {
	glColor3fv((float *)&_mgc->astk->ap.mat->edgecolor);
	i = count; v = V;
	do {
	    register int k = 4;
	    glBegin(GL_LINE_LOOP);
	    do { mgopengl_v4ffunc(v++); } while(--k > 0);
	    glEnd();
	} while(--i > 0);
    }

    if (flag & APF_NORMALDRAW && N) {
	glColor3fv((float *)&_mgc->astk->ap.mat->normalcolor);
	i = count*4; v = V; n = N;
	do {
	    mgopengl_drawnormal(v++, n++);
	} while(--i > 0);
    }
    if(_mgopenglc->znudge) mgopengl_farther();
  }
}


void mgopengl_line( HPoint3 *p1, HPoint3 *p2 )
{
  DONT_LIGHT();
  glBegin(GL_LINE_STRIP);
  glVertex4fv((float *)p1);
  glVertex4fv((float *)p2);
  glEnd();
}

void mgopengl_point(register HPoint3 *v)
{
  int i;
  HPoint3 a;
  register HPoint3 *p, *q;
  float vw;

  DONT_LIGHT();

  if(_mgc->astk->ap.linewidth > 1) {
    
    if(!(_mgc->has & HAS_POINT))
      mg_makepoint();
    /* Compute w component of point after projection to screen */
    vw = v->x * _mgc->O2S[0][3] + v->y * _mgc->O2S[1][3]
      + v->z * _mgc->O2S[2][3] + v->w * _mgc->O2S[3][3];
    if(vw <= 0) return;
    
#define  PUT(p)  					\
    a.x = v->x + p->x*vw; a.y = v->y + p->y*vw;	\
      a.z = v->z + p->z*vw; a.w = v->w + p->w*vw;	\
	glVertex4fv((float *)&a)
	  
	  p = VVEC(_mgc->point, HPoint3);
    q = p + VVCOUNT(_mgc->point);
    glBegin(GL_TRIANGLE_STRIP);
    PUT(p);
    do {
	p++;
	PUT(p);
	if(p >= q) break;
	q--;
	PUT(q);
    } while(p < q);
    glEnd();
  } else {
    glBegin(GL_POINTS);
    glVertex4fv((float *)v);
    glEnd();
  }
}
  

void mgopengl_polyline( int nv, HPoint3 *v, int nc, ColorA *c, int wrapped )
{
  int remain;

  DONT_LIGHT();

  /* note we don't reset to current material color because we could be
   * in the middle of a list of lines and should inherit the color from 
   * the last color call.
   */
  
  if(!(wrapped & 2)) {
	/* First member of batch */
      if(_mgopenglc->znudge) mgopengl_closer();
      if(nc)
	  glDisable(GL_COLOR_MATERIAL);
  }
  if (nv == 1) {
    if(nc > 0) glColor4fv((float *)c);
    mgopengl_point(v);
  } 
  else if(nv > 0) {
    glBegin(GL_LINE_STRIP);
    if(wrapped & 1) {
      if(nc > 0) glColor4fv((float *)(c + nc - 1));
	mgopengl_v4ffunc(v + nv - 1);
    }

    do {
	if(--nc >= 0) glColor4fv((float *)c++);
	mgopengl_v4ffunc(v++);
    } while(--nv > 0);
    glEnd();
  }
  if(!(wrapped & 4) && _mgopenglc->znudge) mgopengl_farther();
}


/* Slave routine for mgopengl_trickypolygon() below. */
static int tessplflags;

void tessvert(Vertex *vp)
{
  if (tessplflags & PL_HASVCOL) D4F(&vp->vcol);
  if (tessplflags & PL_HASVN) N3F(&vp->vn, &vp->pt);
  if (tessplflags & PL_HASST) glTexCoord2fv((GLfloat *)&vp->st);
  glVertex4fv(&vp->pt.x);
}


/*
 * Called when we're asked to deal with a possibly-concave polygon.
 * Note we can only be called if APF_CONCAVE mode is set.
 * Returns 0 if the polygon wasn't concave, and can be rendered by the
 * normal algorithm.
 */
static int
mgopengl_trickypolygon( Poly *p, int plflags ) 
{
  int i;
  Vertex *vprev, *vp, *vnext;
  Point3 e, eprev, cross;
  int cw = 0, ccw = 0;
  float orient;
  static GLUtriangulatorObj *glutri;

  vprev = (p->v)[p->n_vertices-1];
  vp = (p->v)[0];
  HPt3SubPt3(&vp->pt, &vprev->pt, &e);
  for(i = 1; i < p->n_vertices; i++) {
    eprev = e;
    vprev = vp;
    vp = (p->v)[i];
    HPt3SubPt3(&vp->pt, &vprev->pt, &e);
    Pt3Cross(&e, &eprev, &cross);
    orient = Pt3Dot(&cross, &p->pn);
    if(orient < -1e-8) cw = 1;
    else if(orient > 1e-8) ccw = 1;
  }
  if(cw && ccw) {
    /* Has some reflex vertices -- it's concave. */
    double *dpts = (double *)alloca(3*p->n_vertices*sizeof(double));
    double *dp;

    if(glutri == NULL) {
	/* Create GLU-library triangulation handle, just once */
	glutri = gluNewTess();
#ifdef _WIN32	/* Windows idiocy.  We shouldn't need to cast standard funcs! */
	gluTessCallback(glutri, GLU_BEGIN, (GLUtessBeginProc)glBegin);
	gluTessCallback(glutri, GLU_VERTEX, (GLUtessVertexProc)tessvert);
	gluTessCallback(glutri, GLU_END, (GLUtessEndProc)glEnd);
#else		/* Any reasonable OpenGL implementation */
	gluTessCallback(glutri, GLU_BEGIN, glBegin);
	gluTessCallback(glutri, GLU_VERTEX, tessvert);
	gluTessCallback(glutri, GLU_END, glEnd);
#endif
    }

    tessplflags = plflags;
    gluBeginPolygon(glutri);
    for(i = 0, dp = dpts; i < p->n_vertices; i++, dp += 3) {
	vp = (p->v)[i];
	dp[0] = vp->pt.x / vp->pt.w;
	dp[1] = vp->pt.y / vp->pt.w;
	dp[2] = vp->pt.z / vp->pt.w;
	gluTessVertex(glutri, dp, vp);
    }
    gluEndPolygon(glutri);
    return 1;			/* It was tricky, but we handled it */
  }
  return 0;			/* It wasn't tricky, you handle it faster */
}


/*-----------------------------------------------------------------------
 * Function:	mgopengl_polylist
 * Description:	draws a Polylist: collection of Polys
 * Author:	munzner
 * Date:	Wed Oct 16 20:21:56 1991
 * Notes:	see mg.doc
 */
void mgopengl_polylist( int np, Poly *_p, int nv, Vertex *V, int plflags )
{
  register int i,j;
  register Poly *p;
  register Vertex **v, *vp;
  register Point3 *n;
  struct mgastk *ma = _mgc->astk;
  int flag,shading;
  int nonsurf = -1;
  flag = ma->ap.flag;
  shading = ma->ap.shading;


  switch(shading) {
  case APF_FLAT: plflags &= ~PL_HASVN; break;
  case APF_SMOOTH: plflags &= ~PL_HASPN; break;
  default: plflags &= ~(PL_HASVN|PL_HASPN); break;
  }

  if((_mgc->astk->mat.override & MTF_DIFFUSE) && !_mgc->astk->useshader)
    plflags &= ~(PL_HASVCOL | PL_HASPCOL);

  if (flag & APF_FACEDRAW) {
    glColorMaterial(GL_FRONT_AND_BACK, _mgopenglc->lmcolor);glEnable(GL_COLOR_MATERIAL);
    MAY_LIGHT();
    /* reestablish correct drawing color if necessary*/
    if (!(plflags & (PL_HASPCOL | PL_HASVCOL)))
	D4F(&(ma->ap.mat->diffuse));
    if((_mgc->astk->ap.flag & APF_TEXTURE) && (_mgc->astk->ap.tex != NULL)) {
        if(plflags & PL_HASST)
            mgopengl_needtexture();
    } else {
        plflags &= ~PL_HASST;
    }

    for (p = _p, i = 0; i < np; i++, p++) {
	if (plflags & PL_HASPCOL)
	    D4F(&p->pcol);
	if (plflags & PL_HASPN)
	    N3F(&p->pn, &(*p->v)->pt);
	v = p->v;
	if((j = p->n_vertices) <= 2) {
	    nonsurf = i;
	} else if(j <= 4 || !(flag & APF_CONCAVE)
			|| !mgopengl_trickypolygon(p, plflags)) {

	    glBegin(GL_POLYGON);
	    switch(plflags & (PL_HASVCOL|PL_HASVN|PL_HASST)) {
	    case 0:
		do {
		    glVertex4fv((float *)&(*v)->pt);
		    v++;
		} while(--j > 0);
		break;
	    case PL_HASVCOL:
		do {
		    D4F(&(*v)->vcol);
		    glVertex4fv((float *)&(*v)->pt);
		    v++;
		} while(--j > 0);
		break;
	    case PL_HASVN:
		do {
		    N3F(&(*v)->vn, &(*v)->pt);
		    glVertex4fv((float *)&(*v)->pt);
		    v++;
		} while(--j > 0);
		break;
	    case PL_HASVCOL|PL_HASVN:
		do {
		    D4F(&(*v)->vcol);
		    N3F(&(*v)->vn, &(*v)->pt);
		    glVertex4fv((float *)&(*v)->pt);
		    v++;
		} while(--j > 0);
		break;
	    default:
		do {
		    if (plflags & PL_HASVCOL) D4F(&(*v)->vcol);
		    if (plflags & PL_HASVN) N3F(&(*v)->vn, &(*v)->pt);
		    if (plflags & PL_HASST) glTexCoord2fv((GLfloat *)&(*v)->st);
		    glVertex4fv((float *)&(*v)->pt);
		    v++;
		} while(--j > 0);
		break;
	    }
	    glEnd();
	}
    }
  }

  if (flag & (APF_EDGEDRAW|APF_NORMALDRAW) || nonsurf >= 0) {
    if(_mgopenglc->znudge) mgopengl_closer();
    glDisable(GL_COLOR_MATERIAL);
    DONT_LIGHT();

    if (flag & APF_EDGEDRAW) {
	glColor3fv((float *)&_mgc->astk->ap.mat->edgecolor);
	for (p = _p, i = 0; i < np; i++, p++) {
	    glBegin(GL_LINE_LOOP);
	    for (j=0, v=p->v; j < p->n_vertices; j++, v++) {
		mgopengl_v4ffunc(&(*v)->pt);
	    }
	    glEnd();
	}
    }

    if (flag & APF_NORMALDRAW) {
	glColor3fv((float *)&_mgc->astk->ap.mat->normalcolor);
	if (plflags & PL_HASPN) {
	    for (p = _p, i = 0; i < np; i++, p++) {
		for (j=0, v=p->v; j < p->n_vertices; j++, v++)
		  mgopengl_drawnormal(&(*v)->pt, &p->pn);
	    }
	} else if (plflags & PL_HASVN) {
	    for (vp = V, i = 0; i < nv; i++, vp++) {
		mgopengl_drawnormal(&vp->pt, &vp->vn);
	    }
	}
    }


    if (nonsurf >= 0) {
      /* reestablish correct drawing color if necessary*/
      if (!(plflags & (PL_HASPCOL | PL_HASVCOL)))
	D4F(&(ma->ap.mat->diffuse));
      
      for(p = _p, i = 0; i <= nonsurf; p++, i++) {
	if (plflags & PL_HASPCOL)
	  D4F(&p->pcol);
	v = p->v;
	switch(j = p->n_vertices) {
	case 1:
	  if(plflags & PL_HASVCOL) glColor4fv((float *)&(*v)->vcol);
	  mgopengl_point(&(*v)->pt);
	  break;
	case 2:
	  glBegin(GL_LINE_STRIP);
	  do {
	    if(plflags & PL_HASVCOL) glColor4fv((float *)&(*v)->vcol);
	    glVertex4fv((float *)&(*v)->pt);
	    v++;
	  } while(--j > 0);
	  glEnd();
	  break;
	}
      }
    }
    if(_mgopenglc->znudge) mgopengl_farther();
  }
}

/*
 * Z-shift routines: for moving edges closer than faces, etc.
 */
void
mgopengl_init_zrange()
{
  register struct mgopenglcontext *gl = _mgopenglc;

  gl->znudge = (double) _mgc->zfnudge * (gl->zmax - gl->zmin);

  gl->znear = gl->zmin + fabs(gl->znudge * (double)MAXZNUDGE);
  gl->zfar  = gl->zmax - fabs(gl->znudge * (double)MAXZNUDGE);
#ifndef NO_ZNUDGE
  glDepthRange(gl->znear, gl->zfar);
#endif
}

void
mgopengl_closer()
{
#ifndef NO_ZNUDGE
  glDepthRange( _mgopenglc->znear -= _mgopenglc->znudge,  _mgopenglc->zfar -= _mgopenglc->znudge );
#endif
}
void
mgopengl_farther()
{
#ifndef NO_ZNUDGE
  glDepthRange( _mgopenglc->znear += _mgopenglc->znudge,  _mgopenglc->zfar += _mgopenglc->znudge );
#endif
}

/* There is a basic problem now with 4-d points and 3-d normal vectors.
For now, we'll just ignore the 4-th coordinate of the point when 
computing the tip of the normal vector.  This will work OK with all
existing models, but for genuine 4-d points it won't work.  But,
come to think of it, what is the correct interpretation of the
normal vector when the points live in 4-d?
*/
void
mgopengl_drawnormal(register HPoint3 *p, Point3 *n)
{
  Point3 end, tp;
  float scale;

  if (p->w <= 0.0) return;
  if(p->w != 1) {
    HPt3ToPt3(p, &tp);
    p = (HPoint3 *)&tp;
  }

  scale = _mgc->astk->ap.nscale;
  if(_mgc->astk->ap.flag & APF_EVERT) {
    register Point3 *cp = &_mgc->cpos;
    if(!(_mgc->has & HAS_CPOS))
	mg_findcam();
    if((p->x-cp->x) * n->x + (p->y-cp->y) * n->y + (p->z-cp->z) * n->z > 0)
	scale = -scale;
  }

  end.x = p->x + scale*n->x;
  end.y = p->y + scale*n->y;
  end.z = p->z + scale*n->z;

  DONT_LIGHT();

  glBegin(GL_LINE_STRIP);
  glVertex3fv((float *)p);
  glVertex3fv((float *)&end);
  glEnd();
}
