;;;
;;; elmo-imap4.el -- IMAP4 Interface for ELMO.
;;; Copyright (C) 1998 Yuuichi Teranishi <teranisi@gohome.org>
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to the
;;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;;; Boston, MA 02111-1307, USA.
;;;

(require 'elmo-vars)
(require 'elmo-util)
(require 'elmo-msgdb)
(require 'elmo-date)
(require 'elmo-cache)

(eval-when-compile
  (condition-case nil
      (progn
	(require 'ssl))
    (error)))

;;
;; internal variables
;;
(defvar elmo-imap4-seq-prefix "elmo-imap4")
(defvar elmo-imap4-seqno 0)
(defvar elmo-imap4-connection-cache nil
  "Cache of imap connection.")
;; buffer local variable
(defvar elmo-imap4-read-point 0)

;; buffer local variable
(defvar elmo-imap4-server-capability nil)
(defvar elmo-imap4-use-uid t)
(defvar elmo-imap4-use-float t)

(defsubst elmo-imap4-spec-username (spec)
  (nth 3 spec))

(defsubst elmo-imap4-spec-hostname (spec)
  (nth 2 spec))

(defsubst elmo-imap4-spec-folder (spec)
  (nth 1 spec))

(defsubst elmo-imap4-spec-auth (spec)
  (nth 4 spec))

(defsubst elmo-imap4-connection-get-process (conn)
  (nth 1 conn))

(defsubst elmo-imap4-connection-get-buffer (conn)
  (nth 0 conn))

(defsubst elmo-imap4-connection-get-cwf (conn)
  (nth 2 conn))

(defun elmo-imap4-flush-connection ()
  (interactive)
  (let ((cache elmo-imap4-connection-cache)
	buffer process)
    (while cache
      (setq buffer (car (cdr (car cache))))
      (if buffer (kill-buffer buffer))
      (setq process (car (cdr (cdr (car cache)))))
      (if process (delete-process process))
      (setq cache (cdr cache)))
    (setq elmo-imap4-connection-cache nil)))

(defun elmo-imap4-process-folder-list (string)
  (save-excursion
    (let ((tmp-buffer (get-buffer-create " *ELMO FOLDER-LIST TMP*"))
	  ret-val)
      (set-buffer tmp-buffer)
      (let ((case-fold-search t))
	(erase-buffer)
	(insert string)
	(goto-char (point-min))
	(while (re-search-forward 
		"\\* LIST (\\([^)]*\\)) \"/\" \\([^ \n]*[^/]\\)$" nil t)
	  (unless (string-match "noselect"
				(elmo-match-buffer 1))
	    (setq ret-val 
		  (append ret-val 
			  (list (elmo-match-buffer 2))))))
	(kill-buffer tmp-buffer)
	ret-val))))

(defun elmo-imap4-list-folders (spec &optional hierarchy)
  (save-excursion
    (let* ((folder (nth 1 spec))
	   (server (nth 2 spec))
	   (user (nth 3 spec))
	   (auth (nth 4 spec))
	   (port (nth 5 spec))
	   (ssl (nth 6 spec))
	   (connection (elmo-imap4-get-connection user server auth port ssl))
	   (buffer  (nth 0 connection))
	   (process (nth 1 connection))
	   response result append-serv)
      (if (equal '/' (nth (- (length folder) 1) (string-to-list folder)))
	  ()
	(setq folder (concat folder "/")))
      (elmo-imap4-send-command buffer process (format "list %s *" folder))
      (setq response (elmo-imap4-read-response buffer process))
      (setq result (elmo-imap4-process-folder-list response))
      (unless (string= user (user-login-name))
	(setq append-serv (concat ":" user)))
      (unless (string= server elmo-default-imap4-server)
	(setq append-serv (concat append-serv "@" server)))
      (unless (eq port elmo-default-imap4-port)
	(setq append-serv (concat append-serv ":" (int-to-string port))))
      (unless (eq ssl elmo-default-imap4-ssl)
	(setq append-serv (concat append-serv "!")))
      (mapcar '(lambda (fld)
		 (concat "%" fld (and append-serv 
				      (eval append-serv))))
	      result))))

(defun elmo-imap4-last-num (string)
  (save-excursion
    (save-match-data
      (let ((tmp-buf (get-buffer-create " *elmo-imap4-last-num*"))
	    ret-val)
	(set-buffer tmp-buf)
	(erase-buffer)
	(insert string)
	(goto-char (point-min))
	(if (re-search-forward " \\([0-9]+\\)$" nil t)
	    (setq ret-val (elmo-match-buffer 1))
	  (setq ret-val "0"))
	(kill-buffer tmp-buf)
	ret-val))))

(defun elmo-imap4-first-num (string)
  (save-excursion
    (save-match-data
      (let ((tmp-buf (get-buffer-create " *elmo-imap4-first-num*"))
	    ret-val)
	(set-buffer tmp-buf)
	(erase-buffer)
	(insert string)
	(goto-char (point-min))
	(if (re-search-forward "^ *\\([0-9]+\\) " nil t)
	    (setq ret-val (elmo-match-buffer 1))
	  (setq ret-val "0"))
	(kill-buffer tmp-buf)
	ret-val))))

(defun elmo-imap4-folder-exists-p (spec)
  (let* ((folder (nth 1 spec))
	 (server (nth 2 spec))
	 (user (nth 3 spec))
	 (auth (nth 4 spec))
	 (port (nth 5 spec))
	 (ssl (nth 6 spec))
	 (connection (elmo-imap4-get-connection user server auth port ssl))
	 (cwf (nth 2 connection)))
    (if (elmo-imap4-select-folder folder connection)
	t
      ;; select failed. so back to current working folder.
      (if cwf
	  (elmo-imap4-select-folder cwf connection))
      nil)))

(defun elmo-imap4-create-folder-maybe (spec dummy)
  "Create folder if necessary."
  (if (not (elmo-imap4-folder-exists-p spec))
      (elmo-imap4-create-folder spec)))

(defun elmo-imap4-create-folder (spec)
  (let* ((folder (nth 1 spec))
	 (server (nth 2 spec))
	 (user (nth 3 spec))
	 (auth (nth 4 spec))
	 (port (nth 5 spec))
	 (ssl (nth 6 spec))
	 (connection (elmo-imap4-get-connection user server auth port ssl))
	 (buffer  (nth 0 connection))
	 (process (nth 1 connection)))
    (when (and (not (null folder)))
      (elmo-imap4-send-command 
       buffer process 
       (format "create %s" folder))
      (if (null (elmo-imap4-read-response buffer process))
	  (error "Create folder %s failed." folder)
	t))))

(defun elmo-imap4-max-of-folder (spec)
  (save-excursion
    (let* ((folder (nth 1 spec))
	   (server (nth 2 spec))
	   (user (nth 3 spec))
	   (auth (nth 4 spec))
	   (port (nth 5 spec))
	   (ssl (nth 6 spec))
	   (connection (elmo-imap4-get-connection user server auth port ssl))
	   response estimated last-val)
      (if (or (null folder)
	      (null (setq response 
			  (elmo-imap4-select-folder 
			   folder connection))))
	  (error "Select folder failed"))
      (if (string-match "\\* \\([0-9]+\\) EXISTS" response)
	  (setq estimated (string-to-int (elmo-match-string 1 response))))
      (if (string-match "\\* OK \\[UIDNEXT \\([0-9]+\\)\\]" response)
	  (setq last-val 
		(- (string-to-int (elmo-match-string 1 response))
		   1)))
      (cons last-val estimated))))

(defun elmo-imap4-get-connection (user server auth port ssl)
  (let ((user-at-host (format "%s@%s" user server))
	ret-val result buffer process proc-stat
	user-at-host-on-port
	)
    (if (not elmo-plugged)
	(error "Unplugged."))
    (setq user-at-host-on-port 
	  (concat user-at-host ":" (int-to-string port)
		  (if ssl "!")))
    (setq ret-val (assoc user-at-host-on-port
			 elmo-imap4-connection-cache))
    (if (and ret-val 
	     (or (eq (setq proc-stat 
			   (process-status (cadr (cdr ret-val)))) 
		     'closed)
		 (eq proc-stat 'exit)))
	;; connection is closed...
	(progn
	  (kill-buffer (car (cdr ret-val)))
	  (setq elmo-imap4-connection-cache 
		(delete ret-val elmo-imap4-connection-cache))
	  (setq ret-val nil)
	  ))
    (if ret-val
	(cdr ret-val)
      (setq result
	    (elmo-imap4-open-connection server user auth port
					(elmo-get-passwd user-at-host)
					ssl))
      (if (null result)
	  (error "Connection failed."))
      (and elmo-debug 
	   (message "Connected to %s" user-at-host-on-port))
      (setq buffer (car result))
      (setq process (cdr result))
      (when (and process (null buffer))
	(elmo-remove-passwd user-at-host)
	(delete-process process)
	(error "Login failed.")
	)
      (setq elmo-imap4-connection-cache 
	    (append elmo-imap4-connection-cache 
		    (list 
		     (cons user-at-host-on-port
			   (setq ret-val (list buffer process 
					       ""; current-folder..
					       ))))))
      ret-val)))

(defun elmo-imap4-process-filter (process output)
  (save-excursion
    (set-buffer (process-buffer process))
    (goto-char (point-max))
    (insert output)))

(defun elmo-imap4-read-response (buffer process &optional not-command)
  (save-excursion
    (set-buffer buffer)
    (let ((case-fold-search nil)
	  (response-string nil)
	  (response-continue t)
	  (return-value nil)
	  match-end)
      (while response-continue
	(goto-char elmo-imap4-read-point)
	(while (not (search-forward "\r\n" nil t))
	  (accept-process-output process)
	  (goto-char elmo-imap4-read-point))
	
	(setq match-end (point))
	(setq response-string
	      (buffer-substring elmo-imap4-read-point (- match-end 2)))
	(goto-char elmo-imap4-read-point)
	(if (looking-at (format "%s[0-9]+ OK.*$\\|\\+.*$" 
				elmo-imap4-seq-prefix))
	    (progn (setq response-continue nil)
		   (setq elmo-imap4-read-point match-end)
		   (setq return-value 
			 (if return-value 
			     (concat return-value "\n" response-string)
			   response-string
			   )))
	  (if (looking-at (format "\\(. BYE.*\\|%s[0-9]+ \\(NO\\|BAD\\).*\\)$" 
				  elmo-imap4-seq-prefix))
	      (progn (setq response-continue nil)
		     (setq elmo-imap4-read-point match-end)
		     (setq return-value nil))
	    (setq elmo-imap4-read-point match-end)
	    (if not-command
		(setq response-continue nil))
	    (setq return-value 
		  (if return-value 
		      (concat return-value "\n" response-string)
		    response-string)))
	  (setq elmo-imap4-read-point match-end)))
      return-value)
    ))

(defun elmo-imap4-read-contents (buffer process)
  "Read OK response"
  (save-excursion
    (set-buffer buffer)
    (let ((case-fold-search nil)
	  (response-string nil)
	  match-end)
      (goto-char elmo-imap4-read-point)
      (while (not (re-search-forward 
		   (format "%s[0-9]+ \\(NO\\|BAD\\|OK\\).*$" 
			   elmo-imap4-seq-prefix)
		   nil t))
	(accept-process-output process)
	(goto-char elmo-imap4-read-point))
      (beginning-of-line)
      (setq match-end (point))
      (setq response-string (elmo-delete-cr
			     (buffer-substring elmo-imap4-read-point match-end)))
      (if (eq (length response-string) 0)
	  nil
	response-string))))

(defun elmo-imap4-read-bytes (buffer process bytes)
  (save-excursion
    (set-buffer buffer)
    (let ((case-fold-search nil)
	  (return-value nil)
	  start gc-message)
      (setq start elmo-imap4-read-point);; starting point
      (while (< (point-max) (+ start bytes))
	(accept-process-output process))
      (setq return-value (buffer-substring
			  start (+ start bytes)))
      (setq return-value (elmo-delete-cr return-value))
      (setq elmo-imap4-read-point bytes)
      return-value)))

(defun elmo-imap4-read-body (buffer process bytes outbuf)
  (save-excursion
    (set-buffer buffer)
    (let ((case-fold-search nil)
	  (return-value nil)
	  start gc-message)
      (setq start elmo-imap4-read-point);; starting point
      (while (< (point-max) (+ start bytes))
	(accept-process-output process))
      (setq return-value (buffer-substring
			  start (+ start bytes)))
      (setq return-value (elmo-delete-cr-get-content-type outbuf return-value))
      (setq elmo-imap4-read-point bytes)
      return-value)))

(defun elmo-imap4-noop (connection)
  (let ((buffer (car connection))
	(process (cadr connection)))
    (save-excursion
      (elmo-imap4-send-command buffer 
			       process "noop")
      (elmo-imap4-read-response buffer process))))

(defun elmo-imap4-commit (folder)
  (let ((spec (elmo-folder-get-spec folder)))
    (elmo-imap4-select-folder (nth 1 spec)
			      (elmo-imap4-get-connection (nth 3 spec)
							 (nth 2 spec)
							 (nth 4 spec)
							 (nth 5 spec)
							 (nth 6 spec)
							 ))))
(defun elmo-imap4-check (connection)
  (let ((buffer (car connection))
	(process (cadr connection)))
    (save-excursion
      (elmo-imap4-send-command buffer 
			       process "check")
      (elmo-imap4-read-response buffer process))))

(defun elmo-imap4-select-folder (folder connection)
  (let ((buffer (car connection))
	(process (cadr connection))
	response)
    (save-excursion
      (unwind-protect
	  (progn
	    (elmo-imap4-send-command buffer 
				     process (format "select %s" folder))
	    (setq response (elmo-imap4-read-response buffer process)))
	(setcar (cddr connection) folder)))
    response
    ))

(defun elmo-imap4-check-validity (spec validity-file)
  "get uidvalidity value from server and compare it with validity-file."
  (let* ((user (nth 3 spec))
	 (server (nth 2 spec))
	 (folder   (nth 1 spec))
	 (auth     (nth 4 spec))
	 (port     (nth 5 spec))
	 (ssl (nth 6 spec))
	 (connection (elmo-imap4-get-connection user server auth port ssl))
	 (buffer  (nth 0 connection))
	 (process (nth 1 connection))
	 response)
    (save-excursion
      (elmo-imap4-send-command buffer 
			       process 
			       (format "status %s (uidvalidity)" folder))
      (setq response (elmo-imap4-read-response buffer process))
      (if (string-match "UIDVALIDITY \\([0-9]+\\)" response)
	  (string= (elmo-get-file-string validity-file)
		   (elmo-match-string 1 response))
	nil))))

(defun elmo-imap4-sync-validity  (spec validity-file)
  "get uidvalidity value from server and save it to validity-file."
  (let* ((user   (nth 3 spec))
	 (server (nth 2 spec))
	 (folder (nth 1 spec))
	 (auth   (nth 4 spec))
	 (port   (nth 5 spec))
	 (ssl (nth 6 spec))
	 (connection (elmo-imap4-get-connection user server auth port ssl))
	 (buffer  (nth 0 connection))
	 (process (nth 1 connection))
	 response)
    (save-excursion
      (elmo-imap4-send-command buffer 
			       process 
			       (format "status %s (uidvalidity)" folder))
      (setq response (elmo-imap4-read-response buffer process))
      (if (string-match "UIDVALIDITY \\([0-9]+\\)" response)
	  (progn
	    (elmo-save-string
	     (elmo-match-string 1 response)
	     validity-file)
	    t)
	nil))))

(defun elmo-imap4-list-folder (spec)
  (save-excursion
    (let* ((user (nth 3 spec))
	   (server (nth 2 spec))
	   (folder   (nth 1 spec))
	   (auth     (nth 4 spec))
	   (port     (nth 5 spec))
	   (ssl (nth 6 spec))
	   (connection (elmo-imap4-get-connection user server auth port ssl))
	   (buffer  (nth 0 connection))
	   (process (nth 1 connection))
	   (cwf     (nth 2 connection))
	   response ret-val beg end)
      (and (not (null folder))
	   (if (not (string= cwf folder))
	       (if (null (setq response 
			       (elmo-imap4-select-folder 
				folder connection)))
		   (error "Select folder failed"))
	     (if elmo-imap4-use-select-to-update-status
		 (elmo-imap4-select-folder 
		  folder connection) ;; for status update.
	       (elmo-imap4-check connection)) ;; for status update.
	     ))
      (elmo-imap4-send-command buffer process (if elmo-imap4-use-uid 
						  "uid search all"
						"search all"))
      (setq response (elmo-imap4-read-response buffer process))
      (if (and response 
	       (string-match "^\\* SEARCH" response))
	  (progn
	    (setq beg (match-end 0))
	    (if (string-match "\n" response)
		(progn
		  (setq end (match-end 0))
		  (setq ret-val (read (concat "(" (substring 
						   response 
						   beg end) ")"))))
	      (error "SEARCH failed."))))
      ret-val)))

(defun elmo-imap4-search (spec key &optional from-msgs)
  (save-excursion
    (let* ((folder (nth 1 spec))
	   (hostname (nth 2 spec))
	   (username (nth 3 spec))
	   (auth     (nth 4 spec))
	   (port     (nth 5 spec))
	   (ssl (nth 6 spec))
	   (connection (elmo-imap4-get-connection username hostname auth port
						  ssl))
	   (buffer  (nth 0 connection))
	   (process (nth 1 connection))
	   (cwf     (nth 2 connection))
	   response ret-val len word search-key search-value)
      (if (and (not (null folder))
	       (not (string= cwf folder))
	       (null (setq response 
			   (elmo-imap4-select-folder 
			    folder connection))))
	  (error "Select folder failed."))
      (when (string-match "\\([^ ]+\\) *\\(.*\\)$" key)
	(setq search-key (elmo-match-string 1 key))
	(setq search-value (elmo-match-string 2 key)))
      (cond
       ((or (string= "since" search-key)
	    (string= "before" search-key))
	(setq search-key (concat "sent" search-key))
	(elmo-imap4-send-command buffer process
				 (format 
				  (if elmo-imap4-use-uid
				      "uid search %s %s" 
				    " search %s %s")
				  search-key
				  (elmo-date-get-description
				   (elmo-date-get-datevec search-value))))
	)
       (t
	(setq word (encode-mime-charset-string search-value elmo-mime-charset))
	(setq len (length word))
	(elmo-imap4-send-command buffer process 
				 (format 
				  (if elmo-imap4-use-uid
				      "uid search CHARSET %s %s {%d}" 
				    " search CHARSET %s %s {%d}")
				  (symbol-name elmo-mime-charset)
				  search-key
				  len))
	(if (null (elmo-imap4-read-response buffer process t))
	    (error "Searching failed because of server capability??"))
	(elmo-imap4-send-string buffer process word)))
      (if (null (setq response (elmo-imap4-read-response buffer process)))
	  (error "Search failed for %s." key))
      (if (string-match "^\\* SEARCH\\([^\n]*\\)$" response)
	  (setq ret-val (read (concat "(" (elmo-match-string 1 response) ")")))
	(error "SEARCH failed."))
      (if from-msgs 
	  (elmo-list-filter from-msgs ret-val)
	ret-val))))
  
(defun elmo-imap4-nth (num ov-list)
  (let ((value (nth num ov-list)))
    (if (eq 'NIL value)
	nil
      value)))

(defun elmo-imap4-create-msgdb-from-overview-string (str
						     folder
						     new-mark
						     already-mark
						     seen-mark
						     important-mark
						     seen-list
						     &optional numlist)
  (let ((case-fold-search t)
	ov-list 
	ov-entity
	num env-list message-id from-list from-string
	to-string cc-string
	overview number-alist mark-alist
	reference subject date-string size flags gmark seen)
    (setq ov-list (elmo-imap4-parse-overview-string str))
    (while ov-list
      (setq ov-entity (car ov-list))
      (setq num (string-to-int (elmo-imap4-nth 0 ov-entity)))
      (when (or (null numlist)
		(memq num numlist))
	(setq env-list (elmo-imap4-nth 1 ov-entity))
	(setq message-id (elmo-imap4-nth 9 env-list))
	(setq from-list (car (elmo-imap4-nth 2 env-list)))
	(setq from-string (elmo-delete-char 
			   ?\"
			   (concat (elmo-imap4-nth 0 from-list)
				   " <"
				   (elmo-imap4-nth 2 from-list)
				   "@"
				   (elmo-imap4-nth 3 from-list)
				   ">")))
	(setq to-string (mapconcat
			 '(lambda (to-entity)
			    (concat (elmo-imap4-nth 0 to-entity)
				    " <"
				    (elmo-imap4-nth 2 to-entity)
				    "@"
				    (elmo-imap4-nth 3 to-entity)
				    ">"))
			 (elmo-imap4-nth 5 env-list) ","))
	(setq cc-string (mapconcat
			 '(lambda (cc-entity)
			    (concat (elmo-imap4-nth 0 cc-entity)
				    " <"
				    (elmo-imap4-nth 2 cc-entity)
				    "@"
				    (elmo-imap4-nth 3 cc-entity)
				    ">"))
			 (elmo-imap4-nth 6 env-list) ","))
	(setq reference 
	      (elmo-msgdb-get-last-message-id
	       (concat (elmo-imap4-nth 2 ov-entity);references
		       (elmo-imap4-nth 8 env-list) ;in-reply-to
		       )))
	(setq subject (or (elmo-imap4-nth 1 env-list) elmo-no-subject))
	(setq date-string (elmo-imap4-nth 0 env-list))
	(setq size (string-to-int (elmo-imap4-nth 3 ov-entity)))
	(setq flags (elmo-imap4-nth 4 ov-entity))
	(setq seen (member message-id seen-list))
	(setq overview
	      (elmo-msgdb-append-element
	       overview
	       (cons message-id
		     (vector num
			     reference
			     (encode-mime-charset-string
			      (elmo-eword-decode-string 
			       (decode-mime-charset-string
				from-string
				elmo-mime-charset))
			      elmo-mime-charset)
			     (encode-mime-charset-string
			      (elmo-eword-decode-string 
			       (decode-mime-charset-string
				subject
				elmo-mime-charset))
			      elmo-mime-charset)
			     date-string
			     to-string
			     cc-string
			     size
			     nil ; extra-field-list
			     )))) 
	(if (memq 'Flagged flags)
	    (elmo-msgdb-global-mark-set message-id important-mark))
	(setq number-alist
	      (elmo-msgdb-number-add number-alist num message-id))
	(if (setq gmark (or (elmo-msgdb-global-mark-get message-id)
			    (if (elmo-cache-exists-p message-id) ;; XXX
				(if (or (memq 'Seen flags) seen)
				    nil
				  already-mark)
			      (if (or (memq 'Seen flags) seen)
				  (if elmo-imap4-use-cache
				      seen-mark)
				new-mark))))
	    (setq mark-alist (elmo-msgdb-mark-append
			      mark-alist
			      num
			      ;; managing mark with message-id is evil.
			      gmark))))
      (setq ov-list (cdr ov-list)))
    (list overview number-alist mark-alist)))

;;
;; set mark
;; read-mark -> "\\Seen"
;; important -> "\\Flagged"
;; 
;; (delete -> \\Deleted)
(defun elmo-imap4-mark-set-on-msgs (spec msgs mark &optional unmark)
  "SET flag of MSGS as MARK.
If optional argument UNMARK is non-nil, unmark."
  (save-excursion
    (let* ((folder   (nth 1 spec))
	   (hostname (nth 2 spec))
	   (username (nth 3 spec))
	   (auth     (nth 4 spec))
	   (port     (nth 5 spec))
	   (ssl (nth 6 spec))
	   (connection (elmo-imap4-get-connection username hostname auth 
						  port ssl))
	   (buffer  (nth 0 connection))
	   (process (nth 1 connection))
	   (cwf     (nth 2 connection))
	   (msg-list (copy-sequence msgs))
	   cont-list ent)
      (if (and (not (null folder))
	       (not (string= cwf folder))
	       (null (elmo-imap4-select-folder folder connection)))
	  (error "Select folder failed"))
      (setq msg-list (sort msg-list '<))
      (while msg-list
	(setq cont-list 
	      (elmo-add-to-cont-list cont-list (car msg-list)))
	(setq msg-list (cdr msg-list)))
      (while cont-list
	(setq ent (car cont-list))
	(if (stringp ent)
	    (setq ent (string-to-int ent)))
	(cond ((integerp ent)
	       (elmo-imap4-send-command buffer process 
					(format 
					 (if elmo-imap4-use-uid 
					     "uid store %d %sflags.silent (%s)"
					   "store %d %sflags.silent (%s)")
					 ent 
					 (if unmark "-" "+")
					 mark)))
	      ((consp ent)
	       (elmo-imap4-send-command 
		buffer 
		process 
		(format 
		 (if elmo-imap4-use-uid 
		     "uid store %d:%d %sflags.silent (%s)"
		   "store %d:%d %sflags.silent (%s)")
		 (car ent)
		 (cdr ent)
		 (if unmark "-" "+")
		 mark))))
	(if (null (elmo-imap4-read-response buffer process))
	    (error "Store %s flag failed." mark)
	  (setq cont-list (cdr cont-list))
	  ))
      (elmo-imap4-send-command buffer process "expunge")
      (if (null (elmo-imap4-read-response buffer process))
	  (error "Expunge failed")))
    t))

(defun elmo-imap4-mark-as-important (spec msgs)
  (elmo-imap4-mark-set-on-msgs spec msgs "\\Flagged")
  )

(defun elmo-imap4-mark-as-read (spec msgs)
  (elmo-imap4-mark-set-on-msgs spec msgs "\\Seen")
  )

(defun elmo-imap4-unmark-important (spec msgs)
  (elmo-imap4-mark-set-on-msgs spec msgs "\\Flagged" 'unmark)  
  )

(defun elmo-imap4-mark-as-unread (spec msgs)
  (elmo-imap4-mark-set-on-msgs spec msgs "\\Seen" 'unmark)
  )

(defun elmo-imap4-delete-msgs (spec msgs)
  (elmo-imap4-mark-set-on-msgs spec msgs "\\Deleted"))

(defun elmo-imap4-msgdb-create-as-numlist (spec numlist new-mark already-mark 
						seen-mark important-mark 
						seen-list)
  "Create msgdb for SPEC for NUMLIST."
  (elmo-imap4-msgdb-create spec numlist new-mark already-mark
			   seen-mark important-mark seen-list t)
  )

(defun elmo-imap4-msgdb-create (spec numlist new-mark already-mark seen-mark 
				     important-mark seen-list &optional as-num)
  "create msgdb for SPEC."
  (when numlist
    (save-excursion
      (let* ((folder (nth 1 spec))
	     (server (nth 2 spec))
	     (user (nth 3 spec))
	     (auth (nth 4 spec))
	     (port (nth 5 spec))
	     (ssl (nth 6 spec))
	     (connection (elmo-imap4-get-connection user server auth port ssl))
	     (buffer     (car connection))
	     (process    (cadr connection))
	     (cwf        (caddr connection))
	     (filter     (and as-num numlist))
	     (case-fold-search t)
	     rfc2060 ret-val
	     beg-num end-num cur length ov-str
	     )
	(setq rfc2060 
	      (save-excursion (set-buffer buffer)
			      (if (memq 'imap4rev1 
					elmo-imap4-server-capability)
				  t
				(if (memq 'imap4
					  elmo-imap4-server-capability)
				    nil
				  (error "No IMAP4 capability!!")))))
	(setq beg-num (car numlist)
	      cur beg-num
	      end-num (nth (1- (length numlist)) numlist)
	      length  (+ (- end-num beg-num) 1))
	(message "getting overview...")
	(if (and (not (null folder))
		 (not (string= cwf folder))
		 (null (elmo-imap4-select-folder folder connection)))
	    (error "Select imap folder %s failed" folder))
	(while (<= cur end-num)
	  (elmo-imap4-send-command 
	   buffer process 
	   ;; get overview entity from IMAP4
	   (format 
	    (if rfc2060
		(if elmo-imap4-use-uid 
		    "uid fetch %d:%d (envelope body.peek[header.fields (references)] rfc822.size flags)"
		  "fetch %d:%d (envelope body.peek[header.fields (references)] rfc822.size flags)")
	      (if elmo-imap4-use-uid
		  "uid fetch %d:%d (envelope rfc822.size flags)"
		"fetch %d:%d (envelope rfc822.size flags)")
	      )
	    cur
	    (min (+ cur 
		    elmo-imap4-overview-fetch-chop-length)
		 end-num)))
	  ;; process string while waiting for response
	  (if ov-str
	      (setq ret-val
		    (elmo-msgdb-append 
		     ret-val
		     (elmo-imap4-create-msgdb-from-overview-string 
		      ov-str
		      folder
		      new-mark already-mark seen-mark important-mark
		      seen-list filter))))
	  (setq cur (+ elmo-imap4-overview-fetch-chop-length cur 1))
	  (setq ov-str (elmo-imap4-read-contents buffer process))
	  (message "getting overview...%d%%" 
		   (/ (* (+ (- (min cur end-num)
			       beg-num) 1) 100) length)))
	(if ov-str
	    (setq ret-val
		  (elmo-msgdb-append 
		   ret-val
		   (elmo-imap4-create-msgdb-from-overview-string 
		    ov-str 
		    folder
		    new-mark already-mark seen-mark important-mark
		    seen-list filter))))
	(message "getting overview...done.")
	ret-val))))

(defconst elmo-imap4-attribute-alist
  (list (cons "UID" 'num)
	(cons "ENVELOPE" 'paren)
	(cons "REFERENCES" nil)
	(cons "RFC822.SIZE" 'num)
	(cons "FLAGS" 'paren)
	))

(defconst elmo-imap4-attribute-regexp
  "\\(FLAGS\\|UID\\|RFC822\\.SIZE\\|ENVELOPE\\|BODY\\[[^\]]+\\] \\(\{[0-9]+\}\n\\)?\\)"
  )

(defun elmo-imap4-make-attributes ()
  (let ((attributes elmo-imap4-attribute-alist)
	(attr-regexp elmo-imap4-attribute-regexp)
	attr-name type beg value ret-val)
    (while (re-search-forward attr-regexp nil t)
      (setq attr-name (elmo-match-buffer 0))
      (setq type (cdr (assoc attr-name attributes)))
      (cond ((eq type 'paren)
	     (search-forward "(" nil t)
	     (setq beg (match-beginning 0))
	     (goto-char (scan-sexps beg 1))
	     (setq value (read (buffer-substring beg (point)))))
	    ((eq type 'num)
	     (search-forward " " nil t)
	     (setq beg (point))
	     (search-forward " " nil t)
	     (setq value (buffer-substring beg (match-beginning 0))))
	    (t ;; references field......UW imapd is buggy??
	     (setq attr-name "REFERENCES")
	     (setq beg (match-end 0))
	     (when (or (re-search-forward attr-regexp nil t)
		       (search-forward ")" nil t))
	       (setq value (buffer-substring beg (match-beginning 0)))
	       (goto-char (match-beginning 0)))))
      (setq ret-val
	    (cons (cons attr-name value) ret-val)))
    ret-val))

(defun elmo-imap4-parse (string)
  (save-match-data
    (elmo-set-work-buf
     (let ((case-fold-search t)
	   (cur-attr elmo-imap4-attribute-alist)
	   bytes attrs ret-val end content)
       (insert string)
       (goto-char (point-min))
       (if (re-search-forward "BODY\\[[^\]]+\\]" nil t)
	   (setq end (match-beginning 0)))
       ;; process strings.
       (goto-char (point-min))
       (while (re-search-forward "\{\\([0-9]+\\)\}\n" end t)
	 (setq bytes (string-to-int
		      (elmo-match-buffer 1)))
	 (setq content (buffer-substring 
			(match-end 0) (+ (match-end 0) bytes)))
	 (delete-region (match-beginning 0) (+ (match-end 0) bytes))
	 (prin1 content (current-buffer)))
       ;; make attribute list
       (goto-char (point-min))
       (setq attrs (elmo-imap4-make-attributes))
       (while cur-attr
	 (setq ret-val (append ret-val
			       (list (cdr (assoc (car (car cur-attr))
						 attrs)))))
	 (setq cur-attr (cdr cur-attr)))
       ret-val))))

(defun elmo-imap4-parse-overview-string (string)
  (if (null string)
      (error "Getting overview failed."))
  (save-excursion
    (let ((tmp-buffer (get-buffer-create " *ELMO Overview TMP*"))
	  ret-val beg ov-list)
      (set-buffer tmp-buffer)
      (erase-buffer)
      (insert string)
      (goto-char (point-min))
      (setq beg (point))
      (if (re-search-forward "^\* \\([0-9]+\\) FETCH" 
			     nil t)
	  (progn
	    (setq beg (point))
	    (while (re-search-forward 
		    "^\* \\([0-9]+\\) FETCH" 
		    nil t)
	      (save-match-data
		(setq ov-list
		      (elmo-imap4-parse
		       (buffer-substring 
			beg (match-beginning 0)))))
	      (if (null (car ov-list))
		  (save-match-data
		    (save-excursion 
		      (goto-char beg)
		      (re-search-backward "^\* \\([0-9]+\\) FETCH" 
			     nil t)
		      (setcar ov-list
			      (elmo-match-buffer 1)))))
	      (setq beg (point))
	      (setq ret-val (append ret-val (list ov-list))))
	    ;; process last one...
	    (setq ov-list (elmo-imap4-parse
			   (buffer-substring beg (point-max))))
	    (if (null (car ov-list))
		(setcar ov-list
			(elmo-match-buffer 1)))
	    (setq ret-val (append ret-val (list ov-list)))
	    (kill-buffer tmp-buffer)))
      ret-val)))

(defun elmo-imap4-parse-capability (string)
  (if (string-match "^\\*\\(.*\\)$" string)
      (read (concat "(" (downcase (elmo-match-string 1 string)) ")"))))

(defun elmo-imap4-open-connection (imap4-server user auth port passphrase ssl)
  "Open Imap connection and returns 
the list of (process session-buffer current-working-folder).
Return nil if connection failed."
  (let ((process nil)
	(host imap4-server)
	;(port elmo-imap4-port)
	process-buffer ret-val response capability)
    (catch 'done
      (as-binary-process
       (setq process-buffer
	     (get-buffer-create (format " *IMAP session to %s:%d" host port)))
       (save-excursion
	 (set-buffer process-buffer)
	 (make-variable-buffer-local 'elmo-imap4-server-capability)
	 (erase-buffer))
       (if (not ssl)
	   (setq process (open-network-stream "IMAP" process-buffer host port))
	 (require 'ssl)
	 (setq process (open-ssl-stream "IMAP" process-buffer host port)))
       (and (null process) (throw 'done nil))
       (set-process-filter process 'elmo-imap4-process-filter)
       (process-kill-without-query process)
       ;; flush connections when exiting...
       (save-excursion
	 (set-buffer process-buffer)
	 (make-local-variable 'elmo-imap4-read-point)
	 (setq elmo-imap4-read-point (point-min))
	 (if (null (setq response
			 (elmo-imap4-read-response process-buffer process t)))
	     (throw 'done nil)
	   (when (string-match "^\\* PREAUTH" response)
	     (setq ret-val (cons process-buffer process))
	     (throw 'done nil)))
	 (elmo-imap4-send-command process-buffer process "capability")
	 (setq elmo-imap4-server-capability
	       (elmo-imap4-parse-capability
		(elmo-imap4-read-response process-buffer process)))
	 (setq capability elmo-imap4-server-capability)
	 (if (or (and (string= "auth" auth)
		      (not (memq 'auth=login capability)))
		 (and (string= "cram-md5" auth)
		      (not (memq 'auth=cram-md5 capability))))
	     (if (or elmo-imap4-force-login
		     (y-or-n-p 
		      (format 
		       "There's no %s capablility in server. continue?" auth)))
		 (setq auth "login")
	       (error "Login aborted.")))
	 (cond 
	  ((string= "auth" auth)
	   (elmo-imap4-send-command 
	    process-buffer process "authenticate login")
	   ;; Base64 
	   (when (null (elmo-imap4-read-response process-buffer process t))
	     (setq ret-val (cons nil process))
	     (throw 'done nil))
	   (elmo-imap4-send-string
	    process-buffer process (elmo-base64-encode-string user))
	   (when (null (elmo-imap4-read-response process-buffer process t))
	     (setq ret-val (cons nil process))
	     (throw 'done nil))
	   (elmo-imap4-send-string
	    process-buffer process (elmo-base64-encode-string passphrase))
	   (when (null (elmo-imap4-read-response process-buffer process))
	     (setq ret-val (cons nil process))
	     (throw 'done nil))
	   (setq ret-val (cons process-buffer process)))
	  (t ;; not auth... try login
	   (elmo-imap4-send-command 
	    process-buffer process 
	    (format "login %s \"%s\"" user 
		    (elmo-replace-in-string passphrase
					    "\"" "\\\\\"")))
	   (if (null (elmo-imap4-read-response process-buffer process))
	       (setq ret-val (cons nil process))
	     (setq ret-val (cons process-buffer process))))))))
    ret-val))
	    
(defun elmo-imap4-get-seqno ()
  (setq elmo-imap4-seqno (+ 1 elmo-imap4-seqno)))

(defun elmo-imap4-setup-send-buffer (string)
  (let ((tmp-buf (get-buffer-create " *elmo-imap4-setup-send-buffer*")))
    (save-excursion
      (save-match-data
	(set-buffer tmp-buf)
	(erase-buffer)
	(insert string)
	(goto-char (point-min))
	(if (eq (re-search-forward "^$" nil t) 
		(point-max))
	    (insert "\n"))
	(goto-char (point-min))
	(while (search-forward "\n" nil t)
	  (replace-match "\r\n"))))
    tmp-buf))

(defun elmo-imap4-send-command (buffer process command)
  "Send COMMAND string to server with sequence number."
  (save-excursion
    (set-buffer buffer)
    (erase-buffer)
    (goto-char (point-min))
    (setq elmo-imap4-read-point (point))
    (process-send-string process (concat (format "%s%d " 
						 elmo-imap4-seq-prefix 
						 (elmo-imap4-get-seqno))
					 command))
    (process-send-string process "\r\n")))

(defun elmo-imap4-send-string (buffer process string)
  "Send STRING to server."
  (save-excursion
    (set-buffer buffer)
    (erase-buffer)
    (goto-char (point-min))
    (setq elmo-imap4-read-point (point))
    (process-send-string process string)
    (process-send-string process "\r\n")))

(defun elmo-imap4-read-part (folder msg part)
  (save-excursion
    (let* ((spec (elmo-folder-get-spec folder))
	   (username (nth 3 spec))
	   (hostname (nth 2 spec))
	   (folder   (nth 1 spec))
	   (auth     (nth 4 spec))
	   (port     (nth 5 spec))
	   (ssl     (nth 6 spec))
	   (connection (elmo-imap4-get-connection username hostname auth 
						  port ssl))
	   (buffer  (nth 0 connection))
	   (process (nth 1 connection))
	   (cwf     (nth 2 connection))
	   response ret-val bytes)
      (when (and (not (null folder)))
	(when (not (string= cwf folder))
	  (if (null (setq response 
			  (elmo-imap4-select-folder 
			   folder connection)))
	      (error "Select folder failed")))
	(elmo-imap4-send-command buffer process 
				 (format 
				  (if elmo-imap4-use-uid 
				      "uid fetch %s body[%s]"
				    "fetch %s body[%s]")
				  msg part))
	(if (null (setq response (elmo-imap4-read-response 
				  buffer process t)))
	    (error "Fetch failed"))
	(save-match-data
	  (while (string-match "^\\* OK" response)
	    (if (null (setq response (elmo-imap4-read-response 
				      buffer process t)))
		(error "Fetch failed"))))
	(save-match-data
	  (if (string-match ".*{\\([0-9]+\\)}" response)
	      (setq bytes
		    (string-to-int
		     (elmo-match-string 1 response)))
	    (error "Fetch failed")
	    )
	  )
	(if (null (setq response (elmo-imap4-read-bytes 
				  buffer process bytes)))
	    (error "Fetch message failed"))
	(setq ret-val response)
	(elmo-imap4-read-response buffer process)) ;; ignore remaining..
      ret-val)))

(defun elmo-imap4-prefetch-msg (spec msg outbuf)
  (elmo-imap4-read-msg spec msg outbuf 'unseen)
  )

(defun elmo-imap4-read-msg (spec msg outbuf 
				 &optional leave-seen-flag-untouched)
  (save-excursion
    (let* ((username (nth 3 spec))
	   (hostname (nth 2 spec))
	   (folder   (nth 1 spec))
	   (auth     (nth 4 spec))
	   (port     (nth 5 spec))
	   (ssl      (nth 6 spec))
	   (connection (elmo-imap4-get-connection username hostname auth port
						  ssl))
	   (buffer  (nth 0 connection))
	   (process (nth 1 connection))
	   (cwf     (nth 2 connection))
	   response ret-val bytes)
      (as-binary-process
       (when (and (not (null folder)))
	 (when (not (string= cwf folder))
	   (if (null (setq response 
			   (elmo-imap4-select-folder 
			    folder connection)))
	       (error "Select folder failed")))
	 (elmo-imap4-send-command buffer process 
				  (format 
				   (if elmo-imap4-use-uid 
				       "uid fetch %s body%s[]" 
				     "fetch %s body%s[]")
				   msg
				   (if leave-seen-flag-untouched
				       ".peek" "")
				   ))
	 (if (null (setq response (elmo-imap4-read-response 
				   buffer process t)))
	     (error "Fetch failed"))
	 (save-match-data
	   (while (string-match "^\\* OK" response)
	     (if (null (setq response (elmo-imap4-read-response 
				       buffer process t)))
		 (error "Fetch failed"))))
	 (save-match-data
	   (if (string-match ".*{\\([0-9]+\\)}" response)
	       (setq bytes
		     (string-to-int
		      (elmo-match-string 1 response)))
	     (error "Fetch failed")
	     )
	   )
	 (setq ret-val (elmo-imap4-read-body
			buffer process bytes outbuf))
	 (elmo-imap4-read-response buffer process)) ;; ignore remaining..
       )
      ret-val)))

(defun elmo-imap4-setup-send-buffer-from-file (file)
  (let ((tmp-buf (get-buffer-create 
		  " *elmo-imap4-setup-send-buffer-from-file*")))
    (save-excursion
      (save-match-data
	(set-buffer tmp-buf)
	(erase-buffer)
	(as-binary-input-file
	 (insert-file-contents file))
	(goto-char (point-min))
	(if (eq (re-search-forward "^$" nil t) 
		(point-max))
	    (insert "\n"))	
	(goto-char (point-min))
	(while (search-forward "\n" nil t)
	  (replace-match "\r\n"))))
    tmp-buf))

(defun elmo-imap4-delete-msgids (spec msgids)
  "If actual message-id is matched, then delete it."
  (let ((message-ids msgids))
    (while message-ids
      (elmo-imap4-delete-msg-by-id spec (car message-ids))
      (setq message-ids (cdr message-ids)))))

(defun elmo-imap4-delete-msg-by-id (spec msgid)
  (save-excursion
    (let* ((folder (nth 1 spec))
	   (hostname (nth 2 spec))
	   (username (nth 3 spec))
	   (auth     (nth 4 spec))
	   (port     (nth 5 spec))
	   (ssl     (nth 6 spec))
	   (connection (elmo-imap4-get-connection username hostname auth 
						  port ssl))
	   (buffer  (nth 0 connection))
	   (process (nth 1 connection))
	   (cwf     (nth 2 connection))
	   ;;(size (length string))
	   response msgs
	   )
      (if (and (not (null folder))
	       (not (string= cwf folder))
	       (null (elmo-imap4-select-folder 
		      folder connection)))
	  (error "Select folder failed"))
      (save-excursion
	(elmo-imap4-send-command buffer process 
				 (format 
				  (if elmo-imap4-use-uid 
				      "uid search header message-id %s"
				    "search header message-id %s")
				  msgid))
	(setq response (elmo-imap4-read-response buffer process))
	(if (and response 
		 (string-match "^\\* SEARCH\\([^\n]*\\)$" response))
	    (setq msgs (read (concat "(" (elmo-match-string 1 response) ")")))
	  (error "SEARCH failed."))
	(elmo-imap4-delete-msgs spec msgs)))))

(defun elmo-imap4-append-msg-by-id (spec msgid)
  (save-excursion
    (let* ((folder (nth 1 spec))
	   (hostname (nth 2 spec))
	   (username (nth 3 spec))
	   (auth     (nth 4 spec))
	   (port     (nth 5 spec))
	   (ssl      (nth 6 spec))
	   (connection (elmo-imap4-get-connection username hostname auth 
						  port ssl))
	   (buffer  (nth 0 connection))
	   (process (nth 1 connection))
	   (cwf     (nth 2 connection))
	   send-buf
	   )
      (if (and (not (null folder))
	       (not (string= cwf folder))
	       (null (elmo-imap4-select-folder 
		      folder connection)))
	  (error "Select folder failed"))
      (save-excursion
	(setq send-buf (elmo-imap4-setup-send-buffer-from-file 
			(elmo-cache-get-path msgid)))
	(set-buffer send-buf)
	(elmo-imap4-send-command buffer process 
				 (format "append %s (\\Seen) {%d}"
					 folder 
					 (buffer-size)))
	(process-send-string process (buffer-string))
	(process-send-string process "\r\n") ; finished appending.
	)
      (kill-buffer send-buf)
      (if (null (elmo-imap4-read-response buffer process))
	  (error "Append failed.")))
    t))

(defun elmo-imap4-append-msg (spec string &optional msg)
  (save-excursion
    (let* ((folder (nth 1 spec))
	   (hostname (nth 2 spec))
	   (username (nth 3 spec))
	   (auth     (nth 4 spec))
	   (port     (nth 5 spec))
	   (ssl     (nth 6 spec))
	   (connection (elmo-imap4-get-connection username hostname auth 
						  port ssl))
	   (buffer  (nth 0 connection))
	   (process (nth 1 connection))
	   (cwf     (nth 2 connection))
	   send-buf
	   )
      (if (and (not (null folder))
	       (not (string= cwf folder))
	       (null (elmo-imap4-select-folder 
		      folder connection)))
	  (error "Select folder failed"))
      (save-excursion
	(setq send-buf (elmo-imap4-setup-send-buffer string))
	(set-buffer send-buf)
	(elmo-imap4-send-command buffer process 
				 (format "append %s (\\Seen) {%d}"
					 folder 
					 (buffer-size)))
	(if (null (elmo-imap4-read-response buffer process))
	    (error "Cannot append messages to this folder."))
	(process-send-string process (buffer-string))
	(process-send-string process "\r\n") ; finished appending.
	)
      (kill-buffer send-buf)
      (current-buffer)
      (if (null (elmo-imap4-read-response buffer process))
	  (error "Append failed.")))
    t))

(defun elmo-imap4-copy-msgs (src-spec msgs dst-spec &optional expunge-it same-number)
  "Equivalence of hostname, username is assumed."
  (save-excursion
    (let* ((src-folder   (nth 1 src-spec))
	   (dst-folder (nth 1 dst-spec))
	   (connection (elmo-imap4-get-connection (nth 3 src-spec)
						  (nth 2 src-spec)
						  (nth 4 src-spec)
						  (nth 5 src-spec)
						  (nth 6 src-spec)
						  ))
	   (buffer  (nth 0 connection))
	   (process (nth 1 connection))
	   (cwf     (nth 2 connection))
	   (mlist msgs))
      (if (and (not (null src-folder))
	       (not (string= cwf src-folder))
	       (null (elmo-imap4-select-folder
		      src-folder connection)))
	  (error "Select folder failed"))
      (while mlist
	(elmo-imap4-send-command buffer process
				 (format
				  (if elmo-imap4-use-uid 
				      "uid copy %s %s" 
				    "copy %s %s")
				  (car mlist) dst-folder))
	(if (null (elmo-imap4-read-response buffer process))
	    (error "Copy failed.")
	  (setq mlist (cdr mlist))))
      (when expunge-it
	(elmo-imap4-send-command buffer process "expunge")
	(if (null (elmo-imap4-read-response buffer process))
	    (error "Expunge failed")))
      t)))

(defun elmo-imap4-use-cache-p (spec number)
  elmo-imap4-use-cache
  )

(defun elmo-imap4-local-file-p (spec number)
  nil
  )

(provide 'elmo-imap4)
