/*
                        VOC File Format Library
	Functions:

	FILE *VocOpenFile(char *filename, char *mode, off_t *size_rtn)
	int VocIsFileVoc(char *filename)
	unsigned int VocGetEnvSampleRate(voc_data_struct *vd)
	void VocDestroyData(voc_data_struct *vd)
	int VocAddDataBlock(voc_data_struct *vd, unsigned char type)
	int VocReadHeader(char *filename, voc_data_struct *vd)
	int VocReadPartialData(  
	        voc_data_struct *vd,
	        off_t offset,  
	        off_t max_chunk_size,
	        int read_opt
	)



	---

 */

#include <stdio.h>
#include <malloc.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>

#include "voc.h"



FILE *VocOpenFile(char *filename, char *mode, off_t *size_rtn)
{

        FILE *fp;
        struct stat stat_buf;


        if(filename == NULL)
        {
            fprintf(stderr, "Cannot open file with no name.\n");
            *size_rtn = 0;
            return(NULL);
        }
        if(mode == NULL)
        {
            fprintf(stderr, "%s: Open mode not givin.\n", filename);
            *size_rtn = 0;
            return(NULL);
        }

        if(stat(filename, &stat_buf))
        {
            fprintf(stderr, "%s: No such file.\n", filename);
            *size_rtn = 0;
            return(NULL);
        }
        *size_rtn = stat_buf.st_size;

        fp = fopen(filename, mode);
        if(fp == NULL)
        {
            fprintf(stderr, "%s: Cannot open.\n", filename);
            *size_rtn = 0;
            return(NULL);
        }


        return(fp);
}



int VocIsFileVoc(char *filename)
{
	int x;
	char magicname[0x14];
	FILE *fp;
	off_t filesize;


	/* Open file. */
	fp = VocOpenFile(filename, "rb", &filesize);
	if(fp == NULL)
	    return(VocErrorNoAccess);
        if(filesize == 0)
            return(VocErrorNoAccess);


	/* Bytes 0x00 to 0x13: Get magic name. */
	for(x = 0; x < 0x13; x++)
	{
	    magicname[x] = (char)fgetc(fp);
	}

	/* Close file. */
	fclose(fp); fp = NULL;

	/* Check magic name. */
	magicname[0x13] = '\0';
	if(strcmp(magicname, VocMagicName) == 0)
	    return(VocSuccess);
	else
	    return(VocErrorNotVoc);
}


unsigned int VocGetEnvSampleRate(voc_data_struct *vd)
{
	int x;
	unsigned int sample_rate;


	/* Set default. */
	sample_rate = VocDefaultSampleRate;

	if(vd == NULL)
	    return(sample_rate);


	for(x = 0; x < vd->total_datablocks; x++)
	{
	    if(vd->datablock[x] == NULL)
		continue;

	    /* Check if this type is a VocBlockTypeSoundData. */
	    if(vd->datablock[x]->type == VocBlockTypeSoundData)
	    {
		sample_rate = vd->datablock[x]->sample_rate;
		break;
	    }
	}

	/* Sanitize sample rate. */
	if(sample_rate == 0)
	    sample_rate = VocDefaultSampleRate;


	return(sample_rate);
}



void VocDestroyData(voc_data_struct *vd)
{
	int i;


	if(vd == NULL)
	    return;


	/* Free datablocks first. */
	if((vd->datablock != NULL) &&
           (vd->total_datablocks > 0)
	)
	{
	    for(i = 0; i < vd->total_datablocks; i++)
	    {
		if(vd->datablock[i] == NULL)
		    continue;

		free(vd->datablock[i]->str);

		free(vd->datablock[i]);
	    }

	    free(vd->datablock);
	    vd->datablock = NULL;

	    vd->total_datablocks = 0;
	}


	/* Reset values. */
	free(vd->filename);
	vd->filename = NULL;
	vd->filesize = 0;

        free(vd->data);
	vd->data = NULL;
        vd->data_len = 0;

	vd->total_data_len = 0;

	vd->major_version = 0;
        vd->minor_version = 0;
        vd->first_data_block = 0;


	return;
}



int VocAddDataBlock(voc_data_struct *vd, unsigned char type)
{
	int prev_total_blocks;


	/* Error checks. */
	if(vd == NULL)
	    return(-1);

	/* Sanitize total. */
	if(vd->total_datablocks < 0)
	    vd->total_datablocks = 0;

	/* Record previous total. */
	prev_total_blocks = vd->total_datablocks;

	/* Set new total. */
	vd->total_datablocks++;


	/* Realloc pointers. */
	vd->datablock = (voc_block_struct **)realloc(
	    vd->datablock, vd->total_datablocks * sizeof(voc_block_struct *)
	);
	if(vd->datablock == NULL)
	{
	    vd->total_datablocks = 0;
	    return(-1);
	}


	/* Allocate new data block. */
	vd->datablock[prev_total_blocks] = (voc_block_struct *)calloc(1,
	    sizeof(voc_block_struct)
	);
	if(vd->datablock[prev_total_blocks] == NULL)
	{
	    vd->total_datablocks = prev_total_blocks;
	    return(-1);
	}


	/* Set type. */
	vd->datablock[prev_total_blocks]->type = type;


	return(prev_total_blocks);
}



int VocReadHeader(char *filename, voc_data_struct *vd)
{
	int x, y, z, len;
	unsigned char type;
	off_t nextseek;

	/* To identify that this is a VOC file. */
	char magicname[0x14];

	/* For calculating that pesky 3 byte unsigned int block offset. */
	unsigned int size_x, size_y, size_z;

	int bytes_read = 0;
	FILE *fp;
	off_t filesize;


        /* Reset values. */
	memset(vd, 0x00, sizeof(voc_data_struct));


        /* Error checks. */
        if(vd == NULL)
            return(VocErrorBadValue);
        if(filename == NULL)
            return(VocErrorBadValue);


	/* Open file. */
	fp = VocOpenFile(filename, "rb", &filesize);
        if(fp == NULL)
            return(VocErrorNoAccess);
        if(filesize == 0)
            return(VocErrorNoAccess);


        /* Record file name. */
	len = strlen(filename);
        vd->filename = (char *)malloc((len + 1) * sizeof(char));
	if(vd->filename != NULL)
	{
            strncpy(vd->filename, filename, len);
	    vd->filename[len] = '\0';
	}

	/* Record file size. */
	vd->filesize = filesize;


	/* *********************************************************** */

	/* 0x00 to 0x12: Magic number (er name for VOC files). */
	for(x = 0; x < 0x13; x++)
	{
	    magicname[x] = (char)fgetc(fp);
	    bytes_read++;
	}
	magicname[0x13] = '\0';
	/* Check to see if this is a Voc file! */
	if(strcmp(magicname, VocMagicName) != 0)
	{
	    VocDestroyData(vd);
	    return(VocErrorNotVoc);
	}


	/* 0x13: Read but skip abort print byte. */
	fgetc(fp);
	bytes_read++;


	/* 0x14 to 0x15: Get offset to first data block. */
	x = fgetc(fp);
	y = fgetc(fp);
	bytes_read += 2;
	/* Calculate, LSBF. */
	vd->first_data_block = x + (y * 256);


	/* 0x16 to 0x17: Minor and major version numbers. */
	vd->minor_version = (unsigned char)fgetc(fp);
	vd->major_version = (unsigned char)fgetc(fp);
	bytes_read += 2;


	/* 0x18 to 0x19: Extended stuff. */
        x = fgetc(fp);
        y = fgetc(fp);
        bytes_read += 2;


	/* Move fp to start of first block. */
        fseek(fp, vd->first_data_block, SEEK_SET);
	nextseek = ftell(fp);

	while(nextseek < vd->filesize)
	{
	    fseek(fp, nextseek, SEEK_SET);

	    /* Read type. */
	    type = (unsigned char)fgetc(fp); bytes_read++;
	    switch(type)
	    {
              case VocBlockTypeTerminator:
		/* Allocate data block. */
		y = VocAddDataBlock((voc_data_struct *)vd, type);
		if(y < 0)
		    return(VocErrorNoBuffers);
		/* *** Terminator type has no size! *** */

		/* Get size. */
		vd->datablock[y]->size = 1;
		/* Set nextseek. */
		nextseek += 1;

		break;


	      case VocBlockTypeSoundData:
		/* Allocate data block. */
                y = VocAddDataBlock((voc_data_struct *)vd, type);
                if(y < 0)
                    return(VocErrorNoBuffers);

		/* Calculate and set data_start_pos. */
		vd->datablock[y]->data_start_pos = nextseek + 6;

		/* Get size. */
		size_x = (unsigned int)fgetc(fp);
                size_y = (unsigned int)fgetc(fp);
                size_z = (unsigned int)fgetc(fp);
		bytes_read += 3;
		vd->datablock[y]->size = size_x + (size_y << 8) +
		    (size_z << 16);
                /* Set nextseek. */
                nextseek += (vd->datablock[y]->size + 4);

		/* Get data_len. */
		if((long)vd->datablock[y]->size >= 2)
		    vd->datablock[y]->data_len = vd->datablock[y]->size - 2;
		else
		    vd->datablock[y]->data_len = 2;

		/* Get sample rate. */
		z = fgetc(fp); bytes_read++;
		if(z >= 256)
		    z = 255;
		vd->datablock[y]->sample_rate = (unsigned int)(
		    (double)(-1000000) / (double)(z - 256)
		);

		/* Get compresion type. */
		vd->datablock[y]->compression_type = (unsigned char)
		    fgetc(fp); bytes_read++;

		/* Calculate bits. */
		switch(vd->datablock[y]->compression_type)
		{
		  case 0x00:
		    vd->datablock[y]->bits = 8;
		    break;

                  case 0x01:
                    vd->datablock[y]->bits = 4;
                    break;

                  case 0x02:
                    vd->datablock[y]->bits = 2.6;
                    break;

                  case 0x03:
                    vd->datablock[y]->bits = 2;
                    break;

                  case 0x04:	/* Multi DAC, not standard */
                    vd->datablock[y]->bits = 8;
                    break;

		  default:
		    vd->datablock[y]->bits = 8;
		    break;
		}

		break;


	      case VocBlockTypeSoundDataContinue:
		/* Allocate data block. */
                y = VocAddDataBlock((voc_data_struct *)vd, type);
                if(y < 0)
                    return(VocErrorNoBuffers);

                /* Calculate and set data_start_pos. */
                vd->datablock[y]->data_start_pos = nextseek + 4;
                    
                /* Get size. */
                size_x = (unsigned int)fgetc(fp);
                size_y = (unsigned int)fgetc(fp);
                size_z = (unsigned int)fgetc(fp);
                bytes_read += 3;
                vd->datablock[y]->size = size_x + (size_y << 8) +
                    (size_z << 16);
                /* Set nextseek. */
                nextseek += (vd->datablock[y]->size + 4);

                /* Get data_len. */
                vd->datablock[y]->data_len = vd->datablock[y]->size;
                break;


	      case VocBlockTypeSilence:
                /* Allocate data block. */  
                y = VocAddDataBlock((voc_data_struct *)vd, type);
                if(y < 0)
                    return(VocErrorNoBuffers);

                /* Get size. */
                size_x = (unsigned int)fgetc(fp);
                size_y = (unsigned int)fgetc(fp);
                size_z = (unsigned int)fgetc(fp);
                bytes_read += 3;
                vd->datablock[y]->size = size_x + (size_y << 8) +
                    (size_z << 16);
                /* Set nextseek. */
                nextseek += (vd->datablock[y]->size + 4);

		/* Get silence length. */
		fread(&(vd->datablock[y]->silence_length), 1, 2, fp);
		bytes_read += 2;

		/* Get sample rate. */
                z = fgetc(fp); bytes_read++;
                if(z >= 256)
                    z = 255;
                vd->datablock[y]->sample_rate = (unsigned int)(
                    (double)(-1000000) / (double)(z - 256)
		);
		break;

	      case VocBlockTypeMarker:
                /* Allocate data block. */
                y = VocAddDataBlock((voc_data_struct *)vd, type);
                if(y < 0)
                    return(VocErrorNoBuffers);

                /* Get size. */
                size_x = (unsigned int)fgetc(fp);
                size_y = (unsigned int)fgetc(fp);
                size_z = (unsigned int)fgetc(fp);
                bytes_read += 3;
                vd->datablock[y]->size = size_x + (size_y << 8) +
                    (size_z << 16);
                /* Set nextseek. */
                nextseek += (vd->datablock[y]->size + 4);

		/* Get marker number. */
		fread(&(vd->datablock[y]->marker_num), 1, 2, fp);
		bytes_read += 2;
		break;

	      case VocBlockTypeASCII:
                /* Allocate data block. */
                y = VocAddDataBlock((voc_data_struct *)vd, type);
                if(y < 0)
                    return(VocErrorNoBuffers);

                /* Get size. */
                size_x = (unsigned int)fgetc(fp);
                size_y = (unsigned int)fgetc(fp);
                size_z = (unsigned int)fgetc(fp);
                bytes_read += 3;
                vd->datablock[y]->size = size_x + (size_y << 8) +
                    (size_z << 16);
                /* Set nextseek. */
                nextseek += (vd->datablock[y]->size + 4);

		/* Get string. */
		vd->datablock[y]->str_len = vd->datablock[y]->size;
		vd->datablock[y]->str = (char *)calloc(1,
		    (vd->datablock[y]->str_len + 1) * sizeof(char)
		);
		for(z = 0; z < vd->datablock[y]->size; z++)
		{
		    vd->datablock[y]->str[z] = (char)fgetc(fp);
		    bytes_read++;
		}
		vd->datablock[y]->str[z] = '\0';
		break;


	      case VocBlockTypeRepeat:
                /* Allocate data block. */
                y = VocAddDataBlock((voc_data_struct *)vd, type);
                if(y < 0)
                    return(VocErrorNoBuffers);

                /* Get size. */
                size_x = (unsigned int)fgetc(fp);
                size_y = (unsigned int)fgetc(fp);
                size_z = (unsigned int)fgetc(fp);
                bytes_read += 3;
                vd->datablock[y]->size = size_x + (size_y << 8) +
                    (size_z << 16);
                /* Set nextseek. */
                nextseek += (vd->datablock[y]->size + 4);

		/* Get repeat count. */
		fread(&(vd->datablock[y]->repeat_count), 1, 2, fp);
		bytes_read += 2;
		break;


	      case VocBlockTypeEndRepeat:
                /* Allocate data block. */
                y = VocAddDataBlock((voc_data_struct *)vd, type);
                if(y < 0)
                    return(VocErrorNoBuffers);

                /* Get size. */
                size_x = (unsigned int)fgetc(fp);
                size_y = (unsigned int)fgetc(fp);
                size_z = (unsigned int)fgetc(fp);
                bytes_read += 3;
                vd->datablock[y]->size = size_x + (size_y << 8) +
                    (size_z << 16);
                /* Set nextseek. */
                nextseek += (vd->datablock[y]->size + 4);

		/* There is no info for this field. */
		break;


	      case VocBlockTypeExtended: 
                /* Allocate data block. */
                y = VocAddDataBlock((voc_data_struct *)vd, type);
                if(y < 0)
                    return(VocErrorNoBuffers);

                /* Get size. */
                size_x = (unsigned int)fgetc(fp);
                size_y = (unsigned int)fgetc(fp);
                size_z = (unsigned int)fgetc(fp);
                bytes_read += 3;
                vd->datablock[y]->size = size_x + (size_y << 8) +
                    (size_z << 16);
                /* Set nextseek. */
                nextseek += (vd->datablock[y]->size + 4);

		/* Get time constant. */
		fread(&(vd->datablock[y]->time_constant), 1, 2, fp);
		bytes_read += 2;

		/* Get pack. */
		vd->datablock[y]->pack = (unsigned char)fgetc(fp);
		bytes_read++;

		/* Get mode (0 = mono, 1 = stereo). */
		vd->datablock[y]->mode = (unsigned char)fgetc(fp);
                bytes_read++;
		break;


              /* Unknown, assume field is just 1 byte line. */
	      default:
                /* Allocate data block. */  
                y = VocAddDataBlock((voc_data_struct *)vd, type);
                if(y < 0)
                    return(VocErrorNoBuffers);

                /* Get size. */
                size_x = (unsigned int)fgetc(fp);
                size_y = (unsigned int)fgetc(fp);
                size_z = (unsigned int)fgetc(fp);
                bytes_read += 3;
                vd->datablock[y]->size = size_x + (size_y << 8) +
                    (size_z << 16);
                /* Set nextseek. */
                nextseek += (vd->datablock[y]->size + 4);

                break;
	    }
	}


	/* Close file. */
	fclose(fp);
	fp = NULL;


	/* Add audio lengths from all data blocks containing audio data. */
	vd->total_data_len = 0;
	for(x = 0; x < vd->total_datablocks; x++)
	{
	    if(vd->datablock[x] == NULL)
		continue;

	    if( (vd->datablock[x]->type != VocBlockTypeSoundData) &&
                (vd->datablock[x]->type != VocBlockTypeSoundDataContinue)
	    )
		continue;

	    vd->total_data_len += vd->datablock[x]->data_len;
	}



	return(VocSuccess);
}



int VocReadPartialData(
        voc_data_struct *vd,
        off_t offset,
        off_t max_chunk_size,
        int read_opt		/* Reading format. */
)
{
	unsigned char type;
        int i;

	FILE *fp;

	int datablock_num;
	int total_datablocks;

	off_t src_buf_pos;
	int tar_buf_pos;

        off_t filesize;
	off_t offset_count;


        /* Error checks. */
        if(vd == NULL)
            return(VocErrorBadValue);
	if(vd->filename == NULL)
	    return(VocErrorBadValue); 


	/* Zero chunk size means free data. */
        if(max_chunk_size == 0)
	{
	    free(vd->data); vd->data = NULL;
	    vd->data_len = 0;

            return(VocSuccess);
	}


	/* Get total_datablocks. */
	total_datablocks = vd->total_datablocks;

	/* Free and allocate data buffer. */
        free(vd->data);
        vd->data_len = max_chunk_size;
	vd->data = (char *)calloc(
	    1,
	    vd->data_len * sizeof(char)
	);
	if(vd->data == NULL)
	{
	    vd->data_len = 0;
	    return(VocErrorNoBuffers);
	}


        /* Open the file. */
        fp = VocOpenFile(vd->filename, "rb", &filesize);
        if(fp == NULL)
            return(VocErrorNoAccess);
        if(filesize == 0)
            return(VocErrorNoAccess);
        

	tar_buf_pos = 0;
	datablock_num = 0;
	offset_count = 0;
	while((datablock_num < total_datablocks) &&
              (tar_buf_pos < max_chunk_size)
	)
	{
	    /* Skip unallocated datablocks. */
	    if(vd->datablock[datablock_num] == NULL)
	    {
		datablock_num++;
		continue;
	    }

	    /* Read only datablock types audio and audio continue. */
	    type = vd->datablock[datablock_num]->type;
	    if((type != VocBlockTypeSoundData) &&
	       (type != VocBlockTypeSoundDataContinue)
	    )
            {
                datablock_num++;
                continue;
            }

	    /* Skip if offset would not put us in this block. */
	    if((offset_count + vd->datablock[datablock_num]->data_len)
		< offset)
	    {
		offset_count += vd->datablock[datablock_num]->data_len;
                datablock_num++;
		continue;
	    }

	    /* Adjust file pointer. */
	    src_buf_pos = (off_t)((int)offset - (int)offset_count);
	    fseek(
		fp,
		vd->datablock[datablock_num]->data_start_pos + src_buf_pos,
		SEEK_SET
	    );

	    offset_count += vd->datablock[datablock_num]->data_len;
	    offset = offset_count;
	    while((tar_buf_pos < max_chunk_size) &&
                  (src_buf_pos < vd->datablock[datablock_num]->data_len)
	    )
	    {
		vd->data[tar_buf_pos] = (char)fgetc(fp);

		tar_buf_pos++;
		src_buf_pos++;
	    }

	    datablock_num++;
	}


	/* Limit data_len. */
	if(tar_buf_pos < vd->data_len)
	    vd->data_len = tar_buf_pos;


	fclose(fp);
	fp = NULL;


        /*   Shift the DSP, voc file format stores the values
	 *   as u_int8_t.
	 */
        switch(read_opt)
        {
          case VocReadSigned8:
            for(i = 0; i < vd->data_len; i++)
                vd->data[i] = (char)((int)vd->data[i] - (int)128);
            break;

          default:
            break;
        }




	return(VocSuccess);
}
