/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_sysaudio.c,v 1.3 1999/12/03 17:50:19 hercules Exp $";
#endif

/* Allow access to a raw mixing buffer (For IRIX 6.5 and higher) */

#include <dmedia/audio.h>

#include "SDL_endian.h"
#include "SDL_audio.h"
#include "SDL_audiomem.h"
#include "SDL_audio_c.h"


/* The handle to the audio device */
static ALport audio_port = NULL;

/* Raw mixing buffer */
static Uint8 *mixbuf;

void SDL_SYS_WaitAudio(void)
{
	Sint32 timeleft;

	timeleft = audio_spec.samples - alGetFillable(audio_port);
	if ( timeleft > 0 ) {
		timeleft /= (audio_spec.freq/1000);
		SDL_Delay((Uint32)timeleft);
	}
}

void SDL_SYS_PlayAudio(void)
{
	/* Write the audio data out */
	if ( alWriteFrames(audio_port, mixbuf, audio_spec.samples) < 0 ) {
		/* Assume fatal error, for now */
		SDL_AudioEnabled = 0;
	}
}

Uint8 *SDL_SYS_GetAudioBuf(void)
{
	return(mixbuf);
}

void SDL_SYS_CloseAudio(void)
{
	if ( mixbuf != NULL ) {
		SDL_FreeAudioMem(mixbuf);
		mixbuf = NULL;
	}
	ALcloseport(audio_port);
}

int SDL_SYS_OpenAudio(SDL_AudioSpec *spec)
{
	ALconfig audio_config;
	ALpv audio_param;
	int width, channels;

	/* Determine the audio parameters from the AudioSpec */
	switch ( spec->format & 0xFF ) {

		case 8: { /* Signed 8 bit audio data */
			spec->format = AUDIO_S8 | (spec->format&AUDIO_STEREO);
			width = AL_SAMPLE_8;
		}
		break;

		case 16: { /* Signed 16 bit audio data */
			spec->format = AUDIO_S16MSB|(spec->format&AUDIO_STEREO);
			width = AL_SAMPLE_16;
		}
		break;

		default: {
			SDL_SetError("Unsupported audio format");
			return(-1);
		}
	}
	if ( spec->format & AUDIO_STEREO ) {
		channels = 2;
	} else {
		channels = 1;
	}

	/* Update the fragment size as size in bytes */
	SDL_CalculateAudioSpec(spec);

	/* Set output frequency */
	audio_param.param = AL_RATE;
	audio_param.value.i = spec->freq;
	if( alSetParams(AL_DEFAULT_OUTPUT, &audio_param, 1) < 0 ) {
		SDL_SetError("alSetParams failed");
		return(-1);
	}

	/* Open the audio port with the requested frequency */
	audio_port = NULL;
	audio_config = alNewConfig();
	if ( audio_config &&
	     (alSetSampFmt(audio_config, AL_SAMPFMT_TWOSCOMP) >= 0) &&
	     (alSetWidth(audio_config, width) >= 0) &&
	     (alSetQueueSize(audio_config, spec->samples*2) >= 0) &&
	     (alSetChannels(audio_config, channels) >= 0) ) {
		audio_port = ALopenport("SDL audio", "w", audio_config);
	}
	alFreeConfig(audio_config);
	if( audio_port == NULL ) {
		SDL_SetError("Unable to open audio port");
		return(-1);
	}

	/* Allocate mixing buffer */
	mixbuf = (Uint8 *)SDL_AllocAudioMem(spec->size);
	if ( mixbuf == NULL ) {
		SDL_OutOfMemory();
		return(-1);
	}
	memset(mixbuf, spec->silence, spec->size);

	/* We're ready to rock and roll. :-) */
	return(0);
}
