/* File "debugger.c":
 * Debugger functions. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* includes =================================================================*/

#include <stdio.h>
#include <setjmp.h>
#include <stdlib.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "symbols.h"
#include "files.h"
#include "input.h"
#include "commands.h"
#include "options.h"
#include "instr_type.h"
#include "rule_type.h"
#include "rules.h"
#include "display_process.h"
#include "breakpoints.h"

#undef GLOBAL
#define GLOBAL
#include "debugger.h"

/* variables ================================================================*/

LOCAL debug_mode_t debug_mode = RUN_MODE; /* current debug mode */

LOCAL struct /* definition of the step properties */
{
  long_t nested_subrules; /* depth of subrule nesting when stepping started */
  long_t backup_top;      /* number of inactive paths when stepping started */
  long_t line;            /* line where stepping started */
  string_t file;          /* file where stepping started */
  short_t repetitions;    /* repetitions of steps still to do */
} step;

LOCAL void (*print_rule) (void); 
/* pointer to a function that prints the current rule name */

LOCAL command_t **debugger_commands;
/* commands that can be executed when the debug mode is invoked */

/* functions ================================================================*/

LOCAL bool_t at_next_line (rule_sys_t *rule_sys, long_t pc)
/* Compare current source file name and source line to previous values.
 * If one of them has changed, return TRUE. */
{
  long_t line;
  string_t file;
  
  switch (debug_mode)
  {
  case NEXT_MODE:
    if (backup_top < step.backup_top)
      /* Current path is finished. Debug older path. */
    {
      step.nested_subrules = nested_subrules;
      step.backup_top = backup_top;
    }
    if (nested_subrules > step.nested_subrules)
      break;
    /* fall through to STEP_MODE */
  case STEP_MODE:
    source_of_instr (rule_sys, pc, &line, NULL, &file, NULL);
    if ((line == -1 && file == NULL) 
	|| (step.line == line && step.file == file))
      break;
    
    step.nested_subrules = nested_subrules;
    step.line = line;
    step.file = file;
    
    /* Ignore interactive mode while number of steps > 1. */
    if (step.repetitions > 1)
      step.repetitions--;
    else
      return TRUE;
    break;
  case WALK_MODE:
    if (rule_sys->rules[executed_rule_number].first_instr == pc)
      return TRUE;
    break;
  default:
    break;
  }
  return FALSE;
}

/*---------------------------------------------------------------------------*/

GLOBAL void set_debug_mode (debug_mode_t new_debug_mode, rule_sys_t *rule_sys)
/* Set debug mode <new_debug_mode> for <rule_sys>. */
{
  switch (new_debug_mode)
    {
    case RUN_MODE:
      debug_rule_sys = NULL;
      break;
    case STEP_MODE:
    case NEXT_MODE:
      step.backup_top = 0;
      step.nested_subrules = 0;
      step.line = -1;
      step.file = NULL;
      step.repetitions = 0;
      debug_rule_sys = rule_sys;
      break;
    case GO_MODE:
    case WALK_MODE:
      debug_rule_sys = rule_sys;
      break;
    }
  debug_mode = new_debug_mode;
}

/*---------------------------------------------------------------------------*/

LOCAL void check_if_rule_start (void)
/* Look if <pc> points at rule start, and print an appropriate message.
 * Called from "debug_rule" before entering the command loop. */
{
  rule_t *rule = executed_rule_sys->rules + executed_rule_number;
  
  if (rule->first_instr == pc) 
    print_rule ();
}

/*---------------------------------------------------------------------------*/

LOCAL void display_variables (void)
{
  long_t stack_top = inspect_stack_pointer ();
  long_t i;

  start_display_process ();

  fprintf (display_stream, "variables\n");

  for (i = first_variable_index (executed_rule_sys, pc); i < stack_top; i++)
  {
    string_t var_name = variable_at_index (executed_rule_sys, i, pc);
    value_t value;
    
    if (var_name != NULL && (value = inspect_stack (i)) != NULL)
    {
      /* Decode variable value. */
      fprintf (display_stream, "\"$%s\" {", var_name);
      fprint_value (display_stream, value);
      fprintf (display_stream, "}\n");
    }
  }
  
  fprintf (display_stream, "end\n");
  fflush (display_stream);
}

/*---------------------------------------------------------------------------*/

LOCAL void debugger_debug_rule (void)
/* Called from "execute_rule" before instruction pointed to by
 * <pc> is executed. */
{
  short_t breakpoint_number;
  
  executing_rule = FALSE;
  in_debugger = TRUE;
  
  breakpoint_number = at_breakpoint (executed_rule_sys, pc);
  if (breakpoint_number != 0 || at_next_line (executed_rule_sys, pc))
  {
    long_t line;
    string_t file, rule;
    
    source_of_instr (executed_rule_sys, pc, &line, NULL, &file, &rule);
    
    if (breakpoint_number != 0)
      printf ("hit breakpoint %d in file \"%s\", line %ld, rule \"%s\"\n", 
	      breakpoint_number, name_in_path (file), line, rule);
    else if (getenv ("MALAGA_MODE") == NULL)
      printf ("step to file \"%s\", line %ld, rule \"%s\"\n", 
	      name_in_path (file), line, rule);
    
    if (getenv ("MALAGA_MODE") != NULL)
      printf ("SHOW \"%s\":%ld:0\n", file, line);
    
    /* check if we are at rule start and print an appropriate message */
    check_if_rule_start ();

    if (show_variables)
      display_variables ();

    command_loop ("debug", debugger_commands);
    if (quit_requested)
      set_debug_mode (RUN_MODE, NULL);
  }
  
  in_debugger = FALSE;
  executing_rule = TRUE;
}

/*---------------------------------------------------------------------------*/

GLOBAL void init_debugger (void (*my_print_rule) (void), 
			   command_t *my_debugger_commands[])
/* Initialise the debugger module.
 * Pass a function <my_print_rule> printing the current rule, and commands
 * <my_debugger_commands> that can be invoked when debug mode is invoked. */
{
  print_rule = my_print_rule;
  debugger_commands = my_debugger_commands;
  debug_rule = debugger_debug_rule;
}

/*---------------------------------------------------------------------------*/

GLOBAL void terminate_debugger (void)
/* Terminate the debugger module. */
{
  stop_display_process ();
  debug_rule = NULL;
}

/*---------------------------------------------------------------------------*/

LOCAL void step_or_next (debug_mode_t debug_mode, string_t argument)
/* Execute "step" or "next" command (see <debug_mode>) with <argument>. */
{
  long_t line;
  string_t file;
  long_t repetitions;

  if (! in_debugger)
    error ("not in debugging mode");
  
  source_of_instr (executed_rule_sys, pc, &line, NULL, &file, NULL);
  
  DB_ASSERT (line != -1);
  
  if (*argument == EOS)
    repetitions = 0;
  else
    repetitions = parse_integer (&argument);
  parse_end (argument);
  
  set_debug_mode (debug_mode, executed_rule_sys);
  step.backup_top = backup_top;
  step.nested_subrules = nested_subrules;
  step.line = line;
  step.file = file;
  step.repetitions = repetitions;
}

/*---------------------------------------------------------------------------*/

LOCAL void do_run (string_t arguments)
/* Continue rule execution in non-debugging mode. */
{
  if (! in_debugger)
    error ("not in debugging mode");

  parse_end (arguments);
  set_debug_mode (RUN_MODE, NULL);
  leave_command_loop = TRUE;
}

GLOBAL command_t run_command =
{
  "run r", do_run,
  "Return to non-debugging mode and complete rule execution.\n"
  "Arguments: (none)\n"
  "\"run\" can only be used in debug mode.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_go (string_t arguments)
/* Continue rule execution until next breakpoint is met or end is reached. */
{
  if (! in_debugger)
    error ("not in debugging mode");

  parse_end (arguments);
  set_debug_mode (GO_MODE, executed_rule_sys);
  leave_command_loop = TRUE;
}

GLOBAL command_t go_command =
{
  "go g", do_go,
  "Execute rules until a breakpoint is hit or analysis is completed.\n"
  "Arguments: (none)\n"
  "\"go\" can only be used in debug mode.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_walk (string_t arguments)
/* Continue rule execution until next rule or breakpoint is met 
 * or end is reached. */
{
  if (! in_debugger)
    error ("not in debugging mode");

  parse_end (arguments);
  set_debug_mode (WALK_MODE, executed_rule_sys);
  leave_command_loop = TRUE;
}

GLOBAL command_t walk_command =
{
  "walk w", do_walk,
  "Walk to the next rule.\n"
  "Arguments: (none)\n"
  "\"walk\" can only be used in debug mode.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_step (string_t argument)
/* Continue until control reaches a different source line. */
{
  step_or_next (STEP_MODE, argument);
  leave_command_loop = TRUE;
}

GLOBAL command_t step_command =
{
  "step s", do_step,
  "Step to the next source line.\n"
  "Arguments:\n"
  "  <repetitions> -- execute command <repetitions> time\n"
  "  (none) -- execute one step\n"
  "\"step\" can only be used in debug mode.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_next (string_t argument)
/* Continue until control reaches a different source line, but jump over
 * subrules. */
{
  step_or_next (NEXT_MODE, argument);
  leave_command_loop = TRUE;
}

GLOBAL command_t next_command =
{
  "next n", do_next,
  "Step to the next source line, but jump over subrules.\n"
  "Arguments:\n"
  "  <repetitions> -- execute <repetitions> steps\n"
  "  (none) -- execute one step\n"
  "\"next\" can only be used in debug mode.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_trace (string_t arguments)
/* Print all active subrules and rules in reverse order. */
{
  long_t index;
  
  if (pc == -1)
    error ("no rule executed");

  parse_end (arguments);

  index = 0;
  do
  {
    string_t file, rule;
    long_t line;
    long_t pc_index;

    get_base_and_pc_indexes (index, &index, &pc_index);
    source_of_instr (executed_rule_sys, pc_index, &line, NULL, &file, &rule);
    printf ("line %ld in file \"%s\", rule \"%s\"\n", 
	    line, name_in_path (file), rule);
  } while (index != 0);
}

GLOBAL command_t trace_command =
{
  "trace", do_trace,
  "Print the active subrules in reverse order.\n"
  "Arguments: (none)\n"
  "\"trace\" can only be used in debug mode or after a rule execution error.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_print (string_t argument)
/* Print content of variables. */
{
  long_t stack_index;
  value_t value;

  if (pc == -1)
    error ("no rule executed");

  if (*argument != EOS) 
  {
    /* If any arguments given, treat each argument as a variable. */
    while (*argument != EOS) 
    {
      string_t path = parse_word (&argument);
      string_t section, sect_start, sect_end;
      
      if (*path != '$')
	error ("variable names start with \"$\"");
      sect_start = path + 1;
      
      /* Find end of variable name. */
      sect_end = sect_start;
      while (*sect_end != EOS && *sect_end != '.')
	sect_end++;
      
      /* Get variable name and its value. */
      section = new_string_section (sect_start, sect_end);
      stack_index = variable_index (executed_rule_sys, section, pc);
      if (stack_index == -1)
	error ("variable \"$%s\" is not defined here", section);
      free (section);
      value = inspect_stack (stack_index);
      
      while (*sect_end != EOS)
      {
	/* Skip "." and remember section start. */
	sect_end++;
	sect_start = sect_end;
	
	/* Find end of section. */
	while (*sect_end != EOS && *sect_end != '.')
	  sect_end++;
	
	section = new_string_section (sect_start, sect_end);
	if (IS_DIGIT (*section) || *section == '-')
	{
	  long_t index;
	  string_t s;

	  s = section;
	  index = parse_integer (&s);
	  value = get_element (value, index);
	}
	else
	  value = get_attribute (value, find_symbol (section));

	free (section);
	if (value == NULL) 
	  break;
      }	  
      
      if (value == NULL)
	printf ("path \"%s\" doesn't exist\n", path);
      else
      {
	printf ("%s = ", path);
	print_value (value);
	printf ("\n");
      }
      
      free (path);
    }
  } 
  else 
  {
    long_t stack_top = inspect_stack_pointer ();
    
    /* Print all variables currently defined. */
    for (stack_index = first_variable_index (executed_rule_sys, pc); 
	 stack_index < stack_top; 
	 stack_index++) 
    {
      string_t var_name;
      
      var_name = variable_at_index (executed_rule_sys, stack_index, pc);
      if (var_name != NULL)
      {
	value = inspect_stack (stack_index);
	printf ("$%s = ", var_name);
	print_value (value);
	printf ("\n");
      }
    }
  }
}

GLOBAL command_t print_command = 
{
  "print p", do_print,
  "Print the values of variables.\n"
  "Arguments:\n"
  "  <variable> ... -- print the values of the specified variables\n"
  "  (none) -- print all currently defined variables\n"
  "Each variable may include an attribute path.\n"
  "\"print\" can only be used in debug mode or after a rule execution error.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_variables (string_t arguments)
/* Generate variables file and start TCL program to display variables. */
{
  if (pc == -1)
    error ("no rule executed");
  
  parse_end (arguments);
  display_variables ();
}

GLOBAL command_t variables_command = 
{
  "variables v", do_variables,
  "Display current variables in matrix form.\n"
  "Arguments: (none)\n"
  "\"variables\" can only be used in debug mode or after a rule execution"
  "error.\n"
};

/*---------------------------------------------------------------------------*/
